(**
 * TCorelGlyphButton v1.0
 * ---------------------------------------------------------------------------
 * A standard TButton which mimic the buttons used in the new Corel products
 * (e.g. WordPerfect Suite and Corel Photopaint).
 *
 * Copyright 1998, Peter Theill.  All Rights Reserved.
 *
 * This component can be freely used and distributed in commercial and private
 * environments, provied this notice is not modified in any way and there is
 * no charge for it other than nomial handling fees.  Contact me directly for
 * modifications to this agreement.
 * ----------------------------------------------------------------------------
 * Feel free to contact me if you have any questions, comments or suggestions
 * at peter@conquerware.dk
 *
 * The latest version will always be available on the web at:
 *   http://www.conquerware.dk/delphi/
 *
 * See CorelButton.txt for notes, known issues and revision history.
 * ----------------------------------------------------------------------------
 * Last modified: September 6, 1998
 * ----------------------------------------------------------------------------
 *)
unit CorGlBtn;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons;

type
  TGlyphKind = (glOK, glCancel, glHelp, glYes,
                 glNo, glClose, glAbort, glRetry, glIgnore, glAll, glNone);

  TCorelGlyphButton = class(TButton)
  private
    FCanvas: TCanvas;
    IsFocused: Boolean;
    FIsMouseOver: Boolean;
    FGlyphKind : TGlyphKind ;
    {FCaption : TCaption ;}
    FGlyphWidth : integer ;
    BMP : TBitMap ;
    {procedure SetCaption (c : TCaption) ;}
    procedure SetGlyphKind (K : TGlyphKind) ;
    procedure CNMeasureItem(var Msg: TWMMeasureItem); message CN_MEASUREITEM;
    procedure CNDrawItem(var Msg: TWMDrawItem); message CN_DRAWITEM;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;

    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;

    procedure DrawItem(const DrawItemStruct: TDrawItemStruct);

  protected
    procedure CreateParams(var Params: TCreateParams); override;

    procedure SetButtonStyle(ADefault: Boolean); override;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

  published
    property GlyphKind : TGlyphKind read FGlyphKind write SetGlyphKind ;
    property GlyphWidth : integer read FGlyphWidth write FGlyphWidth ;
    {property Caption : TCaption read FCaption write SetCaption ;}
  end;

procedure Register;

implementation

{$R CorGlBtn.res}

function DelAllOccurances( text, del: String ): String;
var      i: LongInt;
begin
     i:=0;
     while (i<Length(text)) do begin
           if text[i]=del then begin
              Delete(text,i,1);
           end else begin
               Inc(i);
           end;
     end;
     Result:=text;
end;

{procedure TCorelGlyphButton.SetCaption (c : TCaption) ;
begin
     FCaption := c ;
     invalidate ;
end ;}

procedure TCorelGlyphButton.SetGlyphKind (K : TGlyphKind) ;
var
 RN : string ;
begin
     FGlyphKind := K ;
     bmp.TransparentColor := clOlive ;
     bmp.TransparentMode := tmFixed ;
     bmp.Transparent := true ;
     try
        { bkOK, bkCancel, bkHelp, bkYes,
          bkNo, bkClose, bkAbort, bkRetry, bkIgnore, bkAll, bkNone }
        case FGlyphKind of
            glOK : RN := 'GLOK' ;
            glCancel : RN := 'GLCANCEL' ;
            glHelp : RN := 'GLHELP' ;
            glYes : RN := 'GLYES' ;
            glNo : RN := 'GLNO' ;
            glClose : RN := 'GLCLOSE' ;
            glAbort : RN := 'GLABORT' ;
            glRetry : RN := 'GLRETRY' ;
            glIgnore : RN := 'GLIGNORE' ;
            glAll : RN := 'GLALL' ;
            glNone : RN := 'GLNONE' ;
        end ;
        bmp.LoadFromResourceName(HInstance,RN);
     except
     end ;

     if FGlyphKind <> glNone
     then FGlyphWidth := bmp.width
     else FGlyphWidth := 0 ;

     //Fcanvas.brush.color := clBlue ;
     //Fcanvas.FillRect (Rect(50,50,50+20,50+20)) ;
     invalidate ;
end ;

constructor TCorelGlyphButton.Create(AOwner: TComponent);
begin

  { Do standard stuff }
  inherited Create(AOwner);

  FCanvas := TCanvas.Create;

  FIsMouseOver := False;
  FGlyphKind := glNone ;
  FGlyphWidth := 0 ;
  {FCaption := '' ;}
  BMP := TBitMap.create ;

  { Set width and height of button }
  Width := 75;
  Height := 25;

end;

destructor TCorelGlyphButton.Destroy;
begin
  FCanvas.Free;

  inherited Destroy;
end;

procedure TCorelGlyphButton.CMMouseEnter(var Message: TMessage);
begin

  if (not FIsMouseOver) then
    Invalidate;

end;

procedure TCorelGlyphButton.CMMouseLeave(var Message: TMessage);
begin

  if (FIsMouseOver) then
    Invalidate;

end;

procedure TCorelGlyphButton.CNMeasureItem(var Msg: TWMMeasureItem);
begin
  with Msg.MeasureItemStruct^ do begin
    itemWidth := Width;
    itemHeight := Height;
  end;
  Msg.Result := 1;
end;

procedure TCorelGlyphButton.CNDrawItem(var Msg: TWMDrawItem);
begin
  DrawItem(Msg.DrawItemStruct^);
  Msg.Result := 1;
end;

procedure TCorelGlyphButton.DrawItem(const DrawItemStruct: TDrawItemStruct);
var
  IsDown, IsDefault: Boolean;
  R: TRect;
  Xb, Yb, Xt : integer ;
//  Flags: Longint;
  CursorPos: TPoint;
  T, BtnRect: TRect;
  SimpleCaption : string ;   //Resource Name variable of glyph
begin

  FCanvas.Handle := DrawItemStruct.hDC;
  try
    R := ClientRect;

    with DrawItemStruct do begin
      IsDown := (itemState and ODS_SELECTED) <> 0;
      IsDefault := (itemState and ODS_FOCUS) <> 0;
    end;

    GetCursorPos(CursorPos);
    BtnRect.TopLeft := Parent.ClientToScreen(Point(Left, Top));
    BtnRect.BottomRight := Parent.ClientToScreen(Point(Left + Width,
       Top + Height));
    FIsMouseOver := PtInRect(BtnRect, CursorPos);

//    Flags := DFCS_BUTTONPUSH or DFCS_ADJUSTRECT;
//    if IsDown then Flags := Flags or DFCS_PUSHED;
//    if DrawItemStruct.itemState and ODS_DISABLED <> 0 then
//      Flags := Flags or DFCS_INACTIVE;

    FCanvas.Brush.Color := clBtnFace;

    if {(csDesigning in ComponentState) OR} (IsFocused) or (IsDefault) then begin

      FCanvas.Pen.Color := clWindowText;
      FCanvas.Pen.Width := 1;
      FCanvas.Brush.Style := bsSolid;
      FCanvas.Rectangle(R.Left, R.Top, R.Right, R.Bottom);

      InflateRect(R, -1, -1);

    end;

    FCanvas.FillRect(R);

    if {(csDesigning in ComponentState) OR} (FIsMouseOver) then begin

      FCanvas.Pen.Color := clWindowText;
      FCanvas.MoveTo(R.Right-1, R.Top);
      FCanvas.LineTo(R.Right-1, R.Bottom-1);
      FCanvas.LineTo(R.Left-1, R.Bottom-1);

      FCanvas.Pen.Color := clBtnHighlight;
      FCanvas.MoveTo(R.Left, R.Bottom-2);
      FCanvas.LineTo(R.Left, R.Top);
      FCanvas.LineTo(R.Right-1, R.Top);

      FCanvas.Pen.Color := clBtnShadow;
      FCanvas.MoveTo(R.Right-2, R.Top+1);
      FCanvas.LineTo(R.Right-2, R.Bottom-2);
      FCanvas.LineTo(R.Left, R.Bottom-2);

    end else begin

      FCanvas.Pen.Color := clBtnHighlight;
      FCanvas.Pen.Width := 1;
      FCanvas.MoveTo(R.Left, R.Bottom-2);
      FCanvas.LineTo(R.Left, R.Top);
      FCanvas.LineTo(R.Right-1, R.Top);

      FCanvas.Pen.Color := clBtnShadow;
      FCanvas.LineTo(R.Right-1, R.Bottom-1);
      FCanvas.LineTo(R.Left-1, R.Bottom-1);

    end;

    if {(csDesigning in ComponentState) OR} (IsDown) then begin

      FCanvas.Brush.Color := clBtnFace;
      FCanvas.FillRect(R);

      FCanvas.Pen.Color := clBtnShadow;
      FCanvas.Pen.Width := 1;
      FCanvas.MoveTo(R.Left, R.Bottom-2);
      FCanvas.LineTo(R.Left, R.Top);
      FCanvas.LineTo(R.Right-1, R.Top);

      FCanvas.Pen.Color := clBtnHighlight;
      FCanvas.LineTo(R.Right-1, R.Bottom-1);
      FCanvas.LineTo(R.Left-1, R.Bottom-1);

    end;

    if {(csDesigning in ComponentState) OR} (IsFocused) and (IsDefault) then begin

      InflateRect(R, -3, -3);
      FCanvas.Pen.Color := clWindowFrame;
      FCanvas.Brush.Color := clBtnFace;
      DrawFocusRect(FCanvas.Handle, R);

    end;


    FCanvas.Font := Self.Font;

    { align text and glyph }
    SimpleCaption := Caption ;
    SimpleCaption := DelAllOccurances (SimpleCaption,'&') ;
    if FGlyphKind <> glNone
    then begin
              Xb := (Clientwidth - FCanvas.TextWidth(SimpleCaption) - bmp.Width - 4) div 2 ;
              Yb := (ClientHeight - bmp.height) div 2 ;
              Xt := Xb + bmp.width + 4 ;
              T := R ;
              T.Left := Xt ;
              if IsDown
              then FCanvas.Draw (Xb+1,Yb+1,BMP)
              else FCanvas.Draw (Xb,Yb,BMP) ;
         end
    else begin
              Xt := (Width - FCanvas.textwidth(SimpleCaption)) div 2 ;
              T := R ;
              T.Left := Xt ;
         end ;

    if (IsDown)
    then begin
              OffsetRect(R, 1, 1);
              OffsetRect(T, 1, 1);
         end ;
    { Draw caption of button }
    with FCanvas do begin
      Brush.Style := bsClear;
      Font.Color := clBtnText;
      if Enabled or ((DrawItemStruct.itemState and ODS_DISABLED) = 0) then begin
        DrawText(Handle, PChar(Caption), Length(Caption), T, DT_LEFT or
         DT_VCENTER or DT_SINGLELINE);
      end else begin
        OffsetRect(R, 1, 1);
        Font.Color := clBtnHighlight;
        DrawText(Handle, PChar(Caption), Length(Caption), T, DT_LEFT or
         DT_VCENTER or DT_SINGLELINE);
        OffsetRect(R, -1, -1);
        Font.Color := clBtnShadow;
        DrawText(Handle, PChar(Caption), Length(Caption), T, DT_LEFT or
         DT_VCENTER or DT_SINGLELINE);
      end;
    end;

  finally
    FCanvas.Handle := 0;
  end;
end;

procedure TCorelGlyphButton.CMFontChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TCorelGlyphButton.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TCorelGlyphButton.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  Perform(WM_LBUTTONDOWN, Message.Keys, Longint(Message.Pos));
end;

procedure TCorelGlyphButton.SetButtonStyle(ADefault: Boolean);
begin

  if ADefault <> IsFocused then begin
    IsFocused := ADefault;
    Refresh;
  end;
end;

procedure TCorelGlyphButton.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style OR BS_OWNERDRAW;
end;


procedure Register;
begin
  RegisterComponents('MyStuff', [TCorelGlyphButton]);
end;

end.

