unit GHUDP;
{ Simple UDP server and Client written by Gordon Hamm
  gordon@yabin.com Uses absolutly no Windows messages
  360-686-8315
}

interface
uses windows, winsock, SysUtils, Classes, StdCtrls, DIALOGS;

const
  ReadBufferSize = 8000; // max to read at a time using Recv() or write

type
  EReadFail = class(Exception); // general read fail, disconnect etc.
  EReadTimeout = class(Exception); // general read fail, disconnect etc.
  EWriteFail = class(Exception); // write fail, disconnect etc.
  ESocketError = class(Exception); // Misc Socket Error.


type
  TGHUDPServer = class;
  TOnData = procedure(TheData: string; addr_remote: string) of object;
  TListenerThread = class(TThread) // thread to watch for incoming data
  private
    ListenSocketHandle: Integer;
    addr: TSockAddr;
    parentOBJ: TGHUDPServer;
    Sbuffer: string; // buff for api calls
    addrsize: integer;
    addr_remote: TSockAddrin;
    function Read: integer;
    function WaitForReadable(Timeout: Integer): boolean;
    procedure execute; override;
  end;


  TGHUDPServer = class(Tcomponent)
  private
    Fport: integer; // listen port
    Factive: Boolean; // system started ??
    FOnData: TOnData;
    WData: TWSAData;
    ListenSocketHandle: Integer;
    ListenerThread: TlistenerThread;
    procedure SetActive(const bValue: Boolean);
    function Start: boolean; // start listen..
    procedure Stop; // stop listen
  public
    GHSOCK_CS: TRTLCriticalSection;
    work_CS: TRTLCriticalSection;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Active: boolean read Factive write SetActive default False;
    // Port to listen on
    property Port: Integer read Fport write Fport default 6711;
    property OnData: TOnData read FOnData write FOnData;
  end;

  TGHUDPClient = class(TComponent)
  private
    FHandle: TSocket;
    WData: TWSAData;
    FCacheDNSLookup: boolean;
    DNSLookupList: TStringList;
    function ResolveHost(SHost: string): u_long;
  protected
  public
    procedure SendTo(sData: string; const sHost: string; Port: Integer);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy;
  published
    property CacheDnsLookup: boolean read FCacheDNSLookup write FCacheDNSLookup;
  end;




procedure Register;


implementation

constructor TGHUDPServer.Create(AOwner: TComponent);
begin
  inherited create(Aowner);
  InitializeCriticalSection(GHSOCK_CS); // initialize my Critical section.
  InitializeCriticalSection(Work_CS); // initialize my Critical section.
  if fport = 0 then // default listening port..
    fport := 6711;
end;


destructor TGHUDPServer.destroy;
begin
  inherited destroy;
end;

procedure TGHUDPServer.SetActive(const bValue: Boolean);
begin
  if (csDesigning in ComponentState) then
  begin
    factive := bvalue;
    exit;
  end;
  if bvalue and not factive then
    start
  else
    if not bvalue and factive then
      Stop;
end;

function TGHUDPServer.Start;
var
  Ierror, x: Integer;
  addr: TSockAddr;
begin
  result := False;
  if (not Factive) then // if its already active, dont start again
  begin

    Ierror := WSAStartup(MakeWord(1, 1), WData);
    if Ierror <> 0 then
    begin
      raise ESocketError.Create('Failed to initialize WinSocket,error #' + inttostr(Ierror));
      //showmessage('Failed to initialize WinSocket,error #' + inttostr(Ierror));
      Exit;
    end;
    try
      // open socket here.
      ListenSocketHandle := socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
      if ListenSocketHandle = INVALID_SOCKET then
      begin
        raise ESocketError.Create('Failed to create a socket, Error #' + Inttostr(WSAGetLastError));
        exit;
      end;

      Addr.sin_family := AF_INET;
      Addr.sin_port := htons(Fport);
      Addr.sin_addr.s_addr := INADDR_ANY;
      // bind the socket..
      if bind(ListenSocketHandle, Addr, SizeOf(Addr)) = SOCKET_ERROR then
      begin
        raise ESocketError.Create('Failed to bind the socket, error #' + Inttostr(WSAGetLastError) + '.'#13#10#13#10 +
          'Probable reason is that another daemon is already running on the same port (' + Inttostr(fport) + ').');

        Ierror := WSACleanup;
        exit
      end;

      ListenerThread := TlistenerThread.create(True);
      ListenerThread.ParentObj := Self;
      ListenerThread.ListenSocketHandle := ListenSocketHandle;
      ListenerThread.addr := addr;
      ListenerThread.resume;

      Factive := True;
      Result := True;
    finally

    end;
  end;

end;

procedure TGHUDPServer.stop;
var
  Ierror, X: integer;
begin
  if (Factive) then
  begin
    ListenerThread.terminate;
    shutdown(ListenerThread.ListenSocketHandle, 2);
    CloseSocket(ListenerThread.ListenSocketHandle);
    EnterCriticalSection(GHSOCK_CS);
    sleep(1000);
    Ierror := WSACleanup;
    waitforsingleobject(ListenerThread.handle, 5000);
    ListenerThread.Free;
    Factive := False;
  end;
end;


procedure TListenerThread.execute;
var
  readlen: integer;
begin
  //Freeonterminate := True; // we will free this thread ourself
  addrsize := SizeOf(Addr_remote);
  setlength(Sbuffer, ReadBufferSize);
  try
    while not terminated do
    begin
      try
        ReadLen := Read;
        if readLen > 0 then // wait for data
        begin

          if assigned(parentobj.OnData) then
            Parentobj.OnData(copy(Sbuffer, 1, readlen), StrPas(inet_ntoa(addr_remote.sin_addr)));
        end;
      except
        // deal with probs here..
      end;
    end;
  finally

  end;
end;

function TListenerThread.WaitForReadable(Timeout: Integer): boolean;
var
  fFDSetSelect: TFDSet;
  FTimeout: TTimeval;
  retval: integer;
begin
  fFDSetSelect.fd_count := 1;
  fFDSetSelect.fd_array[0] := ListenSocketHandle;
  if Timeout < 0 then // if -1 wait forever...
  begin
    retval := Select(-1, @fFDSetSelect, nil, nil, nil);
    result := retval = 1;
  end
  else
  begin
    Ftimeout.tv_sec := Timeout;
    Ftimeout.tv_usec := 0; // Timeout;
    retval := Select(-1, @fFDSetSelect, nil, nil, @Ftimeout);
    result := retval = 1; // if returns a 1 then data is avail or disconnect. If 0 then timeout..
  end;
end;

function TListenerThread.Read: integer;
begin
  begin
    if WaitForReadable(-1) then
    begin

      Result := RecvFrom(ListenSockethandle, Sbuffer[1], ReadBufferSize, 0, addr_remote, addrsize);
      if result <= 0 then
        raise EReadFail.Create('Read Failed'); // prob lost connection
    end
    else // timeout..
      raise EReadTimeout.Create('Read TimedOut ' + inttostr(-1) + ' Seconds');
  end;

end;

constructor TGHUDPClient.Create(AOwner: TComponent);
var
  retval: Integer;
begin
  inherited create(AOwner);
  DNSLookupList := TStringList.Create;
  Retval := WSAStartup(MakeWord(1, 1), WData);
  if Retval <> 0 then
    raise ESocketError.Create('Failed to initialize WinSocket,error #' + inttostr(Retval));
end;

destructor TGHUDPClient.Destroy;
begin
  WSACleanup;
  DNSLookupList.free;
  inherited destroy;
end;




function TGHUDPClient.ResolveHost(SHost: string): u_long;
var
  pa: PChar;
  sIn_addr: TInAddr;
  aHost: PHostEnt;
  retval: Integer;
begin
  if Shost <> '' then
  begin
    if Pos('LOCALHOST', uppercase(Shost)) > 0 then
    begin
      sIn_addr.S_un_b.s_b1 := #127;
      sIn_addr.S_un_b.s_b2 := #0;
      sIn_addr.S_un_b.s_b3 := #0;
      sIn_addr.S_un_b.s_b4 := #1;
      Result := sIn_addr.s_addr;
    end
    else
    begin
      result := u_long(INADDR_NONE);
      if Pos(Shost[1], '0123456789') > 0 then // is dot notation already
        Result := WinSock.Inet_Addr(PChar(Shost));

      if Result = u_long(INADDR_NONE) then
        if CacheDNSLookup then
        begin
          Retval := DNSLookupList.IndexOf(SHost);
          if Retval >= 0 then
            Result := U_Long(DNSLookupList.Objects[Retval])
        end;

      if Result = u_long(INADDR_NONE) then
      begin
        aHost := gethostbyname(PChar(sHost));
        if aHost = nil then
        begin
          raise ESocketError.Create('Error in Resolving host ' + Inttostr(WSAGetLastError));
        end
        else
        begin
          pa := aHost^.h_addr_list^;
          sIn_addr.S_un_b.s_b1 := pa[0];
          sIn_addr.S_un_b.s_b2 := pa[1];
          sIn_addr.S_un_b.s_b3 := pa[2];
          sIn_addr.S_un_b.s_b4 := pa[3];
        end;
        Result := sIn_addr.s_addr;
        if CacheDNSLookup then
          DNSLookupList.AddObject(Shost, Pointer(Result));
      end;

    end;
  end
  else
    raise ESocketError.Create('Null Host name ');
end;


procedure TGHUDPClient.SendTo(sData: string; const sHost: string; Port: Integer);
var
  BytesSent: Integer;
  addr_remote: TSockAddrin;
begin
  try
    FHandle := Socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if FHandle = INVALID_SOCKET then
      raise ESocketError.Create('Cant allocate UDP Socket');

    addr_remote.sin_family := PF_INET;
    addr_remote.sin_port := htons(Port);
    addr_remote.sin_addr.S_addr := ResolveHost(sHost);

    BytesSent := Winsock.SendTo(FHandle, sData[1], Length(sData), 0, addr_remote, sizeof(addr_remote));
    if BytesSent = 0 then
    begin
      raise Exception.Create('0 bytes were sent.')
    end
    else
      if BytesSent = SOCKET_ERROR then
      begin
        if WSAGetLastError() = WSAEMSGSIZE then
          raise Exception.Create('Package Size Too Big')
        else
          raise Exception.Create(IntToStr(WSAGetLastError()));
      end;
  finally
    Winsock.CloseSocket(fHandle);
    fHandle := INVALID_SOCKET;
  end;
end;



procedure Register;
begin
  RegisterComponents('Internet', [TGHUDPServer]);
  RegisterComponents('Internet', [TGHUDPClient]);
end;


initialization


end.

