unit GHSock;
{ Simple multitheaded server and Client written by Gordon Hamm
  gordon@yabin.com Uses absol no Windows messages
  360-686-8315
}

interface
uses windows, winsock, SysUtils, Classes, StdCtrls, dialogs;

const
  MaxConnections = 100; // maximum threads at once..
  ReadBufferSize = 5000; // max to read at a time using Recv() or write

type
  EReadFail = class(Exception); // general read fail, disconnect etc.
  EReadTimeout = class(Exception); // general read fail, disconnect etc.
  EWriteFail = class(Exception); // write fail, disconnect etc.
  ESocketError = class(Exception); // Misc Socket Error.



type
  // this Object contains all the socket IO funtions.
  TSocketIO = class
  private
    IOStream: TmemoryStream; // used for multi use..
    Pbuffer: array[0..ReadBufferSize] of byte; // buff for api calls
  public
    SocketHandle: Tsocket;
    ReadTimeout: integer;

    constructor Create;
    destructor Destroy; override;
    { Rread from a socket to a stream. An exception
      will be raised if timed out or read fail / disconnect.
     }
    procedure ReadToStream(TheStream: Tstream; BytesToRead: integer);
    { Read from a socket to a string. An exception
      will be raised if timed out or read fail / disconnect.
     }
    function ReadToString(BytesToRead: integer): string;
    { Rread from a socket to char #10 , to a String. An exception
      will be raised if timed out or read fail / disconnect.
     }
    function ReadlnToString: string;
    { Rread from a socket to char #10 , to a Stream. An exception
      will be raised if timed out or read fail / disconnect.
     }
    procedure ReadLnToStream(TheStream: Tmemorystream);
    { Determine if this socket is readable. Pass a
      timeout parameter for max time to wait in seconds.
     }

    function ReadTerminatorToStream(TheStream: Tmemorystream; TermChar: Char): string;
    {Added by Kevin to allow for reading packets with known termination characters}

    function ReadTerminatorToString(TermCh: Char): string;
    {Added by Kevin to allow for reading packets with known termination characters}


    function WaitForReadable(Timeout: Integer): Boolean;
    // Write to a socket from a stream.
    function WriteFromStream(TheStream: Tmemorystream): Boolean;
    // Write to a socket from a string.
    function WriteFromString(StringTOWrite: string): boolean;
    // Get the address of the remote peer
    function GetRemoteAddress: string;
    // Get the name of the remote peer.
    function GetRemoteName: string;
    // Get the name of the local.
    function GetLocalName: string;
    // Get the address of the local.
    function GetLocalAddress: string;

  end;


type
  TGHServer = class;
  TWorkerThread = class;
  TWorkerCode = procedure(TheThread: TWorkerThread) of object;
  { This Object is where all the Action happens when a client
    successfully connects to the server.
  }
  TWorkerThread = class(TThread)
  private
    //ThreadRefNumber: Integer; // Which element is this thread in Thread array.
    IsThreadCached: Boolean;
    WorkerCode: TWorkerCode; // late bound method that handles actual processing
    parentOBJ: TGHServer;
    procedure execute; override;
  public
    ThreadRefNumber: Integer; // Which element is this thread in Thread array.
    // Object containing all IO functions to read and Write to sockets
    SocketIO: TSocketIO;
    work_CS: TRTLCriticalSection;
    constructor Create(Suspend: Boolean);
    destructor Destroy; override;
  end;

  // record structure used by workerthread list..

  TWorkerThreadsList = record
    InUse: Boolean;
    CacheThisThread: boolean;
    WorkerThread: Tworkerthread;
  end;



  TListenerThread = class(TThread) // thread to watch for incoming requests
  private
    WorkerCode: TWorkerCode; // pass pointer for worker code, will need it later when worker threads are created.
    ListenSocketHandle: TSocket;
    addr: TSockAddr;
    parentOBJ: TGHServer;
    procedure execute; override;
  end;


  { Drag and Drop server component. Simply set the listening port,
    Set active to true, and place some code in the workercode and
    you have an instant Server. Place as many servers as you want on a
    form
   }
  TGHServer = class(Tcomponent)
  private
    Fport: integer; // listen port
    Factive: Boolean; // system started ??
    Fworkercode: TWorkerCode;
    FThreadCacheSize: integer;
    WData: TWSAData;
    ListenSocketHandle: TSocket;
    ListenerThread: TlistenerThread;
    ThreadCount: Integer;
    ThreadCountHigh: Integer;
    procedure SetActive(const bValue: Boolean);
    function Start: boolean; // start listen..
    procedure Stop; // stop listen
  public
    GHSOCK_CS: TRTLCriticalSection;
    work_CS: TRTLCriticalSection;
    WorkerThreadsList: array[1..MaxConnections] of TWorkerThreadsList; // list of sockets/workerthreads
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetThreadCount: integer;
    function GetThreadCountHigh: integer;
  published
    property Active: boolean read Factive write SetActive default False;
    // Port to listen on
    property Port: Integer read Fport write Fport default 6711;
    { This event / code is executed on a connection to the server.
      This is the real guts to the component. Server will take care of
      all the socket clean up and disconnect after this code terminates.
      }
    property Workercode: TWorkerCode read Fworkercode write Fworkercode; // assigned to all worker threads
    { for best perfomance, threads can be cached or reused V.S. creating
      them with each call }
    property ThreadCacheSize: integer read FThreadCacheSize write FThreadCacheSize default 0;
  end;

  // generic winsock client.
  TGHClient = class(Tcomponent)
  private
    Fport: integer;
    Fhost: string;
    WData: TWSAData;
    FConnected: Boolean;
    FCacheDNSLookup: Boolean;
    DNSLookupList: TStringList;
    function ResolveHost(SHost: string): u_long;
  public
    // Object with all IO Methods to read and write to sockets.
    SocketIO: TSocketIO;
    function Connected: Boolean;
    { connect to Host. make sure you clean up by calling
      disconnect when exception etc. occurs.
      }
    procedure connect;
    // Disconnect socket from host.
    procedure Disconnect;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    // Host port to connect to
    property Port: Integer read Fport write Fport;
    // Host adddress to connect to
    property Host: string read Fhost write Fhost;
    property CacheDNSLookup: Boolean read FCacheDNSLookup write FCacheDNSLookup default True;
  end;

function IPAddToDomain(Ip: string): string;

procedure Register;


implementation

constructor TSocketIO.Create;
begin
  inherited create;
  IOstream := TmemoryStream.create; // create a general IO stream.
  ReadTimeout := -1;
end;

destructor TsocketIO.Destroy;
begin
  IOstream.free;
  inherited Destroy;
end;


constructor TWorkerThread.Create(Suspend: Boolean);
begin
  inherited create(Suspend);
  SocketIO := TSocketIO.create;
end;

destructor TWorkerThread.Destroy;
begin
  SocketIO.Free;
  inherited destroy;
end;

procedure TSocketIO.ReadToStream(TheStream: Tstream; BytesToRead: integer);
var
  retval, BytesSoFar, BytesLeftToRead, AmountToRead: integer;

begin
  if TheStream is Tmemorystream then
    Tmemorystream(Thestream).clear;
  BytesSoFar := 0;
  if BytesToRead >= -1 then
    repeat
      begin
        if WaitForReadable(ReadTimeout) then
        begin
          BytesLeftToRead := (BytesToRead - BytesSoFar);

          if BytesLeftToRead < ReadBufferSize then
            AmountToRead := BytesLeftToRead
          else
            AmountToRead := ReadBufferSize;

          if BytesToRead = -1 then // If they passed -1 then give all that is in the buffer now.
            AmountToRead := ReadBufferSize;

          retval := recv(Sockethandle, Pbuffer, AmountToRead, 0);
          if retval > 0 then
          begin
            TheStream.Write(Pbuffer, Retval);
            BytesSoFar := BytesSoFar + retval;
          end
          else
            raise EReadFail.Create('Read Failed'); // prob lost connection

          if BytesToRead = -1 then
            Break;

        end
        else // timeout..
          raise EReadTimeout.Create('Read TimedOut ' + inttostr(ReadTimeout) + ' Seconds');
      end;
    until BytesSoFar >= BytesToRead;
end;



procedure TSocketIO.ReadLnToStream(TheStream: Tmemorystream);
var
  retval: integer;

begin
  Thestream.clear;
  while True do
  begin
    //if WaitForReadable(ReadTimeout) then
    //begin
    retval := recv(Sockethandle, Pbuffer, 1, 0);
    if retval > 0 then
    begin
      TheStream.Write(Pbuffer, Retval);
      if pbuffer[0] = byte(#10) then
        Break;
    end
    else
      raise EReadFail.Create('Readln Failed');
    //end
    //else
    //  raise EReadTimeout.Create('Read TimedOut ' + inttostr(ReadTimeout) + ' Seconds');
  end;
end;



function TSocketIO.ReadToString(BytesToRead: integer): string;
begin
  IOstream.clear;
  try
    ReadToStream(IOStream, BytesToRead);
  finally
    if IOstream.Size > 0 then
    begin
      setlength(result, IOstream.size);
      IOstream.position := 0;
      IOstream.read(result[1], IOstream.size);
    end;
  end;
end;

function TSocketIO.ReadlnToString: string;
begin
  IOstream.clear;
  try
    if WaitForReadable(ReadTimeout) then
      ReadLnToStream(IOStream)
    else
      raise EReadTimeout.Create('Read TimedOut ' + inttostr(ReadTimeout) + ' Seconds');
  finally
    if IOstream.Size > 0 then
    begin
      setlength(result, IOstream.size);
      IOstream.position := 0;
      IOstream.read(result[1], IOstream.size);
    end;
  end;
end;


function TSocketIO.ReadTerminatorToStream(TheStream: Tmemorystream; TermChar: Char): string;
var
  retval: integer;
begin
  Thestream.clear;
  while True do
  begin
    if WaitForReadable(ReadTimeout) then
    begin
      retval := recv(Sockethandle, Pbuffer, 1, 0);
      if retval > 0 then
      begin
        TheStream.Write(Pbuffer, Retval);
        if pbuffer[0] = byte(TermChar) then
          Break;
      end
      else
        raise EReadFail.Create('Read Failed');
    end
    else
      raise EReadTimeout.Create('ReadTerminator TimedOut ' + inttostr(ReadTimeout) + ' Seconds');
  end;
end;

function TSocketIO.ReadTerminatorToString(TermCh: Char): string;
begin
  IOstream.clear;
  try
    ReadTerminatorToStream(IOStream, TermCh);
  finally
    if IOstream.Size > 0 then
    begin
      setlength(result, IOstream.size);
      IOstream.position := 0;
      IOstream.read(result[1], IOstream.size);
    end;
  end;
end;


function TSocketIO.WriteFromStream(TheStream: Tmemorystream): Boolean;
var
  BytesToSend, BufferPos, SuccessBytesSent: integer;
begin
  Thestream.Position := 0;
  while (Thestream.Position < Thestream.size) do
  begin
    BufferPos := 0;
    BytesToSend := Thestream.read(Pbuffer[BufferPos], ReadBufferSize - 1);
    while true do // since there is no guarantee that the entire buffer will be sent, we will loop until the job is done or error...
    begin
      SuccessBytesSent := send(Sockethandle, Pbuffer[bufferpos], BytesToSend, 0);

      if SuccessBytesSent < 0 then // error condition
        raise EWriteFail.create('Error Writing to socket');

      if SuccessBytesSent < BytesToSend then // send broke up our data, send the rest..
      begin
        Bufferpos := BufferPos + SuccessBytesSent; // adjust the pointer
        BytesToSend := BytesToSend - SuccessBytesSent; // adjust the length to send
        Continue;
      end
      else
        Break; // go get the next chunk to send
    end;
  end;
  Result := True;
end;

function TSocketIO.WriteFromString(StringTOWrite: string): boolean;
var
  byteswritten: integer;
begin
  IOstream.clear;
  byteswritten := IOstream.write(StringToWrite[1], Length(stringToWrite));
  result := WriteFromStream(IOStream);
end;



// this funtion looks for readability..

function TSocketIO.WaitForReadable(Timeout: Integer): boolean;
var
  fFDSetSelect: TFDSet;
  FTimeout: TTimeval;
  retval: integer;
begin
  fFDSetSelect.fd_count := 1;
  fFDSetSelect.fd_array[0] := Sockethandle;
  if Timeout < 0 then // if -1 wait forever...
  begin
    retval := Select(0, @fFDSetSelect, nil, nil, nil);
    result := retval = 1;
  end
  else
  begin
    Ftimeout.tv_sec := Timeout;
    Ftimeout.tv_usec := 0; // Timeout;
    retval := Select(0, @fFDSetSelect, nil, nil, @Ftimeout);
    result := retval = 1; // if returns a 1 then data is avail or disconnect. If 0 then timeout..
  end;
end;

function TSocketIO.GetRemoteAddress: string;
var
  SockAddrIn: TSockAddrIn;
  Size: Integer;
begin
  Result := '';
  Size := SizeOf(SockAddrIn);
  if getpeername(Sockethandle, SockAddrIn, Size) = 0 then
    Result := inet_ntoa(SockAddrIn.sin_addr);
end;

function TSocketIO.GetRemoteName: string;
var
  SockAddrIn: TSockAddrIn;
  Size: Integer;
  HostEnt: PHostEnt;
begin
  Result := '';
  Size := SizeOf(SockAddrIn);
  if getpeername(Sockethandle, SockAddrIn, Size) = 0 then
    HostEnt := gethostbyaddr(@SockAddrIn.sin_addr.s_addr, 4, PF_INET);
  if HostEnt <> nil then Result := HostEnt.h_name;
end;

function TSocketIO.GetLocalName: string;
var
  s1: Shortstring;
begin
  GetHostName(@s1[1], SizeOf(s1));
  Result := StrPas(@s1[1]);
end;

function TSocketIO.GetLocalAddress: string;
var
  sHost: string;
  pa: PChar;
  sa: TInAddr;
  aHost: PHostEnt;
begin
  sHost := GetLocalName + #0;
  aHost := gethostbyname(@sHost[1]);
  if aHost = nil then begin
    raise Exception.Create('Could not get local address.');
  end
  else
  begin
    pa := aHost^.h_addr_list^;
    sa.S_un_b.s_b1 := pa[0];
    sa.S_un_b.s_b2 := pa[1];
    sa.S_un_b.s_b3 := pa[2];
    sa.S_un_b.s_b4 := pa[3];
  end;
  pa := inet_ntoa(sa);
  if pa = nil then
    raise Exception.Create('Inet_NToA Error.');
  Result := StrPas(pa);
end;



procedure TWorkerThread.execute;
begin
  FreeOnTerminate := True;
  while not terminated do
  begin
    try
      try
        workercode(self); // call the actual processing code here.
      except
        //on e : exception do
        //  showmessage(e.message);
        //Deal with any thead socket exceptions here..
        // right now, well ignore..
        //EReadTimeout
        //EReadFail
      end;

    finally
      shutdown(SocketIO.Sockethandle, 2);
      Closesocket(SocketIO.Sockethandle);
      EnterCriticalSection(parentOBJ.GHSOCK_CS);
      try
        if parentobj.WorkerThreadsList[ThreadRefNumber].inuse then
          parentobj.WorkerThreadsList[ThreadRefNumber].inuse := False;
        Dec(parentobj.ThreadCount);
      finally
        LeaveCriticalSection(parentOBJ.GHSOCK_CS);
      end;
    end;

    if IsThreadCached then // sleep if its a cached thread ..
      Suspend
    else
      Break; // end thread..
    // bottom of loop ..
  end;
end;

constructor TGHServer.Create(AOwner: TComponent);
begin
  inherited create(Aowner);
  InitializeCriticalSection(GHSOCK_CS); // initialize my Critical section.
  InitializeCriticalSection(Work_CS); // initialize my Critical section.
  if fport = 0 then // default listening port..
    fport := 6711;

  factive := factive;
end;


destructor TGHServer.destroy;
begin
  inherited destroy;
end;

procedure TGHServer.SetActive(const bValue: Boolean);
begin
  if (csDesigning in ComponentState) then
  begin
    factive := bvalue;
    exit;
  end;
  if bvalue and not factive then
    start
  else
    if not bvalue and factive then
      Stop;
end;

function TGHServer.Start;
var
  Ierror, x: Integer;
  addr: TSockAddr;
begin
  result := False;
  if (not Factive) then // if its already active, dont start again
  begin

    /// initialize worker Thread array
    for X := 1 to Maxconnections do
    begin
      WorkerThreadsList[x].inuse := False;
      WorkerThreadsList[x].CacheThisThread := False;
      // If these threads are from the first up to the TheadCachesize,
      // mark them, they will stay alive once created..
      if x <= FThreadCacheSize then
        WorkerThreadsList[x].CacheThisThread := True;
      WorkerThreadsList[x].WorkerThread := nil;
    end;



    Ierror := WSAStartup(MakeWord(1, 1), WData);
    if Ierror <> 0 then
    begin
      raise ESocketError.Create('Failed to initialize WinSocket,error #' + inttostr(Ierror));
      //showmessage('Failed to initialize WinSocket,error #' + inttostr(Ierror));
      Exit;
    end;
    try
      // open socket here.
      ListenSocketHandle := socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
      if ListenSocketHandle = INVALID_SOCKET then
      begin
        raise ESocketError.Create('Failed to create a socket, Error #' + Inttostr(WSAGetLastError));
        //Showmessage('Failed to create a socket, Error #' + Inttostr(WSAGetLastError));
        exit;
      end;

      Addr.sin_family := AF_INET;
      Addr.sin_port := htons(Fport);
      Addr.sin_addr.s_addr := INADDR_ANY;
      // bind the socket..
      if bind(ListenSocketHandle, Addr, SizeOf(Addr)) = SOCKET_ERROR then
      begin
        raise ESocketError.Create('Failed to bind the socket, error #' + Inttostr(WSAGetLastError) + '.'#13#10#13#10 +
          'Probable reason is that another daemon is already running on the same port (' + Inttostr(fport) + ').');

        //showmessage('Failed to bind the socket, error #' + Inttostr(WSAGetLastError) + '.'#13#10#13#10 +
        //  'Probable reason is that another daemon is already running on the same port (' + Inttostr(fport) + ').');
        Ierror := WSACleanup;
        exit
      end;
      // listen, cache up to 5 hits..
      winsock.listen(ListenSocketHandle, 5); // use listen from winsock.pas

      // if we got here, life is good. Start the listening thread..
      ThreadCount := 0;
      ThreadCountHigh := 0;
      ListenerThread := TlistenerThread.create(True);
      ListenerThread.ParentObj := Self;
      ListenerThread.ListenSocketHandle := ListenSocketHandle;
      ListenerThread.addr := addr;
      ListenerThread.WorkerCode := WorkerCode;
      ListenerThread.resume;

      Factive := True;
      Result := True;
    finally

    end;
  end;

end;

procedure TGHServer.stop;
var
  Ierror, X: integer;
begin
  if (Factive) then
  begin
    ListenerThread.terminate;
    shutdown(ListenerThread.ListenSocketHandle, 2);
    CloseSocket(ListenerThread.ListenSocketHandle);
    EnterCriticalSection(GHSOCK_CS);
    try
      /// Free any outstanding threads, they will free on thread terminate..
      for X := 1 to Maxconnections do
      begin
        if WorkerThreadsList[x].inuse then
        begin
          WorkerThreadsList[x].inuse := False;
          WorkerThreadsList[x].WorkerThread.terminate;
          if WorkerThreadsList[x].CacheThisThread then
            WorkerThreadsList[x].workerthread.resume;
          shutdown(WorkerThreadsList[x].WorkerThread.SocketIO.SocketHandle, 2);
          Closesocket(WorkerThreadsList[x].WorkerThread.SocketIO.SocketHandle);
        end
        else // if its not inuse and cached, clear that thread...
          if (WorkerThreadsList[x].workerthread <> nil) and (WorkerThreadsList[x].CacheThisThread) then
          begin
            WorkerThreadsList[x].WorkerThread.terminate;
            WorkerThreadsList[x].workerthread.resume;
          end;
      end;
    finally
      LeaveCriticalSection(GHSOCK_CS);
    end;
    sleep(1000);
    Ierror := WSACleanup;
    waitforsingleobject(ListenerThread.handle, 5000);
    ListenerThread.Free;
    Factive := False;
  end;
end;

// these two functions gather thread counts/stats

function TGHServer.GetThreadCount: integer;
begin
  EnterCriticalSection(GHSOCK_CS);
  try
    result := Threadcount;
  finally
    LeaveCriticalSection(GHSOCK_CS);
  end;
end;

function TGHServer.GetThreadCountHigh: integer;
begin
  EnterCriticalSection(GHSOCK_CS);
  try
    result := Threadcounthigh;
  finally
    LeaveCriticalSection(GHSOCK_CS);
  end;
end;


// this code sits in a loop and looks
// for incoming requests in its own thread..

procedure TListenerThread.execute;
var
  addrsize, x: integer;
  TempNewSocketHandle : Tsocket;
  WOrkerthread: TworkerThread; // needs to be an array..
  FoundResFlag: Boolean;
  Temp_int: Integer;
  Temp_str: string;
begin
  //Freeonterminate := True; // we will free this thread ourself
  while not terminated do
  begin
    try
      addrsize := SizeOf(Addr);
      TempNewSocketHandle := accept(ListenSocketHandle, @Addr, @addrsize);
      if TempNewSocketHandle = INVALID_SOCKET then
        continue;

      EnterCriticalSection(parentOBJ.GHSOCK_CS);
      try
        /// look for a free resource... up to  Maxconnections..
        /// store the thread in a free array member..
        FoundResFlag := False;
        for X := 1 to Maxconnections do
        begin
          if not parentobj.WorkerThreadsList[x].inuse then
          begin
            {if the thread has never been created or its not a cache thread
             then create / initialize it.. it doesnt exist yet
            }
            with parentobj do
            begin
              if (WorkerThreadsList[x].WorkerThread = nil) or (not WorkerThreadsList[x].CacheThisThread) then
              begin
                WorkerThreadsList[x].WorkerThread := TWorkerThread.create(True);
                WorkerThreadsList[x].WorkerThread.IsThreadCached := WorkerThreadsList[x].CacheThisThread;
                WorkerThreadsList[x].WorkerThread.parentOBJ := parentOBJ;
                WorkerThreadsList[x].WorkerThread.work_cs := parentOBJ.work_cs;
                WorkerThreadsList[x].WorkerThread.ThreadRefNumber := x;
                WorkerThreadsList[x].WorkerThread.workercode := Workercode;
              end;
              WorkerThreadsList[x].WorkerThread.SocketIO.SocketHandle := TempNewSocketHandle;
              WorkerThreadsList[x].inuse := True;
              Inc(ThreadCount);
              if ThreadCount > ThreadCountHigh then
                ThreadCountHigh := ThreadCount;
              WorkerThreadsList[x].WorkerThread.resume;
              FoundResFlag := True;
              break;
            end;
          end;
        end;
      finally
        LeaveCriticalSection(parentobj.GHSOCK_CS);
      end;
      // if no resource, then refuse connection..
      if not FoundResFlag then
      begin
        Temp_str := 'Connection Refused';
        Temp_int := length(temp_str);
        send(TempNewSocketHandle, Temp_str[1], Temp_int, 0);
        shutdown(TempNewSocketHandle, 2);
        Closesocket(TempNewSocketHandle);
      end;
    except
      // deal with probs here..
    end;
  end;
end;


constructor TGHClient.Create(AOwner: TComponent);
var
  retval: Integer;
begin
  inherited create(AOwner);
  DNSLookupList := TStringList.Create;
  SocketIO := TSocketIO.create;
  port := 6711;
  Retval := WSAStartup(MakeWord(1, 1), WData);
  if Retval <> 0 then
    raise ESocketError.Create('Failed to initialize WinSocket,error #' + inttostr(Retval));
end;

destructor TGHClient.Destroy;
begin
  try
    if fconnected then
      disconnect;
  finally
    DNSLookupList.free;
    SocketIO.free;
  end;
  inherited destroy;
end;

procedure TGHclient.connect;
var
  Sockaddr: TSockAddr;
  Retval: Integer;
begin
  if Fconnected then
    disconnect;
  begin
    SocketIO.SocketHandle := Socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    Sockaddr.sin_family := PF_INET;
    Sockaddr.sin_port := htons(fport);
    Sockaddr.sin_addr.S_addr := ResolveHost(FHost);
    try
      Retval := Winsock.Connect(SocketIO.SocketHandle, Sockaddr, SizeOf(Sockaddr));
      if retval < 0 then
        raise ESocketError.Create('Error in Connecting to host ' + Inttostr(WSAGetLastError));
      Fconnected := True;
    except
      on E: Exception do
      begin
        Disconnect;
        raise;
      end;
    end;
  end;
end;


procedure TGHclient.disconnect;
begin
  Fconnected := False;
  Winsock.Shutdown(SocketIO.SocketHandle, 1);  // need to be here ??
  Winsock.CloseSocket(SocketIO.SocketHandle);
end;



function TGHClient.ResolveHost(SHost: string): u_long;
var
  pa: PChar;
  sIn_addr: TInAddr;
  aHost: PHostEnt;
  retval: Integer;
begin
  if Shost <> '' then
  begin
    if Pos('LOCALHOST', uppercase(Shost)) > 0 then
    begin
      sIn_addr.S_un_b.s_b1 := #127;
      sIn_addr.S_un_b.s_b2 := #0;
      sIn_addr.S_un_b.s_b3 := #0;
      sIn_addr.S_un_b.s_b4 := #1;
      Result := sIn_addr.s_addr;
    end
    else
    begin
      result := u_long(INADDR_NONE);
      if (Pos(Shost[1], '0123456789') > 0) then // is dot notation already
        Result := WinSock.Inet_Addr(PChar(Shost));

      if Result = u_long(INADDR_NONE) then
        if CacheDNSLookup then
        begin
          Retval := DNSLookupList.IndexOf(SHost);
          if Retval >= 0 then
            Result := U_Long(DNSLookupList.Objects[Retval])
        end;

      if Result = u_long(INADDR_NONE) then
      begin
        aHost := gethostbyname(PChar(sHost));
        if aHost = nil then
        begin
          raise ESocketError.Create('Error in Resolving host ' + Inttostr(WSAGetLastError));
        end
        else
        begin
          pa := aHost^.h_addr_list^;
          sIn_addr.S_un_b.s_b1 := pa[0];
          sIn_addr.S_un_b.s_b2 := pa[1];
          sIn_addr.S_un_b.s_b3 := pa[2];
          sIn_addr.S_un_b.s_b4 := pa[3];
        end;
        Result := sIn_addr.s_addr;
        if CacheDNSLookup then
          DNSLookupList.AddObject(Shost, Pointer(Result));
      end;

    end;
  end
  else
    raise ESocketError.Create('Null Host name ');
end;



function TGHClient.Connected: boolean;
begin
  Result := Fconnected;
end;

function IPAddToDomain(Ip: string): string;
var
  HostEnt: PHostEnt;
  addr: Integer;
begin
  Result := '';
  Addr := Inet_Addr(PChar(IP));
  HostEnt := GetHostByAddr(@addr, 4, PF_INET);
  if HostEnt <> nil then
    Result := HostEnt.h_name;
end;



procedure Register;
begin
  RegisterComponents('Internet', [TGHServer]);
  RegisterComponents('Internet', [TGHClient]);
end;


initialization


end.

