unit TransCanvas;
{
	TTransCanvas By Paul van Dinther Copyright Diprode 24-01-2000
	e-mail: paul@diprode.com
	Website: http://www.diprode.com

	Having strugled for a while with several methods to display a semi transparent
	area, I thought it would be usefull to encapsulate the whole thing into a
	component. TransCanvas is similar to TPaintBox and would make a great control
	to inherit from to create other semi transparent controls. TTransCanvas
	controls can quite happily be stacked on top of each other with each level
	clearly visible.

	Just set the Transparency type to ttAlpha and set the transparency percentage
	(0 to 100) and presto. The Graphic controls (be aware that windowed controls
	such as buttons are always on top of Graphic controls) behind TTransCanvas
	show through!.

	Transparency types are:

	ttnone		Is like having a transparent canvas to start with.
	ttKey			Key color transparency. to be used with TransKeyColor
	ttAlpha		Full range of transparency from 0 percent to 100
	ttRed			Red Screening. More red means more transparent.
	ttGreen   Green Screening. More red means more transparent.
	ttBlue		Blue Screening. More red means more transparent.

	Note: The last 3 types are slower to render because additional calculations
	are performed for each pixel. Still pretty fast though.

	Use ScreenBiasPercent to improve the Bluescreening effect.The result is often
	a better blue screen effect because it reduces transperency even more in
	colors that are less that 100% blue. (Try it!)
}

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	extctrls, math;
type
	TRGB = record
	R,G,B: Word;
end;

type
	TCalcEvent = procedure (Sender: TObject; ForeColor,BackColor:TRGB; var MergeColor: TRGB; X,Y: Integer) of object;
	TPaintEvent =  procedure (Sender: TObject; Canvas: TCanvas) of object;
	TCanvasType = (ctTransparent, ctLumFilter);
	TTransFade = (tfNone,tfLeft,tfRight,tfUp,tfDown,tfLeftDown,tfRightDown,tfLeftUp,tfRightUp,tfCenter,tfPeak, tfHorizon, tfVertical, tfButton, tfRoundButton);
	TTransType = (ttNone,ttKey,ttAlpha,ttRed,ttGreen,ttBlue);
	TCustomTransCanvas = class(TGraphicControl)
	private
		FCanvasType: TCanvasType;
		FTransBiasPercent: Integer;
		FTransBias: Double;
		FScreenBias: Double;
		FScreenBiasPercent: Integer;
		FTransFade: TTransFade;
		FOnCalc: TCalcEvent;
		FUseCalcEvent: Boolean;
		FonPaint: TPaintEvent;
		FTransMinCutoff: Integer;
		FTransMaxCutoff: Integer;
		FInverse: Boolean;
		FTransType: TTransType;
		FTransPercent: Integer;
		FTransKeyColor: TColor;
		FBackground :TBitmap;
		FTransBand: Integer;
		procedure CanvasToBitmap;
		procedure SetCanvasType(Value : TCanvasType);
		procedure SetScreenBiasPercent(Value: Integer);
		procedure SetTransBiasPercent(Value: Integer);
		function bias(PValue,PBias: Double):Double;
		procedure SetTransFade(Value: TTransFade);
		procedure SetTransBand(Value: Integer);
		procedure SetTransMinCutoff(Value: Integer);
		procedure SetTransMaxCutoff(Value: Integer);
		procedure SetInverse(Value: Boolean);
		procedure SetTransType(Value: TTransType);
		procedure SetTransPercent(Value: Integer);
		procedure SetTransKeyColor(Value: TColor);
		procedure PaintTransArea;
	protected
		procedure paint; override;
		procedure DoPaint(PCanvas: TCanvas); virtual;
		function CalculateTransFade(PX,PY: Integer; PTransPercent: Integer): Integer;
		property CanvasType: TCanvasType read FCanvasType write SetCanvasType;
		property TransBiasPercent: Integer read FTransBiasPercent write SetTransBiasPercent;
		property ScreenBiasPercent: Integer read FScreenBiasPercent write SetScreenBiasPercent;
		property TransFade: TTransFade read FTransFade write SetTransFade;
		property TransBand: Integer read FTransBand write SetTransBand;
		property UseCalcEvent: Boolean read FUseCalcEvent write FUseCalcEvent;
		property OnCalc: TCalcEvent read FOnCalc write FOnCalc;
		property TransType: TTransType read FTransType write SetTransType;
		property TransPercent: Integer read FTransPercent write SetTransPercent;
		property TransMinCutoff: Integer read FTransMinCutoff write SetTransMinCutoff;
		property TransMaxCutoff: Integer read FTransMaxCutoff write SetTransMaxCutoff;
		property TransKeyColor: TColor read FTransKeyColor write SetTransKeyColor;
		property Inverse: Boolean read FInverse write SetInverse;
		property OnPaint: TPaintEvent read FOnPaint write FOnPaint;
	public
		procedure Refresh;
		constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
	end;

	{TTransCanvas}
	TTransCanvas = class(TCustomTransCanvas)
	published
		//New Properties
		property CanvasType;
		property UseCalcEvent;
		property OnCalc;
		property TransFade;
		property TransType;
		property TransPercent;
		property TransMinCutoff;
		property TransMaxCutoff;
		property TransKeyColor;
		property ScreenBiasPercent;
		property TransBiasPercent;
		property Inverse;
		property OnPaint;
		//Standard Properties
		property Align;
		property Color;
		property DragCursor;
		property DragMode;
		property Enabled;
		property Font;
		property ParentColor;
		property ParentFont;
		property ParentShowHint;
		property PopupMenu;
		property ShowHint;
		property Visible;
		property OnClick;
		property OnDblClick;
		property OnDragDrop;
		property OnDragOver;
		property OnEndDrag;
		property OnMouseDown;
		property OnMouseMove;
		property OnMouseUp;
		property OnStartDrag;
	end;

procedure Register;

implementation

procedure Register;
begin
	RegisterComponents('Diprode', [TTransCanvas]);
end;

constructor TCustomTransCanvas.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
	//Setting of the default values
	FTransType := ttNone;
	FBackground := TBitmap.Create;
	FBackground.PixelFormat := pf24Bit;
	FTransPercent := 50;
	FCanvasType := ctTransparent;
	FTransMaxCutoff := 100;
	Width := 50;
	Height := 50;
end;

destructor TCustomTransCanvas.Destroy;
begin
	FBackGround.Free;
	inherited Destroy;
end;

procedure TCustomTransCanvas.CanvasToBitmap;
var
	LPoint: Tpoint;
	HDC: Integer;
	function Min(PValue1,PValue2:Integer): Integer;
	begin
		if PValue1 <= PValue2 then Result := PValue1
		else result := PValue2;
	end;
	function Max(PValue1,PValue2:Integer): Integer;
	begin
		if PValue1 > PValue2 then Result := PValue1
		else result := PValue2;
	end;
begin
	if FBackground.Width <> width then FBackground.Width := Width;
	if FBackground.Height <> Height then FBackground.Height := Height;
	{
		Translate the Top-Left of the control to screen coordinates
		Grab the screen device, take a snapshot and copy the picture accross to FBackground
	}
	LPoint := ClientToScreen(point(Left,Top));
	HDC := GetDC(0);
	BitBlt(FBackground.Canvas.Handle,0,0,min(TPanel(parent).Width - Left,Width),min(TPanel(Parent).Height - Top,Height),HDC,LPoint.X - Left,LPoint.Y - Top, SRCCOPY);
	ReleaseDc(0,HDC);
end;

{
	This procedure calculates the resulting bitmap pixel by pixel using the
	foreground and backgound bitmaps. The calculation method depends on the TransType
	property value selected. An onCalc event exposes the calculation to the
	user and let's the user apply it's own merge calculation.
}
procedure TCustomTransCanvas.PaintTransArea;
var
	LWidth,LHeight: Integer;
	FForeground: TBitmap;
	FCombined: TBitmap;
	LLumPercent: Integer;
	LFCol,LBCol,LMCol: TRGB;
	LTransPercent: Integer;
	x,y : Integer;
	LForeScan: PByteArray;
	LBackScan: PByteArray;
	LCombinedScan: PByteArray;
	function CalcPartLum(PValue1,PValue2,Part: Integer): Integer;
	var
		LLum: Integer;
	begin
		if PValue1 = 0 then begin
			LLum := Part - 50;
			if LLum = 0 then Result := PValue2;
			if LLum > 0 then Result := trunc(PValue2 + ((256 - PValue2) * 0.02 * LLum));
			if LLum < 0 then Result := trunc(PValue2 + (PValue2 * 0.02 * LLum));
		end else Result := PValue2;
	end;
	function CalcPartValue(PValue1,PValue2,Part: Integer): Integer;
	begin
		Result := ((PValue1 * (100 - Part)) + (PValue2 * Part)) div 100;
	end;
	function Min(PValue1,PValue2:Integer): Integer;
	begin
		if PValue1 <= PValue2 then Result := PValue1
		else result := PValue2;
	end;
	function Max(PValue1,PValue2:Integer): Integer;
	begin
		if PValue1 > PValue2 then Result := PValue1
		else result := PValue2;
	end;
begin
	//Crate and Adjust bitmaps
	FForeGround := TBitmap.Create;
	FForeGround.PixelFormat := pf24Bit;
	FCombined := TBitmap.Create;
	FCombined.PixelFormat := pf24Bit;
	FForeGround.Width := Width;
	FForeGround.Height := Height;
	FCombined.Width := Width;
	FCombined.Height := Height;
	DoPaint(FForeground.Canvas);
	LHeight := min(FBackground.Height,TPanel(parent).Height - Top);
	LWidth := Min(FBackground.Width, TPanel(parent).Width  - Left);
	if FCanvasType <> ctTransparent then begin
		{
		This procedure modifies the luminosity value of the background pixel in  those
		locations were the foreground pixel is painted. The amount of Luminosity change
		is defined by the FilterFadeType, MinLum and MaxLum properties. Luminosity is
		defined as a value from 0 to 255. Background luminosity is seen as a value of 0
		and the range from that luminosity value to MinLum and MaxLum is always + and - 100
		}
		for y := 0 to LHeight - 1 do	begin
			LForeScan := FForeground.ScanLine[y];
			LBackScan := FBackground.ScanLine[y];
			LCombinedScan := FCombined.ScanLine[y];
			X := 0;
			while X < LWidth * 3 do begin
				LLumPercent := CalculateTransFade(X div 3,Y,FTransPercent);
				if FInverse then LLumPercent := 100 - LLumPercent;
				LCombinedScan[x] := CalcPartLum(LForeScan[X]+LForeScan[X+1]+LForeScan[X+2],LBackScan[X],LLumPercent);
				LCombinedScan[x+1] := CalcPartLum(LForeScan[X]+LForeScan[X+1]+LForeScan[X+2],LBackScan[X+1],LLumPercent);
				LCombinedScan[x+2] := CalcPartLum(LForeScan[X]+LForeScan[X+1]+LForeScan[X+2],LBackScan[X+2],LLumPercent);
				inc(X,3);
			end;
		end
	end	else begin
		LTransPercent := FTranspercent;
		//these two nested For loops using Y and X provide a step through for each pixel
		for y := 0 to LHeight - 1 do
		begin
			LForeScan := FForeground.ScanLine[y];
			LBackScan := FBackground.ScanLine[y];
			LCombinedScan := FCombined.ScanLine[y];
			X := 0;
			//Width * 3 because the internal bitmaps are always 24 Bit (3 Bytes per pixel)
			while X < LWidth * 3 do begin
				if (assigned(FOnCalc)) and FUseCalcEvent then begin
					//Collect the foreground color for this pixel
					LFCol.R := LForeScan[X + 2];
					LFCol.G := LForeScan[X + 1];
					LFCol.B := LForeScan[X];
					//collect the background color for this pixel
					LBCol.R := LBackScan[X + 2];
					LBCol.G := LBackScan[X + 1];
					LBCol.B := LBackScan[X];
					//Call the event handler
					FOnCalc(self,LFCol,LBCol,LMCol,X,Y);
					//Assign the merged result to the scanline pixel of the destination
					LCombinedScan[X+2] := LMCol.R;
					LCombinedScan[X+1] := LMCol.G;
					LCombinedScan[X] := LMCol.B;
				end else begin
					Case FTransType of
						ttNone:
							LTransPercent := FTransPercent;
						ttKey:
							begin
							if FForeGround.Canvas.Pixels[x div 3,y] = FTransKeyColor then LTransPercent := 100
							else LTransPercent := FTransPercent;
							end;
						ttAlpha:
							LTransPercent := FTransPercent;
						ttBlue:
							if LForeScan[X] + LForeScan[X+1] + LForeScan[X+2] > 0 then begin
								LTransPercent := 100 * LForeScan[X] div (LForeScan[X] + LForeScan[X+1] + LForeScan[X+2]);
							end else LTransPercent := 0;
						ttGreen:
							if LForeScan[X] + LForeScan[X+1] + LForeScan[X+2] > 0 then begin
							LTransPercent := 100 * LForeScan[X + 1] div (LForeScan[X] + LForeScan[X+1] + LForeScan[X+2]);
							end else LTransPercent := 0;
						ttRed:
							if LForeScan[X] + LForeScan[X+1] + LForeScan[X+2] > 0 then begin
							LTransPercent := 100 * LForeScan[X + 2] div (LForeScan[X] + LForeScan[X+1] + LForeScan[X+2]);
							end else LTransPercent := 0;
						end;
						if FTransType in [ttred,ttgreen,ttblue] then begin
							{
								Changes the relation between transparency and screen color from
								linear to exponential. This results in better clips
							}
							LTransPercent := round(bias((LTransPercent / 100),FScreenBias) * 100);
							{
								Apply minimum and max cutoff filters
								These filters will push transparency values below minimum cutoff down
								to zero and boost transparency values above maximum cutoff to Transpercent
							}
							if LTransPercent < FTransMinCutoff  then LTransPercent := 0;
							if LTransPercent > FTransMaxCutoff then LTransPercent := FTransPercent;
							//ensure that remainder of the picture is shown at transparency value
							if LTransPercent < FTranspercent then LTransPercent := FTranspercent;
						end;

						LTransPercent := CalculateTransFade(X div 3,Y,LTransPercent);
						{
							Inverts the transparency values. Transparent area's become opaque and
							vice versa.
						}
						if FInverse then LTransPercent := 100 - LTransPercent;
						//Once the transparency is calculated the actual merge calculation is done
						//Merging Blue
						LCombinedScan[x] := CalcPartValue(LForeScan[X],LBackScan[X],LTransPercent);
						//Merging Green
						LCombinedScan[x+1] := CalcPartValue(LForeScan[X+1],LBackScan[X+1],LTransPercent);
						//Merging Red
						LCombinedScan[x+2] := CalcPartValue(LForeScan[X+2],LBackScan[X+2],LTransPercent);
					end;
				//jump to the next pixel by skipping 3 bytes (remember 24 bit/3 bytes per pixel?)
				inc(X,3);
			end;
		end;
	end;
	Canvas.Draw(0,0,FCombined);
	FForeground.Free;
	FCombined.Free;
end;

function TCustomTransCanvas.CalculateTransFade(PX,PY: Integer; PTransPercent: Integer): Integer;
var
	FactorX,FactorY: double;
	LHalf: Integer;
begin
	case FTransFade of
		tfNone:
			begin
			Result := PTransPercent;
			exit;
			end;
		tfLeft:
			begin
			Result := 100 - round( bias((PX / Width),FTransBias) * (100 - PTransPercent));
			exit;
			end;
		tfRight:
			begin
			Result := 100 - round( bias((1 - PX / Width),FTransBias) * (100 - PTransPercent));
			exit;
			end;
		tfUp:
			begin
			Result := 100 - round(bias((PY / Height),FTransBias) * (100 - PTransPercent));
			exit;
			end;
		tfDown:
			begin
			Result := 100 - round(bias((1 - PY / Height),FTransBias) * (100 - PTransPercent));
			exit;
			end;
		tfLeftUp:
			begin
			Result := 100 - round( bias((PX / Width) * (PY / Height),FTransBias) * (100 - PTransPercent));			if Result > 100 then Result := 100;
			end;
		tfLeftDown:
			begin
			Result := 100 - round( bias((PX / Width) * (1 - (PY / Height)),FTransBias) * (100 - PTransPercent));			if Result > 100 then Result := 100;
			end;
		tfRightDown:
			begin
			Result := 100 - round( bias((1 - (PX / Width)) * (1 - (PY / Height)),FTransBias) * (100 - PTransPercent));			if Result > 100 then Result := 100;
			end;
		tfRightUp:
			begin
			Result := 100 - round( bias((1 - (PX / Width)) * (PY / Height),FTransBias) * (100 - PTransPercent));			if Result > 100 then Result := 100;
			end;
		tfCenter:
			begin
				LHalf := Height div 2;
				if LHalf > 0 then begin
					FactorX := round(Sqrt(Sqr(LHalf - PY) + Sqr((PX - (Width div 2)) * (Height / Width))));
					if FactorX > LHalf then FactorX := LHalf;
					Result := 100 - round(bias((1 - FactorX / LHalf),FTransBias) * (100 - PTranspercent));
				end;
			end;
		tfPeak:
			begin
				FactorY := PY / (Height div 2);
				if FactorY > 1 then FactorY := 2 - FactorY;
				FactorX := PX / (Width div 2);
				if FactorX > 1 then FactorX := 2 - FactorX;
				Result := 100 - round((100 - PTransPercent) * bias((FactorX * FactorY),FTransBias));
			end;
		tfHorizon:
			begin
				FactorY := PY / (Height div 2);
				if FactorY > 1 then FactorY := 2 - FactorY;
				Result := 100 - round((100 - PTransPercent) * bias(FactorY,FTransBias));
			end;
		tfVertical:
			begin
				FactorX := PX / (Width div 2);
				if FactorX > 1 then FactorX := 2 - FactorX;
				Result := 100 - round((100 - PTransPercent) * bias(FactorX,FTransBias));
			end;
		tfButton:
			begin
				if (PX >= PY) and (PY < Height div 2) and ((PX - Width)+PY < 0) then
					Result := 80  //Top
				else if (PX + (PY - Height) >= 0) and (PY > Height - (Height div 2)) and ((PX - Width) < (PY - Height)) then
					Result := 30 //Bottom
				else if (PX <= PY) and (PX < Width div 2) and (PX + (PY - Height) < 0) then
					Result := 60 //Left
				else if ((PX - Width)+PY >= 0)  and (PX >= Width - (Width div 2)) and ((PX - Width) >= (PY - Height)) then
					Result := 40  //Right
				else Result := 0;
				exit;
			end;
		tfRoundbutton:
			begin
				if (PX >= PY) and (PY < canvas.pen.width) and ((PX - Width)+PY < 0) then
					Result := 50 + round(40 * (1 - (PY / canvas.pen.width)))  //Top
				else if (PX + (PY - Height) >= 0) and (PY >= Height - canvas.pen.width) and ((PX - Width) < (PY - Height)) then
					Result := round(50 - round(30 * (1 - ((Height - PY) / canvas.pen.width))))
				else if (PX <= PY) and (PX < canvas.pen.width) and (PX + (PY - Height) < 0) then
					Result := 50 + round(20 * (1 - (PX / canvas.pen.width)))  //Left
				else if ((PX - Width)+PY >= 0)  and (PX >= Width - canvas.pen.width) and ((PX - Width) >= (PY - Height)) then
					Result := 50 - round(20 * (1 - ((Width - PX) / canvas.pen.width)))  //Right
				else Result := 0;
				exit;
			end;
	end;
end;

procedure TCustomTransCanvas.paint;
begin
	//Produce the resulting output bitmap by merging foreground and background
	CanvasToBitmap;
	PaintTransArea;
	//Paint a dashed line around the control to show the boundaries when not painted
	if csDesigning in ComponentState then
	with Canvas do begin
		Pen.Style := psDash;
		Brush.Style := bsClear;
		Rectangle(0, 0, Width, Height);
	end;
end;

procedure TCustomTransCanvas.DoPaint(PCanvas: TCanvas);
begin
	{
		this is a little trick (be it limiting) to support painting on a transparent
		canvas. Only the painted stuff after this will be rendered semitransparent.
	}
	if FTransType = ttNone then PCanvas.Draw(0,0,FBackGround);
	if assigned(FonPaint) then begin
		FOnPaint(self,PCanvas);
	end;
end;

procedure TCustomTransCanvas.SetTransType(Value: TTransType);
begin
	if Value = FTransType then exit;
	FTransType := Value;
	invalidate;
end;

procedure TCustomTransCanvas.SetTransPercent(Value: Integer);
begin
	if Value = FTransPercent then exit;
	if (Value < 0) or (value > 100) then exit;
	FTransPercent := Value;
	if FTransType in [ttAlpha,ttRed,ttGreen,ttBlue,ttNone] then begin
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetTransMinCutoff(Value: Integer);
begin
	if Value = FTransMinCutoff then exit;
	if (Value < 0) or (value > 100) then exit;
	FTransMinCutoff := Value;
	if FTransType in [ttAlpha,ttRed,ttGreen,ttBlue] then begin
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetTransMaxCutoff(Value: Integer);
begin
	if Value = FTransMaxCutoff then exit;
	if (Value < 0) or (value > 100) then exit;
	FTransMaxCutoff := Value;
	if FTransType in [ttAlpha,ttRed,ttGreen,ttBlue] then begin
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetTransKeyColor(Value: TColor);
begin
	if Value = FTransKeyColor then exit;
	FTransKeyColor := Value;
	if FTransType = ttKey then begin
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetInverse(Value: Boolean);
begin
	if Value = FInverse then exit;
	FInverse := Value;
	invalidate;
end;

procedure TCustomTransCanvas.Refresh;
begin
	PaintTransArea;
end;

procedure TCustomTransCanvas.SetTransBand(Value: Integer);
begin
	if Value <> FTRansBand then begin
		FTransBand := Value;
		Invalidate;
	end;
end;

procedure TCustomTransCanvas.SetTransFade(Value: TTransFade);
begin
	if Value <> FTransFade then begin
		FTransFade := Value;
		Invalidate;
	end;
end;

{
	This function Biases PValue. assuming PValue in the range of 0 to 1
	Bias can be positive or negative ranging from 1 through 0 to -1
	A Bias of 0 will not modify PValue.	Result will range between 0 and 1
}
function TCustomTransCanvas.bias(PValue,PBias: Double):Double;
begin
	//Invalid values means not bias calculation
	if (PBias <= 1) and (PBias >= -1) and (PValue >=0) and (PValue <= 1) then begin
		// a Bias of 0 is a linear relationship. Let's save some time here
		if PBias = 0 then begin
			Result := PValue;
			exit;
		end;
		//PBias ranges from 1 through 0 to -1. Actual bias should be between 0 and 1
		if PBias >= 0 then begin
			//Positive bias
			Result := Power(PValue,1 - PBias);
		end else begin
			//mirrored positive bias
			Result := 1 - power(1 - PValue,1 + PBias);
		end;
	end else begin
		Result := PValue;
	end;
end;

procedure TCustomTransCanvas.SetTransBiasPercent(Value: Integer);
begin
	if Value <> FTransBiasPercent then begin
		FTransBiasPercent := Value;
		FTransBias := FTransBiasPercent / 100;
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetScreenBiasPercent(Value: Integer);
begin
	if Value <> FScreenBiasPercent then begin
		FScreenBiasPercent := Value;
		FScreenBias := FScreenBiasPercent / 100;
		PaintTransArea;
	end;
end;

procedure TCustomTransCanvas.SetCanvasType(Value: TCanvasType);
begin
	if Value <> FCanvasType then begin
		FCanvasType := Value;
		Invalidate;
	end;
end;

end.
