{ $Id: TranslatorEditor.pas,v 1.12 2002/01/08 14:11:34 mvj Exp $ }

{
    This file is part of the TTranslator 

    TTranslator is a Delphi component for localizing String and TStrings 
    properties of components dropped on a form. You can also localize your 
    code strings with TTranslator.
    Copyright (C) 2002 Polycon Ab

    This is a licensed version of TTranslator, it may be used as described
    in the TTranslator license agreement. If you have not acquired a 
    commercial TTranslator license, your are using this product illegaly.    
}

unit TranslatorEditor;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Translator, Storages, Criteria, DataElements, StdCtrls, ExtCtrls, DataType,
  Menus, ComCtrls,
  DataEditor,
{$ifndef DBUTEST}
  DataEditorCell, PolyGrid;
{$else}
  DBUDataEditorCell, DBUGrid;
{$endif DBUTEST}

type
  TdlgStringsEditor = class(TForm)
    Menu: TMainMenu;
    MenuLanguages: TMenuItem;
    MenuLanguageAdd: TMenuItem;
    menuLanguageRename: TMenuItem;
    MenuLanguageRemove: TMenuItem;
    MenuLanguagesSeparator: TMenuItem;
    MenuClose: TMenuItem;
    PageControl: TPageControl;
    tsComponentProperties: TTabSheet;
    tsCodeStrings: TTabSheet;
    pnlTop: TPanel;
    lblShowProperties: TLabel;
    cboShow: TComboBox;
    bvlTop: TBevel;
    pnlRight: TPanel;
    cmdAddString: TButton;
    cmdDeleteString: TButton;
    pnlBottom: TPanel;
    tsClassesProperties: TTabSheet;
    plCPRight: TPanel;
    spltrCRLeft: TSplitter;
    spltrCRRight: TSplitter;
    TreeViewComponents: TTreeView;
    ListBoxProps: TListBox;
    pnlCPButtons: TPanel;
    ButtonAdd: TButton;
    ButtonRemove: TButton;
    pnlCodeStrings: TPanel;
    menuEdit: TMenuItem;
    menuEditCut: TMenuItem;
    menuEditCopy: TMenuItem;
    menuEditPaste: TMenuItem;
    cmdDuplicateString: TButton;
    PopupMenuCodeStrings: TPopupMenu;
    menuPopupCut: TMenuItem;
    menuPopupCopy: TMenuItem;
    menuPopupPaste: TMenuItem;
    N1: TMenuItem;
    menuPopupAddString: TMenuItem;
    menuPopupDuplicateString: TMenuItem;
    menuPopupDeleteString: TMenuItem;
    procedure FormShow(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure cboShowChange(Sender: TObject);
    procedure MenuLanguageAddClick(Sender: TObject);
    procedure menuLanguageRenameClick(Sender: TObject);
    procedure MenuLanguageRemoveClick(Sender: TObject);
    procedure cmdAddStringClick(Sender: TObject);
    procedure cmdDeleteStringClick(Sender: TObject);
    procedure MenuCloseClick(Sender: TObject);
    procedure MenuLanguagesClick(Sender: TObject);
    procedure RefreshListBoxProps;
    procedure BuildTreeNodeComponents( ATreeView : TTreeView );
    procedure DeleteTreeNodeItems( TreeView : TTreeView );
    procedure ButtonAddClick(Sender: TObject);
    procedure ButtonRemoveClick(Sender: TObject);
    procedure TreeViewComponentsChange(Sender: TObject; Node: TTreeNode);
    procedure PageControlChange(Sender: TObject);
    procedure menuEditCutClick(Sender: TObject);
    procedure menuEditCopyClick(Sender: TObject);
    procedure menuEditPasteClick(Sender: TObject);
    procedure menuEditAddStringClick(Sender: TObject);
    procedure cmdDuplicateStringClick(Sender: TObject);
    procedure menuEditDuplicateStringClick(Sender: TObject);
    procedure menuEditDeleteStringClick(Sender: TObject);
  private
    FFirstPageChange : Boolean;
    FStrings : IEditableTranslatedStrings;
    FComponentPropertyEditor,
    FClassesPropertyEditor,
    FCodeStringEditor : TDataEditor;
    procedure CreateEditors;
    procedure DestroyEditors;
{$ifndef DBUTEST}
    procedure ComponentEditorCellChangeValue(Sender: TObject; Cell : TDataEditorCell);
{$else DBUTEST}
    procedure ComponentEditorCellChangeValue(Sender: TObject; Cell : TDataInplaceEditorCell);
{$endif DBUTEST}
    function ActiveEditor : TDataEditor;

    procedure ReadPosition;
    procedure StorePosition;
  public
    property Strings : IEditableTranslatedStrings read FStrings write FStrings;
  end;

var
  dlgStringsEditor: TdlgStringsEditor;

implementation

uses DataEditorLib, GridEditor, LanguageEditor, TypInfo,
     DataClipboard, DataTypes, Registry;

var
  FClipBoard : TDataClipboard;

{$R *.DFM}

function TdlgStringsEditor.ActiveEditor : TDataEditor;
begin
  if PageControl.ActivePage = tsComponentProperties then
    Result := FComponentPropertyEditor
  else if PageControl.ActivePage = tsClassesProperties then
    Result := FClassesPropertyEditor
  else
    Result := FCodeStringEditor;
end;

procedure TdlgStringsEditor.CreateEditors;

  procedure LangFieldWidths(ADataEditor : TDataEditor {AGridEditor : TGridEditor});
  var
    iField, FirstIndex : Integer;
    TotalFieldWidth : Integer;
  begin
    with ADataEditor.ActiveGridEditor, ADataEditor.ActiveGridEditor.Grid do
    begin
      FirstIndex := GridIndexOfField[FStrings.LanguageFields[0]];
      TotalFieldWidth := ADataEditor.Parent.Width - 36; // AGridEditor.Grid.Width - 28;
      for iField := 0 to ColCount - 1 do
        if (iField < FirstIndex) or (iField >= FirstIndex + FStrings.LanguageCount) then
          Dec(TotalFieldWidth, ColWidths[iField]);

      TotalFieldWidth := TotalFieldWidth div FStrings.LanguageCount;

      for iField := 0 to FStrings.LanguageCount - 1 do
        ColWidths[iField + FirstIndex] := TotalFieldWidth;
    end;
  end;

begin
  FStrings.Translations.ArrangeRows;

  FComponentPropertyEditor := TDataEditor.CreateWithStandardView(tsComponentProperties, FStrings.Translations, False,
                                                                FStrings.EditorStandardView, nil, nil);
  FComponentPropertyEditor.ShowKeys := False;
  FComponentPropertyEditor.SetFocusOnExec := False;
  FComponentPropertyEditor.ShowAllBooleansAsCheckBoxes := True;
  FComponentPropertyEditor.ShowMarkRowCalcField := True;
  FComponentPropertyEditor.Execute;

  LangFieldWidths(FComponentPropertyEditor {.ActiveGridEditor});
  FComponentPropertyEditor.OnChangeValue := Self.ComponentEditorCellChangeValue;
  FComponentPropertyEditor.UpdateType := utRow;

  FCodeStringEditor := TDataEditor.CreateDefault( pnlCodeStrings, FStrings.StringTranslations, False, nil, nil);
  FCodeStringEditor.ShowKeys := True;
  FCodeStringEditor.SetFocusOnExec := False;
  FCodeStringEditor.ShowAllBooleansAsCheckBoxes := True;
  FCodeStringEditor.ShowMarkRowCalcField := True;
  FCodeStringEditor.Execute;
  FCodeStringEditor.AllowMultipleAutoCreatedRows := True;
  FCodeStringEditor.AutoCreateRows := True;
  FCodeStringEditor.AutoDeleteRows := True;

  LangFieldWidths(FCodeStringEditor {.ActiveGridEditor});

  if ActiveEditor.ActivegridEditor.Grid.Showing then
    ActiveEditor.ActivegridEditor.Grid.SetFocus;

  FClassesPropertyEditor := TDataEditor.CreateDefault( plCPRight, FStrings.AddedProperties, False, nil, nil);
  FClassesPropertyEditor.Definition.DefaultStandardView.SetReadOnly( [FieldClass, FieldProperty], True );
  FClassesPropertyEditor.ShowKeys := True;
  FClassesPropertyEditor.SetFocusOnExec := False;
  FClassesPropertyEditor.ShowAllBooleansAsCheckBoxes := True;
  FClassesPropertyEditor.ShowMarkRowCalcField := True;
  FClassesPropertyEditor.Execute;
end;

procedure TdlgStringsEditor.DestroyEditors;
begin
  try
    FComponentPropertyEditor.Free;
    FCodeStringEditor.Free;
    FClassesPropertyEditor.Free;
  finally
    FComponentPropertyEditor := nil;
    FCodeStringEditor := nil;
    FClassesPropertyEditor := nil;
  end;
end;

procedure TdlgStringsEditor.FormShow(Sender: TObject);
begin
  FFirstPageChange := True;
  PageControl.ActivePage := tsComponentProperties;
  FFirstPageChange := False;
  PageControlChange(Sender); // Force event...

  cboShow.ItemIndex := Integer(FStrings.ShowProperties);

  CreateEditors;

  cmdDeleteString.Enabled := FStrings.StringTranslations.RowCount > 0;
  BuildTreeNodeComponents(TreeViewComponents);
  TreeViewComponents.FullExpand;

  ReadPosition;
end;

procedure TdlgStringsEditor.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  if Self.WindowState = wsNormal then
    StorePosition;

  DestroyEditors;
{$ifdef D4_OR_HIGHER}
  FStrings.FillStrings;
{$else}
  FStrings.FillStrings(-1);
{$endif D4_OR_HIGHER}
end;

procedure TdlgStringsEditor.StorePosition;
var
  reg : TRegistry;
begin
  reg := TRegistry.Create;
  try
    reg.RootKey := HKEY_CURRENT_USER;

    if reg.OpenKey('SOFTWARE\Polycon\Translator', True) then
    begin
      reg.WriteInteger( 'EditorTop', Self.Top );
      reg.WriteInteger( 'EditorHeight', Self.Height );
      reg.WriteInteger( 'EditorLeft', Self.Left );
      reg.WriteInteger( 'EditorWidth', Self.Width );
    end;
  finally
    reg.Free;
  end;
end;

procedure TdlgStringsEditor.ReadPosition;
var
  reg : TRegistry;
begin
  reg := TRegistry.Create;
  try
    reg.RootKey := HKEY_CURRENT_USER;

    if reg.OpenKeyReadOnly('SOFTWARE\Polycon\Translator') then
    begin
      if reg.ValueExists( 'EditorTop' ) then
        Self.Top := reg.ReadInteger( 'EditorTop' );
      if reg.ValueExists( 'EditorHeight' ) then
        Self.Height := reg.ReadInteger( 'EditorHeight' );
      if reg.ValueExists( 'EditorLeft' ) then
        Self.Left := reg.ReadInteger( 'EditorLeft' );
      if reg.ValueExists( 'EditorWidth' ) then
        Self.Width := reg.ReadInteger( 'EditorWidth' );
    end;
  finally
    reg.Free;
  end;
end;


{$ifndef DBUTEST}
procedure TdlgStringsEditor.ComponentEditorCellChangeValue(Sender: TObject; Cell : TDataEditorCell);
{$else DBUTEST}
procedure TdlgStringsEditor.ComponentEditorCellChangeValue(Sender: TObject; Cell : TDataInplaceEditorCell);
{$endif DBUTEST}
begin
  if (Cell.DataRow <> nil) and
     (FStrings.LanguageIndexByField(Cell.DataField) <> ANYLANGUAGE) then
    Cell.DataRow[FieldDoTranslate] := TrueValue;
end;

procedure TdlgStringsEditor.cboShowChange(Sender: TObject);
begin
  FComponentPropertyEditor.ActiveGridEditor.Disable;
  if cboShow.ItemIndex = 0 then
    FStrings.ShowProperties := spAllAdded
  else
    FStrings.ShowProperties := spTranslatedOnly;
  FComponentPropertyEditor.ActiveGridEditor.Enable;
end;

procedure TdlgStringsEditor.MenuLanguageAddClick(Sender: TObject);
var
  LangEditor : TdlgLangEditor;
begin
  Application.CreateForm(TdlgLangEditor, LangEditor);
  DestroyEditors;

  try
    LangEditor.AddLanguage(FStrings);
  finally
    LangEditor.Release;
    CreateEditors;
  end;
end;

procedure TdlgStringsEditor.menuLanguageRenameClick(Sender: TObject);
var
  LangEditor : TdlgLangEditor;
begin
  Application.CreateForm(TdlgLangEditor, LangEditor);

  try
    LangEditor.RenameLanguage(FStrings);
  finally
    LangEditor.Release;

    // Repaint headers 
    FComponentPropertyEditor.ActiveGridEditor.Grid.RedrawRow(0);
    FComponentPropertyEditor.ActiveGridEditor.Grid.RedrawRow(0);
  end;
end;

procedure TdlgStringsEditor.MenuLanguageRemoveClick(Sender: TObject);
var
  LangEditor : TdlgLangEditor;
begin
  Application.CreateForm(TdlgLangEditor, LangEditor);
  DestroyEditors;

  try
    LangEditor.RemoveLanguage(FStrings);
  finally
    LangEditor.Release;
    CreateEditors;
  end;
end;

procedure TdlgStringsEditor.cmdAddStringClick(Sender: TObject);
begin
  FCodeStringEditor.ActiveGridEditor.NewRow;
  cmdDeleteString.Enabled := True;
  FCodeStringEditor.SetFocus;
end;

procedure TdlgStringsEditor.cmdDuplicateStringClick(Sender: TObject);
begin
  FCodeStringEditor.ActiveGridEditor.DuplicateRow;
  cmdDeleteString.Enabled := FCodeStringEditor.ActiveGridEditor.CanDeleteRows;
  FCodeStringEditor.SetFocus;
end;

procedure TdlgStringsEditor.cmdDeleteStringClick(Sender: TObject);
begin
  if FCodeStringEditor.ActiveGridEditor.CanDeleteRows then
    FCodeStringEditor.ActiveGridEditor.DeleteRows;

  cmdDeleteString.Enabled := FCodeStringEditor.ActiveGridEditor.CanDeleteRows;
end;

procedure TdlgStringsEditor.MenuCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TdlgStringsEditor.MenuLanguagesClick(Sender: TObject);
begin
  menuLanguageRename.Enabled := (FStrings.LanguageCount > 1);
end;

procedure TdlgStringsEditor.RefreshListBoxProps;

  function GetClass : TClass;
  begin
    if TreeViewComponents.Selected = nil then
      Result := nil
    else
      Result := TClass(TTreeNode(TreeViewComponents.Selected).Data);
  end;

  function SelfSelected(AClass : String; AProperty : String) : Boolean;
  var
    ARow : TDataRow;
  begin
    ARow := TDataRow(FStrings.AddedProperties.LocateRow([AClass, AProperty]));
    if (ARow <> nil) then
      result := True
    else
      result := False;
  end;

  procedure AddProperties(SelectedClass : String; ANode : TTreeNode; AList : TStringList);
  var
    AClass : TClass;
    APropList : TStringList;
    i : Integer;
  begin
    AClass := TClass(ANode.Data);

    if AClass = nil then
      Exit;

    APropList := FStrings.CreateClassPropertyList(AClass);
    for i := 0 to APropList.Count - 1 do
      if not SelfSelected(SelectedClass, APropList[i]) then
        AList.Add(APropList[i]);

    APropList.Free;

    for i := 0 to ANode.Count - 1 do
      AddProperties(SelectedClass, ANode.Item[i], AList);
  end;

var
  i : Integer;
  AClassName : String;
  ANode : TTreeNode;
  APropertyList : TStringList;
begin
  ListBoxProps.Items.Clear;

  ANode := TreeViewComponents.Selected;
  if (ANode = nil) or (ANode.Data = nil) then
    Exit;

  AClassName := TClass(ANode.Data).ClassName;
  APropertyList := TStringList.Create;
  APropertyList.Sorted := True;
  APropertyList.Duplicates := dupIgnore;

  AddProperties(AClassName, ANode, APropertyList);
  ListBoxProps.Items.Assign(APropertyList);
  APropertyList.Free;

  for i := 0 to ListBoxProps.Items.Count - 1 do
    ListBoxProps.Selected[i] := True;
end;

procedure TdlgStringsEditor.BuildTreeNodeComponents( ATreeView : TTreeView );
var
  i : Integer;
  AComponent : TComponent;
  AClassType : TClass;
  ABaseTreeNode, ATreeNode : TTreeNode;
  FNodes : TList;
  AList : TList;

  function CreateParentNode( AClass : TClass ) : TTreeNode;
  var
    AParentClass : TClass;
    tmpTreeNode : TTreeNode;
    i : Integer;
  begin
    tmpTreeNode := nil;
    AParentClass := AClass.ClassParent;
    for i := 0 to ATreeView.Items.Count - 1 do
    begin
      tmpTreeNode := ATreeView.Items.Item[i];
      if tmpTreeNode.Data = AParentClass then
        break;
      tmpTreeNode := nil;
    end;
    if tmpTreeNode = nil then
    begin
      if AParentClass.ClassParent <> TComponent then  // Stop at TComponent!
        tmpTreeNode := CreateParentNode(AParentClass)
      else
        tmpTreeNode := ABaseTreeNode;
      result := ATreeView.Items.AddChildObject(tmpTreeNode, AParentClass.ClassName, AParentClass);
    end
    else
      result := tmpTreeNode;
  end;

begin
  DeleteTreeNodeItems(ATreeView);
  ABaseTreeNode := ATreeView.Items.AddObject(nil, TComponent.ClassName, TComponent); // Base item is TComponent
  FNodes := TList.Create;
  AList := FStrings.GetComponents;  
  for i := 0 to AList.Count - 1 do
  begin
    AComponent := TComponent(AList.Items[i]);
    if not (AComponent is TTranslator) then
    begin
      AClassType := AComponent.ClassType;
      if FNodes.IndexOf(AClassType) = -1 then
      begin
        FNodes.Add(AClassType);
        ATreeNode := CreateParentNode(AClassType);
        ATreeView.Items.AddChildObject(ATreeNode, AComponent.ClassName, AClassType);
      end;
    end;
  end;
end;

procedure TdlgStringsEditor.DeleteTreeNodeItems( TreeView : TTreeView );
var
  TreeNode : TTreeNode;
  i : Integer;
begin
  for i := TreeView.Items.Count - 1 downto 0 do
  begin
    TreeNode := TreeView.Items[i];
    if TreeNode.Data <> nil then
      TreeNode.Free
  end;
  TreeView.Items.Clear;
end;

procedure TdlgStringsEditor.ButtonAddClick(Sender: TObject);
var
  i : integer;
  AClass : TClass;
  NewRow : TDataRow;
begin
  if TreeViewComponents.Selected <> nil then
    AClass := TClass(TTreeNode(TreeViewComponents.Selected).Data)
  else
    AClass := nil;

  if (AClass = nil) or
     (ListBoxProps.Items.Count = 0) or
     (ListBoxProps.SelCount = 0) then
    Exit;

  FClassesPropertyEditor.ActiveGridEditor.Disable;

  for i := 0 to ListBoxProps.Items.Count - 1 do
  begin
    if ListBoxProps.Selected[i] then
    begin
      NewRow := TDataRow.Create(AddedPropertiesTable);
      NewRow.StringValue[FieldClass] := AClass.ClassName;
      NewRow.StringValue[FieldProperty] := ListBoxProps.Items[i];
      NewRow.BooleanValue[FieldDoTranslate] := True;
      NewRow.BooleanValue[FieldSubClasses] := True;
      FStrings.AddedProperties.PutRow(NewRow, paDontOverwriteKeys);
    end;
  end;
  RefreshListBoxProps;

  FClassesPropertyEditor.ActiveGridEditor.RowStorage.ArrangeRows;
  FClassesPropertyEditor.ActiveGridEditor.Enable;
end;

procedure TdlgStringsEditor.ButtonRemoveClick(Sender: TObject);
begin
  FClassesPropertyEditor.ActiveGridEditor.DeleteRows;

  RefreshListBoxProps;
end;

procedure TdlgStringsEditor.TreeViewComponentsChange(Sender: TObject;
  Node: TTreeNode);
begin
  RefreshListBoxProps;
end;

procedure TdlgStringsEditor.PageControlChange(Sender: TObject);
begin
  if (not FFirstPageChange) and
     (PageControl.ActivePage = tsComponentProperties) then
  begin
    if FComponentPropertyEditor <> nil then
      FComponentPropertyEditor.ActiveGridEditor.Disable;

{$ifdef D4_OR_HIGHER}
    FStrings.FillStrings;
{$else}
    FStrings.FillStrings(-1);
{$endif D4_OR_HIGHER}
    FStrings.DisableUnused;

    if FComponentPropertyEditor <> nil then
      FComponentPropertyEditor.ActiveGridEditor.Enable;
  end;
end;

procedure TdlgStringsEditor.menuEditCutClick(Sender: TObject);
begin
  ActiveEditor.ActiveGridEditor.Cut(FClipBoard);
end;

procedure TdlgStringsEditor.menuEditCopyClick(Sender: TObject);
begin
  ActiveEditor.ActiveGridEditor.Copy(FClipBoard);
end;

procedure TdlgStringsEditor.menuEditPasteClick(Sender: TObject);
begin
  ActiveEditor.ActiveGridEditor.Paste(FClipBoard);
end;

procedure TdlgStringsEditor.menuEditAddStringClick(Sender: TObject);
begin
  cmdAddStringClick(Sender);
end;

procedure TdlgStringsEditor.menuEditDuplicateStringClick(Sender: TObject);
begin
  cmdDuplicateStringClick(Sender);
end;

procedure TdlgStringsEditor.menuEditDeleteStringClick(Sender: TObject);
begin
  cmdDeleteStringClick(Sender);
end;

initialization

  FClipBoard := TDataClipboard.Create( True, False );

finalization

  FClipBoard.Free;

end.
