{ $Id: Translator.pas,v 1.37 2002/08/01 09:33:13 laa Exp $ }

{
    This file is part of the TTranslator 

    TTranslator is a Delphi component for localizing String and TStrings 
    properties of components dropped on a form. You can also localize your 
    code strings with TTranslator.
    Copyright (C) 2002 Polycon Ab

    This is a licensed version of TTranslator, it may be used as described
    in the TTranslator license agreement. If you have not acquired a 
    commercial TTranslator license, your are using this product illegaly.    
}

unit Translator;

interface
{$i common.inc}

uses
  RowList, Classes, Dialogs, Forms, SysUtils, Menus, stdctrls, TypInfo, DataType,
  Storages, Criteria, DataElements, StandardView, TranslatorInterfaces, CalcField;

const
  ANYLANGUAGE : Integer = -1;

type
  TShowProperties = (spAllAdded, spTranslatedOnly);
  TLanguage = String;
  TAboutTranslator = class(TObject);

  TTranslator = class;
  TOnTranslateEvent = procedure(Sender : TTranslator; OldLanguage, NewLanguage : String) of object;

  IEditableTranslatedStrings = interface
    function GetLanguageCount : Integer;
    function GetLanguages(idx : Integer) : String;
    procedure SetLanguages(idx : Integer; LangName : String);
    function GetLanguageFields(idx : Integer) : TDataField;
    function GetComponents : TList;
    function CreateSubClassList(AClassName : string) : TStrings;
    function CreateClassPropertyList(AClass : TClass) : TStringList;
    procedure GetTranslatedProperties(Component : TComponent; List : TDataRowList);

    property LanguageCount : Integer read GetLanguageCount;
    property Languages[idx : Integer] : String read GetLanguages write SetLanguages;
    property LanguageFields[idx : Integer] : TDataField read GetLanguageFields;
    function IndexOfLanguage(LangName : String) : Integer;
    function LanguageIndexByField(AField : TDataField) : Integer;

    function GetEditorStandardView : TSingletonStandardView;
    property EditorStandardView : TSingletonStandardView read GetEditorStandardView;

    function GetTranslations : TRowStorage;
    property Translations : TRowStorage read GetTranslations;

    function GetStringTranslations : TRowStorage;
    property StringTranslations : TRowStorage read GetStringTranslations;

    function GetAddedProperties : TRowStorage;
    property AddedProperties : TRowStorage read GetAddedProperties;

    function GetShowProperties : TShowProperties;
    procedure SetShowProperties(Show : TShowProperties);
    property ShowProperties : TShowProperties read GetShowProperties write SetShowProperties;

    procedure InsertLanguage(Index : Integer; LanguageName : String; CopyFromLanguage : Integer);
    procedure RemoveLanguage(Index : Integer);

{$ifdef D4_OR_HIGHER}
    procedure FillStrings(ALangIndex : Integer = -1 {ANYLANGUAGE});
{$else}
    procedure FillStrings(ALangIndex : Integer {ANYLANGUAGE});
{$endif D4_OR_HIGHER}

    procedure UpdateUI;
    procedure UpdateLanguage;

    procedure DisableUnused;
  end;

  TTranslatedStrings = class(TObject, IEditableTranslatedStrings)
  private
    FTranslationTable : TDataTable;
    FStringTranslationTable : TDataTable;
    FStandardViewLangEditor : TSingletonStandardView;

    FOwner : TTranslator;
    FTranslations : TRowStorage;
    FAddedProperties : TRowStorage;
    FStringTranslations : TRowStorage;
    FLanguages : TStringList;
    FShowProperties : TShowProperties;

{$ifdef D4_OR_HIGHER}
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
{$else}
    function QueryInterface(const IID: TGUID; out Obj): Integer; stdcall;
{$endif D4_OR_HIGHER}
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;

    function GetComponents : TList;
    function CreateSubClassList(AClassName : string) : TStrings;
    function CreateClassPropertyList(AClass : TClass) : TStringList;

    function GetTranslations : TRowStorage;
    function GetStringTranslations : TRowStorage;
    function GetAddedProperties : TRowStorage;

    procedure CreateDataTables;
    procedure DestroyDataTables;

    function GetShowProperties : TShowProperties;
    procedure SetShowProperties(Show : TShowProperties);
    function GetLanguageCount : Integer;
    function GetLanguages(idx : Integer) : String;
    procedure SetLanguages(idx : Integer; LangName : String);
    function GetLanguageFields(idx : Integer) : TDataField;

    function GetTranslationItemByName(Component : TComponent; PropertyName : String) : TDataRow;

{$ifdef D4_OR_HIGHER}
    function HasProperty(Component : TComponent; PropertyName : String; var PropInfo : PPropInfo) : Boolean; overload;
    function HasProperty(Component : TComponent; PropertyName : String) : Boolean; overload;
{$else}
    function HasProperty(Component : TComponent; PropertyName : String; var PropInfo : PPropInfo) : Boolean;
    function HasPropertyOL(Component : TComponent; PropertyName : String) : Boolean;
{$endif D4_OR_HIGHER}
    function SetProperty(Component : TComponent; PropertyName, Value : String) : Boolean;
    function GetProperty(Component : TComponent; PropertyName : String; var Value : String) : Boolean;

    function HasStringsProperty(Component : TComponent; PropertyName : String; var Strings : TStrings) : Boolean;

    function GetEditorStandardView : TSingletonStandardView;

    constructor Create(Owner : TTranslator);

    function FormHasComponent(Form : TForm; ComponentName : String; var Component : TComponent) : Boolean;
    property AddedProperties : TRowStorage read FAddedProperties;

    property TranslationItemByName[Component : TComponent; PropertyName : String] : TDataRow read GetTranslationItemByName;

    function PropertyTranslated(ClassType : TClass; AProperty : String) : Boolean;

    property LanguageFields[idx : Integer] : TDataField read GetLanguageFields;
    function LanguageIndexByField(AField : TDataField) : Integer;

{$ifdef D4_OR_HIGHER}
    procedure FillStrings(ALangIndex : Integer = -1 {ANYLANGUAGE});
    function GetString(const AProperty : String; LangIndex : Integer) : String; overload;
    function GetString(const AProperty : String; LangIndex : Integer; Variables : array of String) : String; overload;
{$else}
    procedure FillStrings(ALangIndex : Integer {ANYLANGUAGE});
    function GetString(const AProperty : String; LangIndex : Integer) : String;
    function GetStringOL(const AProperty : String; LangIndex : Integer; Variables : array of String) : String;
{$endif D4_OR_HIGHER}
    procedure DisableUnused;

    procedure AddStringsTranslation(Component : TComponent; PropertyName : String; Strings : TStrings; ALangIndex : Integer; DoTranslate : Boolean);
    procedure AddTranslation(Component : TComponent; PropertyName : String; ALangIndex : Integer; DoTranslate : Boolean);
    function GetTranslation(Component : TComponent; PropertyName : String; ALangIndex : Integer; var Value : String) : Boolean;
    procedure GetTranslatedProperties(Component : TComponent; List : TDataRowList);

    procedure UpdateUI;
    procedure UpdateLanguage;

    procedure CheckLanguageNames;
  public
    procedure InsertLanguage(Index : Integer; LanguageName : String; CopyFromLanguage : Integer);
    procedure RemoveLanguage(Index : Integer);

    property Owner : TTranslator read FOwner;
    property ShowProperties : TShowProperties read GetShowProperties write SetShowProperties;

    property LanguageCount : Integer read GetLanguageCount;
    property Languages[idx : Integer] : String read GetLanguages write SetLanguages;
    function IndexOfLanguage(LangName : String) : Integer;

    destructor Destroy; override;
  end;

  TLangSortField = class(TCalcField)
  private
    FLangField : TDataField;
    constructor CreateOld(LangField : TDataField);
  public
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    destructor Destroy; override;
  end;

  TLangNameField = class(TDataField)
  private
    constructor CreateOld;
  public
    destructor Destroy; override;
  end;

  TTranslator = class(TComponent, IStringTranslator)
  private
    fIgnoreLanguageNameCheck : Boolean;
    FStrings : TTranslatedStrings;
    FLangIndex : Integer;
    FTranslateApplication : Boolean;

    FBeforeTranslate : TOnTranslateEvent;
    FAfterTranslated : TOnTranslateEvent;

    function GetAbout : TAboutTranslator;
    function GetStrings : TTranslatedStrings;
    function GetForm : TForm;

    procedure LoadProperty(Reader : TReader);
    procedure StoreProperty(Writer : TWriter);
    procedure UpdateCurrentLanguageFromUI;
    procedure UpdateUIFromCurrentLanguage;
    function GetActiveLanguage : String;
    procedure TranslateTo(LangName : String);
    procedure TranslateRecursive(LangName : String; Recursive : Boolean);
    function MainTranslator : TTranslator;

  protected
    procedure TranslateStrings(ALangIndex : Integer);
    procedure DefineProperties(Filer: TFiler); override;
    procedure Loaded; override;
  public
    class function TranslatorVersion : String;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

{$ifdef D4_OR_HIGHER}
    function GetString(const AProperty : String) : String; overload;
    function GetString(const AProperty : String; Variables : array of String) : String; overload;
{$else}
    function GetString(const AProperty : String) : String;
    function GetStringOL(const AProperty : String; Variables : array of String) : String;
{$endif D4_OR_HIGHER}
  published
    property About : TAboutTranslator read GetAbout;
    property Strings : TTranslatedStrings read GetStrings;
    property Language : TLanguage read GetActiveLanguage write TranslateTo;
    property TranslateApplication : Boolean read FTranslateApplication write FTranslateApplication;

    property BeforeTranslate : TOnTranslateEvent read FBeforeTranslate write FBeforeTranslate;
    property AfterTranslated : TOnTranslateEvent read FAfterTranslated write FAfterTranslated;
  end;

var
  FieldClass : TKeyField;
  FieldProperty : TKeyField;

  FieldSubClasses : TDataField;
  FieldDoTranslate : TDataField;

  FieldDoTranslateText : TDataField;
  FieldSubClassesText : TDataField;

  AddedPropertiesTable : TDataTable;


  function PerformVersionNotificationAtInstall : Boolean;

implementation

uses Controls, DataTypes, CommonCalcFields, Registry, Windows;

const
  CURRENTVERSION : String = '2.10';
  STREAMINGVERSION : String = '2.00';

var
  FieldComponent : TKeyField;
  FieldComponentPointer : TDataField;
  TranslatorList : TList;



type
  TTranslatedProperties = class
  private
    fTranslatedStrings : TTranslatedStrings;
    fList : TStringList;
  public
    constructor Create(TranslatedStrings : TTranslatedStrings);
    destructor Destroy; override;
    function GetList(Component : TComponent) : TDataRowList;
  end;

function PerformVersionNotificationAtInstall : Boolean;
var
  reg : TRegistry;
begin
  Result := True;
  reg := TRegistry.Create;
  try
    reg.RootKey := HKEY_CURRENT_USER;

    if reg.OpenKey('SOFTWARE\Polycon\Translator', True) then
    begin
      if not reg.ValueExists( 'VersionNotification' ) then
      begin

        if MessageDlg('Translator ' + CURRENTVERSION + #13#10#13#10 +
                   'You have now started to use the Translator v' +CURRENTVERSION+ '. ' +
                   'Please note that this software is licensed under the GNU General Public License. ' +
                   'You may use, modify and distribute the product as long as the entire source code ' +
                   'of your project is freely available as stated in the GPL license. For more ' +
                   'information about the licensing, please refer to ' + #13#10 +
                   'http://www.polycon.fi/translator/licensing.html' + #13#10#13#10 +
                   'By clicking "OK" now, you agree to all terms of the license.', mtConfirmation, [mbOK, mbCancel], 0) = mrOK then
        begin
          reg.WriteInteger( 'VersionNotification', 2 );
        end
        else
          result := False;

      end;
    end;
  finally
    reg.Free;
  end;
end;

function ClassRefIsClass(AClass, ParentClass : TClass) : Boolean;
begin
  Result := (AClass <> nil) and
            ((AClass = ParentClass) or ClassRefIsClass(AClass.ClassParent, ParentClass));
end;

function IsStringsProperty(AProperty : String; var RealPropertyName : String) : Boolean;
var
  p : Integer;
begin
  p := Pos('[', AProperty);
  Result := (p >= 1);
  if Result then
    RealPropertyName := Copy(AProperty, 1, p-1)
  else
    RealPropertyName := AProperty;
end;

function SubStrPos(FirstSearchPos : Integer; SubStr, S : String) : Integer;
begin
  if FirstSearchPos < 1 then
    FirstSearchPos := 1;

  S := Copy(S, FirstSearchPos, Length(S) - FirstSearchPos + 1);
  Result := Pos(SubStr, S);
  if Result > 0 then
    Result := Result + FirstSearchPos - 1;
end;

{ TLangSortField }

constructor TLangSortField.CreateOld(LangField : TDataField);
begin
  inherited CreateOld('', LangField.DataType, True, True);
  FLangField := LangField;
end;

function TLangSortField.CalcValue(ARow : TAbstractRow) : TValue;
var
  sVal : String;
  p : Integer;
begin
  sVal := AnsiLowerCase(ARow.StringValue[FLangField]);
  while True do
  begin
    p := pos('&', sVal);
    if p < 1 then
      Break;

    sVal := Copy(sVal, 1, p-1) + Copy(sVal, p+1, Length(sVal));
  end;

  Result := ValueFromString(sVal);
end;

destructor TLangSortField.Destroy;
begin
  inherited Destroy;
end;

{ TLangNameField }

constructor TLangNameField.CreateOld;
begin
  inherited CreateOld('', StringType(255, False));
  Self.SortField := TLangSortField.CreateOld(Self);
end;

destructor TLangNameField.Destroy;
begin
  Self.SortField.Free;
  inherited Destroy;
end;


{ TTranslatedStrings }

function ModifyStorage(NewTable : TDataTable; OldStorage : TRowStorage; OldField, NewField : TDataField) : TRowStorage;
var
  RowList : TDataRowList;
  iRow : Integer;
  NewRow : TDataRow;
begin
  RowList := TDataRowList.Create;

  Result := TRowStorage.Create(NewTable, nil, nil, nil, False);
  OldStorage.GetRows(RowList, nil, gaReference);
  for iRow := 0 to RowList.Count - 1 do
  begin
    NewRow := TDataRow.Create(NewTable);
    NewRow.SetDefaultsFrom(RowList[iRow]);
    if OldField <> nil then
      NewRow[NewField] := NewRow[OldField];
    Result.PutRow(NewRow, paDontOverwriteKeys);
  end;

  RowList.Free;
  OldStorage.Free;
end;

{$ifdef D4_OR_HIGHER}
function TTranslatedStrings.QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
const
  E_NOINTERFACE = HResult($80004002);
{$else}
function TTranslatedStrings.QueryInterface(const IID: TGUID; out Obj): Integer; stdcall;
const
  E_NOINTERFACE = $80004002;
{$endif D4_OR_HIGHER}
begin
  if GetInterface(IID, Obj) then Result := 0 else Result := E_NOINTERFACE;
end;

function TTranslatedStrings._AddRef: Integer; stdcall;
begin
  result := 1;
end;

function TTranslatedStrings._Release: Integer; stdcall;
begin
  result := 1;
end;

function TTranslatedStrings.GetTranslations : TRowStorage;
begin
  Result := FTranslations;
end;

function TTranslatedStrings.GetStringTranslations : TRowStorage;
begin
  Result := FStringTranslations;
end;

function TTranslatedStrings.GetAddedProperties : TRowStorage;
begin
  Result := FAddedProperties;
end;

procedure TTranslatedStrings.InsertLanguage(Index : Integer; LanguageName : String; CopyFromLanguage : Integer);
var
  NewLang, OldLang : TDataField;
  NewTranslationTable, NewStringTranslationTable : TDataTable;
  FirstLangField : TDataField;
begin
  if (not Owner.fIgnoreLanguageNameCheck) and (IndexOfLanguage(LanguageName) <> ANYLANGUAGE) then
    raise Exception.Create('Languagename ' + QuotedStr(LanguageName) + ' is already used!');

  if (Index < 0) or (Index > LanguageCount) then
    raise Exception.Create('Invalid index ' + IntToStr(Index) + '!');

  if CopyFromLanguage = ANYLANGUAGE then
    OldLang := nil
  else
    OldLang := LanguageFields[CopyFromLanguage];

  FirstLangField := LanguageFields[0];

  NewLang := TLangNameField.CreateOld;
  NewLang.SetAllDescriptions([LanguageName]);
  FLanguages.InsertObject(Index, '', NewLang);

  NewTranslationTable := TDataTable.CreateCopy(FTranslationTable, nil, nil);
  NewTranslationTable.AddField(NewLang);
  NewTranslationTable.MoveField(NewLang, NewTranslationTable.IndexOfField(FirstLangField) + Index);

  NewStringTranslationTable := TDataTable.CreateCopy(FStringTranslationTable, nil, nil);
  NewStringTranslationTable.AddField(NewLang);
  NewStringTranslationTable.MoveField(NewLang, NewStringTranslationTable.IndexOfField(FirstLangField) + Index);

  FStandardViewLangEditor.AbstractPageView[0].CustomRowView[0].FieldList.Insert(
    FStandardViewLangEditor.AbstractPageView[0].CustomRowView[0].FieldList.IndexOf(FirstLangField) + Index, NewLang);

  FTranslations := ModifyStorage(NewTranslationTable, FTranslations, OldLang, NewLang);
  FStringTranslations := ModifyStorage(NewStringTranslationTable, FStringTranslations, OldLang, NewLang);

  FTranslationTable.Free;
  FTranslationTable := NewTranslationTable;

  FStringTranslationTable.Free;
  FStringTranslationTable := NewStringTranslationTable;

  if Index <= Owner.FLangIndex then
    Inc(Owner.FLangIndex);
end;

procedure TTranslatedStrings.RemoveLanguage(Index : Integer);
var
  OldLang : TDataField;
  NewTranslationTable, NewStringTranslationTable : TDataTable;
begin
  if Index = ANYLANGUAGE then
    Exit;

  if Index = Owner.FLangIndex then
    raise Exception.Create('You are not allowed to remove the currently selected language!');

  OldLang := LanguageFields[Index];

  NewTranslationTable := TDataTable.CreateCopy(FTranslationTable, nil, nil);
  NewTranslationTable.RemoveDataField(OldLang);

  NewStringTranslationTable := TDataTable.CreateCopy(FStringTranslationTable, nil, nil);
  NewStringTranslationTable.RemoveDataField(OldLang);

  FStandardViewLangEditor.AbstractPageView[0].CustomRowView[0].FieldList.Remove(OldLang);

  FTranslations := ModifyStorage(NewTranslationTable, FTranslations, nil, nil);
  FStringTranslations := ModifyStorage(NewStringTranslationTable, FStringTranslations, nil, nil);

  FTranslationTable.Free;
  FTranslationTable := NewTranslationTable;

  FStringTranslationTable.Free;
  FStringTranslationTable := NewStringTranslationTable;

  OldLang.Free;
  FLanguages.Delete(Index);

  if Index <= Owner.FLangIndex  then
    Dec(Owner.FLangIndex);
end;

procedure TTranslatedStrings.CreateDataTables;

  procedure AddLangField(ALangField : TDataField);
  begin
    FTranslationTable.AddField(ALangField);
    FTranslationTable.MoveField(ALangField, 3 + FLanguages.Count);

    FStringTranslationTable.AddField(ALangField);

    FStandardViewLangEditor.SingletonPageView[0].SingletonRowView[0].FieldList.Insert(3 + FLanguages.Count, ALangField);

    FLanguages.AddObject('', ALangField);
  end;

var
  LangField : TDataField;
  i : Integer;

  MT : TTranslator;
begin
  FLanguages := TStringList.Create;

  FTranslationTable := TDataTable.CreateOld('', nil,
                         [FieldComponent, FieldProperty],
                         [FieldClass, {langfields here, } FieldComponentPointer, FieldDoTranslate],
                         nil);
  FStringTranslationTable := TDataTable.CreateOld('', nil,
                         [FieldProperty],
                         [{langfields here, } nil],
                         nil);

  FStandardViewLangEditor := TSingletonStandardView.Create(FTranslationTable, '', '');
  FStandardViewLangEditor.AddPageView(TSingletonPageView.Create(nil, ''));
  FStandardViewLangEditor.AddRowView(TSingletonRowView.Create([FieldClass, FieldComponent, FieldProperty, {langfields here, } FieldDoTranslate], nil));
  FStandardViewLangEditor.SetReadOnly([FieldClass, FieldComponent, FieldProperty], True);

  MT := Owner.MainTranslator;
  if MT <> nil then
  begin
    for i := 0 to MT.Strings.LanguageCount - 1 do
    begin
      LangField := TLangNameField.CreateOld;
      LangField.CopyDescriptionsFrom(MT.Strings.LanguageFields[i]);
      AddLangField(LangField);
    end;
  end
  else
  begin
    // we create two languages by default...
    for i := 1 to 2 do
    begin
      LangField := TLangNameField.CreateOld;
      LangField.SetAllDescriptions(['Language ' + IntToStr(i)]);
      AddLangField(LangField);
    end;
  end;

  FTranslations := TRowStorage.Create(FTranslationTable, nil, nil, nil, False);
  FAddedProperties := TRowStorage.Create(AddedPropertiesTable, nil, nil, nil, False);
  FStringTranslations := TRowStorage.Create(FStringTranslationTable, nil, nil, nil, False);
end;

procedure TTranslatedStrings.DestroyDataTables;
var
  i : Integer;
begin
  FTranslationTable.Free;
  FStringTranslationTable.Free;
  FStandardViewLangEditor.Free;

  for i := 0 to LanguageCount - 1 do
    LanguageFields[i].Free;

  FLanguages.Free;
end;

function TTranslatedStrings.GetShowProperties : TShowProperties;
begin
  Result := FShowProperties;
end;

procedure TTranslatedStrings.SetShowProperties(Show : TShowProperties);
begin
  FShowProperties := Show;
  DisableUnused;
end;

{
procedure TTranslatedStrings.Edit(Designer : IFormDesigner);
var
  Editor: TdlgStringsEditor;
begin
  Owner.UpdateCurrentLanguageFromUI;

  Application.CreateForm(TdlgStringsEditor, Editor);
  try
    Editor.Strings := Self;
    Editor.Caption := 'Editing ' + Owner.GetForm.Name + '.' +Owner.Name;
    Editor.ShowModal;

    if Designer <> nil then // Enable debugging
      Designer.Modified;
  finally
    Editor.Release;
    Owner.UpdateUIFromCurrentLanguage;
  end;
end;
}

function TTranslatedStrings.GetLanguageCount : Integer;
begin
  Result := FLanguages.Count;
end;

function TTranslatedStrings.GetLanguages(idx : Integer) : String;
begin
  Result := LanguageFields[idx].ShortDescription;
end;

procedure TTranslatedStrings.SetLanguages(idx : Integer; LangName : String);
begin
  if Languages[idx] = LangName then
    Exit
  else if (not Owner.fIgnoreLanguageNameCheck) and (IndexOfLanguage(LangName) <> ANYLANGUAGE) then
    raise Exception.Create('Languagename ' + QuotedStr(LangName) + ' is already used!')
  else
    LanguageFields[idx].SetAllDescriptions([LangName]);
end;

function TTranslatedStrings.GetLanguageFields(idx : Integer) : TDataField;
begin
  Result := TDataField(FLanguages.Objects[idx]);
end;

function TTranslatedStrings.LanguageIndexByField(AField : TDataField) : Integer;
begin
  Result := FLanguages.IndexOfObject(AField);
end;

function TTranslatedStrings.FormHasComponent(Form : TForm; ComponentName : String; var Component : TComponent) : Boolean;
begin
  if (Form.Name = ComponentName) then
  begin
    Result := True;
    Component := Form;
  end
  else
  begin
    Component := Form.FindComponent(ComponentName);
    Result := (Component <> nil);
  end;
end;

procedure TTranslatedStrings.DisableUnused;
var
  i : Integer;
  ComponentName : String;
  Form : TForm;
  RowList : TDataRowList;
  Component : TComponent;
begin
  FTranslations.ShowSubTotals := False;
  FTranslations.DetailTreeKey.Visible := True;

  Form := FOwner.GetForm;
  RowList := TDataRowList.Create;
  FTranslations.GetRows(RowList, nil, gaReference);

  for i := 0 to RowList.Count - 1 do
  begin
    ComponentName := RowList.DataRows[i].StringValue[FieldComponent];
    RowList.DataRows[i].Visible := FormHasComponent(Form, ComponentName, Component) and
                                   PropertyTranslated(Component.ClassType, RowList.DataRows[i].StringValue[FieldProperty]);

    if RowList.DataRows[i].Visible and
       (ShowProperties = spTranslatedOnly) and
       (not RowList.DataRows[i].BooleanValue[FieldDoTranslate]) then
      RowList.DataRows[i].Visible := False;
  end;

  FTranslations.ArrangeRows;
  RowList.Free;
end;

function TTranslatedStrings.PropertyTranslated(ClassType : TClass; AProperty : String) : Boolean;
var
  Criteria : TCriteria;
  List : TDataRowList;
begin
  if IsStringsProperty(AProperty, AProperty) then
    AProperty := AProperty + '[]';

  Criteria := TCriteria.Create;
  List := TDataRowList.Create;
  Criteria[FieldClass].AddString(ClassType.ClassName);
  Criteria[FieldProperty].AddString(AProperty);
  FAddedProperties.GetRows(List, Criteria, gaReference);

  Criteria[FieldSubClasses].AddValue(TrueValue);
  Criteria[FieldClass].AcceptNone;

  ClassType := ClassType.ClassParent;
  while ClassType <> nil do
  begin
    Criteria[FieldClass].AddString(ClassType.ClassName);
    ClassType := ClassType.ClassParent;
  end;
  FAddedProperties.GetRows(List, Criteria, gaReference);
  Criteria.Free;

  Result := (List.Count > 0);
  List.Free;
end;

{$ifdef D4_OR_HIGHER}
procedure TTranslatedStrings.FillStrings(ALangIndex : Integer = -1 {ANYLANGUAGE});
{$else}
procedure TTranslatedStrings.FillStrings(ALangIndex : Integer {ANYLANGUAGE});
{$endif D4_OR_HIGHER}
var
  Form : TForm;
  iComponent : Integer;
  TranslatedProperties : TTranslatedProperties;

  procedure ProcessComponent(Component : TComponent);
  var
    iProperty : Integer;
    PropertyName : String;
    Strings : TStrings;
    List : TDataRowList;
  begin
    List := TranslatedProperties.GetList(Component);

    for iProperty := 0 to List.Count - 1 do
    begin
      if IsStringsProperty(List.Strings[iProperty], PropertyName) then
      begin
        if HasStringsProperty(Component, PropertyName, Strings) then
          AddStringsTranslation(Component, PropertyName, Strings, ALangIndex,
                                List.DataRows[iProperty].BooleanValue[FieldDoTranslate]);
      end
      else
      begin
{$ifdef D4_OR_HIGHER}
        if HasProperty(Component, PropertyName) then
{$else}
        if HasPropertyOL(Component, PropertyName) then
{$endif D4_OR_HIGHER}
          AddTranslation(Component, PropertyName, ALangIndex,
                         List.DataRows[iProperty].BooleanValue[FieldDoTranslate]);
      end;
    end;
  end;

begin
  Form := Owner.GetForm;
  TranslatedProperties := TTranslatedProperties.Create(Self);

  ProcessComponent(Form);
  for iComponent := 0 to Form.ComponentCount - 1 do
    ProcessComponent(Form.Components[iComponent]);

  TranslatedProperties.Free;
  FTranslations.ArrangeRows;
end;

function TTranslatedStrings.GetComponents : TList;
var
  Form : TForm;
  iComponent : Integer;
  AList : TList;
begin
  AList := TList.Create;
  Form := Owner.GetForm;

  AList.Add(Form);
  for iComponent := 0 to Form.ComponentCount - 1 do
    AList.Add(Form.Components[iComponent]);
  Result := AList;
end;

function TTranslatedStrings.CreateSubClassList(AClassName : string) : TStrings;
var
  Form : TForm;
  i : Integer;
  AList, ABranch : TStrings;

  function IsParentOf(ASubClass : TClass) : boolean;
  begin
    if (TClass(ASubClass.ClassParent).ClassName = AClassName) or
       ((ASubClass.ClassParent <> nil) and
        IsParentOf(ASubClass.ClassParent)) then
      result := True
    else
      result := False;
  end;

  function CreateBranch(ASubClass : TClass) : TStrings;
  var
    tmpList : TStringList;
    tmpClass : TClass;
  begin
    tmpList := TStringList.Create;
    Result := tmpList;
    if TClass(ASubClass.ClassParent).ClassName = AClassName then
      exit;
    tmpClass := ASubClass;
    while tmpClass.ClassParent <> nil do
    begin
      tmpClass := tmpClass.ClassParent;
      tmpList.AddObject(tmpClass.ClassName, TObject(tmpClass));
      if TClass(tmpClass.ClassParent).ClassName = AClassName then
        Break;
    end;
  end;

begin
  AList := TStringList.Create;
  Form := Owner.GetForm;
  AList.AddObject(Form.ClassName, TObject(Form.ClassType));

  for i := 0 to Form.ComponentCount - 1 do
    if (AList.IndexOf(Form.Components[i].ClassName) = -1) and
       (Form.Components[i].ClassType <> TTranslator) then
      AList.AddObject(Form.Components[i].ClassName, TObject(Form.Components[i].ClassType));

  for i := AList.Count - 1 downto 0 do
  begin
    if not IsParentOf(TClass(AList.Objects[i])) then
      AList.Delete(i)
    else
    begin
      ABranch := CreateBranch(TClass(AList.Objects[i]));
      if ABranch <> nil then
        AList.AddStrings(ABranch);
      ABranch.Free;
    end;
  end;
  Result := AList;
end;

function TTranslatedStrings.CreateClassPropertyList(AClass : TClass) : TStringList;
var
  ATypeKinds : TTypeKinds;
  APropList : PPropList;
  i, ACount, ASize : Integer;
  AList : TStringList;
begin
  if AClass <> nil then
  begin
    AList := TStringList.Create;

    ATypeKinds := [tkClass, tkString, tkLString, tkWString];

    ACount := TypInfo.GetPropList(AClass.ClassInfo, ATypeKinds, nil);
    ASize := ACount * SizeOf(Pointer);
    GetMem(APropList, ASize);

    TypInfo.GetPropList(AClass.ClassInfo, ATypeKinds, APropList);

    for i := 0 to ACount - 1 do
      if APropList^[i].Name <> 'Name' then
      begin
        if (APropList^[i].PropType^.Kind <> tkClass) then
          AList.Add(APropList^[i]^.Name)
        else if ClassRefIsClass(GetTypeData(APropList^[i].PropType^).ClassType, TStrings) then
          AList.Add(APropList^[i]^.Name + '[]');
      end;

    FreeMem(APropList);
    Result := AList;
  end
  else
    Result := nil;
end;

function TTranslatedStrings.HasProperty(Component : TComponent; PropertyName : String; var PropInfo : PPropInfo) : Boolean;
begin
  // Do not allow translation of TTranslator.Language property
  if (Component is TTranslator) and
     (PropertyName = 'Language') then
  begin
    Result := False;
    Exit;
  end;

{$ifdef D4_OR_HIGHER}
  PropInfo := TypInfo.GetPropInfo(Component, PropertyName);
{$else}
  PropInfo := TypInfo.GetPropInfo(Component.ClassInfo, PropertyName);
{$endif D4_OR_HIGHER}
  Result := (PropInfo <> nil);
end;

{$ifdef D4_OR_HIGHER}
function TTranslatedStrings.HasProperty(Component : TComponent; PropertyName : String) : Boolean;
{$else}
function TTranslatedStrings.HasPropertyOL(Component : TComponent; PropertyName : String) : Boolean;
{$endif D4_OR_HIGHER}
var
  PropInfo : PPropInfo;
begin
  Result := HasProperty(Component, PropertyName, PropInfo);
end;

function TTranslatedStrings.SetProperty(Component : TComponent; PropertyName, Value : String) : Boolean;
var
  PropInfo : PPropInfo;
begin
  Result := HasProperty(Component, PropertyName, PropInfo);
  if Result then
    TypInfo.SetStrProp(Component, PropInfo, Value);
end;

function TTranslatedStrings.GetProperty(Component : TComponent; PropertyName : String; var Value : String) : Boolean;
var
  PropInfo : PPropInfo;
begin
  Result := HasProperty(Component, PropertyName, PropInfo);
  if Result then
    Value := TypInfo.GetStrProp(Component, PropInfo);
end;

function TTranslatedStrings.HasStringsProperty(Component : TComponent; PropertyName : String; var Strings : TStrings) : Boolean;
var
  PropInfo : PPropInfo;
begin
{$ifdef D4_OR_HIGHER}
  PropInfo := TypInfo.GetPropInfo(Component, PropertyName);
{$else}
  PropInfo := TypInfo.GetPropInfo(Component.ClassInfo, PropertyName);
{$endif D4_OR_HIGHER}
  Result := (PropInfo <> nil) and (PropInfo.PropType^.Kind = tkClass);
  if Result then
  begin
{$ifdef D4_OR_HIGHER}
    Strings := TStrings(TypInfo.GetObjectProp( Component, PropInfo, TStrings ));
{$else}
    Strings := nil; // FIXA!!! -- denna hittades inte under Delphi3
{$endif D4_OR_HIGHER}
    Result := (Strings <> nil);
  end
  else
    Strings := nil;
end;

{function TTranslatedStrings.HasStringsProperty(Component : TComponent; PropertyName : String) : Boolean;
var
  Strings : TStrings;
begin
  Result := HasStringsProperty(Component, PropertyName, Strings);
end;
}

function TTranslatedStrings.GetEditorStandardView : TSingletonStandardView;
begin
  Result := FStandardViewLangEditor;
end;

function TTranslatedStrings.GetTranslationItemByName(Component : TComponent; PropertyName : String) : TDataRow;
var
  Criteria : TCriteria;
begin
  if (Component = nil) or (Component.Name = '') then
  begin
    Result := nil;
    Exit;
  end;

  Result := FTranslations.LocateRow([Component.Name, PropertyName]);
  if Result <> nil then
  begin
    Result.PointerValue[FieldComponentPointer] := Component;
    Result.StringValue[FieldClass] := Component.ClassName;
  end
  else if (csDesigning in Owner.ComponentState) then // Only do this when designing because it slown down opening of forms. We do not have any FieldComponentPointer-info when not in design mode!
  begin
    Criteria := TCriteria.Create;
    Criteria[FieldClass].AddString(Component.ClassName);
    Criteria[FieldProperty].AddString(PropertyName);
    Criteria[FieldComponentPointer].AddValue(ValueFromObject(Component));
    Result := FTranslations.LocateRowByCriteria(Criteria, True);
    Criteria.Free;

    if Result <> nil then
    begin
//      if Component.Name <> Result.StringValue[FieldComponent] then
//        ShowMessage(Result.StringValue[FieldComponent] + ' -> ' + Component.Name);
      Result.SetFieldValue( FieldComponent, ValueFromString(Component.Name), saOverwriteOnKeyChange); // correct the component's name-link if changed
    end;
  end;
end;

procedure TTranslatedStrings.AddStringsTranslation(Component : TComponent; PropertyName : String; Strings : TStrings; ALangIndex : Integer; DoTranslate : Boolean);
var
  Item : TDataRow;
  iItem, iLangIndex : Integer;
begin
  for iItem := 0 to Strings.Count - 1 do
  begin
    Item := TranslationItemByName[Component, PropertyName + '[' + IntToStr(iItem) + ']'];
    if Item = nil then
    begin
      Item := TDataRow.Create(FTranslationTable);
      Item.StringValue[FieldComponent] := Component.Name;
      Item.StringValue[FieldProperty] := PropertyName + '[' + IntToStr(iItem) + ']';
      Item.PointerValue[FieldComponentPointer] := Component;
      Item.StringValue[FieldClass] := Component.ClassName;

      for iLangIndex := 0 to LanguageCount - 1 do
        Item.StringValue[LanguageFields[iLangIndex]] := Strings[iItem];

      Item.BooleanValue[FieldDoTranslate] := DoTranslate;
      FTranslations.PutRow(Item, paDontOverwriteKeys);
    end
    else if Item.BooleanValue[FieldDoTranslate] then
    begin
      if ALangIndex <> ANYLANGUAGE then
        Item.StringValue[LanguageFields[ALangIndex]] := Strings[iItem];
    end
    else // if property not translated -> update all languages
    begin
      for iLangIndex := 0 to LanguageCount - 1 do
        Item.StringValue[LanguageFields[iLangIndex]] := Strings[iItem];
    end;
  end;
end;

procedure TTranslatedStrings.AddTranslation(Component : TComponent; PropertyName : String; ALangIndex : Integer; DoTranslate : Boolean);
var
  Item : TDataRow;
  PropertyValue : String;
  iLangIndex : Integer;
begin
  if not GetProperty(Component, PropertyName, PropertyValue) then
    Exit;

  Item := TranslationItemByName[Component, PropertyName];
  if Item = nil then
  begin
    Item := TDataRow.Create(FTranslationTable);
    Item.StringValue[FieldComponent] := Component.Name;
    Item.StringValue[FieldProperty] := PropertyName;
    Item.PointerValue[FieldComponentPointer] := Component;
    Item.StringValue[FieldClass] := Component.ClassName;

    for iLangIndex := 0 to LanguageCount - 1 do
     { if (ALangIndex = ANYLANGUAGE) or  // LGE says: update all languages for new properties
         (ALangIndex = iLangIndex) then }
        Item.StringValue[LanguageFields[iLangIndex]] := PropertyValue;

    Item.BooleanValue[FieldDoTranslate] := DoTranslate;
    FTranslations.PutRow(Item, paDontOverwriteKeys);
  end
  else if Item.BooleanValue[FieldDoTranslate] then
  begin
    if ALangIndex <> ANYLANGUAGE then
      Item.StringValue[LanguageFields[ALangIndex]] := PropertyValue;
  end
  else // update all languages if property not translated
  begin
    for iLangIndex := 0 to LanguageCount - 1 do
      Item.StringValue[LanguageFields[iLangIndex]] := PropertyValue;
  end;
end;

constructor TTranslatedStrings.Create(Owner : TTranslator);
begin
  inherited Create;
  FOwner := Owner;

  CreateDataTables;
end;

destructor TTranslatedStrings.Destroy;
begin
  inherited Destroy;

  FTranslations.Free;
  FAddedProperties.Free;
  FStringTranslations.Free;

  DestroyDataTables;
end;

function TTranslatedStrings.GetString(const AProperty : String; LangIndex : Integer) : String;
var
  Row : TDataRow;
begin
  Row := FStringTranslations.LocateRow([AProperty]);
  if Row = nil then
    Result := '[' + AProperty + ']'
  else
    Result := Row.StringValue[LanguageFields[LangIndex]];
end;

{$ifdef D4_OR_HIGHER}
function TTranslatedStrings.GetString(const AProperty : String; LangIndex : Integer; Variables : array of String) : String;
{$else}
function TTranslatedStrings.GetStringOL(const AProperty : String; LangIndex : Integer; Variables : array of String) : String;
{$endif D4_OR_HIGHER}
var
  iVar : Integer;
  VarNr : Integer;
  VarPos : Integer;
  VarEndPos : Integer;
begin
  Result := GetString(AProperty, LangIndex);

  iVar := 1;
  VarPos := 1;
  repeat
    VarPos := SubStrPos(VarPos, '<$', Result);
    if VarPos < 1 then
      Break;
    VarEndPos := SubStrPos(VarPos, '>', Result);
    if VarEndPos < 1 then
      Break;

    VarNr := iVar;

    if VarEndPos - VarPos > 2 then
    begin
      try
        VarNr := StrToInt(Copy(Result, VarPos + 2, VarEndPos - VarPos - 2));
      except
        VarPos := VarEndPos;
        Continue;
      end;
    end;

    if VarNr - 1 <= High(Variables) - Low(Variables) then
      Result := Copy(Result, 1, VarPos - 1) +
                Variables[VarNr - 1 + Low(Variables)] +
                Copy(Result, VarEndPos + 1, Length(Result))
    else
      Result := Copy(Result, 1, VarPos - 1) + Copy(Result, VarEndPos + 1, Length(Result));

    Inc(iVar);
  until False;
end;

function TTranslatedStrings.GetTranslation(Component : TComponent; PropertyName : String; ALangIndex : Integer; var Value : String) : Boolean;
var
  Item : TDataRow;
begin
  Item := TranslationItemByName[Component, PropertyName];
  if (Item = nil) or (not Item.BooleanValue[FieldDoTranslate]) then
    Result := False
  else
  begin
    Result := True;
    Value := Item.StringValue[LanguageFields[ALangIndex]];
  end;
end;

function TTranslatedStrings.IndexOfLanguage(LangName : String) : Integer;
var
  i : Integer;
begin
  Result := ANYLANGUAGE;

  for i := 0 to LanguageCount - 1 do
    if Languages[i] = LangName then
    begin
      Result := i;
      Break;
    end;
end;

procedure TTranslatedStrings.GetTranslatedProperties(Component : TComponent; List : TDataRowList);
var
  Criteria : TCriteria;
  AClass : TClass;
  ListSuperClass : TDataRowList;
  i : Integer;
begin
  if (Component = nil) or (Component.Name = '') then
    Exit;

  Criteria := TCriteria.Create;
  Criteria[FieldClass].AddString(Component.ClassName);
  FAddedProperties.GetRows(List, Criteria, gaReference);
  List.FillStringsOptional(FieldProperty, dvKeyOnly);

  Criteria[FieldSubClasses].AddValue(TrueValue);
  Criteria[FieldClass].AcceptNone;

  ListSuperClass := TDataRowList.Create;

  AClass := Component.ClassType.ClassParent;
  while AClass <> nil do
  begin
    ListSuperClass.Clear;
    Criteria[FieldClass].AddString(AClass.ClassName);
    FAddedProperties.GetRows(ListSuperClass, Criteria, gaReference);
    for i := 0 to ListSuperClass.Count - 1 do
      if List.IndexOf(ListSuperClass.DataRows[i].StringValue[FieldProperty]) = -1 then
        List.AddObject(ListSuperClass.DataRows[i].StringValue[FieldProperty], ListSuperClass.DataRows[i]);
    AClass := AClass.ClassParent;
  end;

  Criteria.Free;
  ListSuperClass.Free;
end;

procedure TTranslatedStrings.UpdateUI;
begin
  if Owner <> nil then
    Owner.UpdateUIFromCurrentLanguage;
end;

procedure TTranslatedStrings.UpdateLanguage;
begin
  if Owner <> nil then
    Owner.UpdateCurrentLanguageFromUI;
end;

procedure TTranslatedStrings.CheckLanguageNames;
var
  i, j : Integer;
begin
  for i := 0 to Self.LanguageCount - 2 do
    for j := i+1 to Self.LanguageCount - 1 do
      if Self.Languages[i] = Self.Languages[j] then
        raise Exception.Create('Languagename ' + QuotedStr(Self.Languages[i]) + ' is used more than once!');
end;

{ TTranslator }

class function TTranslator.TranslatorVersion : String;
begin
  Result := CURRENTVERSION;
end;

function TTranslator.GetAbout : TAboutTranslator;
begin
  Result := nil; // Result is meaningless...
end;

constructor TTranslator.Create(AOwner: TComponent);
begin
  fIgnoreLanguageNameCheck := False;

  if TranslatorList <> nil then
    TranslatorList.Add(Self);

  inherited Create(AOwner);

  FLangIndex := 0;
  FTranslateApplication := False;
  FStrings := TTranslatedStrings.Create(Self);
end;

function TTranslator.GetString(const AProperty : String) : String;
begin
  Result := FStrings.GetString(AProperty, FLangIndex);
end;

{$ifdef D4_OR_HIGHER}
function TTranslator.GetString(const AProperty : String; Variables : array of String) : String;
begin
  Result := FStrings.GetString(AProperty, FLangIndex, Variables);
end;
{$else}
function TTranslator.GetStringOL(const AProperty : String; Variables : array of String) : String;
begin
  Result := FStrings.GetStringOL(AProperty, FLangIndex, Variables);
end;
{$endif D4_OR_HIGHER}

destructor TTranslator.Destroy;
begin
  inherited Destroy;
  FStrings.Free;

  if TranslatorList <> nil then
    TranslatorList.Remove(Self);
end;

function TTranslator.GetForm : TForm;
begin
  Result := TForm(Self.Owner);
end;

function TTranslator.GetStrings : TTranslatedStrings;
begin
  Result := FStrings;
end;

procedure TTranslator.UpdateCurrentLanguageFromUI;
begin
  FStrings.FillStrings(FLangIndex);
end;

procedure TTranslator.UpdateUIFromCurrentLanguage;
begin
  TranslateStrings(FLangIndex);
end;

function TTranslator.GetActiveLanguage : String;
begin
  Result := FStrings.Languages[FLangIndex];
end;

procedure TTranslator.TranslateTo(LangName : String);
var
  LangIndex : Integer;
begin
  if csReading in ComponentState then
    Exit;

  LangIndex := FStrings.IndexOfLanguage(LangName);

  if LangIndex = ANYLANGUAGE then
    raise Exception.Create('Unknown language ' + QuotedStr(LangName));

  TranslateRecursive(LangName, TranslateApplication);
end;

procedure TTranslator.TranslateRecursive(LangName : String; Recursive : Boolean);
var
  LangIndex : Integer;
  iForm, iControl : Integer;
begin
  LangIndex := FStrings.IndexOfLanguage(LangName);
  if LangIndex = ANYLANGUAGE then
    Exit;

  if LangIndex <> FLangIndex then
  begin
    if Assigned(BeforeTranslate) then
      BeforeTranslate(Self, FStrings.Languages[FLangIndex], LangName);

    UpdateCurrentLanguageFromUI;
    TranslateStrings(LangIndex);
    FLangIndex := LangIndex;

    if Recursive then
      for iForm := 0 to Application.ComponentCount - 1 do
        for iControl := 0 to Application.Components[iForm].ComponentCount - 1 do
          if Application.Components[iForm].Components[iControl] is TTranslator then
            TTranslator(Application.Components[iForm].Components[iControl]).TranslateRecursive(LangName, False);

    if Assigned(AfterTranslated) then
      AfterTranslated(Self, FStrings.Languages[FLangIndex], LangName);
  end;
end;

function TTranslator.MainTranslator : TTranslator;
var
  i : Integer;
begin
  Result := nil;

  if TranslatorList <> nil then
  for i := 0 to TranslatorList.Count - 1 do
    if (TranslatorList.Items[i] <> Self) and
       TTranslator(TranslatorList.Items[i]).TranslateApplication then
    begin
      Result := TTranslator(TranslatorList.Items[i]);
      Exit;
    end;
end;

procedure TTranslator.Loaded;
var
  MT : TTranslator;
begin
  inherited Loaded;

  MT := MainTranslator;

  // When a new form is loaded, we translate it
  // a) to the selected language -- if we are designing or we have no main translator
  // b) to the language of the Main TTranslator -- when running

  // Fixa LGE: Behver vi verstta ifall vi redan har rtt language?
  if (MT = nil) or (csDesigning in ComponentState) then
  begin
    UpdateUIFromCurrentLanguage;
  end
  else
  begin
    FLangIndex := FStrings.IndexOfLanguage(MT.Language);
    if FLangIndex = ANYLANGUAGE then
      FLangIndex := 0;
    UpdateUIFromCurrentLanguage;
  end;
end;

procedure TTranslator.TranslateStrings(ALangIndex : Integer);
var
  Form : TForm;
  iComponent : Integer;
  TranslatedProperties : TTranslatedProperties;

  procedure ProcessComponent(Component : TComponent);
  var
    iItem, iProperty : Integer;
    Value : String;
    PropertyName : String;
    Strings : TStrings;
    List : TDataRowList;
  begin
    List := TranslatedProperties.GetList(Component);

    for iProperty := 0 to List.Count - 1 do
    begin
      if IsStringsProperty(List.Strings[iProperty], PropertyName) then
      begin
        if FStrings.HasStringsProperty(Component, PropertyName, Strings) then
          for iItem := 0 to Strings.Count - 1 do
            if FStrings.GetTranslation(Component, PropertyName + '[' + IntToStr(iItem) + ']', ALangIndex, Value) then
              Strings[iItem] := Value;
      end
      else
      begin
        if FStrings.GetTranslation(Component, PropertyName, ALangIndex, Value) then
          FStrings.SetProperty(Component, PropertyName, Value);
      end;
    end;
  end;

begin
  Form := GetForm;
  TranslatedProperties := TTranslatedProperties.Create(FStrings);

  ProcessComponent(Form);
  for iComponent := 0 to Form.ComponentCount - 1 do
  begin
    ProcessComponent(Form.Components[iComponent]);
  end;

  TranslatedProperties.Free;
end;



{
type
  TReaderLink = class(TReader); }

procedure TTranslator.LoadProperty(Reader : TReader);
var
  DfmVersion : Double;

  function StrAsBool(B : String) : Boolean;
  begin
    Result := (LowerCase(B) = 'true');
  end;

  function StrAsPointer(Str : String) : Pointer;
  begin
    Result := Pointer(StrToInt(Str));
  end;

  function ValueFromNonSepStr(Str : String) : TValue;
  var
    p : Integer;
  begin
    p := 1;
    while p > 0 do
    begin
      p := SubStrPos(p, '/_', Str);
      if p > 0 then
      begin
        Str := Copy(Str, 1, p-1) + '/' + Copy(Str, p+2, Length(Str));
        Inc(p);
      end;
    end;
    Result := ValueFromString(Str);
  end;

  function CutNTrimText(var AStr : String; const Separator : String) : String;
  var
    p : Integer;
  begin
    p := Pos(Separator, AStr);
    if p < 1 then
    begin
      Result := Trim(AStr);
      AStr := '';
    end
    else
    begin
      Result := Trim(Copy(AStr, 1, p-1));
      AStr := Trim(Copy(AStr, p+Length(Separator), Length(AStr)));
    end;
  end;

  procedure ReadPropStorage(Storage : TRowStorage);
  var
    Row : TDataRow;
    StrVal : String;
  begin
    Reader.ReadListBegin;
    while not Reader.EndOfList do
    begin
      StrVal := Reader.ReadString;

      Row := TDataRow.Create(Storage.DataTable);
      Row.StringValue[FieldClass] := CutNTrimText(StrVal, '.');
      Row.StringValue[FieldProperty] := CutNTrimText(StrVal, ':');
      Row.BooleanValue[FieldSubClasses] := StrAsBool(CutNTrimText(StrVal, ','));
      Row.BooleanValue[FieldDoTranslate] := StrAsBool(Trim(StrVal));
      Storage.PutRow(Row, paOverwriteonKeyChange);
    end;
    Reader.ReadListEnd;
  end;

  procedure ReadComponentStorage(Storage : TRowStorage);
  var
    j : Integer;
    Row : TDataRow;
    StrVal : String;
  begin
    Reader.ReadListBegin;

(*    if Version >= 1.05 then
    begin
      // class loop
      while not Reader.EndOfList do
      begin
        ClassName := Reader.ReadIdent;
        Reader.ReadListBegin;

        // component loop
        while not Reader.EndOfList do
        begin
          ComponentName := Reader.ReadIdent;
          Reader.ReadListBegin;

          // property loop
          while not Reader.EndOfList do
          begin
            Row := TDataRow.Create(Storage.DataTable);

            Row.StringValue[FieldClass] := ClassName;
            Row.StringValue[FieldComponent] := ComponentName;
            Row.PointerValue[FieldComponentPointer] := nil;

            StrVal := Reader.ReadString;
            Row.StringValue[FieldProperty] := CutNTrimText(StrVal, ':');
            Row.BooleanValue[FieldDoTranslate] := StrAsBool(Trim(StrVal));

            if Row.BooleanValue[FieldDoTranslate] then
            begin
              if Reader.NextValue = vaList then
              begin
                Reader.ReadListBegin;
                for j := 0 to FStrings.LanguageCount - 1 do
                  Row.StringValue[FStrings.LanguageFields[j]] := Reader.ReadString;
                Reader.ReadListEnd;
              end
              else // property isn't really translated -- use only value...
              begin
                StrVal := Reader.ReadString;
                for j := 0 to FStrings.LanguageCount - 1 do
                  Row.StringValue[FStrings.LanguageFields[j]] := StrVal;
              end;
            end;

            Storage.PutRow(Row, paOverwriteonKeyChange);
          end;

          Reader.ReadListEnd;
        end;

        Reader.ReadListEnd;
      end;
    end
    else
    begin *)
      while not Reader.EndOfList do
      begin
        Row := TDataRow.Create(Storage.DataTable);

        StrVal := Reader.ReadString;
        if DfmVersion >= 1.02 then
        begin
          Row.StringValue[FieldClass] := CutNTrimText(StrVal, '(');
          Row.PointerValue[FieldComponentPointer] := nil;
          Row.StringValue[FieldComponent] := CutNTrimText(StrVal, ').');
        end
        else
        begin
          Row.StringValue[FieldClass] := CutNTrimText(StrVal, '(');
          Row.PointerValue[FieldComponentPointer] := StrAsPointer(CutNTrimText(StrVal, '),'));
          Row.StringValue[FieldComponent] := CutNTrimText(StrVal, '.');
        end;
        Row.StringValue[FieldProperty] := CutNTrimText(StrVal, ':');
        Row.BooleanValue[FieldDoTranslate] := StrAsBool(Trim(StrVal));

        if DfmVersion >= 1.04 then
        begin
          if Row.BooleanValue[FieldDoTranslate] then
          begin
            if Reader.NextValue = vaList then
            begin
              Reader.ReadListBegin;
              for j := 0 to FStrings.LanguageCount - 1 do
                Row.StringValue[FStrings.LanguageFields[j]] := Reader.ReadString;
              Reader.ReadListEnd;
            end
            else // property isn't really translated -- use only value...
            begin
              StrVal := Reader.ReadString;
              for j := 0 to FStrings.LanguageCount - 1 do
                Row.StringValue[FStrings.LanguageFields[j]] := StrVal;
            end;
          end;
        end
        else if DfmVersion >= 1.03 then
        begin
          Reader.ReadListBegin;
          for j := 0 to FStrings.LanguageCount - 1 do
            Row.StringValue[FStrings.LanguageFields[j]] := Reader.ReadString;
          Reader.ReadListEnd;
        end
        else
        begin
          StrVal := Reader.ReadString;
          for j := 3 to 4 do
            Row.ValueByIndex[j] := ValueFromNonSepStr(CutNTrimText(StrVal, '//'));
        end;

        Storage.PutRow(Row, paOverwriteonKeyChange);
      end;
//    end;

    Reader.ReadListEnd;
  end;

  procedure ReadTranslationStorage(Storage : TRowStorage);
  var
    j : Integer;
    Row : TDataRow;
    StrVal : String;
  begin
    Reader.ReadListBegin;
    while not Reader.EndOfList do
    begin
      Row := TDataRow.Create(Storage.DataTable);

      StrVal := Reader.ReadString;
      if DfmVersion >= 1.03 then
      begin
        Row.StringValue[FieldProperty] := StrVal;
        Reader.ReadListBegin;
        for j := 0 to FStrings.LanguageCount - 1 do
          Row.StringValue[FStrings.LanguageFields[j]] := Reader.ReadString;
        Reader.ReadListEnd;
      end
      else
      begin
        Row.StringValue[FieldProperty] := CutNTrimText(StrVal, ':');
        for j := 1 to 2 do
          Row.ValueByIndex[j] := ValueFromNonSepStr(CutNTrimText(StrVal, '//'));
      end;

      Storage.PutRow(Row, paOverwriteonKeyChange);
    end;
    Reader.ReadListEnd;
  end;

  function VersionToFloat(Str : String) : Double;
  begin
    Result := StrToInt(CutNTrimText(Str, '.'));
    Result := Result + StrToInt(Str) / 100.0;

    if Length(Str) <> 2 then
      raise Exception.Create('Invalid version number!');
  end;

var
  VersionStr, AllLanguages, ALanguageName, SelectedLanguage, ShowPropertiesStr : String;
  iShowProperties : TShowProperties;
  iLanguage, iRemoveLang : Integer;
begin
  Reader.ReadListBegin;
  VersionStr := Reader.ReadString;
  CutNTrimText(VersionStr, ':');
  DfmVersion := VersionToFloat(VersionStr);
  if DfmVersion > VersionToFloat(STREAMINGVERSION) then
    raise Exception.Create('Cannot read dfm! Your TTranslator version is ' + CURRENTVERSION +
                           ', but the dfm is saved using a newer version!') // ' + VersionStr + '!')
  else


  if DfmVersion < 1.03 then
  begin
    AllLanguages := Reader.ReadString;
    CutNTrimText(AllLanguages, ':');
    FStrings.Languages[0] := CutNTrimText(AllLanguages, ',');
    FStrings.Languages[1] := AllLanguages;
  end
  else
  begin
    if Reader.ReadString <> 'Languages' then
      raise Exception.Create('Error reading Languages!');

    Reader.ReadListBegin;
    iLanguage := 0;
    fIgnoreLanguageNameCheck := True;
    try
      while not Reader.EndOfList do
      begin
        ALanguageName := Reader.ReadString;
        if iLanguage < FStrings.LanguageCount then
          FStrings.Languages[iLanguage] := ALanguageName
        else
          FStrings.InsertLanguage(iLanguage, ALanguageName, ANYLANGUAGE);

        Inc(iLanguage);
      end;

      for iRemoveLang := FStrings.LanguageCount - 1 downto iLanguage do
        FStrings.RemoveLanguage(iRemoveLang);
    finally
      fIgnoreLanguageNameCheck := False;
    end;

    FStrings.CheckLanguageNames;

    Reader.ReadListEnd;
  end;

  SelectedLanguage := Reader.ReadString;
  CutNTrimText(SelectedLanguage, ':');
  FLangIndex := FStrings.IndexOfLanguage(SelectedLanguage);
  if FLangIndex = ANYLANGUAGE then
    FLangIndex := 0;

  if DfmVersion >= 1.01 then
  begin
    ShowPropertiesStr := Reader.ReadString;
    CutNTrimText(ShowPropertiesStr, ':');
    for iShowProperties := Low(TShowProperties) to High(TShowProperties) do
      if TypInfo.GetEnumName(TypeInfo(TShowProperties), Ord(iShowProperties)) = ShowPropertiesStr then
      begin
        FStrings.ShowProperties := iShowProperties;
        Break;
      end;
  end;

  ReadPropStorage(FStrings.AddedProperties);
  ReadComponentStorage(FStrings.FTranslations);
  ReadTranslationStorage(FStrings.FStringTranslations);

  Reader.ReadListEnd;
end;

procedure TTranslator.StoreProperty(Writer : TWriter);

  function BoolStr(Bool : Boolean) : String;
  begin
    if Bool then
      Result := 'True'
    else
      Result := 'False';
  end;

  procedure BeforeWriteStorage(Storage : TRowStorage);
  begin
    Storage.DetailTreeKey.Visible := True;
    Storage.ShowSubTotals := False;
    Storage.ArrangeRows;
  end;

  procedure WritePropStorage(Storage : TRowStorage);
  var
    i : Integer;
    Row : TDataRow;
  begin
    BeforeWriteStorage(Storage);

    Writer.WriteListBegin;
    for i := 0 to Storage.RowCount - 1 do
    begin
      Row := TDataRow(Storage.Rows[i]);
      Writer.WriteString(Row.StringValue[FieldClass] + '.' +
                         Row.StringValue[FieldProperty] + ': ' +
                         BoolStr(Row.BooleanValue[FieldSubClasses]) + ', ' +
                         BoolStr(Row.BooleanValue[FieldDoTranslate]));
    end;
    Writer.WriteListEnd;
  end;

  procedure WriteComponentStorage(Storage : TRowStorage);
  var
    i, j : Integer;
    Row : TDataRow;
    OnlyValue : String;
    OneValue : Boolean;
  begin
    BeforeWriteStorage(Storage);

(*
    Writer.WriteListBegin;
    ClassName := '';
    ComponentName := '';
    for i := 0 to Storage.RowCount - 1 do
    begin
      Row := TDataRow(Storage.Rows[i]);

      ClassEquals := (ClassName <> Row.StringValue[FieldClass]);
      ComponentEquals := (ComponentName <> Row.StringValue[FieldComponent]);


      if (i > 0) and
         ((not ClassEquals) or (not ComponentEquals)) then
        Writer.WriteListEnd;

      if (i = 0) or (not ClassEquals) then
      begin
        if i > 0 then
          Writer.WriteListEnd;

        ClassName := Row.StringValue[FieldClass];
        Writer.WriteIdent(ClassName);
        Writer.WriteListBegin;
      end;

      if (i = 0) or (not ClassEquals) or (not ComponentEquals) then
      begin
        ComponentName := Row.StringValue[FieldComponent];
        Writer.WriteIdent(ComponentName);
        Writer.WriteListBegin;
      end;

      Writer.WriteString(Row.StringValue[FieldProperty] + ': ' +
                         BoolStr(Row.BooleanValue[FieldDoTranslate]));

      if Row.BooleanValue[FieldDoTranslate] then
      begin
        if FStrings.LanguageCount = 0 then
        begin
          OneValue := False;
          OnlyValue := '';
        end
        else
        begin
          OneValue := True;
          OnlyValue := Row.StringValue[FStrings.LanguageFields[0]];
          for j := 1 to FStrings.LanguageCount - 1 do
            if OnlyValue <> Row.StringValue[FStrings.LanguageFields[j]] then
            begin
              OneValue := False;
              Break;
            end;
        end;

        if OneValue then
        begin
          Writer.WriteString(OnlyValue);
        end
        else
        begin
          Writer.WriteListBegin;
          for j := 0 to FStrings.LanguageCount - 1 do
            Writer.WriteString(Row.StringValue[FStrings.LanguageFields[j]]);
          Writer.WriteListEnd;
        end;
      end;
    end;

    if Storage.RowCount > 0 then
    begin
      Writer.WriteListEnd;
      Writer.WriteListEnd;
    end;

    Writer.WriteListEnd; *)

    Writer.WriteListBegin;
    for i := 0 to Storage.RowCount - 1 do
    begin
      Row := TDataRow(Storage.Rows[i]);

      Writer.WriteString(Row.StringValue[FieldClass] + '(' +
                         Row.StringValue[FieldComponent] + ').' +
                         Row.StringValue[FieldProperty] + ': ' +
                         BoolStr(Row.BooleanValue[FieldDoTranslate]));

      if Row.BooleanValue[FieldDoTranslate] then
      begin
        if FStrings.LanguageCount = 0 then
        begin
          OneValue := False;
          OnlyValue := '';
        end
        else
        begin
          OneValue := True;
          OnlyValue := Row.StringValue[FStrings.LanguageFields[0]];
          for j := 1 to FStrings.LanguageCount - 1 do
            if OnlyValue <> Row.StringValue[FStrings.LanguageFields[j]] then
            begin
              OneValue := False;
              Break;
            end;
        end;

        if OneValue then
        begin
          Writer.WriteString(OnlyValue);
        end
        else
        begin
          Writer.WriteListBegin;
          for j := 0 to FStrings.LanguageCount - 1 do
            Writer.WriteString(Row.StringValue[FStrings.LanguageFields[j]]);
          Writer.WriteListEnd;
        end;
      end;
    end;
    Writer.WriteListEnd;
  end;

  procedure WriteTranslationStorage(Storage : TRowStorage);
  var
    i, j : Integer;
    Row : TDataRow;
  begin
    BeforeWriteStorage(Storage);

    Writer.WriteListBegin;
    for i := 0 to Storage.RowCount - 1 do
    begin
      Row := TDataRow(Storage.Rows[i]);
      Writer.WriteString(Row.StringValue[FieldProperty]);
      Writer.WriteListBegin;
      for j := 0 to FStrings.LanguageCount - 1 do
        Writer.WriteString(Row.StringValue[FStrings.LanguageFields[j]]);
      Writer.WriteListEnd;
    end;
    Writer.WriteListEnd;
  end;

var
  i : Integer;
begin
//  if Writer.Ancestor <> nil then
//    ShowMessage('Writer: ' + Writer.Ancestor.ClassName);

  UpdateCurrentLanguageFromUI;

  Writer.WriteListBegin;
  Writer.WriteString('Version: ' + STREAMINGVERSION);

  Writer.WriteString('Languages');
  Writer.WriteListBegin;
  for i := 0 to FStrings.LanguageCount - 1 do
    Writer.WriteString(FStrings.Languages[i]);
  Writer.WriteListEnd;

  Writer.WriteString('Currentlanguage: ' + Self.Language);
  Writer.WriteString('ShowProperties: ' + TypInfo.GetEnumName(TypeInfo(TShowProperties), Ord(FStrings.ShowProperties)));

  WritePropStorage(FStrings.AddedProperties);
  WriteComponentStorage(FStrings.FTranslations);
  WriteTranslationStorage(FStrings.FStringTranslations);

  Writer.WriteListEnd;
end;

procedure TTranslator.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('Translations', LoadProperty, StoreProperty, True);
end;

{ TTranslatedProperties }

constructor TTranslatedProperties.Create(TranslatedStrings : TTranslatedStrings);
begin
  inherited Create;

  fTranslatedStrings := TranslatedStrings;

  fList := TStringList.Create;
  fList.Sorted := True;
  fList.Duplicates := dupError;
end;

destructor TTranslatedProperties.Destroy;
var
  i : Integer;
begin
  for i := 0 to fList.Count - 1 do
    fList.Objects[i].Free;
  fList.Free;

  inherited Destroy;
end;

function TTranslatedProperties.GetList(Component : TComponent) : TDataRowList;
var
  idx : Integer;
begin
  if fList.Find( Component.ClassName, idx ) then
  begin
    Result := TDataRowList(fList.Objects[idx]);
  end
  else
  begin
    Result := TDataRowList.Create;
    fTranslatedStrings.GetTranslatedProperties(Component, Result);
    fList.AddObject(Component.ClassName, Result);
  end;
end;

initialization

  TranslatorList := TList.Create;

  FieldClass := TKeyField.CreateNonAuxtabled('', StringType(64, False));
  FieldClass.SetAllDescriptions(['Class']);
  FieldComponent := TKeyField.CreateNonAuxtabled('', StringType(64, False));
  FieldComponent.SetAllDescriptions(['Component']);
  FieldProperty := TKeyField.CreateNonAuxtabled('', StringType(64, False));
  FieldProperty.SetAllDescriptions(['Property']);

  FieldDoTranslate := TDataField.CreateOld('', BooleanType);
  FieldDoTranslate.SetAllDescriptions(['Translate']);
  FieldDoTranslateText := TBooleanTextField.CreateOld('', FieldDoTranslate);

  FieldSubClasses := TDataField.CreateOld('', BooleanType);
  FieldSubClasses.SetAllDescriptions(['Include subclasses']);
  FieldSubClassesText := TBooleanTextField.CreateOld('', FieldSubClasses);

  FieldComponentPointer := TDataField.CreateOld('', ObjectType);
  FieldComponentPointer.SetAllDescriptions(['Pointer to component']);

  AddedPropertiesTable := TDataTable.CreateOld('', nil,
                         [FieldClass, FieldProperty],
                         [FieldSubClasses, FieldDoTranslate],
                         nil);

finalization

  FieldClass.Free;
  FieldComponent.Free;
  FieldProperty.Free;
  FieldDoTranslate.Free;
  FieldSubClasses.Free;
  FieldComponentPointer.Free;
  AddedPropertiesTable.Free;
  FieldDoTranslateText.Free;
  FieldSubClassesText.Free;

  TranslatorList.Free;
  TranslatorList := nil;

end.

