{ $Id: Interruptable.pas,v 1.2 2001/10/11 11:52:06 laa Exp $}

{
    This file is part of the TTranslator 

    TTranslator is a Delphi component for localizing String and TStrings 
    properties of components dropped on a form. You can also localize your 
    code strings with TTranslator.
    Copyright (C) 2002 Polycon Ab

    This is a licensed version of TTranslator, it may be used as described
    in the TTranslator license agreement. If you have not acquired a 
    commercial TTranslator license, your are using this product illegaly.    
}

unit Interruptable;

interface

uses
  DataTypes;

type
  TInterruptable = class;

  TOperationEvent = procedure(Sender : TInterruptable; Operation : TOperationType) of object;

  TInterruptable = class
  private
    FInterruptInterval : Integer;
    FOnStart : TOperationEvent;
    FOnInterrupt : TOperationEvent;
    FOnFinished : TOperationEvent;
    FRowsAffected : Integer;
    FInterrupted : Boolean;

    function GetInterruptInterval : Integer;
    function GetOnStart : TOperationEvent;
    function GetOnInterrupt : TOperationEvent;
    function GetOnFinished : TOperationEvent;
    function GetRowsAffected : Integer;
    procedure SetRowsAffected(RA : Integer);
    function GetInterrupted : Boolean;
    procedure SetInterrupted(I : Boolean);
  public
    property InterruptInterval : Integer read GetInterruptInterval write FInterruptInterval;

    property OnStart : TOperationEvent read GetOnStart write FOnStart;
    property OnInterrupt : TOperationEvent read GetOnInterrupt write FOnInterrupt;
    property OnFinished : TOperationEvent read GetOnFinished write FOnFinished;

    procedure InitParams(Operation : TOperationType);
    function CheckInterrupt(OperationType : TOperationType) : Boolean;
    procedure AddRow(Operation : TOperationType);
    procedure Finished(Operation : TOperationType);

    property RowsAffected : Integer read GetRowsAffected write SetRowsAffected;
    property Interrupted : Boolean read GetInterrupted write SetInterrupted;

    constructor Create;
    destructor Destroy; override;
  end;

implementation

{ TInterruptable }

constructor TInterruptable.Create;
begin
  inherited Create;

  FInterruptInterval := 10;
  FOnInterrupt := nil;
  Interrupted := False;
end;

destructor TInterruptable.Destroy;
begin
  inherited Destroy;
end;

function TInterruptable.CheckInterrupt(OperationType : TOperationType) : Boolean;
begin
  if (Self <> nil) and
     (InterruptInterval > 0) and
     Assigned(Self.OnInterrupt) and
     (RowsAffected mod InterruptInterval = 0) then
    Self.OnInterrupt( Self, OperationType );

  Result := Interrupted;
end;

procedure TInterruptable.AddRow(Operation : TOperationType);
begin
  RowsAffected := RowsAffected + 1;
  Self.CheckInterrupt(Operation);
end;

procedure TInterruptable.InitParams(Operation : TOperationType);
begin
  RowsAffected := 0;

  if (Self <> nil) and Assigned(Self.OnStart) then
    Self.OnStart(Self, Operation);
end;

procedure TInterruptable.Finished(Operation : TOperationType);
begin
  if (Self <> nil) and Assigned(Self.OnFinished) then
    Self.OnFinished(Self, Operation);
end;

function TInterruptable.GetInterruptInterval : Integer;
begin
  if Self = nil then
    Result := 0
  else
    Result := FInterruptInterval;
end;

function TInterruptable.GetRowsAffected : Integer;
begin
  if Self = nil then
    Result := 0
  else
    Result := FRowsAffected;
end;

procedure TInterruptable.SetRowsAffected(RA : Integer);
begin
  if Self <> nil then
    FRowsAffected := RA;
end;

function TInterruptable.GetInterrupted : Boolean;
begin
  if Self = nil then
    Result := False
  else
    Result := FInterrupted;
end;

procedure TInterruptable.SetInterrupted(I : Boolean);
begin
  if Self <> nil then
    FInterrupted := I;
end;

function TInterruptable.GetOnStart : TOperationEvent;
begin
  if Self = nil then
    Result := nil
  else
    Result := FOnStart;
end;

function TInterruptable.GetOnInterrupt : TOperationEvent;
begin
  if Self = nil then
    Result := nil
  else
    Result := FOnInterrupt;
end;

function TInterruptable.GetOnFinished : TOperationEvent;
begin
  if Self = nil then
    Result := nil
  else
    Result := FOnFinished;
end;

end.
