{ $Id: CommonCalcFields.pas,v 1.88 2002/07/30 13:23:31 laa Exp $}

{
    This file is part of the TTranslator 

    TTranslator is a Delphi component for localizing String and TStrings 
    properties of components dropped on a form. You can also localize your 
    code strings with TTranslator.
    Copyright (C) 2002 Polycon Ab

    This is a licensed version of TTranslator, it may be used as described
    in the TTranslator license agreement. If you have not acquired a 
    commercial TTranslator license, your are using this product illegaly.    
}

{----------------------------------------------------------------------------
  CommonCalcFields   Commonly used calc fields

  What:              TConstantField
                     TSumField
                     TDifferenceField
                     TDifferencePctField
                     TRatioField
                     TSubTotalPercentageField

  Company:           Polycon Ab
  Authors:           VJN, LAA, MJM, MVJ
----------------------------------------------------------------------------}

unit CommonCalcFields;

interface

{$i common.inc}

uses
  RowList, Classes, SysUtils,
{$ifndef LINUX}
  Graphics,
{$else LINUX}
  QGraphics,
{$endif LINUX}
{IFNDEF TRANSLATOR}
  CommonLib,
{ENDIF TRANSLATOR}
  DataElements, DataType, DataTypes, CalcField;

type
  {/** Always returns a constant TValue  */}
  TConstantField = class(TCalcField)
  private
    FConstant : TValue;
  public
    {/** Constructor */}
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(const FieldName : String; Constant : TValue);
    constructor CreateNoName(Constant : TValue);

    {/** Destructor */}
    destructor Destroy; override;

    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;

    property Constant : TValue read FConstant write FConstant;
  end;

  {/** Abstract base class for all fields, consisting of two fields joined by
       an operator -- i.e. normal addition, multiplication etc. */}
  TBinaryField = class(TClosedField)
  private
    FFirstField, FOtherField : TDataField;
    lstDependentFields : TList;
  protected
    fDistributeAction : TDistributeAction;
    // inherited from TClosedField
    function GetFieldCount : Integer; override;
    function GetField(idx : Integer) : TDataField; override;
    property FirstField : TDataField read fFirstField;
    property OtherField : TDataField read fOtherField;
    property DistributeAction : TDistributeAction read fDistributeAction;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; FirstField, OtherField : TDataField; DataType : TDataType;
                       DistributeAction : TDistributeAction; ReadOnly,CalcBeforeAggregating : Boolean); virtual;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;  override;

    procedure AddDependentField(AField : TDataField); virtual;
    procedure AddDependentToList(lstFields : TList); virtual;
  end;

  TBinaryFieldClass = class of TBinaryField;

  {/** FieldsUserd as Headers in a HeaderRowView */}
  THeaderField = class(TConstantField)
  private
    FDataField : TDataField;
    FUseShortDescription : Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField : TDataField; Constant : TValue);

    {/** Pointer to the DataField that decides this fields description */}
    property DataField : TDataField read FDataField;
    property UseShortDescription : Boolean read FUseShortDescription write FUseShortDescription;
  end;

  TDerivedHeaderField = class(THeaderField)
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField : TDataField);

    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TCommonSumField = class(TClosedField)
  private
    FFields : TList;
  protected
    function GetFieldCount : Integer; override;
    function GetField(idx : Integer) : TDataField; override;
  public
    {/** AllocationShareField for aloocation table */}
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; FieldArray : array of TDataField);
    constructor CreateFromList(const FieldName : String; FieldList : TList; LowIndex, HighIndex : Integer; ReadOnly : Boolean);
    {/** Destructor */}
    destructor Destroy; override;

    property Fields : TList read fFields;
    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField; override;
  end;

  TDifferenceField = class(TBinaryField)
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField; override;
    // inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TBinarySumField = class(TBinaryField)
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField; override;
    // inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TDifferencePctField = class(TBinaryField)
  private
    fNegateResult : Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField; override;
    // inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;

    property NegateResult : Boolean read fNegateResult write fNegateResult;
  end;

  TRatioField = class(TBinaryField)
  private
    FDecimalCount : Integer;
    procedure SetDecimalCount(ACount : Integer);
  published
    property DecimalCount : Integer read FDecimalCount write SetDecimalCount default -1;
  public
//    constructor CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
//    function CreateCopy(FieldName : String; GetNewField : TQueryFieldFunction) : TClosedField; override;
    constructor Create(AOwner: TComponent); override;
    function DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult; override;
    // inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TPctField = class(TBinaryField)
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField; override;
    // inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult; override;
  end;

  {/** Multiply two fields, always calculated in memory. */}
  TCalcMultiplyField = class(TBinaryField)
  public
     //inherited from TCalcField
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  {/** Multiply two fields, and generate the needed SQL if needed. */}
  TMultiplyField = class(TCalcMultiplyField)
  protected
     //inherited from TCalcField
    function GetSQLString(FullFieldName : TQueryStringFunction; DoAggregate : Boolean) : String; override;
    function GetCanBeInDB : Boolean; override;
  public
    constructor CreateOld(FieldName : String; FirstField, OtherField : TDataField; DataType : TDataType;
                       DistributeAction : TDistributeAction; ReadOnly,CalcBeforeAggregating : Boolean); override;
  end;

  TAndField = class(TCalcField)
  private
    Fields : TList;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; BooleanFields : array of TDataField);
    constructor CreateFromList(FieldName : String; BooleanFields : TList);
    {/** Destructor */}
    destructor Destroy; override;

    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TOrField = class(TCalcField)
  private
    Fields : TList;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; BooleanFields : array of TDataField);
    constructor CreateFromList(FieldName : String; BooleanFields : TList);
    {/** Destructor */}
    destructor Destroy; override;

    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TNotField = class(TCalcField)
  private
    Field : TDataField;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; BooleanField : TDataField);

    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  {/** Decides wether a DataRow is modified according to DataRow.ContainsChanges */}
  TRowModifiedField = class(TCalcField)
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String);
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  {/** Field that generates "COUNT *" SQL for counting the amount of grouped rows
       in a select statement. */}
  TRowCountField = class(TCalcField)
  private
    fDataTable : TDataTable;
  protected
    function GetSQLString(FullFieldName : TQueryStringFunction; DoAggregate : Boolean) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String);
    property DataTable : TDataTable read fDataTable write fDataTable;
  end;

  {/** A field that shows a primary, unless the primary and the secondary fields
       are equal, when a defualt value is shown (normally blank). */}
  TDefaultIfEqualField = class(TCalcField)
  private
    fDefaultValue : TValue;
    fPrimaryField, fSecondaryField : TDataField;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(const FieldName : String; PrimaryField, SecondaryField : TDataField;
                          DataType:TDataType; ReadOnly:Boolean; DefaultValue:TValue);
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;
  end;

{ifndef WEBPROFFA}

{endif WEBPROFFA}

  {/** Check if a DataRow (not) has the given values */}
  THasValueField = class(TCalcField)
  private
    FDataField : TDataField;
    FValues : TStringList;
    FValuesLegal : Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; DataField : TDataField; Values : array of String; ValuesLegal : Boolean);
    destructor Destroy; override;
    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    procedure GetLegalValues(var Strings : TStrings); virtual;
  end;

  {/** Contains an erroemessage for the given DataField */}
  TErrorMessageField = class(TCalcField)
  private
    FDataField : TDataField;
  protected
    constructor CreateOld(FieldName : String; ADataField : TDataField);
    function GetErrorString(ARow : TAbstractRow) : String; virtual; abstract;
  public
    constructor Create(AOwner: TComponent); override;
    property DataField : TDataField read FDataField;
    property ErrorString[ARow : TAbstractRow] : String read GetErrorString;

    procedure GetAdditionalLegalValues(var Strings : TStrings); virtual; abstract;
    function IsRowValid( const FieldMsgObjectList : TStringList; ARow : TAbstractRow;
                         DisabledFieldList : TAbstractFieldList ) : Boolean; virtual;
  end;

  {/** Check wether a DataRow has legal values or not. Possile to get messages definig problem*/}
  TRowIsValidField = class(TClosedField)
  private
    FObjects : TList;
    function GetSourceField(idx : Integer) : TDataField;
    function GetConditionFieldForSource(AField : TDataField) : TErrorMessageField;

    property SourceField[idx : Integer] : TDataField read GetSourceField;
  protected
    function GetFieldCount : Integer; override;
    function GetField(idx : Integer) : TDataField; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ConditionObjects : array of TErrorMessageField);
    destructor Destroy; override;
    function IsRowValid(const FieldMsgObjectList : TStringList; ARow : TAbstractRow;
      DisabledFieldList : TAbstractFieldList; BreakOnFalse : Boolean) : Boolean; virtual;
    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;  override;

    procedure GetAdditionalLegalValues(AField : TDataField; var Strings : TStrings); virtual;
    property ConditionFieldForSource[AField : TDataField]  : TErrorMessageField read GetConditionFieldForSource;
  end;

  {/** Check if the value for the defined DataField is defined in its AuxTable */}
  THasValueDefinedInTableField = class(TErrorMessageField)
  protected
    function GetErrorString(ARow : TAbstractRow) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField : TDataField);
    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    procedure GetAdditionalLegalValues(var Strings : TStrings); override;
  end;

  {/** Returns true if the Condition holds */}
  TSimpleValueLegalField = class(TErrorMessageField)
  private
    FConditionField : TDataField;
    FErrMsg : String;
  protected
    function GetErrorString(ARow : TAbstractRow) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField, ConditionField : TDataField; ErrorMsg : String);

    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    procedure GetAdditionalLegalValues(var Strings : TStrings); override;
    property ConditionField : TDataField read FConditionField;
  end;

  {/** Check if either of the boolean value for two fields is legal */}
  TValueLegalField = class(TErrorMessageField)
  private
    FConditionField1 : TDataField;
    FConditionField2 : TDataField;
    FErrCond1True : String;
    FErrCond1False : String;
    FTrueForBothEqual : Boolean;
  protected
    function GetErrorString(ARow : TAbstractRow) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField, ConditionField1, ConditionField2 : TDataField;
         TrueForBothEqual : Boolean; ErrorMsgIfCond1True, ErrorMsgIfCond2True : String);

    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;

    property ConditionField1 : TDataField read FConditionField1;
    property ConditionField2 : TDataField read FConditionField2;
    property TrueForBothEqual : Boolean read FTrueForBothEqual;
    procedure GetAdditionalLegalValues(var Strings : TStrings); override;
  end;

  TValueLegalORField = class(TErrorMessageField)
  private
    FConditionField1 : TDataField;
    FConditionField2 : TDataField;
    FFalseForBothFalse : Boolean;
    FErrStr : String;
  protected
    function GetErrorString(ARow : TAbstractRow) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField, ConditionField1, ConditionField2 : TDataField;
         FalseForBothFalse : Boolean; ErrorMsg : String);

    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;

    property ConditionField1 : TDataField read FConditionField1;
    property ConditionField2 : TDataField read FConditionField2;
    property FalseForBothFalse : Boolean read FFalseForBothFalse;
    procedure GetAdditionalLegalValues(var Strings : TStrings); override;
  end;

  TValueLegalANDField = class(TErrorMessageField)
  private
    FConditionField1 : TDataField;
    FConditionField2 : TDataField;
    FTrueForBothTrue : Boolean;
    FErrStr : String;
  protected
    function GetErrorString(ARow : TAbstractRow) : String; override;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String; ADataField, ConditionField1, ConditionField2 : TDataField;
         TrueForBothTrue : Boolean; ErrorMsg : String);

    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    procedure GetAdditionalLegalValues(var Strings: TStrings); override;

    property ConditionField1 : TDataField read FConditionField1;
    property ConditionField2 : TDataField read FConditionField2;
    property TrueForBothTrue : Boolean read FTrueForBothTrue;
  end;

  {/** Shows FirstField if ConditionField returns true (false), otherwise shows SecondField */}
  TConditionalShowField = class(TCalcField)
  private
    FFirstValueField : TDataField;
    FSecondValueField : TDataField;
    FConditionField : TDataField;
    FShowFirstForTrue : Boolean;
  protected
    property FirstValueField : TDataField read FFirstValueField;
    property SecondValueField : TDataField read FSecondValueField;
    property ConditionField : TDataField read FConditionField;
    property ShowFirstForTrue : Boolean read FShowFirstForTrue;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(FieldName : String ; DataType : TDataType; ReadOnly,
                       CalcBeforeAggregating : Boolean; FirstValueField, SecondValueField,
                       ConditionField : TDataField; ShowFirstForTrue : Boolean);

    destructor Destroy; override;
    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult; override;
  end;

  TEditAuxTableField = class(TKeyField)
  protected
    function GetReadOnly(ARow : TAbstractRow) : Boolean; override;
    function GetLookupRow(SrcRow : TAbstractRow) : TDataRow;
    function RowHasKeys(ARow : TAbstractRow) : Boolean;
    function ValueLegal(SrcRow : TAbstractRow; Value : TValue) : Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateOld(AFieldName : String; ASrcField : TKeyField);
    destructor Destroy; override;


//    function CalcValue(ARow : TAbstractRow) : TValue; override;
    function GetExternValue(ARow : TAbstractRow) : TValue; override;
    function SetExternValue(ARow : TAbstractRow; Value : TValue; Action : TSetAction) : TSetResult; override;
    procedure GetValues(Results : TStrings; Table : TDataTable; Row : TAbstractRow; Condition : TCondition); override;
//    function DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult; override;
  end;

  TPictureField = class(TCalcField)
  private
    FPicture : TPicture;
  protected
    property Picture : TPicture read FPicture;
  public
    {/** Constructor */}
    constructor CreateOld(FieldName : String; APicture : TPicture);
    {/** Destructor */}
    destructor Destroy; override;
    {/** See @TCalcField */}
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  {/** A field that always show the ratio of the previous subtotallevel. E.g
       in the row hierarchy

                      DataField    TSubTotalPercentageField
       ====================================================
           ROW1              45                         75%  (45/60)
           ROW2              15                         25%  (15/60)
         SUBTOTAL12          60                         60%  (60/100)
           ROW3              40                        100%  (40/40)
         SUTOTAL2            40                         40%  (40/100)
       GRANDTOTAL           100                        100%  (Total)
  */}
  TSubTotalPercentageField = class(TClosedField)
  private
    fDataField : TDataField;
  protected
    function GetFieldCount : Integer; override;
    function GetField(idx : Integer) : TDataField; override;
  public
    constructor CreateOld(const FieldName : String; DataField : TDataField);
    function CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;  override;

    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

  TBooleanTextField = class(TCalcField)
  private
    FOwner : TDataField;
  public
    constructor CreateOld(const FieldName : String; Owner : TDataField);
    destructor Destroy; override;
    function CalcValue(ARow : TAbstractRow) : TValue; override;
  end;

implementation

uses

  Math,
  Criteria, DerivedDataType, DataLib;

const
  MSGE_ConstValueCantBeConvertedToType = 'The constant value can''t be converted to type';
  MSGE_IndexOutOfBounds = 'Index out of bounds!';
  MSGI_ValueFor = 'The value for';
  MSGI_NotDefinedInTable = 'is not defined in table!';

{ TBooleanTextField }

constructor TBooleanTextField.CreateOld(const FieldName : String; Owner : TDataField);
begin
  inherited CreateOld(FieldName, StringType(5, False), True, True);
  FOwner := Owner;
  FOwner.SetAllTextFields([Self]);
  FOwner.DisplayValues := dvTextOnly;
end;

destructor TBooleanTextField.Destroy;
begin
  inherited Destroy;
end;

function TBooleanTextField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if ARow.BooleanValue[FOwner] then
    Result := ValueFromString('True')
  else
    Result := ValueFromString('False');
end;

// -------------------------------- TConstantField ---------------------------------------

constructor TConstantField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FConstant := ZeroVal;
  IsAggregable := True;
end;

constructor TConstantField.CreateOld(const FieldName : String; Constant : TValue);
begin
  inherited CreateOld(FieldName, Constant.DataType, true, false);
  FConstant := Constant;
  IsAggregable := True;
end;

constructor TConstantField.CreateNoName(Constant : TValue);
begin
  CreateOld('', Constant);
end;

destructor TConstantField.Destroy;
begin
  inherited Destroy;
end;

function TConstantField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := FConstant;
end;

function TConstantField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  result := srReadOnly;
end;

{-----------------------TBinaryField-------------------------------------------}

constructor TBinaryField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  fFirstField := nil;
  fOtherField := nil;
  fDistributeAction := daFirst;
  lstDependentFields := nil;
end;

constructor TBinaryField.CreateOld(FieldName : String; FirstField, OtherField : TDataField; DataType : TDataType;
                                DistributeAction : TDistributeAction; ReadOnly, CalcBeforeAggregating : Boolean);
begin
  inherited CreateOld(FieldName, DataType, (* round(MaxValue([FirstField.FieldSize,OtherField.FieldSize])),*) ReadOnly, CalcBeforeAggregating);
  fFirstField := FirstField;
  fOtherField := OtherField;
  fDistributeAction := DistributeAction;
  lstDependentFields := nil;
end;

function TBinaryField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
var
  FieldClass : TBinaryFieldClass;
  iField : integer;
  ACopy : TBinaryField;
begin
  FieldClass := TBinaryFieldClass(Self.ClassType);
  ACopy := FieldClass.CreateOld(FieldName, GetNewField(FirstField, OwnedList), GetNewField(OtherField, OwnedList), DataType,
                                fDistributeAction, DefaultReadOnly, CalcBeforeAggregating);
  if lstDependentFields<>nil then
    for iField := 0 to lstDependentFields.Count-1 do
      ACopy.AddDependentField( TDataField(lstDependentFields[iField]) );

  Result := ACopy;
  OwnedList.Add( Result );
end;

function TBinaryField.GetFieldCount : Integer;
begin
  result := 2;
end;

function TBinaryField.GetField(idx : Integer) : TDataField;
begin
  case idx of
    0: result := FirstField;
    1: result := OtherField;
    else raise Exception.Create(Self.ClassName + '.GetField: Index out of bounds');
  end;
end;

procedure TBinaryField.AddDependentField(AField : TDataField);
begin
  if lstDependentFields=nil then
    lstDependentFields := TList.Create;
  lstDependentFields.Add( AField );
end;

procedure TBinaryField.AddDependentToList(lstFields : TList);
var
  iField : integer;
begin
  if lstDependentFields<>nil then
    for iField := 0 to lstDependentFields.Count-1 do
      lstFields.Add( lstDependentFields[iField] );
end;

// ---------------------------------- THeaderField -------------------------------

constructor THeaderField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FDataField := nil;
  FUseShortDescription := False;
end;

constructor THeaderField.CreateOld(FieldName : String; ADataField : TDataField;
            Constant : TValue);
begin
  inherited CreateOld(FieldName, Constant);

  FDataField := ADataField;
  FUseShortDescription := False;
end;

// ---------------------------------- TDerivedHeaderField -------------------------------

constructor TDerivedHeaderField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor TDerivedHeaderField.CreateOld(FieldName : String; ADataField : TDataField);
begin
  Assert(ADataField <> nil);
  Inherited CreateOld(FieldName, ADataField, ValueFromString(''));
end;

function TDerivedHeaderField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if FUseShortDescription then
    result := ValueFromString(FDataField.ShortDescription)
  else
    result := ValueFromString(FDataField.LongDescription);
end;

{---------------------- TSumField -------------------------------------------}

constructor TCommonSumField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFields := TList.Create;
end;

constructor TCommonSumField.CreateOld(FieldName : String; FieldArray : array of TDataField);
var
  i : Integer;
begin
  inherited CreateOld(FieldName, FieldArray[Low(FieldArray)].DataType, False, True);

  FFields := TList.Create;
  for i := Low(FieldArray) to High(FieldArray) do
    FFields.Add(FieldArray[i]);
end;

function TCommonSumField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
var
  NewFields : TList;
  i : Integer;
begin
  NewFields := TList.Create;
  for i := 0 to Self.FieldCount - 1 do
    NewFields.Add( GetNewField(Self.Field[i],OwnedList) );
  Result := TCommonSumField.CreateFromList(FieldName, NewFields, 0, NewFields.Count - 1, DefaultReadOnly);
  OwnedList.Add( Result );
  NewFields.Free;
end;

constructor TCommonSumField.CreateFromList(const FieldName : String; FieldList : TList; LowIndex,
            HighIndex : Integer; ReadOnly : Boolean);
var
  i : Integer;
  ADataType : TDataType;
begin
  if FieldList.Count>0 then
    ADataType := TDataField(FieldList.Items[LowIndex]).DataType
  else
    ADataType := AnyStringType;
  inherited CreateOld(FieldName, ADataType, ReadOnly, True);

  FFields := TList.Create;
  for i := LowIndex to HighIndex do
    FFields.Add(FieldList.Items[i]);
end;

destructor TCommonSumField.Destroy;
begin
  FFields.Free;
  inherited Destroy;
end;

function TCommonSumField.GetFieldCount : Integer;
begin
  Result := FFields.Count;
end;

function TCommonSumField.GetField(idx : Integer) : TDataField;
begin
  Assert(idx<FFields.Count, 'TSumField.GetField: Index out of bounds');
  Result := TDataField(FFields.Items[idx]);
end;

function TCommonSumField.CalcValue(ARow : TAbstractRow) : TValue;
var
  i : Integer;
begin
  Result := DataType.DefaultValue;
  for i := 0 to FieldCount - 1 do
    Result := DataType.Sum(Result, ARow.Value[Field[i]]);
end;

{ --- implementation of TDifferenceField --- }

constructor TDifferenceField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor TDifferenceField.CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
begin
  inherited CreateOld(FieldName, FirstField, OtherField, FirstField.DataType, daNone, ReadOnly, True);
end;

function TDifferenceField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  Result := TDifferenceField.CreateDefault(FieldName, GetNewField(fFirstField, OwnedList), GetNewField(fOtherField, OwnedList), DefaultReadOnly);
  OwnedList.Add( Result );
end;

function TDifferenceField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  try
    result := DataType.Difference(ARow.Value[fFirstField], ARow.Value[fOtherField]);
  except
    Result := DataType.DefaultValue;;
  end;
end;

function TDifferenceField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  ARow.Value[fFirstField] := DataType.Sum(ARow.Value[fOtherField], Value);
  Result := srOK;
end;

{----------------- TBinarySumField -------------------------------------------}

constructor TBinarySumField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor TBinarySumField.CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
begin
  inherited CreateOld(FieldName, FirstField, OtherField, FirstField.DataType, daNone, ReadOnly, True);
end;

function TBinarySumField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  Result := TBinarySumField.CreateDefault(FieldName, GetNewField(fFirstField, OwnedList), GetNewField(fOtherField, OwnedList), DefaultReadOnly);
  OwnedList.Add( Result );
end;

function TBinarySumField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  try
    result := DataType.Sum(ARow.Value[fFirstField], ARow.Value[fOtherField]);
  except
    Result := DataType.DefaultValue;;
  end;
end;

function TBinarySumField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  ARow.Value[fFirstField] := DataType.Difference(Value, ARow.Value[fOtherField]);
  Result := srOK;
end;

{ ---implementation of TDifferencePctField ---}

constructor TDifferencePctField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fNegateResult := False;
  IsAggregable := True;
end;

constructor TDifferencePctField.CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
begin
  inherited CreateOld(FieldName, FirstField, OtherField, PercentType, daSecond, ReadOnly, false);
  fNegateResult := False;
  IsAggregable := True;
end;

function TDifferencePctField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  Result := TDifferencePctField.CreateDefault(FieldName, GetNewField(fFirstField, OwnedList), GetNewField(fOtherField, OwnedList), DefaultReadOnly);
  OwnedList.Add( Result );
end;

function TDifferencePctField.CalcValue(ARow : TAbstractRow) : TValue;
var
  Diff, OtherFieldValue, FirstFieldValue : TValue;
begin
  OtherFieldValue := ARow.Value[fOtherField];
  if fOtherField.DataType.Equals(ZeroVal, OtherFieldValue) then
    SetValueUndefined( Result )
  else
  begin
    FirstFieldValue := ARow.Value[fFirstField];
    if fNegateResult then
      Diff := OtherFieldValue.DataType.Difference(OtherFieldValue, FirstFieldValue)
    else
      Diff := OtherFieldValue.DataType.Difference(FirstFieldValue, OtherFieldValue);
//    Diff := OtherFieldValue.DataType.Product(Diff, ValueFromInteger(100));
    result := DataType.Quota(Diff, OtherFieldValue);
  end;
end;

function TDifferencePctField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  ARow.Value[fFirstField] := DataType.Sum(ARow.Value[fOtherField], Value);
  Result := srOK;
end;

{ --- implementation of TRatioField --- }

{constructor TRatioField.CreateDefault(FieldName : String; FirstField, OtherField : TDataField; DataType:TDataType; ReadOnly : Boolean);
begin
  inherited Create(FieldName, FirstField, OtherField, DataType, daSecond, ReadOnly, false);
end;

function TRatioField.CreateCopy(FieldName : String; GetNewField : TQueryFieldFunction) : TClosedField;
begin
  Result := TRatioField.CreateDefault(FieldName, GetNewField(fFirstField), GetNewField(fOtherField), Self.DataType, ReadOnly);
end; }

constructor TRatioField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDecimalCount := -1;
end;

procedure TRatioField.SetDecimalCount(ACount : Integer);
begin
  if ACount < -1 then
    raise Exception.Create('DecimalCount must be greater or equal to "-1"!')
  else
    FDecimalCount := ACount;
end;

function TRatioField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  try
    if DataType.Equals(ZeroVal, ARow.Value[fOtherField]) then
      Result := DataType.DefaultValue
    else
    begin
      Result := DataType.Quota(ARow.Value[fFirstField], ARow.Value[fOtherField]);
      if ( FDecimalCount > 0) then
        Result :=  ValueFromCurrency(CDecimalRound(AsCurrency(Result), Round(Power(10, DecimalCount))))
      else if (FDecimalCount = 0) then
        Result :=  ValueFromCurrency(Round(AsCurrency(Result)));
    end;
  except
    Result := DataType.DefaultValue;
  end;
end;

function TRatioField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  ARow.Value[fFirstField] := DataType.Product(ARow.Value[fOtherField], Value);
  Result := srOK;
end;

{------------------------TPctField-------------------------------------------}

constructor TPctField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  IsAggregable := True;
end;

constructor TPctField.CreateDefault(FieldName : String; FirstField, OtherField : TDataField; ReadOnly : Boolean);
begin
  inherited CreateOld(FieldName, FirstField, OtherField, PercentType, daSecond, ReadOnly, false);
  IsAggregable := True;
end;

function TPctField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  Result := TPctField.CreateDefault(FieldName, GetNewField(fFirstField, OwnedList), GetNewField(fOtherField, OwnedList), DefaultReadOnly);
  OwnedList.Add( Result );
end;

function TPctField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if fOtherField.DataType.Equals(ZeroVal, ARow.Value[fOtherField]) then
    SetValueUndefined( Result )
  else
//    Result := DataType.Product(ValueFromInteger(100), DataType.Quota(ARow.Value[fFirstField], ARow.Value[fOtherField]));
    Result := DataType.Quota(ARow.Value[fFirstField], ARow.Value[fOtherField]);
end;

function TPctField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  if fOtherField.DataType.Equals(ZeroVal, ARow.Value[fOtherField]) then
    ARow.Value[fFirstField] := DataType.DefaultValue
  else
    ARow.Value[fFirstField] := DataType.Quota( DataType.Product(Value,ARow.Value[fOtherField]) ,ValueFromInteger(100));
  result := srOK;

(*
  ARow.Value[fFirstField] := DataType.Quota( DataType.Product(Value,ARow.Value[fOtherField]) ,ValueFromInteger(100));
  Result := srOk;
*)
end;

{ TCalcMultiplyField }

function TCalcMultiplyField.CalcValue(ARow: TAbstractRow): TValue;
begin
  result := DataType.Product(ARow[OtherField], ARow[FirstField]);
end;

{ TMultiplyField }

constructor TMultiplyField.CreateOld(FieldName: String; FirstField,
  OtherField: TDataField; DataType: TDataType;
  DistributeAction: TDistributeAction; ReadOnly,
  CalcBeforeAggregating: Boolean);
begin
  inherited;
  CanGenerateValueSQL := TRue;
  CanGenerateCondSQL := False;
end;

function TMultiplyField.GetCanBeInDB: Boolean;
begin
  Result := True;
end;

function TMultiplyField.GetSQLString(FullFieldName: TQueryStringFunction;
  DoAggregate: Boolean): String;
  function AddField(AField : TDataField) : string;
  var
    iField : integer;
  begin
    if AField is TClosedField then
      for iField := 0 to TClosedField(AField).FieldCount -1 do
      begin
        if iField > 0 then result := result + '+';
        result := result + FullFieldName(TClosedField(AField).Field[iField])
      end
    else
      Result := FullFieldName(AField);
  end;
begin
  Result := ' (' + AddField(FirstField) + ')*(' + AddField(OtherField) + ') ';
  if DoAggregate then
    Result := 'Sum(' + Result + ')';

end;

{---------------------- TAndField -------------------------------------------}

constructor TAndField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Fields := TList.Create;
end;

constructor TAndField.CreateOld(FieldName : String; BooleanFields : array of TDataField);
var
  i : Integer;
begin
  inherited CreateOld(FieldName, BooleanType, True, True);

  Fields := TList.Create;
  for i := Low(BooleanFields) to High(BooleanFields) do
    Fields.Add(BooleanFields[i]);
end;

constructor TAndField.CreateFromList(FieldName : String; BooleanFields : TList);
var
  i : Integer;
begin
  inherited CreateOld(FieldName, BooleanType, True, True);
  Fields := TList.Create;
  for i := 0 to BooleanFields.Count - 1 do
    Fields.Add(BooleanFields.Items[i]);
end;

destructor TAndField.Destroy;
begin
  Fields.Free;
  inherited Destroy;
end;

function TAndField.CalcValue(ARow : TAbstractRow) : TValue;
var
  i : Integer;
begin
  if Fields.Count = 0 then
    Result := ValueFromBoolean(False)
  else
  begin
    Result := ValueFromBoolean(True);

    for i := 0 to Fields.Count - 1 do
      Result := DataType.AndOperator(Result, ARow.Value[TDataField(Fields[i])]);
  end;
end;

{---------------------- TOrField -------------------------------------------}

constructor TOrField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Fields := TList.Create;
end;

constructor TOrField.CreateOld(FieldName : String; BooleanFields : array of TDataField);
var
  i : Integer;
begin
  inherited CreateOld(FieldName, BooleanType, True, True);

  Fields := TList.Create;
  for i := Low(BooleanFields) to High(BooleanFields) do
    Fields.Add(BooleanFields[i]);
end;

constructor TOrField.CreateFromList(FieldName : String; BooleanFields : TList);
var
  i : Integer;
begin
  inherited CreateOld(FieldName, BooleanType, True, True);
  Fields := TList.Create;
  for i := 0 to BooleanFields.Count do
    Fields.Add(BooleanFields.Items[i]);
end;

destructor TOrField.Destroy;
begin
  Fields.Free;
  inherited Destroy;
end;

function TOrField.CalcValue(ARow : TAbstractRow) : TValue;
var
  i : Integer;
begin
  Result := ValueFromBoolean(False);

  if Fields.Count > 0 then
    for i := 0 to Fields.Count - 1 do
      Result := DataType.OrOperator(Result, ARow.Value[TDataField(Fields[i])]);
end;

{---------------------- TNotField -------------------------------------------}

constructor TNotField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Field := nil;
end;

constructor TNotField.CreateOld(FieldName : String; BooleanField : TDataField);
begin
  inherited CreateOld(FieldName, BooleanType, True, True);
  Field := BooleanField;
end;

function TNotField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := DataType.NegateValue(ARow.Value[Field]);
end;

{---------------------- TRowModifiedField -----------------------------------}

constructor TRowModifiedField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor TRowModifiedField.CreateOld(FieldName : String);
begin
  Inherited CreateOld(FieldName, BooleanType, True, False);
end;

function TRowModifiedField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := ValueFromBoolean(ARow.ContainsChanges);
end;

//-------------------------------TRowCountField---------------------------------

constructor TRowCountField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Self.CanGenerateValueSQL := True;
  Self.CanGenerateCondSQL := False;
end;

constructor TRowCountField.CreateOld(FieldName : String);
begin
  Inherited CreateOld(FieldName, IntegerType, True, True);
  Self.CanGenerateValueSQL := True;
  Self.CanGenerateCondSQL := False;
end;

function TRowCountField.GetSQLString(FullFieldName : TQueryStringFunction; DoAggregate : Boolean) : String;
begin
  if DoAggregate then
    Result := 'Count(*)'
  else
    Result := '1';
end;

//----------------------------TDefaultIfEqualField------------------------------

constructor TDefaultIfEqualField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fDefaultValue := ZeroVal;
  fPrimaryField := nil;
  fSecondaryField := nil;
end;

constructor TDefaultIfEqualField.CreateOld(const FieldName : String; PrimaryField, SecondaryField : TDataField;
                     DataType:TDataType; ReadOnly:Boolean; DefaultValue:TValue);
begin
  inherited CreateOld(FieldName, DataType, ReadOnly, False);
  fDefaultValue := DefaultValue;
  fPrimaryField := PrimaryField;
  fSecondaryField := SecondaryField;
end;

function TDefaultIfEqualField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if DataType.Equals( ARow[fPrimaryField], ARow[fSecondaryField]) then
    result := fDefaultValue
  else
    result := ARow[ fPrimaryField ];
end;

function TDefaultIfEqualField.DistributeValue(ARow : TabstractRow; Value : TValue) : TSetResult;
begin
  ARow[ fPrimaryField ] := Value;
  result := srOK;
end;

//---------------------TConstantSQLField----------------------------------------

{ifndef WEBPROFFA}

{endif WEBPROFFA}

// -------------------------------- THasValueField ---------------------------------------

constructor THasValueField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FDataField := nil;
  FValuesLegal := True;
  FValues := TStringList.Create;
  FValues.Sorted := True;
  FValues.Duplicates := dupIgnore;
end;

constructor THasValueField.CreateOld(FieldName : String; DataField : TDataField; Values : array of String; ValuesLegal : Boolean);
var
  idx : Integer;
begin
  FValues := nil;
  inherited CreateOld(FieldName, BooleanType, True, False);
  FDataField := DataField;
  FValuesLegal := ValuesLegal;
  if FValues=nil then
  begin
    FValues := TStringList.Create;
    FValues.Sorted := True;
    FValues.Duplicates := dupIgnore;
  end;

  for idx := Low(Values) to High(Values) do
    FValues.Add(Values[idx]);
end;

destructor THasValueField.Destroy;
begin
  FValues.Free;
  inherited Destroy;
end;

function THasValueField.CalcValue(ARow : TAbstractRow) : TValue;
var
  idx : Integer;
begin
  Result := ValueFromBoolean(FValuesLegal = FValues.Find(ARow.StringValue[FDataField], idx));
end;

procedure THasValueField.GetLegalValues(var Strings: TStrings);
var
  iItem : Integer;
begin
  if FValuesLegal then
    for iItem := 0 to FValues.Count -1 do
      Strings.Add(FValues[iItem]);
end;

// -------------------------------- TErrorMessageField ---------------------------------------

constructor TErrorMessageField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FDataField := nil;
end;

constructor TErrorMessageField.CreateOld(FieldName : String; ADataField : TDataField);
begin
  Inherited CreateOld(FieldName, BooleanType, True, False);

  FDataField := ADataField;
end;

function TErrorMessageField.IsRowValid(const FieldMsgObjectList : TStringList; ARow: TAbstractRow;
  DisabledFieldList: TAbstractFieldList): Boolean;
begin
  Result := ARow.BooleanValue[Self];

  if not Result and (FieldMsgObjectList <> nil) then
    FieldMsgObjectList.Add(ErrorString[ARow]);
end;

// -------------------------------- TRowIsValidField ---------------------------------------

constructor TRowIsValidField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FObjects := TList.Create;
end;

constructor TRowIsValidField.CreateOld(FieldName : String; ConditionObjects : array of TErrorMessageField);
var
  idx : Integer;
begin
  FObjects := nil;
  inherited CreateOld(FieldName, BooleanType, True, True);
  if FObjects=nil then
   FObjects := TList.Create;

  if (TObject(ConditionObjects[Low(ConditionObjects)]) <> nil) then
    for idx := Low(ConditionObjects) to High(ConditionObjects) do
      FObjects.Add(ConditionObjects[idx]);
end;

function TRowIsValidField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  raise Exception.Create('TRowIsValidField.CreateCopy: Method not implemented!');
end;

destructor TRowIsValidField.Destroy;
begin
  FObjects.Free;
  inherited Destroy;
end;

function TRowIsValidField.GetFieldCount : Integer;
begin
  Result := FObjects.Count;
end;

function TRowIsValidField.GetField(idx : Integer) : TDataField;
begin
  Assert(idx < FieldCount, Self.ClassName + '.GetField: ' + MSGE_IndexOutOfBounds
          + ' ' + IntToStr(idx) + ' >= ' + IntToStr(FieldCount));
  Result := TErrorMessageField(FObjects[idx]);
end;

function TRowIsValidField.GetSourceField(idx : Integer) : TDataField;
begin
  Assert(idx < FieldCount, Self.ClassName + '.GetSourceField: ' + MSGE_IndexOutOfBounds
          + ' ' + IntToStr(idx) + ' >= ' + IntToStr(FieldCount));
  Result := TErrorMessageField(FObjects[idx]).DataField;
end;

function TRowIsValidField.GetConditionFieldForSource(AField : TDataField) : TErrorMessageField;
var
  idx : Integer;
begin
  Result := nil;
  for idx := 0 to FieldCount -1 do
    if SourceField[idx] = AField then
      Result := TErrorMessageField( GetField(idx) );
end;

function TRowIsValidField.IsRowValid(const FieldMsgObjectList : TStringList;
  ARow : TAbstractRow; DisabledFieldList : TAbstractFieldList; BreakOnFalse : Boolean) : Boolean;

  function CheckFieldHasLegalValue(AField : TErrorMessageField) : Boolean;
  begin
    Result := AField.IsRowValid( FieldMsgObjectList, ARow, DisabledFieldList );
  end;

var
  idx : Integer;
  FieldIsValid : Boolean;
  aField : TDataField;
  aConditionField : TErrorMessageField;
  ATable : TDataTable;
  AKeyList : TList;
begin
  Result := True;
  if FieldMsgObjectList <> nil then
    FieldMsgObjectList.Clear;
  ATable := ARow.DataTable;
  AKeyList := TList.Create;
  ATable.KeysToList( AKeyList );

  for idx := 0 to FieldCount -1 do
  begin
    aConditionField := TErrorMessageField( GetField(idx) );
    aField := aConditionField.DataField;
    AKeyList.Remove( AField );
    if (DisabledFieldList <> nil) and
       DisabledFieldList.ContainsField(aField) then
      Continue;

    FieldIsValid := CheckFieldHasLegalValue(aConditionField);
    Result := Result and FieldIsValid;
    if not Result and BreakOnFalse then
      Break;
  end;

  if Result or not BreakOnFalse then
    for idx := 0 to AKeyList.Count -1 do
    begin
      AField := TDataField( AKeyList[idx] );
      if (DisabledFieldList <> nil) and
         DisabledFieldList.ContainsField(aField) then
        Continue;
      if aField.HasAuxTable then
        FieldIsValid := ( AField.AuxTable.Cache.LocateByRowValues(ARow, [aField]) <> nil )
      else
        FieldIsValid := True;

      if not FieldIsValid and (FieldMsgObjectList <> nil) then
        FieldMsgObjectList.Add('Illegal value for field ' + aField.FieldName + '!');

      Result := Result and FieldIsValid;
      if not Result and BreakOnFalse then
        Break;
    end;
    
  AKeyList.Free;
end;

function TRowIsValidField.CalcValue(ARow : TAbstractRow) : TValue;
var
  dummy : TStringList;
begin
  dummy := nil;
  Result := ValueFromBoolean(IsRowValid(dummy, ARow, nil, True));
end;

procedure TRowIsValidField.GetAdditionalLegalValues(AField : TDataField; var Strings : TStrings);
var
  ACondField : TErrorMessageField;
begin
  ACondField := ConditionFieldForSource[AField];
  if ACondField <> nil then
    ACondField.GetAdditionalLegalValues(Strings);
end;

// -------------------------------- TSimpleValueLegalField ---------------------

constructor TSimpleValueLegalField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FConditionField := nil;
  FErrMsg := '';
end;

constructor TSimpleValueLegalField.CreateOld(FieldName : String; ADataField,
                    ConditionField : TDataField; ErrorMsg : String);
begin
  Inherited CreateOld(FieldName, ADataField);

  FConditionField := ConditionField;
  FErrMsg := ErrorMsg;
end;

function TSimpleValueLegalField.GetErrorString(ARow : TAbstractRow) : String;
begin
  Result := FErrMsg;
end;

function TSimpleValueLegalField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := ARow[ConditionField];
end;

procedure TSimpleValueLegalField.GetAdditionalLegalValues(var Strings: TStrings);

  procedure GetLegalValues(AField : THasValueField);
  begin
    THasValueField(AField).GetLegalValues(Strings);
  end;

begin
  if ConditionField is THasValueField then
    GetLegalValues(THasValueField(ConditionField));
end;

// -------------------------------- TValueLegalField ---------------------------------------

constructor TValueLegalField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FConditionField1 := nil;
  FConditionField2 := nil;
  FErrCond1True := '';
  FErrCond1False := '';
  FTrueForBothEqual := True;
end;

constructor TValueLegalField.CreateOld(FieldName : String; ADataField, ConditionField1,
               ConditionField2 : TDataField; TrueForBothEqual : Boolean; ErrorMsgIfCond1True, ErrorMsgIfCond2True : String);
begin
  inherited CreateOld(FieldName, ADataField);

  FConditionField1 := ConditionField1;
  FConditionField2 := ConditionField2;
  FErrCond1True := ErrorMsgIfCond1True;
  FErrCond1False := ErrorMsgIfCond2True;
  FTrueForBothEqual := TrueForBothEqual;
end;

function TValueLegalField.GetErrorString(ARow : TAbstractRow) : String;
begin
  Result := FErrCond1False;
  if ARow.BooleanValue[FConditionField1] then
    Result := FErrCond1True;
end;

function TValueLegalField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := ValueFromBoolean(FTrueForBothEqual = (ARow.BooleanValue[FConditionField1] = ARow.BooleanValue[FConditionField2]));
end;

procedure TValueLegalField.GetAdditionalLegalValues(var Strings: TStrings);

  procedure GetLegalValues(AField : THasValueField);
  begin
    THasValueField(AField).GetLegalValues(Strings);
  end;

begin
  if ConditionField1 is THasValueField then
    GetLegalValues(THasValueField(ConditionField1));
  if ConditionField2 is THasValueField then
    GetLegalValues(THasValueField(ConditionField2));
end;

// -------------------------- TValueLegalORField -------------------------------

constructor TValueLegalORField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FConditionField1 := nil;
  FConditionField2 := nil;
  FErrStr := '';
  FFalseForBothFalse := True;
end;

constructor TValueLegalORField.CreateOld(FieldName : String; ADataField, ConditionField1, ConditionField2 : TDataField;
     FalseForBothFalse : Boolean; ErrorMsg : String);
begin
  inherited CreateOld(FieldName, ADataField);

  FConditionField1 := ConditionField1;
  FConditionField2 := ConditionField2;
  FErrStr := ErrorMsg;
  FFalseForBothFalse := FalseForBothFalse;
end;

function TValueLegalORField.GetErrorString(ARow : TAbstractRow) : String;
begin
  Result := FErrStr;
end;

function TValueLegalORField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if FalseForBothFalse then
    Result := ValueFromBoolean(ARow.BooleanValue[FConditionField1] or ARow.BooleanValue[FConditionField2])
  else
    Result := ValueFromBoolean(not ARow.BooleanValue[FConditionField1] or not ARow.BooleanValue[FConditionField2]);
end;

procedure TValueLegalORField.GetAdditionalLegalValues(var Strings: TStrings);

  procedure GetLegalValues(AField : THasValueField);
  begin
    AField.GetLegalValues(Strings);
  end;

begin
  if ConditionField1 is THasValueField then
    GetLegalValues(THasValueField(ConditionField1));
  if ConditionField2 is THasValueField then
    GetLegalValues(THasValueField(ConditionField2));
end;

// -------------------------- TValueLegalORField -------------------------------

constructor TValueLegalANDField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FConditionField1 := nil;
  FConditionField2 := nil;
  FErrStr := '';
  FTrueForBothTrue := True;
end;

constructor TValueLegalANDField.CreateOld(FieldName : String; ADataField, ConditionField1, ConditionField2 : TDataField;
         TrueForBothTrue : Boolean; ErrorMsg : String);
begin
  inherited CreateOld(FieldName, ADataField);

  FConditionField1 := ConditionField1;
  FConditionField2 := ConditionField2;
  FErrStr := ErrorMsg;
  FTrueForBothTrue := TrueForBothTrue;
end;

function TValueLegalANDField.GetErrorString(ARow : TAbstractRow) : String;
begin
  Result := FErrStr;
end;

function TValueLegalANDField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := ValueFromBoolean( ARow.BooleanValue[FConditionField1] and
                              ARow.BooleanValue[FConditionField2] and
                              TrueForBothTrue );
end;

procedure TValueLegalANDField.GetAdditionalLegalValues(var Strings: TStrings);

  procedure GetLegalValues(AField : THasValueField);
  begin
    THasValueField(AField).GetLegalValues(Strings);
  end;

begin
  if ConditionField1 is THasValueField then
    GetLegalValues(THasValueField(ConditionField1));
  if ConditionField2 is THasValueField then
    GetLegalValues(THasValueField(ConditionField2));
end;

// -------------------------- THasValueDefinedInTableField ------------------------

constructor THasValueDefinedInTableField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor THasValueDefinedInTableField.CreateOld(FieldName : String; ADataField : TDataField);
begin
  Inherited CreateOld(FieldName, ADataField);
end;

function THasValueDefinedInTableField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if DataField.HasAuxTable then
    Result := ValueFromBoolean(TKeyField(DataField).AuxTable.Cache.LocateByRowValues(ARow, [TKeyField(DataField)]) <> nil)
  else
    Result := ValueFromBoolean(True);
end;

function THasValueDefinedInTableField.GetErrorString(ARow : TAbstractRow) : String;
begin
  Result := MSGI_ValueFor + ' Field ' + DataField.FieldName + ' ' + MSGI_NotDefinedInTable;
end;

procedure THasValueDefinedInTableField.GetAdditionalLegalValues(
  var Strings: TStrings);
begin
  // Nothing
end;

{ TConditionalShowField }

constructor TConditionalShowField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FFirstValueField := nil;
  FSecondValueField  := nil;
  FConditionField := nil;
  FShowFirstForTrue := True;
end;

constructor TConditionalShowField.CreateOld(FieldName : String ; DataType : TDataType; ReadOnly,
  CalcBeforeAggregating : Boolean; FirstValueField, SecondValueField,
  ConditionField : TDataField; ShowFirstForTrue : Boolean);
begin
  Inherited CreateOld(FieldName, DataType, ReadOnly, CalcbeforeAggregating);

  FFirstValueField := FirstValueField;
  FSecondValueField  := SecondValueField;
  FConditionField := ConditionField;
  FShowFirstForTrue := ShowFirstForTrue;
end;

destructor TConditionalShowField.Destroy;
begin
  inherited;
end;

function TConditionalShowField.CalcValue(ARow: TAbstractRow): TValue;

  function ValueForField(ValueField : TDataField) : TValue;
  begin
    if ValueField = nil then
      Result := Self.DataType.DefaultValue
    else
      Result := ARow[ValueField];
  end;

begin
  if ARow.BooleanValue[ConditionField] = ShowFirstForTrue then
    Result := ValueForField(FirstValueField)
  else
    Result := ValueForField(SecondValueField);
end;

function TConditionalShowField.DistributeValue(ARow: TAbstractRow;
  Value: TValue): TSetResult;
begin
  Result := ARow.SetFieldValue(FirstValueField, Value, saDontOverwriteKeys);
end;

{ TEditAuxTableField }

constructor TEditAuxTableField.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
end;

constructor TEditAuxTableField.CreateOld(AFieldName : String; ASrcField : TKeyField);
begin
  Inherited CreateDependent(AFieldName, ASrcField, nil, nil);

  IsAggregable := True;
//  LookupTable := ASrcField.LookupTable;
{  if ASrcField = nil then
    raise Exception.Create(Self.ClassName + '.Create: No SrcField supplied!')
  else if ASrcField.LookupTable = nil then
    raise Exception.Create(Self.ClassName + '.Create: SrcField ' + ASrcField.FieldName +
       ' doesn''r have any lookup table!');
}
//  DirectAuxTableField := ASrcField;
end;

destructor TEditAuxTableField.Destroy;
begin
  Inherited Destroy;
end;

procedure TEditAuxTableField.GetValues(Results : TStrings; Table : TDataTable; Row : TAbstractRow; Condition : TCondition);
begin
  if Assigned( DirectAuxTableField ) then
    DirectAuxTableField.GetRows(Results, Condition)
  else
    GetRows(Results, Condition);
end;

function TEditAuxTableField.GetReadOnly(ARow : TAbstractRow) : Boolean;
begin
  if ARow = nil then
    Result := FDefaultReadOnly
  else
    Result := FDefaultReadOnly or ( GetLookupRow(ARow) = nil );
end;



function TEditAuxTableField.GetLookupRow(SrcRow : TAbstractRow) : TDataRow;
var
  ACrit : TCriteria;
  LookupTable : TAuxTable;
  Rows : TDataRowList;
begin
  Result := nil;
  ACrit := TCriteria.CreateFromRowKeys(SrcRow);
{  if Debug then
    DisplayCondition( ACrit );
}
  Rows := TDataRowList.Create;
  try
//    ACrit[DirectAuxTableField].AddValue( SrcRow[Self] );
    LookupTable := DirectAuxTableField.LookupTable;
    Assert(LookupTable <> nil);
    LookupTable.Cache.GetRows( Rows, ACrit, gaReference );
    if Rows.Count = 1 then
      Result := Rows.FirstRow;
  finally
    ACrit.Free;
    Rows.Free;
  end;
end;

function TEditAuxTableField.RowHasKeys(ARow : TAbstractRow) : Boolean;
var
  iKey : Integer;
  AList : TList;
  SubTotal : TSubTotalRow;
begin
  Assert(ARow <> nil);
  Result := True;
  if not (ARow is TSubTotalRow) then
    Exit;

  SubTotal := TSubTotalRow(ARow);
  AList := TList.Create;
  Assert(DirectAuxTableField.LookupTable <> nil);
  DirectAuxTableField.LookupTable.KeysToList(AList);

  while SubTotal <> nil do
  begin
    iKey := AList.IndexOf(SubTotal.SubTotalKey.TreeKey);
    if iKey >= 0 then
      AList.Delete(iKey);
    SubTotal := SubTotal.SubTotalRow;
  end;

  Result := (AList.Count = 0);
  AList.Free;
end;

function TEditAuxTableField.ValueLegal(SrcRow : TAbstractRow; Value : TValue) : Boolean;
var
  ACrit : TCriteria;
  AList : TDataRowList;
begin
  ACrit := TCriteria.CreateFromRowFields(SrcRow);
  ACrit[Self].Reset;
  ACrit[DirectAuxTableField].Reset;
  ACrit[DirectAuxTableField].AddValue(Value);

  AList := TDataRowList.Create;
  DirectAuxTableField.GetRows(AList, ACrit); // AuxTable.Cache.LocateRowByCriteria(ACrit);
  Result := AList.Count = 1;

  ACrit.Free;
  AList.Free;
end;

function TEditAuxTableField.GetExternValue(ARow : TAbstractRow) : TValue;
var
  i : Integer;
  subRow : TSubTotalRow;
begin
  if ARow is TSubTotalRow then
  begin
    Result := EmptyString;
    subRow := TSubTotalRow(ARow);
    for i := 0 to subRow.SubRowCount -1 do
    begin
      if i = 0 then
        Result := GetExternValue(subRow.SubRows[i])
      else if not DataType.Equals(Result, GetExternValue(subRow.SubRows[i])) then
      begin
        Result := EmptyString;
        Break;
      end;
    end;
  end
  else
    Result := ARow.GetFieldValue(DirectAuxTableField);
end;

function TEditAuxTableField.SetExternValue(ARow : TAbstractRow; Value : TValue; Action : TSetAction) : TSetResult;
var
  DestRow : TDataRow;
begin
  if not RowHasKeys(ARow) then
  begin
    Result := srReadOnly;
    Exit;
  end;

  if ValueLegal(ARow, Value) then
  begin
    DestRow := GetLookupRow( ARow );
    if DestRow <> nil then
      Result := DestRow.SetFieldValue(DirectAuxTableField, Value, Action)
    else
      Result := srInvalidValue;
  end
  else
    Result := srInvalidValue;
end;

{
function TEditAuxTableField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if RowHasKeys(ARow) then
    Result := ARow.GetFieldValue(DirectAuxTableField)
  else
    Result := ValueFromString('');
end;

function TEditAuxTableField.DistributeValue(ARow : TAbstractRow; Value : TValue) : TSetResult;
var
  ACrit : TCriteria;
  LookupRow : TDataRow;
begin
  if not RowHasKeys(ARow) then
  begin
    Result := srReadOnly;
    Exit;
  end;

  ACrit := TCriteria.CreateFromRowFields(ARow);
  ACrit[DirectAuxTableField].Reset;
  ACrit[DirectAuxTableField].AddValue(Value);
  LookupRow := GetLookupRow(ACrit);

  if LookupRow = nil then
    Result := srReadOnly
  else
    Result := LookupRow.SetFieldValue(DirectAuxTableField, Value, saOverwriteOnKeyChange);
end;
}

{ TPictureField }

constructor TPictureField.CreateOld(FieldName : String; APicture : TPicture);
begin
  Inherited CreateOld(FieldName, PictureType, True, False);
  Self.IsAggregable := True;
  FPicture := APicture;
end;

destructor TPictureField.Destroy;
begin
  inherited Destroy;
end;

function TPictureField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  Result := ValueFromPicture(FPicture);
end;

constructor TSubTotalPercentageField.CreateOld(const FieldName : String; DataField : TDataField);
begin
  inherited CreateOld( FieldName, PercentType, True, True );
  fDataField := DataField;
end;

function TSubTotalPercentageField.GetFieldCount : Integer;
begin
  Result := 1;
end;

function TSubTotalPercentageField.GetField(idx : Integer) : TDataField;
begin
  Result := fDataField;
end;

function TSubTotalPercentageField.CreateCopy(const FieldName : String; GetNewField : TQueryFieldFunction; OwnedList:TList) : TClosedField;
begin
  Result := TSubTotalPercentageField.CreateOld( FieldName, GetNewField( fDataField, OwnedList ) );
  OwnedList.Add( Result );
end;

function TSubTotalPercentageField.CalcValue(ARow : TAbstractRow) : TValue;
begin
  if ARow.SubTotalRow <> nil then
  begin
    if ARow.SubTotalRow.DoubleValue[fDataField] <> 0 then
      Result := DataType.Quota( ARow.Value[fDataField], ARow.SubTotalRow.Value[fDataField])
    else
      Result := ValueFromPercent(0);
  end
  else    
    Result := ValueFromPercent(1);
end;

end.
