{$INCLUDE OPTIONS.INC}

{*****************************************************************************}
{                                                                             }
{       Maps v0.93 Generic Associative Containers for Delphi 2, 3 & 4         }
{                                                                             }
{                 Copyright (c) 1999 Robert R. Marsh, S.J.                    }
{               & the British Province of the Society of Jesus                }
{                                                                             }
{                This source code may *not* be redistributed                  }
{                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  }
{                                                                             }
{       If you like Maps and find yourself using it please consider           }
{       making a donation to your favorite charity. I would also be           }
{       pleased if you would acknowledge Maps in any projects that            }
{       make use of it.                                                       }
{                                                                             }
{       Maps is supplied as is. The author disclaims all warranties,          }
{       expressed or implied, including, without limitation, the              }
{       warranties of merchantability and of fitness for any purpose.         }
{       The author assumes no liability for damages, direct or                }
{       consequential, which may result from the use of QDB.                  }
{                                                                             }
{                           rrm@sprynet.com                                   }
{                     http://home.sprynet.com/~rrm                            }
{                                                                             }
{*****************************************************************************}

(*

  The Variables unit provides support for the use of TVarRecs in Maps

*)

unit Variables;

interface

// note the strict type definition -- it means that, e.g., even though
// Indicator is actually a Pointer, when an Indicator is called for a
// simple Pointer won't do.
type
  Variable = type System.TVarRec;
  Indicator = type Pointer;

  // System.pas defines the following constants as possible
  // values for a Variable's type.

  //      vtInteger    = 0;
  //      vtBoolean    = 1;
  //      vtChar       = 2;
  //      vtExtended   = 3;
  //      vtString     = 4;
  //      vtPointer    = 5;
  //      vtPChar      = 6;
  //      vtObject     = 7;
  //      vtClass      = 8;
  //      vtWideChar   = 9;
  //      vtPWideChar  = 10;
  //      vtAnsiString = 11;
  //      vtCurrency   = 12;
  //      vtVariant    = 13;
  //      vtInterface  = 14; // D3 and up
  //      vtWideString = 15; // D4 and up
  //      vtInt64      = 16; // D4 and up

  // we add our own definition -- you have to imagine that vtVoid
  // is of no type and any type
  //      vtVoid       = 255;

const
  vtVoid = High(Byte);

  // a global variable set in the initialization section,
  // Void can be used in place of an actual Variable, e.g.,
  // if you just want to add a key without a datum use a
  // Void in place of the datum.
var
  Void : Variable;

function IsVoid(const v : Variable) : Boolean;

// Returns a byte value according to the above table
function TypeOf(const v : Variable) : Byte;

// Sets an occupied Variable to Void, releasing any
// memory and finalizing any internal values.
procedure ClearVariable(var v : Variable);

// Copies the value of the Source Variable to Dest.
// Dest must first be made Void or leaks will occur.
procedure CopyVariable(const Source : Variable; var Dest : Variable);

// Compares Variables and returns a negative result if
// V1 is less than V2, 0 if they are equal, and a positve
// result otherwise. V1 and V2 must hold the same kind of
// value or an exception is raised. Void Variables, however,
// are taken to be equal to all others.
// If the Variables hold string values the comparison is
// governed by the CaseSensitive and Ansi parameters. Ansi
// instructs the comparison to take account of the default
// windows locale (with quite an additonal overhead).
function CompareVariables(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;

// Returns a 32-bit value by randomizing the bits of
// v's value according to its type.
function HashVariable(const v : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Cardinal;

// Returns a 32-bit value based on the Variable v. Kinds
// of Variable that fit into 32-bits (e.g., integer, char)
// are not actually hashed. Others are hashed using the
// method above.
function ScatterVariable(const v : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Cardinal;

// Returns a string representation of v. Some kinds
// of Variable make more sense than others.
function VariableToString(const v : Variable) : string;

// Returns a numeric representation of v. Some kinds
// of Variable make more sense than others.
{$IFDEF D4}
function VariableToNumber(const v : Variable) : Int64;
{$ELSE}
function VariableToNumber(const v : Variable) : Integer;
{$ENDIF}

// Returns a single Variable holding whatever was in
// Value.
function AsVariable(Value : array of const) : Variable;

// Extract a reference of the right kind from the
// Variable v. If the Variable is of the wrong kind
// a null result is returned, e.g., 0, '', or nil.
function AsInteger(const v : Variable) : Integer;
function AsBoolean(const v : Variable) : Boolean;
function AsChar(const v : Variable) : Char;
function AsExtended(const v : Variable) : Extended;
function AsPointer(const v : Variable) : Pointer;
function AsPChar(const v : Variable) : PChar;
function AsObject(const v : Variable) : TObject;
function AsClass(const v : Variable) : TClass;
function AsString(const v : Variable) : string;
function AsCurrency(const v : Variable) : Currency;
function AsVariant(const v : Variable) : Variant;
{$IFDEF D3}
function AsInterface(const v : Variable) : Pointer;
{$ENDIF}
{$IFDEF D4}
function AsInt64(const v : Variable) : Int64;
{$ENDIF}

// Use a message box to show a string representation
// of Variable v. Intended as a quick and dirty
// debugging technique.
procedure ShowVariable(const v : Variable);

implementation

uses
  Windows,
  Classes,
  SysUtils,
  Hashing;

function TypeOf(const v : Variable) : Byte;
begin
  Result := v.VType;
end;

function IsVoid(const v : Variable) : Boolean;
begin
  Result := v.VType = vtVoid;
end;

function AsVariable(Value : array of const) : Variable;
begin
  Result := Variable(Value[0]);
end;

function AsInteger(const v : Variable) : Integer;
begin
  Result := 0;
  if v.VType = vtInteger then
    Result := v.VInteger;
end;

function AsBoolean(const v : Variable) : Boolean;
begin
  Result := Low(Boolean);
  if v.VType = vtBoolean then
    Result := v.VBoolean;
end;

function AsChar(const v : Variable) : Char;
begin
  Result := #0;
  if v.VType = vtChar then
    Result := v.VChar;
end;

function AsExtended(const v : Variable) : Extended;
begin
  Result := 0.0;
  if v.VType = vtExtended then
    Result := v.VExtended^;
end;

function AsPointer(const v : Variable) : Pointer;
begin
  Result := nil;
  if v.VType = vtPointer then
    Result := v.VPointer;
end;

function AsPChar(const v : Variable) : PChar;
begin
  Result := PChar('');
  if v.VType = vtPChar then
    Result := v.VPChar;
end;

function AsObject(const v : Variable) : TObject;
begin
  Result := nil;
  if v.VType = vtObject then
    Result := v.VObject;
end;

function AsClass(const v : Variable) : TClass;
begin
  Result := nil;
  if v.VType = vtClass then
    Result := v.VClass;
end;

function AsString(const v : Variable) : string;
begin
  Result := '';
  case v.VType of
    vtAnsiString : Result := string(v.VAnsiString);
    vtString : Result := v.VString^;
  else
  end;
end;

function AsCurrency(const v : Variable) : Currency;
begin
  Result := 0.0;
  if v.VType = vtCurrency then
    Result := v.VCurrency^;
end;

function AsVariant(const v : Variable) : Variant;
begin
  Result := Null;
  if v.VType = vtVariant then
    Result := v.VVariant^;
end;

{$IFDEF D3}

function AsInterface(const v : Variable) : Pointer;
begin
  Result := nil;
  if v.VType = vtInterface then
    Result := v.VInterface;
end;
{$ENDIF}

{$IFDEF D4}

function AsInt64(const v : Variable) : Int64;
begin
  Result := 0;
  if v.VType = vtInt64 then
    Result := v.VInt64^;
end;
{$ENDIF}

// As far as I can tell GetMem.inc defines the same
// "structure" for allocated pointers in D2-4.
// SizeOfMem peeks into the stuff before the pointer
// itself and pulls out the stored info on how much
// memory was requested. FreeMem uses that info
// to return the pointer to the heap -- we use it
// to copy the contents of memory from one place to
// another.

function SizeOfMem(P : Pointer) : Integer;
type
  PUsed = ^TUsed;
  TUsed = record
    sizeflags : Integer;
  end;
const
  cflags = 2 or 1 or Integer($80000000);
var
  u : PUsed;
begin
  if P = nil then
    Result := 0
  else
  begin
    u := P;
    u := PUsed(PChar(u) - SizeOf(TUsed));
    Result := u.sizeflags;
    Result := Result and not cflags - SizeOf(TUsed);
  end;
end;

// Helper type used to carry persistent objects back
// and forth from streams.
type
  TPC = class(TComponent)
  private
    FCarried : TPersistent;
  published
    property Carried : TPersistent Read FCarried Write FCarried;
  end;

procedure CopyVariable(const Source : Variable; var Dest : Variable);
var
  Len : Integer;
  m : TMemoryStream;
  c : TPC;
begin
  Dest.VType := vtInteger;
  Dest.VInteger := 0;
  Dest.VType := Source.VType;
  // some types can be directly assigned...
  if (Source.VType in [vtInteger, vtBoolean, vtChar, vtClass, vtWideChar{$IFDEF D3}, vtInterface{$ENDIF}, vtVoid]) then
    Dest := Source
  else
  begin
    // ...other types need special treatment
    case Source.VType of
      vtExtended :
        begin
          GetMem(Dest.VExtended, SizeOf(Extended));
          Dest.VExtended^ := Source.VExtended^;
        end;
      vtString :
        begin
          New(Dest.VString);
          Dest.VString^ := Source.VString^;
        end;
      vtPointer :
        begin
          Len := SizeOfMem(Source.VPointer);
          GetMem(Dest.VPointer, Len);
          Move(Source.VPointer^, Dest.VPointer^, Len);
        end;
      vtPChar :
        begin
          Len := StrLen(Source.VPChar) + 1;
          GetMem(Dest.VPChar, Len);
          Move(Source.VPChar^, Dest.VPChar^, Len);
        end;
      vtPWideChar :
        begin
          Len := 2 * StrLen(PChar(Source.VPWideChar)) + 1;
          GetMem(Dest.VPWideChar, Len);
          Move(Source.VPWideChar^, Dest.VPWideChar^, Len);
        end;
      vtObject :
        begin
          m := TMemoryStream.Create;
          try
            if Source.VObject is TComponent then
            begin
              m.WriteComponent(TComponent(Source.VObject));
              m.Position := 0;
              Dest.VObject := TComponentClass(Source.VObject.ClassType).Create(nil);
              m.ReadComponent(TComponent(Dest.VObject));
            end
            else
            begin
              c := TPC.Create(nil);
              try
                c.Carried := TPersistent(Source.VObject);
                m.WriteComponent(c);
                m.Position := 0;
                c.Carried := TPersistentClass(Source.VObject.ClassType).Create;
                m.ReadComponent(c);
                Dest.VObject := c.Carried;
              finally
                c.Free;
              end;
            end;
          finally
            m.Free;
          end;
        end;
      vtAnsiString :
        begin
          if Source.VAnsiString <> nil then
          begin
            string(Dest.VAnsiString) := string(Source.VAnsiString);
            UniqueString(string(Dest.VAnsiString));
          end
          else
            Dest.VAnsiString := nil;
        end;
      vtCurrency :
        begin
          New(Dest.VCurrency);
          Dest.VCurrency^ := Source.VCurrency^;
        end;
      vtVariant :
        begin
          New(Dest.VVariant);
          Dest.VVariant^ := Source.VVariant^;
        end;
{$IFDEF D3}
      vtWideString :
        begin
          if Source.VWideString <> nil then
          begin
            WideString(Dest.VWideString) := string(Source.VWideString);
            UniqueString(string(Dest.VWideString));
          end
          else
            Dest.VWideString := nil;
        end;
{$ENDIF}
{$IFDEF D4}
      vtInt64 :
        begin
          GetMem(Dest.VInt64, SizeOf(Int64));
          Dest.VInt64^ := Source.VInt64^;
        end;
{$ENDIF}
    end;
  end;
end;

//procedure CopyEasy(const Source : Variable; var Dest : Variable);
//begin
//  Dest := Source;
//end;
//
//procedure CopyExtended(const Source : Variable; var Dest : Variable);
//begin
//  GetMem(Dest.VExtended, SizeOf(Extended));
//  Dest.VExtended^ := Source.VExtended^;
//end;
//
//procedure CopyString(const Source : Variable; var Dest : Variable);
//begin
//  New(Dest.VString);
//  Dest.VString^ := Source.VString^;
//end;
//
//procedure CopyPointer(const Source : Variable; var Dest : Variable);
//var
//  Len : Cardinal;
//begin
//  Len := SizeOfMem(Source.VPointer);
//  GetMem(Dest.VPointer, Len);
//  Move(Source.VPointer^, Dest.VPointer^, Len);
//end;
//
//procedure CopyPChar(const Source : Variable; var Dest : Variable);
//var
//  Len : Cardinal;
//begin
//  Len := StrLen(Source.VPChar) + 1;
//  GetMem(Dest.VPChar, Len);
//  Move(Source.VPChar^, Dest.VPChar^, Len);
//end;
//
//procedure CopyObject(const Source : Variable; var Dest : Variable);
//var
//  m : TMemoryStream;
//  c : TPC;
//begin
//  m := TMemoryStream.Create;
//  try
//    c := TPC.Create(nil);
//    try
//      c.Carried := TPersistent(Source.VObject);
//      m.WriteComponent(c);
//      m.Position := 0;
//      c.Carried := TPersistentClass(Source.VObject.ClassType).Create;
//      m.ReadComponent(c);
//      Dest.VObject := c.Carried;
//    finally
//      c.Free;
//    end;
//  finally
//    m.Free;
//  end;
//end;
//
//procedure CopyPWideChar(const Source : Variable; var Dest : Variable);
//var
//  Len : Cardinal;
//begin
//  Len := 2 * StrLen(PChar(Source.VPWideChar)) + 1;
//  GetMem(Dest.VPWideChar, Len);
//  Move(Source.VPWideChar^, Dest.VPWideChar^, Len);
//end;
//
//procedure CopyAnsiString(const Source : Variable; var Dest : Variable);
//begin
//  if Source.VAnsiString <> nil then
//  begin
//    string(Dest.VAnsiString) := string(Source.VAnsiString);
//    UniqueString(string(Dest.VAnsiString));
//  end
//  else
//    Dest.VAnsiString := nil;
//end;
//
//procedure CopyCurrency(const Source : Variable; var Dest : Variable);
//begin
//  New(Dest.VCurrency);
//  Dest.VCurrency^ := Source.VCurrency^;
//end;
//
//procedure CopyVariant(const Source : Variable; var Dest : Variable);
//begin
//  New(Dest.VVariant);
//  Dest.VVariant^ := Source.VVariant^;
//end;
//
//procedure CopyWideString(const Source : Variable; var Dest : Variable);
//begin
//  if Source.VWideString <> nil then
//  begin
//    WideString(Dest.VWideString) := string(Source.VWideString);
//    UniqueString(string(Dest.VWideString));
//  end
//  else
//    Dest.VWideString := nil;
//end;
//
//procedure CopyInt64(const Source : Variable; var Dest : Variable);
//begin
//  GetMem(Dest.VInt64, SizeOf(Int64));
//  Dest.VInt64^ := Source.VInt64^;
//end;
//
//procedure CopyVariable2(const Source : Variable; var Dest : Variable);
//type
//  CopyProc = procedure(const Source : Variable; var Dest : Variable);
//const
//  CopyLookup : array[vtInteger..vtInt64] of CopyProc =
//    (
//    CopyEasy,
//    CopyEasy,
//    CopyEasy,
//    CopyExtended,
//    CopyString,
//    CopyPointer,
//    CopyPChar,
//    CopyObject,
//    CopyEasy,
//    CopyEasy,
//    CopyPWideChar,
//    CopyAnsiString,
//    CopyCurrency,
//    CopyVariant,
//    CopyEasy,
//    CopyWideString,
//    CopyInt64
//    );
//begin
//  Dest.VInteger := 0;
//  Dest.VType := Source.VType;
//  if Source.VType = vtVoid then
//    Exit;
//  CopyLookup[Source.VType](Source, Dest);
//end;
//
//procedure Dummy(var v : Variable);
//begin
//end;
//
//procedure ClearExtended(var v : Variable);
//begin
//  FreeMem(v.VExtended);
//end;
//
//procedure ClearString(var v : Variable);
//begin
//  FreeMem(v.VString);
//end;
//
//procedure ClearPointer(var v : Variable);
//begin
//  FreeMem(v.VPointer);
//end;
//
//procedure ClearPChar(var v : Variable);
//begin
//  FreeMem(v.VPChar);
//end;
//
//procedure ClearObject(var v : Variable);
//begin
//  v.VObject.Free;
//end;
//
//procedure ClearPWideChar(var v : Variable);
//begin
//  FreeMem(v.VPWideChar);
//end;
//
//procedure ClearAnsiString(var v : Variable);
//begin
//  Finalize(string(v.VAnsiString));
//end;
//
//procedure ClearCurrency(var v : Variable);
//begin
//  FreeMem(v.VCurrency);
//end;
//
//procedure ClearVariant(var v : Variable);
//begin
//  Finalize(v.VVariant^);
//end;
//
//procedure ClearWideString(var v : Variable);
//begin
//  Finalize(WideString(v.VWideString));
//end;
//
//procedure ClearInt64(var v : Variable);
//begin
//  FreeMem(v.VInt64);
//end;
//
//procedure ClearVariable2(var v : Variable);
//type
//  ClearProc = procedure(var v : Variable);
//const
//  ClearLookup : array[vtInteger..vtInt64] of ClearProc =
//    (
//    Dummy,
//    Dummy,
//    Dummy,
//    ClearExtended,
//    ClearString,
//    ClearPointer,
//    ClearPChar,
//    ClearObject,
//    Dummy,
//    Dummy,
//    ClearPWideChar,
//    ClearAnsiString,
//    ClearCurrency,
//    ClearVariant,
//    Dummy,
//    ClearWideString,
//    ClearInt64
//    );
//begin
//  if v.VType = vtVoid then
//    Exit;
//  ClearLookup[V.VType](V);
//  v.VInteger := 0;
//  v.VType := vtVoid;
//end;
//

procedure ClearVariable(var v : Variable);
begin
  if v.VType in [vtExtended, vtString, vtPointer, vtPChar, vtObject, vtPWideChar, vtAnsiString, vtCurrency, vtVariant{$IFDEF D3}, vtWideString{$ENDIF}{$IFDEF D4}, vtInt64{$ENDIF}] then
    case v.VType of
      vtExtended :
        FreeMem(v.VExtended);
      vtString :
        FreeMem(v.VString);
      vtPointer :
        FreeMem(v.VPointer);
      vtPChar :
        FreeMem(v.VPChar);
      vtObject :
        v.VObject.Free;
      vtPWideChar :
        FreeMem(v.VPWideChar);
      vtAnsiString :
        Finalize(string(v.VAnsiString));
      vtCurrency :
        FreeMem(v.VCurrency);
      vtVariant :
        Finalize(v.VVariant^);
{$IFDEF D3}
      vtWideString :
        Finalize(WideString(v.VWideString));
{$ENDIF}
{$IFDEF D4}
      vtInt64 :
        FreeMem(v.VInt64);
{$ENDIF}
    end;
  v := Void;
end;

// Quick and dirty message box showing the Variable as a string

procedure ShowVariable(const v : Variable);
const
{$IFDEF D4}
  VTypeToStr : array[vtInteger..vtInt64] of string =
{$ELSE}
{$IFDEF D3}
  VTypeToStr : array[vtInteger..vtWideString] of string =
{$ELSE}
{$IFDEF D2}
  VTypeToStr : array[vtInteger..vtVariant] of string =
{$ENDIF}
{$ENDIF}
{$ENDIF}
  ('vtInteger', 'vtBoolean', 'vtChar', 'vtExtended', 'vtString',
    'vtPointer', 'vtPChar', 'vtObject', 'vtClass', 'vtWideChar',
    'vtPWideChar', 'vtAnsiString', 'vtCurrency', 'vtVariant'{$IFDEF D3},
    'vtInterface', 'vtWideString'{$ENDIF}{$IFDEF D4}, 'vtInt64'{$ENDIF});
var
  VariableKind : string;
  VariableAsString : string;
begin
  if IsVoid(v) then
  begin
    VariableKind := 'VOID';
    VariableAsString := 'VOID';
  end
  else
  begin
    VariableKind := VTypeToStr[v.VType];
    VariableAsString := VariableToString(v);
  end;
  MessageBox(0, PChar(VariableAsString), PChar(VariableKind), MB_OK or MB_ICONINFORMATION or MB_TASKMODAL);
end;

// Thank you Abner Fog for the code which inspired this ...
// If (!) I understand correctly E1 and E2 will appear to
// be equal if they are within ~0.001% of each other
// e.g., 1.0 and 1.00001 are "equal" but 1.0 and 1.000001
// are not. I think this serves the purpose we need ...

function CompareExtended(E1, E2 : Extended) : Integer;
asm
      push    ebx
      MOV     EAX, [EBP+14]
      MOV     EBX, [EBP+26]
      MOV     ECX, EAX
      MOV     EDX, EBX
      SAR     ECX, 31              // copy sign bit
      AND     EAX, 7FFFFFFFH       // remove sign bit
      SAR     EDX, 31
      AND     EBX, 7FFFFFFFH
      XOR     EAX, ECX      // make 2-complement if sign bit was set
      XOR     EBX, EDX
      SUB     EAX, ECX
      SUB     EBX, EDX
      CMP     EAX, EBX
      JL      @LT
      JZ      @EQ
      JG      @GT
@LT:  MOV     EAX, 1
      JMP     @EXIT
@EQ:  MOV     EAX, 0
      JMP     @EXIT
@GT:  MOV     EAX, -1
      JMP     @EXIT
@EXIT:pop     ebx
end;

function CompareCurrency(C1, C2 : Currency) : Integer;
asm
      push    ebx
      MOV     EAX, [EBP+8]
      MOV     EBX, [EBP+16]
      MOV     ECX, EAX
      MOV     EDX, EBX
      SAR     ECX, 31              // copy sign bit
      AND     EAX, 7FFFFFFFH       // remove sign bit
      SAR     EDX, 31
      AND     EBX, 7FFFFFFFH
      XOR     EAX, ECX      // make 2-complement if sign bit was set
      XOR     EBX, EDX
      SUB     EAX, ECX
      SUB     EBX, EDX
      CMP     EAX, EBX
      JL      @LT
      JZ      @EQ
      JG      @GT
@LT:  MOV     EAX, 1
      JMP     @EXIT
@EQ:  MOV     EAX, 0
      JMP     @EXIT
@GT:  MOV     EAX, -1
      JMP     @EXIT
@EXIT:pop     ebx
end;

function CompareMem(p1, p2 : Pointer; Length : Integer) : Boolean; assembler;
asm
        PUSH    ESI
        PUSH    EDI
        MOV     ESI,P1
        MOV     EDI,P2
        MOV     EDX,ECX
        XOR     EAX,EAX
        AND     EDX,3
        SHR     ECX,1
        SHR     ECX,1
        REPE    CMPSD
        JNE     @@2
        MOV     ECX,EDX
        REPE    CMPSB
        JNE     @@2
@@1:    INC     EAX
@@2:    POP     EDI
        POP     ESI
end;

//function CompInteger(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := V1.VInteger - V2.VInteger;
//end;
//
//function CompBoolean(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Ord(V1.VBoolean) - Ord(V2.VBoolean);
//end;
//
//function CompChar(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Ord(V1.VChar) - Ord(V2.VChar);
//end;
//
//function CompExtended(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := CompareExtended(V1.VExtended^, V2.VExtended^);
//end;
//
//function CompString(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  if Ansi then
//  begin
//    if CaseSensitive then
//    begin
//      Result := AnsiCompareStr(V1.VString^, V2.VString^);
//    end
//    else
//    begin
//      Result := AnsiCompareText(V1.VString^, V2.VString^);
//    end;
//  end
//  else
//  begin
//    if CaseSensitive then
//    begin
//      Result := CompareStr(V1.VString^, V2.VString^);
//    end
//    else
//    begin
//      Result := CompareText(V1.VString^, V2.VString^);
//    end;
//  end;
//end;
//
//function CompPointer(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//var
//  Len1 : Cardinal;
//  Len2 : Cardinal;
//begin
//end;
//
//function CompPChar(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  if Ansi then
//  begin
//    if CaseSensitive then
//    begin
//      Result := AnsiCompareStr(V1.VPChar, V2.VPChar);
//    end
//    else
//    begin
//      Result := AnsiCompareText(V1.VPChar, V2.VPChar);
//    end;
//  end
//  else
//  begin
//    if CaseSensitive then
//    begin
//      Result := CompareStr(V1.VPChar, V2.VPChar);
//    end
//    else
//    begin
//      Result := CompareText(V1.VPChar, V2.VPChar);
//    end;
//  end;
//end;
//
//function CompObject(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Cardinal(V1.VObject) - Cardinal(V2.VObject);
//end;
//
//function CompClass(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Cardinal(V1.VClass) - Cardinal(V2.VClass);
//end;
//
//function CompWideChar(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Ord(V1.VWideChar) - Ord(V2.VWideChar);
//end;
//
//function CompPWideChar(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  if Ansi then
//  begin
//    if CaseSensitive then
//    begin
//      Result := AnsiCompareStr(PChar(V1.VPWideChar), PChar(V2.VPWideChar));
//    end
//    else
//    begin
//      Result := AnsiCompareText(PChar(V1.VPWideChar), PChar(V2.VPWideChar));
//    end;
//  end
//  else
//  begin
//    if CaseSensitive then
//    begin
//      Result := CompareStr(PChar(V1.VPWideChar), PChar(V2.VPWideChar));
//    end
//    else
//    begin
//      Result := CompareText(PChar(V1.VPWideChar), PChar(V2.VPWideChar));
//    end;
//  end;
//end;
//
//function CompAnsiString(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  if Ansi then
//  begin
//    if CaseSensitive then
//    begin
//      Result := AnsiCompareStr(string(V1.VAnsiString), string(V2.VAnsiString));
//    end
//    else
//    begin
//      Result := AnsiCompareText(string(V1.VAnsiString), string(V2.VAnsiString));
//    end;
//  end
//  else
//  begin
//    if CaseSensitive then
//    begin
//      Result := CompareStr(string(V1.VAnsiString), string(V2.VAnsiString));
//    end
//    else
//    begin
//      Result := CompareText(string(V1.VAnsiString), string(V2.VAnsiString));
//    end;
//  end;
//end;
//
//function CompCurrency(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := CompareCurrency(V1.VCurrency^, V2.VCurrency^);
//end;
//
//function CompVariant(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := V1.VVariant^ - V2.VVariant^;
//end;
//
//function CompInterface(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := Cardinal(V1.VInterface) - Cardinal(V2.VInterface);
//end;
//
//function CompWideString(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  if CaseSensitive then
//  begin
//    Result := CompareStr(WideString(V1.VWideString), WideString(V2.VWideString));
//  end
//  else
//  begin
//    Result := CompareText(WideString(V1.VWideString), WideString(V2.VWideString));
//  end;
//end;
//
//function CompInt64(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//begin
//  Result := V1.VInt64^ - V2.VInt64^;
//end;
//
//function CompareVariables2(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//type
//  CompFunc = function(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
//const
//  CompareLookup : array[vtInteger..vtInt64] of CompFunc =
//    (
//    CompInteger,
//    CompBoolean,
//    CompChar,
//    CompExtended,
//    CompString,
//    CompPointer,
//    CompPChar,
//    CompObject,
//    CompClass,
//    CompWideChar,
//    CompPWideChar,
//    CompAnsiString,
//    CompCurrency,
//    CompVariant,
//    CompInterface,
//    CompWideString,
//    CompInt64
//    );
//begin
//  // Void is equal to everything !!
//  Result := CompareLookup[V1.VType](V1, V2, Ansi, CaseSensitive);
//end;
//

function CompareVariables(const V1, V2 : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Integer;
var
  Len1 : Integer;
  Len2 : Integer;
begin
  Result := 0;
  if (V1.VType = vtVoid) or (V2.VType = vtVoid) then
  begin
    Exit;
  end;
  case V1.VType of
    vtInteger :
      Result := V1.VInteger - V2.VInteger;
    vtAnsiString :
      if Ansi then
      begin
        if CaseSensitive then
        begin
          Result := AnsiCompareStr(string(V1.VAnsiString), string(V2.VAnsiString));
        end
        else
        begin
          Result := AnsiCompareText(string(V1.VAnsiString), string(V2.VAnsiString));
        end;
      end
      else
      begin
        if CaseSensitive then
        begin
          Result := CompareStr(string(V1.VAnsiString), string(V2.VAnsiString));
        end
        else
        begin
          Result := CompareText(string(V1.VAnsiString), string(V2.VAnsiString));
        end;
      end;
    vtBoolean :
      Result := Ord(V1.VBoolean) - Ord(V2.VBoolean);
    vtChar :
      Result := Ord(V1.VChar) - Ord(V2.VChar);
    vtExtended :
      Result := CompareExtended(V1.VExtended^, V2.VExtended^);
    vtString :
      if Ansi then
      begin
        if CaseSensitive then
        begin
          Result := AnsiCompareStr(V1.VString^, V2.VString^);
        end
        else
        begin
          Result := AnsiCompareText(V1.VString^, V2.VString^);
        end;
      end
      else
      begin
        if CaseSensitive then
        begin
          Result := CompareStr(V1.VString^, V2.VString^);
        end
        else
        begin
          Result := CompareText(V1.VString^, V2.VString^);
        end;
      end;
    vtPointer :
      begin
        Len1 := SizeOfMem(V1.VPointer);
        Len2 := SizeOfMem(V2.VPointer);
        if Len1 = Len2 then
        begin
          if CompareMem(V1.VPointer, V2.VPointer, Len1) then
            Result := 0
          else
            Result := 1;
        end
        else
          Result := Len1 - Len2;
      end;
    vtPChar :
      Result := Cardinal(V1.VPChar) - Cardinal(V2.VPChar);
    vtObject :
      Result := Cardinal(V1.VObject) - Cardinal(V2.VObject);
    vtClass :
      Result := Cardinal(V1.VClass) - Cardinal(V2.VClass);
    vtWideChar :
      Result := Ord(V1.VWideChar) - Ord(V2.VWideChar);
    vtPWideChar :
      Result := Cardinal(V1.VPWideChar) - Cardinal(V2.VPWideChar);
    vtCurrency :
      Result := CompareCurrency(V1.VCurrency^, V2.VCurrency^);
    vtVariant :
      Result := V1.VVariant^ - V2.VVariant^;
{$IFDEF D3}
    vtInterface :
      Result := Cardinal(V1.VInterface) - Cardinal(V2.VInterface);
    vtWideString :
      if CaseSensitive then
      begin
        Result := AnsiCompareStr(WideString(V1.VWideString), WideString(V2.VWideString));
      end
      else
      begin
        Result := AnsiCompareText(WideString(V1.VWideString), WideString(V2.VWideString));
      end;
{$ENDIF}
{$IFDEF D4}
    vtInt64 :
      Result := V1.VInt64^ - V2.VInt64^;
{$ENDIF}
  else
    ;
  end;
end;

{$IFDEF D4}

function VariableToNumber(const v : Variable) : Int64;
{$ELSE}

function VariableToNumber(const v : Variable) : Integer;
{$ENDIF}
begin
  Result := 0;
  case v.VType of
    vtInteger :
      Result := v.VInteger;
    vtBoolean :
      Result := Ord(v.VBoolean);
    vtChar :
      Result := Ord(v.VChar);
    vtExtended :
      Result := Round(v.VExtended^);
    vtString :
{$IFDEF D4}
      Result := StrToInt64(v.VString^);
{$ELSE}
      Result := StrToInt(v.VString^);
{$ENDIF}
    vtPointer :
      Result := Cardinal(v.VPointer);
    vtPChar :
{$IFDEF D4}
      Result := StrToInt64(v.VPChar);
{$ELSE}
      Result := StrToInt(v.VPChar);
{$ENDIF}
    vtObject :
      Result := Cardinal(v.VObject);
    vtClass :
      Result := Cardinal(v.VClass);
    vtWideChar :
      Result := Ord(v.VWideChar);
    vtPWideChar :
{$IFDEF D4}
      Result := StrToInt64(v.VPWideChar);
{$ELSE}
      Result := StrToInt(PChar(v.VPWideChar));
{$ENDIF}
    vtAnsiString :
{$IFDEF D4}
      Result := StrToInt64(string(v.VAnsiString));
{$ELSE}
      Result := StrToInt(string(v.VAnsiString));
{$ENDIF}
    vtCurrency :
      Result := Round(v.VCurrency^);
    vtVariant :
      Result := Integer(v.VVariant^);
{$IFDEF D3}
    vtInterface :
      Result := Cardinal(v.VInterface);
    vtWideString :
{$IFDEF D4}
      Result := StrToInt64(WideString(v.VWideString));
{$ELSE}
      Result := StrToInt(WideString(v.VWideString));
{$ENDIF}
{$ENDIF}
{$IFDEF D4}
    vtInt64 :
      Result := v.VInt64^;
{$ENDIF}
  else
    ;
  end;
end;

const
  tf : array[False..True] of string = ('true', 'false');

function VariableToString(const v : Variable) : string;
type
  TArray = array[0..Maxint - 1] of Byte;
  PArray = ^TArray;
var
  i : Integer;
begin
  Result := '';
  case v.VType of
    vtInteger :
      Result := IntToStr(v.VInteger);
    vtBoolean :
      Result := tf[v.VBoolean];
    vtChar :
      Result := v.VChar;
    vtExtended :
      Result := FloatToStr(v.VExtended^);
    vtString :
      Result := v.VString^;
    vtPointer :
      begin
        SetLength(Result, 3 * SizeOfMem(v.VPointer));
        Result := '';
        for i := 1 to SizeOfmem(v.VPointer) do
          Result := Result + IntToHex(PArray(v.VPointer)^[i - 1], 2) + ' ';
      end;
    vtPChar :
      Result := v.VPChar;
    vtObject :
      Result := v.VObject.Classname;
    vtClass :
      Result := v.VClass.Classname;
    vtWideChar :
      Result := Char(v.VWideChar);
    vtPWideChar :
      Result := PChar(v.VPWideChar);
    vtAnsiString :
      begin
        Result := string(v.VAnsiString);
        UniqueString(Result);
      end;
    vtCurrency :
      Result := CurrToStr(v.VCurrency^);
    vtVariant :
      Result := v.VVariant^;
{$IFDEF D3}
    //vtInterface: (VInterface: Pointer);
    vtWideString :
      Result := WideString(v.VWideString);
{$ENDIF}
{$IFDEF D4}
    vtInt64 :
      Result := IntToStr(v.VInt64^);
{$ENDIF}
  else
    Result := 'VOID';
  end;
end;

// Hash a Variable appropriately (i.e., preserving as
// wide a scatter as possible). Produces a full 32-bit
// value which the caller may reduce to a specific range
// by integer division.
// Note that types less than 4 bytes wide are NOT hashed
// since hashing would increase rather than decrease
// collisions (consider that the n integers from 1 to n
// have themselves, by definition, no collisions whereas
// any hashing scheme based on bit-randomization can only
// hope to achieve a 50% collision rate).

function ScatterVariable(const v : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Cardinal;
var
  S : string;
begin
  Result := 0;
  case v.VType of
    vtInteger :
      Result := Cardinal(v.VInteger);
    vtAnsiString :
      begin
        if not CaseSensitive then
        begin
          if Ansi then
            S := AnsiUpperCase(string(v.VAnsiString))
          else
            S := UpperCase(string(v.VAnsiString));
          Result := Hash(S[1], Length(S));
        end
        else
          Result := Hash(string(v.VAnsiString)[1], Length(string(v.VAnsiString)));
      end;
    vtBoolean :
      Result := Cardinal(v.VBoolean);
    vtChar :
      Result := Cardinal(v.VChar);
    vtExtended :
      Result := Hash(v.VExtended^, SizeOf(Extended));
    vtString :
      begin
        if not CaseSensitive then
        begin
          if Ansi then
            S := AnsiUpperCase(v.VString^)
          else
            S := UpperCase(v.VString^);
          Result := Hash(S[1], Length(S));
        end
        else
          Result := Hash(v.VString^, Length(v.VString^));
      end;
    vtPointer :
      Result := Hash(v.VPointer^, SizeOfMem(v.VPointer));
    vtPChar :
      begin
        if not CaseSensitive then
        begin
          if Ansi then
            S := AnsiStrUpper(v.VPChar)
          else
            S := StrUpper(v.VPChar);
          Result := Hash(S[1], Length(S));
        end
        else
          Result := Hash(v.VPChar^, StrLen(v.VPChar));
      end;
    vtObject :
      Result := Cardinal(v.VObject);
    vtClass :
      Result := Cardinal(v.VClass);
    vtWideChar :
      Result := Cardinal(v.VWideChar);
    vtPWideChar :
      begin
        if not CaseSensitive then
        begin
          S := AnsiUpperCase(v.VPWideChar);
          Result := Hash(S[1], Length(S));
        end
        else
          Result := Hash(v.VPWideChar^, Length(PChar(v.VPWideChar)));
      end;
    vtCurrency :
      Result := Hash(v.VCurrency^, SizeOf(Currency));
    vtVariant :
      Result := Hash(v.VVariant^, SizeOf(Variant));
{$IFDEF D3}
    vtInterface :
      Result := Cardinal(v.VInterface);
    vtWideString :
      begin
        if not CaseSensitive then
        begin
          S := AnsiUpperCase(WideString(v.VWideString));
          Result := Hash(S[1], Length(S));
        end
        else
          Result := Hash(string(v.VAnsiString)[1], Length(string(v.VAnsiString)));
      end;
{$ENDIF}
{$IFDEF D4}
    vtInt64 :
      Result := Hash(v.VInt64^, SizeOf(Int64));
{$ENDIF}
  end;
end;

function HashVariable(const v : Variable; Ansi : Boolean; CaseSensitive : Boolean) : Cardinal;
begin
  Result := 0;
  case v.VType of
    vtInteger :
      Result := Hash(v.VInteger, SizeOf(Integer));
    vtAnsiString :
      Result := Hash(string(v.VAnsiString)[1], Length(string(v.VAnsiString)));
    vtBoolean :
      Result := Hash(v.VBoolean, SizeOf(Boolean));
    vtChar :
      Result := Hash(v.VChar, SizeOf(Char));
    vtExtended :
      Result := Hash(v.VExtended^, SizeOf(Extended));
    vtString :
      Result := Hash(v.VString^, Length(v.VString^));
    vtPointer :
      Result := Hash(v.VPointer^, SizeOfMem(v.VPointer));
    vtPChar :
      Result := Hash(v.VPChar^, StrLen(v.VPChar));
    vtObject :
      Result := Hash(v.VObject, SizeOf(TObject));
    vtClass :
      Result := Hash(v.VClass, SizeOf(TClass));
    vtWideChar :
      Result := Hash(v.VWideChar, SizeOf(WideChar));
    vtPWideChar :
      Result := Hash(v.VPWideChar^, Length(PChar(v.VPWideChar)));
    vtCurrency :
      Result := Hash(v.VCurrency^, SizeOf(Currency));
    vtVariant :
      Result := Hash(v.VVariant^, SizeOf(Variant));
{$IFDEF D3}
    vtInterface :
      Result := Hash(v.VInterface, SizeOf(Pointer));
    vtWideString :
      Result := Hash(WideString(v.VWideString)[1], Length(WideString(v.VWideString)));
{$ENDIF}
{$IFDEF D4}
    vtInt64 :
      Result := Hash(v.VInt64^, SizeOf(Int64));
{$ENDIF}
  end;
end;

initialization

  FillChar(Void, SizeOf(Void), vtVoid);

end.

