{$INCLUDE OPTIONS.INC}

{*****************************************************************************}
{                                                                             }
{       Maps v0.93 Generic Associative Containers for Delphi 2, 3 & 4         }
{                                                                             }
{                 Copyright (c) 1999 Robert R. Marsh, S.J.                    }
{               & the British Province of the Society of Jesus                }
{                                                                             }
{                This source code may *not* be redistributed                  }
{                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  }
{                                                                             }
{       If you like Maps and find yourself using it please consider           }
{       making a donation to your favorite charity. I would also be           }
{       pleased if you would acknowledge Maps in any projects that            }
{       make use of it.                                                       }
{                                                                             }
{       Maps is supplied as is. The author disclaims all warranties,          }
{       expressed or implied, including, without limitation, the              }
{       warranties of merchantability and of fitness for any purpose.         }
{       The author assumes no liability for damages, direct or                }
{       consequential, which may result from the use of QDB.                  }
{                                                                             }
{                           rrm@sprynet.com                                   }
{                     http://home.sprynet.com/~rrm                            }
{                                                                             }
{*****************************************************************************}

(*

      Maps: Generic Associative Containers for Delphi (2, 3, & 4)

  Maps are probably the most generally useful of all the abstract data types
  yet they are missing from Delphis VCL. What are they? First of all they
  are containersthey are designed to hold and manipulate data. What kind of
  data? Any kindnearly! The maps provided in this library are genericthey
  can hold Delphis atomic types (integer, real, string, etc.) and they can
  also hold objects. And like Delphis TStringList, maps are associative
  containersthey store things in pairs so that given one you can get the
  other. But where TStringList can only associate objects with strings, maps
  can be used to associate strings with objects, or integers with characters,
    or whatever you want.

  The trick behind the ability to handle generic data is Delphis own array
  of const idiom which is used in a few strategic places in the VCL, e.g.,
  in the Format function. You wont find it much used elsewhere though
  mainly because programmers imagine it to be slow and unwieldy, perhaps
  confusing it with another of Delphis mechanisms, the Variant type.
  Ross Judson, however, has shown it to be a powerful, robust, and efficient
  way of manipulating generic data. The inspiration for this library comes
  from Ross Judson's SDL (Standard Delphi Library) which uses the "array of
  const" technique to implement a very powerful generic container and
  algorithm library modeled after the C++ Standard Template Library (STL).
  Delphi programmers should overcome their distrust of anything remotely
  related to C++ and give it a careful look at Ross website
  (http://www.soletta.com).

  The aim of the Maps library presented here is much more modest than
  Judsons. Delphi's "built-in" containers, TList and TStringList, are so
  easy to use that most programmers turn to them without further thought
  despite two great disadvantages. First, Delphi's lists manage to be
  generic only by handling all data as pointers, thereby ignoring a great
  deal of information that could be used to make life easier for the
  programmer. Second, and more important, the array-based architecture of
  Delphi's lists trades off the efficiency of some operations against
  others. In particular efficiency of insertion and deletion is sacrificed
  for rapid random access and searching. Every implementation makes such
  tradeoffswhat the programmer needs is a variety of containers offering
  a range of behaviors. With the library presented here you have the choices
  you need. If you want the fastest addition and deletion, for example, you
  could use the list-based map. For the fastest search, the hash map is
  the best choice but if, in addition to efficient searching, you need to
  access data in order the map based on a treap proves a good choice.

  See Maps.html for more information and examples or the demonstration
  program Anagram.

  A note about programming style. I try to keep implementation details
  hidden away and out of the interface section so you'll see in many
  places class definitions contain plain pointers even though the entity
  being declared is really another class. This keeps the distracting
  private details out of the way until they are needed. The downside is
  that the implementation code has to typecast such pointers appropriately
  every time they are used. It makes no difference at run-time and an
  invalid typecast could wreak havoc but I prefer to pay that price
  rather than have to parade all the details of, say, the custom memory
  allocator where such information is only confusing. Not a standard
  Delphi style but it works for me. I hope you can understand it too.

*)

{$DEFINE TYPECHECK} // activate run-time type safety

unit Maps;

(*

  v.0.92
    Hashing now takes account of Ansi and CaseSensitive
    Added clipboard support -- use Assign or LoadFrom/SaveToClipboard

*)

interface

uses
  SysUtils,
  Classes,
  HATList,
  Variables,
  Rand;

type
  EMapError = class(Exception)
  end;

var
  CF_MAP : Word; // clipboard format for maps

  // This is the abstract Map class and should never be instantiated.
type
  TAbstractMap = class(TPersistent)
  private
    FAnsi : Boolean;
    FCaseSensitive : Boolean;
    FCount : Cardinal;
    MA : Pointer; // a custom memory allocator
    SP : Indicator; // an internal pointer used during traversal
{$IFDEF TYPECHECK}
    KeyType : Byte; // the type of permitted Keys
    DatumType : Byte; // the type of permitted Data
{$ENDIF}
    function GetDatumByKey(const Key : Variable) : Variable;
    procedure SetDatumByKey(const Key, Datum : Variable);
  protected
    // handles TMap to TStrings assignments
    procedure AssignTo(Dest : TPersistent); override;
{$IFDEF TYPECHECK}
    procedure CheckKey(const Key : Variable);
    procedure CheckDatum(const Datum : Variable);
{$ENDIF}
    // handles map persistence
    procedure DefineProperties(Filer : TFiler); override;
    function GetCapacity : Cardinal; virtual; abstract;
    procedure SetAnsi(Value : Boolean);
    procedure SetCapacity(NewCapacity : Cardinal); virtual; abstract;
    procedure SetCaseSensitive(Value : Boolean);
  public
    constructor Create; virtual;
    // Add a copy of a single item
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; virtual; abstract;
    // Add a reference to a single item
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; virtual; abstract;
    // Add a copy of a number of items
    procedure AddItems(Keys : array of const; Data : array of const);
    // Make this Map a copy of another, or of a TStrings
    procedure Assign(Source : TPersistent); override;
    // Empty the Map and dispose of all Variables
    procedure Clear; virtual; abstract;
    // If there is an item with a matching Key it is deleted
    procedure DeleteByKey(const Key : Variable); virtual; abstract;
    // If there are items with matching Keys they are deleted
    procedure DeleteByKeys(Keys : array of const);
    // If there is an item with a matching Datum it is deleted
    // Caution -- much slower than DeleteByKey
    procedure DeleteByDatum(const Datum : Variable); virtual; abstract;
    // If there are items with matching Data they are deleted
    // Caution -- much slower than DeleteByKeys
    procedure DeleteByData(const Data : array of const);
    class procedure Error(const Msg : string);
    // Locate an item with a matching Key and sets Index to point to it.
    // Return value indicates success or failure. Index is only defined in case of success.
    function Find(const Key : Variable; var Index : Indicator) : Boolean; virtual; abstract;
    // Return an Indicator to the first item.
    function First : Indicator; virtual; abstract;
    // Return a reference to the Key at Index
    function GetKeyRef(const Index : Indicator) : Variable; virtual; abstract;
    // Return a copy of the Key at Index
    function GetKeyVal(const Index : Indicator) : Variable; virtual; abstract;
    // Return a reference to the Datum at Index
    function GetDatumRef(const Index : Indicator) : Variable; virtual; abstract;
    // Return a copy of the Datum at Index
    function GetDatumVal(const Index : Indicator) : Variable; virtual; abstract;
    // Return an Indicator to the item matching the Key (or invalid if no match).
    function IndexOfKey(const Key : Variable) : Indicator;
    // Return an Indicator to the item matching the Datum (or invalid if no match).
    // Caution -- much slower than IndexOfKey
    function IndexOfDatum(const Datum : Variable) : Indicator;
    // Make Index an invalid Indicator.
    procedure Invalidate(var Index : Indicator); virtual; abstract;
    // Return an Indicator to the last item.
    function Last : Indicator; virtual; abstract;
    // Make this Map a copy of the one stored on the Clipboard
    procedure LoadFromClipboard;
    // Make this Map a copy of the one stored in FilenName
    procedure LoadFromFile(const FileName : string); virtual;
    // Make this Map a copy of the one stored in Stream
    procedure LoadFromStream(Stream : TStream); virtual;
    // Return an Indicator to the next item (or invalid if none).
    function Next : Indicator; virtual; abstract;
    // Return an Indicator to the previous item (or invalid if none).
    function Prev : Indicator; virtual; abstract;
    // Change the Key in the item pointed to by Index. No copy.
    procedure PutKeyRef(const Index : Indicator; const Key : Variable); virtual; abstract;
    // Change the Key in the item pointed to by Index. Copies.
    procedure PutKeyVal(const Index : Indicator; const Key : Variable); virtual; abstract;
    // Change the Datum in the item pointed to by Index. No copy.
    procedure PutDatumRef(const Index : Indicator; const Datum : Variable); virtual; abstract;
    // Change the Datum in the item pointed to by Index. Copies.
    procedure PutDatumVal(const Index : Indicator; const Datum : Variable); virtual; abstract;
    // Store the state of this Map to the Clipboard
    procedure SaveToClipboard;
    // Store the state of this Map to FileName
    procedure SaveToFile(const FileName : string); virtual;
    // Store the state of this Map to Stream
    procedure SaveToStream(Stream : TStream); virtual;
    // Is Index a valid Indicator for this Map?
    function Valid(const Index : Indicator) : Boolean; virtual; abstract;
    // Should string comparisons take account of locale or not (much faster).
    property Ansi : Boolean Read FAnsi Write SetAnsi Default True;
    // Indicate the expected capacity of the Map. (Crucial for HAMap).
    property Capacity : Cardinal Read GetCapacity Write SetCapacity;
    // Should string comparisons take account of case?
    property CaseSensitive : Boolean Read FCaseSensitive Write SetCaseSensitive Default True;
    // The number of items in the Map
    property Count : Cardinal Read FCount;
    // Gives access to Keys via an Indicator
    property Keys[const Index : Indicator] : Variable Read GetKeyRef Write PutKeyVal;
    // Gives access to Data via an Indicator
    property Data[const Index : Indicator] : Variable Read GetDatumRef Write PutDatumVal;
    // Gives access to Data via a Key
    property DataByKey[const Key : Variable] : Variable Read GetDatumByKey Write SetDatumBykey;
  end;

type
  TAbstractMapClass = class of TAbstractMap;

  // Uses an Unsorted Doubly-Linked List as the underlying structure.
  // Add       : Fastest
  // Delete    : Fast given an Indicator, Very slow otherwise
  // Find      : Very slow
  // Traversal : Very fast
  // Capacity irrelevant
type
  TListMap = class(TAbstractMap)
  private
    FAddAtFront : Boolean;
    Head : Pointer;
    Tail : Pointer;
  protected
    function GetCapacity : Cardinal; override;
    procedure SetCapacity(NewCapacity : Cardinal); override;
  public
    constructor Create; override;
    destructor Destroy; override;
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; override;
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; override;
    procedure Clear; override;
    // ListMaps can delete an indicated item
    procedure Delete(var Index : Indicator);
    procedure DeleteByKey(const Key : Variable); override;
    procedure DeleteByDatum(const Datum : Variable); override;
    function Find(const Key : Variable; var Index : Indicator) : Boolean; override;
    function First : Indicator; override;
    function GetKeyRef(const Index : Indicator) : Variable; override;
    function GetKeyVal(const Index : Indicator) : Variable; override;
    function GetDatumRef(const Index : Indicator) : Variable; override;
    function GetDatumVal(const Index : Indicator) : Variable; override;
    procedure Invalidate(var Index : Indicator); override;
    function Last : Indicator; override;
    function Next : Indicator; override;
    function Prev : Indicator; override;
    procedure PutKeyRef(const Index : Indicator; const Key : Variable); override;
    procedure PutKeyVal(const Index : Indicator; const Key : Variable); override;
    procedure PutDatumRef(const Index : Indicator; const Datum : Variable); override;
    procedure PutDatumVal(const Index : Indicator; const Datum : Variable); override;
    function Valid(const Index : Indicator) : Boolean; override;
    // should items be added at front or back
    property AddAtFront : Boolean Read FAddAtFront Write FAddAtFront;
  end;

  // Uses a Hashed Array Tree as the underlying structure (based on C
  // code by Edward Sitarski, DDJ, September 1996, 107-110).
  // Can be sorted or not
  // Add       : Very slow when sorted, Moderate if not
  // Delete    : Slow in general, Fast at end
  // Find      : Moderate when sorted, Very slow if not
  // Traversal : Fastest
  // Setting Capacity before adding many items makes Add Fast
type
  THATMap = class(TAbstractMap)
  private
    FList : THATList;
    FSorted : Boolean;
    FDuplicates : TDuplicates;
    procedure InsertionSort(l, r : Integer);
    procedure QuickSort(l, r : Integer);
  protected
    function GetCapacity : Cardinal; override;
    procedure SetAnsi(Value : Boolean);
    procedure SetCapacity(NewCapacity : Cardinal); override;
    procedure SetCaseSensitive(Value : Boolean);
    procedure SetDuplicates(Value : TDuplicates);
    procedure SetSorted(Value : Boolean);
  public
    constructor Create; override;
    destructor Destroy; override;
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; override;
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; override;
    procedure Clear; override;
    // HATMaps can delete an indicated item
    procedure Delete(var Index : Indicator);
    procedure DeleteByKey(const Key : Variable); override;
    procedure DeleteByDatum(const Datum : Variable); override;
    procedure Exchange(Here, There : Indicator);
    function Find(const Key : Variable; var Index : Indicator) : Boolean; override;
    function First : Indicator; override;
    function GetKeyRef(const Index : Indicator) : Variable; override;
    function GetKeyVal(const Index : Indicator) : Variable; override;
    function GetDatumRef(const Index : Indicator) : Variable; override;
    function GetDatumVal(const Index : Indicator) : Variable; override;
    // HATMaps can insert at an indicated place
    procedure InsertItemVal(Index : Indicator; Key : array of const; Datum : array of const);
    procedure Invalidate(var Index : Indicator); override;
    function Last : Indicator; override;
    procedure Move(Here, There : Indicator);
    function Next : Indicator; override;
    function Prev : Indicator; override;
    procedure PutKeyRef(const Index : Indicator; const Key : Variable); override;
    procedure PutKeyVal(const Index : Indicator; const Key : Variable); override;
    procedure PutDatumRef(const Index : Indicator; const Datum : Variable); override;
    procedure PutDatumVal(const Index : Indicator; const Datum : Variable); override;
    procedure Reverse(SetSorted : Boolean);
    procedure Sort; virtual;
    function Valid(const Index : Indicator) : Boolean; override;
    property Duplicates : TDuplicates Read FDuplicates Write SetDuplicates;
    property Sorted : Boolean Read FSorted Write SetSorted;
  end;

  // Uses a Hash Array with Chaining as the underlying structure.
  // You should specify the likely Capacity of the table in advance.
  // Initially this is set to 17 but it should usually be changed by
  // setting Capacity. This is much cheaper if done when the table is
  // empty. When the nominal capacity is exceeded performance falls
  // off rapidly.
  // Add       : Very Fast
  // Delete    : Fast
  // Find      : Very fast and almost independant of size
  // Traversal : Fast but not in Key order
  // N.B. Reverse traversal is slower but since the order of
  // traversal hardly matters just avoid it.
type
  THashMap = class(TAbstractMap)
  private
    FCapacity : Cardinal;
    FArray : Pointer;
  protected
    function GetCapacity : Cardinal; override;
    procedure SetCapacity(NewCapacity : Cardinal); override;
  public
    constructor Create; override;
    constructor CreateWithCapacity(ACapacity : Cardinal); virtual;
    destructor Destroy; override;
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; override;
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; override;
    procedure Clear; override;
    procedure Delete(var Index : Indicator);
    procedure DeleteByKey(const Key : Variable); override;
    procedure DeleteByDatum(const Datum : Variable); override;
    function Find(const Key : Variable; var Index : Indicator) : Boolean; override;
    function First : Indicator; override;
    function GetKeyRef(const Index : Indicator) : Variable; override;
    function GetKeyVal(const Index : Indicator) : Variable; override;
    function GetDatumRef(const Index : Indicator) : Variable; override;
    function GetDatumVal(const Index : Indicator) : Variable; override;
    procedure Invalidate(var Index : Indicator); override;
    function Last : Indicator; override;
    function Next : Indicator; override;
    function Prev : Indicator; override;
    procedure PutKeyRef(const Index : Indicator; const Key : Variable); override;
    procedure PutKeyVal(const Index : Indicator; const Key : Variable); override;
    procedure PutDatumRef(const Index : Indicator; const Datum : Variable); override;
    procedure PutDatumVal(const Index : Indicator; const Datum : Variable); override;
    function Valid(const Index : Indicator) : Boolean; override;
  end;

  // A generic abstract Map which implements the methods shared among
  // the concrete trees which descend from it. DO NOT create such a
  // tree -- it has no useful functions.
type
  TAbstractTree = class(TAbstractMap)
  private
    Root : Pointer;
    NullNode : Pointer;
    TraversalStack : Pointer;
  protected
    function GetCapacity : Cardinal; override;
    procedure SetCapacity(NewCapacity : Cardinal); override;
  public
    constructor Create; override;
    destructor Destroy; override;
    procedure DeleteByDatum(const Datum : Variable); override;
    function First : Indicator; override;
    function GetKeyRef(const Index : Indicator) : Variable; override;
    function GetKeyval(const Index : Indicator) : Variable; override;
    function GetDatumRef(const Index : Indicator) : Variable; override;
    function GetDatumVal(const Index : Indicator) : Variable; override;
    procedure Invalidate(var Index : Indicator); override;
    function Last : Indicator; override;
    function Next : Indicator; override;
    function Prev : Indicator; override;
    procedure PutKeyRef(const Index : Indicator; const Key : Variable); override;
    procedure PutKeyVal(const Index : Indicator; const Key : Variable); override;
    procedure PutDatumRef(const Index : Indicator; const Datum : Variable); override;
    procedure PutDatumVal(const Index : Indicator; const Datum : Variable); override;
    function Valid(const Index : Indicator) : Boolean; override;
  end;

  // Uses a simple Binary Search Tree as the underlying structure but
  // ordered by the hash value of the Key rather than the Key itself.
  // The randomness of the hashing tends to keep the tree balanced.
  // Performance is very good but key-ordering is forfeited.
  // Add       : Fast
  // Delete    : Fastest
  // Find      : Very fast
  // Traversal : Fast but out of key-order.
  // Capacity irrelevant
type
  TTreeMap = class(TAbstractTree)
  private
  protected
  public
    constructor Create; override;
    destructor Destroy; override;
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; override;
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; override;
    procedure Clear; override;
    procedure DeleteByKey(const Key : Variable); override;
    function Find(const Key : Variable; var Index : Indicator) : Boolean; override;
  end;

  // Uses a Treap as the underlying structure.
  // Add       : Moderate but acceptable
  // Delete    : Moderate
  // Find      : Moderate
  // Traversal : Fast and ordered
  // Capacity irrelevant
type
  TTreapMap = class(TAbstractTree)
  private
    FDuplicates : TDuplicates;
    RandNum : TRandGen;
  protected
  public
    constructor Create; override;
    destructor Destroy; override;
    function AddItemVal(const Key : Variable; const Datum : Variable) : Indicator; override;
    function AddItemRef(const Key : Variable; const Datum : Variable) : Indicator; override;
    procedure Clear; override;
    procedure DeleteByKey(const Key : Variable); override;
    function Find(const Key : Variable; var Index : Indicator) : Boolean; override;
    property Duplicates : TDuplicates Read FDuplicates Write FDuplicates;
  end;

implementation

uses
  Windows,
  Clipbrd,
  Allocators,
  Resources;

{ TAbstractMap }

procedure TAbstractMap.AddItems(Keys, Data : array of const);
var
  n : Integer;
begin
  for n := 0 to High(Keys) do
    AddItemVal(Variable(Keys[n]), Variable(Data[n]));
end;

procedure TAbstractMap.Assign(Source : TPersistent);
var
  i : Integer;
  l : Indicator;
begin
  if Source is TAbstractMap then
  begin
    Clear;
    l := TAbstractMap(Source).First;
    while TAbstractMap(Source).Valid(l) do
    begin
      AddItemVal(TAbstractMap(Source).GetKeyRef(l), TAbstractMap(Source).GetDatumRef(l));
      l := TAbstractMap(Source).Next;
    end;
    Exit;
  end;
  if Source is TStrings then
  begin
    Clear;
    for i := 0 to TStrings(Source).Count - 1 do
    begin
      AddItems([TStrings(Source).Strings[i]], [TStrings(Source).Objects[i]]);
    end;
    Exit;
  end;
  if Source is TClipboard then
  begin
    LoadFromClipboard;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TAbstractMap.AssignTo(Dest : TPersistent);
var
  i : Indicator;
  Key : Variable;
  Item : Variable;
  S : string;
  o : TObject;
begin
  if Dest is TStrings then
  begin
    TStrings(Dest).Clear;
    i := First;
    while Valid(i) do
    begin
      Key := GetKeyRef(i);
      S := VariableToString(Key);
      // this is a mess really
      // it might be better to ignore the datum
      Item := GetDatumRef(i);
      case Item.VType of
        vtInteger :
          o := TObject(Item.VInteger);
        vtObject :
          o := Item.VObject;
        vtPointer :
          o := TObject(Item.VPointer);
      else
        o := nil;
      end;
      TStrings(Dest).AddObject(S, o);
      i := Next;
    end;
    Exit;
  end;
  if Dest is TClipboard then
  begin
    SaveToClipboard;
    Exit;
  end;
  inherited AssignTo(Dest);
end;

{$IFDEF TYPECHECK}

procedure TAbstractMap.CheckKey(const Key : Variable);
begin
  if Key.VType = vtVoid then
    Exit;
  if (Key.VType = vtObject) and not (Key.VObject is TPersistent) then
    Error(SNotPersistent);
  if KeyType = vtVoid then
    KeyType := Key.VType
  else if Key.VType <> KeyType then
    Error(SWrongKeyType);
end;
{$ENDIF}

{$IFDEF TYPECHECK}

procedure TAbstractMap.CheckDatum(const Datum : Variable);
begin
  if Datum.VType = vtVoid then
    Exit;
  if (Datum.VType = vtObject) and not (Datum.VObject is TPersistent) then
    Error(SNotPersistent);
  if DatumType = vtVoid then
    DatumType := Datum.VType
  else if Datum.VType <> DatumType then
    Error(SWrongDatumType);
end;
{$ENDIF}

constructor TAbstractMap.Create;
begin
  inherited Create;
  FCount := 0;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
  Ansi := True;
  CaseSensitive := True;
end;

procedure TAbstractMap.DefineProperties(Filer : TFiler);

  function DoWrite : Boolean;
  begin
    Result := Count > 0;
  end;

begin
  Filer.DefineBinaryProperty('Map', LoadFromStream, SaveToStream, DoWrite);
end;

procedure TAbstractMap.DeleteByKeys(Keys : array of const);
var
  n : Integer;
begin
  for n := 0 to High(Keys) do
    DeleteByKey(Variable(Keys[n]));
end;

procedure TAbstractMap.DeleteByData(const Data : array of const);
var
  n : Integer;
begin
  for n := 0 to High(Data) do
    DeleteByDatum(Variable(Data[n]));
end;

class procedure TAbstractMap.Error(const Msg : string);
var
  StackTop : record
  end;
  Stack : record
    BPorEBP : Integer;
    ReturnAddress : Pointer;
  end absolute StackTop;
  n : string;
begin
  n := Classname;
  raise EMapError.CreateFmt('%s: %s', [PChar(@n[2]), Msg])at Stack.ReturnAddress;
end;

function TAbstractMap.GetDatumByKey(const Key : Variable) : Variable;
var
  Index : Indicator;
begin
  Index := IndexOfKey(Key);
  if not Valid(Index) then
  begin
    Error(SNoSuchKey);
  end;
  Result := GetDatumRef(Index);
end;

function TAbstractMap.IndexOfKey(const Key : Variable) : Indicator;
begin
  if not Find(Key, Result) then
    Invalidate(Result);
end;

function TAbstractMap.IndexOfDatum(const Datum : Variable) : Indicator;
begin
  Result := First;
  while Valid(Result) do
  begin
    if CompareVariables(GetDatumRef(Result), Datum, FAnsi, FCaseSensitive) = 0 then
      Exit;
    Result := Next;
  end;
end;

procedure TAbstractMap.LoadFromClipboard;
var
  AData : THandle;
  PData : Pointer;
  m : TMemoryStream;
begin
  OpenClipboard(0);
  try
    AData := GetClipboardData(CF_MAP);
    if AData = 0 then
      Exit;
    PData := GlobalLock(AData);
    if PData = nil then
      Exit;
    try
      m := TMemoryStream.Create;
      try
        m.WriteBuffer(PData^, GlobalSize(AData));
        m.Position := 0;
        LoadFromStream(m);
      finally
        m.Free;
      end;
    finally
      GlobalUnlock(AData);
    end;
  finally
    CloseClipboard
  end;
end;

procedure TAbstractMap.LoadFromFile(const FileName : string);
var
  Stream : TStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

type
  TPC = class(TComponent)
  private
    FCarried : TPersistent;
  published
    property Carried : TPersistent Read FCarried Write FCarried;
  end;

const
  BufSize = 16 * 1024;

type
  TAbstractMapItem = (tvaNull, tvaList, tvaInt8, tvaInt16, tvaInt32, tvaExtended,
    tvaString, tvaIdent, tvaFalse, tvaTrue, tvaBinary, tvaSet, tvaLString,
    tvaNil, tvaCollection, tvaSingle, tvaCurrency, tvaDate, tvaWString,
    tvaChar, tvaPointer, tvaPChar, tvaPersistent, tvaComponent, tvaClass,
    tvaWideChar, tvaPWideChar, tvaVariant, tvaInt64, tvaByte);

const
{$IFDEF D4}
  ValTtoVarT : array[TValueType] of Integer = (varNull, varError, varByte,
    varSmallint, varInteger, varDouble, varString, varError, varBoolean,
    varBoolean, varError, varError, varString, varEmpty, varError, varSingle,
    varCurrency, varDate, varOleStr);
{$ELSE}
  ValTtoVarT : array[TValueType] of Integer = (varNull, varError, varByte,
    varSmallint, varInteger, varDouble, varString, varError, varBoolean,
    varBoolean, varError, varError, varString, varEmpty, varError);
{$ENDIF}

type
  THackReader = class(TReader);

procedure TAbstractMap.LoadFromStream(Stream : TStream);
var
  Reader : TReader;
  Key : Variable;
  Item : Variable;

  procedure ReadVariable(var v : Variable);
  var
    ValType : TAbstractMapItem;
    VarValType : TValueType;
    LenByte : Byte;
    Len : Integer;
    S : string;
    pc : TPC;

    function ReadItem : TAbstractMapItem;
    begin
      Reader.Read(Result, SizeOf(Result));
    end;

  begin
    ValType := ReadItem;
    case ValType of
      tvaInt32 :
        begin
          v.VType := vtInteger;
          Reader.Read(v.VInteger, SizeOf(Integer));
        end;
      tvaExtended :
        begin
          v.VType := vtExtended;
          GetMem(v.VExtended, SizeOf(Extended));
          Reader.Read(v.VExtended^, SizeOf(Extended));
        end;
      tvaChar :
        begin
          v.VType := vtChar;
          Reader.Read(v.VChar, SizeOf(Char));
        end;
      tvaString :
        begin
          v.VType := vtString;
          Reader.Read(LenByte, SizeOf(Byte));
          GetMem(v.VString, LenByte + 1);
          v.VString^[0] := Char(LenByte);
          Reader.Read(v.VString^[1], LenByte);
        end;
      tvaFalse :
        begin
          v.VType := vtBoolean;
          v.VBoolean := False;
        end;
      tvaTrue :
        begin
          v.VType := vtBoolean;
          v.VBoolean := True;
        end;
      tvaCurrency :
        begin
          v.VType := vtCurrency;
          GetMem(v.VCurrency, SizeOf(Currency));
          Reader.Read(v.VCurrency^, SizeOf(Currency));
        end;
      tvaLString :
        begin
          v.VType := vtAnsiString;
          Len := 0;
          Reader.Read(Len, SizeOf(Integer));
          v.VAnsiString := nil;
          SetLength(string(v.VAnsiString), Len);
          Reader.Read(Pointer(v.VAnsiString)^, Len);
        end;
      tvaPChar :
        begin
          v.VType := vtPChar;
          Len := 0;
          Reader.Read(Len, SizeOf(Integer));
          GetMem(v.VPChar, Len + 1);
          FillChar(v.VPChar^, Len + 1, 0);
          Reader.Read(v.VPChar^, Len);
        end;
      tvaPersistent :
        begin
          v.VType := vtObject;
          Reader.Read(Len, SizeOf(Integer));
          SetLength(S, Len);
          Reader.Read(S[1], Len);
          pc := TPC.Create(nil);
          pc.Carried := TPersistentClass(FindClass(S)).Create;
          Reader.FlushBuffer;
          Stream.ReadComponent(pc);
          v.VObject := pc.carried;
          pc.Free;
        end;
      tvaComponent :
        begin
          v.VType := vtObject;
          Reader.FlushBuffer;
          v.VObject := Stream.ReadComponent(nil);
        end;
      tvaVariant :
        begin
          VarValType := THackReader(Reader).NextValue;
          case VarValType of
            vaNil, vaNull :
              begin
                if Reader.ReadValue = vaNil then
                  VarClear(v.VVariant^)
                else
                  v.VVariant^ := Null;
              end;
            vaInt8 : TVarData(v.VVariant^).VByte := Byte(Reader.ReadInteger);
            vaInt16 : TVarData(v.VVariant^).VSmallint := Smallint(Reader.ReadInteger);
            vaInt32 : TVarData(v.VVariant^).VInteger := Reader.ReadInteger;
            vaExtended : TVarData(v.VVariant^).VDouble := Reader.ReadFloat;
            vaString, vaLString : v.VVariant^ := Reader.ReadString;
            vaFalse, vaTrue : TVarData(v.VVariant^).VBoolean := Reader.ReadValue = vaTrue;
{$IFDEF D4}
            vaSingle : TVarData(v.VVariant^).VSingle := Reader.ReadSingle;
            vaCurrency : TVarData(v.VVariant^).VCurrency := Reader.ReadCurrency;
            vaDate : TVarData(v.VVariant^).VDate := Reader.ReadDate;
            vaWString : v.VVariant^ := Reader.ReadWideString;
{$ENDIF}
          else
            Error(SVariantReadError);
          end;
          TVarData(v.VVariant^).VType := ValTtoVarT[VarValType];
        end;
      tvaPointer :
        begin
          v.VType := vtPointer;
          Reader.Read(Len, SizeOf(Len));
          GetMem(v.VPointer, Len);
          Reader.Read(v.VPointer^, Len);
        end;
{$IFDEF D4}
      tvaInt64 :
        begin
          v.VType := vtInt64;
          GetMem(v.VInt64, SizeOf(Int64));
          Reader.Read(v.VInt64^, SizeOf(Int64));
        end;
{$ENDIF}
{$IFDEF D3}
      tvaWString :
        begin
          v.VType := vtWideString;
          Reader.Read(Len, SizeOf(Integer));
          v.VWideString := nil;
          SetLength(WideString(v.VWideString), Len);
          Reader.Read(Pointer(v.VWideString)^, Len * 2);
        end;
{$ENDIF}
    else
    end;
  end;

begin
  Reader := TReader.Create(Stream, BufSize);
  Reader.Position := Stream.Position;
  try
    if Reader.ReadString <> Classname then
      Error(SNotValidMapStream);
    if not Reader.EndOfList then
      Clear;
    while not Reader.EndOfList do
    begin
      Reader.ReadListBegin;
      ReadVariable(Key);
      ReadVariable(Item);
      AddItemRef(Key, Item);
      Reader.ReadListEnd;
    end;
    Reader.ReadListEnd;
  finally
    Reader.Free;
  end;
end;

procedure TAbstractMap.SaveToClipboard;
var
  AData : THandle;
  PData : Pointer;
  m : TMemoryStream;
begin
  OpenClipboard(0);
  try
    m := TMemoryStream.Create;
    try
      SaveToStream(m);
      m.Position := 0;
      AData := GlobalAlloc(GMEM_MOVEABLE + GMEM_DDESHARE, m.Size);
      try
        PData := GlobalLock(AData);
        try
          Move(m.Memory^, PData^, m.Size);
          EmptyClipboard;
          SetClipboardData(CF_MAP, AData);
        finally
          GlobalUnlock(AData);
        end;
      except
        GlobalFree(AData);
        raise;
      end;
    finally
      m.Free;
    end;
  finally
    CloseClipboard
  end;
end;

procedure TAbstractMap.SaveToFile(const FileName : string);
var
  Stream : TStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TAbstractMap.SaveToStream(Stream : TStream);
var
  Writer : TWriter;
  i : Indicator;

  procedure WriteVariable(const v : Variable);
  var
    Len : Integer;
    S : string;
    pc : TPC;

    procedure WriteItem(Item : TAbstractMapItem);
    begin
      Writer.Write(Item, SizeOf(Item));
    end;

  begin
    case v.VType of
      vtInteger :
        begin
          WriteItem(tvaInt32);
          Writer.Write(v.VInteger, SizeOf(Integer));
        end;
      vtBoolean :
        begin
          if v.VBoolean then
            WriteItem(tvaTrue)
          else
            WriteItem(tvaFalse);
        end;
      vtChar :
        begin
          WriteItem(tvaChar);
          Writer.Write(v.VChar, SizeOf(Char));
        end;
      vtExtended :
        begin
          WriteItem(tvaExtended);
          Writer.Write(v.VExtended^, SizeOf(Extended));
        end;
      vtString :
        begin
          WriteItem(tvaString);
          Writer.Write(v.VString^[0], SizeOf(Byte));
          Writer.Write(v.VString^[1], Integer(v.VString^[0]));
        end;
      vtPointer :
        begin
          WriteItem(tvaPointer);
          //??        Len:=SizeOfMem(v.VPointer);
          Writer.Write(Len, SizeOf(Len));
          Writer.Write(v.VPointer^, Len);
        end;
      vtPChar :
        begin
          WriteItem(tvaPChar);
          Len := Length(string(v.VPChar));
          Writer.Write(Len, SizeOf(Len));
          Writer.Write(v.VPChar^, Len);
        end;
      vtObject :
        begin
          if v.VObject is TComponent then
          begin
            WriteItem(tvaComponent);
            Writer.FlushBuffer;
            Stream.WriteComponent(TComponent(v.VObject));
          end
          else if v.VObject is TPersistent then
          begin
            WriteItem(tvaPersistent);
            S := v.VObject.Classname;
            Len := Length(S);
            Writer.Write(Len, SizeOf(Integer));
            Writer.Write(S[1], Len);
            Writer.FlushBuffer;
            pc := TPC.Create(nil);
            pc.Carried := TPersistent(v.VObject);
            Stream.WriteComponent(pc);
            pc.Free;
          end
          else
            //Error('streaming objects implemented only for TPersistent and TComponent');
        end;
      vtClass : ;
      //        Error('streaming classes not implemented', 0);
      vtWideChar :
        begin
          WriteItem(tvaWideChar);
          Writer.Write(v.VWideChar, SizeOf(WideChar));
        end;
      vtPWideChar :
        begin
          WriteItem(tvaPWideChar);
          Len := Length(PChar(v.VPWideChar));
          Writer.Write(Len, SizeOf(Integer));
          Writer.Write(v.VPWideChar^, Len * 2);
        end;
      vtAnsiString :
        begin
          WriteItem(tvaLString);
          Len := Length(string(v.VAnsiString));
          Writer.Write(Len, SizeOf(Integer));
          Writer.Write(Pointer(v.VAnsiString)^, Len);
        end;
      vtCurrency :
        begin
          WriteItem(tvaCurrency);
          Writer.Write(v.VCurrency^, SizeOf(Currency));
        end;
      vtVariant :
        begin
          WriteItem(tvaVariant);
          case VarType(v.VVariant^) and varTypeMask of
            varEmpty : WriteItem(tvaNil);
            varNull : WriteItem(tvaNull);
            varString : Writer.WriteString(v.VVariant^);
            varByte, varSmallint, varInteger : Writer.WriteInteger(v.VVariant^);
            varDouble : Writer.WriteFloat(v.VVariant^);
            varBoolean : Writer.WriteBoolean(v.VVariant^);
{$IFDEF D4}
            varOleStr : Writer.WriteWideString(v.VVariant^);
            varSingle : Writer.WriteSingle(v.VVariant^);
            varCurrency : Writer.WriteCurrency(v.VVariant^);
            varDate : Writer.WriteDate(v.VVariant^);
{$ENDIF}
          else
            try
              Writer.WriteString(v.VVariant^);
            except
              raise EWriteError.Create(SVariantWriteError);
            end;
          end;
        end;
{$IFDEF D3}
      vtInterface :
        ; //Error('streaming interfaces not implemented', 0);
      vtWideString :
        begin
          WriteItem(tvaWString);
          Len := Length(WideString(v.VWideString));
          Writer.Write(Len, SizeOf(Integer));
          Writer.Write(Pointer(v.VWideString)^, Len * 2);
        end;
{$ENDIF}
{$IFDEF D4}
      vtInt64 :
        begin
          WriteItem(tvaInt64);
          Writer.Write(v.VInt64^, SizeOf(Int64));
        end;
{$ENDIF}
    else
    end;
  end;

  procedure WriteElement(Index : Indicator);
  var
    Key : Variable;
    Item : Variable;
  begin
    Writer.WriteListBegin;
    Key := GetKeyRef(Index);
    WriteVariable(Key);
    Item := GetDatumRef(Index);
    WriteVariable(Item);
    Writer.WriteListEnd;
  end;

begin
  Writer := TWriter.Create(Stream, BufSize);
  try
    Writer.WriteString(Classname);
    // we go backwards so that the unordered lists preserve
    // their order when reloaded
    i := Last;
    while Valid(i) do
    begin
      WriteElement(i);
      i := Prev;
    end;
    Writer.WriteListEnd;
  finally
    Writer.Free;
  end;
end;

// We only allow this to change when empty -- otherwise
// we would have to rebuild everything in case the
// ordering changed

procedure TAbstractMap.SetAnsi(Value : Boolean);
begin
  if FCount = 0 then
    FAnsi := Value
  else
    Error(SCannotChangeOrder);
end;

// We only allow this to change when empty -- otherwise
// we would have to rebuild everything in case the
// ordering changed

procedure TAbstractMap.SetCaseSensitive(Value : Boolean);
begin
  if FCount = 0 then
    FCaseSensitive := Value
  else
    Error(SCannotChangeOrder);
end;

procedure TAbstractMap.SetDatumBykey(const Key, Datum : Variable);
var
  Index : Indicator;
begin
  Index := IndexOfKey(Key);
  if not Valid(Index) then
  begin
    // not there so add it
    AddItemVal(Key, Datum);
  end
  else
    PutDatumVal(Index, Datum);
end;

{ TListMap }

type
  PLNode = ^TLNode;
  TLNode = record
    Key : Variable;
    Datum : Variable;
    Left : PLNode;
    Right : PLNode;
  end;

function TListMap.AddItemVal(const Key, Datum : Variable) : Indicator;
var
  P : PLNode;
begin
  P := TAllocator(MA).Allocate;
  CopyVariable(Key, P.Key);
  CopyVariable(Datum, P.Datum);
  if FAddAtFront then
  begin
    P.Left := nil;
    P.Right := Head;
    if (Tail = nil) then
    begin
      { previously empty }
      Tail := P;
      Head := Tail;
    end
    else
    begin
      PLNode(Head).Left := P;
      Head := P;
    end;
  end
  else
  begin
    P.Right := nil;
    P.Left := Tail;
    if (Head = nil) then
    begin
      { previously empty }
      Head := P;
      Tail := Head;
    end
    else
    begin
      PLNode(Tail).Right := P;
      Tail := P;
    end;
  end;
  Inc(FCount);
  Result := P;
end;

function TListMap.AddItemRef(const Key, Datum : Variable) : Indicator;
var
  P : PLNode;
begin
  P := TAllocator(MA).Allocate;
  P.Key := Key;
  P.Datum := Datum;
  if FAddAtFront then
  begin
    P.Left := nil;
    P.Right := Head;
    if (Tail = nil) then
    begin
      { previously empty }
      Tail := P;
      Head := Tail;
    end
    else
    begin
      PLNode(Head).Left := P;
      Head := P;
    end;
  end
  else
  begin
    P.Right := nil;
    P.Left := Tail;
    if (Head = nil) then
    begin
      { previously empty }
      Head := P;
      Tail := Head;
    end
    else
    begin
      PLNode(Tail).Right := P;
      Tail := P;
    end;
  end;
  Inc(FCount);
  Result := P;
end;

procedure TListMap.Clear;
var
  P : PLNode;
begin
  while Head <> nil do
  begin
    P := Head;
    Head := PLNode(Head).Right;
    ClearVariable(P.Key);
    ClearVariable(P.Datum);
    TAllocator(MA).Dispose(P);
  end;
  FCount := 0;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
  Tail := nil;
end;

constructor TListMap.Create;
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(TLNode));
  FAddAtFront := True;
  Head := nil;
  Tail := nil;
end;

procedure TListMap.Delete(var Index : Indicator);
begin
  if FCount = 1 then
  begin
    Head := nil;
    Tail := nil
  end
  else if PLNode(Index).Right = nil then
  begin
    { last element }
    Tail := PLNode(Index).Left;
    PLNode(Tail).Right := nil;
  end
  else if PLNode(Index).Left = nil then
  begin
    { first element }
    Head := PLNode(Index).Right;
    PLNode(Head).Left := nil;
  end
  else
  begin
    { element found between first and last }
    PLNode(Index).Left.Right := PLNode(Index).Right;
    PLNode(Index).Right.Left := PLNode(Index).Left;
  end;
  ClearVariable(PLNode(Index).Key);
  Clearvariable(PLNode(Index).Datum);
  TAllocator(MA).Dispose(Index);
  Dec(FCount);
  Invalidate(Index);
end;

procedure TListMap.DeleteByKey(const Key : Variable);
var
  i : Indicator;
begin
  if Find(Key, i) then
    Delete(i);
end;

procedure TListMap.DeleteByDatum(const Datum : Variable);
var
  i : Indicator;
begin
  i := IndexOfDatum(Datum);
  if Valid(i) then
    Delete(i);
end;

destructor TListMap.Destroy;
begin
  Clear;
  TAllocator(MA).Free;
  inherited Destroy;
end;

// exhaustive search

function TListMap.Find(const Key : Variable; var Index : Indicator) : Boolean;
var
  P : PLNode;
begin
  Result := False;
  P := Head;
  while P <> nil do
  begin
    if CompareVariables(Key, P.Key, FAnsi, FCaseSensitive) = 0 then
    begin
      Result := True;
      Index := P;
      Exit;
    end;
    P := P.Right;
  end;
  Invalidate(Index);
end;

function TListMap.First : Indicator;
begin
  SP := Head; // SP keeps a note of where we are
  Result := SP;
end;

// Capacity is unlimited

function TListMap.GetCapacity : Cardinal;
begin
  Result := High(Cardinal);
end;

function TListMap.GetKeyRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PLNode(Index).Key;
end;

function TListMap.GetKeyVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PLNode(Index).Key, Result);
end;

function TListMap.GetDatumRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PLNode(Index).Datum;
end;

function TListMap.GetDatumVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PLNode(Index).Datum, Result);
end;

procedure TListMap.Invalidate(var Index : Indicator);
begin
  Index := nil;
end;

function TListMap.Last : Indicator;
begin
  SP := Tail;
  Result := SP;
end;

function TListMap.Next : Indicator;
begin
  if SP <> nil then
  begin
    SP := PLNode(SP).Right;
  end;
  Result := SP;
end;

function TListMap.Prev : Indicator;
begin
  if SP <> nil then
  begin
    SP := PLNode(SP).Left;
  end;
  Result := SP;
end;

procedure TListMap.PutKeyRef(const Index : Indicator; const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PLNode(Index).Key);
  PLNode(Index).Key := Key;
end;

procedure TListMap.PutKeyVal(const Index : Indicator; const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PLNode(Index).Key);
  CopyVariable(Key, PLNode(Index).Key);
end;

procedure TListMap.PutDatumRef(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PLNode(Index).Datum);
  PLNode(Index).Datum := Datum;
end;

procedure TListMap.PutDatumVal(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PLNode(Index).Datum);
  CopyVariable(Datum, PLNode(Index).Datum);
end;

procedure TListMap.SetCapacity(NewCapacity : Cardinal);
begin
  // do nothing
end;

function TListMap.Valid(const Index : Indicator) : Boolean;
begin
  Result := Index <> nil;
end;

{ THATMap }

const
  InsertionSortThreshold = 10;

type
  PListItem = ^TListItem;
  TListItem = record
    Key : Variable;
    Datum : Variable;
  end;

function THATMap.AddItemVal(const Key, Datum : Variable) : Indicator;
var
  P : PListItem;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  if not Sorted then
    Result := Indicator(FCount)
  else if Find(Key, Result) then
    case Duplicates of
      dupIgnore :
        begin
          Invalidate(Result);
          Exit;
        end;
      dupError :
        begin
          Invalidate(Result);
          Error(SDuplicateKey);
        end;
    end;
  P := TAllocator(MA).Allocate;
  CopyVariable(Key, P.Key);
  CopyVariable(Datum, P.Datum);
  FList.Insert(Cardinal(Result), P);
  Inc(FCount);
end;

function THATMap.AddItemRef(const Key, Datum : Variable) : Indicator;
var
  P : PListItem;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  if not Sorted then
    Result := Indicator(FCount)
  else if Find(Key, Result) then
    case Duplicates of
      dupIgnore :
        begin
          Invalidate(Result);
          Exit;
        end;
      dupError :
        begin
          Invalidate(Result);
          Error(SDuplicateKey);
        end;
    end;
  P := TAllocator(MA).Allocate;
  P.Key := Key;
  P.Datum := Datum;
  FList.Insert(Cardinal(Result), P);
  Inc(FCount);
end;

procedure THATMap.Clear;
var
  i : Longint;
begin
  if FCount <= 0 then
    Exit;
  for i := 1 to FCount do
  begin
    ClearVariable(PListItem(FList[i - 1])^.Key);
    ClearVariable(PListItem(FList[i - 1])^.Datum);
    TAllocator(MA).Dispose(FList[i - 1]);
    FList[i - 1] := nil;
  end;
  FList.Clear;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
  FCount := 0;
end;

constructor THATMap.Create;
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(TListItem));
  FList := THATList.Create;
end;

procedure THATMap.Delete(var Index : Indicator);
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Key);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Datum);
  TAllocator(MA).Dispose(FList[Cardinal(Index)]);
  FList.Delete(Cardinal(Index));
  Dec(FCount);
  Invalidate(Index);
end;

procedure THATMap.DeleteByKey(const Key : Variable);
var
  Loc : Indicator;
begin
  Loc := IndexOfKey(Key);
  if Valid(Loc) then
    Delete(Loc);
end;

procedure THATMap.DeleteByDatum(const Datum : Variable);
var
  i : Indicator;
begin
  i := IndexOfDatum(Datum);
  if Valid(i) then
  begin
    Delete(i);
  end;
end;

destructor THATMap.Destroy;
begin
  Clear;
  FList.Free;
  TAllocator(MA).Free;
  inherited Destroy;
end;

procedure THATMap.Exchange(Here, There : Indicator);
begin
  if not Valid(Here) then
    Error(SInvalidIndicator);
  if not Valid(There) then
    Error(SInvalidIndicator);
  FList.Exchange(Cardinal(Here), Cardinal(There));
end;

function THATMap.Find(const Key : Variable; var Index : Indicator) : Boolean;
var
  l, h, i, c, n : Longint;
begin
  Result := False;
  if FCount = 0 then
  begin
    Index := Indicator(0);
    Exit;
  end;
  //??CheckKey(Key);
  if not Sorted then
  begin
    for n := 0 to FCount - 1 do
    begin
      if CompareVariables(PListItem(FList[n]).Key, Key, Ansi, CaseSensitive) = 0 then
      begin
        Result := True;
        Index := Indicator(n);
        Exit;
      end;
    end;
    Invalidate(Index);
  end
  else
  begin
    l := 0;
    h := FCount - 1;
    while l <= h do
    begin
      i := (l + h) shr 1;
      c := CompareVariables(PListItem(FList[i]).Key, Key, Ansi, CaseSensitive);
      if c < 0 then
        l := i + 1
      else
      begin
        h := i - 1;
        if c = 0 then
        begin
          Result := True;
          if Duplicates <> dupAccept then
            l := i;
        end;
      end;
    end;
    Index := Indicator(l);
  end;
end;

function THATMap.First : Indicator;
begin
  SP := Indicator(0);
  Result := SP;
end;

function THATMap.GetCapacity : Cardinal;
begin
  Result := FList.Capacity;
end;

function THATMap.GetKeyRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PListItem(FList[Cardinal(Index)]).Key;
end;

function THATMap.GetKeyVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PListItem(FList[Cardinal(Index)]).Key, Result);
end;

function THATMap.GetDatumRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PListItem(FList[Cardinal(Index)]).Datum;
end;

function THATMap.GetDatumVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PListItem(FList[Cardinal(Index)]).Datum, Result);
end;

procedure THATMap.InsertionSort(l, r : Integer);
var
  i : Longint;
  J : Longint;
  v : TListItem;
begin
  for i := l + 1 to r do
    if CompareVariables(PListItem(FList[i]).Key, PListItem(FList[l]).Key, Ansi, CaseSensitive) < 0 then
      FList.Exchange(l, i);
  for i := l + 2 to r do
  begin
    J := i;
    v := PListItem(FList[i])^;
    while CompareVariables(v.Key, PListItem(FList[J - 1]).Key, Ansi, CaseSensitive) < 0 do
    begin
      PListItem(FList[J])^ := PListItem(FList[J - 1])^;
      Dec(J);
    end;
    PListItem(FList[J])^ := v;
  end;
end;

procedure THATMap.InsertItemVal(Index : Indicator; Key, Datum : array of const);
var
  P : PListItem;
begin
  if (High(Key) <> 0) or (High(Datum) <> 0) then
    Error(SInvalidArgument);
{$IFDEF TYPECHECK}
  CheckKey(Variable(Key[0]));
  CheckDatum(Variable(Datum[0]));
{$ENDIF}
  if Sorted then
    Error(SInvalidIfSorted);
  if not Valid(Index) then
    Error(SInvalidIndicator);
  P := TAllocator(MA).Allocate;
  CopyVariable(Variable(Key[Low(Key)]), P.Key);
  CopyVariable(Variable(Datum[Low(Datum)]), P.Datum);
  FList.Insert(Cardinal(Index), P);
  Inc(FCount);
end;

procedure THATMap.Invalidate(var Index : Indicator);
begin
  Index := Indicator(High(Cardinal));
end;

function THATMap.Last : Indicator;
begin
  SP := Indicator(FCount - 1);
  Result := SP;
end;

procedure THATMap.Move(Here, There : Indicator);
begin
  if not Valid(Here) then
    Error(SInvalidIndicator);
  if not Valid(There) then
    Error(SInvalidIndicator);
  FList.Move(Cardinal(Here), Cardinal(There));
end;

function THATMap.Next : Indicator;
begin
  if not Valid(SP) then
    Result := First
  else if Cardinal(SP) < Pred(FCount) then
  begin
    SP := Indicator(Cardinal(SP) + 1);
    Result := SP;
  end
  else
  begin
    Invalidate(SP);
    Invalidate(Result);
  end;
end;

function THATMap.Prev : Indicator;
begin
  if not Valid(SP) then
    Result := Last
  else if Cardinal(SP) > 0 then
  begin
    SP := Indicator(Cardinal(SP) - 1);
    Result := SP;
  end
  else
  begin
    Invalidate(SP);
    Invalidate(Result);
  end;
end;

procedure THATMap.PutKeyRef(const Index : Indicator; const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if Sorted then
    Error(SInvalidIfSorted);
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Key);
  PListItem(FList[Cardinal(Index)]).Key := Key;
end;

procedure THATMap.PutKeyVal(const Index : Indicator; const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if Sorted then
    Error(SInvalidIfSorted);
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Key);
  CopyVariable(Key, PListItem(FList[Cardinal(Index)]).Key);
end;

procedure THATMap.PutDatumRef(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Datum);
  PListItem(FList[Cardinal(Index)]).Datum := Datum;
end;

procedure THATMap.PutDatumVal(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PListItem(FList[Cardinal(Index)]).Datum);
  CopyVariable(Datum, PListItem(FList[Cardinal(Index)]).Datum);
end;

procedure THATMap.QuickSort(l, r : Integer);
var
  i, J : Longint;
  P : Variable;
begin
  if r - l <= InsertionSortThreshold then
  begin
    InsertionSort(l, r);
    Exit;
  end;
  repeat
    i := l;
    J := r;
    P := PListItem(FList[(l + r) shr 1]).Key;
    repeat
      while CompareVariables(PListItem(FList[i]).Key, P, Ansi, CaseSensitive) < 0 do
        Inc(i);
      while CompareVariables(PListItem(FList[J]).Key, P, Ansi, CaseSensitive) > 0 do
        Dec(J);
      if i <= J then
      begin
        FList.Exchange(i, J);
        Inc(i);
        Dec(J);
      end;
    until i > J;
    if l < J then
      QuickSort(l, J);
    l := i;
  until i >= r;
end;

procedure THATMap.Reverse(SetSorted : Boolean);
var
  n, i : Longint;
begin
  if Sorted then
    Exit;
  n := 1;
  i := FCount;
  while n < i do
  begin
    FList.Exchange(n - 1, i - 1);
    Inc(n);
    Dec(i);
  end;
  if SetSorted then
    FSorted := True;
end;

procedure THATMap.SetAnsi(Value : Boolean);
var
  n : Longint;
  i : Indicator;
  J : Indicator;
begin
  if FCount = 0 then
  begin
    FAnsi := Value;
    Exit;
  end;
  if FAnsi <> Value then
  begin
    FAnsi := Value;
    if Sorted then
    begin
      // re-sort with new ordering
      Sort;
      if Duplicates <> dupAccept then
      begin
        // remove any new duplicates
        n := Count - 1;
        while n > 0 do
        begin
          i := Indicator(n - 1);
          J := Indicator(n);
          if CompareVariables(GetKeyRef(i), GetKeyRef(J), Ansi, CaseSensitive) = 0 then
            Delete(J);
          Dec(n);
        end;
      end;
    end;
  end;
end;

procedure THATMap.SetCapacity(NewCapacity : Cardinal);
begin
  FList.Capacity := NewCapacity;
end;

procedure THATMap.SetCaseSensitive(Value : Boolean);
var
  n : Longint;
  i : Indicator;
  J : Indicator;
begin
  if FCount = 0 then
  begin
    FCaseSensitive := Value;
    Exit;
  end;
  if FCaseSensitive <> Value then
  begin
    FCaseSensitive := Value;
    if Sorted then
    begin
      // re-sort with new ordering
      Sort;
      if Duplicates <> dupAccept then
      begin
        // remove any new duplicates
        n := Count - 1;
        while n > 0 do
        begin
          i := Indicator(n - 1);
          J := Indicator(n);
          if CompareVariables(GetKeyRef(i), GetKeyRef(J), Ansi, CaseSensitive) = 0 then
            Delete(J);
          Dec(n);
        end;
      end;
    end;
  end;
end;

procedure THATMap.SetDuplicates(Value : TDuplicates);
var
  n : Longint;
  i : Indicator;
  J : Indicator;
begin
  if FCount = 0 then
  begin
    FDuplicates := Value;
    Exit;
  end;
  if FDuplicates <> Value then
  begin
    if (FDuplicates = dupAccept) and Sorted then
    begin
      // remove any new duplicates
      n := Count - 1;
      while n > 0 do
      begin
        i := Indicator(n - 1);
        J := Indicator(n);
        if CompareVariables(GetKeyRef(i), GetKeyRef(J), Ansi, CaseSensitive) = 0 then
          Delete(J);
        Dec(n);
      end;
    end;
    FDuplicates := Value;
  end;
end;

procedure THATMap.SetSorted(Value : Boolean);
begin
  if FSorted <> Value then
  begin
    if Value then
      Sort;
    FSorted := Value;
  end;
end;

procedure THATMap.Sort;
begin
  if not Sorted and (FCount > 1) then
  begin
    QuickSort(0, FCount - 1);
  end;
end;

function THATMap.Valid(const Index : Indicator) : Boolean;
begin
  //Result := (Cardinal(Index) < FCount);
  Result := (Cardinal(Index) <> High(Cardinal));
end;

{ THashMap }

// from  Prime v1.2 (C) 1997 Allen Cheng. All Rights Reserved.
// Homepage:  http://www.geocities.com/SiliconValley/Park/8979/
// Email:     ac@4u.net

function IsPrime(Num : Longint) : Boolean;
var
  x : Longint;
  i : Integer;
const
  y : array[1..25] of Integer = (2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97);
  {All Primes from 1 to 100}

begin
  Result := True;
  if Num = 1 then
  begin
    Result := False;
    Exit;
  end;
  for i := 1 to 25 do
    if ((Num mod y[i]) = 0) and (Num <> y[i]) then
    begin
      Result := False;
      Exit;
    end;
  x := 99; { The Prime After 97 is 101 ( 99+2 = 101 )}
  while (Sqr(x) < Num) do
  begin
    x := x + 2; { No need to check Even numbers }
    if (Num mod x) = 0 then
    begin
      Result := False;
      Exit;
    end;
  end;
end;

function GuessCapacity(Capacity : Longint) : Longint;
begin
  Result := Capacity + ((Capacity + 1) mod 2); // make sure it's odd
  while not IsPrime(Result) do
  begin
    Inc(Result, 2);
  end;
  if Capacity >= $3FFFFF then
  begin
    Result := $3FFFFE;
    while not IsPrime(Result) do
    begin
      Dec(Result, 2);
    end;
  end
end;

type
  PHNode = ^THNode;
  THNode = record
    Key : Variable;
    Datum : Variable;
    Next : PHNode;
  end;

type
  PArray = ^TArray;
  TArray = array[0..(MaxLongint div SizeOf(PHNode)) - 1] of PHNode;

  // There's skullduggery afoot in these methods! The hash array
  // either contains nil or a pointer to a singly-linked list for
  // collision resolution. The HashMap indicator is basically a
  // pointer directly to the item in question which speeds up that
  // kind of access but leaves us with a problem: how do we get
  // from the indicator to the array for such operations as deletion
  // etc. This is where the trick comes in. Honest pointers from
  // Delphi's heap are byte-aligned (I think... but the details
  // don't matter) and thus always even when cast as numbers. This
  // lets us use odd valued indicators to store something else--
  // in this case an index into the hash array. Each link list is
  // terminated by an imitation pointer which, with suitable bit
  // shifting locates the head of the list.
  // This whole approach relies upon the list being very short--if
  // for any reason the list grow long all that running up and down
  // lists will really slow things down. But, if the hash array has
  // the recommended Capacity (i.e., roughly equal to Count) and the
  // hash function is behaving well lists should not be more than
  // four or five items long and usually less.

function THashMap.AddItemVal(const Key, Datum : Variable) : Indicator;
var
  ALoc : Cardinal;
  P : PHNode;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  ALoc := ScatterVariable(Key, Ansi, CaseSensitive) mod FCapacity;
  if PArray(FArray)^[ALoc] = nil then
  begin
    PArray(FArray)^[ALoc] := TAllocator(MA).Allocate;
    CopyVariable(Key, PArray(FArray)^[ALoc].Key);
    CopyVariable(Datum, PArray(FArray)^[ALoc].Datum);
    // terminate the list with a magic pointer back to the hash array
    PArray(FArray)^[ALoc].Next := PHNode((ALoc shl 1) + 1);
    Result := PArray(FArray)^[ALoc];
  end
  else
  begin
    P := TAllocator(MA).Allocate;
    CopyVariable(Key, P.Key);
    CopyVariable(Datum, P.Datum);
    P.Next := PArray(FArray)^[ALoc];
    PArray(FArray)^[ALoc] := P;
    Result := PArray(FArray)^[ALoc];
  end;
  Inc(FCount);
end;

function THashMap.AddItemRef(const Key, Datum : Variable) : Indicator;
var
  ALoc : Cardinal;
  P : PHNode;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  ALoc := ScatterVariable(Key, Ansi, CaseSensitive) mod FCapacity;
  if PArray(FArray)^[ALoc] = nil then
  begin
    PArray(FArray)^[ALoc] := TAllocator(MA).Allocate;
    PArray(FArray)^[ALoc].Key := Key;
    PArray(FArray)^[ALoc].Datum := Datum;
    // terminate the list with a magic pointer back to the hash array    PArray(FArray)^[ALoc].Next := PHNode((ALoc shl 1) + 1);
    Result := PArray(FArray)^[ALoc];
  end
  else
  begin
    P := TAllocator(MA).Allocate;
    P.Key := Key;
    P.Datum := Datum;
    P.Next := PArray(FArray)^[ALoc];
    PArray(FArray)^[ALoc] := P;
    Result := PArray(FArray)^[ALoc];
  end;
  Inc(FCount);
end;

procedure THashMap.Clear;
var
  n : Cardinal;
  P : PHNode;
begin
  for n := 0 to FCapacity - 1 do
  begin
    if PArray(FArray)^[n] <> nil then
    begin
      // the list terminator is identified by being odd
      while not Odd(Cardinal(PArray(FArray)^[n].Next)) do
      begin
        P := PHNode(PArray(FArray)^[n]);
        PArray(FArray)^[n] := P.Next;
        ClearVariable(P.Key);
        ClearVariable(P.Datum);
        TAllocator(MA).Dispose(P);
      end;
      ClearVariable(PArray(FArray)^[n].Key);
      ClearVariable(PArray(FArray)^[n].Datum);
      TAllocator(MA).Dispose(PArray(FArray)^[n]);
      PArray(FArray)^[n] := nil;
    end;
  end;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
  FCount := 0;
end;

constructor THashMap.Create;
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(THNode));
  SetCapacity(17);
end;

constructor THashMap.CreateWithCapacity(ACapacity : Cardinal);
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(THNode));
  FCapacity := GuessCapacity(ACapacity);
  GetMem(FArray, (FCapacity * SizeOf(PHNode)));
  FillChar(FArray^, FCapacity * SizeOf(PHNode), 0);
end;

procedure THashMap.Delete(var Index : Indicator);
var
  ALoc : Cardinal;
  P : PHNode;
  n : PHNode;
begin
  // first go to the end of the chain to find the array location
  P := PHNode(Index);
  while not Odd(Cardinal(P.Next)) do
    P := P.Next;
  // find the array index buried in the imitation pointer
  ALoc := Cardinal(P.Next) shr 1;
  if PArray(FArray)^[ALoc] = Index then
  begin
    // already at the last Node
    ClearVariable(P.Key);
    ClearVariable(P.Datum);
    TAllocator(MA).Dispose(P);
    PArray(FArray)^[ALoc] := nil;
  end
  else
  begin
    // now find the link to node to be deleted
    n := PArray(FArray)^[ALoc];
    P := n.Next;
    while P <> Index do
    begin
      n := P;
      P := n.Next;
    end;
    n.Next := P.Next;
    ClearVariable(PHNode(Index).Key);
    ClearVariable(PHNode(Index).Datum);
    TAllocator(MA).Dispose(Index);
  end;
  Dec(FCount);
  Invalidate(Index);
end;

procedure THashMap.DeleteByKey(const Key : Variable);
var
  i : Indicator;
begin
  if Find(Key, i) then
    Delete(i);
end;

procedure THashMap.DeleteByDatum(const Datum : Variable);
var
  i : Indicator;
begin
  i := IndexOfDatum(Datum);
  if Valid(i) then
  begin
    Delete(i);
  end;
end;

destructor THashMap.Destroy;
begin
  Clear;
  FreeMem(FArray);
  FCapacity := 0;
  TAllocator(MA).Free;
  inherited Destroy;
end;

function THashMap.Find(const Key : Variable; var Index : Indicator) : Boolean;
var
  ALoc : Cardinal;
  P : PHNode;
begin
  Result := False;
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  // hash the key
  ALoc := ScatterVariable(Key, Ansi, CaseSensitive) mod FCapacity;
  if PArray(FArray)^[ALoc] <> nil then
  begin
    // then do a linear search of the list
    P := PArray(FArray)^[ALoc];
    repeat
      if CompareVariables(P.Key, Key, Ansi, CaseSensitive) = 0 then
      begin
        Result := True;
        Index := Indicator(P);
        Exit;
      end;
      P := P.Next;
    until Odd(Cardinal(P));
  end;
  Invalidate(Index);
end;

// the navigation methods are awkward -- navigating a hash
// table doesn't make much sense really!

function THashMap.First : Indicator;
var
  ALoc : Cardinal;
begin
  ALoc := 0;
  // find the first list in the array
  while (PArray(FArray)^[ALoc] = nil) and (ALoc < FCapacity) do
    Inc(ALoc);
  if ALoc = FCapacity then
    Invalidate(SP)
  else
    SP := PArray(FArray)^[ALoc]; // the first item in the list
  Result := SP;
end;

function THashMap.GetCapacity : Cardinal;
begin
  Result := FCapacity;
end;

function THashMap.GetKeyRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PHNode(Index).Key;
end;

function THashMap.GetKeyVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PHNode(Index).Key, Result);
end;

function THashMap.GetDatumRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PHNode(Index).Datum;
end;

function THashMap.GetDatumVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PHNode(Index).Datum, Result);
end;

procedure THashMap.Invalidate(var Index : Indicator);
begin
  Index := nil;
end;

// See First

function THashMap.Last : Indicator;
var
  ALoc : Integer;
  P : PHNode;
begin
  // find the last list in the array
  ALoc := FCapacity - 1;
  while (ALoc >= 0) and (PArray(FArray)^[ALoc] = nil) do
    Dec(ALoc);
  if ALoc = 0 then
    Invalidate(SP)
  else
  begin
    // find the end of the list
    P := PArray(FArray)^[ALoc];
    while not Odd(Cardinal(P.Next)) do
      P := P.Next;
    SP := P;
  end;
  Result := SP;
end;

function THashMap.Next : Indicator;
var
  ALoc : Cardinal;
begin
  if Valid(SP) then
  begin
    // if the next item's odd we're at the end of a list
    if Odd(Cardinal(PHNode(SP).Next)) then
    begin
      // go looking for the next list
      ALoc := Cardinal(PHNode(SP).Next) shr 1;
      Inc(ALoc);
      while (ALoc < FCapacity) and (PArray(FArray)^[ALoc] = nil) do
      begin
        Inc(ALoc);
      end;
      if ALoc < FCapacity then
      begin
        SP := PArray(FArray)^[ALoc];
      end
      else
      begin
        Invalidate(SP);
      end;
    end
    else
      // just move one down the list
    begin
      SP := PHNode(SP).Next;
    end;
    begin
    end;
  end;
  Result := SP;
end;

function THashMap.Prev : Indicator;
var
  ALoc : Integer;
  P : PHNode;
  n : PHNode;
begin
  if Valid(SP) then
  begin
    // find the end of the current list
    P := SP;
    while not Odd(Cardinal(P.Next)) do
      P := P.Next;
    // now we can find the head of the list
    ALoc := Cardinal(P.Next) shr 1;
    // go looking for the previous item
    if PArray(FArray)^[ALoc] = SP then
    begin
      // we are at the top so we have to find the previous list
      Dec(ALoc);
      while (ALoc >= 0) and (PArray(FArray)^[ALoc] = nil) do
      begin
        Dec(ALoc);
      end;
      // and find the end of that list
      if ALoc >= 0 then
      begin
        P := PArray(FArray)^[ALoc];
        while not Odd(Cardinal(P.Next)) do
          P := P.Next;
        SP := P;
        Result := SP;
      end
      else
      begin
        Invalidate(SP);
        Invalidate(Result);
      end;
    end
    else
    begin
      // find the predecessor
      n := PArray(FArray)^[ALoc];
      P := n.Next;
      while P <> SP do
      begin
        n := P;
        P := n.Next;
      end;
      SP := n;
      Result := SP;
    end;
  end;
end;

procedure THashMap.PutKeyVal(const Index : Indicator; const Key : Variable);
begin
  Error(SCannotChangeKey);
end;

procedure THashMap.PutKeyRef(const Index : Indicator; const Key : Variable);
begin
  Error(SCannotChangeKey);
end;

procedure THashMap.PutDatumRef(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PHNode(Index).Datum);
  PHNode(Index).Datum := Datum;
end;

procedure THashMap.PutDatumVal(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PHNode(Index).Datum);
  CopyVariable(Datum, PHNode(Index).Datum);
end;

procedure THashMap.SetCapacity(NewCapacity : Cardinal);
var
  NewMap : THashMap;
  TmpMap : PArray;
  TmpCap : Cardinal;
  TmpCount : Cardinal;
  i : Indicator;
  Key, Datum : Variable;
begin
  if FArray = nil then
    // Capacity hasn't been set yet so it's easy
  begin
    FCapacity := GuessCapacity(NewCapacity);
    GetMem(FArray, (FCapacity * SizeOf(PHNode)));
    FillChar(FArray^, FCapacity * SizeOf(PHNode), 0);
  end
  else
  begin
    // otherwise we have to build a new map
    NewMap := THashMap.CreateWithCapacity(NewCapacity);
    try
      if FCount <> 0 then
        // if the curent map isn't empty we have to move
        // everything over to the new one
      begin
        i := First;
        while Valid(i) do
        begin
          Key := GetKeyRef(i);
          Datum := GetDatumRef(i);
          NewMap.AddItemVal(Key, Datum);
          i := Next;
        end;
      end;
      // then we do some swapping to move the innards of
      // the new map into the old one and vice versa
      tmpMap := FArray;
      FArray := NewMap.FArray;
      NewMap.FArray := tmpMap;
      tmpCap := FCapacity;
      FCapacity := NewMap.FCapacity;
      NewMap.FCapacity := tmpCap;
      tmpCount := FCount;
      FCount := NewMap.FCount;
      NewMap.FCount := tmpCount;
    finally
      // the idea being that we can free the new map
      // and "really" be getting rid of the old one
      NewMap.Free;
    end;
  end;
end;

function THashMap.Valid(const Index : Indicator) : Boolean;
begin
  Result := Index <> nil;
end;

// TPStack is a simple stack of pointers. We use it to
// remove recursion in some of the tree operations.

type
  TPStack = class(TObject)
  private
    FCount : Cardinal;
    MA : Pointer;
    Head : Pointer;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Peek : Pointer;
    function Pop : Pointer;
    procedure Push(Item : Pointer);
    property Count : Cardinal Read FCount;
  end;

  { TPStack }

type
  PPSNode = ^TPSNode;
  TPSNode = record
    Next : PPSNode;
    Item : Pointer;
  end;

procedure TPStack.Clear;
var
  P : PPSNode;
begin
  while FCount > 0 do
  begin
    P := PPSNode(Head).Next;
    TAllocator(MA).Dispose(Head);
    Head := P;
    Dec(FCount);
  end;
  Head := nil;
end;

constructor TPStack.Create;
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(TPSNode));
  Head := nil;
  FCount := 0;
end;

destructor TPStack.Destroy;
begin
  Clear;
  TAllocator(MA).Free;
  inherited Destroy;
end;

function TPStack.Peek : Pointer;
begin
  Result := PPSNode(Head).Item;
end;

function TPStack.Pop : Pointer;
var
  P : PPSNode;
begin
  Result := PPSNode(Head).Item;
  P := PPSNode(Head).Next;
  TAllocator(MA).Dispose(Head);
  Head := P;
  Dec(FCount);
end;

procedure TPStack.Push(Item : Pointer);
var
  P : PPSNode;
begin
  P := TAllocator(MA).Allocate;
  P.Item := Item;
  P.Next := Head;
  Head := P;
  Inc(FCount);
end;

{ TAbstractTree }

type
  PNode = ^TNode;
  TNode = record
    Key : Variable;
    Datum : Variable;
    Left : PNode;
    Right : PNode;
    Code : Cardinal; // gets used differently in the Tree and the Treap
  end;

procedure RotateWithLeftChild(var K2 : PNode);
var
  K1 : PNode;
begin
  K1 := K2.Left;
  K2.Left := K1.Right;
  K1.Right := K2;
  K2 := K1;
end;

procedure RotateWithRightChild(var K1 : PNode);
var
  K2 : PNode;
begin
  K2 := K1.Right;
  K1.Right := K2.Left;
  K2.Left := K1;
  K1 := K2;
end;

constructor TAbstractTree.Create;
begin
  inherited Create;
  MA := TAllocator.Create(SizeOf(TNode));
  TraversalStack := TPStack.Create;
end;

procedure TAbstractTree.DeleteByDatum(const Datum : Variable);
var
  i : Indicator;
begin
  i := IndexOfDatum(Datum);
  if Valid(i) then
  begin
    DeleteByKey(GetKeyRef(i));
  end;
end;

destructor TAbstractTree.Destroy;
begin
  TPStack(TraversalStack).Free;
  TAllocator(MA).Free;
  inherited Destroy;
end;

// we run to the smallest element in the tree
// and leave things set up for Next

function TAbstractTree.First : Indicator;
var
  AlreadyFound : Boolean;
begin
  TPStack(TraversalStack).Clear;
  AlreadyFound := False;
  SP := Root;
  Result := NullNode;
  repeat
    while SP <> NullNode do
    begin
      TPStack(TraversalStack).Push(SP);
      SP := PNode(SP).Left;
    end;
    if TPStack(TraversalStack).Count > 0 then
    begin
      SP := TPStack(TraversalStack).Pop;
      if AlreadyFound then
        Exit
      else
      begin
        Result := SP;
        AlreadyFound := True;
      end;
      SP := PNode(SP).Right;
    end;
  until (TPStack(TraversalStack).Count = 0) and (SP = NullNode);
  TPSTack(TraversalStack).Clear;
end;

function TAbstractTree.GetCapacity : Cardinal;
begin
  Result := High(Cardinal);
end;

function TAbstractTree.GetKeyRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PNode(Index).Key;
end;

function TAbstractTree.GetKeyVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PNode(Index).Key, Result);
end;

function TAbstractTree.GetDatumRef(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  Result := PNode(Index).Datum;
end;

function TAbstractTree.GetDatumVal(const Index : Indicator) : Variable;
begin
  if not Valid(Index) then
    Error(SInvalidIndicator);
  CopyVariable(PNode(Index).Datum, Result);
end;

procedure TAbstractTree.Invalidate(var Index : Indicator);
begin
  Index := NullNode;
end;

function TAbstractTree.Last : Indicator;
var
  AlreadyFound : Boolean;
begin
  TPStack(TraversalStack).Clear;
  AlreadyFound := False;
  SP := Root;
  Result := NullNode;
  repeat
    while SP <> NullNode do
    begin
      TPStack(TraversalStack).Push(SP);
      SP := PNode(SP).Right;
    end;
    if TPStack(TraversalStack).Count > 0 then
    begin
      SP := TPStack(TraversalStack).Pop;
      if AlreadyFound then
        Exit
      else
      begin
        Result := SP;
        AlreadyFound := True;
      end;
      SP := PNode(SP).Left;
    end;
  until (TPStack(TraversalStack).Count = 0) and (SP = NullNode);
  TPStack(TraversalStack).Clear;
end;

function TAbstractTree.Next : Indicator;
begin
  Result := SP;
  if SP = NullNode then
  begin
    TPStack(TraversalStack).Clear;
    Exit;
  end;
  SP := PNode(SP).Right;
  repeat
    while SP <> NullNode do
    begin
      TPStack(TraversalStack).Push(SP);
      SP := PNode(SP).Left;
    end;
    if TPStack(TraversalStack).Count > 0 then
    begin
      SP := TPStack(TraversalStack).Pop;
      Exit
    end;
  until (TPStack(TraversalStack).Count = 0) and (SP = NullNode);
end;

function TAbstractTree.Prev : Indicator;
begin
  Result := SP;
  if SP = NullNode then
  begin
    TPStack(TraversalStack).Clear;
    Exit;
  end;
  SP := PNode(SP).Left;
  repeat
    while SP <> NullNode do
    begin
      TPStack(TraversalStack).Push(SP);
      SP := PNode(SP).Right;
    end;
    if TPStack(TraversalStack).Count > 0 then
    begin
      SP := TPStack(TraversalStack).Pop;
      Exit
    end;
  until (TPStack(TraversalStack).Count = 0) and (SP = NullNode);
end;

procedure TAbstractTree.PutKeyRef(const Index : Indicator;
  const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PNode(Index).Key);
  PNode(Index).Key := Key;
end;

procedure TAbstractTree.PutKeyVal(const Index : Indicator; const Key : Variable);
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PNode(Index).Key);
  CopyVariable(Key, PNode(Index).Key);
end;

procedure TAbstractTree.PutDatumRef(const Index : Indicator;
  const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PNode(Index).Datum);
  PNode(Index).Datum := Datum;
end;

procedure TAbstractTree.PutDatumVal(const Index : Indicator; const Datum : Variable);
begin
{$IFDEF TYPECHECK}
  CheckDatum(Datum);
{$ENDIF}
  if not Valid(Index) then
    Error(SInvalidIndicator);
  ClearVariable(PNode(Index).Datum);
  CopyVariable(Datum, PNode(Index).Datum);
end;

procedure TAbstractTree.SetCapacity(NewCapacity : Cardinal);
begin
  // do nothing;
end;

function TAbstractTree.Valid(const Index : Indicator) : Boolean;
begin
  Result := Index <> NullNode;
end;

{ TTreeMap }

function TTreeMap.AddItemVal(const Key, Datum : Variable) : Indicator;
var
  KeyHash : Cardinal;
  P : PNode;
  X : PNode;
  sgn : Integer;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  Keyhash := HashVariable(Key, Ansi, CaseSensitive);
  if Root = NullNode then
  begin
    PNode(Root) := TAllocator(MA).Allocate;
    CopyVariable(Key, PNode(Root).Key);
    CopyVariable(Datum, PNode(Root).Datum);
    PNode(Root).Left := NullNode;
    PNode(Root).Right := NullNode;
    PNode(Root).Code := keyhash;
    Inc(FCount);
    Result := Root;
    Exit;
  end;
  P := Root;
  X := P;
  while X <> NullNode do
  begin
    P := X;
    if keyhash < X.Code then
      X := X.Left
    else if X.Code < keyhash then
      X := X.Right
    else
    begin
      sgn := CompareVariables(Key, X.Key, Ansi, CaseSensitive);
      if sgn < 0 then
        X := X.Left
      else if sgn > 0 then
        X := X.Right;
    end;
  end;
  X := TAllocator(MA).Allocate;
  CopyVariable(Key, X.Key);
  CopyVariable(Datum, X.Datum);
  X.Left := NullNode;
  X.Right := NullNode;
  X.Code := keyhash;
  Inc(FCount);
  Result := X;
  if keyhash < P.Code then
    P.Left := X
  else if P.Code < keyhash then
    P.Right := X
  else
  begin
    sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
    if sgn < 0 then
      P.Left := X
    else if sgn > 0 then
      P.Right := X;
  end;
end;

function TTreeMap.AddItemRef(const Key, Datum : Variable) : Indicator;
var
  KeyHash : Cardinal;
  P : PNode;
  X : PNode;
  sgn : Integer;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  Keyhash := HashVariable(Key, Ansi, CaseSensitive);
  if Root = NullNode then
  begin
    PNode(Root) := TAllocator(MA).Allocate;
    PNode(Root).Key := Key;
    PNode(Root).Datum := Datum;
    PNode(Root).Left := NullNode;
    PNode(Root).Right := NullNode;
    PNode(Root).Code := keyhash;
    Inc(FCount);
    Result := Root;
    Exit;
  end;
  P := Root;
  X := P;
  while X <> NullNode do
  begin
    P := X;
    sgn := keyhash - X.Code;
    if sgn < 0 then
      X := X.Left
    else if sgn > 0 then
      X := X.Right
    else
    begin
      sgn := CompareVariables(Key, X.Key, Ansi, CaseSensitive);
      if sgn < 0 then
        X := X.Left
      else if sgn > 0 then
        X := X.Right;
    end;
  end;
  X := TAllocator(MA).Allocate;
  X.Key := Key;
  X.Datum := Datum;
  X.Left := NullNode;
  X.Right := NullNode;
  X.Code := keyhash;
  Inc(FCount);
  Result := X;
  sgn := keyhash - P.Code;
  if sgn < 0 then
    P.Left := X
  else if sgn > 0 then
    P.Right := X
  else
  begin
    sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
    if sgn < 0 then
      P.Left := X
    else if sgn > 0 then
      P.Right := X;
  end;
end;

procedure TTreeMap.Clear;

  procedure _Clear(var P : PNode);
  begin
    if P <> NullNode then
    begin
      _Clear(P.Left);
      _Clear(P.Right);
      ClearVariable(P.Key);
      ClearVariable(P.Datum);
      TAllocator(MA).Dispose(P);
    end;
  end;

begin
  _Clear(PNode(Root));
  Root := NullNode;
  FCount := 0;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
end;

constructor TTreeMap.Create;
begin
  inherited Create;
  NullNode := TAllocator(MA).Allocate;
  PNode(NullNode).Left := NullNode;
  PNode(NullNode).Right := NullNode;
  Root := NullNode;
end;

procedure TTreeMap.DeleteByKey(const Key : Variable);
var
  KeyHash : Cardinal;

  function _FindMin(P : PNode) : PNode;
  begin
    if P.Left = NullNode then
      Result := P
    else
      Result := _FindMin(P.Left);
  end;

  procedure _Delete(var P : PNode);
  var
    Tmp : PNode;
    sgn : Integer;
  begin
    if P <> NullNode then
    begin
      if KeyHash < P.Code then
        _Delete(P.Left)
      else if KeyHash > P.Code then
        _Delete(P.Right)
      else
      begin
        sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
        if sgn < 0 then
          _Delete(P.Left)
        else if sgn > 0 then
          _Delete(P.Right)
        else
        begin
          if P.Left = NullNode then // only a right child
          begin
            Tmp := P;
            P := P.Right;
            ClearVariable(Tmp.Key);
            ClearVariable(Tmp.Datum);
            TAllocator(MA).Dispose(Tmp);
            Dec(FCount);
          end
          else if P.Right = NullNode then // only left child
          begin
            Tmp := P;
            P := P.Left;
            ClearVariable(Tmp.Key);
            ClearVariable(Tmp.Datum);
            TAllocator(MA).Dispose(Tmp);
            Dec(FCount);
          end
          else // two children
          begin
            Tmp := _FindMin(P.Right);
            ClearVariable(P.Key);
            CopyVariable(Tmp.Key, P.Key);
            ClearVariable(P.Datum);
            CopyVariable(Tmp.Datum, P.Datum);
            P.Code := Tmp.Code;
            _Delete(P.Right);
          end;
        end
      end;
    end;
  end;

begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  KeyHash := HashVariable(Key, Ansi, CaseSensitive);
  _Delete(PNode(Root));
end;

destructor TTreeMap.Destroy;
begin
  Clear;
  TAllocator(MA).Dispose(NullNode);
  inherited Destroy;
end;

function TTreeMap.Find(const Key : Variable; var Index : Indicator) : Boolean;
var
  P : PNode;
  h : Cardinal;
  sgn : Integer;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  h := HashVariable(Key, Ansi, CaseSensitive);
  Index := Root;
  P := Root;
  while P <> NullNode do
  begin
    if h < P.Code then
      P := P.Left
    else if P.Code < h then
      P := P.Right
    else
    begin
      sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
      if sgn < 0 then
        P := P.Left
      else if sgn > 0 then
        P := P.Right
      else
      begin
        Index := P;
        Result := True;
        Exit;
      end;
    end;
  end;
  Result := False;
  Invalidate(Index);
end;

{ TTreapMap }

function TTreapMap.AddItemVal(const Key, Datum : Variable) : Indicator;

  function _Insert(var P : PNode) : PNode;
  var
    sgn : Integer;
  begin
    Result := NullNode;
    if P = NullNode then
    begin
      P := TAllocator(MA).Allocate;
      CopyVariable(Key, P.Key);
      CopyVariable(Datum, P.Datum);
      P.Code := RandNum.Random32;
      P.Left := NullNode;
      P.Right := NullNode;
      Inc(FCount);
      Result := P;
    end
    else
    begin
      sgn := CompareVariables(Key, P.Key, FAnsi, FCaseSensitive);
      if sgn < 0 then
      begin
        Result := _Insert(P.Left);
        if P.Left.Code < P.Code then
          RotateWithLeftChild(P);
      end
      else if sgn > 0 then
      begin
        Result := _Insert(P.Right);
        if P.Right.Code < P.Code then
          RotateWithRightChild(P);
      end
      else if sgn = 0 then
      begin
        case Duplicates of
          dupIgnore :
            begin
              Result := NullNode;
              Exit;
            end;
          dupError :
            begin
              Result := NullNode;
              Error(SDuplicateKey);
              Exit; // just to put down a compler warning
            end;
        else
        end;
        Result := _Insert(P.Right);
        if P.Right.Code < P.Code then
          RotateWithRightChild(P);
      end;
    end;
  end;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  Result := _Insert(PNode(Root));
end;

function TTreapMap.AddItemRef(const Key, Datum : Variable) : Indicator;

  function _Insert(var P : PNode) : PNode;
  var
    sgn : Integer;
  begin
    Result := NullNode;
    if P = NullNode then
    begin
      P := TAllocator(MA).Allocate;
      P.Key := Key;
      P.Datum := Datum;
      P.Code := RandNum.Random32;
      P.Left := NullNode;
      P.Right := NullNode;
      Inc(FCount);
      Result := P;
    end
    else
    begin
      sgn := CompareVariables(Key, P.Key, FAnsi, FCaseSensitive);
      if sgn < 0 then
      begin
        Result := _Insert(P.Left);
        if P.Left.Code < P.Code then
          RotateWithLeftChild(P);
      end
      else if sgn > 0 then
      begin
        Result := _Insert(P.Right);
        if P.Right.Code < P.Code then
          RotateWithRightChild(P);
      end
      else if sgn = 0 then
      begin
        case Duplicates of
          dupIgnore :
            begin
              Result := NullNode;
              Exit;
            end;
          dupError :
            begin
              Result := NullNode;
              Error(SDuplicateKey);
              Exit; // just to put down a compler warning
            end;
        end;
        Result := _Insert(P.Right);
        if P.Right.Code < P.Code then
          RotateWithRightChild(P);
      end;
    end;
  end;

begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
  CheckDatum(Datum);
{$ENDIF}
  Result := _Insert(PNode(Root));
end;

procedure TTreapMap.Clear;

  procedure _Clear(var P : PNode);
  begin
    if P <> NullNode then
    begin
      _Clear(P.Left);
      _Clear(P.Right);
      ClearVariable(P.Key);
      ClearVariable(P.Datum);
      TAllocator(MA).Dispose(P);
    end;
  end;

begin
  _Clear(PNode(Root));
  Root := NullNode;
  FCount := 0;
{$IFDEF TYPECHECK}
  KeyType := vtVoid;
  DatumType := vtVoid;
{$ENDIF}
end;

constructor TTreapMap.Create;
begin
  inherited Create;
  RandNum := TRandGen.Create(1);
  if NullNode = nil then
  begin
    NullNode := TAllocator(MA).Allocate;
    PNode(NullNode).Left := NullNode;
    PNode(NullNode).Right := NullNode;
    PNode(NullNode).Code := High(Cardinal);
  end;
  Root := NullNode;
  FDuplicates := dupIgnore;
end;

procedure TTreapMap.DeleteByKey(const Key : Variable);

  procedure _Delete(var P : PNode);
  var
    sgn : Integer;
  begin
    if P <> NullNode then
    begin
      sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
      if sgn < 0 then
      begin
        _Delete(P.Left);
      end
      else if sgn > 0 then
      begin
        _Delete(P.Right);
      end
      else
      begin
        if P.Left.Code < P.Right.Code then
        begin
          RotateWithLeftChild(P)
        end
        else
        begin
          RotateWithRightChild(P)
        end;
        if P <> NullNode then
        begin
          _Delete(P);
        end
        else
        begin
          ClearVariable(P.Left.Key);
          ClearVariable(P.Left.Datum);
          TAllocator(MA).Dispose(P.Left);
          P.Left := NullNode;
        end;
      end;
    end;
  end;
begin
  _Delete(PNode(Root));
end;

destructor TTreapMap.Destroy;
begin
  Clear;
  TAllocator(MA).Dispose(NullNode);
  RandNum.Free;
  inherited Destroy;
end;

function TTreapMap.Find(const Key : Variable; var Index : Indicator) : Boolean;
var
  P : PNode;
  sgn : Integer;
begin
{$IFDEF TYPECHECK}
  CheckKey(Key);
{$ENDIF}
  Index := Root;
  P := Root;
  while P <> NullNode do
  begin
    sgn := CompareVariables(Key, P.Key, Ansi, CaseSensitive);
    if sgn < 0 then
      P := P.Left
    else if sgn > 0 then
      P := P.Right
    else
    begin
      Index := P;
      Result := True;
      Exit;
    end;
  end;
  Result := False;
  Invalidate(Index);
end;

initialization

  // so that FindClass recognizes out maps
  RegisterClasses([TStringList, TAbstractMap, THATMap, TListMap,
    THashMap, TTreeMap, TTreapMap]);
  CF_MAP := RegisterClipboardFormat('Delphi Map');

finalization

  UnRegisterClasses([TStringList, TAbstractMap, THATMap, TListMap,
    THashMap, TTreeMap, TTreapMap]);

end.

