{$INCLUDE OPTIONS.INC}

{*****************************************************************************}
{                                                                             }
{       Maps v0.93 Generic Associative Containers for Delphi 2, 3 & 4         }
{                                                                             }
{                 Copyright (c) 1999 Robert R. Marsh, S.J.                    }
{               & the British Province of the Society of Jesus                }
{                                                                             }
{                This source code may *not* be redistributed                  }
{                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  }
{                                                                             }
{       If you like Maps and find yourself using it please consider           }
{       making a donation to your favorite charity. I would also be           }
{       pleased if you would acknowledge Maps in any projects that            }
{       make use of it.                                                       }
{                                                                             }
{       Maps is supplied as is. The author disclaims all warranties,          }
{       expressed or implied, including, without limitation, the              }
{       warranties of merchantability and of fitness for any purpose.         }
{       The author assumes no liability for damages, direct or                }
{       consequential, which may result from the use of QDB.                  }
{                                                                             }
{                           rrm@sprynet.com                                   }
{                     http://home.sprynet.com/~rrm                            }
{                                                                             }
{*****************************************************************************}

(*

   The HATList unit implements a TList clone using a Hashed Array Tree as
   described by Edward Sitarski (Dr. Dobb's Journal, September 1996, 107-110).
   The advantage is an overall reduced memory requirement, much reduced memory
   fragmentation, and more efficient growth of the list. The trade-off is with
   access time which is about doubled.

   Fuller versions of this unit are available from my website under the name
   QLists.

*)

unit HATList;

interface

uses
  Windows,
  SysUtils,
  Classes;

{ THATList class }

const
  MaxBranchSize = (65536 div SizeOf(Pointer)) - 1;
  MaxListSize = MaxBranchSize * MaxBranchSize;

  SListCapacityError = 'Invalid capacity value (%d)';

type
  EHATListError = class(Exception);

type
  PLeafList = ^TLeafList;
  TLeafList = array[0..MaxBranchSize - 1] of Pointer;
  PTopList = ^TTopList;
  TTopList = array[0..MaxBranchSize - 1] of PLeafList;

type
  THATList = class(TObject)
  private
    FList : PTopList;
    FCount : Longint;
    FCapacity : Longint;
    LeafMask : Longint; { used to find the index into a leaf }
    LeafLength : Longint; { the length of the Leaf array       }
    LeafSize : Longint; { the memory-size of the Leaf        }
    TopSize : Longint; { the memory-size of the Top array   }
    Power : Longint; { the power of two giving the length }
    TopUsed : Longint; { the number of active leaves        }
    procedure AddLeaf;
    procedure SetPower(P : Longint);
  protected
    function Get(Index : Longint) : Pointer;
    procedure Grow; virtual;
    procedure Put(Index : Longint; Item : Pointer);
    procedure SetCapacity(NewCapacity : Longint);
    procedure SetCount(NewCount : Longint);
  public
    constructor Create;
    destructor Destroy; override;
    function Add(Item : Pointer) : Longint;
    procedure Clear;
    procedure Delete(Index : Longint);
    procedure Exchange(Index1, Index2 : Longint);
    function Expand : THATList;
    procedure Insert(Index : Longint; Item : Pointer);
    procedure Move(CurIndex, NewIndex : Longint);
    property Capacity : Longint Read FCapacity Write SetCapacity;
    property Count : Longint Read FCount Write SetCount;
    property Items[Index : Longint] : Pointer Read Get Write Put; Default;
  end;

implementation

{ THATList }

const
  PowerMin = 1;

constructor THATList.Create;
begin
  inherited Create;
  FCount := 0;
  FList := nil;
  TopUsed := 0;
  SetPower(PowerMin);
  SetCapacity(0);
end;

destructor THATList.Destroy;
begin
  while TopUsed > 0 do
  begin
    FreeMem(FList^[TopUsed - 1], LeafSize);
    Dec(TopUsed);
  end;
  if FList <> nil then
  begin
    FreeMem(FList, TopSize);
    FList := nil;
  end;
  inherited Destroy;
end;

function THATList.Add(Item : Pointer) : Longint;
begin
  Result := FCount;
  if Result = FCapacity then
    Grow;
  if (Result and LeafMask) = 0 then
    AddLeaf;
  FList^[(Result shr Power)]^[(Result and LeafMask)] := Item;
  Inc(FCount);
end;

procedure THATList.AddLeaf;
var
  NewLeaf : PLeafList;
begin
  GetMem(NewLeaf, LeafSize);
  FList^[TopUsed] := NewLeaf;
  Inc(TopUsed);
end;

procedure THATList.Clear;
begin
  while TopUsed > 0 do
  begin
    FreeMem(FList^[TopUsed - 1], LeafSize);
    Dec(TopUsed);
  end;
  FCount := 0;
  SetCapacity(0);
end;

procedure THATList.Delete(Index : Longint);
var
  i : Longint;
  amount : Longint;
begin
  amount := LeafLength - 1 - (Index and LeafMask);
  { move the first chunk }
  if amount > 0 then
    System.Move(FList^[(Index shr Power)]^[(Index + 1) and LeafMask],
      FList^[(Index shr Power)]^[Index and LeafMask], amount * SizeOf(Pointer));
  { then for each leaf upwards }
  for i := (Index shr Power) to TopUsed - 2 do
  begin
    { bring one item down }
    FList^[i]^[LeafLength - 1] := FList^[i + 1]^[0];
    { shift the rest by one }
    System.Move(FList^[i + 1]^[1], FList^[i + 1]^[0], LeafSize - SizeOf(Pointer));
  end;
  Dec(FCount);
  if (FCount = 0) or (((FCount - 1) shr Power) < (TopUsed - 1)) then
  begin
    FreeMem(FList^[TopUsed - 1], LeafSize);
    Dec(TopUsed);
  end;
end;

procedure THATList.Exchange(Index1, Index2 : Longint);
var
  Item : Pointer;
begin
  Item := FList^[(Index1 shr Power)]^[(Index1 and LeafMask)];
  FList^[(Index1 shr Power)]^[(Index1 and LeafMask)] := FList^[(Index2 shr Power)]^[(Index2 and LeafMask)];
  FList^[(Index2 shr Power)]^[(Index2 and LeafMask)] := Item;
end;

function THATList.Expand : THATList;
begin
  if FCount = FCapacity then
    Grow;
  Result := Self;
end;

function THATList.Get(Index : Longint) : Pointer;
begin
  Result := FList^[(Index shr Power)]^[(Index and LeafMask)];
end;

procedure THATList.Grow;
begin
  { SetCapacity will choose a suitable new value -- the list }
  { capacity grows by powers of two                          }
  SetCapacity(FCapacity + 1);
end;

procedure THATList.Insert(Index : Longint; Item : Pointer);
var
  i : Longint;
  amount : Longint;
begin
  if FCount = FCapacity then
    Grow;
  if (FCount and LeafMask) = 0 then
    AddLeaf;
  { for each leaf down to the place of insertion }
  for i := TopUsed - 1 downto ((Index shr Power) + 1) do
  begin
    { shift one place up }
    System.Move(FList^[i]^[0], FList^[i]^[1], LeafSize - SizeOf(Pointer));
    { bring one item up }
    FList^[i]^[0] := FList^[i - 1]^[LeafLength - 1];
  end;
  amount := LeafLength - 1 - (Index and LeafMask);
  { shift to make room for new item }
  System.Move(FList^[Index shr Power]^[(Index and LeafMask)],
    FList^[Index shr Power]^[(Index and LeafMask) + 1], amount * SizeOf(Pointer));
  { insert the item itself }
  FList^[(Index shr Power)]^[(Index and LeafMask)] := Item;
  Inc(FCount);
end;

procedure THATList.Move(CurIndex, NewIndex : Longint);
var
  Item : Pointer;
begin
  if CurIndex <> NewIndex then
  begin
    Item := Get(CurIndex);
    Delete(CurIndex);
    Insert(NewIndex, Item);
  end;
end;

procedure THATList.Put(Index : Longint; Item : Pointer);
begin
  FList^[(Index shr Power)]^[(Index and LeafMask)] := Item;
end;

{ this is one of the chief engines of the Hashed Array Tree ... }

procedure THATList.SetCapacity(NewCapacity : Longint);
var
  NewPower : Longint;
  NewSize : Longint;
  NewList : PTopList;
  NewLeaf : PLeafList;
  NewTopUsed : Longint;
  Ratio : Longint;
  i, J : Longint;

  function RecommendedPower(NewCapacity : Longint) : Longint;
  begin
    { compute the root of s to the nearest greater power of 2 }
    Result := PowerMin;
    while NewCapacity >= (1 shl (Result shl 1)) do
      Inc(Result);
  end;

begin
  { calculate the parameters of the 'new' hatlist }
  NewPower := RecommendedPower(NewCapacity);
  NewSize := (1 shl NewPower) * SizeOf(Pointer);
  NewCapacity := (1 shl (NewPower shl 1));
  if (NewCapacity < FCount) or (NewCapacity > MaxListSize) then
    raise EHATListError.CreateFmt(SListCapacityError, [NewCapacity]);
  if NewCapacity <> FCapacity then
  begin
    { begin to build a new hatlist }
    GetMem(NewList, NewSize);
    if FCount > 0 then
    begin
      { only relevant if the list has members }
      NewTopUsed := ((FCount - 1) shr NewPower) + 1;
      { how many old leaves fit into a new one }
      Ratio := (NewSize div LeafSize);
      { for each old leaf }
      for i := 0 to TopUsed - 1 do
      begin
        { if a new leaf is needed }
        if i mod Ratio = 0 then
        begin
          try
            { add a new leaf }
            GetMem(NewLeaf, NewSize);
          except
            on EOutOfMemory do
            begin { get rid of the partly built hatlist }
              J := i;
              Dec(J, Ratio);
              while J >= 0 do
                FreeMem(NewList^[J], NewSize);
              FreeMem(NewList, NewSize);
              raise;
            end
          else
            raise;
          end;
          { put the leaf into the tree }
          NewList^[i div Ratio] := NewLeaf;
        end;
        { move the old leaf to its place in the new }
        System.Move(FList^[i]^[0], NewList^[i div Ratio]^[(LeafLength * (i mod Ratio))], LeafSize);
        { get rid of the old leaf }
        FreeMem(FList^[i], LeafSize);
      end;
      TopUsed := NewTopUsed;
    end;
    { get rid of the old hatlist }
    if FList <> nil then
      FreeMem(FList, TopSize);
    { assign the new hatlist instead }
    FList := NewList;
    { adjust the hatlist parameters }
    SetPower(NewPower);
    FCapacity := NewCapacity;
  end;
end;

procedure THATList.SetCount(NewCount : Longint);
var
  i : Longint;
begin
  if NewCount > FCapacity then
    SetCapacity(NewCount);
  { if we are shrinking the list we blank out the unwanted }
  { items -- if they point to anything there'll be a leak  }
  if NewCount > FCount then
    for i := FCount to NewCount do
      FList^[(i shr Power)]^[(i and LeafMask)] := nil;
  FCount := NewCount;
end;

procedure THATList.SetPower(P : Longint);
begin
  Power := P;
  LeafLength := (1 shl Power);
  LeafSize := LeafLength * SizeOf(Pointer);
  LeafMask := LeafLength - 1;
  TopSize := LeafSize;
end;

end.

