{*****************************************************************************}
{                                                                             }
{       Maps v0.93 Generic Associative Containers for Delphi 2, 3 & 4         }
{                                                                             }
{                 Copyright (c) 1999 Robert R. Marsh, S.J.                    }
{               & the British Province of the Society of Jesus                }
{                                                                             }
{                This source code may *not* be redistributed                  }
{                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  }
{                                                                             }
{       If you like Maps and find yourself using it please consider           }
{       making a donation to your favorite charity. I would also be           }
{       pleased if you would acknowledge Maps in any projects that            }
{       make use of it.                                                       }
{                                                                             }
{       Maps is supplied as is. The author disclaims all warranties,          }
{       expressed or implied, including, without limitation, the              }
{       warranties of merchantability and of fitness for any purpose.         }
{       The author assumes no liability for damages, direct or                }
{       consequential, which may result from the use of QDB.                  }
{                                                                             }
{                           rrm@sprynet.com                                   }
{                     http://home.sprynet.com/~rrm                            }
{                                                                             }
{*****************************************************************************}

(*

   The Allocators unit defines a simple fixed-size memory re-allocator
   for Maps.

*)

unit Allocators;

{$DEFINE USEALLOCATOR} // if not GetMem is used directly

interface

// A siimple fixed-size memory re-allocator designed
// to be faster than Delphi's own and less prone to
// fragmentation.

// Allocates page-sized blocks from the Delphi heap
// as required and makes them available in chunks
// of the desired size. The chunks are guaranteed to
// be appropriately aligned for their size.

// If USEALLOCATOR is not DEFINEd allocation is
// redirected to the Delphi memory manager.

type
  TAllocator = class(TObject)
  private
    fPageList : Pointer;
    fNextFree : Pointer;
    FPageSize : Cardinal;
    fChunkSize : Cardinal;
    procedure Clear;
    function GrowPageList : Pointer;
  public
    constructor Create(ChunkSize : Integer);
    destructor Destroy; override;
    function Allocate : Pointer;
    procedure Dispose(P : Pointer);
  end;

implementation

uses
  Windows,
  Classes;

type
  PPointer = ^Pointer;

type
  PPageNode = ^TPageNode;
  TPageNode = record
    Page : Pointer;
    NChunks : Integer;
    Next : PPageNode;
  end;

var
  PageSize : Cardinal;

function GetPageSize : Cardinal;
var
  Info : TSystemInfo;
begin
  GetSystemInfo(Info);
  Result := Info.dwPageSize;
end;

constructor TAllocator.Create(ChunkSize : Integer);
begin
  inherited Create;
  if ChunkSize < SizeOf(Pointer) then
    ChunkSize := SizeOf(Pointer);
  fChunkSize := ChunkSize;
  FPageSize := (PageSize div fChunkSize) * fChunkSize;
  GetMem(fPageList, SizeOf(TPageNode));
  PPageNode(fPageList).Next := nil;
end;

destructor TAllocator.Destroy;
begin
  if fPageList <> nil then
    Clear;
  FreeMem(fPageList);
  inherited Destroy;
end;

procedure TAllocator.Clear;
var
  P : PPageNode;
begin
  while PPageNode(fPageList).Next <> nil do
  begin
    P := PPageNode(fPageList).Next;
    PPageNode(fPageList).Next := P.Next;
    FreeMem(P.Page);
    FreeMem(P);
  end;
end;

function TAllocator.Allocate : Pointer;
begin
{$IFDEF USEALLOCATOR}
  if fNextFree = nil then
    fNextFree := GrowPageList;
  Result := fNextFree;
  fNextFree := PPointer(fNextFree)^;
{$ELSE}
  GetMem(Result, FChunkSize);
{$ENDIF}
end;

procedure TAllocator.Dispose(P : Pointer);
begin
{$IFDEF USEALLOCATOR}
  PPointer(P)^ := fNextFree;
  fNextFree := P;
{$ELSE}
  FreeMem(P);
{$ENDIF}
end;

function TAllocator.GrowPageList : Pointer;
var
  P : PChar;
  i : Integer;
  T : PPageNode;
  NewPage : Pointer;
  Offset : Integer;
  NumChunks : Integer;
begin
  GetMem(NewPage, FPageSize);
  try
    FillChar(NewPage^, FPageSize, $CC);
    // we make sure that a page is aligned correctly for the chunks it holds
    Offset := Cardinal(NewPage) mod FChunkSize;
    NumChunks := FPageSize div fChunkSize;
    if Offset <> 0 then
      Dec(NumChunks);
    P := PChar(NewPage) + Offset;
    for i := 1 to pred(NumChunks) do
    begin
      PPointer(P)^ := PPointer(P + fChunkSize);
      Inc(P, fChunkSize);
    end;
    PPointer(P)^ := nil;
    Result := PChar(NewPage) + Offset;
    GetMem(T, SizeOf(TPageNode));
    T.Page := NewPage;
    T.Next := PPageNode(fPageList).Next;
    T.NChunks := NumChunks;
    PPageNode(fPageList).Next := T;
  except
    FreeMem(NewPage);
    //    Result := nil;
    raise;
  end;
end;

initialization

  PageSize := GetPageSize;

end.

