{ -------------------------------------------------------------------------------------}
{ A list view control that enables access to the new style types provieded by the      }
{ updated list view control.                                                           }
{ Copyright 1996, Brad Stowers.  All Rights Reserved.                                  }
{ This component can be freely used and distributed in commercial and private          }
{ environments, provied this notice is not modified in any way.                        }
{ -------------------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions at        }
{ bstowers@pobox.com or 72733,3374 on CompuServe.                                      }
{ The lateset version will always be available on the web at:                          }
{   http://www.pobox.com/~bstowers/delphi/                                             }
{ -------------------------------------------------------------------------------------}
{ Date last modified:  Feb. 7, 1997                                                    }
{ -------------------------------------------------------------------------------------}

{ -------------------------------------------------------------------------------------}
{ TExtListView v2.04                                                                   }
{ -------------------------------------------------------------------------------------}
{                                                                                      }
{ Description:                                                                         }
{   A list view control that enables access to the new style types provieded by the    }
{   updated list view control.  The updated list view is provided in the COMCTL32.DLL  }
{   file that comes with Microsoft's new internet software.  The version I have tested }
{   this component with is dated August 26, 1996 at 09:39.                             }
{                                                                                      }
{ Notes:                                                                               }
{   The new column stuff (see TLVColumnEx) can not be properly implemented without     }
{    changes to the TListColumn source in Source\VCL\ComCtrls.pas.  I may do that in   }
{    the future, but for now, you will have to use the new stuff manually with the     }
{    appropriate new functions, i.e. ListView_GetColumnEx, etc.  I suggest you stay    }
{    away from them unless you know what you are doing.                                }
{                                                                                      }
{   There are some serious limitations that I did not realize when I first released    }
{    the virtual mode feature.  If you are not using virtual mode (VirtualMode         }
{    set to TRUE) then you need not be concerned.  These problems only affect virutal  }
{    mode.  The problem is that since items are not stored by the list view, but       }
{    rather provided as needed by an event handler, the Items property does not        }
{    reflect the actual items in the list.  In fact, it is always empty.  Items.Count  }
{    will return the proper number of items, but anything trying to use Items[x] will  }
{    never get any valid data.  This affects several things, such as automatic drag    }
{    and drop (DragMode = dmAutomatic), the Selected property and more.  This is       }
{    because the implementation of TListView uses the Items property extensively.      }
{    I have had some limited success in overriding the Items property and supplying    }
{    my own methods for getting and setting TListItems for it, so I believe that I     }
{    will eventually be able to corret this problem.  However, I do not know when I    }
{    will be able to get to it, so for now I suggest you not use virtual mode unless   }
{    you completely understand all of the problems that may arise from it.             }
{ -------------------------------------------------------------------------------------}
{                                                                                      }
{ Revision History:                                                                    }
{ 1.00:  + Initial release.                                                            }
{ 2.00:  + Added SaveSettings property.  Allows automatic saving and restoring of      }
{          column order and sizes.                                                     }
{        + Now covers all new additions in the latest ActiveX SDK release.             }
{          Briefly, here's most of what's new:                                         }
{            Properties for Virtual Mode, subitem info, hot item, hot cursor, work     }
{            area and item checked.                                                    }
{            Methods for item spacing, get sub item at give coords, getting and        }
{            setting column order.                                                     }
{            New events for MarqueeBegin (bounding box selection), ItemActivate, and   }
{            owner data ("Virtual Mode") events.                                       }
{        + Now includes COMCTL32.DLL and API documentation ("Chapter 11.doc").         }
{        + Now includes far less comments than usual because I have no time.  If you   }
{          see something that looks wrong or doesn't make sense, email me.             }
{ 2.01:  + Added two new properties, AutoColumnSort and AutoSortAscending, that allow  }
{          the list view columns to be automatically sorted when the user clicks on    }
{          them.  Thanks to Peter Thvrnqvist (email NMA96PTH@lustudat.student.lu.se)   }
{          for writing the sorting code that I never seemed to get to.                 }
{        + There are some serious limitations to virtual mode.  See discussion in the  }
{          "Notes" section above.                                                      }
{ 2.02:  + Changed property AutoColumnSort to have three different states (acsNoSort,  }
{          acsSort, acsSortToggle). New one toggles sort order between ascending and   }
{          descending order.   By Christian Holzner <cholzner@ping.at>.                }
{ 2.03:  + Reworked the sorting routines to work with bigger numbers and to make it a  }
{          little smarter at detecting valid numbers.  Thanks for much of this to      }
{          Rod Cullison (RCullis@worldnet.att.net).  Also added basic date sorting     }
{          capabilities.                                                               }
{ 2.04:  + TLVColumnEx record had two members declared in the wrong order.  Thanks to  }
{          Remi Sellem for catching this.                                              }
{        + The COMCTL32.H did not include a macro for setting an item's checked state, }
{          only for getting it.  I have figured one out, and now the IsChecked         }
{          property is read/write not read-only.                                       }
{--------------------------------------------------------------------------------------}


unit ExtListView;

interface

{$IFNDEF WIN32}
  ERROR!  This unit only available for Delphi 2.0/3.0!!!
{$ENDIF}

uses
  Windows, Messages, Classes, Controls, ComCtrls, CommCtrl, SysUtils,
  Menus;


type
  TLVDispInfo = TLVDispInfoA; // Borland forgot this one.

const
  LVIF_INDENT             = $0010;
  LVIF_NORECOMPUTE        = $0800;

type
  PLVItemEx = ^TLVItemEx;
  TLVItemEx = packed record
    mask: UINT;
    iItem: Integer;
    iSubItem: Integer;
    state: UINT;
    stateMask: UINT;
    pszText: PAnsiChar;
    cchTextMax: Integer;
    iImage: Integer;
    lParam: LPARAM;
    iIndent: integer;
  end;

  PLVDispInfoEx = ^TLVDispInfoEx;
  TLVDispInfoEx = packed record
    hdr:   TNMHDR;
    item:  TLVItemEx;
  end;

const
  LVCF_IMAGE              = $0010; // index of image in the image list for column header.
  LVCF_ORDER              = $0020; // 0 based column offset, left to right order.

  LVCFMT_IMAGE            = $0800; // Item displays an image from an image list.
  LVCFMT_BITMAP_ON_RIGHT  = $1000; // Image appears to right of text.
  LVCFMT_COL_HAS_IMAGES   = $8000; // Undocumented.

type
  TLVColumnEx = packed record
    mask: UINT;
    fmt: Integer;
    cx: Integer;
    pszText: PAnsiChar;
    cchTextMax: Integer;
    iSubItem: Integer;
    iImage: integer; // New
    iOrder: integer; // New
  end;

  TAutoColumnSort = (acsNoSort,acsSort,acsSortToggle);

// These functions already exist, and there is no way to override them, so I'll just
// rename them and you can use them as best you can.
function ListView_GetColumnEx(LVWnd: HWND; iCol: Integer; var pcol: TLVColumnEx): Bool;
function ListView_SetColumnEx(LVWnd: HWnd; iCol: Integer; const pcol: TLVColumnEx): Bool;
function ListView_InsertColumnEx(LVWnd: HWND; iCol: Integer;
                                 const pcol: TLVColumnEx): Integer;


const
  LVM_GETHEADER           = LVM_FIRST + 31;

function ListView_GetHeader(LVWnd: HWnd): HWnd;

const
  LVM_SETICONSPACING      = LVM_FIRST + 53;

  // -1 for cx and cy means we'll use the default (system settings)
  // 0 for cx or cy means use the current setting (allows you to change just one param)
function ListView_SetIconSpacing(LVWnd: HWnd; cx, cy: integer): DWORD;

const
  LVS_EX_GRIDLINES             = $00000001;  // Report mode only.
  LVS_EX_SUBITEMIMAGES         = $00000002;  // Report mode only.
  LVS_EX_CHECKBOXES            = $00000004;
  LVS_EX_TRACKSELECT           = $00000008;
  LVS_EX_HEADERDRAGDROP        = $00000010;  // Report mode only.
  LVS_EX_FULLROWSELECT         = $00000020;  // Report mode only.
  LVS_EX_ONECLICKACTIVATE      = $00000040;
  LVS_EX_TWOCLICKACTIVATE      = $00000080;

  LVM_SETEXTENDEDLISTVIEWSTYLE = LVM_FIRST + 54;

function ListView_SetExtendedListViewStyle(LVWnd: HWnd; ExStyle: LPARAM): DWORD;

const
  LVM_GETEXTENDEDLISTVIEWSTYLE = LVM_FIRST + 55;

function ListView_GetExtendedListViewStyle(LVWnd: HWnd): DWORD;

const
  LVIR_BOUNDS             = 0;
  LVIR_ICON               = 1;
  LVIR_LABEL              = 2;
  LVIR_SELECTBOUNDS       = 3;

  LVM_GETSUBITEMRECT      = LVM_FIRST + 56;

function ListView_GetSubItemRect(LVWnd: HWnd; ParentItem, SubItem, Code: integer;
                                 var Rect: TRect): boolean;

const
  LVM_SUBITEMHITTEST      = LVM_FIRST + 57;

type
  PLVHitTestInfoEx = ^TLVHitTestInfoEx;
  TLVHitTestInfoEx = packed record
    pt: TPoint;
    flags: UINT;
    iItem: integer;
    iSubItem: integer;
  end;

function ListView_SubItemHitTest(LVWnd: HWnd; var HitTestInfo: TLVHitTestInfoEx): integer;

const
  LVM_SETCOLUMNORDERARRAY = LVM_FIRST + 58;

function ListView_SetColumnOrderArray(LVWnd: HWnd; Count: integer;
                                      const IntArray: array of integer): boolean;

const
  LVM_GETCOLUMNORDERARRAY = LVM_FIRST + 59;

function ListView_GetColumnOrderArray(LVWnd: HWnd; Count: integer;
                                      var IntArray: array of integer): boolean;

const
  LVM_SETHOTITEM  = LVM_FIRST + 60;

function ListView_SetHotItem(LVWnd: HWnd; Item: integer): integer;

const
  LVM_GETHOTITEM  = LVM_FIRST + 61;

function ListView_GetHotItem(LVWnd: HWnd): integer;

const
  LVM_SETHOTCURSOR  = LVM_FIRST + 62;

function ListView_SetHotCursor(LVWnd: HWnd; Cursor: HCursor): HCursor;

const
  LVM_GETHOTCURSOR  = LVM_FIRST + 63;

function ListView_GetHotCursor(LVWnd: HWnd): HCursor;

const
  LVM_APPROXIMATEVIEWRECT = LVM_FIRST + 64;

function ListView_ApproximateViewRect(LVWnd: HWnd; Width, Height, Count: integer): DWORD;

const
  LVM_SETWORKAREA         = LVM_FIRST + 65;

function ListView_SetWorkArea(LVWnd: HWnd; const Rect: TRect): boolean;

function ListView_GetCheckState(LVWnd: HWnd; Index: UINT): boolean;

procedure ListView_SetCheckState(LVWnd: HWnd; Index: UINT; Checked: boolean);

const
  LVSICF_NOINVALIDATEALL  = $00000001;
  LVSICF_NOSCROLL         = $00000002;

procedure ListView_SetItemCountEx(LVWnd: HWnd; Items: integer; Flags: DWORD);

const
  { New list view style flags.                                                           }
  LVS_OWNERDATA                = $1000; // Specifies a "virtual" list veiw control.

  { New notification messages.                                                           }
  LVN_ODCACHEHINT              = LVN_FIRST-13;
  LVN_ODFINDITEMA              = LVN_FIRST-52;
  LVN_ODFINDITEMW              = LVN_FIRST-79;
  LVN_ODFINDITEM               = LVN_ODFINDITEMA;

  LVN_ITEMACTIVATE             = LVN_FIRST-14;
  LVN_ODSTATECHANGED           = LVN_FIRST-15;
  LVN_MARQUEEBEGIN             = LVN_FIRST-56;

type
  PLVCacheHint = ^TLVCacheHint;
  TLVCacheHint = packed record
    hdr:       TNMHDR;
    iFrom:     integer;
    iTo:       integer;
  end;

  PLVFindItem = ^TLVFindItem;
  TLVFindItem = packed record
    hdr:       TNMHDR;
    iStart:    integer;
    lvif:      TLVFindInfo;
  end;

  PLVODStateChange = ^TLVODStateChange;
  TLVODStateChange = packed record
    hdr:       TNMHDR;
    iFrom:     integer;
    iTo:       integer;
    uNewState: UINT;
    uOldState: UINT;
  end;


type
  { New extended style flags converted to set format.                                    }
  {   lvxGridlines: Adds grid lines to seperate items and columns. Report mode only.     }
  {   lvxSubItemImages: Allows images to be displayed for subitems.  Report mode only.   }
  {   lvxCheckboxes: Adds checkboxes to items.  Checked items are stored internally as   }
  {       selected items.                                                                }
  {   lvxTrackSelect: Tracks the mouse and highlights the item it currently positioned   }
  {       over by changing it's color.  If mouse is left over an item for a brief period }
  {       of time, it will be automatically selected.                                    }
  {   lvxHeaderDragDrop: Allows headers to be dragged to new positions and dropped,      }
  {       allowing users to reorder column information.                                  }
  {   lvxFullRowSelect: Allows user to click anywhere on an item to select it,           }
  {       highlighting the entire length of the item.  Without this style, users must    }
  {       click inside the text of column 0.  It is only useful in vsReport view style.  }
  {   lvxOneClickActivate: Sends an LVN_ITEMACTIVATE notification message to the parent  }
  {       when the user clicks an item.                                                  }
  {   lvxTwoClickActivate: Sends an LVN_ITEMACTIVATE notification message to the parent  }
  {       when the user double clicks an item.                                           }
  TLVExtendedStyle = (lvxGridLines, lvxSubItemImages, lvxCheckboxes, lvxTrackSelect,
                      lvxHeaderDragDrop, lvxFullRowSelect, lvxOneClickActivate,
                      lvxTwoClickActivate);

  { A set of the new style bits.                                                         }
  TLVExtendedStyles = set of TLVExtendedStyle;

  TLVItemCountFlag = (lvsicfNoInvalidateAll, lvsicfNoScroll);
  TLVItemCountFlags = set of TLVItemCountFlag;
  TLVODMaskItem = (lvifText, lvifImage, lvifParam, lvifState, lvifIndent);
  TLVODMaskItems = set of TLVODMaskItem;

  TLVMarqueeBeginEvent   = procedure(Sender: TObject; var CanBegin: boolean) of object;
  TLVItemActivateEvent   = TNotifyEvent;
  TLVODGetItemInfoEvent  = procedure(Sender: TObject; Item, SubItem: integer;
                                     Mask: TLVODMaskItems; var Image: integer;
                                     var Param: LPARAM; var State: UINT;
                                     var Indent: integer; var Text: string) of object;
  TLVODCacheHintEvent    = procedure(Sender: TObject; var HintInfo: TLVCacheHint) of object;
  TLVODFindItemEvent     = procedure(Sender: TObject; var FindInfo: TLVFindItem;
                                     var Found: boolean) of object;
  TLVODStateChangedEvent = procedure(Sender: TObject; var StateInfo: TLVODStateChange) of object;

  { Class for saved settings                                                             }
  TELVSaveSettings = class(TPersistent)
  private
    FAutoSave: boolean;
    FRegistryKey: string;
    FSaveColumnOrder: boolean;
    FSaveColumnSizes: boolean;
  public
    constructor Create; virtual;
    procedure StoreColumnOrder(ColCount: integer; const IntArray: array of integer);
    procedure ReadColumnOrder(ColCount: integer; var IntArray: array of integer);
    procedure StoreColumnSizes(ColCount: integer; const IntArray: array of integer);
    procedure ReadColumnSizes(ColCount: integer; var IntArray: array of integer);
  published
    property AutoSave: boolean read FAutoSave write FAutoSave default FALSE;
    property RegistryKey: string read FRegistryKey write FRegistryKey;
    property SaveColumnOrder: boolean read FSaveColumnOrder write FSaveColumnOrder default TRUE;
    property SaveColumnSizes: boolean read FSaveColumnSizes write FSaveColumnSizes default TRUE;
  end;

  { The new class.                                                                       }
  TExtListView = class(TListView)
  private
    FAutoColumnSort: TAutoColumnSort;
    FAutoSortAscending: boolean;
    FTmpAutoSortAscending: boolean;
    FLastColumnClicked: Integer;
    FVirtualMode: boolean;
    FSaveSettings: TELVSaveSettings;
    FOnMarqueeBegin: TLVMarqueeBeginEvent;
    FOnItemActivate: TLVItemActivateEvent;
    FOnODGetItemInfo: TLVODGetItemInfoEvent;
    FOnODCacheHint: TLVODCacheHintEvent;
    FOnODFindItem: TLVODFindItemEvent;
    FOnODStateChanged: TLVODStateChangedEvent;
    { Function to convert from our set type to expected API value.                       }
    function SetValueToAPIValue(Styles: TLVExtendedStyles): LPARAM;
    function SetValueFromAPIValue(Styles: DWORD): TLVExtendedStyles;
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
    procedure WMDestroy(var Message: TWMDestroy); message WM_DESTROY;
    procedure SetAutoColumnSort(Value: TAutoColumnSort);
    procedure SetAutoSortAscending(Value: boolean);
  protected
    { Overriden ancestor methods }
    procedure ColClick(Column: TListColumn); override;
    { Property method for setting styles.                                                }
    procedure SetExtendedStyles(Val: TLVExtendedStyles);
    function GetExtendedStyles: TLVExtendedStyles;
    function GetHeaderHandle: HWnd;
    function GetSubItemRect(Item, SubItem: integer; Index: integer): TRect;
    procedure SetHotItem(Val: integer);
    function GetHotItem: integer;
    procedure SetHotCursor(const Val: HCursor);
    function GetHotCursor: HCursor;
    procedure SetWorkArea(Rect: TRect);
    procedure SetCheckState(Index: integer; Checked: boolean);
    function GetCheckState(Index: integer): boolean;
    procedure SetVirtualMode(Val: boolean);
    procedure CreateParams(var Params: TCreateParams); override;
    procedure Loaded; override;
    { Event method handlers -- fire the events if they exist.                            }
    function MarqueeBegin: boolean; virtual;
    procedure ItemActivate; virtual;
    procedure ODGetDispInfo(var ItemInfo: TLVItemEx); virtual;
    procedure ODCacheHint(var HintInfo: TLVCacheHint); virtual;
    function ODFindItem(var FindInfo: TLVFindItem): boolean; virtual;
    procedure ODStateChanged(var StateInfo: TLVODStateChange); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetIconSpacing(X, Y: integer);
    function GetSubItemAt(X, Y: integer): string;
    procedure SetColumnOrder(Count: integer; const IntArray: array of integer);
    function GetColumnOrder(Count: integer; var IntArray: array of integer): boolean;
    function ApproximateViewRect(Count: integer; const Proposed: TPoint): TPoint;
    procedure SetItemCountEx(Count: integer; Flags: TLVItemCountFlags);
    procedure StoreSettings;
    procedure LoadSettings;
    {************ Added by John Newlin 12/20/97 *************}
    function IsSelected(index : integer) : boolean;
    function ItemIndex : integer;
    {********************************************************}
    procedure DefaultSort(ColumnIndex:integer; Ascending:boolean); virtual;
    procedure Resort;

    property HeaderHandle: HWnd
             read GetHeaderHandle;
    property SubItem_BoundsRect[Item: integer; SubItem: integer]: TRect
             index LVIR_BOUNDS
             read GetSubItemRect;
    property SubItem_IconRect[Item: integer; SubItem: integer]: TRect
             index LVIR_ICON
             read GetSubItemRect;
    property SubItem_LabelRect[Item: integer; SubItem: integer]: TRect
             index LVIR_LABEL
             read GetSubItemRect;
    property SubItem_SelectBoundsRect[Item: integer; SubItem: integer]: TRect
             index LVIR_SELECTBOUNDS
             read GetSubItemRect;
    property HotItem: integer
             read GetHotItem write SetHotItem;
    property HotCursor: HCursor
             read GetHotCursor write SetHotCursor;
    property WorkArea: TRect
             write SetWorkArea;
    property IsChecked[Index: integer]: boolean
             read GetCheckState write SetCheckState;
  published
    { Property for new styles.                                                           }
    property ExtendedStyles: TLVExtendedStyles
             read GetExtendedStyles write SetExtendedStyles default [];
    property VirtualMode: boolean
             read FVirtualMode write SetVirtualMode default FALSE;

    { Auto column sorting property.                                                      }
    property AutoColumnSort: TAutoColumnSort
             read FAutoColumnSort write SetAutoColumnSort default acsNoSort;
    property AutoSortAscending: boolean
             read FAutoSortAscending write SetAutoSortAscending default TRUE;

    { Autosave settings property.                                                        }
    property SaveSettings: TELVSaveSettings
             read FSaveSettings write FSaveSettings;

    { Events                                                                             }
    property OnMarqueeBegin: TLVMarqueeBeginEvent
             read FOnMarqueeBegin write FOnMarqueeBegin;
    property OnItemActivate: TLVItemActivateEvent
             read FOnItemActivate write FOnItemActivate;
    property OnODGetItemInfo: TLVODGetItemInfoEvent
             read FOnODGetItemInfo write FOnODGetItemInfo;
    property OnODCacheHint: TLVODCacheHintEvent
             read FOnODCacheHint write FOnODCacheHint;
    property OnODFindItem: TLVODFindItemEvent
             read FOnODFindItem write FOnODFindItem;
    property OnODStateChanged: TLVODStateChangedEvent
             read FOnODStateChanged write FOnODStateChanged;
  end;


function __CustomSortProc1__(Item1, Item2: TListItem; Data: integer): integer; stdcall;
procedure Register;

implementation

uses DsgnIntf, Registry;

var
  FDirection: integer;

function __CustomSortProc1__(Item1, Item2: TListItem; Data: integer): integer;
  function IsValidNumber(const S: string; var V: extended): boolean;
  var
    NumCode: integer;
  begin
    Val(S, V, NumCode);
    Result := (NumCode = 0);
  end;

  function IsValidDate(const S: string; var D: TDateTime): boolean;
  begin
    try
      D := StrToDate(S);
      Result := TRUE;
    except
      D := 0;
      Result := FALSE;
    end;
  end;
var
  Str1, Str2: string;
  Val1, Val2: extended;
  Date1, Date2: TDateTime;
begin
  try
    if Data = -1 then begin
      Str1 := Item1.Caption;
      Str2 := Item2.Caption;
    end else begin
      Str1 := Item1.SubItems[Data];
      Str2 := Item2.SubItems[Data];
    end;

    if IsValidNumber(Str1, Val1) and IsValidNumber(Str2, Val2) then
      if   Val1 < Val2 then Result := -1
      else if Val1 > Val2 then Result := 1
      else Result := 0
    else if IsValidDate(Str1, Date1) and IsValidDate(Str2, Date2) then
      Result := Trunc(Date1 - Date2)
    else // date check?
      Result := AnsiCompareStr(Str1, Str2);

    Result := FDirection * Result; // Set direction flag.
  except
    Result := 0;  // Something went bad in the comparison.  Say they are equal.
  end;
end;

procedure Register;
begin
  RegisterComponents('Tools', [TExtListView]);
  RegisterPropertyEditor(TypeInfo(TELVSaveSettings), nil, '', TClassProperty);
end;


function ListView_GetColumnEx(LVWnd: HWND; iCol: Integer; var pcol: TLVColumnEx): bool;
begin
  Result := bool(SendMessage(LVWnd, LVM_GETCOLUMN, iCol, Longint(@pcol)));
end;

function ListView_SetColumnEx(LVWnd: HWnd; iCol: Integer; const pcol: TLVColumnEx): Bool;
begin
  Result := bool(SendMessage(LVWnd, LVM_SETCOLUMN, iCol, Longint(@pcol)));
end;

function ListView_InsertColumnEx(LVWnd: HWND; iCol: Integer;
                                 const pcol: TLVColumnEx): Integer;
begin
  Result := SendMessage(LVWnd, LVM_INSERTCOLUMN, iCol, Longint(@pcol));
end;

function ListView_GetHeader(LVWnd: HWnd): HWnd;
begin
  Result := HWnd(SendMessage(LVWnd, LVM_GETHEADER, 0, 0));
end;

function ListView_SetIconSpacing(LVWnd: HWnd; cx, cy: integer): DWORD;
begin
  Result := SendMessage(LVWnd, LVM_SETICONSPACING, 0, MAKELONG(cx,cy));
end;

function ListView_SetExtendedListViewStyle(LVWnd: HWnd; ExStyle: LPARAM): DWORD;
begin
  Result := SendMessage(LVWnd, LVM_SETEXTENDEDLISTVIEWSTYLE, 0, ExStyle);
end;

function ListView_GetExtendedListViewStyle(LVWnd: HWnd): DWORD;
begin
  Result := SendMessage(LVWnd, LVM_GETEXTENDEDLISTVIEWSTYLE, 0, 0);
end;

function ListView_GetSubItemRect(LVWnd: HWnd; ParentItem, SubItem, Code: integer;
                                 var Rect: TRect): boolean;
begin
  Rect.Top := SubItem;
  Rect.Left := Code;
  Result := (SendMessage(LVWnd, LVM_GETSUBITEMRECT, ParentItem, LPARAM(@Rect)) <> 0);
end;

function ListView_SubItemHitTest(LVWnd: HWnd; var HitTestInfo: TLVHitTestInfoEx): integer;
begin
  Result := SendMessage(LVWnd, LVM_SUBITEMHITTEST, 0, LPARAM(@HitTestInfo));
end;

function ListView_SetColumnOrderArray(LVWnd: HWnd; Count: integer;
                                      const IntArray: array of integer): boolean;
begin
  Result := (SendMessage(LVWnd, LVM_SETCOLUMNORDERARRAY, Count, LPARAM(@IntArray)) <> 0);
end;

function ListView_GetColumnOrderArray(LVWnd: HWnd; Count: integer;
                                      var IntArray: array of integer): boolean;
begin
  Result := (SendMessage(LVWnd, LVM_GETCOLUMNORDERARRAY, Count, LPARAM(@IntArray)) <> 0);
end;

function ListView_SetHotItem(LVWnd: HWnd; Item: integer): integer;
begin
  Result := SendMessage(LVWnd, LVM_SETHOTITEM, Item, 0);
end;

function ListView_GetHotItem(LVWnd: HWnd): integer;
begin
  Result := SendMessage(LVWnd, LVM_GETHOTITEM, 0, 0);
end;

function ListView_SetHotCursor(LVWnd: HWnd; Cursor: HCursor): HCursor;
begin
  Result := HCursor(SendMessage(LVWnd, LVM_SETHOTCURSOR, 0, LPARAM(Cursor)));
end;

function ListView_GetHotCursor(LVWnd: HWnd): HCursor;
begin
  Result := HCursor(SendMessage(LVWnd, LVM_GETHOTCURSOR, 0, 0));
end;

function ListView_ApproximateViewRect(LVWnd: HWnd; Width, Height, Count: integer): DWORD;
begin
  Result := SendMessage(LVWnd, LVM_APPROXIMATEVIEWRECT, Count, MAKELPARAM(Width, Height));
end;

function ListView_SetWorkArea(LVWnd: HWnd; const Rect: TRect): boolean;
begin
  Result := (SendMessage(LVWnd, LVM_SETWORKAREA, 0, LPARAM(@Rect)) <> 0);
end;

function ListView_GetCheckState(LVWnd: HWnd; Index: UINT): boolean;
begin
  Result := (((SendMessage(LVWnd, LVM_GETITEMSTATE, Index, LVIS_STATEIMAGEMASK) SHR 12)-1)
             <> 0);
end;

procedure ListView_SetCheckState(LVWnd: HWnd; Index: UINT; Checked: boolean);
const
  LVIS_UNCHECKED = $1000;
  LVIS_CHECKED = $2000;
var
  Data: integer;
begin
  if Checked then Data := LVIS_CHECKED
  else Data := LVIS_UNCHECKED;
  ListView_SetItemState(LVWnd, Index, Data, LVIS_STATEIMAGEMASK);
end;

procedure ListView_SetItemCountEx(LVWnd: HWnd; Items: integer; Flags: DWORD);
begin
  SendMessage(LVWnd, LVM_SETITEMCOUNT, Items, Flags);
end;


constructor TELVSaveSettings.Create;
begin
  FAutoSave := FALSE;
  FRegistryKey := '';
  FSaveColumnOrder := TRUE;
  FSaveColumnSizes := TRUE;
end;

procedure TELVSaveSettings.StoreColumnOrder(ColCount: integer; const IntArray: array of integer);
var
  Reg: TRegIniFile;
  x: integer;
  s: string;
begin
  if ColCount < 1 then exit;
  s := '';
  for x := 0 to ColCount-1 do
    s := s + IntToStr(IntArray[x]) + ',';
  SetLength(s, Length(s)-1);
  Reg := TRegIniFile.Create(FRegistryKey);
  try
    Reg.WriteString('Columns', 'Order', s);
  finally
    Reg.Free;
  end;
end;

procedure TELVSaveSettings.ReadColumnOrder(ColCount: integer; var IntArray: array of integer);
var
  Reg: TRegIniFile;
  x,y: integer;
  s: string;
begin
  if ColCount < 1 then exit;
  s := '';
  Reg := TRegIniFile.Create(FRegistryKey);
  try
    s := Reg.ReadString('Columns', 'Order', '');
  finally
    Reg.Free;
  end;
  if s = '' then begin
    for x := 0 to ColCount-1 do
      IntArray[x] := x;
    exit;
  end;
  y := 0;
  for x := 0 to ColCount-1 do begin
    try
      y := Pos(',', s);
      if y = 0 then
        y := Length(s)+1;
      IntArray[x] := StrToInt(Copy(s, 1, y-1));
    except
      IntArray[x] := 0;
    end;
    s := copy(s, y+1, length(s));
    if s = '' then break;
  end;
end;

procedure TELVSaveSettings.StoreColumnSizes(ColCount: integer; const IntArray: array of integer);
var
  Reg: TRegIniFile;
  x: integer;
  s: string;
begin
  if ColCount < 1 then exit;
  s := '';
  for x := 0 to ColCount-1 do
    s := s + IntToStr(IntArray[x]) + ',';
  SetLength(s, Length(s)-1);
  Reg := TRegIniFile.Create(FRegistryKey);
  try
    Reg.WriteString('Columns', 'Sizes', s);
  finally
    Reg.Free;
  end;
end;

procedure TELVSaveSettings.ReadColumnSizes(ColCount: integer; var IntArray: array of integer);
var
  Reg: TRegIniFile;
  x,y: integer;
  s: string;
begin
  if ColCount < 1 then exit;
  s := '';
  Reg := TRegIniFile.Create(FRegistryKey);
  try
    s := Reg.ReadString('Columns', 'Sizes', '');
  finally
    Reg.Free;
  end;
  if s = '' then begin
    IntArray[0] := -1;
    exit;
  end;
  y := 0;
  for x := 0 to ColCount-1 do begin
    try
      y := Pos(',', s);
      if y = 0 then
        y := Length(s)+1;
      IntArray[x] := StrToInt(Copy(s, 1, y-1));
    except
      IntArray[x] := 0;
    end;
    s := copy(s, y+1, length(s));
    if s = '' then break;
  end;
end;

{ Override constructor to "zero out" our internal variable.                              }
constructor TExtListView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSaveSettings := TELVSaveSettings.Create;
  FVirtualMode := FALSE;
  FAutoColumnSort := acsNoSort;
  FAutoSortAscending := TRUE;
  FTmpAutoSortAscending := FAutoSortAscending;
//  AutoSortAscending := TRUE;    instead of previous two lines - which do you prefer??
  FLastColumnClicked := -1;
  FOnMarqueeBegin := NIL;
  FOnItemActivate := NIL;
  FOnODGetItemInfo := NIL;
  FOnODCacheHint := NIL;
  FOnODFindItem := NIL;
  FOnODStateChanged := NIL;
end;

destructor TExtListView.Destroy;
begin
  FSaveSettings.Free;
  inherited Destroy;
end;

procedure TExtListView.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  if FVirtualMode then
    Params.Style := Params.Style or LVS_OWNERDATA;
end;

procedure TExtListView.Loaded;
begin
  inherited Loaded;
  LoadSettings;
end;

const
  API_STYLES: array[lvxGridLines..lvxTwoClickActivate] of LPARAM = (
              LVS_EX_GRIDLINES, LVS_EX_SUBITEMIMAGES, LVS_EX_CHECKBOXES,
              LVS_EX_TRACKSELECT, LVS_EX_HEADERDRAGDROP, LVS_EX_FULLROWSELECT,
              LVS_EX_ONECLICKACTIVATE, LVS_EX_TWOCLICKACTIVATE
              );

{ Function to convert our style set type into the value expected by the API.             }
function TExtListView.SetValueToAPIValue(Styles: TLVExtendedStyles): LPARAM;
var
  x: TLVExtendedStyle;
begin
  Result := 0;
  { Check for each possible style. }
  for x := lvxGridLines to lvxTwoClickActivate do
    { If the style is set... }
    if x in Styles then
      { OR the appropriate value into the result. }
      Result := Result OR API_STYLES[x];
end;

{ Function to convert from the API values to our style set type.                         }
function TExtListView.SetValueFromAPIValue(Styles: DWORD): TLVExtendedStyles;
var
  x: TLVExtendedStyle;
begin
  Result := [];
  { Check for each possible style. }
  for x := lvxGridLines to lvxTwoClickActivate do
    { If the style is set... }
    if (API_STYLES[x] and Styles) <> 0 then
      { OR the appropriate value into the result. }
      Result := Result + [x];
end;

{ Property method to get the extended style bits.                                        }
function TExtListView.GetExtendedStyles: TLVExtendedStyles;
begin
  Result := SetValueFromAPIValue(ListView_GetExtendedListViewStyle(Handle));
end;

{ Property method to set new style bits.                                                 }
procedure TExtListView.SetExtendedStyles(Val: TLVExtendedStyles);
begin
  { Update the window with the new styles. }
  ListView_SetExtendedListViewStyle(Handle, SetValueToAPIValue(Val));
end;

function TExtListView.GetHeaderHandle: HWnd;
begin
  Result := ListView_GetHeader(Handle);
end;

{ Not sure about how to update the view after changing this.  Refresh doesn't do the job.
  Seems the best way to do it is something like:

  SetIconSpacing(X, Y);
  if ViewStyle = vsIcon then begin
    SendMessage(Handle, WM_SETREDRAW, 0, 0);
    try
      ViewStyle := vsSmallIcon;
      ViewStyle := vsIcon;
    finally
      SendMessage(Handle, WM_SETREDRAW, 1, 0);
    end;
  end;

Does strange things if ViewStyle is not set to vsIcon!
}
procedure TExtListView.SetIconSpacing(X, Y: integer);
begin
  if ViewStyle = vsIcon then
    ListView_SetIconSpacing(Handle, X, Y);
end;

function TExtListView.GetSubItemRect(Item, SubItem: integer; Index: integer): TRect;
begin
  ListView_GetSubItemRect(Handle, Item, SubItem, Index, Result);
end;

function TExtListView.GetSubItemAt(X, Y: integer): string;
var
  Info: TLVHitTestInfoEx;
begin
  Result := '';
  Info.pt := Point(X, Y);
  if ListView_SubItemHitTest(Handle, Info) <> -1 then begin
    if Info.iItem > -1 then
      if Info.iSubItem = 0 then
        Result := Items[Info.iItem].Caption
      else
        Result := Items[Info.iItem].SubItems[Info.iSubItem-1];
  end;
end;

procedure TExtListView.SetColumnOrder(Count: integer; const IntArray: array of integer);
begin
  ListView_SetColumnOrderArray(Handle, Count, IntArray);
  Refresh;
end;

function TExtListView.GetColumnOrder(Count: integer;
                                     var IntArray: array of integer): boolean;
begin
  Result := ListView_GetColumnOrderArray(Handle, Count, IntArray);
end;

procedure TExtListView.SetHotItem(Val: integer);
begin
  ListView_SetHotItem(Handle, Val);
end;

function TExtListView.GetHotItem: integer;
begin
  Result := ListView_GetHotItem(Handle);
end;

procedure TExtListView.SetHotCursor(const Val: HCursor);
begin
  ListView_SetHotCursor(Handle, Val);
end;

function TExtListView.GetHotCursor: HCursor;
begin
  Result := ListView_GetHotCursor(Handle);
end;

function TExtListView.ApproximateViewRect(Count: integer; const Proposed: TPoint): TPoint;
var
  Res: DWORD;
begin
  Res := ListView_ApproximateViewRect(Handle, Count, Proposed.X, Proposed.Y);
  Result := Point(LoWord(Res), HiWord(Res));
end;

procedure TExtListView.SetWorkArea(Rect: TRect);
begin
  ListView_SetWorkArea(Handle, Rect);
end;

procedure TExtListView.SetCheckState(Index: integer; Checked: boolean);
begin
  ListView_SetCheckState(Handle, Index, Checked);
end;

function TExtListView.GetCheckState(Index: integer): boolean;
begin
  Result := ListView_GetCheckState(Handle, Index);
end;

procedure TExtListView.SetItemCountEx(Count: integer; Flags: TLVItemCountFlags);
var
  APIFlags: DWORD;
begin
  APIFlags := 0;
  if lvsicfNoInvalidateAll in Flags then
    APIFlags := LVSICF_NOINVALIDATEALL;
  if lvsicfNoScroll in Flags then
    APIFlags := APIFlags or LVSICF_NOSCROLL;
  ListView_SetItemCountEx(Handle, Count, APIFlags);
end;

procedure TExtListView.SetVirtualMode(Val: boolean);
begin
  if Val = FVirtualMode then exit;
  FVirtualMode := Val;
  RecreateWnd;
end;


procedure TExtListView.CNNotify(var Message: TWMNotify);
begin
  with Message.NMHdr^ do begin
    Message.Result := 0;
    if FVirtualMode then begin
      case code of
        LVN_GETDISPINFO:    ODGetDispInfo(PLVDispInfoEx(pointer(Message.NMHdr))^.item);
        LVN_ODCACHEHINT:    ODCacheHint(PLVCacheHint(pointer(Message.NMHdr))^);
        LVN_ODSTATECHANGED: ODStateChanged(PLVODStateChange(pointer(Message.NMHdr))^);
        LVN_ODFINDITEM:
          if not ODFindItem(PLVFindItem(pointer(Message.NMHdr))^) then
            Message.Result := -1;
      else
        inherited;
      end;
    end else begin
      case code of
        LVN_ITEMACTIVATE:
          begin
            ItemActivate;
            Message.Result := 0;
          end;
        LVN_MARQUEEBEGIN:
          begin
            if MarqueeBegin then
              Message.Result := 0
            else
              Message.Result := 1;
          end;
      else
        inherited;
      end;
    end;
  end;
end;

function TExtListView.MarqueeBegin: boolean;
begin
  Result := TRUE;
  if assigned(FOnMarqueeBegin) then
    FOnMarqueeBegin(Self, Result);
end;

procedure TExtListView.ItemActivate;
begin
  if assigned(FOnItemActivate) then
    FOnItemActivate(Self);
end;

procedure TExtListView.ODGetDispInfo(var ItemInfo: TLVItemEx);
  function MaskFlagsToSet(Mask: UINT): TLVODMaskItems;
  begin
    Result := [];
    if (Mask and LVIF_TEXT) = LVIF_TEXT then
      Include(Result, lvifText);
    if (Mask and LVIF_IMAGE) = LVIF_IMAGE then
      Include(Result, lvifImage);
    if (Mask and LVIF_PARAM) = LVIF_PARAM then
      Include(Result, lvifParam);
    if (Mask and LVIF_STATE) = LVIF_STATE then
      Include(Result, lvifState);
    if (Mask and LVIF_INDENT) = LVIF_INDENT then
      Include(Result, lvifIndent);
  end;
var
  Text: string;
  NewState: integer;
  GetMask: TLVODMaskItems;
begin
  if ItemInfo.iItem = -1 then exit;  // No way.
  Text := '';
  NewState := ItemInfo.State;
  GetMask := MaskFlagsToSet(ItemInfo.Mask);
  if assigned(FOnODGetItemInfo) then begin
    with ItemInfo do
      FOnODGetItemInfo(Self, iItem, iSubItem, GetMask,
                       iImage, lParam, NewState, iIndent, Text);
    if (ItemInfo.mask and LVIF_TEXT) = LVIF_TEXT then
      StrLCopy(ItemInfo.pszText, PChar(Text), ItemInfo.cchTextMax);
    ItemInfo.State := NewState;
  end;
end;

procedure TExtListView.ODCacheHint(var HintInfo: TLVCacheHint);
begin
  if assigned(FOnODCacheHint) then
    FOnODCacheHint(Self, HintInfo);
end;

function TExtListView.ODFindItem(var FindInfo: TLVFindItem): boolean;
begin
  Result := FALSE;
  if assigned(FOnODFindItem) then
    FOnODFindItem(Self, FindInfo, Result);
end;

procedure TExtListView.ODStateChanged(var StateInfo: TLVODStateChange);
begin
  if assigned(FOnODStateChanged) then
    FOnODStateChanged(Self, StateInfo);
end;

procedure TExtListView.WMDestroy(var Message: TWMDestroy);
begin
  StoreSettings;
  inherited;
end;

type
  PIntArray = ^TIntArray;
  TIntArray = array[0..30000] of integer;

procedure TExtListView.StoreSettings;
var
  ColCount: integer;
  ColArray: PIntArray;
  x: integer;
begin
  if FSaveSettings.AutoSave and (not(csDesigning in ComponentState)) then begin
    ColCount := Columns.Count;
    if ColCount > 0 then begin
      GetMem(ColArray, SizeOf(Integer)*ColCount);
      try
        if FSaveSettings.SaveColumnOrder then begin
          GetColumnOrder(ColCount, ColArray^);
          FSaveSettings.StoreColumnOrder(ColCount, ColArray^);
        end;
        if FSaveSettings.SaveColumnSizes then begin
          for x := 0 to ColCount-1 do
            ColArray[x] := Columns[x].Width;
          FSaveSettings.StoreColumnSizes(ColCount, ColArray^);
        end;
      finally
        FreeMem(ColArray);
      end;
    end;
  end;
end;

procedure TExtListView.LoadSettings;
var
  ColCount: integer;
  ColArray: PIntArray;
  x: integer;
begin
  if FSaveSettings.AutoSave and (not(csDesigning in ComponentState)) then begin
    ColCount := Columns.Count;
    if ColCount > 0 then begin
      GetMem(ColArray, SizeOf(Integer)*ColCount);
      try
        if FSaveSettings.SaveColumnOrder then begin
          FSaveSettings.ReadColumnOrder(ColCount, ColArray^);
          SetColumnOrder(ColCount, ColArray^);
        end;
        if FSaveSettings.SaveColumnSizes then begin
          FSaveSettings.ReadColumnSizes(ColCount, ColArray^);
          if ColArray[0] <> -1 then
            for x := 0 to ColCount-1 do
              Columns[x].Width := ColArray[x];
        end;
      finally
        FreeMem(ColArray);
      end;
    end;
  end;
end;

procedure TExtListView.DefaultSort(ColumnIndex: integer; Ascending: boolean);
begin
  if Ascending then
    FDirection := 1
  else
    FDirection := -1;
  CustomSort(@__CustomSortProc1__, ColumnIndex-1);
end;

procedure TExtListView.ColClick(Column: TListColumn);
begin
  inherited ColClick(Column);
  // Check if the sort order should be toggled
  if FAutoColumnSort = acsSortToggle then
    if FLastColumnClicked = Column.Index then
      FTmpAutoSortAscending := not FTmpAutoSortAscending
    else
      FTmpAutoSortAscending := FAutoSortAscending;
  if (FAutoColumnSort <> acsNoSort) and (Column.Index < Columns.Count) then begin
    DefaultSort(Column.Index, FTmpAutoSortAscending);
    FLastColumnClicked := Column.Index;
  end;
end;

procedure TExtListView.SetAutoColumnSort(Value: TAutoColumnSort);
begin
  if FAutoColumnSort <> Value then
    FAutoColumnSort := Value;
end;

procedure TExtListView.SetAutoSortAscending(Value: Boolean);
begin
  if FAutoSortAscending <> Value then begin
    FAutoSortAscending := Value;
    FTmpAutoSortAscending := Value;
  end;
end;

procedure TExtListView.Resort;
begin
  if (FAutoColumnSort <> acsNoSort) and (FLastColumnClicked >= 0) and
     (FLastColumnClicked < Columns.Count) then
    DefaultSort(FLastColumnClicked, FTmpAutoSortAscending);
end;

function TExtListView.IsSelected(index : integer) : boolean;
begin
  Result :=
   ListView_GetItemState(Handle,index , LVIS_SELECTED) and LVIS_SELECTED <> 0;
end;

function TExtListView.ItemIndex : integer;
begin
  result := ListView_GetNextItem(Handle, -1, LVNI_FOCUSED);
end;

END.

