unit SwColor;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;
type
  TFontComboBox = class(TCustomComboBox)
  private
    { Private declarations }
    FBitmap: TBitmap;
    FCanvas: TControlCanvas;
    FTTOnly: Boolean;
    FUseItemFont: Boolean;
    function  IsTrueType(Index: Integer): Boolean;
    procedure DrawTT(Background: TColor);
    procedure SetTTOnly(Value : boolean);
    procedure SetUseItemFont(Value : boolean);
    procedure ReSetItemHeight ;
    procedure Change;override;
  protected
    { Protected declarations }
    procedure CreateWnd; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure   DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
  published
    { Published declarations }
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property DropDownCount;
    property Enabled;
    property Font;
    property Items;
    property ItemHeight;
    property ParentColor;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property TTonly : boolean read FTTonly write SetTTonly;
    property UseItemFont : boolean read FUseItemFont write SetUseItemFont;
  end;

  TFontListBox = class(TCustomListBox)
  private
    { Private declarations }
    FBitmap: TBitmap;
    FCanvas: TControlCanvas;
    FTTOnly: Boolean;
    FUseItemFont: Boolean;
    function  IsTrueType(Index: Integer): Boolean;
    procedure DrawTT(Background: TColor);
    procedure SetTTOnly(Value : boolean);
    procedure SetUseItemFont(Value : boolean);
    procedure ReSetItemHeight ;
    procedure Click;override;
  protected
    { Protected declarations }
    procedure CreateWnd; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure   DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
  published
    { Published declarations }
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    property Items;
    property ItemIndex;
    property IntegralHeight;
    property ParentColor;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property TTonly : boolean read FTTonly write SetTTonly;
    property UseItemFont : boolean read FUseItemFont write SetUseItemFont;
  end;

type
  TSortBy = (ccsUnsorted, ccsColor, ccsText);
  TShowMode = (ccsmBoth, ccsmColorRect, ccsmText);
  TSwComboColor = class(TCustomComboBox)
  private
    { Private declarations }
    FColorWidth: Integer;
    FSortBy: TSortBy;
    FShowMode : TShowMode ;
    FCustomColor:TColor ;
    FCustomColorText:String ;
    function GetSelectedColor: TColor;
    function GetSelectedColorText: String;
    procedure SetColorWidth(Value: Integer);
    procedure SetSelectedColor(Value: TColor);
    procedure SetSelectedColorText(Value: String);
    procedure SetCustomColorText(Value: String);
    procedure SetSortBy(Value: TSortBy);
    procedure DrawItem(Index: Integer; Rect: TRect;
                       State: TOwnerDrawState); override;
    procedure SetShowMode(Value:TShowMode) ;
    procedure Change;override;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    procedure AddColor(ColorText: String; Color: TColor);
    property SelectedColor:TColor read GetSelectedColor write SetSelectedColor;
    property SelectedColorText:String read GetSelectedColorText write SetSelectedColorText;
  published
    { Published declarations }
    property ShowMode: TShowMode read FShowMode write SetShowMode default ccsmBoth;
    property CustomColorText:String read FCustomColorText write SetCustomColorText;
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property DropDownCount;
    property Enabled;
    property Font;
    property ItemHeight;
    property Items;
    property MaxLength;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
    property ColorWidth: Integer read FColorWidth write SetColorWidth default 18;
    property SortBy: TSortBy read FSortBy write SetSortBy default ccsUnsorted;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Samples', [TFontComboBox]);
  RegisterComponents('Samples', [TFontListBox]);
  RegisterComponents('Samples', [TSwComboColor]);
end;

constructor TFontComboBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Sorted := True;
  Style := csOwnerDrawVariable ;
  ShowHint:=True ;
  ItemHeight := 16;
  {Create a bitmap for drawing}
  FBitmap := TBitmap.Create;
  FBitmap.Height := 12;
  FBitmap.Width := 12;
  {Create a Canvas for checking True Type property}
  FCanvas := TControlCanvas.Create;
  FCanvas.Control := Self;
end;

procedure TFontComboBox.SetTTOnly(value : boolean);
var I:longint;
begin
  FTTonly := Value;
  if FTTonly = True then
  begin
   Items.clear;
   Items.Assign(Screen.Fonts);
   I := 0;
   repeat
   if Items[I] = 'Default' then
   begin
     Items.delete(I);
   end else inc(I);
   until I = Items.count;
   ItemIndex := Items.Count-1;
   I := 0;
   if FTTonly then
     repeat
       if not IsTrueType(I) then
       begin
          Items.delete(I);
       end else inc(I);
     until I = Items.count;
  end else
  begin
   Items.clear;
   Items.Assign(Screen.Fonts);
   I := 0;
   repeat
    if Items[I] = 'Default' then
    begin
      Items.delete(I);
    end else inc(I);
   until I = Items.count;
   ItemIndex := Items.Count-1;
  end;
  Hint:=Items[ItemIndex] ;
  ReSetItemHeight ;
end;

procedure TFontComboBox.SetUseItemFont(value : boolean);
begin
 FUseItemFont := Value;
 SetTTOnly(FTTOnly);
end;

destructor TFontComboBox.Destroy;
begin
  FBitmap.Free;
  FCanvas.Free;
  inherited Destroy;
end;

procedure TFontComboBox.CreateWnd;
var I:longint;
begin
  inherited CreateWnd;
  Items.Assign(Screen.Fonts);
  I := 0;
  repeat
   if Items[I] = 'Default' then
   begin
        Items.delete(I);
   end else inc(I);
  until I = Items.count;
  ItemIndex := Items.Count-1;
  Hint:=Items[ItemIndex] ;
  ReSetItemHeight;
end;

procedure TFontComboBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
begin
if FUseItemFont then
begin
  Canvas.Font.Charset:=DEFAULT_CHARSET ;
  Canvas.Font.Name:=Items[Index] ;
end;
  { Rect.Left = 3 means drawing on the Static portion of the ComboBox }
  with Canvas do begin
    FillRect(Rect);
    if IsTrueType(Index) and (Rect.Left <> 3) then begin
       DrawTT(Brush.Color);
       Draw(Rect.Left+2, Rect.Top+2, FBitmap);
    end;
    if (Rect.Left <> 3) then
       TextOut(Rect.Left+16, Rect.Top, Items[Index])
    else
       TextOut(Rect.Left, Rect.Top, Items[Index])
  end;
end;

function TFontComboBox.IsTrueType(Index: Integer): Boolean;
var
  Metrics: TTextMetric;
  lf: TLogFont;
  oldFont, newFont: HFont;
begin
  with lf do begin
    lfHeight := 10;
    lfWidth := 10;
    lfEscapement := 0;
    lfWeight := FW_REGULAR;
    lfItalic := 0;
    lfUnderline := 0;
    lfStrikeOut := 0;
    lfCharSet := DEFAULT_CHARSET;
    lfOutPrecision := OUT_DEFAULT_PRECIS;
    lfClipPrecision := CLIP_DEFAULT_PRECIS;
    lfQuality := DEFAULT_QUALITY;
    lfPitchAndFamily := DEFAULT_PITCH or FF_DONTCARE;
    StrPCopy(lfFaceName, Items[Index]);
  end;
  newFont := CreateFontIndirect(lf);
  oldFont := SelectObject(FCanvas.Handle, newFont);
  GetTextMetrics(FCanvas.Handle, Metrics);
  Result := (Metrics.tmPitchAndFamily and TMPF_TRUETYPE) <> 0;
  SelectObject(FCanvas.Handle, oldFont);
  DeleteObject(newFont);
end;

procedure TFontComboBox.DrawTT(Background: TColor);
  procedure DrawT(OrgX, OrgY: Integer; Color: TColor);
  begin
   with FBitmap.Canvas do begin
     Brush.Style := bsSolid;
     Pen.Color := Color;
     MoveTo(OrgX,OrgY);
     LineTo(OrgX+7,OrgY);
     LineTo(OrgX+7,OrgY+3);
     MoveTo(OrgX,OrgY);
     LineTo(OrgX,OrgY+3);
     MoveTo(OrgX+1,OrgY);
     LineTo(OrgX+1,OrgY+1);
     MoveTo(OrgX+6,OrgY);
     LineTo(OrgX+6,OrgY+1);
     MoveTo(OrgX+3,OrgY);
     LineTo(OrgX+3,OrgY+8);
     MoveTo(OrgX+4,OrgY);
     LineTo(OrgX+4,OrgY+8);
     MoveTo(OrgX+1,OrgY+8);
     LineTo(OrgX+6,OrgY+8);
   end;
  end;
begin
  with FBitmap.Canvas do begin
    Brush.Style := bsClear;
    Brush.Color := background;
    FillRect(Rect(0,0,12,12));
    DrawT(0,0,clGray);
    DrawT(4,3,clBlack);
  end;
end;

procedure TFontComboBox.Change;
begin
    Hint:=Items[ItemIndex] ;
    inherited ;
end;

procedure TFontComboBox.ReSetItemHeight ;
var i,h1,h2:Integer ;
begin
     FCanvas.Font:=Font ;
     FCanvas.Font.CharSet:=DEFAULT_CHARSET;
     h1:=0;
     for i:=0 to Items.Count-1 do
     begin
          h2:=FCanvas.TextHeight('0');
          if h1<h2 then h1:=h2 ;
     end;
     if h1>0 then ItemHeight:=h1 ;
end;

constructor TFontListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Sorted := True;
  Style := lbOwnerDrawFixed;
  ShowHint:=True ;
  ItemHeight := 16;
  {Create a bitmap for drawing}
  FBitmap := TBitmap.Create;
  FBitmap.Height := 12;
  FBitmap.Width := 12;
  {Create a Canvas for checking True Type property}
  FCanvas := TControlCanvas.Create;
  FCanvas.Control := Self;
end;

procedure TFontListBox.SetTTOnly(value : boolean);
var I:longint;
begin
  FTTonly := Value;
  if FTTonly = True then
  begin
   Items.clear;
   Items.Assign(Screen.Fonts);
   I := 0;
   repeat
   if Items[I] = 'Default' then
   begin
     Items.delete(I);
   end else inc(I);
   until I = Items.count;
   ItemIndex := Items.Count-1;
   I := 0;
   if FTTonly then
     repeat
       if not IsTrueType(I) then
       begin
          Items.delete(I);
       end else inc(I);
     until I = Items.count;
  end else
  begin
   Items.clear;
   Items.Assign(Screen.Fonts);
   I := 0;
   repeat
    if Items[I] = 'Default' then
    begin
      Items.delete(I);
    end else inc(I);
   until I = Items.count;
   ItemIndex := Items.Count-1;
  end;
  Hint:=Items[ItemIndex] ;
  ReSetItemHeight ;
end;

procedure TFontListBox.SetUseItemFont(value : boolean);
begin
 FUseItemFont := Value;
 SetTTOnly(FTTOnly);
end;

destructor TFontListBox.Destroy;
begin
  FBitmap.Free;
  FCanvas.Free;
  inherited Destroy;
end;

procedure TFontListBox.CreateWnd;
var I:longint;
begin
  inherited CreateWnd;
  Items.Assign(Screen.Fonts);
  I := 0;
  repeat
   if Items[I] = 'Default' then
   begin
        Items.delete(I);
   end else inc(I);
  until I = Items.count;
  ItemIndex := Items.Count-1;
  Hint:=Items[ItemIndex] ;
  ReSetItemHeight ;
end;

procedure TFontListBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
begin
if FUseItemFont then
begin
  Canvas.Font.Charset:=DEFAULT_CHARSET ;
  Canvas.Font.Name:=Items[Index] ;
end;
  { Rect.Left = 3 means drawing on the Static portion of the ComboBox }
  with Canvas do begin
    FillRect(Rect);
    if IsTrueType(Index) and (Rect.Left <> 3) then begin
       DrawTT(Brush.Color);
       Draw(Rect.Left+2, Rect.Top+2, FBitmap);
    end;
    if (Rect.Left <> 3) then
       TextOut(Rect.Left+16, Rect.Top, Items[Index])
    else
       TextOut(Rect.Left, Rect.Top, Items[Index])
  end;
end;

function TFontListBox.IsTrueType(Index: Integer): Boolean;
var
  Metrics: TTextMetric;
  lf: TLogFont;
  oldFont, newFont: HFont;
begin
  with lf do begin
    lfHeight := 10;
    lfWidth := 10;
    lfEscapement := 0;
    lfWeight := FW_REGULAR;
    lfItalic := 0;
    lfUnderline := 0;
    lfStrikeOut := 0;
    lfCharSet := DEFAULT_CHARSET;
    lfOutPrecision := OUT_DEFAULT_PRECIS;
    lfClipPrecision := CLIP_DEFAULT_PRECIS;
    lfQuality := DEFAULT_QUALITY;
    lfPitchAndFamily := DEFAULT_PITCH or FF_DONTCARE;
    StrPCopy(lfFaceName, Items[Index]);
  end;
  newFont := CreateFontIndirect(lf);
  oldFont := SelectObject(FCanvas.Handle, newFont);
  GetTextMetrics(FCanvas.Handle, Metrics);
  Result := (Metrics.tmPitchAndFamily and TMPF_TRUETYPE) <> 0;
  SelectObject(FCanvas.Handle, oldFont);
  DeleteObject(newFont);
end;

procedure TFontListBox.DrawTT(Background: TColor);
  procedure DrawT(OrgX, OrgY: Integer; Color: TColor);
  begin
   with FBitmap.Canvas do begin
     Brush.Style := bsSolid;
     Pen.Color := Color;
     MoveTo(OrgX,OrgY);
     LineTo(OrgX+7,OrgY);
     LineTo(OrgX+7,OrgY+3);
     MoveTo(OrgX,OrgY);
     LineTo(OrgX,OrgY+3);
     MoveTo(OrgX+1,OrgY);
     LineTo(OrgX+1,OrgY+1);
     MoveTo(OrgX+6,OrgY);
     LineTo(OrgX+6,OrgY+1);
     MoveTo(OrgX+3,OrgY);
     LineTo(OrgX+3,OrgY+8);
     MoveTo(OrgX+4,OrgY);
     LineTo(OrgX+4,OrgY+8);
     MoveTo(OrgX+1,OrgY+8);
     LineTo(OrgX+6,OrgY+8);
   end;
  end;
begin
  with FBitmap.Canvas do begin
    Brush.Style := bsClear;
    Brush.Color := background;
    FillRect(Rect(0,0,12,12));
    DrawT(0,0,clGray);
    DrawT(4,3,clBlack);
  end;
end;

procedure TFontListBox.Click;
begin
    Hint:=Items[ItemIndex] ;
    inherited ;
end;

procedure TFontListBox.ReSetItemHeight ;
var i,h1,h2:Integer ;
begin
     FCanvas.Font:=Font ;
     FCanvas.Font.CharSet:=DEFAULT_CHARSET;
     h1:=0;
     for i:=0 to Items.Count-1 do
     begin
          h2:=FCanvas.TextHeight('0');
          if h1<h2 then h1:=h2 ;
     end;
     if h1>0 then ItemHeight:=h1 ;
end;

constructor TSwComboColor.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColorWidth := 18;
  FSortBy := ccsUnsorted;
  Style := csOwnerDrawFixed;
  FCustomColor := clWhite ;
  FCustomColorText := 'Custom Color';
end;

procedure TSwComboColor.AddColor(ColorText: String; Color: TColor);
var
  X: Integer;
begin
  if (FSortBy = ccsUnsorted) or (Items.Count = 0) then
    Items.AddObject(ColorText, Pointer(Color))
  else if FSortBy = ccsColor then
    begin
      for X := 0 to Items.Count - 1 do
      begin
        if TColor(Items.Objects[X]) > Color then
        begin
          Break;
        end;
      end;
      Items.InsertObject(X , ColorText, Pointer(Color));
    end
  else
    begin
      for X := 0 to Items.Count - 1 do
      begin
        if AnsiLowerCase(Items[X]) > AnsiLowerCase(ColorText) then
        begin
          Break;
        end;
      end;
      Items.InsertObject(X , ColorText, Pointer(Color));
    end;
end;

function TSwComboColor.GetSelectedColor: TColor;
begin
  if ItemIndex = -1 then
    Result := -1
  else
     if (ItemIndex=Items.Count-1) then
         Result := FCustomColor
     else
         Result := TColor(Items.Objects[ItemIndex]);
end;

function TSwComboColor.GetSelectedColorText: String;
begin
  if ItemIndex = -1 then
    Result := ''
  else
     if (ItemIndex=Items.Count-1) then
         Result := FCustomColorText
     else
         Result := Items[ItemIndex];
end;

procedure TSwComboColor.SetColorWidth(Value: Integer);
begin
  if (FColorWidth <> Value) and (Value > 4) then
    begin
      FColorWidth := Value;
      if not (csDesigning in ComponentState) then
        Invalidate;
    end;
end;

procedure TSwComboColor.SetSelectedColor(Value: TColor);
var
  X: Integer;
begin
  ItemIndex:=-1 ;
  for X := 0 to Items.Count - 1 do
  begin
    if TColor(Items.Objects[X]) = Value then
    begin
      ItemIndex := X;
      Break;
    end;
  end;
  if (ItemIndex=-1) then
  begin
    FCustomColor:=Value;
    ItemIndex := Items.Count - 1;
  end;
end;

procedure TSwComboColor.SetCustomColorText(Value: String);
begin
  if FCustomColorText <> Value then begin
    FCustomColorText := Value;
    Invalidate;
  end;
end;

procedure TSwComboColor.SetSelectedColorText(Value: String);
var
  X: Integer;
begin
  for X := 0 to Items.Count - 1 do
  begin
    if Items[X] = Value then
    begin
      ItemIndex := X;
      Break;
    end;
  end;
end;

procedure TSwComboColor.SetSortBy(Value: TSortBy);
var
  C: TColor;
  X: Integer;
  Y: Integer;
begin
  if FSortBy <> Value then
    FSortBy := Value;
    { Use a "Buble Sort". Not the fastest algorithm, but it works fine here! }
    if FSortBy <> ccsUnsorted then
    begin
      C := SelectedColor;
      X := 0;
      while X < Items.Count - 1 do
      begin
        Y := Items.Count -1;
        while Y > X do
        begin
          if FSortBy = ccsColor then
            begin
              if TColor(Items.Objects[Y]) < TColor(Items.Objects[Y - 1]) then
                Items.Exchange(Y, Y - 1);
            end
          else
            begin
              if AnsiLowerCase(Items[Y]) < AnsiLowerCase(Items[Y - 1]) then
                Items.Exchange(Y, Y - 1);
            end;
          Y := Y - 1;
        end;
        X := X + 1;
      end;
      SelectedColor := C;
    end
end;

procedure TSwComboColor.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  ColorR: TRect;
  TextR: TRect;
  FullRect : TRect ;
  OldColor: TColor;
  DrawItemColor : TColor ;
  DrawItemText : PChar ;
  TempShowMode : TShowMode ;
begin
  ColorR.Left := Rect.Left + 1;
  ColorR.Top := Rect.Top + 1;
  ColorR.Right := Rect.Left + FColorWidth - 1;
  ColorR.Bottom := Rect.Top + ItemHeight - 1;
  TextR.Left := Rect.Left + FColorWidth + 4;
  TextR.Top := Rect.Top + 1;
  TextR.Right := Rect.Right;
  TextR.Bottom := Rect.Bottom - 1;
  FullRect.TopLeft:=ColorR.TopLeft ;
  FullRect.BottomRight:=TextR.BottomRight;
  TempShowMode := FShowMode ;
  if (Index=Items.Count-1) then
  begin
      DrawItemColor := FCustomColor ;
      DrawItemText := PChar(FCustomColorText) ;
      TempShowMode := ccsmBoth ;
  end
  else
  begin
      DrawItemColor := TColor(Items.Objects[Index]) ;
      DrawItemText := PChar(Items[Index]) ;
  end;
  with Canvas do
    begin
      FillRect(Rect);	{ clear the rectangle }
      OldColor := Brush.Color;
      Brush.Color := DrawItemColor;
      case TempShowMode of
           ccsmBoth:
           begin
              Rectangle(ColorR.Left, ColorR.Top, ColorR.Right, ColorR.Bottom);
              Brush.Color := OldColor;
              DrawText(Handle, DrawItemText, -1, TextR, DT_VCENTER or DT_SINGLELINE);
           end;
           ccsmColorRect:
           begin
              Rectangle(Rect.Left + 1, Rect.Top+1, Rect.Right-1, Rect.Bottom-1);
              Brush.Color := OldColor;
           end;
           ccsmText:
           begin
              Brush.Color := OldColor;
              DrawText(Handle, DrawItemText, -1, FullRect, DT_VCENTER or DT_SINGLELINE);
           end;
      end;
    end;
end;

procedure TSwComboColor.SetShowMode(Value: TShowMode);
begin
  if FShowMode<>Value then
  begin
      FShowMode := Value ;
      if not (csDesigning in ComponentState) then
        Invalidate;
  end;
end;

procedure TSwComboColor.Change ;
var SelectColorDialog: TColorDialog;
begin
    if ItemIndex=Items.Count-1 then
    begin
        SelectColorDialog := TColorDialog.Create(Self) ;
        SelectColorDialog.Color := FCustomColor ;
        if SelectColorDialog.Execute then
        begin
            SetSelectedColor(SelectColorDialog.Color) ;
        end;
        SelectColorDialog.Free ;
    end;
    inherited ;
end;

end.
