{ Author: Jose Sebastian Battig
  Version: 1.0
  E-Mail: k2xt@iname.com

  Program parameters:
  /? : Show help
  /start ServiceName [Param1, Param2, ..., ParamN] : Starts the specified service
  /stop ServiceName : Stops the specified service
  /pause ServiceName : Pauses the specified service
  /continue ServiceName : Continues a the paused specified service
  /list : Lists all the services and it the state of each one }

program svccont;
{$APPTYPE CONSOLE}
uses
  SysUtils, Windows, WinSvc;

procedure ShowHelp;
begin
  Writeln ('Control windows services. It can start, pause, stop and list services.'#13#10);
  Writeln ('Syntax:');
  Writeln ('svccont /?                    Shows this help');
  Writeln ('svccont /start ServiceName [Param1, Param2, ..., ParamN] Starts the specified service');
  Writeln ('svccont /stop ServiceName     Stops the specified service');
  Writeln ('svccont /pause ServiceName    Pauses the specified service');
  Writeln ('svccont /continue ServiceName Continues a the paused specified service');
  Writeln ('svccont /list               Lists all the services and it the state of each one');
end;

procedure ShowError (const ErrorMsg : string);
begin
  Writeln (#13#10#13#10'Error trying to peform the specified action.');
  Writeln ('Error Message: ' + ErrorMsg + '.');
end;

procedure ListServices;
var
  hscManager : THandle;
  EnumServiceStatus : pointer;
  BytesNeeded, ServicesReturned, ResumeHandle : DWORD;
  BytesAllocated : DWORD;
  procedure ShowServices;
  type
    TArrayEnumServicesStatus = array of TEnumServiceStatus;
  const
    StatusText : array [SERVICE_STOPPED..SERVICE_PAUSED] of string =
                       ('Stopped', 'Starting', 'Stopping', 'Started',
                        'Restarting', 'Pausing', 'Paused');
  var
    i : Integer;
  begin
    for i := 0 to ServicesReturned - 1 do
      with TArrayEnumServicesStatus ((@EnumServiceStatus)^) [i] do
        begin
          Writeln (Format ('%s - %s: %s', [lpServiceName, lpDisplayName, uppercase (Statustext [ServiceStatus.dwCurrentState])]));
        end;
  end;
  procedure CallEnumServices;
  begin
    if WinSvc.EnumServicesStatus (hScManager,
                                 SERVICE_TYPE_ALL,
                                 SERVICE_STATE_ALL,
                                 TEnumServiceStatus (EnumServiceStatus^),
                                 BytesAllocated,
                                 BytesNeeded, ServicesReturned, ResumeHandle)
      then
      begin
        ShowServices;
      end
      else
      case GetLastError of
        ERROR_ACCESS_DENIED : ShowError ('The specified handle was not opened with SC_MANAGER_ENUMERATE_SERVICE access');
        ERROR_INVALID_HANDLE : ShowError ('The specified handle is invalid');
        ERROR_INVALID_PARAMETER : ShowError ('A parameter that was specified is invalid');
        ERROR_MORE_DATA :
          begin
            ShowServices;
            ReallocMem (EnumServiceStatus, BytesNeeded);
            BytesAllocated := BytesNeeded;
            CallEnumServices;
          end;
        else ShowError ('Unknown Error');
      end
  end;
begin
  ResumeHandle := 0;
  hScManager := OpenSCManager (nil, nil, GENERIC_READ or GENERIC_EXECUTE);
  try
    if hscManager <> 0
      then
      begin
        BytesAllocated := 1024;
        GetMem (EnumServiceStatus, BytesAllocated);
        try
          CallEnumServices;
        finally
          FreeMem (EnumServiceStatus, BytesAllocated);
        end;
      end
      else
      case GetLastError of
        ERROR_ACCESS_DENIED : ShowError ('Acces Denied');
        ERROR_DATABASE_DOES_NOT_EXIST : ShowError ('Database doesn''t exists');
        ERROR_INVALID_PARAMETER : ShowError ('Invalid Parameter');
        else ShowError ('Unknown Error');
      end;
  finally
    if hscManager <> 0
      then CloseServiceHandle (hscManager);
  end;
end;

procedure UnknownCommand;
begin
  Writeln ('Unknown Command Option.');
end;

type
  TServiceAction = (saStart, saStop, saPause, saContinue);
  PPChar = ^PChar;

function CreateArgsArray : PChar;
var
  i : Integer;
  p : PPChar;
begin
  if ParamCount - 2 > 0
    then
    begin
      GetMem (Result, (ParamCount - 2) * SizeOf (PChar));
      p := @Result [0];
      for i := 3 to ParamCount do
        begin
          p^ := StrNew (PChar (ParamStr (i)));
          Inc (p);
        end;
    end
    else Result := nil;
end;

procedure DestroyArgsArray (Args : PChar);
var
  i : Integer;
  p : PPChar;
begin
  if Args <> nil
    then
    begin
      p := @Args [0];
      for i := 3 to ParamCount do
        begin
          StrDispose (p^);
          Inc (p);
        end;
      FreeMem (Args, (ParamCount - 2) * SizeOf (PChar));
    end;
end;

procedure PerformAction (const ServiceName : string; Action : TServiceAction);
const
  ActionDoing : array [TServiceAction] of string = ('Starting', 'Stopping', 'Pausing', 'Continuing');
  ActionDone : array [TServiceAction] of String = ('Started', 'Stopped', 'Paused', 'Continued');
  WaitFor : array [TServiceAction] of DWORD = (SERVICE_RUNNING, SERVICE_STOPPED, SERVICE_PAUSED, SERVICE_RUNNING);
var
  hscManager, hService : THandle;
  Args : PChar;
  ServiceStatus : TServiceStatus;
  ActionResult : Boolean;
begin
  SetLastError ($FFFFFFFF);
  hScManager := OpenSCManager (nil, nil, GENERIC_READ or GENERIC_EXECUTE);
  try
    if hscManager <> 0
      then
      begin
        hService := OpenService (hScManager, PChar (ServiceName), GENERIC_READ or GENERIC_EXECUTE
                                 or SERVICE_USER_DEFINED_CONTROL);
        try
          if hService <> 0
            then
            begin
              Write (ActionDoing [Action] + ' ' + ServiceName);
              case Action of
                saStart :
                  begin
                    Args := CreateArgsArray;
                    try
                      ActionResult := WinSvc.StartService (hService, ParamCount - 2, Args);
                    finally
                      DestroyArgsArray (Args);
                    end;
                  end;
                saStop : ActionResult := WinSvc.ControlService (hService, SERVICE_CONTROL_STOP, ServiceStatus);
                saPause : ActionResult := WinSvc.ControlService (hService, SERVICE_CONTROL_PAUSE, ServiceStatus);
                saContinue : ActionResult := winsvc.ControlService (hService, SERVICE_CONTROL_CONTINUE, ServiceStatus);
                else ActionResult := False;
              end;
              if ActionResult
                then
                begin
                  repeat
                    Write ('.');
                    Sleep (1000);
                  until QueryServiceStatus (hService, ServiceStatus) and (WaitFor [Action] = ServiceStatus.dwCurrentState);
                  Writeln (#13#10 + ActionDone [Action] + ' ' + ServiceName + #13#10);
                end
                else
                case GetLastError of
                  ERROR_ACCESS_DENIED :  ShowError ('The specified handle was not opened with the necessary access');
                  ERROR_DEPENDENT_SERVICES_RUNNING : ShowError ('The service cannot be stopped because other running services are dependent on it');
                  ERROR_INVALID_SERVICE_CONTROL : ShowError ('The requested control code is not valid, or it is unacceptable to the service');
                  ERROR_SERVICE_CANNOT_ACCEPT_CTRL : ShowError ('The requested control code cannot be sent to the service because the state of the service is SERVICE_STOPPED, SERVICE_START_PENDING, or SERVICE_STOP_PENDING');
                  ERROR_SERVICE_NOT_ACTIVE : ShowError ('The service has not been started');
                  ERROR_SERVICE_REQUEST_TIMEOUT : ShowError ('The service did not respond to the start request in a timely fashion');
                  ERROR_INVALID_HANDLE : ShowError ('The specified handle is invalid');
                  ERROR_PATH_NOT_FOUND : ShowError ('The service binary file could not be found');
                  ERROR_SERVICE_ALREADY_RUNNING : ShowError ('An instance of the service is already running');
                  ERROR_SERVICE_DATABASE_LOCKED : ShowError ('The database is locked');
                  ERROR_SERVICE_DEPENDENCY_DELETED : ShowError ('The service depends on a service that does not exist or has been marked for deletion');
                  ERROR_SERVICE_DEPENDENCY_FAIL	: ShowError ('The service depends on another service that has failed to start');
                  ERROR_SERVICE_DISABLED : ShowError ('The service has been disabled');
                  ERROR_SERVICE_LOGON_FAILED : ShowError ('The service could not be logged on');
                  ERROR_SERVICE_MARKED_FOR_DELETE : ShowError ('The service has been marked for deletion');
                  ERROR_SERVICE_NO_THREAD : ShowError ('A thread could not be created for the Win32 service');
                  else ShowError ('Unknown Error');
                end;
            end
            else
            case GetLastError of
              ERROR_ACCESS_DENIED : ShowError ('The specified service control manager database handle does not have access to the service');
              ERROR_INVALID_HANDLE : ShowError ('The specified handle is invalid');
              ERROR_INVALID_NAME : ShowError ('The specified service name is invalid');
              ERROR_SERVICE_DOES_NOT_EXIST : ShowError ('The specified service does not exist');
              else ShowError ('Unknown Error');
            end;
        finally
          if hService <> 0
            then CloseServiceHandle (hService);
        end;
      end
      else
      case GetLastError of
        ERROR_ACCESS_DENIED : ShowError ('Acces Denied');
        ERROR_DATABASE_DOES_NOT_EXIST : ShowError ('Database doesn''t exists');
        ERROR_INVALID_PARAMETER : ShowError ('Invalid Parameter');
        else ShowError ('Unknown Error');
      end;
  finally
    if hscManager <> 0
      then CloseServiceHandle (hscManager);
  end;
end;

procedure StartService (const ServiceName : string);
begin
  PerformAction (ServiceName, saStart);
end;

procedure StopService (const ServiceName : string);
begin
  PerformAction (ServiceName, saStop);
end;

procedure PauseService (const ServiceName : string);
begin
  PerformAction (ServiceName, saPause);
end;

procedure ContinueService (const ServiceName : string);
begin
  PerformAction (ServiceName, saContinue);
end;

begin
  if ParamCount = 1
    then if ParamStr (1) = '/?'
      then ShowHelp
      else if UpperCase (ParamStr (1)) = '/LIST'
        then ListServices
        else UnknownCommand
    else if ParamCount >= 2
      then if UpperCase (ParamStr (1)) = '/START'
        then StartService (ParamStr (2))
        else if UpperCase (ParamStr (1)) = '/STOP'
          then StopService (ParamStr (2))
          else if UpperCase (ParamStr (1)) = '/PAUSE'
            then PauseService (ParamStr (2))
            else if UpperCase (ParamStr (1)) = '/CONTINUE'
              then ContinueService (ParamStr (2))
              else UnknownCommand
      else if ParamCount = 0
        then ShowHelp;
end.
