{

Copyright  2000, StreamSec HB - http://www.streamsec.com/
All rights reserved.

}
unit Macro;

interface

uses
  Classes, SysUtils, Controls, Windows, FilerSup, DesignPoint;

const
  diNone = -1;
  diBack = -2;
  diPrev = -3;
  diNext = -4;
  diDown = -5;
  diUp = -6;
  diUpNext = -7;
  diMin = -7;

  S_ERR_INVALID_ARG = 'Invalid argument in macro %s on line %d.';

type
  TMacroCmd = (mcNotify,mcExecute,mcSelect,mcMouseToControl,mcMouseToPoint,
               mcMouseDown,mcMouseUp,mcSleep,mcRepaint);

  TMacroItem = class(TOwnedObjectCollectionItem)
  private
    FCmd: TMacroCmd;
    FmSeconds: Cardinal;
    FSelectDocIndex: Integer;
    FControl: TComponentContainer;
    FNotify: TNotifyContainer;
    FExecute: TExecuteContainer;
    FPoint: TComponentContainer;
    procedure SetCmd(const Value: TMacroCmd);
    procedure SetControl(const Value: TComponentContainer);
    procedure SetExecute(const Value: TExecuteContainer);
    procedure SetmSeconds(const Value: Cardinal);
    procedure SetNotify(const Value: TNotifyContainer);
    procedure SetSelectDocIndex(const Value: Integer);
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure DoExecute; virtual;
    function GetDisplayName: string; override;
    procedure MouseToPoint(Q: TDesignPoint);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    function ControlStored: Boolean; virtual;
    function Execute: Boolean; virtual; 
    function GetMaxDocIndex: Integer; virtual;
    function mSecondsStored: Boolean; virtual;
    function PointStored: Boolean; virtual;
    function SelDocIndexStored: Boolean; virtual;
    property MaxDocIndex: Integer read GetMaxDocIndex;
  published
    property Cmd: TMacroCmd read FCmd write SetCmd;
    property Control: TComponentContainer read FControl
                                          write SetControl stored False;
    property mSeconds: Cardinal read FmSeconds
                                write SetmSeconds;
    property Method: TExecuteContainer read FExecute
                                       write SetExecute; 
    property Point: TComponentContainer read FPoint write FPoint;
    property SelectDocIndex: Integer read FSelectDocIndex
                                     write SetSelectDocIndex;
    property Notify: TNotifyContainer read FNotify write SetNotify;
  end;

  TCustomMacro = class(TOwnedObjectCollection)
  private
    function GetItems(index: Integer): TMacroItem;
    procedure SetItems(index: Integer; const Value: TMacroItem);
  public
    function Add: TMacroItem;
    procedure ExecuteMacro;
    function Owner: TPersistent;
    property Items[index: Integer]: TMacroItem read GetItems write SetItems;
  end;

  TMacro = class(TCustomMacro)
  public
    constructor Create(AOwner: TPersistent);
  end;

  TFilerMacroItem = class(TCollectionItem)
  private
    FmSeconds: Cardinal;
    FSelectDocIndex: Integer;
    FControl: string;
    FPoint: string;
    FExecute: string;
    FCmd: TMacroCmd;
    FNotify: string;
    FControlStored: Boolean;
    FmSecondsStored: Boolean;
    FExecuteStored: Boolean;
    FPointStored: Boolean;
    FSelDocIndexStored: Boolean;
    FNotifyStored: Boolean;
    procedure SetCmd(const Value: TMacroCmd);
  protected
    procedure AssignTo(Dest: TPersistent); override;
  public
    procedure Assign(Source: TPersistent); override;
  published
    property Cmd: TMacroCmd read FCmd write SetCmd;
    property Control: string read FControl
                             write FControl
                             stored FControlStored;
    property mSeconds: Cardinal read FmSeconds
                                write FmSeconds
                                stored FmSecondsStored;
    property Method: string read FExecute
                            write FExecute
                            stored FExecuteStored;
    property Point: string read FPoint write FPoint stored FPointStored;
    property SelectDocIndex: Integer read FSelectDocIndex
                                     write FSelectDocIndex
                                     stored FSelDocIndexStored;
    property Notify: string read FNotify
                            write FNotify
                            stored FNotifyStored;
  end;

  TFilerMacro = class(TCollection)
  public
    constructor Create;
  end;


  EInvalidArgument = class(Exception);

  procedure MouseTo(Control: TControl);

implementation

uses
  Docs, Messages, Forms;

function GetDoc(Item: TMacroItem): TDoc;
begin
  Result := nil;
  if Item = nil then Exit;
  if Item.Collection = nil then Exit;
  if (Item.Collection as TCustomMacro).Owner = nil then Exit;
  Result := (Item.Collection as TCustomMacro).Owner as TDoc;
end;       

procedure InternalMouseToPoint(x, y: Integer; Delay: Cardinal);
var
  P: TPoint;
  T: Integer;
  R, dx, dy: Real;
begin
  P := Mouse.CursorPos;
  R := sqrt(sqr(P.x - x) + sqr(P.y - y));
  dx := (P.x - x)/R;
  dy := (P.y - y)/R;
  T := Trunc(R);
  while T > 0 do begin
    P.x := Trunc(T*dx) + x;
    P.y := Trunc(T*dy) + y;
    Mouse.CursorPos := P;
    Sleep(Delay);
    Dec(T);
  end;
end;

procedure MouseTo(Control: TControl);
var
  Q, C: TPoint;
begin
  Q := Control.ClientOrigin;
  C := Control.ClientRect.BottomRight;
  Q.x := Q.x + (C.x div 2);
  Q.y := Q.y + (C.y div 2);
  InternalMouseToPoint(Q.x,Q.y,5);
end;

procedure MouseDown(Control: TControl);
var
  Msg: TWMMouse;
begin
  Msg.Msg := WM_LBUTTONDOWN;
  Msg.Keys := 0;
  Msg.XPos := Mouse.CursorPos.x;
  Msg.YPos := Mouse.CursorPos.y;
  Control.Dispatch(Msg);
end;    

procedure MouseUp(Control: TControl);
var
  Msg: TWMMouse;
begin
  Msg.Msg := WM_LBUTTONUP;
  Msg.Keys := 0;
  Msg.XPos := Mouse.CursorPos.x;
  Msg.YPos := Mouse.CursorPos.y;
  Control.Dispatch(Msg);
end;

{ TMacroItem }

procedure TMacroItem.AssignTo(Dest: TPersistent);
begin
  if Dest is TMacroItem then begin
    (Dest as TMacroItem).Cmd := Cmd;
    (Dest as TMacroItem).Control.Assign(Control);
    (Dest as TMacroItem).Method.Assign(Method);
    (Dest as TMacroItem).mSeconds := mSeconds;
    (Dest as TMacroItem).Notify.Assign(Notify);
    (Dest as TMacroItem).Point.Assign(Point);
    (Dest as TMacroItem).FSelectDocIndex := SelectDocIndex;
  end else inherited AssignTo(Dest);
end;

function TMacroItem.ControlStored: Boolean;
begin
  Result := FCmd in [mcMouseToControl,mcMouseDown,mcMouseUp,mcRepaint];
end;

constructor TMacroItem.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FSelectDocIndex := diNone;
  FExecute := TExecuteContainer.Create(Self);
  FNotify := TNotifyContainer.Create(Self);
  FPoint := TComponentContainer.Create(Self);
  FPoint.ComponentClass := TDesignPoint;
  FControl := TComponentContainer.Create(Self);
  FControl.ComponentClass := TControl;
end;

destructor TMacroItem.Destroy;
begin
  FControl.Free;
  FPoint.Free;
  FNotify.Free;
  FExecute.Free;
  inherited
end;

procedure TMacroItem.DoExecute;
begin
  case Cmd of
    mcMouseToControl: MouseTo(Control.Component as TControl);
    mcMouseToPoint:   MouseToPoint(Point.Component as TDesignPoint);
    mcMouseUp:        MouseUp(Control.Component as TControl);
    mcMouseDown:      MouseDown(Control.Component as TControl);
    mcSleep:          Sleep(FmSeconds);
    mcRepaint:        (Control.Component as TControl).Repaint;
    mcSelect:         case SelectDocIndex of
                        diNone:;
                        diBack: GetDoc(Self).Owner.Back;
                        diDown: GetDoc(Self).Owner.Down;
                        diNext: GetDoc(Self).Owner.Next;
                        diPrev: GetDoc(Self).Owner.Prev;
                        diUp:   GetDoc(Self).Owner.Up;
                        diUpNext: GetDoc(Self).Owner.UpNext;
                      else
                        GetDoc(Self).Owner[SelectDocIndex].Selected := True;
                      end;
    mcExecute:        TObjectMethod(FExecute.Method);
    mcNotify:         begin
                        TNotifyEvent(FNotify.Method)(GetDoc(Self));
                      end;
  end;
end;

function TMacroItem.Execute: Boolean;
begin
  case Cmd of
    mcMouseToControl,
    mcMouseUp,
    mcMouseDown,
    mcRepaint:        begin
                        Control.Name := Control.Name;
                        Result := Assigned(Control.Component);
                      end;
    mcMouseToPoint:   begin
                        Point.Name := Point.Name;
                        Result := Assigned(Point.Component);
                      end;
    mcSleep:          Result := True;
    mcSelect:         Result := (GetDoc(Self) <> nil) and
                                (GetDoc(Self).Owner <> nil) and
                                (FSelectDocIndex >= diMin) and
                                (FSelectDocIndex <= GetMaxDocIndex);
    mcExecute:        begin
                        Method.Name := Method.Name;
                        Result := Assigned(Method.Ptr);
                      end;
    mcNotify:         begin
                        Notify.Name := Notify.Name;
                        Result := Assigned(Notify.Ptr);
                      end;  
  else
    Result := False;
  end;
  if Result then DoExecute;
end;

function TMacroItem.GetDisplayName: string;

  function CompCoord(Comp: TComponent): string;
  begin
    Result := IntToStr((Comp as TDesignPoint).Left) + ',' +
              IntToStr((Comp as TDesignPoint).Top);
  end;

begin
  case Cmd of     
    mcExecute:
              if Assigned(FExecute) then
                Result := 'Execute(' + FExecute.Name + ')'
              else
                Result := 'Execute(nil)';
    mcMouseDown:
              if Assigned(FControl) then
                Result := 'MouseDown(' + Control.Name + ')'
              else
                Result := 'MouseDown(nil)';
    mcMouseToControl:
              if Assigned(FControl) then
                Result := 'MouseToControl(' + Control.Name + ')'
              else
                Result := 'MouseToControl(nil)';
    mcMouseToPoint:
              if Point.Ptr = nil then
                Result := 'MouseToPoint(,)'
              else
                Result := 'MouseToPoint(' + CompCoord(Point.Component) + ')';
    mcMouseUp:
              if Assigned(FControl) then
                Result := 'MouseUp(' + Control.Name + ')'
              else
                Result := 'MouseUp(nil)';
    mcNotify: if Assigned(FNotify) then
                Result := 'Notify(' + FNotify.Name + ')'
              else
                Result := 'Notify(nil)'; 
    mcRepaint:
              if Assigned(FControl) then 
                Result := 'Repaint(' + Control.Name + ')'
              else
                Result := 'Repaint(nil)';
    mcSelect: case SelectDocIndex of
                diBack: Result := 'Select(diBack)';
                diDown: Result := 'Select(diDown)';
                diNext: Result := 'Select(diNext)';
                diNone: Result := 'Select(diNone)';
                diPrev: Result := 'Select(diPrev)';
                diUp:   Result := 'Select(diUp)';
                diUpNext: Result := 'Select(diUpNext)';
              else
                Result := 'Select(' + IntToStr(SelectDocIndex) + ')';
              end;
    mcSleep:  Result := 'Sleep(' + IntToStr(mSeconds) + ')';
  end;
end;
function TMacroItem.GetMaxDocIndex: Integer;
var
  Doc: TDoc;
begin
  Result := MaxInt;
  Doc := GetDoc(Self);
  if Doc = nil then Exit;  
  Result := -1;
  if Doc.Owner = nil then Exit;
  Result := Doc.Owner.Count - 1;
end;

procedure TMacroItem.MouseToPoint(Q: TDesignPoint);
var
  P: TPoint;
begin
  P.x := Q.Left;
  P.y := Q.Top;
  P := GetForm.ClientToScreen(P);
  InternalMouseToPoint(P.x,P.y,Q.SleepPerPixel);
end;

function TMacroItem.mSecondsStored: Boolean;
begin
  Result := FCmd = mcSleep;
end;

function TMacroItem.PointStored: Boolean;
begin
  Result := FCmd in [mcMouseToPoint];
end;

function TMacroItem.SelDocIndexStored: Boolean;
begin
  Result := FCmd = mcSelect;
end;

procedure TMacroItem.SetCmd(const Value: TMacroCmd);
begin
  FCmd := Value;
  if Cmd = mcSelect then SelectDocIndex := diNone;
  Changed(False);
end;

procedure TMacroItem.SetControl(const Value: TComponentContainer);
begin
  FControl.Assign(Value);
  Changed(False);
end;

procedure TMacroItem.SetExecute(const Value: TExecuteContainer);
begin
  FExecute.Assign(Value);
  Changed(False);
end;

procedure TMacroItem.SetmSeconds(const Value: Cardinal);
begin
  FmSeconds := Value; 
  Changed(False);
end;

procedure TMacroItem.SetNotify(const Value: TNotifyContainer);
begin
  FNotify.Assign(Value);
  Changed(False);
end;

procedure TMacroItem.SetSelectDocIndex(const Value: Integer);
begin
  FSelectDocIndex := Value;
  Changed(False);
end;

{ TCustomMacro }

function TCustomMacro.Add: TMacroItem;
begin
  Result := TMacroItem(inherited Add)
end;

procedure TCustomMacro.ExecuteMacro;
var
  I: Integer;
begin
  I := 0;
  while I < Count do begin
    if not Items[I].Execute then
      raise EInvalidArgument.CreateFmt(S_ERR_INVALID_ARG,[GetNamePath,I]);
    Inc(I);
  end;
end;

function TCustomMacro.GetItems(index: Integer): TMacroItem;
begin
  Result := TMacroItem(inherited GetItem(index));
end;

function TCustomMacro.Owner: TPersistent;
begin
  Result := GetOwner;
end;

procedure TCustomMacro.SetItems(index: Integer; const Value: TMacroItem);
begin
  inherited SetItem(index,Value);
end;

{ TFilerMacro }

constructor TFilerMacro.Create;
begin
  inherited Create(TFilerMacroItem);
end;

{ TMacro }

constructor TMacro.Create(AOwner: TPersistent);
begin
  inherited Create(AOwner,TMacroItem);
end;

{ TFilerMacroItem }

procedure TFilerMacroItem.Assign(Source: TPersistent);
var
  MI: TMacroItem;
  FMI: TFilerMacroItem;
begin
  if Source is TMacroItem then begin
    MI := Source as TMacroItem;
    Cmd := MI.Cmd;
    Notify := MI.Notify.Name;
    Method := MI.Method.Name;
    Control := MI.Control.Name;
    mSeconds := MI.mSeconds;
    Point := MI.Point.Name;
    SelectDocIndex := MI.SelectDocIndex;
  end else if Source is TFilerMacroItem then begin
    FMI := Source as TFilerMacroItem;
    Cmd := FMI.Cmd;
    Notify := FMI.Notify;
    Method := FMI.Method;
    Control := FMI.Control;
    mSeconds := FMI.mSeconds;
    Point := FMI.Point;
    SelectDocIndex := FMI.SelectDocIndex;
  end else inherited Assign(Source);
end;

procedure TFilerMacroItem.AssignTo(Dest: TPersistent);
var
  MI: TMacroItem;
begin
  if Dest is TMacroItem then begin
    MI := Dest as TMacroItem;
    MI.Cmd := Cmd;
    MI.Notify.Name := Notify;
    MI.Method.Name := Method;
    MI.Control.Name := Control;
    MI.mSeconds := mSeconds;
    MI.Point.Name := Point;
    MI.SelectDocIndex := SelectDocIndex;
  end else inherited AssignTo(Dest);
end;

procedure TFilerMacroItem.SetCmd(const Value: TMacroCmd);
begin
  FCmd := Value;
  FControlStored := Value in [mcMouseToControl,mcMouseDown,mcMouseUp,mcRepaint];
  FmSecondsStored := Value = mcSleep;
  FExecuteStored := Value = mcExecute;
  FPointStored := Value = mcMouseToPoint;
  FSelDocIndexStored := Value = mcSelect;
  FNotifyStored := Value = mcNotify;
end;

end.
