{

Copyright  2000, StreamSec HB - http://www.streamsec.com/
All rights reserved.

}

// Filer support classes

unit FilerSup;

interface

uses
  Classes, SysUtils, TypInfo, Forms;

const
  ERR_OWNED_COLLECTION = '%s is not TOwnedObjectCollection';
  ERR_OWNED_CLASS = '%s is not an owned object class';
  ERR_NO_FORM = 'Class %s: Owner did not return any form.';
  ERR_NO_OWNER = 'Class %s: Object does not have an owner.';

  IID_OwnerObject: TGUID = '{C3DCF240-9D18-11D4-AD6B-0000B4B70EE6}';

type
  EOwnedObject = class(Exception);

  IOwnerObject = interface(IUnknown)
    ['{C3DCF240-9D18-11D4-AD6B-0000B4B70EE6}']
    function AsPersistent: TPersistent;
    function GetForm: TCustomForm;
    function GetOwner: TPersistent;
    function OwnerInterface: IOwnerObject;
  end;

  TOwnedObject = class(TPersistent, IOwnerObject, IUnknown)
  private
    FOwner: TPersistent;
  protected
    function GetForm: TCustomForm; virtual;
    function GetOwner: TPersistent; override;   
    function OwnerInterface: IOwnerObject;
  public
    constructor Create(AOwner: TPersistent);
    function AsPersistent: TPersistent;
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  end;

  TOwnedObjectCollection = class(TCollection, IOwnerObject, IUnknown)
  private
    FOwner: TPersistent;
  protected
    function GetForm: TCustomForm; virtual;
    function GetOwner: TPersistent; override;   
    function OwnerInterface: IOwnerObject;
  public
    constructor Create(AOwner: TPersistent; ItemClass: TCollectionItemClass); 
    function AsPersistent: TPersistent;
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  end;

  TOwnedObjectCollectionItem = class(TCollectionItem, IOwnerObject, IUnknown)
  protected
    function GetForm: TCustomForm; virtual;  
    function OwnerInterface: IOwnerObject;
  public
    constructor Create(Collection: TCollection); override;
    function AsPersistent: TPersistent;   
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  end;

  TPropContainer = class(TOwnedObject)
  private
    FName: string;
    FPtr: Pointer;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    function GetComponent(Name: string): TComponent;
    function GetComponentName(Component: TComponent): string;
    function GetMethodAddress(const MethodName: string): Pointer;
    function GetMethodName(Address: Pointer): string;
    procedure SetName(const Value: string); virtual;
    procedure SetPtr(const Value: Pointer); virtual;
  public
    property Name: string read FName write SetName;
    property Ptr: Pointer read FPtr write SetPtr;
  end;

  TMethodContainer = class(TPropContainer)
  private
    FMethodTypeInfo: PTypeInfo;
    function GetMethod: TMethod;
    procedure SetMethod(const Value: TMethod);
    procedure SetMethodTypeInfo(const Value: PTypeInfo);
  protected
    procedure SetName(const Value: string); override;
    procedure SetPtr(const Value: Pointer); override;
  public
    property MethodTypeInfo: PTypeInfo read FMethodTypeInfo;
    property Method: TMethod read GetMethod write SetMethod;
  end;

  TNotifyContainer = class(TMethodContainer)
  public
    constructor Create(AOwner: TPersistent);
  end;

  TObjectMethod = procedure of object;

  TExecuteContainer = class(TMethodContainer)
  public
    constructor Create(AOwner: TPersistent);
  end;

  TComponentContainer = class(TPropContainer)
  private
    FComponentClass: TComponentClass;
    function GetComp: TComponent;
    procedure SetComponent(const Value: TComponent);
    procedure SetComponentClass(const Value: TComponentClass);
    function GetComponentTypeInfo: PTypeInfo;
  protected
    procedure SetName(const Value: string); override;
    procedure SetPtr(const Value: Pointer); override;
  public
    property Component: TComponent read GetComp write SetComponent;
    property ComponentClass: TComponentClass read FComponentClass write SetComponentClass;
    property ComponentTypeInfo: PTypeInfo read GetComponentTypeInfo;
  end;

implementation

{ TOwnedObject }

constructor TOwnedObject.Create(AOwner: TPersistent);
begin
  FOwner := AOwner;
end;

function _GetForm(OwnedObject: IOwnerObject): TCustomForm;
begin
  if (OwnedObject.GetOwner = nil) and (OwnedObject.OwnerInterface = nil) then
    raise EOwnedObject.CreateFmt(ERR_NO_OWNER,[OwnedObject.AsPersistent.ClassName]);
  if OwnedObject.GetOwner is TCustomForm then
    Result := OwnedObject.GetOwner as TCustomForm
  else if OwnedObject.OwnerInterface <> nil then
    Result := OwnedObject.OwnerInterface.GetForm
  else Result := nil;
  if Result = nil then
    raise EOwnedObject.CreateFmt(ERR_NO_FORM,[OwnedObject.AsPersistent.ClassName]);
end;

function TOwnedObject.GetForm: TCustomForm;
begin
  Result := _GetForm(Self);
end;

function TOwnedObject.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TOwnedObject._AddRef: Integer;
begin
  Result := -1;
end;

function TOwnedObject._Release: Integer;
begin
  Result := -1;
end;

function TOwnedObject.QueryInterface(const IID: TGUID; out Obj): HResult;
const
  E_NOINTERFACE = $80004002;
begin
  if GetInterface(IID, Obj) then Result := 0 else Result := E_NOINTERFACE;
end;

function TOwnedObject.AsPersistent: TPersistent;
begin
  Result := Self;
end;

function _OwnerInterface(AOwner: TPersistent): IOwnerObject;
begin                               
  Result := nil;
  if not Assigned(AOwner) then Exit;
  AOwner.GetInterface(IID_OwnerObject,Result);
end;

function TOwnedObject.OwnerInterface: IOwnerObject;
begin
  Result := _OwnerInterface(FOwner);
end;

{ TOwnedObjectCollection }

constructor TOwnedObjectCollection.Create(AOwner: TPersistent;
  ItemClass: TCollectionItemClass);
begin
  FOwner := AOwner;
  inherited Create(ItemClass);
end;

function TOwnedObjectCollection.GetForm: TCustomForm;
begin
  Result := _GetForm(Self);
end;         

function TOwnedObjectCollection.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TOwnedObjectCollection._AddRef: Integer;
begin
  Result := -1;
end;

function TOwnedObjectCollection._Release: Integer;
begin
  Result := -1;
end;

function TOwnedObjectCollection.QueryInterface(const IID: TGUID;
  out Obj): HResult;
const
  E_NOINTERFACE = $80004002;
begin
  if GetInterface(IID, Obj) then Result := 0 else Result := E_NOINTERFACE;
end;

function TOwnedObjectCollection.AsPersistent: TPersistent;
begin
  Result := Self;
end;

function TOwnedObjectCollection.OwnerInterface: IOwnerObject;
begin
  Result := _OwnerInterface(FOwner);
end;

{ TOwnedObjectCollectionItem }

constructor TOwnedObjectCollectionItem.Create(
  Collection: TCollection);
begin
  if not (Collection is TOwnedObjectCollection) then
    raise EOwnedObject.CreateFmt(ERR_OWNED_COLLECTION,[Collection.ClassName]);
  inherited Create(Collection);
end;

function TOwnedObjectCollectionItem.GetForm: TCustomForm;
begin
  Result := (Collection as TOwnedObjectCollection).GetForm; 
  if Result = nil then
    raise EOwnedObject.CreateFmt(ERR_NO_FORM,[ClassName]);
end;

function TOwnedObjectCollectionItem._AddRef: Integer;
begin
  Result := -1;
end;

function TOwnedObjectCollectionItem._Release: Integer;
begin
  Result := -1;
end;

function TOwnedObjectCollectionItem.QueryInterface(const IID: TGUID;
  out Obj): HResult;
const
  E_NOINTERFACE = $80004002;
begin
  if GetInterface(IID, Obj) then Result := 0 else Result := E_NOINTERFACE;
end;

function TOwnedObjectCollectionItem.AsPersistent: TPersistent;
begin
  Result := Self;
end;

function TOwnedObjectCollectionItem.OwnerInterface: IOwnerObject;
begin
  if Collection is TOwnedObjectCollection then
    Result := Collection as TOwnedObjectCollection
  else Result := nil;
end;

{ TPropContainer }

procedure TPropContainer.AssignTo(Dest: TPersistent);
begin
  if Dest is TPropContainer then
    (Dest as TPropContainer).Name := Name
  else inherited
end;

function TPropContainer.GetComponent(Name: string): TComponent;
var
  I: Integer;
  OwnerName: string;
  Owner: TComponent;
begin
  I := Pos('.',Name);
  if I > 0 then begin
    OwnerName := Copy(Name,1,I-1);
    Delete(Name,1,I);
  end;
  if GetForm = nil then Result := nil
  else if (csLoading in GetForm.ComponentState) then Result := nil
  else if (OwnerName = '') then Result := GetForm.FindComponent(Name)
  else begin
    Owner := GetForm.FindComponent(OwnerName);
    if Assigned(Owner) then Result := Owner.FindComponent(Name)
    else begin
      Owner := GetForm.Owner;
      if Assigned(Owner) then begin
        Owner := Owner.FindComponent(OwnerName);
        if Assigned(Owner) then Result := Owner.FindComponent(Name)
        else Result := nil;
      end else Result := nil;
    end;
  end;
end;

function TPropContainer.GetComponentName(Component: TComponent): string;
begin
  if Component.Owner = nil then Result := ''
  else if Component.Owner = GetForm then Result := Component.Name
  else Result := Component.Owner.Name + '.' + Component.Name;
end;

function TPropContainer.GetMethodAddress(
  const MethodName: string): Pointer;
begin
  if GetForm <> nil then
    Result := GetForm.MethodAddress(MethodName)
  else Result := nil;
end;

function TPropContainer.GetMethodName(Address: Pointer): string;
begin
  if GetForm <> nil then
    Result := GetForm.MethodName(Address)
  else Result := '';
end;

procedure TPropContainer.SetName(const Value: string);
begin
  FName := Value;
end;

procedure TPropContainer.SetPtr(const Value: Pointer);
begin
  FPtr := Value;
end;

{ TMethodContainer }

function TMethodContainer.GetMethod: TMethod;
begin
  Result.Data := GetForm;
  Result.Code := Ptr;
end;

procedure TMethodContainer.SetMethod(const Value: TMethod);
begin
  Ptr := Value.Code;
end;

procedure TMethodContainer.SetMethodTypeInfo(const Value: PTypeInfo);
begin
  FMethodTypeInfo := Value;
end;

procedure TMethodContainer.SetName(const Value: string);
begin
  inherited SetPtr(GetMethodAddress(Value));
  inherited SetName(Value);
end;

procedure TMethodContainer.SetPtr(const Value: Pointer);
begin
  SetName(GetMethodName(Value));
end;

{ TComponentContainer }

function TComponentContainer.GetComp: TComponent;
begin
  Result := Ptr;
end;

function TComponentContainer.GetComponentTypeInfo: PTypeInfo;
begin
  if ComponentClass = nil then Result := nil
  else Result := ComponentClass.ClassInfo;
end;

procedure TComponentContainer.SetComponent(const Value: TComponent);
begin
  Ptr := Value;
end;

procedure TComponentContainer.SetComponentClass(
  const Value: TComponentClass);
begin
  FComponentClass := Value;
end;

procedure TComponentContainer.SetName(const Value: string);
begin
  inherited SetName(Value);
  inherited SetPtr(GetComponent(Value));
end;

procedure TComponentContainer.SetPtr(const Value: Pointer);
var
  Comp: TComponent;
begin
  Comp := Value;
  if Assigned(Comp) and (Comp is FComponentClass) then
    SetName(GetComponentName(Comp))
  else SetName('');
end;

{ TNotifyContainer }

constructor TNotifyContainer.Create(AOwner: TPersistent);
begin
  inherited Create(AOwner);
  SetMethodTypeInfo(TypeInfo(TNotifyEvent));
end;

{ TExecuteContainer }

constructor TExecuteContainer.Create(AOwner: TPersistent);
begin
  inherited Create(AOwner);
  SetMethodTypeInfo(TypeInfo(TObjectMethod));
end;

end.
