{

Copyright  2000, StreamSec HB - http://www.streamsec.com/
All rights reserved.

}
unit Docs;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ActnList, Macro, ImgList, FilerSup;

const
  ERR_DOCLIST_ADD = 'Could not insert an item at level %d.';

  IID_RTFDesigner: TGUID = '{C3DCF241-9D18-11D4-AD6B-0000B4B70EE6}';
  IID_DocsDesigner: TGUID = '{7ABFC1E0-9D32-11D4-AD6B-0000B4B70EE6}';

type
  TDocList = class;
  TDoc = class;

  TRTFStream = class(TStringStream)
  private
    FOwner: TDoc;
    function GetTitle: string;
    procedure SetOwner(const Value: TDoc);
  public
    constructor Create(const AValue: string; AOwner: TDoc);
    property Title: string read GetTitle;
    property Owner: TDoc read FOwner write SetOwner;
  end;

  TDoc = class(TOwnedObject)
  private        
    FData: TObject;
    FDeleted: Boolean;
    FStream: TRTFStream;
    FTag: Integer;
    FMacroNext: TCustomMacro;
    FMacroBack: TCustomMacro;
    FMacroDemo: TCustomMacro;
    FMacroPrev: TCustomMacro;
    FVisible: Boolean;
    FImageIndex: Integer;
    FStateIndex: Integer;
    function GetAsStream: TStream;
    function GetAsString: string;
    function GetCaption: string;
    function GetIndex: Integer;
    function GetLevel: Integer;
    procedure SetAsString(const Value: string);
    procedure SetCaption(const Value: string);
    procedure SetData(const Value: TObject);
    procedure SetTag(const Value: Integer);
    procedure SetDeleted(const Value: Boolean);
    function GetSelected: Boolean;
    procedure SetSelected(const Value: Boolean);
    procedure SetMacroBack(const Value: TCustomMacro);
    procedure SetMacroDemo(const Value: TCustomMacro);
    procedure SetMacroNext(const Value: TCustomMacro);
    procedure SetMacroPrev(const Value: TCustomMacro);
    procedure SetVisible(const Value: Boolean);
    procedure SetAsStream(const Value: TStream);
    procedure SetImageIndex(const Value: Integer);
    procedure SetStateIndex(const Value: Integer);
  protected
    function CheckDelete: Boolean;
    procedure DoChange;
    procedure DoSelected; 
    function GetDocList: TDocList;
  public
    constructor Create(AOwner: TDocList);
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    procedure Change;
    procedure Delete;
    procedure DeleteChildren;
    function GetParent: TDoc;
    function GetFirstChild: TDoc;
    function GetNamePath: string; override;
    function GetNextBranch: TDoc;
    function GetNextChild(Item: TDoc): TDoc;
    function GetNextSibling: TDoc;           
    function GetPrevSibling: TDoc;
    function HasSubString(const Value: string): Boolean;
    property Data: TObject read FData write SetData;
    property Deleted: Boolean read FDeleted write SetDeleted;
    property Index: Integer read GetIndex;
    property Level: Integer read GetLevel;   
    property Owner: TDocList read GetDocList; 
    property Selected: Boolean read GetSelected write SetSelected;
    property Text: string read GetAsString write SetAsString;
  published
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property MacroBack: TCustomMacro read FMacroBack write SetMacroBack;
    property MacroDemo: TCustomMacro read FMacroDemo write SetMacroDemo;
    property MacroNext: TCustomMacro read FMacroNext write SetMacroNext;
    property MacroPrev: TCustomMacro read FMacroPrev write SetMacroPrev;
    property StateIndex: Integer read FStateIndex write SetStateIndex;
    property Tag: Integer read FTag write SetTag;
    property TextAsStream: TStream read GetAsStream write SetAsStream;       
    property Title: string read GetCaption write SetCaption;
    property Visible: Boolean read FVisible write SetVisible;
  end;

  TDocCollection = class;

  TDocCollectionItem = class(TCollectionItem)
  private
    FTag: Integer;
    FLevel: Integer;
    FTitle: string;
    FRTF: string;
    FMacroNext: TFilerMacro;
    FMacroBack: TFilerMacro;
    FMacroDemo: TFilerMacro;
    FMacroPrev: TFilerMacro;
    FVisible: Boolean;
    FStateIndex: Integer;
    FImageIndex: Integer;
    procedure SetLevel(const Value: Integer);
    procedure SetTag(const Value: Integer);
    procedure SetRTF(const Value: string);
    procedure SetVisible(const Value: Boolean);
    procedure SetStateIndex(const Value: Integer);
    procedure SetImageIndex(const Value: Integer);
  protected
    procedure AssignTo(Dest: TPersistent); override;
    function GetDisplayName: string; override;
    function GetOwner: TPersistent; override;
    procedure SetTitle(const Value: string);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    function GetNamePath: string; override;
  published
    property Title: string read FTitle write SetTitle;
    property ImageIndex: Integer read FImageIndex write SetImageIndex default -1;
    property Level: Integer read FLevel write SetLevel default 0;
    property MacroBack: TFilerMacro read FMacroBack write FMacroPrev;
    property MacroDemo: TFilerMacro read FMacroDemo write FMacroPrev;
    property MacroNext: TFilerMacro read FMacroNext write FMacroPrev;
    property MacroPrev: TFilerMacro read FMacroPrev write FMacroPrev;
    property RTF: string read FRTF write SetRTF;
    property StateIndex: Integer read FStateIndex write SetStateIndex default -1;
    property Tag: Integer read FTag write SetTag default 0;
    property Visible: Boolean read FVisible write SetVisible default True;
  end;

  TDocCollection = class(TCollection)
  private
    function GetItem(index: Integer): TDocCollectionItem;
    procedure SetItem(index: Integer; const Value: TDocCollectionItem);
  public
    constructor Create;
    function Add: TDocCollectionItem;
    property Items[index: Integer]: TDocCollectionItem read GetItem write SetItem;
  end;

  EDocList = class(Exception);

  TDocs = class;

  TAddDocAction = (adaAddFirst,adaAddLast);

  TDocList = class(TOwnedObject)
  private
    FHistory: TStrings;
    FItems: TStrings;
    FItemsChanged: TList;
    FItemsDeleted: TList;
    FLines: TStrings;
    FSelected: TDoc;
    FUpdateCount: Integer;
    FOwnerName: string;
    function GetCount: Integer;
    function GetDocsComponent: TDocs;
    function GetItems(index: Integer): TDoc;    
    function GetLines: TStrings;  
    function GetUpdating: Boolean;
    procedure InternalEndUpdate;
    procedure LinesChange(Sender: TObject);
    procedure ReadList(Reader: TReader);
    procedure SetItems(index: Integer; const Value: TDoc);
    procedure SetLines(const Value: TStrings);
    procedure SetOwnerName(const Value: string);     
    procedure WriteList(Writer: TWriter);
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure DefineProperties(Filer: TFiler); override;
    procedure DoAddItem(RefItem, NewItem: TDoc; Action: TAddDocAction);
    procedure DoChangeItem(Item: TDoc);
    function DoDeleteItem(Item: TDoc): Boolean;
    procedure DoSelect(Item: TDoc);
    procedure DoUpdate;
    function InternalInsert(Index, Level: Integer): TDoc;
  public
    constructor Create(const AOwner: IOwnerObject);
    destructor Destroy; override;
    function Add(Level: Integer): TDoc;
    function AddFirst: TDoc;
    function AddFirstChild(Item: TDoc): TDoc;
    function AddFirstSibling(Item: TDoc): TDoc;
    function AddLast: TDoc;
    function AddLastChild(Item: TDoc): TDoc;
    function AddLastSibling(Item: TDoc): TDoc;
    procedure Back;
    procedure BeginUpdate;
    procedure Clear;
    procedure Down;
    procedure EndUpdate;
    function GetForm: TCustomForm; override;
    function GetNamePath: string; override;
    procedure LoadFromFile(const FileName: string);
    procedure LoadFromStream(AStream: TStream);
    procedure MakeIndex(const SearchStr: string; List: TStrings);
    procedure Next;
    procedure Prev;
    procedure SaveToFile(const FileName: string);
    procedure SaveToStream(AStream: TStream);
    procedure Up;
    procedure UpNext;
    property Count: Integer read GetCount;
    property History: TStrings read FHistory;
    property Items[index: Integer]: TDoc read GetItems write SetItems; default;
    property Owner: TDocs read GetDocsComponent;
    property OwnerName: string read FOwnerName write SetOwnerName;
    property Selected: TDoc read FSelected;
    property InternalUpdating: Boolean read GetUpdating; 
  published
    property Lines: TStrings read GetLines write SetLines stored False;
  end;        

  IRTFDesigner = interface(IUnknown)
    ['{C3DCF241-9D18-11D4-AD6B-0000B4B70EE6}']
    procedure AddAll;
    procedure AddDoc(Doc: TDoc);
    procedure ChangeDoc(Doc: TDoc);
    function Closed: Boolean;
    procedure Close(Sender: TObject);
    function GetDocsComponent: TDocs;
    procedure RemoveDoc(Doc: TDoc);
    procedure SelectDoc(Doc: TDoc);
    procedure SetDocsComponent(const Value: TDocs);
    procedure Show;
  end;

  IDocsDesigner = interface(IUnknown)
    ['{7ABFC1E0-9D32-11D4-AD6B-0000B4B70EE6}']
    function Closed: Boolean;
    procedure Close(Sender: TObject);
    function GetDocsComponent: TDocs;
    function GetEditorDlg: TForm;
    procedure SetDocsComponent(const Value: TDocs);
    procedure Show;
  end;

  TDocEvent = procedure (Sender: TObject; Doc: TDoc) of object;

  TDocs = class(TComponent, IOwnerObject, IUnknown)
  private       
    FDocDisplay: TTreeView;  
    FDocs: TDocList;
    FOnChange: TDocEvent;  
    FOnGetData: TDocEvent;
    FOnDeletion: TDocEvent;   
    FOnSelection: TDocEvent;
    FOnUpdate: TNotifyEvent;
    FRichEdit: TRichEdit;
    FForceVisible: Boolean;
    FRTFDesigner: IRTFDesigner;
    FDocsDesigner: IDocsDesigner;
    FUpWhenBack: Boolean;
    procedure ReadList(Reader: TReader);
    procedure ReadSel(Reader: TReader);
    procedure SetDocDisplay(const Value: TTreeView);
    procedure SetDocs(const Value: TDocList);
    procedure SetOnChange(const Value: TDocEvent);
    procedure SetOnGetData(const Value: TDocEvent);
    procedure SetOnDeletion(const Value: TDocEvent);
    procedure SetOnSelection(const Value: TDocEvent);
    procedure SetOnUpdate(const Value: TNotifyEvent);
    procedure SetRichEdit(const Value: TRichEdit);
    procedure SetForceVisible(const Value: Boolean);
    function GetCount: Integer;
    function GetSelIndex: Integer;
    procedure SetSelIndex(const Value: Integer);
    procedure WriteList(Writer: TWriter);
    procedure WriteSel(Writer: TWriter);
    procedure SetCount(const Value: Integer);
    procedure SetRTFDesigner(const Value: IRTFDesigner);
    procedure SetDocsDesigner(const Value: IDocsDesigner);
    procedure SetUpWhenBack(const Value: Boolean);
  protected
    procedure DefineProperties(Filer: TFiler); override;
    procedure DoAddDoc(RefItem, NewItem: TDoc; Action: TAddDocAction);
    procedure DoChangeDoc(Item: TDoc);
    procedure DoDeleteDoc(Item: TDoc);
    procedure DoSelectDoc(Item: TDoc);
    procedure DoUpdate;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    function OwnerInterface: IOwnerObject;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;   
    function AsPersistent: TPersistent; 
    procedure BackExecute(Sender: TObject); dynamic;
    procedure DemoExecute(Sender: TObject); dynamic;
    function GetForm: TCustomForm; virtual;
    procedure NextExecute(Sender: TObject); dynamic;
    procedure PrevExecute(Sender: TObject); dynamic;    
    property DocsDesigner: IDocsDesigner read FDocsDesigner
                                       write SetDocsDesigner
                                       stored False;
    property RTFDesigner: IRTFDesigner read FRTFDesigner
                                       write SetRTFDesigner
                                       stored False;
  published
    property Count: Integer read GetCount write SetCount stored False;
    property DocDisplay: TTreeView read FDocDisplay write SetDocDisplay;
    property DocList: TDocList read FDocs write SetDocs stored False;
    property ForceVisible: Boolean read FForceVisible
                                   write SetForceVisible default False;
    property RichEdit: TRichEdit read FRichEdit write SetRichEdit;
    property SelIndex: Integer read GetSelIndex write SetSelIndex stored False;
    property UpWhenBack: Boolean read FUpWhenBack write SetUpWhenBack default False;
    property OnChange: TDocEvent read FOnChange write SetOnChange;
    property OnGetData: TDocEvent read FOnGetData write SetOnGetData;
    property OnDeletion: TDocEvent read FOnDeletion write SetOnDeletion;
    property OnSelection: TDocEvent read FOnSelection write SetOnSelection;
    property OnUpdate: TNotifyEvent read FOnUpdate write SetOnUpdate;
  end;

  TDocsCompCont = class(TDocs, IUnknown, IOwnerObject)
  private
    FForm: TCustomForm;
  public
    constructor Create(AOwner: TComponent); override;
    function GetForm: TCustomForm; override;
  end;

  TDocAction = class(TAction)
  private
    FDocs: TDocs;
    procedure SetDocs(Value: TDocs);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public                              
    function Execute: Boolean; override;
    function HandlesTarget(Target: TObject): Boolean; override;
    function Update: Boolean; override;
    procedure UpdateTarget(Target: TObject); override;
  published         
    property Docs: TDocs read FDocs write SetDocs;
  end;

  TDocBack = class(TDocAction)
  public
    procedure ExecuteTarget(Target: TObject); override;
  end;       

  TDocDemo = class(TDocAction)
  public
    procedure ExecuteTarget(Target: TObject); override;
  end;

  TDocNext = class(TDocAction)
  public
    procedure ExecuteTarget(Target: TObject); override;
  end;     

  TDocPrev = class(TDocAction)
  public
    procedure ExecuteTarget(Target: TObject); override;
  end;

  procedure ExtractCmdPath(const CmdLine: string; var Cmd, Path: string);
  function ExtractPoint(APath: string): TPoint;

  procedure Register;

implementation

procedure Register;
begin 
  RegisterComponents('StreamSec', [TDocs]);    
  RegisterActions('Docs',[TDocBack,TDocNext,TDocDemo,TDocPrev],nil);
end;

{ TDoc }

procedure TDoc.Assign(Source: TPersistent);
begin
  if Source is TDoc then begin
    Title := (Source as TDoc).Title;
    Tag := (Source as TDoc).Tag;
    Text := (Source as TDoc).Text;
    MacroBack.Assign((Source as TDoc).MacroBack);
    MacroDemo.Assign((Source as TDoc).MacroDemo);
    MacroNext.Assign((Source as TDoc).MacroNext);
    MacroPrev.Assign((Source as TDoc).MacroPrev);
    Visible := (Source as TDoc).Visible;
    ImageIndex := (Source as TDoc).ImageIndex;
    StateIndex := (Source as TDoc).StateIndex;
  end else inherited
end;

procedure TDoc.Change;
begin
  DoChange;
end;

function TDoc.CheckDelete: Boolean;
begin
  if Assigned(Owner) then Result := Owner.DoDeleteItem(Self)
  else Result := True;
end;

constructor TDoc.Create(AOwner: TDocList);
begin
  inherited Create(AOwner);
  FStream := TRTFStream.Create('',Self);
  FMacroBack := TMacro.Create(Self);
  FMacroDemo := TMacro.Create(Self);
  FMacroNext := TMacro.Create(Self);
  FMacroPrev := TMacro.Create(Self);
  Visible := True;
  FImageIndex := -1;
  FStateIndex := -1;
end;

procedure TDoc.Delete;
begin
  if Deleted then Exit;
  { Calling DeleteChildren before CheckDelete will cause the Child with the
    highest level to be delete first. This is essential in particular if
    a TreeView is assigned as a DocDisplay to the TDocs component. }
  DeleteChildren;

  if CheckDelete then Free
  else
    { This will happen if the owner is updating. Setting Deleted to True will
      cause the owner to free this object when the owner ends updating.}
    Deleted := True;
end;

procedure TDoc.DeleteChildren;
var
  Doc: TDoc;
begin
  Doc := GetFirstChild;
  while Assigned(Doc) do begin
    Doc.Delete;
    Doc := GetFirstChild;
  end;
end;

destructor TDoc.Destroy;
begin
  FStream.Free;
  FMacroBack.Free;
  FMacroDemo.Free;
  FMacroNext.Free;
  FMacroPrev.Free;
end;

procedure TDoc.DoChange;
begin
  if Assigned(Owner) then Owner.DoChangeItem(Self);
end;

procedure TDoc.DoSelected;
begin
  if Assigned(Owner) then Owner.DoSelect(Self);
end;

function TDoc.GetAsStream: TStream;
begin
  FStream.Position := 0;
  Result := FStream;
end;

function TDoc.GetAsString: string;
begin
  Result := FStream.DataString;
end;

function TDoc.GetCaption: string;
var
  I: Integer;
begin
  I := Index;
  if I > -1 then begin
    Result := Owner.FItems[I];
    if Result <> '' then begin
      I := 1;
      while (I <= Length(Result)) and (Result[I] = #9) do Inc(I);
      if I > 1 then System.Delete(Result,1,I-1);
      I := Pos(#0,Result);
      if I > 0 then Result := Copy(Result,1,I-1);
    end;
  end else Result := '';
end;

function TDoc.GetFirstChild: TDoc;
var
  I: Integer;
begin
  if Assigned(Owner) then begin
    I := Index;
    Result := Owner.Items[I + 1];
    if Assigned(Result) and (Result.Level <= Level) then Result := nil;
  end else Result := nil;
end;

function TDoc.GetIndex: Integer;
begin
  Result := -1;
  if Assigned(Owner) then Result := Owner.FItems.IndexOfObject(Self);
end;

function TDoc.GetLevel: Integer;
var
  I: Integer;
  S: string;
begin
  I := Index;
  if I > -1 then begin
    S := Owner.FItems[I];
    if S <> '' then begin
      Result := 0;
      while (Result < Length(S)) and (S[Result + 1] = #9) do Inc(Result);
    end else Result := 0;
  end else Result := -1;
end;

function TDoc.GetNamePath: string;
begin
  if Assigned(Owner) then
    Result := Owner.GetNamePath + '[' + IntToStr(Index) + ']'
  else Result := 'Doc';
end;

function TDoc.GetNextBranch: TDoc;
var
  I, L: Integer;
begin
  if Assigned(Owner) then begin
    I := Index;
    L := Level;
    Result := nil;
    while I < Owner.FItems.Count-1 do begin
      Inc(I);
      Result := Owner.Items[I];
      if Assigned(Result) then
        if Result.Level <= L then Exit;
    end;
  end else Result := nil;
end;

function TDoc.GetNextChild(Item: TDoc): TDoc;
begin
  if Assigned(Item) and Assigned(Owner) and (Item.GetParent = Self) then
    Result := Item.GetNextSibling
  else Result := nil;
end;

function TDoc.GetNextSibling: TDoc;
var
  I, L: Integer;
begin
  if Assigned(Owner) then begin
    I := Index;
    L := Level;
    Result := nil;
    while I < Owner.FItems.Count-1 do begin
      Inc(I);
      Result := Owner.Items[I];
      if Assigned(Result) then begin
        if Result.Level <= L then begin
          if Result.Level < L then Result := nil;
          Exit;
        end;
      end;
    end;
  end else Result := nil;
end;

function TDoc.GetDocList: TDocList;
begin
  Result := GetOwner as TDocList;
end;

function TDoc.GetParent: TDoc;
var
  I: Integer;
begin
  if Level > 0 then begin
    I := Index;
    while I > 0 do begin
      Dec(I);
      Result := Owner.Items[I];
      if Result.Level < Level then Exit;
    end;
  end;
  Result := nil;
end;

function TDoc.GetPrevSibling: TDoc;
var
  I, L: Integer;
begin
  if Assigned(Owner) then begin
    I := Index;
    L := Level;
    Result := nil;
    while I > 0 do begin
      Dec(I);
      Result := Owner.Items[I];
      if Assigned(Result) then begin
        if Result.Level <= L then begin
          if Result.Level < L then Result := nil;
          Exit;
        end;
      end;
    end;
  end else Result := nil;
end;

function TDoc.GetSelected: Boolean;
begin
  Result := False;
  if Assigned(Owner) then Result := Owner.FSelected = Self;
end;

function TDoc.HasSubString(const Value: string): Boolean;
var
  I, J, Len: Integer;
  C: AnsiChar;
begin
  Result := False;
  if Length(Value) = 0 then Exit;
  Len := Length(Text);
  I := 1;
  J := 1;
  while I <= Len do begin
    C := Text[I];
    if C = '\' then begin
      Inc(I);
      if I <= Len then begin
        if Text[I] in ['{','}','\'] then C := Text[I]
        else if Text[I] = '''' then begin
          Inc(I);
          C := AnsiChar(StrToIntDef('$' + Copy(Text,I,2),0));
          Inc(I,2);
        end else begin
          C := #0;
          while (I <= Len) and not (Text[I] in [#13,#10,' ','\','{','}']) do
            Inc(I);
        end;
      end else C := #0;
    end else if C in ['{','}'] then C := #0;
    if C = #0 then J := 0
    else begin
      Result := AnsiUpperCase(C) = AnsiUpperCase(Value[J]);
      if Result then begin
        Inc(J);
        if J > Length(Value) then Exit;
      end else J := 1;
    end;
    Inc(I);
  end;
end;

procedure TDoc.SetAsStream(const Value: TStream);
begin
  Value.Position := 0;
  FStream.Size := 0;
  FStream.CopyFrom(Value,0); 
  DoChange;
end;

procedure TDoc.SetAsString(const Value: string);
begin
  FStream.Size := 0;
  FStream.WriteString(Value); 
  DoChange;
end;

procedure TDoc.SetCaption(const Value: string);
var
  I: Integer;
  S: string;
begin
  S := StringReplace(Value,#9,' ',[rfReplaceAll]);
  I := Index;
  if I > -1 then
    Owner.FItems[I] := StringOfChar(#9,Level) + S;
  DoChange;
end;

procedure TDoc.SetData(const Value: TObject);
begin
  FData := Value;
end;

procedure TDoc.SetDeleted(const Value: Boolean);
begin
  FDeleted := Value;
end;

procedure TDoc.SetImageIndex(const Value: Integer);
begin
  FImageIndex := Value;  
  DoChange;
end;

procedure TDoc.SetMacroBack(const Value: TCustomMacro);
begin
  FMacroBack.Assign(Value);
end;

procedure TDoc.SetMacroDemo(const Value: TCustomMacro);
begin
  FMacroDemo.Assign(Value);
end;

procedure TDoc.SetMacroNext(const Value: TCustomMacro);
begin
  FMacroNext.Assign(Value);
end;

procedure TDoc.SetMacroPrev(const Value: TCustomMacro);
begin
  FMacroPrev.Assign(Value);
end;

procedure TDoc.SetSelected(const Value: Boolean);
begin
  if (Value = GetSelected) or not Assigned(Owner) then Exit;
  if Value then begin
    if Assigned(Owner.FSelected) and
       ((Owner.FHistory.Count = 0) or
        (Owner.FHistory.Objects[Owner.FHistory.Count-1] <> Owner.Selected)) then
      Owner.FHistory.AddObject(Owner.FSelected.Title,Owner.Selected);
    Owner.FSelected := Self
  end else if Self = Owner.Selected then Owner.FSelected := nil;
  DoSelected;
end;

procedure TDoc.SetStateIndex(const Value: Integer);
begin
  FStateIndex := Value;  
  DoChange;
end;

procedure TDoc.SetTag(const Value: Integer);
begin
  FTag := Value;   
  DoChange;
end;

procedure TDoc.SetVisible(const Value: Boolean);
begin
  FVisible := Value;     
  DoChange;
end;

{ TDocList }

function TDocList.Add(Level: Integer): TDoc;
begin
  Result := InternalInsert(FItems.Count,Level);
  if not Assigned(Result) then
    raise EDocList.CreateFmt(ERR_DOCLIST_ADD,[Level]);
  if Level = 0 then         
    DoAddItem(nil,Result,adaAddLast)
  else
    DoAddItem(Result.GetParent,Result,adaAddLast);
end;

function TDocList.AddFirst: TDoc;
begin
  Result := InternalInsert(0,0);   
  DoAddItem(nil,Result,adaAddFirst);
end;

function TDocList.AddFirstChild(Item: TDoc): TDoc;
begin
  if not Assigned(Item) then Result := AddFirst
  else begin
    Result := InternalInsert(Item.Index + 1,Item.Level + 1);
    DoAddItem(Item,Result,adaAddFirst);
  end;
end;

function TDocList.AddFirstSibling(Item: TDoc): TDoc;
begin
  if not Assigned(Item) then Result := AddFirst
  else begin
    Item := Item.GetParent;
    if Assigned(Item) then Result := AddFirstChild(Item)
    else Result := AddFirst;
  end;
end;

function TDocList.AddLast: TDoc;
begin
  Result := Add(0);
end;

function TDocList.AddLastChild(Item: TDoc): TDoc;
var
  Next: TDoc;
begin
  if not Assigned(Item) then Result := AddLast
  else begin
    Next := Item.GetNextBranch;
    if Assigned(Next) then begin
      Result := InternalInsert(Next.Index,Item.Level + 1);
      DoAddItem(Item,Result,adaAddLast);
    end else Result := Add(Item.Level + 1);
  end;
end;

function TDocList.AddLastSibling(Item: TDoc): TDoc;
var
  Parent: TDoc;
begin
  if Assigned(Item) then begin
    Parent := Item.GetParent;
    if Assigned(Parent) then Result := AddLastChild(Parent)
    else Result := AddLast;
  end else Result := AddLast;
end;

procedure TDocList.AssignTo(Dest: TPersistent);
var
  D: TDocList;
  I: Integer;
  Doc: TDoc;
begin
  if Dest is TDocList then begin
    D := Dest as TDocList;
    D.Clear;
    I := 0;
    while I < FItems.Count do begin
      Doc := D.Add(Items[I].Level);
      Doc.Assign(Items[I]);
      Inc(I);
    end;
    if Assigned(FSelected) then
      D.FSelected := D.Items[Selected.Index]
    else
      D.FSelected := nil;
  end else inherited AssignTo(Dest);
end;

procedure TDocList.Back;
var
  Doc: TDoc;
begin
  if FHistory.Count = 0 then Exit;
  Doc := FHistory.Objects[FHistory.Count-1] as TDoc; 
  FHistory.Delete(FHistory.Count-1);
  FSelected := nil;
  Doc.Selected := True;
end;

procedure TDocList.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

procedure TDocList.Clear;
begin
  while Count > 0 do Items[0].Delete;
  FSelected := nil;
  FHistory.Clear;
  FItems.Clear;
end;

constructor TDocList.Create(const AOwner: IOwnerObject);
begin
  inherited Create(AOwner.AsPersistent);
  FLines := TStringList.Create;    
  (FLines as TStringList).OnChange := LinesChange;
  FItems := TStringList.Create;
  FHistory := TStringList.Create;
  SetOwnerName('');
end;

procedure TDocList.DefineProperties(Filer: TFiler);
begin

end;

destructor TDocList.Destroy;
begin
  Clear;
  FItems.Free;
  FHistory.Free;
  if Assigned(FItemsChanged) then FItemsChanged.Free;
  if Assigned(FItemsDeleted) then FItemsDeleted.Free;
  if Assigned(FLines) then FLines.Free;
end;

procedure TDocList.DoAddItem(RefItem, NewItem: TDoc;
  Action: TAddDocAction);
begin
  if GetDocsComponent <> nil then GetDocsComponent.DoAddDoc(RefItem,NewItem,Action);
end;

procedure TDocList.DoChangeItem(Item: TDoc);
begin
  if FUpdateCount > 0 then begin
    if not Assigned(FItemsChanged) then FItemsChanged := TList.Create;
    if FItemsChanged.IndexOf(Item) < 0 then FItemsChanged.Add(Item);
  end else
    if GetDocsComponent <> nil then GetDocsComponent.DoChangeDoc(Item);
end;

function TDocList.DoDeleteItem(Item: TDoc): Boolean;
var
  I: Integer;
begin
  if FUpdateCount > 0 then begin
    if not Assigned(FItemsDeleted) then FItemsDeleted := TList.Create;
    FItemsDeleted.Add(Item);
    Result := False;
  end else begin
    if GetDocsComponent <> nil then GetDocsComponent.DoDeleteDoc(Item);
    I := FItems.IndexOfObject(Item);
    if I > -1 then FItems.Delete(I);
    repeat
      I := FHistory.IndexOfObject(Item);
      if I > -1 then FHistory.Delete(I);
    until I < 0;
    if Assigned(Item) and Item.Deleted then Item.Free;
    Result := True;
  end;
end;

procedure TDocList.DoSelect(Item: TDoc);
begin
  if GetDocsComponent <> nil then GetDocsComponent.DoSelectDoc(Item);
end;

procedure TDocList.DoUpdate;
begin
  if GetDocsComponent <> nil then GetDocsComponent.DoUpdate;
end;

procedure TDocList.Down;
var
  Doc: TDoc;
begin
  if Assigned(FSelected) then begin
    Doc := FSelected.GetFirstChild;
    if Assigned(Doc) then Doc.Selected := True;
  end;
end;

procedure TDocList.EndUpdate;
begin
  Dec(FUpdateCount);
  if FUpdateCount <= 0 then InternalEndUpdate;
end;

function TDocList.GetCount: Integer;
begin
  if not Assigned(FItems) then Result := 0
  else Result := FItems.Count;
end;

function TDocList.GetDocsComponent: TDocs;
begin
  if (GetOwner <> nil) and (GetOwner is TDocs) then
    Result := GetOwner as TDocs
  else Result := nil;
end;

function TDocList.GetForm: TCustomForm;
begin
  if GetDocsComponent <> nil then Result := GetDocsComponent.GetForm
  else Result := inherited GetForm;
end;

function TDocList.GetItems(index: Integer): TDoc;
begin
  Result := nil;
  if not Assigned(FItems) then Exit;
  if (Index < 0) or (Index >= FItems.Count) then Exit;
  if not (FItems.Objects[Index] is TDoc) then Exit;
  Result := FItems.Objects[Index] as TDoc;
end;

function TDocList.GetLines: TStrings;
var
  SS: TStringStream;
begin
  SS := TStringStream.Create('');
  try
    SaveToStream(SS);
    SS.Position := 0;
    (FLines as TStringList).OnChange := nil;
    FLines.LoadFromStream(SS);     
    (FLines as TStringList).OnChange := LinesChange;
    Result := FLines;
  finally
    SS.Free;
  end;
end;

function TDocList.GetNamePath: string;
begin
  if Owner <> nil then Result := Owner.Name + '.DocList'
  else Result := 'DocList';
end;

function TDocList.GetUpdating: Boolean;
begin
  Result := FUpdateCount > 0;
end;

procedure TDocList.InternalEndUpdate;
var
  I: Integer;
  Doc: TDoc;
begin
  FUpdateCount := 0;
  DoUpdate;  
  if Assigned(FItemsDeleted) then begin
    { The order of deletion is of essence. The items must be deleted
      in the same order as the notification arrived. }
    I := 0;
    while I < FItemsDeleted.Count do begin
      Doc := FItemsDeleted[I];
      DoDeleteItem(Doc);
      Inc(I);
    end;
    FItemsDeleted.Free;
    FItemsDeleted := nil;
  end;
  if Assigned(FItemsChanged) then begin
    I := 0;
    while I < FItemsChanged.Count do begin
      Doc := FItemsChanged[I];
      // Items might have been deleted. Check first:
      if FItems.IndexOfObject(Doc) > -1 then DoChangeItem(Doc);
      Inc(I);
    end; 
    FItemsChanged.Free;
    FItemsChanged := nil;
  end;
end;

function TDocList.InternalInsert(Index, Level: Integer): TDoc;
var
  Allowed: Boolean;
begin
  Allowed := (Index >= 0) and (Index <= FItems.Count);
  if Allowed then begin
    if Index = 0 then Level := 0
    else if Items[Index-1].Level < Level - 1 then
      Level := Items[Index-1].Level + 1;
  end;
  if Allowed then begin
    Result := TDoc.Create(Self);
    FItems.InsertObject(Index,StringOfChar(#9,Level),Result);
  end else Result := nil;
end;

procedure TDocList.LinesChange(Sender: TObject);
begin
  SetLines(FLines);
end;

procedure TDocList.LoadFromFile(const FileName: string);
var
  FS: TFileStream;
begin
  FS := TFileStream.Create(FileName,fmOpenRead);
  try
    LoadFromStream(FS);
  finally
    FS.Free;
  end;
end;

procedure TDocList.LoadFromStream(AStream: TStream);
var
  Docs: TDocsCompCont;
  tmpStrm: TMemoryStream;
begin
  try
    Docs := TDocsCompCont.Create(GetForm);
  except
    raise
      EDocList.Create('Could not create an internal TDocs component.');
  end;
  try
    tmpStrm := TMemoryStream.Create;
    try
      ObjectTextToResource(AStream,tmpStrm);
      tmpStrm.Position := 0;
      try
        tmpStrm.ReadComponentRes(Docs);
      except     
        raise
          EDocList.Create('Could not read internal TDocs component from stream.');
      end;
      Assign(Docs.DocList);
    finally
      tmpStrm.Free;
    end;
  finally
    Docs.Free;
  end;
end;

procedure TDocList.MakeIndex(const SearchStr: string; List: TStrings);
var
  I: Integer;
  Doc: TDoc;
begin
  List.Clear;
  I := 0;
  while I < FItems.Count do begin
    Doc := Items[I];
    if Doc.HasSubString(SearchStr) then
      List.AddObject(Doc.Title,Doc);
    Inc(I);
  end;
end;

procedure TDocList.Next;
var
  Doc: TDoc;
begin
  if Assigned(FSelected) then begin
    Doc := FSelected.GetNextSibling;
    if Assigned(Doc) then Doc.Selected := True;
  end else if Count > 0 then
    Items[0].Selected := True;
end;

procedure TDocList.Prev;
var
  Doc: TDoc;
begin
  if Assigned(FSelected) then begin
    Doc := FSelected.GetPrevSibling;
    if Assigned(Doc) then Doc.Selected := True;
  end else if Count > 0 then
    Items[Count-1].Selected := True;
end;

procedure TDocList.ReadList(Reader: TReader);
var
  Col: TDocCollection;
  ColItem: TDocCollectionItem;
  Doc: TDoc;
  I: Integer;
begin
  if Reader.ReadValue <> vaCollection then Exit;
  Clear;
  Col := TDocCollection.Create;
  try
    try
      Reader.ReadCollection(Col);
      I := 0;
      while I < Col.Count do begin
        ColItem := Col.Items[I];
        Doc := Add(ColItem.Level);
        ColItem.AssignTo(Doc);
        Inc(I);
      end;
    except
      on E: Exception do
        ShowMessage(E.Message);
    end;
  finally
    Col.Free;
  end;
end;

procedure TDocList.SaveToFile(const FileName: string);
var
  FS: TFileStream;
begin
  if FileExists(FileName) then
    FS := TFileStream.Create(FileName,fmOpenWrite or fmShareExclusive)
  else
    FS := TFileStream.Create(FileName,fmCreate);
  try
    SaveToStream(FS);
  finally
    FS.Free;
  end;
end;

procedure TDocList.SaveToStream(AStream: TStream);
var
  vName: string;
  Docs: TDocsCompCont;
  tmpStrm: TMemoryStream;
begin
  if GetDocsComponent <> nil then vName := GetDocsComponent.Name + '_Res'
  else vName := 'Docs';
  Docs := TDocsCompCont.Create(GetForm);
  try
    tmpStrm := TMemoryStream.Create;
    try
      AssignTo(Docs.DocList);
      tmpStrm.WriteComponentRes(vName,Docs);
      tmpStrm.Position := 0;
      ObjectResourceToText(tmpStrm,AStream);
    finally
      tmpStrm.Free;
    end;
  finally
    Docs.Free;
  end;
end;

procedure TDocList.SetItems(index: Integer; const Value: TDoc);
begin
  if (Index < 0) or (Index >= FItems.Count) then Exit;
  Items[Index].Assign(Value);
end;

procedure TDocList.SetLines(const Value: TStrings);
var
  SS: TStringStream;
begin
  SS := TStringStream.Create('');
  try
    if Assigned(Value) then Value.SaveToStream(SS);
    SS.Position := 0;
    LoadFromStream(SS);
  finally
    SS.Free;
  end;
end;

procedure TDocList.SetOwnerName(const Value: string);
begin
  if (Value = '') and (GetDocsComponent <> nil) then
    FOwnerName := GetDocsComponent.Name
  else
    FOwnerName := Value;
end;

procedure TDocList.Up;
var
  Doc: TDoc;
begin
  if Assigned(FSelected) then begin
    Doc := FSelected.GetParent;
    if Assigned(Doc) then Doc.Selected := True;
  end;
end;

procedure TDocList.UpNext;
var
  Doc: TDoc;
begin
  if Assigned(FSelected) then begin
    Doc := FSelected.GetParent;
    if Assigned(Doc) then begin
      Doc := Doc.GetNextSibling;
      if Assigned(Doc) then Doc.Selected := True;
    end
  end;
end;

procedure TDocList.WriteList(Writer: TWriter);
var
  Col: TDocCollection;
  ColItem: TDocCollectionItem;
  I: Integer;
begin
  try
    Col := TDocCollection.Create;
    try
      I := 0;
      while I < Count do begin
        ColItem := Col.Add;
        ColItem.Assign(Items[I]);
        Inc(I);
      end;
      Writer.WriteCollection(Col);
    finally
      Col.Free;
    end;
  except
    showmessage('Could not write DocList resource.');
  end;
end;

{ TDocs }

function TDocs.AsPersistent: TPersistent;
begin
  Result := Self;
end;

procedure TDocs.BackExecute(Sender: TObject);
begin
  if Assigned(FDocs.Selected) and
     (FDocs.Selected.MacroBack.Count > 0)  then
    FDocs.Selected.MacroBack.ExecuteMacro
  else if FUpWhenBack then
    FDocs.Up
  else if (FDocs.FHistory.Count > 0) then
    FDocs.Back;
end;

constructor TDocs.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);  
  FDocs := TDocList.Create(Self);
end;

procedure TDocs.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('List',ReadList,WriteList,True);
  Filer.DefineProperty('Selected',ReadSel,WriteSel,SelIndex <> -1);
end;

procedure TDocs.DemoExecute(Sender: TObject);
begin
  if Assigned(FDocs.Selected) then begin
    if FDocs.Selected.MacroDemo.Count > 0 then
      FDocs.Selected.MacroDemo.ExecuteMacro
    else
      FDocs.Down;
  end;
end;

destructor TDocs.Destroy;
begin  
  RTFDesigner := nil;
  DocsDesigner := nil;
  FDocs.Free;
  FDocs := nil;
  inherited Destroy;
end;

procedure TDocs.DoAddDoc(RefItem, NewItem: TDoc; Action: TAddDocAction);
var
  OldNode: TTreeNode;
begin
  if Assigned(FDocDisplay) and (ForceVisible or NewItem.Visible) then begin
    if Assigned(RefItem) then OldNode := RefItem.Data as TTreeNode
    else OldNode := nil;
    with FDocDisplay.Items do
      case Action of
        adaAddFirst: NewItem.Data :=
                       AddChildObjectFirst(OldNode,NewItem.Title,NewItem);
        adaAddLast:  NewItem.Data :=
                       AddChildObject(OldNode,NewItem.Title,NewItem);
      end;
    FDocDisplay.Refresh;
  end;
  if Assigned(FRTFDesigner) then
    FRTFDesigner.AddDoc(NewItem);
  if not (csDesigning in ComponentState) then
    if Assigned(FOnGetData) then FOnGetData(Self,NewItem);
end;

procedure TDocs.DoChangeDoc(Item: TDoc);
var
  TN: TTreeNode;
begin
  if Assigned(Item) then begin
    if Item.Selected and Assigned(FRichEdit) and
       not (csDesigning in ComponentState) then begin
      Item.TextAsStream.Position := 0;
      FRichEdit.Lines.LoadFromStream(Item.TextAsStream);
    end;
    if Assigned(Item.Data) and (Item.Data is TTreeNode) then begin
      TN := Item.Data as TTreeNode;
      if not (ForceVisible or Item.Visible) then begin
        TN.Delete;
        Item.Data := nil;
      end else begin
        TN.Text := Item.Title;
        TN.ImageIndex := Item.ImageIndex;
        TN.StateIndex := Item.StateIndex;
      end;
    end;
  end;
  if Assigned(FRTFDesigner) then FRTFDesigner.ChangeDoc(Item);
  if not (csDesigning in ComponentState) then
    if Assigned(FOnChange) then FOnChange(Self,Item);
end;

procedure TDocs.DoDeleteDoc(Item: TDoc);
var
  TN: TTreeNode;
begin
  if Assigned(Item) and Assigned(Item.Data) and (Item.Data is TTreeNode) then begin;
    TN := Item.Data as TTreeNode;
    TN.Data := nil;
    TN.Delete;
    Item.Data := nil;
  end;
  if Assigned(FRTFDesigner) then
    FRTFDesigner.RemoveDoc(Item);
  if Assigned(FOnDeletion) then FOnDeletion(Self,Item);
end;

procedure TDocs.DoSelectDoc(Item: TDoc);
var
  TN: TTreeNode;
begin
  if Assigned(Item) and Item.Selected then begin
    if Assigned(FRichEdit) and not (csDesigning in ComponentState) then begin
      Item.TextAsStream.Position := 0;
      FRichEdit.Lines.LoadFromStream(Item.TextAsStream);
    end;
    if Assigned(Item.Data) and (Item.Data is TTreeNode) then begin
      TN := Item.Data as TTreeNode;
      if not TN.Selected then TN.Selected := True;
    end;
    if Assigned(FRTFDesigner) then
      FRTFDesigner.SelectDoc(Item);
  end;
  if Assigned(FOnSelection) then FOnSelection(Self,Item);
end;

procedure TDocs.DoUpdate;
begin
  if Assigned(FOnUpdate) then FOnUpdate(Self);
end;

procedure ExtractCmdPath(const CmdLine: string; var Cmd, Path: string);
var
  S: string;
  P: Integer;
begin
  S := Trim(CmdLine);
  if S = '' then begin
    Cmd := '';
    Path := '';
  end else begin
    P := Pos(' ',S);
    if P = 0 then begin
      Cmd := '';
      Path := S;
    end else begin
      Cmd := Copy(S,1,P-1);
      Path := Copy(S,P+1,MaxInt);
    end;
  end;
end;             

function ExtractPoint(APath: string): TPoint;
var
  P: Integer;
begin
  Result := Mouse.CursorPos;
  if APath = '' then Exit;
  if (APath[1] <> '(') or (APath[Length(APath)] <> ')') then Exit;
  APath := Copy(APath,2,Length(APath)-2);
  P := Pos(',',APath);
  Result.x := StrToIntDef(Copy(APath,1,P),Mouse.CursorPos.x);
  Delete(APath,1,P);
  Result.y := StrToIntDef(Copy(APath,1,P),Mouse.CursorPos.x);
end;

function TDocs.GetCount: Integer;
begin
  if not Assigned(FDocs) then Result := 0
  else Result := FDocs.Count;
end;

function TDocs.GetForm: TCustomForm;
begin
  if (Owner = nil) or not (Owner is TCustomForm) then Result := nil
  else Result := Owner as TCustomForm;
end;

function TDocs.GetSelIndex: Integer;
begin
  if Assigned(FDocs.FSelected) then Result := FDocs.FSelected.Index
  else Result := -1;
end;

procedure TDocs.NextExecute(Sender: TObject);
begin              
  if Assigned(FDocs.Selected) then begin
    if FDocs.Selected.MacroNext.Count > 0 then
      FDocs.Selected.MacroNext.ExecuteMacro
    else
      FDocs.Next;
  end;
end;

procedure TDocs.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (AComponent = FDocDisplay) and (Operation = opRemove) then
    SetDocDisplay(nil)
  else if (AComponent = FRichEdit) and (Operation = opRemove) then
    SetRichEdit(nil);
  inherited Notification(AComponent,Operation);
end;

function TDocs.OwnerInterface: IOwnerObject;
begin
  Owner.ComObject.QueryInterface(IID_OwnerObject,Result);
end;

procedure TDocs.PrevExecute(Sender: TObject);
begin
  if Assigned(FDocs.Selected) then begin
    if FDocs.Selected.MacroPrev.Count > 0 then
      FDocs.Selected.MacroPrev.ExecuteMacro
    else
      FDocs.Prev;
  end;
end;

procedure TDocs.ReadList(Reader: TReader);
begin
  if Assigned(FDocs) then FDocs.ReadList(Reader)
  else
    raise EDocList.Create('No doclist.');
end;

procedure TDocs.ReadSel(Reader: TReader);
begin
  SelIndex := Reader.ReadInteger;
end;

procedure TDocs.SetCount(const Value: Integer);
begin
  // Do nothing. We just want the current value to show in the designer.
end;

procedure TDocs.SetDocDisplay(const Value: TTreeView);
var
  Doc: TDoc;
  I: Integer;

  procedure Add(ParentNode: TTreeNode; ADoc: TDoc);
  var
    Child: TDoc;
    Node: TTreeNode;
  begin
    if not (ForceVisible or ADoc.Visible) then Exit;
    Node := Value.Items.AddChildObject(ParentNode,ADoc.Title,ADoc);
    Node.ImageIndex := ADoc.ImageIndex;
    Node.StateIndex := ADoc.StateIndex;
    ADoc.Data := Node;
    Child := ADoc.GetFirstChild;
    while Assigned(Child) do begin
      Add(Node,Child);
      Child := ADoc.GetNextChild(Child);
    end;
  end;

begin
  if FDocDisplay = Value then Exit;
  if Assigned(Value) then begin  
    Value.FreeNotification(Self);
    Value.Items.Clear;
    if Count > 0 then begin
      Value.Items.BeginUpdate;
      try
        Doc := FDocs[0];
        repeat
          Add(nil,Doc);
          Doc := Doc.GetNextSibling;
        until not Assigned(Doc);
      finally
        Value.Items.EndUpdate;
      end;
    end;
  end else if Assigned(FDocDisplay) then begin
    I := 0;
    while I < Count do begin
      DocList.Items[I].Data := nil;
      Inc(I);
    end;
  end;
  FDocDisplay := Value;
end;

procedure TDocs.SetDocs(const Value: TDocList);
begin
  if Assigned(Value) then Value.AssignTo(FDocs)
  else FDocs.Clear;
end;

procedure TDocs.SetDocsDesigner(const Value: IDocsDesigner);
begin
  if Value = FDocsDesigner then Exit;
  if Assigned(FDocsDesigner) then
    FDocsDesigner.SetDocsComponent(nil);
  FDocsDesigner := Value;
  if Assigned(Value) then Value.SetDocsComponent(Self);
end;

procedure TDocs.SetForceVisible(const Value: Boolean);
begin
  FForceVisible := Value;
end;

procedure TDocs.SetOnChange(const Value: TDocEvent);
begin
  FOnChange := Value;
end;

procedure TDocs.SetOnDeletion(const Value: TDocEvent);
begin
  FOnDeletion := Value;
end;

procedure TDocs.SetOnGetData(const Value: TDocEvent);
begin
  FOnGetData := Value;
end;

procedure TDocs.SetOnSelection(const Value: TDocEvent);
begin
  FOnSelection := Value;
end;

procedure TDocs.SetOnUpdate(const Value: TNotifyEvent);
begin
  FOnUpdate := Value;
end;

procedure TDocs.SetRichEdit(const Value: TRichEdit);
begin
  if Value = FRichEdit then Exit;
  if Assigned(Value) then begin
    Value.Clear;
    Value.FreeNotification(Self);
    if Assigned(FDocs.Selected) then
      Value.Lines.LoadFromStream(FDocs.Selected.TextAsStream);
  end;
  FRichEdit := Value;
end;

procedure TDocs.SetRTFDesigner(const Value: IRTFDesigner);
begin
  if Value = FRTFDesigner then Exit;
  if Assigned(FRTFDesigner) then
    FRTFDesigner.SetDocsComponent(nil);
  FRTFDesigner := Value;
  if Assigned(Value) then Value.SetDocsComponent(Self);
end;

procedure TDocs.SetSelIndex(const Value: Integer);
var
  Doc: TDoc;
begin
  if Value = GetSelIndex then Exit;
  Doc := FDocs.GetItems(Value);
  if Assigned(Doc) then Doc.Selected := True
  else if Assigned(FDocs.FSelected) then FDocs.FSelected.Selected := False;
end;

procedure TDocs.SetUpWhenBack(const Value: Boolean);
begin
  FUpWhenBack := Value;
end;

procedure TDocs.WriteList(Writer: TWriter);
begin
  if Assigned(FDocs) then
    FDocs.WriteList(Writer);
end;

procedure TDocs.WriteSel(Writer: TWriter);
begin
  Writer.WriteInteger(SelIndex);
end;

{ TDocCollectionItem }

procedure TDocCollectionItem.Assign(Source: TPersistent);
begin
  if Source is TDoc then begin
    Title := (Source as TDoc).Title;
    Tag := (Source as TDoc).Tag;
    Level := (Source as TDoc).Level;
    RTF := (Source as TDoc).Text;
    MacroBack.Assign((Source as TDoc).MacroBack);
    MacroDemo.Assign((Source as TDoc).MacroDemo);
    MacroNext.Assign((Source as TDoc).MacroNext);
    MacroPrev.Assign((Source as TDoc).MacroPrev);
    Visible := (Source as TDoc).Visible;
    ImageIndex := (Source as TDoc).ImageIndex;
    StateIndex := (Source as TDoc).StateIndex;
  end else inherited Assign(Source);
end;

procedure TDocCollectionItem.AssignTo(Dest: TPersistent);
begin
  if Dest is TDoc then begin
    (Dest as TDoc).Title := Title;
    (Dest as TDoc).Tag := Tag;
    (Dest as TDoc).Text := RTF;
    (Dest as TDoc).MacroBack.Assign(MacroBack);
    (Dest as TDoc).MacroDemo.Assign(MacroDemo);
    (Dest as TDoc).MacroNext.Assign(MacroNext);
    (Dest as TDoc).MacroPrev.Assign(MacroPrev);
    (Dest as TDoc).Visible := Visible;
    (Dest as TDoc).ImageIndex := ImageIndex;
    (Dest as TDoc).StateIndex := StateIndex;
  end else inherited AssignTo(Dest);
end;

function TDocCollectionItem.GetDisplayName: string;
begin
  Result := FTitle;
  if Result = '' then Result := inherited GetDisplayName;
end;

procedure TDocCollectionItem.SetTitle(const Value: string);
begin
  FTitle := Value;
end;

procedure TDocCollectionItem.SetLevel(const Value: Integer);
begin
  FLevel := Value;
end;

procedure TDocCollectionItem.SetTag(const Value: Integer);
begin
  FTag := Value;
end;

procedure TDocCollectionItem.SetRTF(const Value: string);
begin
  FRTF := Value;
end;

function TDocCollectionItem.GetOwner: TPersistent;
begin
  Result := nil;
end;

function TDocCollectionItem.GetNamePath: string;
begin
  Result := ClassName;
end;

procedure TDocCollectionItem.SetVisible(const Value: Boolean);
begin
  FVisible := Value;
end;

constructor TDocCollectionItem.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FMacroBack := TFilerMacro.Create;
  FMacroDemo := TFilerMacro.Create;
  FMacroNext := TFilerMacro.Create;
  FMacroPrev := TFilerMacro.Create;
  FImageIndex := -1;
  FStateIndex := -1;
  FVisible := True;
end;

destructor TDocCollectionItem.Destroy;
begin
  FMacroBack.Free;
  FMacroDemo.Free;
  FMacroNext.Free;
  FMacroPrev.Free;
  inherited
end;

procedure TDocCollectionItem.SetStateIndex(const Value: Integer);
begin
  FStateIndex := Value;
end;

procedure TDocCollectionItem.SetImageIndex(const Value: Integer);
begin
  FImageIndex := Value;
end;

{ TDocCollection }

function TDocCollection.Add: TDocCollectionItem;
begin
  Result := TDocCollectionItem(inherited Add) 
end;

constructor TDocCollection.Create;
begin
  inherited Create(TDocCollectionItem);
end;

function TDocCollection.GetItem(index: Integer): TDocCollectionItem;
begin
  Result := TDocCollectionItem(inherited GetItem(index));
end;

procedure TDocCollection.SetItem(index: Integer;
  const Value: TDocCollectionItem);
begin
  inherited SetItem(index,Value);
end;

{ TDocAction }

function TDocAction.Execute: Boolean;
begin
  Result := inherited Execute;
  if Enabled and Assigned(FDocs) and not Result then begin
    Result := True;
    ExecuteTarget(FDocs);
  end;
end;

function TDocAction.HandlesTarget(Target: TObject): Boolean;
begin
  Result := ((Docs <> nil) and (Target = Docs)) or
    ((Docs = nil) and (Target is TDocs));
end;

procedure TDocAction.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = Docs) then Docs := nil;
end;

procedure TDocAction.SetDocs(Value: TDocs);
begin
  if Value <> FDocs then
  begin
    FDocs := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TDocAction.Update: Boolean;

  function TryUpdate: Boolean;
  begin
    if Assigned(OnUpdate) then begin
      OnUpdate(Self);
      Result := True;
    end else Result := False;
  end;

begin
  Result := (ActionList <> nil) and ActionList.UpdateAction(Self) or
    Application.UpdateAction(Self) or TryUpdate;
  if (not Result) and Assigned(FDocs) then
    Result := FDocs.UpdateAction(Self);
end;

procedure TDocAction.UpdateTarget(Target: TObject);
var
  Docs: TDocs;
begin
  if (Target = nil) or not (Target is TDocs) then Exit;
  Docs := Target as TDocs;
  if Self is TDocBack then
    Enabled := (Assigned(Docs.FDocs.Selected) and
                ((Docs.FDocs.Selected.MacroBack.Count > 0) or
                 (Docs.UpWhenBack and (Docs.FDocs.Selected.GetParent <> nil)))) or
               (Docs.FDocs.History.Count > 0)
  else if Self is TDocDemo then  
    Enabled := Assigned(Docs.FDocs.Selected) and
               ((Docs.FDocs.Selected.MacroDemo.Count > 0) or
                (Docs.FDocs.Selected.GetFirstChild <> nil))   
  else if Self is TDocNext then
    Enabled := Assigned(Docs.FDocs.Selected) and
               ((Docs.FDocs.Selected.MacroNext.Count > 0) or
                (Docs.FDocs.Selected.GetNextSibling <> nil))         
  else if Self is TDocPrev then
    Enabled := Assigned(Docs.FDocs.Selected) and
               ((Docs.FDocs.Selected.MacroPrev.Count > 0) or
                (Docs.FDocs.Selected.GetPrevSibling <> nil))

end;

{ TDocBack }

procedure TDocBack.ExecuteTarget(Target: TObject);
begin
  (Target as TDocs).BackExecute(Self);
end;

{ TDocDemo }

procedure TDocDemo.ExecuteTarget(Target: TObject);
begin
  (Target as TDocs).DemoExecute(Self);
end;

{ TDocNext }

procedure TDocNext.ExecuteTarget(Target: TObject);
begin
  (Target as TDocs).NextExecute(Self);
end;

{ TDocPrev }

procedure TDocPrev.ExecuteTarget(Target: TObject);
begin
  (Target as TDocs).PrevExecute(Self);
end;

{ TRTFStream }

constructor TRTFStream.Create(const AValue: string; AOwner: TDoc);
begin
  inherited Create(AValue);
  FOwner := AOwner;
end;

function TRTFStream.GetTitle: string;
begin
  if Assigned(FOwner) then Result := FOwner.Title
  else Result := '';
end;

procedure TRTFStream.SetOwner(const Value: TDoc);
begin
  FOwner := Value;
end;

{ TDocsCompCont }

constructor TDocsCompCont.Create(AOwner: TComponent);
begin
  FForm := AOwner as TCustomForm;
  inherited Create(nil);
end;

function TDocsCompCont.GetForm: TCustomForm;
begin
  Result := FForm;
end;

end.
