unit Spellers;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Langs, ComCtrls, RichEdit, StdCtrls, Math;

type
  TSpellOption = (spoSuggestFromUserDict, spoIgnoreAllCaps, spoIgnoreMixedDigits,
                  spoIgnoreRomanNumerals, spoFindUncappedSentences,
                  spoFindMissingSpaces, spoFindRepeatWord, spoFindExtraSpaces,
                  spoFindSpacesBeforePunc, spoFindSpacesAfterPunc, spoRateSuggestions,
                  spoFindInitialNumerals);

  TSpellOptions = set of TSpellOption;

  TSpellCommand = (scVerifyWord, scVerifyBuffer, scSuggest, scSuggestMore,
                   scHyphInfo, scWildCard, scAnagram);

  TSpellReturnCode = (srNoErrors, srUnknownInputWord, srReturningChangeAlways,
                      srReturningChangeOnce, srInvalidHyphenation,
                      srErrorCapitalization, srWordConsideredAbbreviation,
                      srHyphChangesSpelling, srNoMoreSuggestions,
                      srMoreInfoThanBufferCouldHold, srNoSentenceStartCap,
                      srRepeatWord, srExtraSpaces, srMissingSpace,
                      srInitialNumeral);

  TMisspellFontOption = (mfoName, mfoSize, mfoHeight, mfoColor, mfoPitch,
                         mfoStyle);

  TMisspellFontOptions = set of TMisspellFontOption;

  TMisspellEvent = procedure (Sender: TObject; SRC: TSpellReturnCode;
                              BufPos, Len: Integer) of object;
  TChangeTextEvent = procedure(Sender: TObject; BufPos, Len: Integer;
                               NewWord: String) of object;

  TGetDictEvent = procedure(Sender: TObject; Language: TLanguage;
                            var Dict: TFileName) of object;

  TSpellerDialog2 = class;
  TAbstractSpeller = class;

  TSpellChecker = class(TComponent)
  private
    { Private declarations }
    FMemo,
    FBackMemo: TCustomMemo;
    FLanguage: TLanguage;
    FOptions: TSpellOptions;
    FDialog: TSpellerDialog2;
    FModalDialog: Boolean;
    FHTML: Boolean;
    FCustomDict: TFileName;
    FCaption: TCaption;
    Spellers: TList;
    CRPos,
    TagPos,
    LangPos: Integer;
    FStartSentence: Boolean;
    FSpellStart,
    FSpellEnd: Integer;
    FSpeller: TAbstractSpeller;
    FMisspellStart,
    FMisspellLen: Integer;
    FMisspellText: String;
    FLangSupport: Boolean;
    FUnicode: Boolean;
    FSRC: TSpellReturnCode;
    FFinishMessage: String;
    FShowFinishMessage: Boolean;
    FOnMisspell: TMisspellEvent;
    FOnChangeText: TChangeTextEvent;
    FOnGetDict: TGetDictEvent;
  protected
    { Protected declarations }
    procedure SetLanguage(Value: TLanguage);
    procedure ChangeOnce(Word: String);
    procedure Change(Word: String);
    procedure ChangeAlways(Word: String);
    procedure Delete;
    procedure Add;
    procedure IgnoreAlways;
    function OpenLanguage(Value: TLanguage): Boolean;
    function FindLanguage(Value: TLanguage): TAbstractSpeller;
    procedure Init;
    procedure GetBlock(From: Integer; var StartPos, EndPos: Integer);
    function GetMemoLanguage: TLanguage;
    procedure GetTag(From: Integer; var Len: Integer);
    procedure GetTextRange(Buf: PChar; StartPos, EndPos: Integer; CP: Word);
    function SentenceCapitalize(const S: String): String;
    procedure ContinueCheck;
    procedure FinishCheck;
    function GetLineFromPos(Pos: Integer; var LineStart: Integer): String;
    procedure GetMemoProperties;
    function GetCurrentLanguage: TLanguage;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    class procedure RegisterEditControl(MemoClass: String; Unicode, Multilanguage: Boolean);
    procedure Check(Memo: TCustomMemo);
    function IsKnownWord(Word: String; Language: TLanguage): Boolean;
    procedure AddWord(Word: String; Language: TLanguage);
    procedure GetVariants(Word: String; Variants: TStrings; Language: TLanguage);
    property CurrentLanguage: TLanguage read GetCurrentLanguage;
  published
    { Published declarations }
    property Language: TLanguage read FLanguage write SetLanguage;
    property Options: TSpellOptions read FOptions write FOptions;
    property OnMisspelling: TMisspellEvent read FOnMisspell write FOnMisspell;
    property OnChangeText: TChangeTextEvent read FOnChangeText write FOnChangeText;
    property OnGetDictionary: TGetDictEvent read FOnGetDict write FOnGetDict;
    property Caption: TCaption read FCaption write FCaption;
//    property MisspellFont: TMisspellFont read FFont write FFont;
    property ModalDialog: Boolean read FModalDialog write FModalDialog;
    property HTMLSupport: Boolean read FHTML write FHTML default False;
    property CustomDict: TFileName read FCustomDict write FCustomDict;
    property FinishMessage: String read FFinishMessage write FFinishMessage;
    property ShowFinishMessage: Boolean read FShowFinishMessage write FShowFinishMessage;
  end;

  TAbstractSpeller = class(TObject)
    FLanguage: TLanguage;
    FOptions: TSpellOptions;
    SpellChecker: TSpellChecker;
    FNotActive: Boolean;
    constructor Create(Language: TLanguage; Owner: TSpellChecker; Options: TSpellOptions); virtual;
    function FindMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode; virtual; abstract;
    function FindNextMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode; virtual; abstract;
    procedure ChangeOnce(Word, NewWord: String); virtual; abstract;
    procedure ChangeAlways(Word, NewWord: String); virtual; abstract;
    procedure Add(Word: String);  virtual; abstract;
    procedure IgnoreAlways(Word: String); virtual; abstract;
    procedure GetVariants(Word: String; Variants: TStrings); virtual; abstract;
    property Language: TLanguage read FLanguage;
    property Options: TSpellOptions read FOptions;
    function GetChangeText: String; virtual; abstract;
    function GetMisspellText: String; virtual; abstract;
    property ChangeText: String read GetChangeText;
    property MisspellText: String read GetMisspellText;
    property NotActive: Boolean read FNotActive;
  end;
  TSpellerClass = class of TAbstractSpeller;

  ESpellError = class(Exception);

  TSpellerDialog2 = class(TForm)
    InfoMsg: TLabel;
    Misspelling: TRichEdit;
    Label2: TLabel;
    Variants: TListBox;
    ChangeButton: TButton;
    ChangeAllButton: TButton;
    SkipButton: TButton;
    SkipAllButton: TButton;
    AddButton: TButton;
    CancelButton: TButton;
    CancelEdit: TButton;
    DelButton: TButton;
    StartButton: TButton;
    procedure FormCreate(Sender: TObject);
    procedure DelButtonClick(Sender: TObject);
    procedure SkipButtonClick(Sender: TObject);
    procedure SkipAllButtonClick(Sender: TObject);
    procedure AddButtonClick(Sender: TObject);
    procedure ChangeButtonClick(Sender: TObject);
    procedure ChangeAllButtonClick(Sender: TObject);
    procedure MisspellingProtectChange(Sender: TObject; StartPos,
      EndPos: Integer; var AllowChange: Boolean);
    procedure MisspellingChange(Sender: TObject);
    procedure CancelEditClick(Sender: TObject);
    procedure CancelButtonClick(Sender: TObject);
    procedure StartButtonClick(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
    Checker: TSpellChecker;
    FShowing: Boolean;
    FBlockStart,
    FBlockEnd: Integer;
    procedure ShowForChange(Msg: TCaption);
    procedure ShowForDelete;
    procedure ShowForEdit(Msg: TCaption);
    procedure ShowMisspelling;
    procedure GetHotArea(var SS, SL: Integer);
  public
    { Public declarations }
    constructor Create(SpellChecker: TSpellChecker);
  end;

const
  langAfrikaans = TLanguage(1078);
  langAlbanian = TLanguage(1052);
  langArabic = TLanguage(1025);
  langBasque = TLanguage(1069);
  langBelgianDutch = TLanguage(2067);
  langBelgianFrench = TLanguage(2060);
  langBrazilianPortuguese = TLanguage(1046);
  langBulgarian = TLanguage(1026);
  langByelorussian = TLanguage(1059);
  langCatalan = TLanguage(1027);
  langCroatian = TLanguage(1050);
  langCzech = TLanguage(1029);
  langDanish = TLanguage(1030);
  langDutch = TLanguage(1043);
  langEnglishAUS = TLanguage(3081);
  langEnglishCanadian = TLanguage(4105);
  langEnglishNewZealand = TLanguage(5129);
  langEnglishSouthAfrica = TLanguage(7177);
  langEnglishUK = TLanguage(2057);
  langEnglishUS = TLanguage(1033);
  langEstonian = TLanguage(1061);
  langFaeroese = TLanguage(1080);
  langFarsi = TLanguage(1065);
  langFinnish = TLanguage(1035);
  langFinnishSwedish = TLanguage(2077);
  langFrench = TLanguage(1036);
  langFrenchCanadian = TLanguage(3084);
  langGerman = TLanguage(1031);
  langGreek = TLanguage(1032);
  langHebrew = TLanguage(1037);
  langHungarian = TLanguage(1038);
  langItalian = TLanguage(1040);
  langIcelandic = TLanguage(1039);
  langIndonesian = TLanguage(1057);
  langJapanese = TLanguage(1041);
  langKorean = TLanguage(1042);
  langLatvian = TLanguage(1062);
  langLithuanian = TLanguage(1063);
  langMacedonian = TLanguage(1071);
  langMalaysian = TLanguage(1086);
  langMexicanSpanish = TLanguage(2058);
  langNorwegianBokmol = TLanguage(1044);
  langNorwegianNynorsk = TLanguage(2068);
  langPolish = TLanguage(1045);
  langPortuguese = TLanguage(2070);
  langRomanian = TLanguage(1048);
  langRussian = TLanguage(1049);
  langSerbianCyrillic = TLanguage(3098);
  langSerbianLatin = TLanguage(2074);
  langSesotho = TLanguage(1072);
  langSimplifiedChinese = TLanguage(2052);
  langSlovak = TLanguage(1051);
  langSlovenian = TLanguage(1060);
  langSpanish = TLanguage(1034);
  langSpanishModernSort = TLanguage(3082);
  langSwedish = TLanguage(1053);
  langSwissFrench = TLanguage(4108);
  langSwissGerman = TLanguage(2055);
  langSwissItalian = TLanguage(2064);
  langThai = TLanguage(1054);
  langTraditionalChinese = TLanguage(1028);
  langTsonga = TLanguage(1073);
  langTswana = TLanguage(1074);
  langTurkish = TLanguage(1055);
  langUkrainian = TLanguage(1058);
  langVenda = TLanguage(1075);
  langVietnamese = TLanguage(1066);
  langXhosa = TLanguage(1076);
  langZulu = TLanguage(1077);

var
  SpellerDialog2: TSpellerDialog2;

procedure Register;
function GetSpellLanguages(Languages: TStrings; Option: TLangOption): Integer;

implementation

uses
  CSAPI, SpellRes, DsgnIntf, Registry;

{$R *.DFM}

type
  TDictFileProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

procedure TDictFileProperty.Edit;
var
  FileOpen: TOpenDialog;
begin
  FileOpen := TOpenDialog.Create(Application);
  FileOpen.Filename := GetValue;
  FileOpen.Filter := 'Dictionary files (*.dic)|*.dic|All files (*.*)|*.*';
  FileOpen.Options := FileOpen.Options + [ofShowHelp, ofPathMustExist,
    ofFileMustExist];
  try
    if FileOpen.Execute then SetValue(FileOpen.Filename);
  finally
    FileOpen.Free;
  end;
end;

function TDictFileProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paRevertable];
end;

procedure CheckSR(SR: TSEC);
begin
  if SR<>secNoErrors then
    raise ESpellError.CreateFmt(spsError, [SR]);
end;

type
  TMemoClass = class of TCustomMemo;

{TAbstractSpeller}
constructor TAbstractSpeller.Create(Language: TLanguage; Owner: TSpellChecker; Options: TSpellOptions);
begin
  inherited Create;
  FLanguage:= Language;
  FOptions:= Options;
  SpellChecker:= Owner;
end;

{TCSAPISpeller}
type
  TCSAPISpeller = class(TAbstractSpeller)
  private
    SpellInstance: THandle;
    DLLName: String;
    LexName: String;
    UserDict: String;
    UserExc: String;
    UnkWord: String;
    FOptions: TSpellOptions;
    SpellVer: TSpellVerFunc;
    SpellInit: TSpellInitFunc;
    SpellOptions: TSpellOptionsFunc;
    SpellCheck: TSpellCheckFunc;
    SpellTerminate: TSpellTerminateFunc;
    SpellVerifyMdr: TSpellVerifyMdrFunc;
    SpellOpenMdr: TSpellOpenMdrFunc;
    SpellOpenUdr: TSpellOpenUdrFunc;
    SpellAddUdr: TSpellAddUdrFunc;
    SpellAddChangeUdr: TSpellAddChangeUdrFunc;
    SpellDelUdr: TSpellDelUdrFunc;
    SpellClearUdr: TSpellClearUdrFunc;
    SpellGetSizeUdr: TSpellGetSizeUdrFunc;
    SpellGetListUdr: TSpellGetListUdrFunc;
    SpellCloseMdr: TSpellCloseMdrFunc;
    SpellCloseUdr: TSpellCloseUdrFunc;
  protected
    Handle: TSPLID;
    SpecChars: TWSC;
    Mdrs: TMDRS;
    Udr: TUDR;
    InputBuffer: TSIB;
    ResultBuffer: TSRB;
    constructor Create(Language: TLanguage; Owner: TSpellChecker; Options: TSpellOptions); override;
    destructor Destroy; override;
    function FindMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode; override;
    function FindNextMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode; override;
    procedure ChangeOnce(Word, NewWord: String); override;
    procedure ChangeAlways(Word, NewWord: String); override;
    procedure Add(Word: String); override;
    procedure IgnoreAlways(Word: String); override;
    procedure GetVariants(Word: String; Variants: TStrings); override;
    function GetChangeText: String; override;
    function GetMisspellText: String; override;
  end;

{$O-}
constructor TCSAPISpeller.Create(Language: TLanguage; Owner: TSpellChecker; Options: TSpellOptions);
var
  UdrRO: Boolean;
  NotFound: Boolean;
  Registry: TRegistry;
begin
  inherited;
  with SpecChars do
    begin
      bIgnore:= #0;
      bHyphenHard:= #45;
      bHyphenSoft:= #31;
      bHyphenNonBreaking:= #30;
      bEmDash:= #151;
      bEnDash:= #150;
      bEllipsis:= #133;
      rgLineBreak:= #11#10;
      rgParaBreak:= #13#10;
    end;
  Registry:= TRegistry.Create;
  Registry.RootKey:= HKEY_LOCAL_MACHINE;
  NotFound:= False;
  if Registry.OpenKey(
       Format('\SOFTWARE\Microsoft\Shared Tools\Proofing Tools\Spelling\%d\Normal', [FLanguage]),
       False) or
     Registry.OpenKey(
       Format('\SOFTWARE\Microsoft\Shared Tools\Proofing Tools\Spelling\%d\Normal', [1024+(FLanguage mod 1024)]),
       False) then
    begin
      DLLName:= Registry.ReadString('Engine');
      LexName:= Registry.ReadString('Dictionary');
    end
  else
    NotFound:= True;
  if not NotFound then
    begin
      if (SpellChecker.CustomDict='') and Registry.OpenKey(
         '\SOFTWARE\Microsoft\Shared Tools\Proofing Tools\Custom Dictionaries', True) then
        UserDict:= Registry.ReadString('1')
      else
        UserDict:= SpellChecker.CustomDict;
      if Assigned(SpellChecker.FOnGetDict) then
        SpellChecker.FOnGetDict(SpellChecker, FLanguage, UserDict);
      if UserDict='' then
        begin
          UserDict:= ExtractFilePath(LexName)+'CUSTOM.DIC';
          Registry.WriteString('1', UserDict);
        end;
    end;
  Registry.Free;
  if NotFound then
    begin
      FNotActive:= True;
      Exit;
    end;
  try
    SpellInstance:= LoadLibrary(PChar(DllName));
  except
    FNotActive:= True;
    raise ESpellError.CreateFmt(spsErrorLoad, [DllName]);
  end;
  try
    @SpellVer:= GetProcAddress(SpellInstance, 'SpellVer');
    @SpellInit:= GetProcAddress(SpellInstance, 'SpellInit');
    @SpellOptions:= GetProcAddress(SpellInstance, 'SpellOptions');
    @SpellCheck:= GetProcAddress(SpellInstance, 'SpellCheck');
    @SpellTerminate:= GetProcAddress(SpellInstance, 'SpellTerminate');
    @SpellVerifyMdr:= GetProcAddress(SpellInstance, 'SpellVerifyMdr');
    @SpellOpenMdr:= GetProcAddress(SpellInstance, 'SpellOpenMdr');
    @SpellOpenUdr:= GetProcAddress(SpellInstance, 'SpellOpenUdr');
    @SpellAddUdr:= GetProcAddress(SpellInstance, 'SpellAddUdr');
    @SpellAddChangeUdr:= GetProcAddress(SpellInstance, 'SpellAddChangeUdr');
    @SpellDelUdr:= GetProcAddress(SpellInstance, 'SpellDelUdr');
    @SpellClearUdr:= GetProcAddress(SpellInstance, 'SpellClearUdr');
    @SpellGetSizeUdr:= GetProcAddress(SpellInstance, 'SpellGetSizeUdr');
    @SpellGetListUdr:= GetProcAddress(SpellInstance, 'SpellGetListUdr');
    @SpellCloseMdr:= GetProcAddress(SpellInstance, 'SpellCloseMdr');
    @SpellCloseUdr:= GetProcAddress(SpellInstance, 'SpellCloseUdr');
  except
    FreeLibrary(SpellInstance);
    FNotActive:= True;
    raise ESpellError.CreateFmt(spsErrorLoad, [DLLName]);
  end;
  FNotActive:= False;
  FOptions:= Options;
  CheckSR(SpellInit(Handle, SpecChars));
  CheckSR(SpellOptions(Handle, Word(FOptions)));
  CheckSR(SpellOpenMdr(Handle, PChar(LexName), nil, False, True, FLanguage, Mdrs));
  CheckSR(SpellOpenUdr(Handle, PChar(UserDict), True, IgnoreAlwaysProp, Udr, UdrRO));
  with InputBuffer do
    begin
      cMdr:= 1;
      cUdr:= 1;
      lrgMdr:= @Mdrs.MDR;
      lrgUdr:= @Udr;
    end;
  with ResultBuffer do
    begin
      cch:= 1024;
      lrgsz:= AllocMem(1024);
      lrgbRating:= AllocMem(255);
      cbRate:= 255;
    end;
end;

destructor TCSAPISpeller.Destroy;
begin
  if not FNotActive then
    begin
      FreeMem(ResultBuffer.lrgsz);
      FreeMem(ResultBuffer.lrgbRating);
      CheckSR(SpellCloseMdr(Handle, Mdrs));
      CheckSR(SpellCloseUdr(Handle, Udr, True));
      CheckSR(SpellTerminate(Handle, True));
      try
        FreeLibrary(SpellInstance);
      except
        raise ESpellError.CreateFmt(spsErrorUnload, [DLLName]);
      end;
    end;
  inherited;
end;

function TCSAPISpeller.FindMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode;
begin
  if FNotActive then
    begin
      Result:= srNoErrors;
      Exit;
    end;
  InputBuffer.cch:= MaxLen;
  InputBuffer.lrgch:= Buf;
  InputBuffer.wSpellState:= fssNoStateInfo;
  CheckSR(SpellCheck(handle, sccVerifyBuffer, InputBuffer, ResultBuffer));
  Result:= TSpellReturnCode(ResultBuffer.scrs);
  if Result<>srNoErrors then
    begin
      Start:= ResultBuffer.ichError;
      Len:= ResultBuffer.cchError;
      SetLength(UnkWord, ResultBuffer.cchError);
      StrLCopy(@UnkWord[1], InputBuffer.lrgch+ResultBuffer.ichError, ResultBuffer.cchError);
    end;
end;

function TCSAPISpeller.FindNextMisspell(Buf: PChar; MaxLen: Integer; var Start, Len: Integer): TSpellReturnCode;
begin
  if FNotActive then
    begin
      Result:= srNoErrors;
      Exit;
    end;
  InputBuffer.cch:= MaxLen;
  InputBuffer.lrgch:= Buf;
  InputBuffer.wSpellState:= fssIsContinued;
  CheckSR(SpellCheck(handle, sccVerifyBuffer, InputBuffer, ResultBuffer));
  Result:= TSpellReturnCode(ResultBuffer.scrs);
  if Result<>srNoErrors then
    begin
      Start:= ResultBuffer.ichError;
      Len:= ResultBuffer.cchError;
      SetLength(UnkWord, ResultBuffer.cchError);
      StrLCopy(@UnkWord[1], InputBuffer.lrgch+ResultBuffer.ichError, ResultBuffer.cchError);
    end;
end;

procedure TCSAPISpeller.ChangeOnce(Word, NewWord: String);
begin
  if FNotActive then
    Exit;
  CheckSR(SpellAddChangeUdr(Handle, udrChangeOnce,
      PChar(Word), PChar(NewWord)));
end;

procedure TCSAPISpeller.ChangeAlways(Word, NewWord: String);
begin
  if FNotActive then
    Exit;
  CheckSR(SpellAddChangeUdr(Handle, udrChangeAlways,
      PChar(Word), PChar(NewWord)));
end;

procedure TCSAPISpeller.Add(Word: String);
begin
  if FNotActive then
    Exit;
  CheckSR(SpellAddUdr(Handle, Udr, PChar(Word)));
end;

procedure TCSAPISpeller.IgnoreAlways(Word: String);
begin
  if FNotActive then
    Exit;
  CheckSR(SpellAddUdr(Handle, udrIgnoreAlways, PChar(Word)));
end;

procedure TCSAPISpeller.GetVariants(Word: String; Variants: TStrings);
var
  SIB: TSIB;
  SRB: TSRB;
  Buf: array[0..2047]of Char;
  Ratings: array[0..255]of Byte;
  P: PChar;
begin
  Variants.Clear;
  if FNotActive then
    Exit;
  with SIB do
    begin
      cch:= Length(Word);
      cMdr:= 1;
      cUdr:= 1;
      wSpellState:= fssNoStateInfo;
      lrgch:= @Word[1];
      lrgMdr:= @Mdrs.MDR;
      lrgUdr:= @Udr;
    end;
  with SRB do
    begin
      cch:= 2047;
      lrgsz:= @Buf;
      lrgbRating:= @Ratings;
      cbRate:= 255;
    end;
  CheckSR(SpellCheck(Handle, sccSuggest, SIB, SRB));
  while SRB.scrs<>scrsNoMoreSuggestions do
    begin
      P:= SRB.lrgsz;
      while P^<>#0 do
        begin
          if Variants.IndexOf(P)=-1 then
            Variants.Add(P);
          while P^<>#0 do
            Inc(P);
          Inc(P);
        end;
      CheckSR(SpellCheck(Handle, sccSuggestMore, SIB, SRB));
    end;
end;

function TCSAPISpeller.GetChangeText: String;
begin
  if FNotActive then
    Result:= ''
  else
    Result:= ResultBuffer.lrgsz;
end;

function TCSAPISpeller.GetMisspellText: String;
begin
  if FNotActive then
    Result:= ''
  else
    Result:= UnkWord;
end;
{$O+}


type
  THTMLBracket = (thbTag, thbComment, thbBasic);

const
  OpenBracket: array[THTMLBracket] of PChar=('<', '<!--', '<%');
  CloseBracket: array[THTMLBracket] of PChar=('>', '-->', '%>');

var
  ControlTypes: TStrings;

{TSpellChecker}
constructor TSpellChecker.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage:= GetSystemDefaultLCID;
  FOptions:= [spoSuggestFromUserDict, spoIgnoreAllCaps, spoIgnoreMixedDigits,
              spoIgnoreRomanNumerals];
  Spellers:= TList.Create;
end;

destructor TSpellChecker.Destroy;
var
  I: Integer;
begin
  for I:= Spellers.Count-1 downto 0 do
    TAbstractSpeller(Spellers.Items[I]).Free;
  Spellers.Free;
  inherited;
end;

class procedure TSpellChecker.RegisterEditControl(MemoClass: String; Unicode, MultiLanguage: Boolean);
begin
  ControlTypes.AddObject(MemoClass, Pointer(Ord(Unicode) or Ord(MultiLanguage)*2));
end;

procedure TSpellChecker.GetMemoProperties;
var
  C: TClass;
  I: Integer;
begin
  C:= FMemo.ClassType;
  repeat
    for I:= 0 to ControlTypes.Count-1 do
      if AnsiCompareText(C.ClassName, ControlTypes[I])=0 then
        begin
          FLangSupport:= Boolean(Integer(ControlTypes.Objects[I]) shr 1);
          FUnicode:= Boolean(Integer(ControlTypes.Objects[I]) and 1);
          Exit;
        end;
    C:= C.ClassParent;
  until C=TCustomEdit;                     
  raise ESpellError.CreateFmt('You can''t spell check %s.', [FMemo.Name]);
end;

procedure TSpellChecker.SetLanguage(Value: TLanguage);
begin
  FLanguage:= Value;
end;

function TSpellChecker.OpenLanguage(Value: TLanguage): Boolean;
var
  Speller: TAbstractSpeller;
begin
  try
    Speller:= TCSAPISpeller.Create(Value, Self, FOptions);
  except
    Speller.Free;
    Result:= False;
    Exit;
  end;
  Spellers.Add(Speller);
  Result:= True;
end;

function TSpellChecker.FindLanguage(Value: TLanguage): TAbstractSpeller;
var
  I: Integer;
begin
  for I:= 0 to Spellers.Count-1 do
    if TAbstractSpeller(Spellers.Items[I]).Language=Value then
      begin
        Result:= TAbstractSpeller(Spellers.Items[I]);
        Exit;
      end;
  if OpenLanguage(Value) then
    Result:= TAbstractSpeller(Spellers.Items[Spellers.Count-1])
  else
    Result:= nil;
end;

function TSpellChecker.IsKnownWord(Word: String; Language: TLanguage): Boolean;
var
  Start, Len: Integer;
begin
  with FindLanguage(Language) do
    Result:= FindMisspell(@Word[1], Length(Word), Start, Len)=srNoErrors;
end;

procedure TSpellChecker.AddWord(Word: String; Language: TLanguage);
begin
  with FindLanguage(Language) do
    Add(Word);
end;

procedure TSpellChecker.GetVariants(Word: String; Variants: TStrings; Language: TLanguage);
begin
  with FindLanguage(Language) do
    GetVariants(Word, Variants);
end;

procedure TSpellChecker.GetTag(From: Integer; var Len: Integer);
var
  P, PP: PChar;
  HTMLTag: THTMLBracket;
  C: Char;
  S: String;
begin
  SetLength(S, FSpellEnd-From);
  GetTextRange(@S[1], From, FSpellEnd, 1252);
  if StrLComp(@S[1], OpenBracket[thbComment], StrLen(OpenBracket[thbComment]))=0 then
    HTMLTag:= thbComment
  else if StrLComp(@S[1], OpenBracket[thbBasic], StrLen(OpenBracket[thbBasic]))=0 then
    HTMLTag:= thbBasic
  else
    HTMLTag:= thbTag;
  P:= StrPos(@S[1], CloseBracket[HTMLTag])+StrLen(CloseBracket[HTMLTag]);
  if HTMLTag<>thbBasic then
    begin
      PP:= StrScan(@S[2], '<');
      if (PP<>nil) and (PP<P) then
        begin
          GetTag(PP-@S[1]+From, Len);
          P:= StrPos(PP+Len, CloseBracket[HTMLTag])+StrLen(CloseBracket[HTMLTag]);
        end;
    end;
  if P=nil then
    Len:= Length(S)
  else
    Len:= P-@S[1];
end;

procedure TSpellChecker.GetTextRange(Buf: PChar; StartPos, EndPos: Integer; CP: Word);
type
{ The declarations of TTextRangeA and TTextRangeW in Richedit.pas are incorrect}
  TTextRangeA = record
    chrg: TCharRange;
    lpstrText: PAnsiChar; {not AnsiChar!}
  end;
var
  TR: TTextRangeA;
  W: WideString;
  S: String;
  GTL: TGetTextLengthEx;
  GT: TGetTextEx;
  L: Integer;
begin
  GTL.flags:= GTL_DEFAULT;
  GTL.codepage:= 1200;
  L:= FMemo.Perform(EM_GETTEXTLENGTHEX, Integer(@GTL), 0);
  if L>0 then
    begin
      SetLength(W, L);
      GT.cb:= L*2+2;
      GT.flags:= GT_DEFAULT;
      GT.codepage:= 1200;
      GT.lpDefaultChar:= nil;
      GT.lpUsedDefChar:= nil;
      FMemo.Perform(EM_GETTEXTEX, Integer(@GT), Integer(@W[1]));
      WideCharToMultiByte(CP, 0, @W[StartPos+1], EndPos-StartPos, Buf, EndPos-StartPos, nil, nil);
      Buf[EndPos-StartPos]:= #0;
    end
  else
    begin
      S:= FBackMemo.Text;
      StrLCopy(Buf, @S[StartPos+1], EndPos-StartPos);
    end;
end;

function TSpellChecker.GetMemoLanguage: TLanguage;
var
  CF: TCharFormat2A;
  CFW: TCharFormat2W;
begin
  if not FLangSupport then
    Result:= FLanguage
  else if FUnicode then
    begin
      FillChar(CFW, SizeOf(CFW), 0);
      CFW.cbSize:= SizeOf(CFW);
      FBackMemo.Perform(EM_GETCHARFORMAT, 1, LongInt(@CFW));
      Result:= CFW.lid;
    end
  else
    begin
      FillChar(CF, SizeOf(CF), 0);
      CF.cbSize:= SizeOf(CF);
      FBackMemo.Perform(EM_GETCHARFORMAT, 1, LongInt(@CF));
      Result:= CF.lid;
    end;
end;

procedure TSpellChecker.GetBlock(From: Integer; var StartPos, EndPos: Integer);
var
  L, Lang: TLanguage;
  FT: TFindTextA;
  FTW: TFindTextW;
  C: Char;
  P, Len, LP: Integer;
  S: String;
  Pos: PChar;
begin
  P:= From-1;
  FStartSentence:= False;
  repeat
    Inc(P);
    GetTextRange(@C, P, P+1, 1252);
    if FHTML and (C='<') then
      begin
        GetTag(P, Len);
        Inc(P, Len-1);
      end
    else if C=#13 then
      FStartSentence:= True
    else if not (C in [#10, #13]) then
      Break;
  until P>=FSpellEnd;
  if P<FSpellEnd then
    begin
      StartPos:= P;
      if FUnicode then
        begin
          if CRPos<=StartPos then
            begin
              FTW.chrg.cpMin:= StartPos;
              FTW.chrg.cpMax:= FSpellEnd;
              FTW.lpstrText:= #13;
              CRPos:= FBackMemo.Perform(EM_FINDTEXTEX, 1, LongInt(@FTW));
              if CRPos=-1 then
                CRPos:= FSpellEnd;
            end;
          if FHTML then
            if (TagPos<=StartPos) then
              begin
                FTW.chrg.cpMin:= StartPos;
                FTW.chrg.cpMax:= FSpellEnd;
                FTW.lpstrText:= '<';
                TagPos:= FBackMemo.Perform(EM_FINDTEXTEX, 1, LongInt(@FTW));
                if TagPos=-1 then
                  TagPos:= FSpellEnd;
              end
            else
          else
            TagPos:= FSpellEnd;
        end
      else if (FBackMemo is TCustomRichEdit) then
        begin
          if CRPos<=StartPos then
            begin
              FT.chrg.cpMin:= StartPos;
              FT.chrg.cpMax:= FSpellEnd;
              FT.lpstrText:= #13;
              CRPos:= FBackMemo.Perform(EM_FINDTEXT, 1, LongInt(@FT));
              if CRPos=-1 then
                CRPos:= FSpellEnd;
            end;
          if FHTML then
            if (TagPos<=StartPos) then
              begin
                FT.chrg.cpMin:= StartPos;
                FT.chrg.cpMax:= FSpellEnd;
                FT.lpstrText:= '<';
                TagPos:= FBackMemo.Perform(EM_FINDTEXT, 1, LongInt(@FT));
                if TagPos=-1 then
                  TagPos:= FSpellEnd;
              end
            else
          else
            TagPos:= FSpellEnd;
        end
      else
        begin
          S:= FMemo.Text;
          Pos:= StrScan(@S[StartPos+1], #13);
          if Pos<>nil then
            CRPos:= Pos-@S[1]
          else
            CRPos:= FSpellEnd;
          Pos:= StrScan(@S[StartPos+1], '<');
          if Pos<>nil then
            TagPos:= Pos-@S[1]
          else
            TagPos:= FSpellEnd;
        end;
      if FLangSupport then
        if LangPos<=StartPos then
          begin
            FBackMemo.Perform(EM_SETSEL, StartPos, StartPos+1);
            L:= GetMemoLanguage;
            LangPos:= StartPos;
            repeat
              LP:= FBackMemo.Perform(EM_FINDWORDBREAK, WB_RIGHT, LangPos);
              if LP<LangPos then
                Break;
              if (LP=LangPos) and (LP<FSpellEnd) then
                Inc(LP);
              LangPos:= LP;
              FBackMemo.Perform(EM_SETSEL, LangPos, LangPos+1);
              Lang:= GetMemoLanguage;
            until (Lang<>L) or (LangPos>=FSpellEnd);
          end
        else
      else
        LangPos:= FSpellEnd;
      EndPos:= MinIntValue([CRPos, TagPos, LangPos]);
    end
  else
    begin
      StartPos:= FSpellEnd;
      EndPos:= FSpellEnd;
    end;
end;

procedure TSpellChecker.Init;
var
  MemoClass: TMemoClass;
  MS: TMemoryStream;
  CFW: TCharFormat2W;
  CF: TCharFormat2A;
  S: String;
  P: PChar;
  L: Integer;
  PT: Boolean;
begin
  GetMemoProperties;
  MemoClass:= TMemoClass(FMemo.ClassType);
  if Assigned(FBackMemo) then
    FBackMemo.Free;
  FBackMemo:= MemoClass.Create(FMemo.Owner);
  FBackMemo.Visible := False;
  FBackMemo.Parent := FMemo.Owner as TWinControl;
  MS:= TMemoryStream.Create;
  if (FMemo is TCustomRichEdit) then
    begin
      PT:= TRichEdit(FMemo).PlainText;
      TRichEdit(FMemo).PlainText:= False;
      TRichEdit(FMemo).Lines.SaveToStream(MS);
      MS.Position:= 0;
      TRichEdit(FBackMemo).Lines.LoadFromStream(MS);
      TRichEdit(FMemo).PlainText:= PT;
    end
  else
    begin
      FMemo.Lines.SaveToStream(MS);
      MS.Position:= 0;
      FBackMemo.Lines.LoadFromStream(MS);
    end;
  MS.Free;
  FMemo.Perform(EM_GETSEL, Integer(@FSpellStart), Integer(@FSpellEnd));
  if FSpellEnd=FSpellStart then
    FSpellEnd:= FMemo.Perform(WM_GETTEXTLENGTH, 0, 0);
  FSpellStart:= FMemo.Perform(EM_FINDWORDBREAK, WB_LEFTBREAK, FSpellStart);
  if FSpellStart>0 then
    FSpellStart:= FMemo.Perform(EM_FINDWORDBREAK, WB_RIGHT, FSpellStart);
  if FHTML then
    begin
      S:= FBackMemo.Text;
      P:= @S[1];
      repeat
        P:= StrPos(P, '<');
        if (P=nil) or (P-@S[1]>FSpellStart) then
          Break;
        GetTag(P-@S[1], L);
        if (P-@S[1]+L>FSpellStart) then
          begin
            if (P-@S[1]+L<FSpellEnd) then
              FSpellStart:= P-@S[1]+L
            else
              FSpellStart:= FSpellEnd;
            Break;
          end;
        Inc(P, L);
      until False;
    end;
  FMisspellStart:= FSpellStart;
  FMisspellLen:= 0;
  CrPos:= 0;
  TagPos:= 0;
  LangPos:= 0;
end;

procedure TSpellChecker.Check(Memo: TCustomMemo);
begin
  FMemo:= Memo;
  if not Assigned(FDialog) then
    FDialog:= TSpellerDialog2.Create(Self);
  Init;
  ContinueCheck;
end;

function TSpellChecker.SentenceCapitalize(const S: String): String;
begin
  Result:= S;
  LCMapString(FLanguage, LCMAP_Uppercase, @Result[1], 1, @Result[1], 1);
end;

procedure TSpellChecker.FinishCheck;
begin
  FDialog.Hide;
  FDialog.Close;
  if Assigned(FBackMemo) then
    FBackMemo.Free;
  FBackMemo:= nil;
  if FShowFinishMessage and (FFinishMessage<>'') then
    MessageBox(Application.Handle, @FFinishMessage[1], PChar(spsFinishCaption),
               mb_Ok+mb_IconInformation);
end;

procedure TSpellChecker.ContinueCheck;
var
  StartPos,
  EndPos: Integer;
  L: TLanguage;
  Buf: PChar;
begin
  FSRC:= srNoErrors;
  repeat
    GetBlock(FMisspellStart+FMisspellLen, StartPos, EndPos);
    if StartPos=EndPos then
      begin
        FinishCheck;
        Break;
      end;
    FBackMemo.Perform(EM_SETSEL, StartPos, StartPos+1);
    L:= GetMemoLanguage;
    FSpeller:= Findlanguage(L);
    Buf:= AllocMem(EndPos-StartPos+1);
    GetTextRange(Buf, StartPos, EndPos, CodePageFromLocale(L));
    if FStartSentence then
      FSRC:= FSpeller.FindMisspell(Buf, EndPos-StartPos, FMisspellStart, FMisspellLen)
    else
      FSRC:= FSpeller.FindNextMisspell(Buf, EndPos-StartPos, FMisspellStart, FMisspellLen);
    FreeMem(Buf);
    if FSRC<>srNoErrors then
      begin
        Inc(FMisspellStart, StartPos);
        FMemo.SelStart:= FMisspellStart;
        FMemo.SelLength:= FMisspellLen;
        FBackMemo.SelStart:= FMisspellStart;
        FBackMemo.SelLength:= FMisspellLen;
        FMisspellText:= FSpeller.MisspellText;
      end
    else
      begin
        FMisspellStart:= EndPos;
        FMisspellLen:= 0;
      end;
    if (FSRC=srReturningChangeAlways) then
      Change(FSpeller.ChangeText);
  until not (FSRC in [srNoErrors, srReturningChangeAlways]);
  if not (FSRC in [srNoErrors, srReturningChangeAlways]) then
    begin
      if not FDialog.Visible then
        if FModalDialog then
          FDialog.ShowModal
        else
          FDialog.Show;
        FDialog.Caption:= Format(Caption, [LanguageName(FSpeller.Language)]);
        case FSRC of
        srUnknownInputWord:
          FDialog.ShowForChange(spsNotFound);
        srReturningChangeOnce:
          begin
            FDialog.ShowForChange(spsNotFound);
            if FDialog.Variants.Items.IndexOf(FSpeller.ChangeText)=-1 then
              FDialog.Variants.Items.Insert(0, FSpeller.ChangeText);
          end;
        srInvalidHyphenation:
          FDialog.ShowForChange(spsHyphen);
        srErrorCapitalization:
          FDialog.ShowForChange(spsCaps);
        srWordConsideredAbbreviation:
          FDialog.ShowForChange(spsAbbrev);
        srNoSentenceStartCap:
          FDialog.ShowForChange(spsNoSentenceCap);
        srRepeatWord:
          FDialog.ShowForDelete;
        srExtraSpaces:
          FDialog.ShowForChange(spsExtraSpaces);
        srMissingSpace:
          FDialog.ShowForEdit(spsMissingSpace);
        srInitialNumeral:
          FDialog.ShowForEdit(spsInitialNumeral);
        end;
      end;
end;

function TSpellChecker.GetLineFromPos(Pos: Integer; var LineStart: Integer): String;
var
  L: Integer;
  Buf: String;
  PS, PE: Integer;
begin
  L:= FBackMemo.Perform(WM_GETTEXTLENGTH, 0, 0);
  SetLength(Buf, L);
  GetTextRange(@Buf[1], 0, L, CodePageFromLocale(FSpeller.Language));
  PS:= Pos;
  while (PS>0) and not (Buf[PS+1] in [#10, #13]) do
    Dec(PS);
  if (Buf[PS+1] in [#10, #13]) then
    Inc(PS);
  PE:= Pos;
  while not (Buf[PE+1] in [#10, #13, #0]) do
    Inc(PE);
  Result:= Copy(Buf, PS+1, PE-PS);
  LineStart:= PS;
end;

function TSpellChecker.GetCurrentLanguage: TLanguage;
begin
  if Assigned(FSpeller) then
    Result:= FSpeller.Language
  else
    Result:= 0;
end;

procedure TSpellChecker.ChangeOnce(Word: String);
var
  N: Integer;
begin
  FSpeller.ChangeOnce(FSpeller.MisspellText, Word);
  FMemo.SelText:= Word;
  FBackMemo.SelText:= Word;
  if Assigned(FOnChangeText) then
    FOnChangeText(Self, FMisspellStart, FMisspellLen, Word);
  N:= Length(Word)-FMisspellLen;
  if N<>0 then
    begin
      Inc(FSpellEnd, N);
      Inc(FMisspellLen, N);
      Inc(CrPos, N);
      Inc(TagPos, N);
      Inc(LangPos, N);
    end;
end;

procedure TSpellChecker.Change(Word: String);
var
  N: Integer;
begin
  FMemo.SelText:= Word;
  FBackMemo.SelText:= Word;
  if Assigned(FOnChangeText) then
    FOnChangeText(Self, FMisspellStart, FMisspellLen, Word);
  N:= Length(Word)-FMisspellLen;
  if N<>0 then
    begin
      Inc(FSpellEnd, N);
      Inc(FMisspellLen, N);
      Inc(CrPos, N);
      Inc(TagPos, N);
      Inc(LangPos, N);
    end;
end;

procedure TSpellChecker.ChangeAlways(Word: String);
var
  N: Integer;
begin
  FSpeller.ChangeAlways(FSpeller.MisspellText, Word);
  FMemo.SelText:= Word;
  FBackMemo.SelText:= Word;
  if Assigned(FOnChangeText) then
    FOnChangeText(Self, FMisspellStart, FMisspellLen, Word);
  N:= Length(Word)-FMisspellLen;
  if N<>0 then
    begin
      Inc(FSpellEnd, N);
      Inc(FMisspellLen, N);
      Inc(CrPos, N);
      Inc(TagPos, N);
      Inc(LangPos, N);
    end;
end;

procedure TSpellChecker.Delete;
begin
  Inc(FMisspellLen);
  FMemo.SelLength:= FMisspellLen;
  FBackMemo.SelLength:= FMisspellLen;
  FMemo.SelText:= '';
  FBackMemo.SelText:= '';
  if Assigned(FOnChangeText) then
    FOnChangeText(Self, FMisspellStart, FMisspellLen, '');
  Dec(FSpellEnd, FMisspellLen);
  Dec(CrPos, FMisspellLen);
  Dec(TagPos, FMisspellLen);
  Dec(LangPos, FMisspellLen);
  FMisspellLen:= 0;
end;

procedure TSpellChecker.Add;
begin
  FSpeller.Add(FSpeller.MisspellText);
  FMisspellLen:= 0;
end;

procedure TSpellChecker.IgnoreAlways;
begin
  FSpeller.IgnoreAlways(FSpeller.MisspellText);
end;

function DelExtraSpaces(const S: String): String;
var
  I: Integer;
begin
  I:= Length(S)-2;
  while S[I]=' ' do
    Dec(I);
  Result:= Copy(S, 1, I)+Copy(S, Length(S)-1, 2);
end;


{TSpellerDialog2}
constructor TSpellerDialog2.Create(SpellChecker: TSpellChecker);
begin
  Checker:= SpellChecker;
  inherited Create(SpellChecker.Owner);
end;

procedure TSpellerDialog2.FormCreate(Sender: TObject);
begin
  Label2.Caption:= spsVariants;
  ChangeButton.Caption:= spsChange;
  ChangeAllButton.Caption:= spsChangeAll;
  SkipButton.Caption:= spsSkip;
  SkipAllButton.Caption:= spsSkipAll;
  DelButton.Caption:= spsDelete;
  CancelButton.Caption:= spsCancel;
  CancelEdit.Caption:= spsCancelEdit;
  StartButton.Caption:= spsStart;
end;

procedure TSpellerDialog2.ShowForChange(Msg: TCaption);
var
  UnkWord: String;
begin
  ShowMisspelling;
  InfoMsg.Caption:= Msg;
  StartButton.Hide;
  CancelEdit.Hide;
  DelButton.Hide;
  ChangeButton.Show;
  SkipButton.Enabled:= True;
  SkipAllButton.Enabled:= (Checker.FSRC=srUnknownInputWord) and (Length(Checker.FMisspellText)<65);
  AddButton.Enabled:= (Checker.FSRC=srUnknownInputWord) and (Length(Checker.FMisspellText)<65);
  Variants.Font.Charset:= CharSetFromLocale(Checker.FSpeller.Language);
  UnkWord:= Checker.FMisspellText;
  case Checker.FSRC of
  srUnknownInputWord,
  srReturningChangeOnce,
  srInvalidHyphenation,
  srErrorCapitalization:
    Checker.GetVariants(UnkWord, Variants.Items, Checker.FSpeller.Language);
  srNoSentenceStartCap:
    begin
      Variants.Items.Clear;
      Variants.Items.Add(Checker.SentenceCapitalize(UnkWord));
    end;
  srExtraSpaces:
    begin
      Variants.Items.Clear;
      Variants.Items.Add(DelExtraSpaces(UnkWord));
    end;
  end;
  Variants.ItemIndex:= 0;
  ChangeButton.Enabled:= Variants.ItemIndex=0;
  ChangeAllButton.Enabled:= (Variants.ItemIndex=0) and (Checker.FSRC in [srUnknownInputWord, srReturningChangeOnce]);
  Variants.Enabled:= Variants.ItemIndex=0;
  Label2.Enabled:= Variants.ItemIndex=0;
  if not Visible then
    if Checker.ModalDialog then
      ShowModal
    else
      Show;
end;

procedure TSpellerDialog2.ShowForEdit(Msg: TCaption);
begin
  ShowMisspelling;
  InfoMsg.Caption:= Msg;
  CancelEdit.Hide;
  DelButton.Hide;
  ChangeButton.Show;
  SkipButton.Enabled:= True;
  SkipAllButton.Enabled:= False;
  AddButton.Enabled:= False;
  Variants.Items.Clear;
  ChangeButton.Enabled:= False;
  ChangeAllButton.Enabled:= False;
  Variants.Enabled:= False;
  if not Visible then
    if Checker.ModalDialog then
      ShowModal
    else
      Show;
end;

procedure TSpellerDialog2.ShowForDelete;
begin
  ShowMisspelling;
  InfoMsg.Caption:= spsRepeatedWord;
  CancelEdit.Hide;
  DelButton.Show;
  ChangeButton.Hide;
  ChangeAllButton.Enabled:= False;
  SkipButton.Show;
  SkipAllButton.Enabled:= False;
  AddButton.Enabled:= False;
  if not Visible then
    if Checker.ModalDialog then
      ShowModal
    else
      Show;
end;

procedure TSpellerDialog2.ShowMisspelling;
var
  CurStr: String;
  PS: Integer;
begin
  Misspelling.Enabled:= True;
  FShowing:= True;
  with Misspelling do
    begin
      Lines.Clear;
      DefAttributes.Charset:= CharsetFromLocale(Checker.CurrentLanguage);
      Text:= Checker.GetLineFromPos(Checker.FMisspellStart, PS);
      SelectAll;
      SelAttributes.Color:= clWindowText;
      SelAttributes.Protected:= True;
      SelStart:= Checker.FMisspellStart-PS;
      SelLength:= Checker.FMisspellLen;
      SelAttributes.Protected:= False;
      SelAttributes.Color:= clRed;
      SelLength:= 0;
    end;
  FShowing:= False;
end;

procedure TSpellerDialog2.MisspellingProtectChange(Sender: TObject;
  StartPos, EndPos: Integer; var AllowChange: Boolean);
begin
  AllowChange:= FShowing;
end;

procedure TSpellerDialog2.MisspellingChange(Sender: TObject);
begin
  if FShowing then
    Exit;
  CancelEdit.Show;
  ChangeButton.Show;
  ChangeButton.Enabled:= True;
  ChangeAllButton.Enabled:= False;
  SkipAllButton.Enabled:= False;
  AddButton.Enabled:= False;
  DelButton.Hide;
  Variants.Enabled:= False;
end;

procedure TSpellerDialog2.GetHotArea(var SS, SL: Integer);
var
  I: Integer;
begin
  with Misspelling do
    begin
      SelLength:= 0;
      for I:= 0 to Length(Misspelling.Text)-1 do
        begin
          SelStart:= I;
          if not SelAttributes.Protected then
            begin
              SS:= I;
              Break;
            end;
        end;
      for I:= SS to Length(Misspelling.Text) do
        begin
          SelStart:= I;
          if SelAttributes.Protected then
            begin
              SL:= I-SS-1;
              Exit;
            end;
        end;
      SL:= Length(Misspelling.Lines[0])-SS;
    end;
end;

procedure TSpellerDialog2.CancelEditClick(Sender: TObject);
var
  SS, SL: Integer;
  UnkWord: String;
begin
  GetHotArea(SS, SL);
  UnkWord:= Checker.FMisspellText;
  with Misspelling do
    begin
      SelStart:= SS;
      SelLength:= SL;
      SelText:= UnkWord;
      SelLength:= 0;
    end;
  case Checker.FSRC of
  srUnknownInputWord,
  srReturningChangeOnce,
  srInvalidHyphenation,
  srErrorCapitalization,
  srNoSentenceStartCap,
  srExtraSpaces:
    begin
      DelButton.Hide;
      ChangeButton.Show;
      SkipButton.Enabled:= True;
      SkipAllButton.Enabled:= Checker.FSRC=srUnknownInputWord;
      AddButton.Enabled:= Checker.FSRC=srUnknownInputWord;
      ChangeAllButton.Enabled:= (Variants.ItemIndex=0) and (Checker.FSRC in [srUnknownInputWord, srReturningChangeOnce]);
      ChangeButton.Enabled:= Variants.ItemIndex=0;
      Variants.Enabled:= Variants.ItemIndex=0;
    end;
  srRepeatWord:
    begin
      DelButton.Show;
      ChangeButton.Hide;
      ChangeAllButton.Enabled:= False;
      SkipButton.Show;
      SkipAllButton.Enabled:= False;
      AddButton.Enabled:= False;
    end;
  srWordConsideredAbbreviation,
  srMissingSpace,
  srInitialNumeral:
    begin
      DelButton.Hide;
      ChangeButton.Show;
      SkipButton.Enabled:= True;
      SkipAllButton.Enabled:= false;
      AddButton.Enabled:= False;
      ChangeButton.Enabled:= False;
      ChangeAllButton.Enabled:= False;
      Variants.Enabled:= False;
    end;
  end;
  CancelEdit.Hide;
end;

procedure TSpellerDialog2.CancelButtonClick(Sender: TObject);
begin
  Checker.FinishCheck;
end;

procedure TSpellerDialog2.DelButtonClick(Sender: TObject);
begin
  Checker.Delete;
  Checker.ContinueCheck;
end;

procedure TSpellerDialog2.SkipButtonClick(Sender: TObject);
begin
  Checker.ContinueCheck;
end;

procedure TSpellerDialog2.SkipAllButtonClick(Sender: TObject);
begin
  Checker.IgnoreAlways;
  Checker.ContinueCheck;
end;

procedure TSpellerDialog2.AddButtonClick(Sender: TObject);
begin
  Checker.Add;
  Checker.ContinueCheck;
end;

procedure TSpellerDialog2.ChangeButtonClick(Sender: TObject);
var
  SS, SL: Integer;
begin
  if CancelEdit.Visible then
    begin
      GetHotArea(SS, SL);
      with Misspelling do
        begin
          SelStart:= SS;
          SelLength:= SL;
          Checker.Change(SelText);
        end;
      Checker.ContinueCheck;
    end
  else
    begin
      if Checker.FSRC=srUnknownInputWord then
        Checker.ChangeOnce(Variants.Items[Variants.ItemIndex])
      else
         Checker.Change(Variants.Items[Variants.ItemIndex]);
     Checker.ContinueCheck;
    end;
end;

procedure TSpellerDialog2.ChangeAllButtonClick(Sender: TObject);
begin
  Checker.ChangeAlways(Variants.Items[Variants.ItemIndex]);
  Checker.ContinueCheck;
end;

procedure TSpellerDialog2.StartButtonClick(Sender: TObject);
var
  ChildForm, ActiveForm, ParentForm: TForm;
  MemoName: String;
  Cmpnt: TComponent;
begin
  StartButton.Hide;
  try
    if not (Checker.FMemo.Owner is TForm) then
      Abort;
    ChildForm:= Checker.FMemo.Owner as TForm;
    if (ChildForm.FormStyle<>fsMDIChild) then
      Abort;
    MemoName:= Checker.FMemo.Name;
    ParentForm:= ChildForm.Owner as TForm;
    ActiveForm:= ParentForm.ActiveMDIChild;
    Cmpnt:= ActiveForm.FindComponent(MemoName);
    if not Assigned(Cmpnt) or not (Cmpnt is TCustomMemo) then
      Abort;
  finally
    Checker.Check(Cmpnt as TCustomMemo);
  end;
end;

procedure TSpellerDialog2.FormDeactivate(Sender: TObject);
begin
  StartButton.Show;
  DelButton.Enabled:= False;
  ChangeButton.Enabled:= False;
  ChangeAllButton.Enabled:= False;
  SkipButton.Enabled:= False;
  SkipAllButton.Enabled:= False;
  AddButton.Enabled:= False;
  Misspelling.Enabled:= False;
  Variants.Enabled:= False;
  InfoMsg.Enabled:= False;
  Label2.Enabled:= False;
  Variants.Items.Clear;
  FShowing:= True;
  Misspelling.Text:= '';
  FShowing:= False;
end;

procedure TSpellerDialog2.FormDestroy(Sender: TObject);
begin
  Checker.FDialog:= nil;
end;



function GetSpellLanguages(Languages: TStrings; Option: TLangOption): Integer;
var
  Registry: TRegistry;
  N, I: Integer;
  LCType: Integer;
  Buf: array[0..255]of Char;
begin
  Registry:= TRegistry.Create;
  Registry.RootKey:= HKEY_LOCAL_MACHINE;
  Registry.OpenKey('\SOFTWARE\Microsoft\Shared Tools\Proofing Tools\Spelling', False);
  Registry.GetKeyNames(Languages);
  Registry.Free;
  case Option of
  loLocalized:
    LCType:= LOCALE_SLanguage;
  loEnglish:
    LCType:= LOCALE_SEngLanguage;
  loNative:
    LCType:= LOCALE_SNativeLangName;
  loAbbrev:
    LCType:= LOCALE_SAbbrevLangName;
  end;
  for I:= Languages.Count-1 downto 0 do
    begin
      N:= StrToIntDef(Languages[I], -1);
      if N=-1 then
        Languages.Delete(I)
      else
        begin
          Languages.Objects[I]:= Pointer(N);
          GetLocaleInfo(N, LCType, Buf, 255);
          Languages[I]:= Buf;
        end;
    end;
  Result:= Languages.Count;
end;


procedure Register;
begin
  RegisterComponents('BRI O&&MR', [TSpellChecker]);
  RegisterPropertyEditor(TypeInfo(TFileName), TSpellChecker, 'CustomDict', TDictFileProperty);
end;

initialization
  ControlTypes:= TStringList.Create;
  TSpellChecker.RegisterEditControl('TCustomMemo', False, False);
  TSpellChecker.RegisterEditControl('TCustomRichEdit', False, False);
  TSpellChecker.RegisterEditControl('TCustomRichEdit98', True, True);

finalization
  ControlTypes.Free;

end.
