unit MDIFormUnit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, Langs, ClipBrd, RichEdit, Menus, RichEdit2,
  GoToWeb, Menus97, anlRuler, ExtCtrls;

type
  TMDIForm = class(TForm)
    RichEdit: TRichEdit98;
    StatusBar: TStatusBar;
    VariantsMenu: TPopupMenu;
    GoToWeb: TGoToWeb;
    VariantsMenu97: TPopupMenus97;
    Ruler: TanlRuler;
    procedure RichEditChange(Sender: TObject);
    procedure RichEditSelectionChange(Sender: TObject);
    procedure VariantsMenuPopup(Sender: TObject);
    procedure SpellMIClick(Sender: TObject);
    procedure AddWordMIClick(Sender: TObject);
    procedure VariantMIClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormActivate(Sender: TObject);
    procedure RichEditURLClick(Sender: TObject; URL: String);
    function VariantsMenu97DrawItem(Control: TMenu; Item: TMenuItem;
      Rect: TRect; State: TOwnerDrawState): Boolean;
    procedure RichEditMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure RulerIndentChange(Sender: TObject; IndentID: TIndentID);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    FChanged: Boolean;
    FFileName: String;
    FIsNew: Boolean;
    OldColor,
    OldBackColor: TColor;
    CheckWord: String;
    CheckStart,
    CheckEnd: Integer;
    procedure SetChanged(Value: Boolean);
    procedure SetFileName(Value: String);
    property IsChanged: Boolean read FChanged write SetChanged;
    property FileName: String read FFileName write SetFileName;
    property IsNew: Boolean read FIsNew write FIsNew;
    procedure UpdateButtons;
  public
    { Public declarations }
    procedure ExecSave;
    procedure ExecSaveAs;
    procedure ExecLoad(FName: String);
    procedure ExecNew;
    procedure CheckForSave;
    procedure SelectAll;
    procedure Clear;
    procedure Cut;
    procedure Copy;
    procedure Paste;
    procedure Find(Sender: TObject);
    procedure Replace;
    procedure Undo;
    procedure Redo;
    procedure Print;
    procedure Spell;
    procedure SetFont(FontName: String);
    procedure SetBold(Bold: Boolean);
    procedure SetItalic(Italic: Boolean);
    procedure SetUnderline(Underline: Boolean);
    procedure SetColor(Color: TColor);
    procedure SetBackColor(Color: TColor);
    procedure SetFontSize(Size: Integer);
    procedure SetAlignment(Alignment: TAlignment98);
    procedure SetNumbering(Numbering: Boolean);
    procedure SetLanguage(Language: TLanguage);
  end;

var
  MDIForm: TMDIForm;

implementation

{$R *.DFM}

uses
  SpellDemoUnit;

const
  Counter: Integer = 0;

procedure TMDIForm.RichEditChange(Sender: TObject);
begin
  IsChanged:= True;
  UpdateButtons;
end;

procedure TMDIForm.RichEditSelectionChange(Sender: TObject);
begin
  UpdateButtons;
  RichEdit.Language:= RichEdit.SelAttributes.Language;
  with RichEdit.SelAttributes, SpellDemoForm do
    begin
      FontCombo.ItemIndex:= FontCombo.Items.IndexOf(RichEdit.SelAttributes.Name);
      FontSize.Text:= IntToStr(Size);
      BoldBtn.Down:= Bold;
      ItalicBtn.Down:= Italic;
      UndrlBtn.Down:= Boolean(UnderlineType);
      if OldColor<>RichEdit.SelAttributes.Color then
        begin
          SetColorButton(TextColorBtn, RichEdit.SelAttributes.Color);
          OldColor:= RichEdit.SelAttributes.Color;
        end;
      if OldBackColor<>BackColor then
        begin
          SetColorButton(BackColorBtn, BackColor);
          OldBackColor:= BackColor;
        end;
      ChLang:= False;
      Languages.Language:= Language;
      ChLang:= True;
    end;
  with RichEdit.Paragraph, SpellDemoForm do
    begin
      BulletsBtn.Down:= Numbering<>nsNone;
      LeftBtn.Down:= Alignment=taLeft;
      CenterBtn.Down:= Alignment=taCenter;
      RightBtn.Down:= Alignment=taRight;
    end;
  StatusBar.Panels[0].Text:= IntToStr(RichEdit.Line+1)+' : '+IntToStr(RichEdit.Col+1);
end;

procedure TMDIForm.SetChanged(Value: Boolean);
begin
  FChanged:= Value;
  if Value then
    StatusBar.Panels[1].Text:= 'Modified'
  else
    StatusBar.Panels[1].Text:= '';
end;

procedure TMDIForm.SetFileName(Value: String);
begin
  FFileName:= Value;
  Caption:= Value;
end;

procedure TMDIForm.UpdateButtons;
begin
  with SpellDemoForm do
    begin
      CopyBtn.Enabled:= RichEdit.SelLength>0;
      CutBtn.Enabled:= RichEdit.SelLength>0;
      FindBtn.Enabled:= RichEdit.Lines.Count>0;
      ReplaceBtn.Enabled:= RichEdit.Lines.Count>0;
      PasteBtn.Enabled:= Clipboard.HasFormat(CF_Text);
      UndoBtn.Enabled:= RichEdit.CanUndo;
      RedoBtn.Enabled:= RichEdit.CanRedo;
      CopyMI.Enabled:= CopyBtn.Enabled;
      CutMI.Enabled:= CutBtn.Enabled;
      FindMI.Enabled:= FindBtn.Enabled;
      ReplaceMI.Enabled:= ReplaceBtn.Enabled;
      PasteMI.Enabled:= PasteBtn.Enabled;
      UndoMI.Enabled:= UndoBtn.Enabled;
      RedoMI.Enabled:= RedoBtn.Enabled;
    end;
end;

procedure TMDIForm.ExecSave;
begin
  if IsNew then
    ExecSaveAs
  else
    begin
      RichEdit.Lines.SaveToFile(FileName);
      IsChanged:= False;
    end;
end;

procedure TMDIForm.ExecSaveAs;
begin
  SpellDemoForm.SaveDialog.FileName:= FileName;
  if SpellDemoForm.SaveDialog.Execute then
    begin
      FileName:= SpellDemoForm.SaveDialog.FileName;
      RichEdit.Lines.SaveToFile(FileName);
      IsNew:= False;
      IsChanged:= False;
    end;
end;

procedure TMDIForm.ExecLoad(FName: String);
begin
  FileName:= SpellDemoForm.OpenDialog.FileName;
  RichEdit.Lines.LoadFromFile(FileName);
  IsNew:= False;
  IsChanged:= False;
  RichEditSelectionChange(Self);
end;

procedure TMDIForm.ExecNew;
begin
end;

procedure TMDIForm.CheckForSave;
var
  Res: Integer;
begin
  if not IsChanged then
    Exit;
  Res:= Application.MessageBox(PChar(Format('File is %s modified.'^M'Save before closing?', [FileName])),
                   'Warning', mb_YesNoCancel+mb_IconQuestion);
  if Res=id_Cancel then
    Abort;
  if Res=id_Yes then
    ExecSave;
end;

procedure TMDIForm.SelectAll;
begin
  RichEdit.SelectAll;
  UpdateButtons;
end;

procedure TMDIForm.Clear;
begin
  RichEdit.Clear;
  UpdateButtons;
end;

procedure TMDIForm.Cut;
begin
  RichEdit.CutToClipboard;
  IsChanged:= True;
  UpdateButtons;
end;

procedure TMDIForm.Copy;
begin
  RichEdit.CopyToClipboard;
  UpdateButtons;
end;

procedure TMDIForm.Paste;
begin
  RichEdit.PasteFromClipboard;
  IsChanged:= True;
  UpdateButtons;
end;

procedure TMDIForm.Find(Sender: TObject);
var
  SearchStart: Integer;
  FS: TFindTextExW;
  Flags: Integer;
  W: WideString;
begin
  with TFindDialog(Sender) do
  begin
    if (frFindNext in Options) then
      SearchStart:= RichEdit.SelStart+RichEdit.SelLength
    else
      SearchStart:= RichEdit.SelStart;
    FS.chrg.cpMin:= SearchStart;
    FS.chrg.cpMax:= -1;
    W:= FindText;
    FS.lpstrText:= @W[1];
    if (frMatchCase in Options) then
      Flags:= FT_MATCHCASE
    else
      Flags:= 0;
    if (frWholeWord in Options) then
      Flags:= Flags or FT_WHOLEWORD;
    if (frDown in Options) then
      Flags:= Flags or FT_DOWNWARD;
    if (RichEdit.Perform(EM_FINDTEXTEX, Flags, Integer(@FS))>0) then
      begin
        RichEdit.SelStart := FS.chrgText.cpMin;
        RichEdit.SelLength := FS.chrgText.cpMax - FS.chrgText.cpMin;
        UpdateButtons;
      end
    else
      MessageBox(Handle, PChar(Format('Text %s not found', [FindText])), 'Message',
                 mb_Ok+mb_IconInformation);
  end;
end;

procedure TMDIForm.Replace;
var
  SearchStart: Integer;
  ReplCount: Integer;
  FS: TFindTextExW;
  Flags: Integer;
  W: wideString;
begin
  with SpellDemoForm.ReplaceDialog do
  begin
    if (frReplaceAll in Options) then
      begin
        ReplCount:= 0;
        FS.chrg.cpMin:= RichEdit.SelStart;
        FS.chrg.cpMax:= -1;
        W:= FindText;
        FS.lpstrText:= @W[1];
        if (frMatchCase in Options) then
          Flags:= FT_MATCHCASE
        else
          Flags:= 0;
        if (frWholeWord in Options) then
          Flags:= Flags or FT_WHOLEWORD;
        if (frDown in Options) then
          Flags:= Flags or FT_DOWNWARD;
        while (RichEdit.Perform(EM_FINDTEXTEX, Flags, Integer(@FS))>-1) do
          begin
            RichEdit.SelStart := FS.chrgText.cpMin;
            RichEdit.SelLength := FS.chrgText.cpMax - FS.chrgText.cpMin;
            RichEdit.SelText:= ReplaceText;
            UpdateButtons;
            Inc(ReplCount);
            FS.chrg.cpMin:= RichEdit.SelStart+Length(ReplaceText);
          end;
        MessageBox(Handle, PChar(Format('%d replaces were made', [ReplCount])), 'Message',
                   mb_Ok+mb_IconInformation);
      end
    else
      begin
        if (frFindNext in Options) then
          SearchStart:= RichEdit.SelStart+RichEdit.SelLength
        else
          SearchStart:= RichEdit.SelStart;
        FS.chrg.cpMin:= SearchStart;
        FS.chrg.cpMax:= -1;
        W:= FindText;
        FS.lpstrText:= @W[1];
        if (frMatchCase in Options) then
          Flags:= FT_MATCHCASE
        else
          Flags:= 0;
        if (frWholeWord in Options) then
          Flags:= Flags or FT_WHOLEWORD;
        if (frDown in Options) then
          Flags:= Flags or FT_DOWNWARD;
        if (RichEdit.Perform(EM_FINDTEXTEX, Flags, Integer(@FS))>-1) then
          begin
            RichEdit.SelStart := FS.chrgText.cpMin;
            RichEdit.SelLength := FS.chrgText.cpMax - FS.chrgText.cpMin;
            RichEdit.SelText:= ReplaceText;
            UpdateButtons;
          end
        else
          MessageBox(Handle, PChar(Format('Text %s not found', [FindText])), 'Message',
                     mb_Ok+mb_IconInformation);
      end;
  end;
end;

procedure TMDIForm.Undo;
begin
  RichEdit.Undo;
  UpdateButtons;
end;

procedure TMDIForm.Redo;
begin
  RichEdit.Redo;
  UpdateButtons;
end;

procedure TMDIForm.Print;
begin
  RichEdit.Print(Caption);
end;

procedure TMDIForm.SetFont(FontName: String);
begin
  RichEdit.SelAttributes.Name:= FontName;
end;

procedure TMDIForm.SetBold(Bold: Boolean);
begin
  RichEdit.SelAttributes.Bold:= Bold;
end;

procedure TMDIForm.SetItalic(Italic: Boolean);
begin
  RichEdit.SelAttributes.Italic:= Italic;
end;

procedure TMDIForm.SetUnderline(Underline: Boolean);
begin
  RichEdit.SelAttributes.UnderlineType:= TUnderlineType(Underline);
end;

procedure TMDIForm.SetColor(Color: TColor);
begin
  RichEdit.SelAttributes.Color:= Color;
end;

procedure TMDIForm.SetBackColor(Color: TColor);
begin
  RichEdit.SelAttributes.BackColor:= Color;
end;

procedure TMDIForm.SetFontSize(Size: Integer);
begin
  RichEdit.SelAttributes.Size:= Size;
end;

procedure TMDIForm.SetAlignment(Alignment: TAlignment98);
begin
  RichEdit.Paragraph.Alignment:= Alignment;
end;

procedure TMDIForm.SetNumbering(Numbering: Boolean);
begin
  if Numbering then
    begin
      RichEdit.Paragraph.Numbering := nsBullet;
      RichEdit.Paragraph.NumberingTab := RichEdit.SelAttributes.Size;
    end
  else
    RichEdit.Paragraph.Numbering := nsNone;
end;

procedure TMDIForm.SetLanguage(Language: TLanguage);
begin
  RichEdit.SelAttributes.Language:= Language;
end;

procedure TMDIForm.VariantsMenuPopup(Sender: TObject);
var
  I: Integer;
  Variants: TStringList;
  MI: TMenuItem;
begin
  for I:= VariantsMenu.Items.Count-1 downto 0 do
    VariantsMenu.Items.Delete(I);
// Checks selected word with current language
// If word is unknown then populates VariantsMenu with alternatives
  if not SpellDemoForm.Speller.IsKnownWord(CheckWord, RichEdit.Language) then
    begin
      MI:= TMenuItem.Create(VariantsMenu);
      MI.Caption:= 'Spelling';
      MI.ShortCut:= TextToShortCut('F7');
      MI.OnClick:= SpellMIClick;
      MI.Tag:= 8;
      VariantsMenu.Items.Add(MI);
      MI:= TMenuItem.Create(VariantsMenu);
      MI.Caption:= 'Add word';
      MI.OnClick:= AddWordMIClick;
      MI.Tag:= -2;
      VariantsMenu.Items.Add(MI);
      Variants:= TStringList.Create;
//Get list of alternatives into  Variants
      SpellDemoForm.Speller.GetVariants(CheckWord, Variants, RichEdit.Language);
      if Variants.Count>0 then
        begin
          MI:= TMenuItem.Create(VariantsMenu);
          MI.Caption:= '-';
          VariantsMenu.Items.Add(MI);
        end;
      for I:= 0 to Variants.Count-1 do
        begin
          MI:= TMenuItem.Create(VariantsMenu);
          MI.Caption:= Variants[I];
          MI.OnClick:= VariantMIClick;
          MI.Tag:= -2;
          MI.Default:= True;
          VariantsMenu.Items.Add(MI);
        end;
      Variants.Free;
    end;
end;

procedure TMDIForm.SpellMIClick(Sender: TObject);
begin
// Executes interactive spell checking
  SpellDemoForm.Speller.Check(RichEdit);
end;

procedure TMDIForm.AddWordMIClick(Sender: TObject);
begin
// Executes interactive spell checking
  SpellDemoForm.Speller.AddWord(CheckWord, RichEdit.Language);
end;

procedure TMDIForm.VariantMIClick(Sender: TObject);
begin
// Replaces selected word with alterntive from popup menu
  RichEdit.SelStart:= CheckStart;
  RichEdit.SelLength:= CheckEnd-CheckStart;
  RichEdit.SelText:= (Sender as TMenuItem).Caption;
end;

procedure TMDIForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  CheckForSave;
end;

procedure TMDIForm.FormCreate(Sender: TObject);
var
  DC: HDC;
  Divider: Double;
begin
  Inc(Counter);
  FileName:= 'Document'+IntToStr(Counter)+'.rtf';
  RichEdit.Clear;
  RichEdit.SelAttributes.BackColor:= clNone;
  RichEditSelectionChange(Self);
  IsNew:= True;
  IsChanged:= False;
  SpellDemoForm.FontCombo.ItemIndex:=
     SpellDemoForm.FontCombo.Items.IndexOf('Arial');
  Ruler.MarginRight:= GetSystemMetrics(SM_CXVSCROLL);
  FormResize(Self);
  DC := GetDC(0);
  Ruler.TabDist := Round(GetDeviceCaps(DC, LOGPIXELSX)/10.16);
  ReleaseDC(0, DC);
end;

procedure TMDIForm.Spell;
begin
// Executes interactive spell checking
  SpellDemoForm.Speller.Check(RichEdit);
end;

procedure TMDIForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Action:= caFree;
end;

procedure TMDIForm.FormActivate(Sender: TObject);
begin
  UpdateButtons;
  OldColor:= -1;
  OldBackColor:= -1;
  RichEditSelectionChange(Self);
end;

procedure TMDIForm.RichEditURLClick(Sender: TObject; URL: String);
begin
  GotoWeb.URL:= URL;
  GotoWeb.Execute;
end;

function TMDIForm.VariantsMenu97DrawItem(Control: TMenu; Item: TMenuItem;
  Rect: TRect; State: TOwnerDrawState): Boolean;
var
  W: WideString;
begin
  Result:= False;
  if Control.Items.IndexOf(Item)<3 then
    Exit;
  with VariantsMenu97.Canvas do
    begin
      FillRect(Rect);
      Font.Style:= Font.Style +[fsBold];
      W:= CharToWide(Item.Caption, CodePageFromLocale(RichEdit.Language));
      TextOutW(Handle, Rect.Left+22, Rect.Top+2, @W[1], Length(W));
    end;
  Result:= True;
end;

procedure TMDIForm.RichEditMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  P: Integer;
  TR: RichEdit2.TTextRangeA;
  CS: Integer;
begin
  if Button=mbRight then
    begin
      RichEdit.Perform(WM_LBUTTONDOWN, MK_LBUTTON, MakeLParam(X, Y));
      P:= RichEdit.SelStart;
      CheckStart:= RichEdit.Perform(EM_FINDWORDBREAK, WB_MOVEWORDLEFT, P);
      CS:= RichEdit.Perform(EM_FINDWORDBREAK, WB_LEFT, P);
      if CS>CheckStart then
        CheckStart:= CS;
      CheckEnd:= RichEdit.Perform(EM_FINDWORDBREAK, WB_RIGHTBREAK, P);
      if Abs(CheckEnd-CheckStart)<2 then
        Exit;
      SetLength(CheckWord, CheckEnd-CheckStart);
      TR.chrg.cpMin:= CheckStart;
      TR.chrg.cpMax:= CheckEnd;
      TR.lpstrText:= @CheckWord[1];
      RichEdit.Perform(EM_GETTEXTRANGE, 0, LongInt(@TR));
      while CheckWord[CheckEnd-CheckStart]<#33 do
        Dec(CheckEnd);
      SetLength(CheckWord, CheckEnd-CheckStart);
      with RichEdit.ClientToScreen(Point(X, Y)) do
        VariantsMenu.Popup(X, Y);
    end;
end;

procedure TMDIForm.RulerIndentChange(Sender: TObject; IndentID: TIndentID);
var
  DC: HDC;
  Divider: Double;
begin
  DC := GetDC(0);
  Divider := GetDeviceCaps(DC, LOGPIXELSX)/72;
  ReleaseDC(0, DC);
  case IndentId of
  riFirst:
    begin
      RichEdit.Paragraph.FirstIndent:= (Ruler.IndentFirst-Ruler.MarginLeft)/Divider+4;
      RichEdit.Paragraph.LeftIndent:= (Ruler.IndentLeft-Ruler.IndentFirst)/Divider;
    end;
  riLeft:
    RichEdit.Paragraph.LeftIndent:= (Ruler.IndentLeft-Ruler.IndentFirst)/Divider;
  riRight:
    RichEdit.Paragraph.RightIndent:= (Ruler.Width - Ruler.MarginRight - Ruler.IndentRight-4)/Divider;
  riBothLeft:
    RichEdit.Paragraph.FirstIndent:= (Ruler.IndentFirst-Ruler.MarginLeft)/Divider+4;
  end;
end;

procedure TMDIForm.FormResize(Sender: TObject);
var
  DC: HDC;
  Divider: Double;
begin
  DC := GetDC(0);
  Divider := GetDeviceCaps(DC, LOGPIXELSX)/72;
  ReleaseDC(0, DC);
  Ruler.IndentRight:= Round(Ruler.Width-Ruler.MarginRight-RichEdit.Paragraph.RightIndent*Divider-4);
end;

end.
