{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}

{*******************************************************}
{ Special thanks to Tibor F. Liska for TExcel component }
{ This one was the base for my DDE components and best  }
{             example of DDE programming                }
{*******************************************************}

unit SohoDDE;

{$I SOHOLIB.INC}

interface

uses Forms, Classes, DDEMan, SysUtils, IniFiles, Graphics;

type

  { DDE-}
  TsohoCustomDDE = class(TComponent)
  private
    FDDE              : TDdeClientConv;
    FExeName          : TFileName;
    FMacro            : TFileName;
    FConnected        : Boolean;
    FOnClose          : TNotifyEvent;
    FOnOpen           : TNotifyEvent;
    procedure SetExeName(const Value: TFileName);
    procedure SetConnect(const Value: Boolean);
  protected
    { }
    function  ServerName              : string;virtual;
    {    }
    function  OpenErrorMessage        : string;virtual;
    { ,    }
    function  CommandNotAcceptMessage : string;virtual;
    { ,    }
    function  MacrosIsNillMessage     : string;virtual;
    {  }
    property  Connected : Boolean        read FConnected write SetConnect;
    { DDE,  -  }
    property  DDE       : TDdeCLientConv read FDDE;
    { -}
    property  ExeName   : TFileName      read FExeName   write SetExeName;
    {    }
    property  OnClose   : TNotifyEvent   read FOnClose   write FOnClose;
    {    }
    property  OnOpen    : TNotifyEvent   read FOnOpen    write FOnOpen;
    { -   ""   Embedding win.ini}
    function  FindExeByEmbedding (const Embedding, ServerName : string) : string;
  public
    procedure OpenLink(Sender: TObject);
    procedure ShutDown(Sender: TObject);
    {  }
    procedure Connect;
    {  }
    procedure Disconnect;
    {   -}
    procedure OpenMacroFile (Fn : TFileName);
    {   -}
    procedure CloseMacroFile;
    {    }
    procedure StartDocument;virtual;
    { }
    procedure EndDocument;virtual;
    {   }
    function  Request (Item : string): string;
    { }
    procedure Exec    (Cmd  : string);
    {  }
    procedure RealExec(Cmd  : PChar);
    {   }
    procedure Run     (Mn   : string);
    constructor Create(AOwner: TComponent); override;
  end;

  {    Excel}
  TExcelColorIndex     = 0..56;
  TExcelColor          = TColor;
  TExcelVertAlignment  = (vaTop, vaCenter, vaBottom);
  TExcelHorAlignment   = (gaLeft, gaCenter, gaRight);
  TExcelOrientation    = (xoHorizontal, xoVertical);
  TExcelRangeBorders   = (rbNone, rbContinous, rbDash, rbDouble, rbDot);
  TExcelGraphicBorders = (gbNone, gbContinous, gbDash, gbDot, gbDashDot,
                          gbDashDotDot, gbGray25, gbGray50, gbGray75,gbAutomatic);
  TExcelBorderWidth    = (bwHairline, bwThin, bwMedium, brThick);

  {   Excel}
  TsohoExcel = class(TsohoCustomDDE)
  protected
    function  ServerName              : string;override;
    function  OpenErrorMessage        : string;override;
    function  CommandNotAcceptMessage : string;override;
    function  MacrosIsNillMessage     : string;override;
  public
    procedure Insert  (s    : string);
    procedure Select  (Row, Col : Integer);
    procedure PutExt  (Row, Col : Integer; e : Extended);
    procedure PutStr  (Row, Col : Integer; s : string);
    procedure PutInt  (Row, Col : Integer; i : Longint);
    procedure PutDay  (Row, Col : Integer; d : TDateTime);
    procedure Range   (Row1,Col1,Row2,Col2 : integer);
    procedure SetFont (FontName : string; FontSize : integer;
                       FontColor : TColor; FontStyle : TFontStyles);
    property  Connected;
    property  DDE;
    procedure StartDocument;override;
    procedure EndDocument;override;
    procedure OpenFile (const FileName : TFileName);
    procedure Replace  (const FromStr, ToStr : string);
    constructor Create(AOwner: TComponent); override;
  published
    property  ExeName;
    property  OnClose;
    property  OnOpen;
  end;

  TWordAlignment = (waLeft, waRight, waCenter, waJustify);
  TWordFontColor = 0..16;

  TsohoWord = class(TsohoCustomDDE)
  protected
    function  ServerName              : string;override;
    function  OpenErrorMessage        : string;override;
    function  CommandNotAcceptMessage : string;override;
    function  MacrosIsNillMessage     : string;override;
  public
    procedure Insert     (s    : string);
    procedure InsertPara;
    procedure MoveUp       (Value : LongInt);
    procedure MoveDown     (Value : LongInt);
    procedure MoveLeft     (Value : LongInt);
    procedure MoveRight    (Value : LongInt);
    procedure SelectLeft   (Value : LongInt);
    procedure SelectRight  (Value : LongInt);
    procedure SelectUp     (Value : LongInt);
    procedure SelectDown   (Value : LongInt);
    procedure SelAlignment (Value : TWordAlignment);
    procedure SetFont      (FontName : string; FontSize : integer;
                            FontColor : TWordFontColor; FontStyle : TFontStyles);
    procedure SetStyle     (StyleName : string);
    procedure StartOfLine;
    procedure EndOfLine;
    procedure StartOfDocument;
    procedure EndOfDocument;
    property  Connected;
    property  DDE;
    procedure StartDocument;override;
    procedure EndDocument;override;
    procedure OpenFile (const FileName : TFileName);
    procedure Replace  (const FromStr, ToStr : string);
    procedure CloseAll (WithSave : boolean);
    constructor Create(AOwner: TComponent); override;
  published
    property  ExeName;
    property  OnClose;
    property  OnOpen;
  end;

function ColorToWordColor (Value : TColor) : TWordFontColor;
function WordColorToColor (Value : TWordFontColor) : TColor;

implementation
uses SoUtils, Dialogs, SoDate;

{$IFDEF WIN32}
  {$R sohoDDE.R32}
{$ELSE}
  {$R sohoDDE.R16}
{$ENDIF}

const
{sohoard DDE messages}
wdde_ExcelOpenError         = 12500; {"   Excel"}
wdde_ExcelCommandNotAccept  = 12501; {"Excel  \377 "}
wdde_ExcelNoMacro           = 12502; {" Excel  "}
wdde_WordOpenError          = 12503; {"   Word"}
wdde_WordCommandNotAccept   = 12504; {"Word  \377 "}
wdde_WordNoMacro            = 12505; {" Word  "}

{---------------------TsohoardCustomDDE-------------------------}
function  TsohoCustomDDE.ServerName : string;
begin
     Result := 'Server';
end;

function  TsohoCustomDDE.OpenErrorMessage        : string;
begin
     Result := 'Server Open Error';
end;

function  TsohoCustomDDE.CommandNotAcceptMessage : string;
begin
     Result:='Server not accepted command:';
end;

function  TsohoCustomDDE.MacrosIsNillMessage     : string;
begin
     Result:='Macro is nill';
end;

function  TsohoCustomDDE.FindExeByEmbedding (const Embedding, ServerName : string) : string;
var Ini        : TIniFile;
    ObjectName : string;
    OpenD      : TOpenDialog;
begin
     Result := '';
     Ini := TIniFile.Create('win.ini');
     ObjectName := Ini.ReadString('Embedding',Embedding,'');
     if ObjectName='' then begin
         ErrorMsg('OLE   : '+Embedding+#13+
                  ' -: '+ServerName);
         OpenD := TOpenDialog.Create(self);
         with OpenD do begin
            Filter := 'Executable (*.EXE)|*.EXE';
            if Execute then begin
               Result := FileName;
               Ini.WriteString('Embedding',Embedding,'empty,empty,'+Result+',document');
            end;
            Free;
         end;
     end
     else begin
        SetDivisers([',']);
        Result := GetSomeWords(ObjectName,WordsInString(ObjectName)-1,
                WordsInString(ObjectName)-1);
     end;
     Ini.Free;
end;

procedure TsohoCustomDDE.SetExeName(const Value: TFileName);
var i : Integer;
begin
     i := Pos('.', Value);
     if i = 0 then FExeName := Value
     else FExeName := Copy(Value, 1, i-1);
     { DDE  design-time!}
     if not (csDesigning in ComponentState) then
     begin
          Disconnect;
          FDDE.ServiceApplication := FExeName;
     end;
end;

procedure TsohoCustomDDE.SetConnect(const Value: Boolean);
begin
     if FConnected = Value then Exit;
     if Value then Connect else Disconnect;
end;

procedure TsohoCustomDDE.OpenLink(Sender: TObject);
begin
     FConnected := True;
     if Assigned(FOnOpen) then FOnOpen(Self);
end;

procedure TsohoCustomDDE.ShutDown(Sender: TObject);
begin
     FConnected := False;
     FMacro := '';
     if Assigned(FOnClose) then FOnClose(Self);
end;

constructor TsohoCustomDDE.Create;
begin
     inherited Create(AOwner);
     { DDE  design-time!}
     if not (csDesigning in ComponentState) then
     begin
          FDDE             := TDDEClientConv.Create(AOwner);
          FDDE.ConnectMode := ddeManual;
          FDDE.SetLink (ServerName, 'System');
          FDDE.OnOpen      := OpenLink;
          FDDE.OnClose     := ShutDown;
     end;
end;

procedure TsohoCustomDDE.Connect;
begin
     if FConnected or FDDE.OpenLink then Exit;
     raise Exception.Create(OpenErrorMessage);
end;

procedure TsohoCustomDDE.Disconnect;
begin
     if FConnected then FDDE.CloseLink;
end;

procedure TsohoCustomDDE.OpenMacroFile(Fn: TFileName);
begin
     if FMacro = ExtractFileName(Fn) then Exit;
     if FMacro <> '' then CloseMacroFile;
     Exec('[OPEN("' + Fn + '")]');
     Exec('[HIDE()]');
     FMacro := ExtractFileName(Fn);
end;

procedure TsohoCustomDDE.CloseMacroFile;
begin
     if FMacro = '' then Exit;
     Exec('[UNHIDE("' + FMacro + '")]');
     Exec('[CLOSE(FALSE)]');
     FMacro := '';
end;

procedure TsohoCustomDDE.StartDocument;
begin
     {abstract}
end;

procedure TsohoCustomDDE.EndDocument;
begin
     {abstract}
end;

function TsohoCustomDDE.Request(Item: string): string;
var Reply : PChar;
begin
     Reply  := FDDE.RequestData(Item);
     Result := StrPas(Reply);
     StrDispose(Reply);
end;

procedure TsohoCustomDDE.RealExec(Cmd  : PChar);
var i : integer;
begin
     if FDDE.ExecuteMacro(Cmd, False) then Exit;
     for i:=0 to 3 do
       Application.ProcessMessages;   { Waiting for server }
     if FDDE.ExecuteMacro(Cmd, True) then Exit;
     raise Exception.Create(CommandNotAcceptMessage+' '+StrPas(Cmd));
end;

procedure TsohoCustomDDE.Exec (Cmd: string);
var a : array[0..255] of Char;
begin
     StrPCopy(a, Cmd);
     RealExec(a);
end;

procedure TsohoCustomDDE.Run(Mn: string);
begin
     if FMacro = '' then raise Exception.Create(MacrosIsNillMessage);
     Exec('[RUN("' + FMacro + '!' + Mn + '";FALSE)]');
end;

{-----------------------TsohoExcel------------------------}
constructor TsohoExcel.Create;
begin
     inherited Create(AOwner);
     SetExeName(FindExeByEmbedding('Excel.Sheet.5','Ms Excel'));
end;

function  TsohoExcel.ServerName;
begin
     Result:='Excel';
end;

procedure TsohoExcel.StartDocument;
begin
     Exec('[APP.MINIMIZE()]');
     Exec('[NEW(1)]');
end;

procedure TsohoExcel.EndDocument;
begin
     Exec('[APP.MAXIMIZE()]');
end;

function  TsohoExcel.OpenErrorMessage;
begin
     Result:=LoadStr(wdde_ExcelOpenError);
end;

function  TsohoExcel.CommandNotAcceptMessage;
begin
     Result:=LoadStr(wdde_ExcelCommandNotAccept);
end;

function  TsohoExcel.MacrosIsNillMessage;
begin
     Result:=LoadStr(wdde_ExcelNoMacro);
end;

procedure TsohoExcel.Insert (s : string);
var i : Integer;
begin
     i := Pos('"', s);
     while i > 0 do
     begin
       s[i] := '''';             { Filter out " }
       i := Pos('"', s);
     end;
     Exec('[FORMULA("'+s+'")]');
end;

procedure TsohoExcel.Select(Row, Col: Integer);
var Fmt : string;
begin
     Fmt := '[SELECT("R%dC%d")]';
     Exec(Format(Fmt, [Row, Col]));
end;

procedure TsohoExcel.Range;
var Fmt : string;
begin
     Fmt := '[SELECT("R%dC%d:R%dC%d")]';
     Exec(Format(Fmt, [Row1, Col1, Row2, Col2]));
end;

procedure TsohoExcel.SetFont (FontName : string; FontSize : integer;
                       FontColor : TExcelColor; FontStyle : TFontStyles);
var Command : string;
begin
     Command:='Selection.Font .Font="'+FontName+'" .Size='+IntToStr(FontSize)+
              ' .Strikethrough='+BoolToStr(fsStrikeOut in FontStyle);
end;


procedure TsohoExcel.PutExt(Row, Col: Integer; e: Extended);
var  Cmd : string;
     i : Integer;
begin
     Select(Row, Col);
     Str(e:0:2, Cmd);
     i := Pos('.', Cmd);
     if i > 0 then Cmd[i] := DecimalSeparator;
     Insert(Cmd);
end;

procedure TsohoExcel.PutStr(Row, Col: Integer; s: string);
begin
     Select(Row, Col);
     Insert(s);
end;

procedure TsohoExcel.PutInt(Row, Col: Integer; i: Longint);
begin
     PutStr(Row, Col, IntToStr(i));
end;

procedure TsohoExcel.PutDay(Row, Col: Integer; d: TDateTime);
begin
     PutStr(Row, Col, ''''+WDateToStr(d));
end;

procedure TsohoExcel.OpenFile (const FileName : TFileName);
begin
     Exec('[OPEN ("'+FileName+'",0)]');
end;

procedure TsohoExcel.Replace  (const FromStr, ToStr : string);
begin
     Exec('[Cells.Replace What:="'+FromStr+'", Replacement:="'+ToStr+
          ', LookAt:= _xlPart, SearchOrder:=xlByRows, MatchCase:=False]');
end;

{------------------------TsohoWord-------------------------}
function ColorToWordColor (Value : TColor) : TWordFontColor;
begin
     Result:=0;
     if Value=clBlack   then Result:=1;
     if Value=clBlue    then Result:=2;
     if Value=clAqua    then Result:=3;
     if Value=clLime    then Result:=4;
     if Value=clFuchsia then Result:=5;
     if Value=clRed     then Result:=6;
     if Value=clYellow  then Result:=7;
     if Value=clWhite   then Result:=8;
     if Value=clNavy    then Result:=9;
     if Value=clTeal    then Result:=10;
     if Value=clGreen   then Result:=11;
     if Value=clPurple  then Result:=12;
     if Value=clMaroon  then Result:=13;
     if Value=clOlive   then Result:=14;
     if Value=clGray    then Result:=15;
     if Value=clSilver  then Result:=16;
end;

function WordColorToColor (Value : TWordFontColor) : TColor;
begin
     Result:=clBlack;
     if Value=1 then Result:=clBlack;
     if Value=2 then Result:=clBlue;
     if Value=3 then Result:=clAqua;
     if Value=4 then Result:=clLime;
     if Value=5 then Result:=clFuchsia;
     if Value=6 then Result:=clRed;
     if Value=7 then Result:=clYellow;
     if Value=8 then Result:=clWhite;
     if Value=9 then Result:=clNavy;
     if Value=11 then Result:=clTeal;
     if Value=12 then Result:=clGreen;
     if Value=13 then Result:=clPurple;
     if Value=14 then Result:=clMaroon;
     if Value=15 then Result:=clOlive;
     if Value=16 then Result:=clGray;
     if Value=17 then Result:=clSilver;
end;

constructor TsohoWord.Create;
begin
     inherited Create(AOwner);
     SetExeName(FindExeByEmbedding('Word.Document.6','Ms Word'));
end;

function  TsohoWord.ServerName;
begin
     Result:='WinWord';
end;

procedure TsohoWord.StartDocument;
begin
     Exec('[AppMinimize]');
     Exec('[FileNewDefault]');
end;

procedure TsohoWord.EndDocument;
begin
     Exec('[APPMAXIMIZE]');
end;

function  TsohoWord.OpenErrorMessage;
begin
     Result:=LoadStr(wdde_WordOpenError);
end;

function  TsohoWord.CommandNotAcceptMessage;
begin
     Result:=LoadStr(wdde_WordCommandNotAccept);
end;

function  TsohoWord.MacrosIsNillMessage;
begin
     Result:=LoadStr(wdde_WordNoMacro);
end;

procedure TsohoWord.Insert;
begin
     { ,    }
     S:=ChangeChars(S,'"',#39);
     Exec('[INSERT "'+S+'"]');
end;

procedure TsohoWord.InsertPara;
begin Exec('[INSERTPARA]'); end;

procedure TsohoWord.MoveUp       (Value : LongInt);
begin Exec('[LineUp '+IntToStr(Value)+']');end;

procedure TsohoWord.MoveDown     (Value : LongInt);
begin Exec('[LineDown '+IntToStr(Value)+']');end;

procedure TsohoWord.MoveLeft     (Value : LongInt);
begin Exec('[CharLeft '+IntToStr(Value)+']');end;

procedure TsohoWord.MoveRight    (Value : LongInt);
begin Exec('[CharRight '+IntToStr(Value)+']');end;

procedure TsohoWord.SelectLeft   (Value : LongInt);
begin Exec('[CharLeft '+IntToStr(Value)+', 1]');
end;

procedure TsohoWord.SelectRight  (Value : LongInt);
begin Exec('[CharRight '+IntToStr(Value)+', 1]');end;

procedure TsohoWord.SelectUp     (Value : LongInt);
begin Exec('[LineUp '+IntToStr(Value)+', 1]');end;

procedure TsohoWord.SelectDown   (Value : LongInt);
begin Exec('[LineDown '+IntToStr(Value)+', 1]');end;

procedure TsohoWord.SelAlignment (Value : TWordAlignment);
const  Align : array [TWordAlignment] of string = ('LeftPara','RightPara',
       'CenterPara','JustifyPara');
begin Exec('['+Align[Value]+']');end;

procedure TsohoWord.SetFont;
var Command     : PChar;
    Part1,Part2 : string;
    Flag    : byte;
begin
     Part1:='[FormatFont .Points="'+IntToStr(FontSize)+'", ';
     Flag:=byte(fsUnderLine in FontStyle);
     Part1:=Part1+'.Underline='+IntToStr(Flag)+', ';
     Part1:=Part1+'.Color='+IntToStr(FontColor)+', ';
     Flag:=byte(fsStrikeOut in FontStyle);
     Part1:=Part1+'.Strikethrough='+IntToStr(Flag)+', ';
     Part1:=Part1+'.Superscript = 0, .Subscript = 0, .Hidden = 0, .SmallCaps = 0, '+#0;

     Part2:='.AllCaps = 0, .Spacing = "0 ", .Position = "0 ", .Kerning = 0, ';
     Part2:=Part2+'.KerningMin = "", .Tab = "0", ';
     Part2:=Part2+'.Font="'+FontName+'", ';
     Flag:=byte(fsBold in FontStyle);
     Part2:=Part2+'.Bold='+IntToStr(Flag)+', ';
     Flag:=byte(fsItalic in FontStyle);
     Part2:=Part2+'.Italic='+IntToStr(Flag)+', .Outline = 0, .Shadow = 0]'+#0;
     GetMem(Command,length(Part1)+length(Part2)+1);
     StrCat(Command,@Part1[1]);
     StrCat(Command,@Part2[1]);
     RealExec(Command);
     FreeMem(Command,length(Part1)+length(Part2)+1);
end;

procedure TsohoWord.StartOfLine;
begin Exec('[StartOfLine]');end;

procedure TsohoWord.EndOfLine;
begin Exec('[EndOfLine]');end;

procedure TsohoWord.StartOfDocument;
begin Exec('[StartOfDocument]');end;

procedure TsohoWord.EndOfDocument;
begin Exec('[EndOfDocument]');end;

procedure TsohoWord.SetStyle;
begin Exec('[Style "'+StyleName+'"]');end;

procedure TsohoWord.CloseAll (WithSave : boolean);
begin
    if not WithSave then Exec('[FileCloseAll 2]')
    else Exec('[FileCloseAll 1]');
end;

procedure TsohoWord.OpenFile (const FileName : TFileName);
begin
    Exec(
    '[FileOpen .Name = "'+FileName+'", .ConfirmConversions = 0, .ReadOnly = 0,'+
    '.AddToMru = 0, .PasswordDoc = "", .PasswordDot = "", .Revert = 0,'+
    '.WritePasswordDoc = "", .WritePasswordDot = ""]');
end;

procedure TsohoWord.Replace  (const FromStr, ToStr : string);
begin
     Exec('[EditReplace .Find = "'+FromStr+'", .Replace = "'+ToStr+'", .Direction = 0,'+
          ' .MatchCase = 0, .WholeWord = 0, .PatternMatch = 0, .SoundsLike = 0,'+
          ' .ReplaceAll, .Format = 0, .Wrap = 1, .FindAllWordForms = 0]');
end;

end.
