{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
    TsohoTextButton, TsohoCustomButton,
 TsohoTransButton  TsohoBitBtn
}

unit SohoBtns;

{$I SOHOLIB.INC}
{$G-}

interface

uses WinTypes, WinProcs, Messages, Classes, Controls, Forms, Graphics,
  StdCtrls, ExtCtrls, Buttons, SoTools, Menus, SoCtrls;

type
  
  TsohoButtonKind = (sbOk, sbCancel, sbYes, sbNo, sbClose, sbCustom);
  
  TsohoTextButton = class(TsohoCustomLabel)
  private
    FBottomColor: TColor;
    FHighlightColor: TColor;
    FOldLight: TLightPos;
    FGlyph: TBitmap;
    FKind: TsohoButtonKind;
    FModalResult: TModalResult;
    FEnabled: boolean;
    FIsClicked: boolean;
    FLayout: TButtonLayout;
    FSound: TsohoSound;
    FSoundName: TWaveFileName;
    procedure SetLayout(Value: TButtonLayout);
    procedure SetGlyph(Value: TBitmap);
    procedure SetEnabled(Value: boolean);
    procedure SetKind(Value: TsohoButtonKind);
    procedure SetSoundName(Value: TWaveFileName);
    procedure SetHighlightColor(Value: TColor);
    procedure SetBottomColor(Value: TColor);
    procedure GlyphChanged(Sender: TObject);
  protected
    function GetPalette: HPALETTE; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    procedure Click; override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Paint; override;
  published
    property HighlightColor: TColor read FHighlightColor write SetHighlightColor default clWhite;
    property BottomColor: TColor read FBottomColor write SetBottomColor default clGray;
    property Sound: TsohoSound read FSound write FSound;
    property ClickSound: TWaveFileName read FSoundName write SetSoundName;
    property ModalResult: TModalResult read FModalResult write FModalResult default mrNone;
    property Kind: TsohoButtonKind read FKind write SetKind default sbCustom;
    property Enabled: boolean read FEnabled write SetEnabled default True;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property Glyph: TBitmap read FGlyph write SetGlyph;
    property Alignment;
    property Bevel;
    property BevelInner;
    property BevelOuter;
    property BorderWidth;
    property BevelWidth;
    property LightColor;
    property ShadowColor;
    property Light;
    property Height3D;
    property Align;
    property Caption;
    property Color;
    property Cursor default crHandPoint;
    property DragCursor;
    property DragMode;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Transparent;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TsohoCustomButton = class(TBUTTON)
  private
    { Private Declarations }
    FInRect: boolean;
    FFlat: boolean;
    FColor: TColor;
    FGlyph: TBitmap;
    FLayout: TButtonLayout;
    FSound: TsohoSound;
    FSoundName: TWaveFileName;
    IsFocused: boolean;
    FCanvas: TCanvas;
    FBottomColor: TColor;
    FHighlightColor: TColor;
    FBevWidth: Integer;
    procedure SetFlat(Value: boolean);
    procedure SetHighlightColor(Value: TColor);
    procedure SetColor(Value: TColor);
    procedure SetBottomColor(Value: TColor);
    procedure SetBevWidth(Value: Integer);
    procedure SetLayout(Value: TButtonLayout);
    procedure SetGlyph(Value: TBitmap);
    procedure SetSoundName(Value: TWaveFileName);
    procedure GlyphChanged(Sender: TObject);
  protected
    { Protected Declarations }
    function MouseUnderButton: boolean;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure CMMouseEnter(var message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var message: TMessage); message CM_MOUSELEAVE;
    procedure CNMeasureItem(var message: TWMMeasureItem); message CN_MEASUREITEM;
    procedure CNDrawItem(var message: TWMDrawItem); message CN_DRAWITEM;
    procedure CMFontChanged(var message: TMessage); message CM_FONTCHANGED;
    procedure CMParentColorChanged(var message: TMessage); message CM_PARENTCOLORCHANGED;
    procedure CMEnabledChanged(var message: TMessage); message CM_ENABLEDCHANGED;
    procedure DrawItem(const DrawItemStruct: TDrawItemStruct); virtual;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure SetButtonStyle(ADefault: boolean); override;
    function GetPalette: HPALETTE; override;
    property BevelWidth: Integer read FBevWidth write SetBevWidth default 1;
    property Canvas: TCanvas read FCanvas;
    property Color: TColor read FColor write SetColor default clBtnFace;
    property HighlightColor: TColor read FHighlightColor write SetHighlightColor default clWhite;
    property BottomColor: TColor read FBottomColor write SetBottomColor default clGray;
    property Sound: TsohoSound read FSound write FSound;
    property ClickSound: TWaveFileName read FSoundName write SetSoundName;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property Glyph: TBitmap read FGlyph write SetGlyph;
    property Flat: boolean read FFlat write SetFlat;
    property ParentColor;
  public
    { Public Declarations }
    procedure DrawButton(ButtonState: TButtonState; Rect: TRect;
      IsDown, IsDefault: boolean); virtual;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
  end;

  TsohoTransButton = class(TsohoCustomButton)
  private
    { Private Declarations }
    FKind: TsohoButtonKind;
    FBack: TBitmap;
  protected
    procedure SetKind(Value: TsohoButtonKind);
  public
    { Public Declarations }
    procedure WMKillFocus(var message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMSetFocus(var message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMEraseBkgnd(var message: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure DrawButton(ButtonState: TButtonState; Rect: TRect;
      IsDown, IsDefault: boolean); override;
    constructor Create(AOwner: TComponent); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    destructor Destroy; override;
    procedure Click; override;
  published
    { Published Declarations }
    property Kind: TsohoButtonKind read FKind write SetKind default sbCustom;
    property BevelWidth;
    property HighlightColor;
    property BottomColor;
    property Sound;
    property ClickSound;
    property Cursor default crHandPoint;
    property Layout;
    property Glyph;
    property Flat;
  end;

  TsohoButtonType = (btRectangle, btEllipse);

  TsohoBitBtn = class(TsohoCustomButton)
  private
    FGlyph: Pointer;
    FStyle: TButtonStyle;
    FKind: TBitBtnKind;
    fType: TsohoButtonType;
    FBackColor: TColor;
    FLayout: TButtonLayout;
    FSpacing: Integer;
    FMargin: Integer;
    IsFocused: boolean;
    FModifiedGlyph: boolean;
    FSound: TsohoSound;
    FSoundName: TWaveFileName;
    procedure SetGlyph(Value: TBitmap);
    function GetGlyph: TBitmap;
    function GetNumGlyphs: TNumGlyphs;
    procedure SetNumGlyphs(Value: TNumGlyphs);
    procedure GlyphChanged(Sender: TObject);
    function IsCustom: boolean;
    function IsCustomCaption: boolean;
    procedure SetStyle(Value: TButtonStyle);
    procedure SetKind(Value: TBitBtnKind);
    procedure SetType(Value: TsohoButtonType);
    procedure SetBackColor(Value: TColor);
    function GetKind: TBitBtnKind;
    procedure SetLayout(Value: TButtonLayout);
    procedure SetSpacing(Value: Integer);
    procedure SetMargin(Value: Integer);
    procedure SetSoundName(Value: TWaveFileName);
  protected
    procedure WMLButtonDown(var message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonDblClk(var message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
    procedure CMParentColorChanged(var message: TMessage); message CM_PARENTCOLORCHANGED;
    function GetPalette: HPALETTE; override;
    procedure SetButtonStyle(ADefault: boolean); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    procedure DrawButton(ButtonState: TButtonState; Rect: TRect;
      IsDown, IsDefault: boolean); override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
  published
    property Flat default False;
    property Color stored True;
    property ParentColor;
    property Cancel stored IsCustom;
    property Caption stored IsCustomCaption;
    property Default stored IsCustom;
    property Glyph: TBitmap read GetGlyph write SetGlyph stored IsCustom;
    property Kind: TBitBtnKind read GetKind write SetKind default bkCustom;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property Margin: Integer read FMargin write SetMargin default - 1;
    property ModalResult stored IsCustom;
    property NumGlyphs: TNumGlyphs read GetNumGlyphs write SetNumGlyphs stored IsCustom default 1;
    property ParentShowHint;
    property Style: TButtonStyle read FStyle write SetStyle default bsAutoDetect;
    property Spacing: Integer read FSpacing write SetSpacing default 4;
    property BtnType: TsohoButtonType read fType write SetType default btRectangle;
    property BackColor: TColor read FBackColor write SetBackColor default clBtnFace;
    property Cursor default crHandPoint;
    property Sound: TsohoSound read FSound write FSound;
    property ClickSound: TWaveFileName read FSoundName write SetSoundName;
  end;

  { TBitPool }
const
  BitsPerInt = SizeOf(Integer) * 8;

type
  TBitEnum = 0..BitsPerInt - 1;
  TBitSet = set of TBitEnum;
  TBitPool = class
  private
    FSize: Integer;
    FBits: Pointer;
    procedure SetSize(Value: Integer);
    procedure SetBit(index: Integer; Value: boolean);
    function GetBit(index: Integer): boolean;
  public
    destructor Destroy; override;
    function OpenBit: Integer;
    property Bits[index: Integer]: boolean read GetBit write SetBit; default;
    property Size: Integer read FSize write SetSize;
  end;
  
  TGlyphList = class(TImageList)
  private
    used: TBitPool;
    fCount: Integer;
    function AllocateIndex: Integer;
  public
    {$IFNDEF Win32}
    constructor Create(AWidth, AHeight: Integer);
    {$ELSE}
    constructor CreateSize(AWidth, AHeight: Integer);
    {$ENDIF}
    destructor Destroy; override;
    function Add(IMAGE, Mask: TBitmap): Integer;
    function AddMasked(IMAGE: TBitmap; MaskColor: TColor): Integer;
    procedure Delete(index: Integer);
    property Count: Integer read fCount;
  end;
  
  TGlyphCache = class
  private
    GlyphLists: TList;
  public
    constructor Create;
    destructor Destroy; override;
    function GetList(AWidth, AHeight: Integer): TGlyphList;
    procedure ReturnList(List: TGlyphList);
    function Empty: boolean;
  end;

  TButtonGlyph = class
  private
    FColor: TColor;
    FOriginal: TBitmap;
    FGlyphList: TGlyphList;
    FIndexs: array[TButtonState] of Integer;
    FTransparentColor: TColor;
    FNumGlyphs: TNumGlyphs;
    FOnChange: TNotifyEvent;
  public
    procedure GlyphChanged(Sender: TObject);
    procedure SetGlyph(Value: TBitmap);
    procedure SetNumGlyphs(Value: TNumGlyphs);
    procedure Invalidate;
    function CreateButtonGlyph(State: TButtonState): Integer;
    procedure DrawButtonGlyph(Canvas: TCanvas; X, Y: Integer;
      State: TButtonState);
    procedure DrawButtonText(Canvas: TCanvas; const Caption: string;
      TextBounds: TRect; State: TButtonState);
    procedure CalcButtonLayout(Canvas: TCanvas; const CLIENT: TRect;
      const Caption: string; Layout: TButtonLayout; Margin, SPACING: Integer;
      var GlyphPos: TPoint; var TextBounds: TRect);
    constructor Create;
    destructor Destroy; override;
    { return the text rectangle }
    function Draw(Canvas: TCanvas; const CLIENT: TRect;
      const Caption: string; Layout: TButtonLayout; Margin, SPACING: Integer;
      State: TButtonState): TRect;
    property Glyph: TBitmap read FOriginal write SetGlyph;
    property NumGlyphs: TNumGlyphs read FNumGlyphs write SetNumGlyphs;
    property Color: TColor read FColor write FColor default clBtnFace;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;
  
type
  PBitArray = ^TBitArray;
  TBitArray = array[0..4096] of TBitSet;

function DrawButtonFace(Canvas: TCanvas; const CLIENT: TRect; FillColor: TColor;
  BevelWidth: Integer; Style: TButtonStyle; IsRounded, IsDown,
  IsFocused: boolean): TRect;

procedure DrawGlyphWithText(const Glyph: TBitmap; Layout: TButtonLayout;
  Text: string; Canvas: TCanvas; Rect: TRect; var TxtRect: TRect);

procedure DrawUsualGlyphWithText(const Glyph: TBitmap; Layout: TButtonLayout;
  Text: string; Canvas: TCanvas; Rect: TRect; var TxtRect: TRect;
  Enabled: boolean);

procedure CreateBrushPattern;

const
  
  {TsohoButtons kinds constants}
  wtb_CaptionOk     = 12600; {"OK"}
  wtb_CaptionCancel = 12601; {""}
  wtb_CaptionYes    = 12602; {""}
  wtb_CaptionNo     = 12603; {""}
  wtb_CaptionClose  = 12604; {""}

  sbResNames: array[TsohoButtonKind] of string = ('sbOK',
    'sbCANCEL', 'sbYES', 'sbNO', 'sbCLOSE', '');
  WFBResNames: array[TsohoButtonKind] of string = ('WFBOK',
    'WFBCANCEL', 'WFBYES', 'WFBNO', 'WFBCLOSE', '');
  sbResCaptions: array[TsohoButtonKind] of Word = (wtb_CaptionOk,
    wtb_CaptionCancel, wtb_CaptionYes, wtb_CaptionNo,
    wtb_CaptionClose, 0);
  sbModalResults: array[TsohoButtonKind] of TModalResult = (mrOk,
    mrCancel, mrYes, mrNo, mrNone, mrNone);

var
  Pattern    : TBitmap;
  ButtonCount: Integer;

implementation

{$IFDEF Win32}
{$R SOHOBTNS.R32}
{$ELSE}
{$R SOHOBTNS.R16}
{$ENDIF}

uses (*{$IFDEF RUSSIAN_MESSAGES}*)
     Consts,
     (*{$ENDIF}*)
     SysUtils, SoUtils, VCLUtils;


{$IFNDEF Delphi3}
ResourceString
  SOKButton = 'OK';
  SCancelButton = '';
  SYesButton = '&';
  SNoButton = '&';
  SHelpButton = '&';
  SCloseButton = '&';
  SIgnoreButton = '&';
  SRetryButton = '&';
  SAbortButton = '&';
  SAllButton = '&';
{$Endif}

const
  { TsohoBitBtn data }
  BitBtnResNames: array[TBitBtnKind] of PChar = (
    nil, 'sbBOK', 'sbBCANCEL', 'sbBHELP', 'sbBYES', 'sbBNO', 'sbBCLOSE',
    'sbBABORT', 'sbBRETRY', 'sbBIGNORE', 'sbBALL');
  {$IFDEF Win32}
    {$IFDEF Delphi4}
        (*BitBtnResNames: array[TBitBtnKind] of PChar = (
          nil, 'BBOK', 'BBCANCEL', 'BBHELP', 'BBYES', 'BBNO', 'BBCLOSE',
          'BBABORT', 'BBRETRY', 'BBIGNORE', 'BBALL');*)
        BitBtnCaptions: array[TBitBtnKind] of Pointer = (
          nil, @SOKButton, @SCancelButton, @SHelpButton, @SYesButton, @SNoButton,
          @SCloseButton, @SAbortButton, @SRetryButton, @SIgnoreButton,
          @SAllButton);
        (*BitBtnModalResults: array[TBitBtnKind] of TModalResult = (
          0, mrOk, mrCancel, 0, mrYes, mrNo, 0, mrAbort, mrRetry, mrIgnore,
          mrAll);*)
    {$ELSE}
        BitBtnCaptions: array[TBitBtnKind] of string = (
          '', SOKButton, SCancelButton, SHelpButton, SYesButton, SNoButton,
          SCloseButton, SAbortButton, SRetryButton, SIgnoreButton,
          SAllButton);
    {$ENDIF}
  {$ELSE}
  BitBtnCaptions: array[TBitBtnKind] of Word = (
    0, SOKButton, SCancelButton, SHelpButton, SYesButton, SNoButton,
    SCloseButton, SAbortButton, SRetryButton, SIgnoreButton,
    SAllButton);
  {$ENDIF}

  BitBtnModalResults: array[TBitBtnKind] of TModalResult = (
    0, mrOk, mrCancel, 0, mrYes, mrNo, 0, mrAbort, mrRetry, mrIgnore,
    mrAll);
  
var
  BitBtnGlyphs: array[TBitBtnKind] of TBitmap;
  
{ DrawButtonFace - returns the remaining usable area inside the Client rect.}
function SohoDrawButtonFace(Canvas: TCanvas; const CLIENT: TRect; FillColor: TColor;
    BevelWidth: Integer; Style: TButtonStyle; IsRounded, IsDown,
    IsFocused: boolean; Transparent : boolean): TRect;
var
  NewStyle: boolean;
  R       : TRect;
begin
  NewStyle := ((Style = bsAutoDetect) and NewStyleControls) or (Style = bsNew);

  R := CLIENT;
  with Canvas do begin
    if NewStyle then begin
      if not Transparent then begin
        BRUSH.Color := FillColor;
        BRUSH.Style := bsSolid;
        FillRect(R);
      end;

      if IsDown then begin
        Frame3D(Canvas, R, clWindowFrame, clBtnHighlight, 1);
        Frame3D(Canvas, R, clBtnShadow, clBtnFace, 1);
      end
      else begin
        Frame3D(Canvas, R, clBtnHighlight, clWindowFrame, 1);
        Frame3D(Canvas, R, clBtnFace, clBtnShadow, 1);
      end;

      if IsFocused then begin
        InflateRect(R, - 1, - 1);
        BRUSH.Style := bsClear;
        Rectangle(R.Left, R.Top, R.Right, R.Bottom);
      end;
      
    end
    else begin
      Pen.Color := clWindowFrame;
      if not Transparent then begin
        BRUSH.Color := FillColor;
        BRUSH.Style := bsSolid;
      end
      else BRUSH.Style := bsClear;

      Rectangle(R.Left, R.Top, R.Right, R.Bottom);

      { round the corners - only applies to Win 3.1 style buttons }
      if IsRounded then begin
        Pixels[R.Left, R.Top] := clBtnFace;
        Pixels[R.Left, R.Bottom - 1] := clBtnFace;
        Pixels[R.Right - 1, R.Top] := clBtnFace;
        Pixels[R.Right - 1, R.Bottom - 1] := clBtnFace;
      end;

      if IsFocused then begin
        InflateRect(R, - 1, - 1);
        BRUSH.Style := bsClear;
        Rectangle(R.Left, R.Top, R.Right, R.Bottom);
      end;

      InflateRect(R, - 1, - 1);
      if not IsDown then Frame3D(Canvas, R, clBtnHighlight, clBtnShadow, BevelWidth)
      else begin
        Pen.Color := clBtnShadow;
        Polyline([Point(R.Left, R.Bottom - 1), Point(R.Left, R.Top),
          Point(R.Right, R.Top)]);
      end;
    end;
  end;

  Result := CLIENT;
  InflateRect(Result, - BevelWidth, - BevelWidth);
  if IsDown then OffsetRect(Result, 1, 1);
end;

function DrawButtonFace(Canvas: TCanvas; const CLIENT: TRect; FillColor: TColor;
    BevelWidth: Integer; Style: TButtonStyle; IsRounded, IsDown,
    IsFocused: boolean): TRect;
begin
  Result := SohoDrawButtonFace(Canvas, CLIENT, FillColor, BevelWidth,
            Style, IsRounded, IsDown, IsFocused, false);
end;

procedure GetCoordsGlyphAndText(const Glyph: TBitmap; Layout: TButtonLayout;
    Text: string; Canvas: TCanvas; Rect: TRect; var TxtRect: TRect;
    GlyphWidth: Integer; var IMAGE: TRect);
var AllW, AllH: Integer;
begin
  AllW := 0;
  AllH := 0;
  with Canvas do begin
    if Glyph <> nil then begin
      case Layout of
        blGlyphLeft,
          blGlyphRight: begin AllW := GlyphWidth + TextWidth(Text);
            AllH := GetMax(Glyph.Height, TextHeight('W'));
          end;
        blGlyphTop,
          blGlyphBottom: begin AllH := Glyph.Height + TextHeight('W');
            AllW := GetMax(GlyphWidth, TextWidth(Text));
          end;
      end;
      case Layout of
        blGlyphLeft: begin with IMAGE do begin
            Left := HorCenter(Rect, AllW);
            Top := VertCenter(Rect, Glyph.Height);
            Right := Left + GlyphWidth;
            Bottom := Top + Glyph.Height;
          end;
          TxtRect.Left := IMAGE.Right + 1;
          TxtRect.Top := VertCenter(Rect, TextHeight('W'));
          TxtRect.Right := TxtRect.Left + TextWidth(Text);
          TxtRect.Bottom := TxtRect.Top + TextHeight('W');
        end;
        blGlyphRight: begin TxtRect.Left := HorCenter(Rect, AllW);;
          TxtRect.Top := VertCenter(Rect, TextHeight('W'));
          TxtRect.Right := TxtRect.Left + TextWidth(Text);
          TxtRect.Bottom := TxtRect.Top + TextHeight('W');
          with IMAGE do begin
            Left := TxtRect.Right + 1;
            Top := VertCenter(Rect, Glyph.Height);
            Right := Left + GlyphWidth;
            Bottom := Top + Glyph.Height;
          end;
        end;
        blGlyphTop: begin with IMAGE do begin
            Left := HorCenter(Rect, GlyphWidth);
            Top := VertCenter(Rect, AllH);
            Right := Left + GlyphWidth;
            Bottom := Top + Glyph.Height;
          end;
          TxtRect.Left := HorCenter(Rect, TextWidth(Text));
          TxtRect.Top := IMAGE.Bottom + 1;
          TxtRect.Right := TxtRect.Left + TextWidth(Text);
          TxtRect.Bottom := TxtRect.Top + TextHeight('W');
        end;
        blGlyphBottom: begin TxtRect.Left := HorCenter(Rect, TextWidth(Text));
          TxtRect.Top := VertCenter(Rect, AllH);
          TxtRect.Right := TxtRect.Left + TextWidth(Text);
          TxtRect.Bottom := TxtRect.Top + TextHeight('W');
          with IMAGE do begin
            Left := HorCenter(Rect, GlyphWidth);
            Top := TxtRect.Bottom + 1;
            Right := Left + GlyphWidth;
            Bottom := Top + Glyph.Height;
          end;
        end;
      end;
    end
    else begin
      TxtRect.Left := HorCenter(Rect, TextWidth(Text));
      TxtRect.Top := VertCenter(Rect, TextHeight('W'));
      TxtRect.Right := TxtRect.Left + TextWidth(Text);
      TxtRect.Bottom := TxtRect.Top + TextHeight('W');
    end;
  end;
end;

procedure DrawUsualGlyphWithText;
var gW: Integer;
  IMAGE: TRect;
begin
  if Glyph.Width = 2 * Glyph.Height then gW := Glyph.Width div 2
  else gW := Glyph.Width;
  GetCoordsGlyphAndText(Glyph, Layout, Text, Canvas, Rect, TxtRect, gW, IMAGE);
  if Glyph <> nil then begin
    if Enabled or (Glyph.Width <> 2 * Glyph.Height) then
      Canvas.BrushCopy(IMAGE, Glyph, Bounds(0, 0, gW, Glyph.Height),
      Glyph.Canvas.Pixels[0, Glyph.Height - 1])
    else Canvas.BrushCopy(IMAGE, Glyph, Bounds(gW, 0, gW, Glyph.Height),
      Glyph.Canvas.Pixels[0, Glyph.Height - 1]);
  end;
end;

procedure DrawGlyphWithText;
var IMAGE: TRect;
  gW: Integer;
begin
  gW := Glyph.Width div 2;
  GetCoordsGlyphAndText(Glyph, Layout, Text, Canvas, Rect, TxtRect, gW, IMAGE);
  {  }
  with Canvas do begin
    CopyMode := cmSrcAnd;
    CopyRect(IMAGE, Glyph.Canvas, Bounds(gW, 0, gW, Glyph.Height));
    {   }
    CopyMode := cmSrcPaint;
    CopyRect(IMAGE, Glyph.Canvas, Bounds(0, 0, gW, Glyph.Height));
  end;
end;

function GetBitBtnGlyph(Kind: TBitBtnKind): TBitmap;
begin
  if BitBtnGlyphs[Kind] = nil then begin
    BitBtnGlyphs[Kind] := TBitmap.Create;
    BitBtnGlyphs[Kind].Handle := LoadBitmap(HInstance, BitBtnResNames[Kind]);
  end;
  Result := BitBtnGlyphs[Kind];
end;


destructor TBitPool.Destroy;
begin
  SetSize(0);
  inherited Destroy;
end;

procedure TBitPool.SetSize(Value: Integer);
var
  NewMem    : Pointer;
  NewMemSize: Integer;
  OldMemSize: Integer;
  
  function Min(X, Y: Integer): Integer;
  begin
    Result := X;
    if X > Y then Result := Y;
  end;
   begin
  if Value <> Size then begin
    NewMemSize := ((Value + BitsPerInt - 1) div BitsPerInt) * SizeOf(Integer);
    OldMemSize := ((Size + BitsPerInt - 1) div BitsPerInt) * SizeOf(Integer);
    if NewMemSize <> OldMemSize then begin
      if NewMemSize <> 0 then begin
        GetMem(NewMem, NewMemSize);
        FillChar(NewMem^, NewMemSize, 0);
      end
      else NewMem := nil;
      if OldMemSize <> 0 then begin
        if NewMem <> nil then
          Move(FBits^, NewMem^, Min(OldMemSize, NewMemSize));
        FreeMem(FBits, OldMemSize);
      end;
      FBits := NewMem;
    end;
    FSize := Value;
  end;
end;

procedure TBitPool.SetBit(index: Integer; Value: boolean);
begin
  if Value then
    Include(PBitArray(FBits)^[index div BitsPerInt], index mod BitsPerInt)
  else
    Exclude(PBitArray(FBits)^[index div BitsPerInt], index mod BitsPerInt);
end;

function TBitPool.GetBit(index: Integer): boolean;
begin
  Result := index mod BitsPerInt in PBitArray(FBits)^[index div BitsPerInt];
end;

function TBitPool.OpenBit: Integer;
var
  I: Integer; 
  B: TBitSet; 
  J: TBitEnum;
  E: Integer; 
begin
  E := (Size + BitsPerInt - 1) div BitsPerInt - 1;
  for I := 0 to E do
    if PBitArray(FBits)^[I] <> [0..BitsPerInt - 1] then begin
      B := PBitArray(FBits)^[I];
      for J := Low(J) to High(J) do begin
        if not (J in B) then begin
          Result := I * BitsPerInt + J;
          if Result >= Size then Result := -1;
          Exit;
        end;
      end;
    end;
  Result := -1;
end;

{ TGlyphList }

{$IFNDEF Win32}
constructor TGlyphList.Create(AWidth, AHeight: Integer);
begin
  inherited Create(AWidth, AHeight);
  used := TBitPool.Create;
end;
{$ELSE}
constructor TGlyphList.CreateSize(AWidth, AHeight: Integer);
begin
  inherited CreateSize(AWidth, AHeight);
  used := TBitPool.Create;
end;
{$ENDIF}

destructor TGlyphList.Destroy;
begin
  used.Free;
  inherited Destroy;
end;

function TGlyphList.AllocateIndex: Integer;
begin
  Result := used.OpenBit;
  if Result = -1 then begin
    Result := inherited Add(nil, nil);
    used.Size := Result + 1;
  end;
  used[Result] := True;
end;

function TGlyphList.Add(IMAGE, Mask: TBitmap): Integer;
begin
  Result := AllocateIndex;
  Replace(Result, IMAGE, Mask);
  Inc(fCount);
end;

function TGlyphList.AddMasked(IMAGE: TBitmap; MaskColor: TColor): Integer;
begin
  Result := AllocateIndex;
  ReplaceMasked(Result, IMAGE, MaskColor);
  Inc(fCount);
end;

procedure TGlyphList.Delete(index: Integer);
begin
  if used[index] then begin
    Dec(fCount);
    used[index] := False;
  end;
end;

{ TGlyphCache }

constructor TGlyphCache.Create;
begin
  inherited Create;
  GlyphLists := TList.Create;
end;

destructor TGlyphCache.Destroy;
begin
  GlyphLists.Free;
  inherited Destroy;
end;

function TGlyphCache.GetList(AWidth, AHeight: Integer): TGlyphList;
var
  I: Integer;
begin
  for I := GlyphLists.Count - 1 downto 0 do begin
    Result := GlyphLists[I];
    with Result do
      if (AWidth = Width) and (AHeight = Height) then Exit;
  end;
  {$IFNDEF Win32}
  Result := TGlyphList.Create(AWidth, AHeight);
  {$ELSE}
  Result := TGlyphList.CreateSize(AWidth, AHeight);
  {$ENDIF}
  GlyphLists.Add(Result);
end;

procedure TGlyphCache.ReturnList(List: TGlyphList);
begin
  if List = nil then Exit;
  if List.Count = 0 then begin
    GlyphLists.Remove(List);
    List.Free;
  end;
end;

function TGlyphCache.EMPTY: boolean;
begin
  Result := GlyphLists.Count = 0;
end;

var
  GlyphCache: TGlyphCache;
  
  { TButtonGlyph }
  
constructor TButtonGlyph.Create;
var
  I: TButtonState;
begin
  inherited Create;
  FColor := clBtnFace;
  FOriginal := TBitmap.Create;
  FOriginal.OnChange := GlyphChanged;
  FTransparentColor := clOlive;
  FNumGlyphs := 1;
  for I := Low(I) to High(I) do
    FIndexs[I] := -1;
  if GlyphCache = nil then GlyphCache := TGlyphCache.Create;
end;

destructor TButtonGlyph.Destroy;
begin
  FOriginal.Free;
  Invalidate;
  if Assigned(GlyphCache) and GlyphCache.EMPTY then begin
    GlyphCache.Free;
    GlyphCache := nil;
  end;
  inherited Destroy;
end;

procedure TButtonGlyph.Invalidate;
var
  I: TButtonState;
begin
  for I := Low(I) to High(I) do begin
    if FIndexs[I] <> -1 then FGlyphList.Delete(FIndexs[I]);
    FIndexs[I] := -1;
  end;
  GlyphCache.ReturnList(FGlyphList);
  FGlyphList := nil;
end;

procedure TButtonGlyph.GlyphChanged(Sender: TObject);
begin
  if Sender = FOriginal then begin
    FTransparentColor := FOriginal.TransparentColor;
    Invalidate;
    if Assigned(FOnChange) then FOnChange(Self);
  end;
end;

procedure TButtonGlyph.SetGlyph(Value: TBitmap);
var
  Glyphs: Integer;
begin
  Invalidate;
  FOriginal.Assign(Value);
  if (Value <> nil) and (Value.Height > 0) then begin
    FTransparentColor := Value.TransparentColor;
    if Value.Width mod Value.Height = 0 then begin
      Glyphs := Value.Width div Value.Height;
      if Glyphs > 4 then Glyphs := 1;
      SetNumGlyphs(Glyphs);
    end;
  end;
end;

procedure TButtonGlyph.SetNumGlyphs(Value: TNumGlyphs);
begin
  if (Value <> FNumGlyphs) and (Value > 0) then begin
    Invalidate;
    FNumGlyphs := Value;
  end;
end;

function TButtonGlyph.CreateButtonGlyph(State: TButtonState): Integer;
var
  TmpImage, MonoBmp: TBitmap;     
  iWidth, IHeight  : Integer;     
  IRect, ORect     : TRect;       
  I                : TButtonState;
begin
  if (State = bsDown) and (NumGlyphs < 3) then State := bsUp;
  Result := FIndexs[State];
  if Result <> -1 then Exit;
  iWidth := FOriginal.Width div FNumGlyphs;
  IHeight := FOriginal.Height;
  if FGlyphList = nil then begin
    if GlyphCache = nil then GlyphCache := TGlyphCache.Create;
    FGlyphList := GlyphCache.GetList(iWidth, IHeight);
  end;
  TmpImage := TBitmap.Create;
  try
    TmpImage.Width := iWidth;
    TmpImage.Height := IHeight;
    IRect := Rect(0, 0, iWidth, IHeight);
    TmpImage.Canvas.BRUSH.Color := FColor;
    TmpImage.Canvas.FillRect(IRect);
    I := State;
    if Ord(I) >= NumGlyphs then I := bsUp;
    ORect := Rect(Ord(I) * iWidth, 0, (Ord(I) + 1) * iWidth, IHeight);
    case State of
      bsUp, bsDown: begin
        TmpImage.Canvas.BrushCopy(IRect, FOriginal, ORect, FTransparentColor);
        FIndexs[State] := FGlyphList.AddMasked(TmpImage, FColor);
      end;
      bsExclusive: begin
        TmpImage.Canvas.CopyRect(IRect, FOriginal.Canvas, ORect);
        FIndexs[State] := FGlyphList.AddMasked(TmpImage, FTransparentColor);
      end;
      bsDisabled:
        if NumGlyphs > 1 then begin
          TmpImage.Canvas.BrushCopy(IRect, FOriginal, ORect, FTransparentColor);
          FIndexs[State] := FGlyphList.Add(TmpImage, nil);
        end
        else begin
          { Create a disabled version }
          MonoBmp := TBitmap.Create;
          try
            with MonoBmp do begin
              Assign(FOriginal);
              Canvas.BRUSH.Color := clBlack;
              Width := iWidth;
              if Monochrome then begin
                Canvas.Font.Color := clWhite;
                Monochrome := False;
                Canvas.BRUSH.Color := clWhite;
              end;
              Monochrome := True;
            end;
            with TmpImage.Canvas do begin
              BRUSH.Color := FColor;
              FillRect(IRect);
              BRUSH.Color := clBlack;
              Font.Color := clWhite;
              CopyMode := MERGEPAINT;
              Draw(IRect.Left + 1, IRect.Top + 1, MonoBmp);
              CopyMode := SRCAND;
              Draw(IRect.Left, IRect.Top, MonoBmp);
              BRUSH.Color := clBtnShadow;
              Font.Color := clBlack;
              CopyMode := SRCPAINT;
              Draw(IRect.Left, IRect.Top, MonoBmp);
              CopyMode := SRCCOPY;
            end;
            FIndexs[State] := FGlyphList.Add(TmpImage, nil);
          finally
            MonoBmp.Free;
          end;
        end;
    end;
  finally
    TmpImage.Free;
  end;
  Result := FIndexs[State];
  FOriginal.Dormant;
end;

procedure TButtonGlyph.DrawButtonGlyph(Canvas: TCanvas; X, Y: Integer;
    State: TButtonState);
var
  index: Integer;
begin
  if FOriginal = nil then Exit;
  if (FOriginal.Width = 0) or (FOriginal.Height = 0) then Exit;
  index := CreateButtonGlyph(State);
  FGlyphList.Draw(Canvas, X, Y, index);
end;

procedure TButtonGlyph.DrawButtonText(Canvas: TCanvas; const Caption: string;
    TextBounds: TRect; State: TButtonState);
var
  CString: array[0..255] of Char;
begin
  StrPCopy(CString, Caption);
  Canvas.BRUSH.Style := bsClear;
  if State = bsDisabled then begin
    with Canvas do begin
      OffsetRect(TextBounds, 1, 1);
      Font.Color := clWhite;
      DrawText(Handle, CString, Length(Caption), TextBounds, 0);
      OffsetRect(TextBounds, - 1, - 1);
      Font.Color := clDkGray;
      DrawText(Handle, CString, Length(Caption), TextBounds, 0);
    end;
  end
  else
    DrawText(Canvas.Handle, CString, - 1, TextBounds,
      DT_CENTER or DT_VCENTER or DT_SINGLELINE);
end;

procedure TButtonGlyph.CalcButtonLayout(Canvas: TCanvas; const CLIENT: TRect;
    const Caption: string; Layout: TButtonLayout; Margin, SPACING: Integer;
    var GlyphPos: TPoint; var TextBounds: TRect);
var
  TextPos                        : TPoint;               
  ClientSize, GlyphSize, TextSize: TPoint;               
  TotalSize                      : TPoint;               
  CString                        : array[0..255] of Char;
  {SpaceLeft                     : Integer;              }
  
begin
  { calculate the item sizes }
  ClientSize := Point(CLIENT.Right - CLIENT.Left, CLIENT.Bottom -
    CLIENT.Top);
  
  if FOriginal <> nil then
    GlyphSize := Point(FOriginal.Width div FNumGlyphs, FOriginal.Height) else
    GlyphSize := Point(0, 0);
  
  if Length(Caption) > 0 then begin
    TextBounds := Rect(0, 0, CLIENT.Right - CLIENT.Left, 0);
    DrawText(Canvas.Handle, StrPCopy(CString, Caption), - 1, TextBounds,
      DT_CALCRECT);
  end
  else
    TextBounds := Rect(0, 0, 0, 0);
  TextSize := Point(TextBounds.Right - TextBounds.Left, TextBounds.Bottom -
    TextBounds.Top);
  
  { If the layout has the glyph on the right or the left, then both the
    text and the glyph are centered vertically.  If the glyph is on the top
    or the bottom, then both the text and the glyph are centered horizontally.}
  if Layout in [blGlyphLeft, blGlyphRight] then begin
    GlyphPos.Y := (ClientSize.Y div 2) - (GlyphSize.Y div 2);
    TextPos.Y := (ClientSize.Y div 2) - (TextSize.Y div 2);
  end
  else begin
    GlyphPos.X := (ClientSize.X div 2) - (GlyphSize.X div 2);
    TextPos.X := (ClientSize.X div 2) - (TextSize.X div 2);
  end;
  
  { if there is no text or no bitmap, then Spacing is irrelevant }
  if (TextSize.X = 0) or (GlyphSize.X = 0) then
    SPACING := 0;
  
  { adjust Margin and Spacing }
  if Margin = -1 then begin
    if SPACING = -1 then begin
      TotalSize := Point(GlyphSize.X + TextSize.X, GlyphSize.Y + TextSize.Y);
      if Layout in [blGlyphLeft, blGlyphRight] then
        Margin := (ClientSize.X - TotalSize.X) div 3
      else
        Margin := (ClientSize.Y - TotalSize.Y) div 3;
      SPACING := Margin;
    end
    else begin
      TotalSize := Point(GlyphSize.X + SPACING + TextSize.X, GlyphSize.Y +
        SPACING + TextSize.Y);
      if Layout in [blGlyphLeft, blGlyphRight] then
        Margin := (ClientSize.X div 2) - (TotalSize.X div 2)
      else
        Margin := (ClientSize.Y div 2) - (TotalSize.Y div 2);
    end;
  end
  else begin
    if SPACING = -1 then begin
      TotalSize := Point(ClientSize.X - (Margin + GlyphSize.X), ClientSize.Y -
        (Margin + GlyphSize.Y));
      if Layout in [blGlyphLeft, blGlyphRight] then
        SPACING := (TotalSize.X div 2) - (TextSize.X div 2)
      else
        SPACING := (TotalSize.Y div 2) - (TextSize.Y div 2);
    end;
  end;
  
  case Layout of
    blGlyphLeft: begin
      GlyphPos.X := Margin;
      TextPos.X := GlyphPos.X + GlyphSize.X + SPACING;
    end;
    blGlyphRight: begin
      GlyphPos.X := ClientSize.X - Margin - GlyphSize.X;
      TextPos.X := GlyphPos.X - SPACING - TextSize.X;
    end;
    blGlyphTop: begin
      GlyphPos.Y := Margin;
      TextPos.Y := GlyphPos.Y + GlyphSize.Y + SPACING;
    end;
    blGlyphBottom: begin
      GlyphPos.Y := ClientSize.Y - Margin - GlyphSize.Y;
      TextPos.Y := GlyphPos.Y - SPACING - TextSize.Y;
    end;
  end;
  
  { fixup the result variables }
  Inc(GlyphPos.X, CLIENT.Left);
  Inc(GlyphPos.Y, CLIENT.Top);
  OffsetRect(TextBounds, TextPos.X + CLIENT.Left, TextPos.Y + CLIENT.Top);
end;

function TButtonGlyph.Draw(Canvas: TCanvas; const CLIENT: TRect;
    const Caption: string; Layout: TButtonLayout; Margin, SPACING: Integer;
    State: TButtonState): TRect;
var
  GlyphPos  : TPoint;               
  TextBounds: TRect;                
  {CString  : array[0..255] of Char;}
begin
  CalcButtonLayout(Canvas, CLIENT, Caption, Layout, Margin, SPACING,
    GlyphPos, TextBounds);
  DrawButtonGlyph(Canvas, GlyphPos.X, GlyphPos.Y, State);
  DrawButtonText(Canvas, Caption, TextBounds, State);
  Result := TextBounds;
end;

procedure CreateBrushPattern;
var
  X, Y: Integer;
begin
  Pattern := TBitmap.Create;
  Pattern.Width := 8;
  Pattern.Height := 8;
  with Pattern.Canvas do begin
    BRUSH.Style := bsSolid;
    BRUSH.Color := clBtnFace;
    FillRect(Rect(0, 0, Pattern.Width, Pattern.Height));
    for Y := 0 to 7 do
      for X := 0 to 7 do
        if (Y mod 2) = (X mod 2) then { toggles between even/odd pixles }
          Pixels[X, Y] := clWhite; { on even/odd rows }
  end;
end;

{ TsohoTextButton }
procedure TsohoTextButton.SetHighlightColor;
begin
  if FHighlightColor = Value then Exit;
  FHighlightColor := Value;
  Invalidate;
end;

procedure TsohoTextButton.SetBottomColor;
begin
  if FBottomColor = Value then Exit;
  FBottomColor := Value;
  Invalidate;
end;

procedure TsohoTextButton.GlyphChanged(Sender: TObject);
begin
  Invalidate;
end;

procedure TsohoTextButton.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if (AComponent = FSound) and (Operation = opRemove) then FSound := nil;
  inherited Notification(AComponent, Operation);
end;

procedure TsohoTextButton.SetSoundName;
begin
  FSoundName := Value;
end;

constructor TsohoTextButton.Create(AOwner: TComponent);
var Form: TForm;
begin
  inherited Create(AOwner);
  AutoSize := False;
  FGlyph := TBitmap.Create;
  FGlyph.OnChange := GlyphChanged;
  Bevel := True;
  BevelWidth := 1;
  BevelInner := bvNone;
  BevelOuter := bvRaised;
  Form := GetOwnerForm(Self);
  if Form <> nil then FSound := FindSound(Form);
  FSoundName := '';
  FModalResult := mrNone;
  FKind := sbCustom;
  FLayout := blGlyphLeft;
  FEnabled := True;
  FHighlightColor := clBtnHighlight;
  FBottomColor := clBlack;
  Cursor := crHandPoint;
  Height := 24;
  Width := 75;
end;

destructor TsohoTextButton.Destroy;
begin
  FGlyph.Free;
  inherited Destroy;
end;

procedure TsohoTextButton.SetLayout(Value: TButtonLayout);
begin
  if FLayout = Value then Exit;
  FLayout := Value; Invalidate;
end;

procedure TsohoTextButton.SetGlyph(Value: TBitmap);
begin
  FGlyph.Assign(Value);
  Invalidate;
end;

procedure TsohoTextButton.SetEnabled(Value: boolean);
begin
  if FEnabled = Value then Exit;
  FEnabled := Value; Invalidate;
end;

procedure TsohoTextButton.SetKind;
begin
  if FKind = Value then Exit;
  FKind := Value;
  case FKind of
    sbOk..sbClose: begin Glyph.Handle := ResBitmap(sbResNames[FKind]);
      if ((csLoading in ComponentState) and (Caption = ''))
        or (not (csLoading in ComponentState)) then
        Caption := ResString(sbResCaptions[FKind]);
      ModalResult := sbModalResults[FKind];
    end;
    sbCustom: ModalResult := mrNone;
  end;
  Invalidate;
end;

procedure TsohoTextButton.Paint;
var TopColor,
  BottomColor,
    FActiveColor: TColor;
  Rect, Txt     : TRect; 
  
begin
  with Canvas do begin
    Rect := GetClientRect;
    if Bevel then begin
      if BevelOuter <> bvNone then begin
        AdjustColors(BevelOuter, TopColor, BottomColor, FHighlightColor, FBottomColor);
        if FIsClicked then SwapLongInt(Longint(TopColor), Longint(BottomColor));
        Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
      end;
      Frame3D(Canvas, Rect, Color, Color, BorderWidth);
      if BevelInner <> bvNone then begin
        AdjustColors(BevelInner, TopColor, BottomColor, FHighlightColor, FBottomColor);
        if FIsClicked then SwapLongInt(Longint(TopColor), Longint(BottomColor));
        Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
      end;
    end;
    if not Transparent then begin
      BRUSH.Style := bsSolid;
      if Alive and MouseOverLabel then BRUSH.Color := ColorActive
      else BRUSH.Color := Self.Color;
      FillRect(Rect);
    end;
    BRUSH.Style := bsClear;
    Font.Assign(Self.Font);
    if FIsClicked then OffsetRect(Rect, 1, 1);
    DrawGlyphWithText(FGlyph, FLayout, Caption, Canvas, Rect, Txt);
    if FEnabled then FActiveColor := Font.Color
    else FActiveColor := clGray;
    InflateRect(Txt, Height3D, Height3D);
    OffsetRect(Txt, Height3D, 0);
    Draw3DText(Caption, Txt, Canvas, Light, FActiveColor, LightColor,
      ShadowColor, Height3D, taLeftJustify, False, False, False);
  end;
end;

procedure TsohoTextButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if not FEnabled then Exit;
  FOldLight := Light;
  FIsClicked := True;
  case Light of
    lpLeft: Light := lpRight;
    lpRight: Light := lpLeft;
    lpLeftTop: Light := lpRightBottom;
    lpLeftBottom: Light := lpRightTop;
    lpRightTop: Light := lpLeftBottom;
    lpRightBottom: Light := lpLeftTop;
    lpTop: Light := lpBottom;
    lpBottom: Light := lpTop;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TsohoTextButton.Click;
var Form: TForm;
begin
  if not FEnabled then Exit;
  if FSound <> nil then begin
    if FSoundName[1] = '#' then
      FSound.PlayResourceWave(Copy(FSoundName, 2, Length(FSoundName)))
    else FSound.PlayWave(FSoundName);
  end;
  Form := GetOwnerForm(Self);
  if Form <> nil then Form.ModalResult := ModalResult;
  inherited Click;
end;

procedure TsohoTextButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if not FEnabled then Exit;
  FIsClicked := False;
  Light := FOldLight;
  inherited MouseUp(Button, Shift, X, Y);
end;

function TsohoTextButton.GetPalette: HPALETTE;
begin
  Result := FGlyph.Palette;
end;

{ TsohoCustomButton }
procedure TsohoCustomButton.CMParentColorChanged(var message: TMessage);
begin
  if ParentColor then Color := GetParentColor(Self);
end;

function TsohoCustomButton.MouseUnderButton: boolean;
begin
  Result := FInRect;
end;

function TsohoCustomButton.GetPalette: HPALETTE;
begin
  Result := FGlyph.Palette;
end;

procedure TsohoCustomButton.GlyphChanged(Sender: TObject);
begin
  Invalidate;
end;

procedure TsohoCustomButton.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do Style := Style or BS_OWNERDRAW;
end;

procedure TsohoCustomButton.SetColor;
begin
  if FColor = Value then Exit;
  FColor := Value;
  Invalidate;
end;

procedure TsohoCustomButton.SetSoundName;
begin
  FSoundName := Value;
end;

procedure TsohoCustomButton.SetButtonStyle(ADefault: boolean);
begin
  if ADefault <> IsFocused then begin
    IsFocused := ADefault;
    Refresh;
  end;
end;

procedure TsohoCustomButton.CNMeasureItem(var message: TWMMeasureItem);
begin
  with message.MeasureItemStruct^ do begin
    ItemWidth := Width;
    ItemHeight := Height;
  end;
end;

procedure TsohoCustomButton.CNDrawItem(var message: TWMDrawItem);
begin
  if csDestroying in ComponentState then Exit;
  DrawItem(message.DrawItemStruct^);
end;

procedure TsohoCustomButton.Click;
begin
  if FSound <> nil then begin
    if FSoundName[1] = '#' then
      FSound.PlayResourceWave(Copy(FSoundName, 2, Length(FSoundName)))
    else FSound.PlayWave(FSoundName);
  end;
  inherited Click;
end;

procedure TsohoCustomButton.DrawButton(ButtonState: TButtonState; Rect: TRect;
    IsDown, IsDefault: boolean);
var InnerRect, Txt: TRect;
begin
  Canvas.Font := Self.Font;
  with InnerRect do begin
    Top := Rect.Top + BevelWidth;
    Left := Rect.Left + BevelWidth;
    Right := Rect.Right - BevelWidth;
    Bottom := Rect.Bottom - BevelWidth;
  end;
  with Canvas do begin
    InnerRect := SohoDrawButtonFace(Canvas, Rect, FColor, BevelWidth, bsNew, false, IsDown,
     false, false);
    DrawGlyphWithText(Glyph, Layout, Caption, Canvas, InnerRect, Txt);
    TextInRect(Canvas, Caption, Txt, taLeftJustify);
    BRUSH.Style := bsSolid;
  end;
  if IsDefault then begin
    Canvas.Pen.Color := HighlightColor;
    InflateRect(Rect, - 2 - BevelWidth, - 2 - BevelWidth);
    if IsDown then OffsetRect(Rect, - 1, - 1);
    DrawFocusRect(Canvas.Handle, Rect);
  end;
end;

procedure TsohoCustomButton.DrawItem(const DrawItemStruct: TDrawItemStruct);
var State: TButtonState;
  IsDown, IsDefault: boolean;
begin
  FCanvas.Handle := DrawItemStruct.HDC;
  with DrawItemStruct do begin
    IsDown := ItemState and ODS_SELECTED <> 0;
    IsDefault := ItemState and ODS_FOCUS <> 0;
    if not Enabled then State := bsDisabled
    else if IsDown then State := bsDown
    else State := bsUp;
  end;
  DrawButton(State, ClientRect, IsDown, IsDefault);
  FCanvas.Handle := 0;
end;

procedure TsohoCustomButton.CMFontChanged(var message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TsohoCustomButton.CMEnabledChanged(var message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TsohoCustomButton.SetHighlightColor;
begin
  if FHighlightColor = Value then Exit;
  FHighlightColor := Value;
  Invalidate;
end;

procedure TsohoCustomButton.SetBottomColor;
begin
  if FBottomColor = Value then Exit;
  FBottomColor := Value;
  Invalidate;
end;

procedure TsohoCustomButton.SetBevWidth;
begin
  if FBevWidth = Value then Exit;
  FBevWidth := Value;
  Invalidate;
end;

procedure TsohoCustomButton.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if (AComponent = FSound) and (Operation = opRemove) then FSound := nil;
  inherited Notification(AComponent, Operation);
end;

procedure TsohoCustomButton.SetLayout(Value: TButtonLayout);
begin
  if FLayout = Value then Exit;
  FLayout := Value; Invalidate;
end;

procedure TsohoCustomButton.SetGlyph(Value: TBitmap);
begin
  FGlyph.Assign(Value);
  Invalidate;
end;

procedure TsohoCustomButton.CMMouseEnter(var message: TMessage);
begin
  if not Flat then Exit;
  FInRect := True;
  Invalidate;
end;

procedure TsohoCustomButton.CMMouseLeave(var message: TMessage);
begin
  if not Flat then Exit;
  FInRect := False;
  Invalidate;
end;

procedure TsohoCustomButton.SetFlat(Value: boolean);
begin
  if FFlat = Value then Exit;
  FFlat := Value;
  Invalidate;
end;

constructor TsohoCustomButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FGlyph := TBitmap.Create;
  FGlyph.OnChange := GlyphChanged;
  if Owner is TForm then FSound := FindSound(AOwner as TForm);
  FSoundName := '';
  FLayout := blGlyphLeft;
  FCanvas := TCanvas.Create;
  FHighlightColor := clWhite;
  FBottomColor := clBlack;
  FBevWidth := 1;
  FColor := clBtnFace;
  FFlat := False;
  FInRect := False;
  Cursor := crHandPoint;
  Height := 24;
  Width := 75;
end;

destructor TsohoCustomButton.Destroy;
begin
  FCanvas.Free;
  FGlyph.Free;
  inherited Destroy;
end;

{ TsohoTransButton }
procedure TsohoTransButton.SetKind(Value: TsohoButtonKind);
begin
  if FKind = Value then Exit;
  FKind := Value;
  case FKind of
    sbOk..sbClose: begin Glyph.Handle := ResBitmap(sbResNames[FKind]);
      default := Value in [sbOk, sbYes];
      CANCEL := Value in [sbCancel, sbNo];
      if ((csLoading in ComponentState) and (Caption = ''))
        or (not (csLoading in ComponentState)) then begin
        default := Value in [sbOk, sbYes];
        CANCEL := Value in [sbCancel, sbNo];
        Caption := ResString(sbResCaptions[FKind]);
      end;
      ModalResult := sbModalResults[FKind];
    end;
    sbCustom: ModalResult := mrNone;
  end;
  Invalidate;
end;

procedure TsohoTransButton.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var DoRecreate : boolean;
begin
  DoRecreate := (Top <> ATop) or (Left <> ALeft) or (Height <> AHeight) or
    (Width <> AWidth);
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
  if DoRecreate then RecreateWnd;
end;

procedure TsohoTransButton.WMEraseBkgnd(var message: TWMEraseBkgnd);
var Rect : TRect;
begin
  if (csDesigning in ComponentState) then inherited
  else begin
    Message.Result := 1;
    Rect := GetClientRect;
    FBack.Free;
    FBack := TBitmap.Create;
    FBack.Height := Rect.Bottom - Rect.Top;
    FBack.Width := Rect.Right - Rect.Left;
    CopyParentImage(Self, FBack.Canvas);
  end;
end;

procedure TsohoTransButton.DrawButton(ButtonState: TButtonState; Rect: TRect;
    IsDown, IsDefault: boolean);
var InnerRect, Txt: TRect;
begin
  if (csDesigning in ComponentState) then
    inherited DrawButton(ButtonState, Rect, IsDown, IsDefault)
  else begin
    Canvas.Font := Self.Font;
    with InnerRect do begin
      Top := Rect.Top + BevelWidth;
      Left := Rect.Left + BevelWidth;
      Right := Rect.Right - BevelWidth;
      Bottom := Rect.Bottom - BevelWidth;
    end;

    with Canvas do begin
      BitBlt(Canvas.Handle, 0, 0, Rect.Right - Rect.Left, Rect.Bottom - Rect.Top,
        FBack.Canvas.Handle, 0, 0, SRCCOPY);

      if IsDown then Frame3D(Canvas, InnerRect, BottomColor, HighlightColor, BevelWidth)
      else begin
        if FFlat and (FInRect or (csDesigning in ComponentState)) then
          InnerRect := SohoDrawButtonFace(Canvas, Rect, FColor, BevelWidth, bsNew, false, IsDown,
            Focused, true);
        if not FFlat then
          InnerRect := SohoDrawButtonFace(Canvas, Rect, FColor, BevelWidth, bsNew, false, IsDown,
            Focused, true);
      end;
      if IsDown then OffsetRect(InnerRect, 1, 1);
      BRUSH.Style := bsClear;
      DrawGlyphWithText(Glyph, Layout, Caption, Canvas, InnerRect, Txt);
      TextInRect(Canvas, Caption, Txt, taLeftJustify);
    end;
    if IsDefault then begin
      Canvas.Pen.Color := clBlack;
      InflateRect(Rect, - 2 - BevelWidth, - 2 - BevelWidth);
      DrawFocusRect(Canvas.Handle, Rect);
    end;
  end;
end;

procedure TsohoTransButton.WMKillFocus(var message: TWMKillFocus);
begin
  Repaint;
  inherited;
end;

procedure TsohoTransButton.WMSetFocus(var message: TWMSetFocus);
begin
  Repaint;
  inherited;
end;

constructor TsohoTransButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FKind := sbCustom;
  FBack := TBitmap.Create;
  FInRect := False;
end;

destructor TsohoTransButton.Destroy;
begin
  FBack.Free;
  inherited Destroy;
end;

procedure TsohoTransButton.Click;
begin
  inherited Click;
  Repaint;
end;

procedure TsohoTransButton.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  if not (csDesigning in ComponentState) then
    Params.ExStyle := Params.ExStyle + WS_EX_TRANSPARENT;
end;

{ TsohoBitBtn }
procedure TsohoBitBtn.CMParentColorChanged(var message: TMessage);
begin
  inherited;
  if ParentColor then begin
    if Parent <> nil then FBackColor := GetParentColor(Self)
    else FBackColor := GetControlColor(Owner as TForm);
  end;
end;

procedure TsohoBitBtn.SetType(Value: TsohoButtonType);
begin
  if fType = Value then Exit;
  fType := Value;
  Invalidate;
end;

procedure TsohoBitBtn.SetBackColor(Value: TColor);
begin
  if FBackColor = Value then Exit;
  if not (csLoading in ComponentState) and
    not (csReading in ComponentState) then ParentColor := False;
  FBackColor := Value;
  Repaint;
end;

procedure TsohoBitBtn.WMLButtonDown(var message: TWMLButtonDown);
var X0, Y0, A, B, X, Y: Integer;
begin
  {   ,   -      ? }
  if fType = btRectangle then inherited
  else begin
    A := Width div 2;
    B := Height div 2;
    X0 := Left + A;
    Y0 := Top + B;
    X := Left + message.XPos;
    Y := Top + message.YPos;
    if ((X0 - X) * (X0 - X) / (A * A) +
      (Y0 - Y) * (Y0 - Y) / (B * B)) <= 1 then inherited;
  end;
end;

constructor TsohoBitBtn.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FGlyph := TButtonGlyph.Create;
  TButtonGlyph(FGlyph).OnChange := GlyphChanged;
  FStyle := bsNew;
  FKind := bkCustom;
  FLayout := blGlyphLeft;
  FSpacing := 4;
  FMargin := -1;
  Cursor := crHandPoint;
  Flat := False;
  Height := 24;
  Width := 75;
  FBackColor := clBtnFace;
  Color := clBtnFace;
  fType := btRectangle;
end;

destructor TsohoBitBtn.Destroy;
begin
  TButtonGlyph(FGlyph).Free;
  inherited Destroy;
end;

procedure TsohoBitBtn.SetButtonStyle(ADefault: boolean);
begin
  if ADefault <> IsFocused then begin
    IsFocused := ADefault;
    Refresh;
  end;
end;

procedure TsohoBitBtn.Click;
var Form: TForm;
  CONTROL: TWinControl;
begin
  if (FSound <> nil) and (ClickSound <> '') then begin
    if FSoundName[1] = '#' then
      FSound.PlayResourceWave(Copy(FSoundName, 2, Length(FSoundName)))
    else FSound.PlayWave(FSoundName);
  end;
  case FKind of
    bkClose: begin
      Form := GetOwnerForm(Self);
      if Form <> nil then Form.Close
      else inherited Click;
    end;
    bkHelp: begin
      CONTROL := Self;
      while (CONTROL <> nil) and (CONTROL.HelpContext = 0) do
        CONTROL := CONTROL.Parent;
      if CONTROL <> nil then Application.HelpContext(CONTROL.HelpContext)
      else inherited Click;
    end;
  else inherited Click;
  end;
end;


procedure TsohoBitBtn.DrawButton(ButtonState: TButtonState; Rect: TRect;
    IsDown, IsDefault: boolean);
var NewStyle: boolean;
  R, TextBounds: TRect;  
  Bevel        : Integer;
  
  procedure DrawRectButton;
  begin
    TButtonGlyph(FGlyph).Color := FColor;
    Canvas.Font := Font;
    TextBounds := TButtonGlyph(FGlyph).Draw(Canvas, R, Caption, FLayout, FMargin, FSpacing,
      ButtonState);
    
    if IsDefault then begin
      Canvas.BRUSH.Color := FColor;
      if NewStyle then TextBounds := R;
      if NewStyle then begin
        InflateRect(TextBounds, - 2, - 2);
        if IsDown then OffsetRect(TextBounds, - 1, - 1);
      end
      else InflateRect(TextBounds, 1, 1);
      DrawFocusRect(FCanvas.Handle, TextBounds);
    end;
  end;
  
  procedure DrawFocusCircle(ACanvas: TCanvas; var Rect: TRect; BtnColor: TColor);
  begin
    with ACanvas do begin
      Pen.Color := clBlack xor BtnColor;
      Pen.Width := 1;
      Pen.Style := psDot;
      Pen.Mode := pmXor;
      BRUSH.Style := bsClear;
      Ellipse(Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
    end;
  end;
  
  procedure DrawEllipseButtonText(const Caption: string; TextBounds: TRect);
  var CString: array[0..255] of Char;
  begin
    StrPCopy(CString, Caption);
    Canvas.BRUSH.Style := bsClear;
    if ButtonState = bsDisabled then begin
      with Canvas do begin
        OffsetRect(TextBounds, 1, 1);
        Font.Color := clBtnHighlight;
        DrawText(Handle, CString, Length(Caption), TextBounds,
          DT_CENTER or DT_VCENTER or DT_SINGLELINE);
        OffsetRect(TextBounds, - 1, - 1);
        Font.Color := clBtnShadow;
        DrawText(Handle, CString, Length(Caption), TextBounds,
          DT_CENTER or DT_VCENTER or DT_SINGLELINE);
      end;
    end
    else begin
      Canvas.Font := Font;
      DrawText(Canvas.Handle, CString, - 1, TextBounds,
        DT_CENTER or DT_VCENTER or DT_SINGLELINE);
    end;
  end;
  
  procedure DrawEllipseButton;
  var X, Y, R1, R2: Integer;
    FaceColor: TColor;
  begin
    TextBounds := R;
    FaceColor := Color;
    with Canvas do begin
      Pen.Style := psSolid;
      Pen.Mode := pmCopy;
      Pen.Width := 1;
      BRUSH.Color := Self.Color;
      BRUSH.Style := bsSolid;
      InflateRect(R, - 1, - 1);
      if IsDown then begin
        Pen.Color := clBtnShadow;
        Ellipse(R.Left, R.Top, R.Right, R.Bottom);
        FloodFill((R.Left + R.Right) div 2, (R.Top + R.Bottom) div 2, FaceColor,
          fsBorder);
        Pen.Color := clBtnHighlight;
        Ellipse(R.Left + 1, R.Top + 1, R.Right, R.Bottom);
        R1 := trunc((R.Right - R.Left) * sin(45) / 2);
        R2 := trunc((R.Bottom - R.Top) * sin(45) / 2);
        X := R.Left + 1 + (R.Right - R.Left) div 2 - R1;
        Y := R.Top + 1 + (R.Bottom - R.Top) div 2 + R2;
        Pen.Color := clWindowFrame;
        Ellipse(R.Left + 1, R.Top + 1, R.Right - 1, R.Bottom - 1);
        Pen.Color := clBtnShadow;
        Arc(R.Left + 1, R.Top + 1, R.Right - 1, R.Bottom - 1, X, Y, X + 2 * R1, Y - 2 * R2);
      end
      else begin
        Pen.Color := clBtnShadow;
        Ellipse(R.Left, R.Top, R.Right - 1, R.Bottom - 1);
        FloodFill((R.Left + R.Right) div 2, (R.Top + R.Bottom) div 2, FaceColor,
          fsBorder);
        Pen.Color := clWindowFrame;
        Ellipse(R.Left + 2, R.Top + 2, R.Right, R.Bottom);
        R1 := trunc((R.Right - R.Left) * sin(45) / 2);
        R2 := trunc((R.Bottom - R.Top) * sin(45) / 2);
        X := R.Left + 1 + (R.Right - R.Left) div 2 - R1;
        Y := R.Top + 1 + (R.Bottom - R.Top) div 2 + R2;
        Pen.Color := clBtnHighlight;
        Ellipse(R.Left + 1, R.Top + 1, R.Right - 1, R.Bottom - 1);
        Pen.Color := clBtnShadow;
        Arc(R.Left + 1, R.Top + 1, R.Right - 1, R.Bottom - 1, X, Y, X + 2 * R1, Y - 2 * R2);
      end;
    end;
    if IsDown then
      OffsetRect(TextBounds, 1, 1);
    DrawEllipseButtonText(Caption, TextBounds);
    InflateRect(R, - 3, - 3);
    if IsDefault then DrawFocusCircle(Canvas, R, Color);
  end;
begin
  NewStyle := ((Style = bsAutoDetect) and NewStyleControls) or (Style = bsNew);

  Canvas.Font := Self.Font;

  if NewStyle then Bevel := 1
  else Bevel := 2;
  
  R := ClientRect;
  if Flat and not (FInRect or (csDesigning in ComponentState)) then begin
    Canvas.BRUSH.Color := FBackColor;
    Canvas.FillRect(R);
    if fType = btRectangle then DrawRectButton
    else DrawEllipseButton;
  end
  else
    case fType of
      btRectangle: begin
        R := DrawButtonFace(Canvas, ClientRect, FColor, Bevel, FStyle, not NewStyle,
          IsDown, IsDefault or IsFocused);
        DrawRectButton;
      end;
      btEllipse: begin
        Canvas.BRUSH.Color := FBackColor;
        Canvas.FillRect(R);
        DrawEllipseButton;
      end;
    end;
end;

procedure TsohoBitBtn.WMLButtonDblClk(var message: TWMLButtonDblClk);
begin
  Perform(WM_LBUTTONDOWN, message.KEYS, Longint(message.Pos));
end;

function TsohoBitBtn.GetPalette: HPALETTE;
begin
  Result := Glyph.Palette;
end;

procedure TsohoBitBtn.SetGlyph(Value: TBitmap);
begin
  TButtonGlyph(FGlyph).Glyph := Value as TBitmap;
  FModifiedGlyph := True;
  Invalidate;
end;

function TsohoBitBtn.GetGlyph: TBitmap;
begin
  Result := TButtonGlyph(FGlyph).Glyph;
end;

procedure TsohoBitBtn.GlyphChanged(Sender: TObject);
begin
  Invalidate;
end;

function TsohoBitBtn.IsCustom: boolean;
begin
  Result := Kind = bkCustom;
end;

procedure TsohoBitBtn.SetStyle(Value: TButtonStyle);
begin
  if Value <> FStyle then begin
    FStyle := Value;
    Invalidate;
  end;
end;

procedure TsohoBitBtn.SetKind(Value: TBitBtnKind);
begin
  if Value <> FKind then begin
    if Value <> bkCustom then begin
      default := Value in [bkOK, bkYes];
      CANCEL := Value in [bkCancel, bkNo];

      if ((csLoading in ComponentState) and (Caption = '')) or
        (not (csLoading in ComponentState)) then begin
        {$IFDEF Win32}
          {$IFDEF Delphi4}
            if BitBtnCaptions[Value] <> nil then
              Caption := LoadResString(BitBtnCaptions[Value]);
          {$ELSE}
            if BitBtnCaptions[Value] <> '' then
                Caption := BitBtnCaptions[Value];
          {$ENDIF}
        {$ELSE}
        if BitBtnCaptions[Value] > 0 then
          Caption := LoadStr(BitBtnCaptions[Value]);
        {$ENDIF}
      end;
      
      ModalResult := BitBtnModalResults[Value];
      TButtonGlyph(FGlyph).Glyph := GetBitBtnGlyph(Value);
      NumGlyphs := 2;
      FModifiedGlyph := False;
    end;
    FKind := Value;
    Invalidate;
  end;
end;

function TsohoBitBtn.IsCustomCaption: boolean;
begin
  {$IFDEF Win32}
    {$IFDEF Delphi3}
     Result := CompareStr(Caption, BitBtnCaptions[FKind]) <> 0;
    {$ELSE}
     Result := AnsiCompareStr(Caption, LoadResString(BitBtnCaptions[FKind])) <> 0;
    {$ENDIF}
  {$ELSE}
  Result := CompareStr(Caption, LoadStr(BitBtnCaptions[FKind])) <> 0;
  {$ENDIF}
end;

function TsohoBitBtn.GetKind: TBitBtnKind;
begin
  if FKind <> bkCustom then
    if ((FKind in [bkOK, bkYes]) and (not default)) or
      ((FKind in [bkCancel, bkNo]) and (not CANCEL)) or
      (ModalResult <> BitBtnModalResults[FKind]) or
      FModifiedGlyph then
      FKind := bkCustom;
  Result := FKind;
end;

procedure TsohoBitBtn.SetLayout(Value: TButtonLayout);
begin
  if FLayout <> Value then begin
    FLayout := Value;
    Invalidate;
  end;
end;

function TsohoBitBtn.GetNumGlyphs: TNumGlyphs;
begin
  Result := TButtonGlyph(FGlyph).NumGlyphs;
end;

procedure TsohoBitBtn.SetNumGlyphs(Value: TNumGlyphs);
begin
  if Value < 0 then Value := 1
  else if Value > 4 then Value := 4;
  if Value <> TButtonGlyph(FGlyph).NumGlyphs then begin
    TButtonGlyph(FGlyph).NumGlyphs := Value;
    Invalidate;
  end;
end;

procedure TsohoBitBtn.SetSpacing(Value: Integer);
begin
  if FSpacing <> Value then begin
    FSpacing := Value;
    Invalidate;
  end;
end;

procedure TsohoBitBtn.SetMargin(Value: Integer);
begin
  if (Value <> FMargin) and (Value >= - 1) then begin
    FMargin := Value;
    Invalidate;
  end;
end;

procedure TsohoBitBtn.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if (AComponent = FSound) and (Operation = opRemove) then FSound := nil;
  inherited Notification(AComponent, Operation);
end;

procedure TsohoBitBtn.SetSoundName;
begin
  FSoundName := Value;
end;

procedure DestroyLocals; far;
var I: TBitBtnKind;
begin
  for I := Low(TBitBtnKind) to High(TBitBtnKind) do BitBtnGlyphs[I].Free;
end;
 begin
  FillChar(BitBtnGlyphs, SizeOf(BitBtnGlyphs), 0);
  AddExitProc(DestroyLocals);
end.

