{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
    TsohoSplitter
}
unit SoSplit;

{$I SOHOLIB.INC}

interface
uses Classes, Controls, Graphics, WinTypes, WinProcs, Messages;

type
  
  {         
      .     ,   "" 
        .   
       ,      design-time,   ,
     ,   TsohoTreeView }
  {example:
    Control1.Align := alLeft;
    Splitter.Align := alLeft;
    Control2.Align := alClient;
  }
  TsohoSplitter = class(TCustomControl)
  private
    FLineDC: HDC;
    FDownPos: TPoint;
    FSplit: Integer;
    FMinSize: Integer;
    FMaxSize: Integer;
    FCONTROL: TControl;
    FNewSize: Integer;
    FActiveControl: TWinControl;
    FOldKeyDown: TKeyEvent;
    FBevel: boolean;
    FLineVisible: boolean;
    FOnMove: TNotifyEvent;
    procedure AllocateLineDC;
    procedure DrawLine;
    procedure ReleaseLineDC;
    procedure UpdateSize(X, Y: Integer);
    procedure FocusKeyDown(Sender: TObject; var KEY: Word; Shift: TShiftState);
    procedure SetBevel(Value: boolean);
    procedure SetAlign (Value : TAlign);
    function  GetAlign : TAlign;
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure StopSizing;
  public
    constructor Create(AOwner: TComponent); override;
  published
    {   }
    property Align : TAlign read GetAlign write SetAlign;
    {  Bevel = true,   "3D"  }
    property Bevel: boolean read FBevel write SetBevel default True;
    property Color;
    {   ( )    }
    property MinSize: Integer read FMinSize write FMinSize default 30;
    property ParentColor;
    {       }
    property OnMove: TNotifyEvent read FOnMove write FOnMove;
  end;


implementation
uses SoUtils, Forms, SysUtils;

type

  THack = class(TWinControl)
  public
    property OnKeyDown;
  end;

function ValidParentForm(CONTROL: TControl): TForm;
begin
  Result := GetOwnerForm(CONTROL);
  if not Assigned(Result) then Abort;
end;

procedure TsohoSplitter.SetAlign (Value : TAlign);
begin
  inherited Align := Value;
  if (Align = alLeft) or (Align = alRight) then Cursor := crHSplit;
  if (Align = alTop) or (Align = alBottom) then Cursor := crVSplit;
end;

function  TsohoSplitter.GetAlign : TAlign;
begin
  Result := inherited Align;
end;

constructor TsohoSplitter.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csDesignInteractive, csOpaque, csCaptureMouse];
  Align := alNone;
  Width := 3;
  Height := 50;
  FMinSize := 30;
  FBevel := True;
end;

procedure TsohoSplitter.AllocateLineDC;
begin
  FLineDC := GetDCEx(Parent.Handle, 0, DCX_CACHE or DCX_CLIPSIBLINGS
    or DCX_LOCKWINDOWUPDATE);
end;

procedure TsohoSplitter.DrawLine;
var P: TPoint;
begin
  FLineVisible := not FLineVisible;
  P := Point(Left, Top);
  if Align in [alLeft, alRight] then P.X := Left + FSplit
  else P.Y := Top + FSplit;
  with P do PatBlt(FLineDC, X, Y, Width, Height, PATINVERT);
end;

procedure TsohoSplitter.ReleaseLineDC;
begin
  ReleaseDC(Parent.Handle, FLineDC);
end;

procedure TsohoSplitter.Paint;
var FrameBrush: HBRUSH;
  R: TRect;
begin
  R := ClientRect;
  Canvas.Brush.Color := Color;
  Canvas.FillRect(ClientRect);
  if Bevel then begin
    if Align in [alLeft, alRight] then InflateRect(R, - 1, 2)
    else InflateRect(R, 2, - 1);
    OffsetRect(R, 1, 1);
    FrameBrush := CreateSolidBrush(ColorToRGB(clBtnHighlight));
    FrameRect(Canvas.Handle, R, FrameBrush);
    DeleteObject(FrameBrush);
    OffsetRect(R, - 2, - 2);
    FrameBrush := CreateSolidBrush(ColorToRGB(clBtnShadow));
    FrameRect(Canvas.Handle, R, FrameBrush);
    DeleteObject(FrameBrush);
  end;
end;

procedure TsohoSplitter.MouseDown(Button: TMouseButton; Shift: TShiftState;
    X, Y: Integer);
  
  function FindControl: TControl;
  var P: TPoint;
    I: Integer;
  begin
    Result := nil;
    P := Point(Left, Top);
    case Align of
      alLeft: Dec(P.X);
      alRight: Inc(P.X, Width);
      alTop: Dec(P.Y);
      alBottom: Inc(P.Y, Height);
    else exit;
    end;
    for I := 0 to Parent.ControlCount - 1 do begin
      Result := Parent.Controls[I];
      if PtInRect(Result.BoundsRect, P) then exit;
    end;
    Result := nil;
  end;
  
var I: Integer; begin
  inherited MouseDown(Button, Shift, X, Y);                                  
  if (Button = mbLeft) or (csDesigning in ComponentState) then begin
    SetCapture(Handle);                                                      
    FCONTROL                              := FindControl;                    
    FDownPos                              := Point(X, Y);                    
    if Assigned(FCONTROL) then begin
      if Align in [alLeft, alRight] then begin
        FMaxSize                          := Parent.ClientWidth - FMinSize;  
        for I                             := 0 to Parent.ControlCount - 1 do
          with Parent.Controls[I] do
            if Align in [alLeft, alRight] then Dec(FMaxSize, Width);         
        Inc(FMaxSize, FCONTROL.Width);
      end
      else begin
        FMaxSize                          := Parent.ClientHeight - FMinSize; 
        for I                             := 0 to Parent.ControlCount - 1 do
          with Parent.Controls[I] do
            if Align in [alTop, alBottom] then Dec(FMaxSize, Height);        
        Inc(FMaxSize, FCONTROL.Height);                                      
      end;                                                                   
      UpdateSize(X, Y);                                                      
      AllocateLineDC;                                                        
      with ValidParentForm(Self) do
        if ActiveControl <> nil then begin
          FActiveControl                  := ActiveControl;                  
          FOldKeyDown                     := THack(FActiveControl).OnKeyDown;
          THack(FActiveControl).OnKeyDown := FocusKeyDown;                   
        end;                                                                 
      DrawLine;                                                              
    end;                                                                     
  end;                                                                       
end;

procedure TsohoSplitter.UpdateSize(X, Y: Integer);
var S: Integer;
begin
  if Align in [alLeft, alRight] then FSplit := X - FDownPos.X
  else FSplit := Y - FDownPos.Y;
  S := 0;
  case Align of
    alLeft: S := FCONTROL.Width + FSplit;
    alRight: S := FCONTROL.Width - FSplit;
    alTop: S := FCONTROL.Height + FSplit;
    alBottom: S := FCONTROL.Height - FSplit;
  end;
  FNewSize := S;
  if S < FMinSize then FNewSize := FMinSize
  else if S > FMaxSize then FNewSize := FMaxSize;
  if S <> FNewSize then begin
    if Align in [alRight, alBottom] then S := S - FNewSize
    else S := FNewSize - S;
    Inc(FSplit, S);
  end;
end;

procedure TsohoSplitter.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseMove(Shift, X, Y);
  if Assigned(FCONTROL) then begin
    DrawLine;
    UpdateSize(X, Y);
    DrawLine;
  end;
end;

procedure TsohoSplitter.MouseUp(Button: TMouseButton; Shift: TShiftState;
    X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if Assigned(FCONTROL) then begin
    ReleaseCapture;
    DrawLine;
    case Align of
      alLeft: FCONTROL.Width := FNewSize;
      alTop: FCONTROL.Height := FNewSize;
      alRight: begin
        Parent.DisableAlign;
        try
          FCONTROL.Left := FCONTROL.Left + (FCONTROL.Width - FNewSize);
          FCONTROL.Width := FNewSize;
        finally
          Parent.EnableAlign;
        end;
      end;
      alBottom: begin
        Parent.DisableAlign;
        try
          FCONTROL.Top := FCONTROL.Top + (FCONTROL.Height - FNewSize);
          FCONTROL.Height := FNewSize;
        finally
          Parent.EnableAlign;
        end;
      end;
    end;
    StopSizing;
  end;
end;

procedure TsohoSplitter.FocusKeyDown(Sender: TObject; var KEY: Word; Shift: TShiftState);
begin
  if KEY = VK_ESCAPE then StopSizing
  else if Assigned(FOldKeyDown) then FOldKeyDown(Sender, KEY, Shift);
end;

procedure TsohoSplitter.SetBevel(Value: boolean);
begin
  FBevel := Value;
  Repaint;
end;

procedure TsohoSplitter.StopSizing;
begin
  if Assigned(FCONTROL) then begin
    if FLineVisible then DrawLine;
    FCONTROL := nil;
    ReleaseLineDC;
    if Assigned(FActiveControl) then begin
      THack(FActiveControl).OnKeyDown := FOldKeyDown;
      FActiveControl := nil;
    end;
  end;
  if Assigned(FOnMove) then FOnMove(Self);
end;

end.

