{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
    TsohoFormStorage  TsohoApplication
}
unit SoRxTls;

{$I SOHOLIB.INC}

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Placemnt, IniFiles, AppEvent, SoTools;

type

  {       TFormStorage  
    RxLib.        :
    TColor, TFont,          
     ini-
  }
  TsohoFormStorage = class(TFormStorage)
  private
    { Private declarations }
    FIni: TIniFile;
    FOnSavePlacement: TNotifyEvent;
    FOnRestorePlacement: TNotifyEvent;
  public
    {   ini-  }
    procedure WriteString(const IniSection, Ident, Value: string);
    {   ini-   }
    procedure WriteInteger(const IniSection, Ident: string; Value: Longint);
    {   ini-   }
    procedure WriteBool(const IniSection, Ident: string; Value: boolean);
    {   ini-  }
    procedure WriteColor(const IniSection, Ident: string; Value: TColor);
    {   ini-  }
    procedure WriteFont(const IniSection, Ident: string; Value: tFont);
    {   ini-  }
    function ReadString(const IniSection, Ident, DefValue: string): string;
    {   ini-  }
    procedure ReadFont(const IniSection, Ident: string; Value, DefValue: tFont);
    {   ini-   }
    function ReadInteger(const IniSection, Ident: string; DefValue: Longint): Longint;
    {   ini-   }
    function ReadBool(const IniSection, Ident: string; DefValue: boolean): boolean;
    {   ini-  }
    function ReadColor(const IniSection, Ident: string; DefValue: TColor): TColor;
    {   ini-     }
    procedure ReadSection(const IniSection: string; Strings: TStrings);
    {   ini-      }
    procedure ReadSectionValues(const IniSection: string; Strings: TStrings);
    {   ini-   }
    procedure ReadKeyValues(const IniSection: string; Strings: TStrings);
    {   ini-  }
    procedure DeleteIdent(const IniSection, Ident: string);
    {   ini-  }
    procedure RenameSection(const OldSection, NewSection: string);
    {   ini-  }
    procedure RenameIdent(const IniSection, OldIdent, NewIdent: string);
    {   ini-  }
    procedure EraseSection(const IniSection: string);
    {  ,    Section }
    procedure Clear;
    procedure SavePlacement; override;
    procedure RestorePlacement; override;
    procedure Loaded; override;
  published
    property OnSavePlacement: TNotifyEvent read FOnSavePlacement write FOnSavePlacement;
    property OnRestorePlacement: TNotifyEvent read FOnRestorePlacement write FOnRestorePlacement;
  end;

  {    :  ,  }
  TsohoHintScrollKind = (skHorizontal, skVertical);

  {       TAppEvents  RxLib }
  TsohoApplication = class(TAppEvents)
  private
    { Private declarations }
    FScrollSteps : LongInt;
    FScrollKind  : TsohoHintScrollKind;
    FHintAlign   : TAlignment;
    FOnMainCreate: TNotifyEvent;
    WasActived: boolean;
    FSearchPrev: boolean;
    FFonts: TStringList;
    FFontsDir: TTTFDirName;
    FApplicationID: string;
    procedure SetSearchPrev(Value: boolean);
    procedure SetFontsDir(Value: TTTFDirName);
    procedure SetFonts(Value: TStringList);
    procedure SetAppId(Value: string);
  protected
    { Protected declarations }
    function GetWinDir: string;
    procedure MainFormCreate(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;
  published
    {  ""  }
    property HintSteps : LongInt read FScrollSteps write FScrollSteps default 100;
    property HintKind  : TsohoHintScrollKind read FScrollKind write FScrollKind default skHorizontal;
    property HintAlign   : TAlignment read FHintAlign write FHintAlign default taLeftJustify;
    {      }
    property SearchPrev: boolean read FSearchPrev write SetSearchPrev;
    {       }
    property Fonts: TStringList read FFonts write SetFonts;
    {    }
    property FontsDir: TTTFDirName read FFontsDir write SetFontsDir;
    {$IFDEF WIN32}
    property ApplicationId: string read FApplicationId write SetAppId;
    {$ENDIF}
  end;

implementation
uses SoUtils, Forms, StrUtils
  {$IFDEF WIN32}
  ,SoSingle, SoHints
  {$ENDIF};

{ TsohoFormStorage }
procedure TsohoFormStorage.Loaded;
begin
  inherited Loaded;
end;

procedure TsohoFormStorage.DeleteIdent(const IniSection, Ident: string);
var S1: TStringList;
  index, I      : Integer;
  aIdent, aValue: string; 
begin
  S1 := TStringList.Create;
  FIni := TIniFile.Create(IniFileName);
  FIni.ReadSection(IniSection, S1);
  index := S1.IndexOf(Ident);
  if index > -1 then begin
    S1.Clear;
    FIni.ReadSectionValues(IniSection, S1);
    FIni.EraseSection(IniSection);
    for I := 0 to S1.Count - 1 do
      if I <> index then begin
        aIdent := Copy(S1[I], 1, Pos('=', S1[I]) - 1);
        aValue := S1.Values[aIdent];
        FIni.WriteString(IniSection, aIdent, aValue);
      end;
  end;
  FIni.Free;
end;

procedure TsohoFormStorage.RenameSection(const OldSection, NewSection: string);
var S1: TStringList;
  aIdent, aValue: string; 
  I             : Integer;
begin
  S1 := TStringList.Create;
  FIni := TIniFile.Create(IniFileName);
  FIni.ReadSectionValues(OldSection, S1);
  FIni.EraseSection(OldSection);
  for I := 0 to S1.Count - 1 do begin
    aIdent := Copy(S1[I], 1, Pos('=', S1[I]) - 1);
    aValue := S1.Values[aIdent];
    FIni.WriteString(NewSection, aIdent, aValue);
  end;
  FIni.Free;
end;

procedure TsohoFormStorage.RenameIdent(const IniSection, OldIdent, NewIdent: string);
var S1: TStringList;
  index, I      : Integer;
  aIdent, aValue: string; 
begin
  S1 := TStringList.Create;
  FIni := TIniFile.Create(IniFileName);
  FIni.ReadSection(IniSection, S1);
  index := S1.IndexOf(OldIdent);
  if index > -1 then begin
    S1.Clear;
    FIni.ReadSectionValues(IniSection, S1);
    FIni.EraseSection(IniSection);
    for I := 0 to S1.Count - 1 do begin
      if I <> index then begin
        aIdent := Copy(S1[I], 1, Pos('=', S1[I]) - 1);
        aValue := S1.Values[aIdent];
      end
      else begin
        aIdent := NewIdent;
        aValue := S1.Values[Copy(S1[I], 1, Pos('=', S1[I]) - 1)];
      end;
      FIni.WriteString(IniSection, aIdent, aValue);
    end;
  end;
  FIni.Free;
end;

procedure TsohoFormStorage.ReadSection(const IniSection: string; Strings: TStrings);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.ReadSection(IniSection, Strings);
  FIni.Free;
end;

procedure TsohoFormStorage.ReadSectionValues(const IniSection: string; Strings: TStrings);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.ReadSectionValues(IniSection, Strings);
  FIni.Free;
end;

procedure TsohoFormStorage.ReadKeyValues(const IniSection: string; Strings: TStrings);
var Tmp : TStringList;
    Index : LongInt;
begin
  Tmp := TStringList.Create;
  FIni := TIniFile.Create(IniFileName);
  Strings.Clear;
  FIni.ReadSection(IniSection, Tmp);
  for Index := 0 to pred(Tmp.Count) do
    Strings.Add(FIni.ReadString(IniSection, Tmp[Index], ''));
  FIni.Free;
  Tmp.Free;
end;

procedure TsohoFormStorage.WriteString(const IniSection, Ident, Value: string);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.WriteString(IniSection, Ident, Value);
  FIni.Free;
end;

procedure TsohoFormStorage.WriteInteger(const IniSection, Ident: string; Value: Longint);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.WriteInteger(IniSection, Ident, Value);
  FIni.Free;
end;

procedure TsohoFormStorage.WriteBool(const IniSection, Ident: string; Value: boolean);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.WriteBool(IniSection, Ident, Value);
  FIni.Free;
end;

procedure TsohoFormStorage.WriteColor(const IniSection, Ident: string; Value: TColor);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.WriteString(IniSection, Ident, ColorToString(Value));
  FIni.Free;
end;

procedure TsohoFormStorage.WriteFont(const IniSection, Ident: string; Value: tFont);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.WriteString(IniSection, Ident, FontToStr(Value));
  FIni.Free;
end;

function TsohoFormStorage.ReadString(const IniSection, Ident, DefValue: string): string;
begin
  FIni := TIniFile.Create(IniFileName);
  Result := FIni.ReadString(IniSection, Ident, DefValue);
  FIni.Free;
end;

procedure TsohoFormStorage.ReadFont(const IniSection, Ident: string; Value, DefValue: tFont);
begin
  FIni := TIniFile.Create(IniFileName);
  StrToFont(FIni.ReadString(IniSection, Ident, FontToStr(DefValue)), Value);
  FIni.Free;
end;

function TsohoFormStorage.ReadInteger(const IniSection, Ident: string; DefValue: Longint): Longint;
begin
  FIni := TIniFile.Create(IniFileName);
  Result := FIni.ReadInteger(IniSection, Ident, DefValue);
  FIni.Free;
end;

function TsohoFormStorage.ReadBool(const IniSection, Ident: string; DefValue: boolean): boolean;
begin
  FIni := TIniFile.Create(IniFileName);
  Result := FIni.ReadBool(IniSection, Ident, DefValue);
  FIni.Free;
end;

function TsohoFormStorage.ReadColor(const IniSection, Ident: string; DefValue: TColor): TColor;
begin
  FIni := TIniFile.Create(IniFileName);
  Result := StringToColor(FIni.ReadString(IniSection, Ident, ColorToString(DefValue)));
  FIni.Free;
end;

procedure TsohoFormStorage.EraseSection(const IniSection: string);
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.EraseSection(IniSection);
  FIni.Free;
end;

procedure TsohoFormStorage.Clear;
begin
  FIni := TIniFile.Create(IniFileName);
  FIni.EraseSection(IniSection);
  FIni.Free;
end;

procedure TsohoFormStorage.SavePlacement;
var Form: TForm;
begin
  inherited SavePlacement;
  Form := GetOwnerForm(Self);
  if Form <> nil then begin
    WriteColor(IniSection, 'FormColor', Form.Color);
    WriteFont(IniSection, 'FormFont', Form.Font);
  end;
  if Assigned(FOnSavePlacement) then FOnSavePlacement(Self);
end;

procedure TsohoFormStorage.RestorePlacement;
var Form: TForm;
begin
  inherited RestorePlacement;
  Form := GetOwnerForm(Self);
  if Form <> nil then begin
    Form.Color := ReadColor(IniSection, 'FormColor', Form.Color);
    ReadFont(IniSection, 'FormFont', Form.Font, Form.Font);
  end;
  if Assigned(FOnRestorePlacement) then FOnRestorePlacement(Self);
end;

{ TsohoApplication }
procedure TsohoApplication.SetAppId(Value: string);
begin
  if Value = '' then exit;
  FApplicationID := Value;
end;

procedure TsohoApplication.SetSearchPrev(Value: boolean);
begin
  FSearchPrev := Value;
end;

procedure TsohoApplication.SetFonts;
begin
  FFonts.Assign(Value);
end;

procedure TsohoApplication.MainFormCreate;
var Index : LongInt;
    FileName : string;
begin
  if not WasActived then
    { ,       }
    if (Owner as TForm) <> nil then begin
      if (CmdShow = SW_SHOWMINNOACTIVE) or
        (CmdShow = SW_SHOWMINIMIZED) then (Owner as TForm).WindowState := wsMinimized
      else
        if CmdShow = SW_SHOWMAXIMIZED then (Owner as TForm).WindowState := wsMaximized;
      WasActived := True;
    end;
  if Assigned(FOnMainCreate) then FOnMainCreate(Owner as TForm);
  for Index := 0 to pred(Fonts.Count) do begin
    FileName := FontsDir + Fonts[Index];
    if FileExists(FileName) then AddFontResource(PChar(FileName));
  end;
end;

constructor TsohoApplication.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSearchPrev := True;
  FFonts := TStringList.Create;
  FFontsDir := '';
  WasActived := False;
  FApplicationID := '{' + Dec2Hex(StrToInt(IntToStr(trunc(Date))             ), 6) +
                          Dec2Hex(StrToInt(DeleteChars(TimeToStr(Time),[':'])), 6) +
                    '}';
  FScrollSteps := 100;
  FScrollKind  := skHorizontal;
  FHintAlign   := taLeftJustify;
end;

destructor TsohoApplication.Destroy;
var Index: Integer;
    FileName: string;
begin
  { Remove fonts resources }
  if not (csDesigning in ComponentState) then begin
    // GetMem(Filename, 255);
    for Index := 0 to pred(Fonts.Count) do begin
      FileName := FontsDir + Fonts[Index];
      if FileExists(FileName) then RemoveFontResource(PChar(Filename));
    end;
    // FreeMem(Filename, 255);
  end;
  FFonts.Free;
  inherited Destroy;
end;

procedure TsohoApplication.SetFontsDir;
begin
  {    '\'}
  FFontsDir := Value;
  if (Length(FFontsDir) <> 0) and (FFontsDir[Length(FFontsDir)] <> '\')
    then FFontsDir := FFontsDir + '\';
end;

procedure TsohoApplication.Loaded;
var K: Integer;
  // Filename : PChar;
  OwnerForm: TForm;
begin
  inherited Loaded;
  if not (csDesigning in ComponentState) then begin
    ScrollSteps := FScrollSteps;
    HorizontalScroll := FScrollKind = skHorizontal;
    SoHints.HintAlign := FHintAlign;
    SetSohoHint;
    // GetMem(Filename, 255);
    // FreeMem(Filename, 255);
    {  ,   CmdShow}
    OwnerForm := GetOwnerForm(Self);
    FOnMainCreate := OwnerForm.OnCreate;
    OwnerForm.OnCreate := MainFormCreate;
    if FSearchPrev then begin
      {$IFNDEF Win32}
      if HPrevInst <> 0 then begin
        GotoPreviousInstance;
        Application.Terminate;
      end;
      {$ELSE}
      if AlreadyRunning(ApplicationId, True, false) then Application.Terminate;
      {$ENDIF}
    end;
  end;
end;

function TsohoApplication.GetWinDir: string;
var Buf: PChar;
begin
  GetMem(Buf, 500);
  GetWindowsDirectory(Buf, 500);
  GetWinDir := StrPas(Buf);
  FreeMem(Buf, 500);
end;

end.

