{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Updated by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
    TsohoGauge
}
unit SoGauges;

{$I SOHOLIB.INC}


interface

uses WinTypes, Messages, Classes, Graphics, Controls, Forms, StdCtrls;

type

  {   }
  TBltBitmap = class(TBitmap)
    procedure MakeLike (ATemplate: TBitmap);
  end;

  {   : ,  , 
    ,  ,   }
  TGaugeKind = (gkText, gkHorizontalBar, gkVerticalBar, gkPie, gkNeedle);
  {    : , "", "",
    ,  }
  TBevelShape = (bshNone, bshRaised, bshLowered, bshBorder, bshFrame);

  {       
     ,     -  -
    ,       -   
  }
  TsohoGauge = class(TGraphicControl)
  private
    FCaption : string; {04/10/98}
    FMinValue: Longint;
    FMaxValue: Longint;
    FCurValue: Longint;
    FPercent : Integer;
    FKind: TGaugeKind;
    FShowText: Boolean;
    FForeColor: TColor;
    FBackColor: TColor;
    FImage : TBitmap;
    FOverlay : TBltBitmap;
    FInitialized : boolean;
  {$IFNDEF WIN32}
    FProcessMessages : boolean; {only for Win16}
  {$ENDIF}
    FBevel : TBevelShape; {broadcast this nonsense}
    FBevelMargin : Integer;
    FProgressBevel : TBevelShape;
    XWidth, XHeight : Integer;
    procedure ImageBackground(AnImage: TBitmap; var aRect : TRect);
    procedure ImageBevel(AnImage: TBitmap; bvKind : TBevelShape; var aRect : TRect);
    procedure OverlayBackground(AnImage: TBitmap; var aRect : TRect);
    procedure PaintAsText(AnImage: TBitmap; PaintRect: TRect);
    procedure PaintAsNothing(AnImage: TBitmap; PaintRect: TRect);
    procedure PaintAsBar(AnImage: TBitmap; PaintRect: TRect);
    procedure PaintAsPie(AnImage: TBitmap; PaintRect: TRect);
    procedure PaintAsNeedle(AnImage: TBitmap; PaintRect: TRect);
    procedure SetBevel(Value : TBevelShape);
    procedure SetBevelMargin(Value : Integer);
    procedure SetProgressBevel(Value : TBevelShape);
    procedure SetGaugeKind(Value: TGaugeKind);
    procedure SetShowText(Value: Boolean);
    procedure SetForeColor(Value: TColor);
    procedure SetBackColor(Value: TColor);
    procedure SetMinValue(Value: Longint);
    procedure SetMaxValue(Value: Longint);
    procedure SetProgress(Value: Longint);
    procedure SetPercent;
    procedure PaintProgress;
    procedure SetCaption (Value : string);{04/10/98}
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    {     Value }
    procedure AddProgress(Value: Longint);
  published
    {    }
    property Caption : string read FCaption write SetCaption;
    property Align;
    property Color;
    property Enabled;
    {     }
    property PercentDone: Integer read FPercent;
{$IFNDEF WIN32}
    property ProcessMessage : boolean read FProcessMessages write FProcessMessages default True;
{$ENDIF}
    {     }
    property Bevel : TBevelShape read FBevel write SetBevel default bshNone;
    {   }
    property BevelMargin : Integer read FBevelMargin write SetBevelMargin default 0;
    property ProgressBevel : TBevelShape read FProgressBevel write SetProgressBevel default bshNone;
    {   }
    property Kind: TGaugeKind read FKind write SetGaugeKind default gkHorizontalBar;
    {      }
    property ShowText: Boolean read FShowText write SetShowText default true;
    property Font;
    {   }
    property ForeColor: TColor read FForeColor write SetForeColor default clBlack;
    {    }
    property BackColor: TColor read FBackColor write SetBackColor default clWhite;
    {    }
    property MinValue: Longint read FMinValue write SetMinValue default 0;
    {    }
    property MaxValue: Longint read FMaxValue write SetMaxValue default 100;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    {    }
    property Progress: Longint read FCurValue write SetProgress;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

implementation
uses WinProcs, SysUtils;

procedure TBltBitmap.MakeLike;
begin
  Width := ATemplate.Width;
  Height := ATemplate.Height;
end;

{ This function solves for x in the equation "x is y% of z". }
function SolveForX(Y, Z: Longint): Integer;
begin
  SolveForX := Trunc( Z * (Y * 0.01) );
end;

{ This function solves for y in the equation "x is y% of z". }
function SolveForY(X, Z: Longint): Integer;
begin
  if Z = 0
  then SolveForY := 0
  else SolveForY := Trunc( (X * 100) / Z );
end;

{ TsohoGauge }

procedure TsohoGauge.SetCaption (Value : string);{04/10/98}
begin
  if FCaption = Value then exit;
  FCaption := Value;
  PaintProgress;
end;

procedure TsohoGauge.SetPercent;
var oldPercent : integer;
begin
  oldPercent := FPercent;
  FPercent := SolveForY(FCurValue - FMinValue, FMaxValue - FMinValue);
  if FPercent > 100
  then FPercent := 100
  else
    if FPercent < 0
    then FPercent := 0;
  if FPercent <> oldPercent then PaintProgress;
end;

constructor TsohoGauge.Create(AOwner: TComponent);
var bad : boolean;
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csFramed, csOpaque];
  { default values }
  FCaption  := '';
  FMinValue := 0;
  FMaxValue := 100;
  FCurValue := 0;
  FPercent := -1;
  FKind := gkHorizontalBar;
  FShowText := true;
  FForeColor := clBlack;
  FBackColor := clWhite;
  FBevel := bshNone;
  FBevelMargin := 0;
  FProgressBevel := bshNone;
  Width := 50;
  Height := 50;
  XWidth := Width;
  XHeight := Height;
  FInitialized := false;
{$IFNDEF WIN32}
  FProcessMessages := true;
{$ENDIF}
  bad := false;
  FImage := TBltBitmap.Create;
  try
    FImage.Height := Height;
    FImage.Width := Width;
    FOverlay := TBltBitmap.Create;
    try
      FOverlay.MakeLike(FImage);
    except
      on EOutOfMemory do
        bad := true;
      else bad := true;
    end;
    if bad
    then FImage.Free;
  except
    on EOutOfMemory do
    else bad := true;
  end;
  if bad
  then begin
    FImage := nil;
    FOverlay := nil;
  end;
end;

destructor TsohoGauge.Destroy;
begin
  FImage.Free;
  FOverlay.Free;
  inherited Destroy;
end;

procedure TsohoGauge.Paint;
begin
  if Assigned(FImage)
  then begin
    if Height <> XHeight
    then begin
      FImage.Height := Height;
      FOverlay.Height := Height;
      XHeight := Height;
      FInitialized := false;
    end;
    if Width <> XWidth
    then begin
      FImage.Width := Width;
      FOverlay.Width := Width;
      XWidth := Width;
      FInitialized := false;
    end;
    if not FInitialized
    then begin
      dec(FPercent);
      SetPercent;
      FInitialized := true;
    end;
    Canvas.Draw(0, 0, FImage);
  end;
end;

procedure TsohoGauge.OverlayBackground;
begin
  with anImage.Canvas do
  begin
    Brush.Style := bsSolid;
    CopyMode := cmBlackness;
    CopyRect(ClipRect, anImage.Canvas, ClipRect);
    CopyMode := cmSrcCopy;
  end
end;

procedure TsohoGauge.ImageBevel;
begin
  with AnImage.Canvas do
  begin
    Brush.Color := Self.Color;
    FillRect(aRect);
    Brush.Style := bsClear;
    with aRect do
      case bvKind of
        bshBorder : begin
          Pen.Color := clBlack;
          rectangle(left, top, right, bottom);
        end;
        bshRaised : begin
          Pen.Color := clBtnShadow;
          Rectangle(left, top, right, bottom);
          Pen.Color := clBtnHighlight;
          moveto(left, bottom - 1);
          lineto(left, top);
          lineto(right - 1, top);
        end;
        bshLowered : begin
          Pen.Color := clBtnHighlight;
          rectangle(left, top, right, bottom);
          Pen.Color := clBtnShadow;
          moveto(left, bottom - 1);
          lineto(left, top);
          lineto(right - 1, top);
        end;
        bshFrame : begin
          brush.style := bsClear;
          Pen.Color := clBtnHighlight;
          rectangle(left + 1, top + 1, right, bottom);
          Pen.Color := clBtnShadow;
          rectangle(left, top, right - 1, bottom - 1);
          InflateRect(aRect, -1, -1);
        end;
      end;
    if bvKind <> bshNone then InflateRect(aRect, -1, -1);
  end;
end;

procedure TsohoGauge.ImageBackground;
begin
  with AnImage.Canvas do
  begin
    Brush.Color := Self.Color;
    FillRect(aRect);
    Brush.Style := bsClear;
    ImageBevel(anImage, FBevel, aRect);
    InflateRect(aRect, -FBevelMargin, -FBevelMargin);
    ImageBevel(anImage, FProgressBevel, aRect);
    Brush.Style := bsSolid;
    Brush.Color := clBlack;
    FillRect(aRect);
  end;
end;

procedure TsohoGauge.PaintProgress;
var
  PaintRect: TRect;
begin
  if Assigned(FImage)
  then
    with Canvas do
    begin
      PaintRect := ClientRect;
      ImageBackground(FImage, PaintRect);
      OverlayBackground(FOverlay, PaintRect);
      case FKind of
        gkText: PaintAsNothing(FOverlay, PaintRect);
        gkHorizontalBar, gkVerticalBar: PaintAsBar(FOverlay, PaintRect);
        gkPie: PaintAsPie(FOverlay, PaintRect);
        gkNeedle: PaintAsNeedle(FOverlay, PaintRect);
      end;
      FImage.Canvas.CopyMode := cmSrcInvert;
      FImage.Canvas.Draw(0, 0, FOverlay);
      FImage.Canvas.CopyMode := cmSrcCopy;
      if ShowText then PaintAsText(FImage, PaintRect);
      Canvas.CopyMode := cmSrcCopy;
      Canvas.Draw(0, 0, FImage);
    end;
{$IFNDEF WIN32}
  if FProcessMessages
  then Application.ProcessMessages;
{$ENDIF}
end;

procedure TsohoGauge.PaintAsText;
var
  S: string;
  X, Y: Integer;
  PaintRectje : TRect;
  temp : TPenMode;
begin
  PaintRectje := ClientRect;
  OverlayBackground(FOverlay, PaintRectje);
  S := Format('%s %d%%', [FCaption, PercentDone]);
  with FOverlay.Canvas do
  begin
    Brush.Style := bsClear;
    Font := Self.Font;
    Font.Color := clWhite;
    temp := Pen.Mode;
    Pen.Mode := pmNot;
    with PaintRectje do
    begin
      X := (Right - Left + 1 - TextWidth(S)) div 2;
      Y := (Bottom - Top + 1 - TextHeight(S)) div 2;
    end;
    TextRect(PaintRectje, X, Y, S);
    Brush.Style := bsSolid;
    Pen.Mode := temp;
  end;
  AnImage.Canvas.CopyMode := cmSrcInvert;
  AnImage.Canvas.Draw(0, 0, FOverlay);
  AnImage.Canvas.CopyMode := cmSrcCopy;
end;

procedure TsohoGauge.PaintAsNothing;
begin
  with AnImage do
  begin
    Canvas.Brush.Color := BackColor;
    Canvas.FillRect(PaintRect);
  end;
end;

procedure TsohoGauge.PaintAsBar;
var
  delta, result : longint;
begin
  with AnImage.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := BackColor;
    FillRect(PaintRect);
    Pen.Color := ForeColor;
    Pen.Width := 1;
    Brush.Color := ForeColor;
    case FKind of
      gkHorizontalBar:
        with PaintRect do
        begin
          delta := Right - Left;
          delta := SolveForX(PercentDone, delta);
          FillRect(Rect(Left, Top, left + delta, bottom));
        end;
      gkVerticalBar:
        with PaintRect do
        begin
          delta := Bottom - Top;
          result := SolveForX(PercentDone, delta);
          FillRect(Rect(Left, bottom - result, Right, bottom));
        end;
    end;
  end;
end;

procedure TsohoGauge.PaintAsPie;
var
  MiddleX, MiddleY: Integer;
  Angle{, ca, sa}: Double;
  W, H: Integer;
begin
  W := PaintRect.Right - PaintRect.Left;
  H := PaintRect.Bottom - PaintRect.Top;
  if ((w div 2) * 2) = w
  then w := w - 2;
  if ((h div 2) * 2) = w
  then h := h - 2;
  with AnImage.Canvas do
  begin
    Brush.Color := Color;
    FillRect(PaintRect);
    Brush.Color := BackColor;
    Pen.Color := ForeColor;
    Pen.Width := 1;
    with PaintRect do
    begin
      Ellipse(Left, Top, Right, Bottom);
      if PercentDone > 0
      then begin
        Brush.Color := ForeColor;
        MiddleX := left + (W div 2);
        MiddleY := bottom - (H div 2);
        Angle := (Pi * ((PercentDone / 50) + 0.5));
        Pie(Left, Top, Right, Bottom,
          Round(MiddleX * (1 - Cos(Angle))),
          Round(MiddleY * (1 - Sin(Angle))),
          MiddleX, top );
      end;
    end;
  end;
end;

procedure TsohoGauge.PaintAsNeedle;
var
  MiddleX: Integer;
  Angle: Double;
  W, H: Integer;
begin
  with PaintRect do
    with anImage.canvas do
    begin
      W := Right - Left;
      H := Bottom - Top;
      if ((w div 2) * 2) = w
      then w := w - 2;
      if ((h div 2) * 2) = w
      then h := h - 2;
      Brush.Color := Color;
      FillRect(PaintRect);
      Brush.Color := BackColor;
      Pen.Color := ForeColor;
      Pen.Width := 1;
      Pie(left, top, right, top + H * 2, right, Bottom - 1, left, Bottom - 1);
      if PercentDone > 0
      then begin
        Pen.Color := ForeColor;
        MiddleX := W div 2;
        MoveTo(left + MiddleX, Bottom - 1);
        Angle := (Pi * ((PercentDone / 100)));
        LineTo(left + Round(MiddleX * (1 - Cos(Angle))),
          Round(top - 1 + H * (1 - Sin(Angle))));
      end;
    end;
end;

procedure TsohoGauge.SetGaugeKind(Value: TGaugeKind);
begin
  if Value <> FKind
  then begin
    FKind := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetShowText(Value: Boolean);
begin
  if Value <> FShowText
  then begin
    FShowText := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetBevel(Value: TBevelShape);
begin
  if Value <> FBevel
  then begin
    FBevel := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetBevelMargin(Value: Integer);
begin
  if Value <> FBevelMargin
  then begin
    FBevelMargin := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetProgressBevel(Value: TBevelShape);
begin
  if Value <> FProgressBevel
  then begin
    FProgressBevel := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetForeColor(Value: TColor);
begin
  if Value <> FForeColor
  then begin
    FForeColor := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.SetBackColor(Value: TColor);
begin
  if Value <> FBackColor
  then begin
    FBackColor := Value;
    PaintProgress; {Refresh;}
  end;
end;

procedure TsohoGauge.CMFontChanged(var Message: TMessage);
begin
  inherited;
  PaintProgress;
end;

procedure TsohoGauge.CMColorChanged(var Message: TMessage);
begin
  inherited;
  PaintProgress;
end;

procedure TsohoGauge.SetMinValue(Value: Longint);
begin
  if Value <> FMinValue
  then begin
    FMinValue := Value;
    SetPercent;
  end;
end;

procedure TsohoGauge.SetMaxValue(Value: Longint);
begin
  if Value <> FMaxValue
  then begin
    FMaxValue := Value;
    SetPercent;
  end;
end;

procedure TsohoGauge.SetProgress(Value: Longint);
begin
  if (FCurValue <> Value) and (Value >= FMinValue) and (Value <= FMaxValue)
  then begin
    FCurValue := Value;
    SetPercent;
  end;
end;

procedure TsohoGauge.AddProgress(Value: Longint);
begin
  Progress := FCurValue + Value;
  SetPercent;
end;

end.
