{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
    TFormEditor -    run-time
}
unit SoFrmEd;

{$I SOHOLIB.INC}

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Menus;

type
  
  {       run-time mode, 
               
     . TsohoFormEditor     
      
  }
  TsohoFormEditor = class(TComponent)
  private
    { Private declarations }
    FDestroyed: Boolean;
    FOldCloseQuery: TCloseQueryEvent;
    FMarkSize: Integer;
    FHooked: Boolean;
    FPopMenu: TPopupMenu;
    FOldPopup: TPopupMenu;
    FEditors: TList;
    FSelectedList: TList;
    FOnExit: TNotifyEvent;
    procedure BeforePopup(Sender: TObject);
    procedure PopupClick(Sender: TObject);
    procedure DoMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure CreateEditControls;
    procedure DestroyEditControls;
    procedure UpdateEditControls(Sender: TObject);
    procedure DoCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure UpdateSelectedList;
    procedure AlignSelectedControls;
    procedure SizeSelectedControls;
    procedure GridOptions;
    procedure SetEvents; virtual;
    procedure RestoreEvents; virtual;
  public
    {     }
    procedure Edit; virtual;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    {          
      .      TsohoEditControl }
    property MarkSize: Integer read FMarkSize write FMarkSize default 2;
    {        }
    property OnExit: TNotifyEvent read FOnExit write FOnExit;
  end;

  {   -   .  
         ,    
      , ,   ,   
         }
  TsohoEditControl = class(TCustomControl)
  private
    { Private declarations }
    FMoving: Boolean;
    FFocused: Boolean;
    FSelected: Boolean;
    FBoxIndex: Integer;
    FLinkControl: TControl;
    FDrawBoxes: array[1..8] of TRect;
    FStartPoint: TPoint;
    FOldFocus: TRect;
    FMarkSize: Integer;
    FText: string;
    FAfterMoving: TNotifyEvent;
    procedure LinkToControl(CONTROL: TControl);
    procedure SetMarkSize(Value: Integer);
    procedure SetSelected(Value: Boolean);
    procedure SetFocused(Value: Boolean);
  protected
    { Protected declarations }
    procedure SetLinkControlPos(NewPoint: TPoint);
    function CursorOverBoxes(Point: TPoint): Integer;
    procedure WMMouseMove(var message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMLButtonUp(var message: TWMMouse); message WM_LBUTTONUP;
    procedure WMLButtonDown(var message: TWMMouse); message WM_LBUTTONDOWN;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure Paint; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    {       }
    procedure SetPosition;
    constructor Create(AOwner: TComponent); override;
    {   }
    property Color;
    {   }
    property Font;
    {   }
    property LinkControl: TControl read FLinkControl write LinkToControl;
    {          
       }
    property MarkSize: Integer read FMarkSize write SetMarkSize default 2;
    {      -   
          (     
        Shift).    Selected = true }
    property Selected: Boolean read FSelected write SetSelected;
    {     () }
    property Focused: Boolean read FFocused write SetFocused;
    {       }
    property AfterMoving: TNotifyEvent read FAfterMoving write FAfterMoving;
    {        }
    property OnMouseDown;
  end;
  
  
implementation
uses SoUtils, SoCtrls, SoAlignF, SoSizeF, SoCmnCns;

{   }

{    - PointTo-PointFrom }
function PointsDifference(PointFrom, PointTo: TPoint): TPoint;
begin
  Result.X := PointTo.X - PointFrom.X;
  Result.Y := PointTo.Y - PointFrom.Y;
end;

{    }
function RectClientToScreen(AControl: TControl; CLIENT: TRect): TRect;
var Pnt: TPoint;
begin
  Pnt := AControl.ClientToScreen(Point(CLIENT.Left, CLIENT.Top));
  Result.Left := Pnt.X;
  Result.Top := Pnt.Y;
  Result.Right := Result.Left + (CLIENT.Right - CLIENT.Left);
  Result.Bottom := Result.Top + (CLIENT.Bottom - CLIENT.Top);
end;

function NewItemWithTag(const ACaption: string; AShortCut: TShortCut;
    AChecked, AEnabled: Boolean; AOnClick: TNotifyEvent; aTag: Longint): TMenuItem;
begin
  Result := NewItem(ACaption, AShortCut, AChecked, AEnabled, AOnClick,
    0, '');
  Result.Tag := aTag;
end;

{ TsohoFormEditor }
procedure TsohoFormEditor.DoMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var index: Integer;
begin
  if ssShift in Shift then exit;
  for index := 0 to pred(FEditors.Count) do begin
    TsohoEditControl(FEditors[index]).Selected := False;
    TsohoEditControl(FEditors[index]).Focused := False;
  end;
  TsohoEditControl(Sender).Focused := True;
end;

procedure TsohoFormEditor.BeforePopup(Sender: TObject);
var EnableItems: Boolean;
begin
  {     ,    "" }
  EnableItems := not (FPopMenu.PopupComponent = Owner);
  {    BringToFront, SentToBack  GridOptions }
  FPopMenu.Items[0].Enabled := EnableItems and False;
  FPopMenu.Items[1].Enabled := EnableItems and False;
  FPopMenu.Items[3].Enabled := EnableItems;
  FPopMenu.Items[4].Enabled := EnableItems;
  FPopMenu.Items[5].Enabled := EnableItems and False;
end;

procedure TsohoFormEditor.RestoreEvents;
begin
  with TForm(Owner) do begin
    OnCloseQuery := FOldCloseQuery;
    PopupMenu := FOldPopup;
  end;
  FHooked := False;
end;

procedure TsohoFormEditor.SetEvents;
begin
  if FHooked then exit;
  with TForm(Owner) do begin
    FOldCloseQuery := OnCloseQuery;
    OnCloseQuery := DoCloseQuery;
    FOldPopup := PopupMenu;
    PopupMenu := FPopMenu;
  end;
  FDestroyed := False;
  FHooked := True;
end;

procedure TsohoFormEditor.Edit;
begin
  SetEvents;
  CreateEditControls;
end;

procedure TsohoFormEditor.UpdateSelectedList;
var index: Integer;
begin
  FSelectedList.Clear;
  for index := 0 to pred(FEditors.Count) do begin
    if (TsohoEditControl(FEditors[index]).Selected) and
      not (TsohoEditControl(FEditors[index]).Focused) then FSelectedList.Add(FEditors[index]);
    if TsohoEditControl(FEditors[index]).Focused then FSelectedList.Insert(0, FEditors[index]);
  end;
end;

procedure TsohoFormEditor.AlignSelectedControls;
var index: Longint;
  aHorAlign, aVertAlign: Integer; 
  Form                 : TForm;   
  Anchor, CNTRL        : TControl;
begin
  UpdateSelectedList;
  if FSelectedList.Count = 0 then exit;
  if not GetControlsAlign(aHorAlign, aVertAlign) then exit;
  Form := TForm(Owner);
  Anchor := TsohoEditControl(FSelectedList[0]).LinkControl;
  if aHorAlign <> 0 then
    for index := 0 to pred(FSelectedList.Count) do begin
      CNTRL := TsohoEditControl(FSelectedList[index]).LinkControl;
      case aHorAlign of
        1: CNTRL.Left := Anchor.Left; { By lefts          }
        2: CNTRL.Left := Anchor.Left + Anchor.Width div 2 - CNTRL.Width div 2; { By centers       }
        3: CNTRL.Left := Anchor.Left + Anchor.Width - CNTRL.Width; { By rights        }
        4: CNTRL.Left := Form.Width div 2 - CNTRL.Width div 2; { By window center }
      end;
    end;
  if aVertAlign <> 0 then
    for index := 0 to pred(FSelectedList.Count) do begin
      CNTRL := TsohoEditControl(FSelectedList[index]).LinkControl;
      case aVertAlign of
        1: CNTRL.Top := Anchor.Top; { By tops          }
        2: CNTRL.Top := Anchor.Top + Anchor.Height div 2 - CNTRL.Height div 2; { By centers       }
        3: CNTRL.Top := Anchor.Top + Anchor.Height - CNTRL.Height; { By rights        }
        4: CNTRL.Top := Form.Height div 2 - CNTRL.Height div 2; { By window center }
      end;
    end;
end;

procedure TsohoFormEditor.SizeSelectedControls;
var index, AWidth, AHeight: Integer;
  aWidthValue, aHeightValue               : Integer; 
  Anchor, CNTRL                           : TControl;
  MaxWidth, MinWidth, MaxHeight, MinHeight: Integer; 
begin
  UpdateSelectedList;
  if FSelectedList.Count = 0 then exit;
  if not GetControlsSize(AWidth, AHeight, aWidthValue, aHeightValue,
    TControl(FSelectedList[0])) then exit;
  Anchor := TsohoEditControl(FSelectedList[0]).LinkControl;
  MaxWidth := Anchor.Width;
  MinWidth := Anchor.Width;
  MaxHeight := Anchor.Height;
  MinHeight := Anchor.Height;
  for index := 0 to pred(FSelectedList.Count) do begin
    CNTRL := TsohoEditControl(FSelectedList[index]).LinkControl;
    if CNTRL.Width < MinWidth then MinWidth := CNTRL.Width;
    if CNTRL.Width > MaxWidth then MaxWidth := CNTRL.Width;
    if CNTRL.Height < MinHeight then MinHeight := CNTRL.Height;
    if CNTRL.Height > MaxHeight then MaxHeight := CNTRL.Height;
  end;
  if AWidth <> 0 then
    for index := 0 to pred(FSelectedList.Count) do begin
      CNTRL := TsohoEditControl(FSelectedList[index]).LinkControl;
      case AWidth of
        1: CNTRL.Width := MinWidth;
        2: CNTRL.Width := MaxWidth;
        3: CNTRL.Width := aWidthValue;
      end;
    end;
  if AHeight <> 0 then
    for index := 0 to pred(FSelectedList.Count) do begin
      CNTRL := TsohoEditControl(FSelectedList[index]).LinkControl;
      case AHeight of
        1: CNTRL.Height := MinHeight;
        2: CNTRL.Height := MaxHeight;
        3: CNTRL.Height := aHeightValue;
      end;
    end;
end;

procedure TsohoFormEditor.GridOptions;
begin
  { Nothing to do now }
end;

procedure TsohoFormEditor.PopupClick(Sender: TObject);
var index: Integer;
begin
  case (Sender as TMenuItem).Tag of
    {     }
    1: for index := 0 to pred(FEditors.Count) do
        if (TsohoEditControl(FEditors[index]).Selected or
          TsohoEditControl(FEditors[index]).Focused
          ) then TsohoEditControl(FEditors[index]).LinkControl.SendToBack;
      {     }
    2: for index := 0 to pred(FEditors.Count) do
        if (TsohoEditControl(FEditors[index]).Selected or
          TsohoEditControl(FEditors[index]).Focused
          ) then begin
          TsohoEditControl(FEditors[index]).LinkControl.BringToFront;
          TsohoEditControl(FEditors[index]).BringToFront;
        end;
      { Size }
    3: SizeSelectedControls;
      { Align }
    4: AlignSelectedControls;
      { Grid }
    5: GridOptions;
      { Close editor }
    6: begin
      RestoreEvents;
      DestroyEditControls;
      if Assigned(FOnExit) then FOnExit(Self);
    end
  else ErrorMsg(soFrmEditMenuItemNotExists);
  end;
  UpdateEditControls(Self);
end;

procedure TsohoFormEditor.UpdateEditControls(Sender: TObject);
var index: Integer;
begin
  for index := 0 to pred(FEditors.Count) do
    TsohoEditControl(FEditors[index]).SetPosition;
end;

procedure TsohoFormEditor.CreateEditControls;
var index: Integer;
  NewEdit: TsohoEditControl;
  Form   : TForm;           
begin
  Form := TForm(Owner);
  for Index := 0 to pred(Form.ControlCount) do begin
    NewEdit := TsohoEditControl.Create(Self);
    with NewEdit do begin
      Parent := Form.Controls[index].Parent;
      LinkControl := Form.Controls[index];
      AfterMoving := UpdateEditControls;
      MarkSize := FMarkSize;
      PopupMenu := FPopMenu;
      OnMouseDown := DoMouseDown;
    end;
    FEditors.Add(NewEdit);
  end;
end;

procedure TsohoFormEditor.DestroyEditControls;
var index: Integer;
begin
  if FDestroyed then exit;
  FDestroyed := True;
  for index := pred(TForm(Owner).ControlCount) downto 0 do
    if TForm(Owner).Controls[index] is TsohoEditControl then
      TForm(Owner).Controls[index].Free;
  FEditors.Clear;
  FSelectedList.Clear;
end;

procedure TsohoFormEditor.DoCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  CanClose := False;
end;

constructor TsohoFormEditor.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPopMenu := NewPopupMenu(Self, '', paLeft, True,[
    NewItemWithTag(soFrmEditBack, 0, False, True, PopupClick, 1),
    NewItemWithTag(soFrmEditFront, 0, False, True, PopupClick, 2),
    NewItemWithTag('-', 0, False, True, nil, 0),
    NewItemWithTag(soFrmEditSize, 0, False, True, PopupClick, 3),
    NewItemWithTag(soFrmEditAlign, 0, False, True, PopupClick, 4),
    NewItemWithTag(soFrmEditGrid, 0, False, True, PopupClick, 5),
    NewItemWithTag(soFrmEditStop, 0, False, True, PopupClick, 6)
    ]);
  FPopMenu.OnPopup := BeforePopup;
  FDestroyed := False;
  FHooked := False;
  FMarkSize := 2;
  FEditors := TList.Create;
  FSelectedList := TList.Create;
end;

destructor TsohoFormEditor.Destroy;
begin
  if not (csDestroying in ComponentState) then begin
    RestoreEvents;
    DestroyEditControls;
  end;
  FEditors.Free;
  FSelectedList.Free;
  inherited Destroy;
end;

{ TsohoEditControl }
procedure TsohoEditControl.SetSelected(Value: Boolean);
begin
  FSelected := Value;
  Repaint;
end;

procedure TsohoEditControl.SetFocused(Value: Boolean);
begin
  FFocused := Value;
  Repaint;
end;

procedure TsohoEditControl.SetMarkSize(Value: Integer);
begin
  if Value < 2 then Value := 2;
  if Value > 16 then Value := 16;
  if FMarkSize = Value then exit;
  FMarkSize := Value;
  SetPosition;
  Repaint;
end;

procedure TsohoEditControl.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if (AComponent = FLinkControl) and (Operation = opRemove) then FLinkControl := nil;
  inherited Notification(AComponent, Operation);
end;

procedure TsohoEditControl.LinkToControl(CONTROL: TControl);
begin
  FLinkControl := CONTROL;
  FText := 'No name';
  if IsProperty(FLinkControl, 'Name') then
    FText := GetStringProperty(FLinkControl, 'Name');
  if IsProperty(FLinkControl, 'Text') then
    FText := GetStringProperty(FLinkControl, 'Text');
  if IsProperty(FLinkControl, 'Caption') then
    FText := GetStringProperty(FLinkControl, 'Caption');
  SetPosition;
end;

procedure TsohoEditControl.SetPosition;
begin
  if FLinkControl = nil then exit;
  Top := FLinkControl.Top - FMarkSize;
  Left := FLinkControl.Left - FMarkSize;
  Width := FLinkControl.Width + 2 * FMarkSize;
  Height := FLinkControl.Height + 2 * FMarkSize;
  FDrawBoxes[1] := Rect(0, 0, 2 * FMarkSize, 2 * FMarkSize);
  FDrawBoxes[2] := Rect(Width div 2 - FMarkSize, 0, Width div 2 + FMarkSize, 2 * FMarkSize);
  FDrawBoxes[3] := Rect(Width - 2 * FMarkSize, 0, Width, 2 * FMarkSize);
  FDrawBoxes[4] := Rect(0, Height div 2 - FMarkSize, 2 * FMarkSize, Height div 2 + FMarkSize);
  FDrawBoxes[5] := Rect(Width - 2 * FMarkSize, Height div 2 - FMarkSize, Width,
    Height div 2 + FMarkSize);
  FDrawBoxes[6] := Rect(0, Height - 2 * FMarkSize, 2 * FMarkSize, Height);
  FDrawBoxes[7] := Rect(Width div 2 - FMarkSize, Height - 2 * FMarkSize,
    Width div 2 + FMarkSize, Height);
  FDrawBoxes[8] := Rect(Width - 2 * FMarkSize, Height - 2 * FMarkSize, Width, Height);
end;

function TsohoEditControl.CursorOverBoxes(Point: TPoint): Integer;
var index: Integer;
begin
  Result := 0;
  for index := Low(FDrawBoxes) to High(FDrawBoxes) do
    if PtInRect(FDrawBoxes[index], Point) then begin
      Result := index;
      exit;
    end;
end;

procedure TsohoEditControl.WMLButtonUp(var message: TWMMouse);
begin
  if FMoving then ReleaseCapture;
  inherited;
  if FLinkControl = nil then exit;
  if FMoving then SetLinkControlPos(Point(message.XPos, message.YPos));
end;

procedure TsohoEditControl.WMMouseMove(var message: TWMMouseMove);
const Cursors: array[0..8] of Integer = (
  crSize,
    crSizeNWSE, crSizeNS, crSizeNESW,
    crSizeWE, crSizeWE,
    crSizeNESW, crSizeNS, crSizeNWSE
    );
var Delta: TPoint;
  Focus   : TRect;
  ScreenDC: HDC;  
begin
  if not FMoving then begin
    FBoxIndex := CursorOverBoxes(Point(message.XPos, message.YPos));
    Cursor := Cursors[FBoxIndex];
    inherited;
    exit;
  end;
  Delta := PointsDifference(FStartPoint, Point(message.XPos, message.YPos));
  case FBoxIndex of
    0: Focus := Bounds(Left + Delta.X, Top + Delta.Y, Width, Height);
    1: Focus := Bounds(Left + Delta.X, Top + Delta.Y,
      Width - Delta.X, Height - Delta.Y);
    2: Focus := Bounds(Left, Top + Delta.Y, Width, Height - Delta.Y);
    3: Focus := Bounds(Left, Top + Delta.Y, Width + Delta.X, Height - Delta.Y);
    4: Focus := Bounds(Left + Delta.X, Top, Width - Delta.X, Height);
    5: Focus := Bounds(Left, Top, Width + Delta.X, Height);
    6: Focus := Bounds(Left + Delta.X, Top, Width - Delta.X, Height + Delta.Y);
    7: Focus := Bounds(Left, Top, Width, Height + Delta.Y);
    8: Focus := Bounds(Left, Top, Width + Delta.X, Height + Delta.Y);
  else Focus := FOldFocus;
  end;
  ScreenDC := GetDC(0);
  WinProcs.DrawFocusRect(ScreenDC, RectClientToScreen(Parent, FOldFocus));
  WinProcs.DrawFocusRect(ScreenDC, RectClientToScreen(Parent, Focus));
  FOldFocus := Focus;
  ReleaseDC(0, ScreenDC);
  inherited;
end;

procedure TsohoEditControl.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  FFocused := not (ssShift in Shift);
  FSelected := not FFocused;
  Repaint;
end;

procedure TsohoEditControl.WMLButtonDown(var message: TWMMouse);
var ScreenDC: HDC;
begin
  inherited;
  if FLinkControl = nil then exit;
  SetCapture(Handle);
  FMoving := True;
  FStartPoint := Point(message.XPos, message.YPos);
  FOldFocus := Bounds(Left, Top, Width, Height);
  ScreenDC := GetDC(0);
  WinProcs.DrawFocusRect(ScreenDC, RectClientToScreen(Parent, FOldFocus));
  ReleaseDC(0, ScreenDC);
end;

procedure TsohoEditControl.SetLinkControlPos;
var ScreenDC: HDC;
begin
  ScreenDC := GetDC(0);
  WinProcs.DrawFocusRect(ScreenDC, RectClientToScreen(Parent, FOldFocus));
  ReleaseDC(0, ScreenDC);
  FMoving := False;
  with FOldFocus do begin
    if Bottom < Top then SwapInteger(Bottom, Top);
    if Right < Left then SwapInteger(Right, Left);
    FLinkControl.SetBounds(Left + FMarkSize, Top + FMarkSize, Right - Left - 2 * FMarkSize,
      Bottom - Top - 2 * FMarkSize);
  end;
  SetPosition;
  if Assigned(FAfterMoving) then FAfterMoving(Self);
end;

procedure TsohoEditControl.Paint;
var index: Integer;
  X, Y : Integer;
  fRect: TRect;  
begin
  if FLinkControl = nil then exit;
  with Canvas do begin
    BRUSH.Color := Self.Color;
    FillRect(ClientRect);
    BRUSH.Color := clBlack;
    fRect := ClientRect;
    FrameRect(fRect);
    if FFocused then BRUSH.Color := clBlue;
    if FSelected then BRUSH.Color := clWhite;
    InflateRect(fRect, - 3, - 3);
    if FFocused or FSelected then FrameRect(fRect);
    BRUSH.Color := Self.Color;
    Font.Assign(Self.Font);
    X := HorCenter(fRect, TextWidth(FText));
    Y := VertCenter(fRect, TextHeight(FText));
    TextOut(X, Y, FText);
    BRUSH.Color := clRed;
    for index := Low(FDrawBoxes) to High(FDrawBoxes) do
      FillRect(FDrawBoxes[index]);
  end;
end;

constructor TsohoEditControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLinkControl := nil;
  FMoving := False;
  FBoxIndex := 0;
  FOldFocus := Rect(0, 0, 0, 0);
  FMarkSize := 2;
end;

end.

