{*******************************************************}
{                                                       }
{           Delphi Visual Component Library             }
{                                                       }
{          Copyright (c) 1996-1997 AllexSoft            }
{                   Written by VSM                      }
{                                                       }
{                   SOHO Components                     }
{                                                       }
{*******************************************************}
{
     - ,
   ,   ...    
       TsohoList, TsohoIntegerList, TsohoFloatList
     ,     
}
unit SoClass;

{$I SOHOLIB.INC }

interface
uses Classes;

type

  {    OnNewItem, OnDeleteItem  TsohoList }
  TOnPntItemNotify = procedure (Sender : TObject; Item : pointer) of object;
  {    OnChangeItem  TsohoList }
  TOnPntItemChange = procedure (Sender : TObject; OldItem, NewItem : pointer) of object;

  {        }
  TsohoList = class(TObject)
  private
    FList: PPointerList;
    FCount: Integer;
    FCapacity: Integer;
    FOnChange : TNotifyEvent;
    FOnNewItem : TOnPntItemNotify;
    FOnDeleteItem : TOnPntItemNotify;
    FOnClear      : TNotifyEvent;
    FOnChangeItem : TOnPntItemChange;
  protected
    procedure Error; virtual;
    function Get(Index: Integer): Pointer;
    procedure Grow; virtual;
    procedure Put(Index: Integer; Item: Pointer);
    procedure SetCapacity(NewCapacity: Integer);
    procedure SetCount(NewCount: Integer);
  public
    destructor Destroy; override;
    function Add(Item: Pointer): Integer;
    procedure Clear;
    procedure Delete(Index: Integer);
    procedure Exchange(Index1, Index2: Integer);
    function Expand: TsohoList;
    function First: Pointer;
    function IndexOf(Item: Pointer): Integer;
    procedure Insert(Index: Integer; Item: Pointer);
    function Last: Pointer;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: Pointer): Integer;
    procedure Pack;
    property Capacity: Integer read FCapacity write SetCapacity;
    property Count: Integer read FCount write SetCount;
    property Items[Index: Integer]: Pointer read Get write Put; default;
    property List: PPointerList read FList;
    property OnNewItem : TOnPntItemNotify read FOnNewItem write FOnNewItem;
    property OnDeleteItem : TOnPntItemNotify read FOnDeleteItem write FOnDeleteItem;
    property OnClearItems : TNotifyEvent read FOnClear write FOnClear;
    property OnChangeItem : TOnPntItemChange read FOnChangeItem write FOnChangeItem;
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  {    OnNewItem, OnDeleteItem  TsohoIntegerList }
  TOnIntItemNotify = procedure (Sender : TObject; Item : LongInt) of object;
  {    OnChangeItem  TsohoIntegerList }
  TOnIntItemChange = procedure (Sender : TObject; OldItem, NewItem : LongInt) of object;

  PIntegerList = ^TIntegerList;
  TIntegerList = array[0..MaxListSize - 1] of LongInt;

  {     (LongInt)     }
  TsohoIntegerList = class(TObject)
  private
    FList: PIntegerList;
    FCount: Integer;
    FCapacity: Integer;
    FOnChange : TNotifyEvent;
    FOnNewItem : TOnIntItemNotify;
    FOnDeleteItem : TOnIntItemNotify;
    FOnClear      : TNotifyEvent;
    FOnChangeItem : TOnIntItemChange;
  protected
    procedure Error; virtual;
    function Get(Index: Integer): LongInt;
    procedure Grow; virtual;
    procedure Put(Index: Integer; Item: LongInt);
    procedure SetCapacity(NewCapacity: Integer);
    procedure SetCount(NewCount: Integer);
  public
    destructor Destroy; override;
    function Add(Item: LongInt): Integer;
    procedure SaveToFile (const FileName : string);
    procedure LoadFromFile (const FileName : string);
    procedure Clear;
    procedure Delete(Index: Integer);
    procedure Exchange(Index1, Index2: Integer);
    function Expand: TsohoIntegerList;
    function First: LongInt;
    function IndexOf(Item: LongInt): Integer;
    procedure Insert(Index: Integer; Item: LongInt);
    function Last: LongInt;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: LongInt): Integer;
    procedure Assign (Donor : TsohoIntegerList);
    property Capacity: Integer read FCapacity write SetCapacity;
    property Count: Integer read FCount write SetCount;
    property Items[Index: Integer]: LongInt read Get write Put; default;
    property List: PIntegerList read FList;
    property OnNewItem : TOnIntItemNotify read FOnNewItem write FOnNewItem;
    property OnDeleteItem : TOnIntItemNotify read FOnDeleteItem write FOnDeleteItem;
    property OnClearItems : TNotifyEvent read FOnClear write FOnClear;
    property OnChangeItem : TOnIntItemChange read FOnChangeItem write FOnChangeItem;
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  PFloatList = ^TFloatList;
  TFloatList = array[0..MaxListSize - 1] of Double;

  {    OnNewItem, OnDeleteItem  TsohoFloatList }
  TOnFltItemNotify = procedure (Sender : TObject; Item : Double) of object;
  {    OnChangeItem  TsohoFloatList }
  TOnFltItemChange = procedure (Sender : TObject; OldItem, NewItem : Double) of object;

  {     (Double)     }
  TsohoFloatList = class(TObject)
  private
    FList: PFloatList;
    FCount: Integer;
    FCapacity: Integer;
    FOnChange : TNotifyEvent;
    FOnNewItem : TOnFltItemNotify;
    FOnDeleteItem : TOnFltItemNotify;
    FOnClear      : TNotifyEvent;
    FOnChangeItem : TOnFltItemChange;
  protected
    procedure Error; virtual;
    function Get(Index: Integer): Double;
    procedure Grow; virtual;
    procedure Put(Index: Integer; Item: Double);
    procedure SetCapacity(NewCapacity: Integer);
    procedure SetCount(NewCount: Integer);
  public
    destructor Destroy; override;
    function Add(Item: Double): Integer;
    procedure Clear;
    procedure SaveToFile (const FileName : string);
    procedure LoadFromFile (const FileName : string);
    procedure Delete(Index: Integer);
    procedure Exchange(Index1, Index2: Integer);
    function Expand: TsohoFloatList;
    function First: Double;
    function IndexOf(Item: Double): Integer;
    procedure Insert(Index: Integer; Item: Double);
    function Last: Double;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: Double): Integer;
    procedure Assign (Donor : TsohoIntegerList);
    property Capacity: Integer read FCapacity write SetCapacity;
    property Count: Integer read FCount write SetCount;
    property Items[Index: Integer]: Double read Get write Put; default;
    property List: PFloatList read FList;
    property OnNewItem : TOnFltItemNotify read FOnNewItem write FOnNewItem;
    property OnDeleteItem : TOnFltItemNotify read FOnDeleteItem write FOnDeleteItem;
    property OnClearItems : TNotifyEvent read FOnClear write FOnClear;
    property OnChangeItem : TOnFltItemChange read FOnChangeItem write FOnChangeItem;
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

implementation
uses SoUtils, SysUtils;

const
  SListIndexError = 0;

procedure ListError(Ident: Word);
begin
  raise Exception.Create(LoadStr(Ident));
  // ErrorRes(Ident);
end;

{ TsohoList }
destructor TsohoList.Destroy;
begin
  Clear;
end;

function TsohoList.Add(Item: Pointer): Integer;
begin
  Result := FCount;
  if Result = FCapacity then Grow;
  FList^[Result] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoList.Clear;
begin
  SetCount(0);
  SetCapacity(0);
  if Assigned(FOnClear) then FOnClear(Self);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoList.Delete(Index: Integer);
var Item : pointer;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Item := Items[Index];
  Dec(FCount);
  if Index < FCount then
    System.Move(FList^[Index + 1], FList^[Index],
      (FCount - Index) * SizeOf(Pointer));
  if Assigned(FOnDeleteItem) then FOnDeleteItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoList.Error;
begin
  ListError(SListIndexError);
end;

procedure TsohoList.Exchange(Index1, Index2: Integer);
var
  Item: Pointer;
begin
  if (Index1 < 0) or (Index1 >= FCount) or
    (Index2 < 0) or (Index2 >= FCount) then Error;
  Item := FList^[Index1];
  FList^[Index1] := FList^[Index2];
  FList^[Index2] := Item;
end;

function TsohoList.Expand: TsohoList;
begin
  if FCount = FCapacity then Grow;
  Result := Self;
end;

function TsohoList.First: Pointer;
begin
  Result := Get(0);
end;

function TsohoList.Get(Index: Integer): Pointer;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Result := FList^[Index];
end;

procedure TsohoList.Grow;
var
  Delta: Integer;
begin
  if FCapacity > 8 then Delta := 16 else
    if FCapacity > 4 then Delta := 8 else
      Delta := 4;
  SetCapacity(FCapacity + Delta);
end;

function TsohoList.IndexOf(Item: Pointer): Integer;
begin
  Result := 0;
  while (Result < FCount) and (FList^[Result] <> Item) do Inc(Result);
  if Result = FCount then Result := -1;
end;

procedure TsohoList.Insert(Index: Integer; Item: Pointer);
begin
  if (Index < 0) or (Index > FCount) then Error;
  if FCount = FCapacity then Grow;
  if Index < FCount then
    System.Move(FList^[Index], FList^[Index + 1],
      (FCount - Index) * SizeOf(Pointer));
  FList^[Index] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoList.Last: Pointer;
begin
  Result := Get(FCount - 1);
end;

procedure TsohoList.Move(CurIndex, NewIndex: Integer);
var
  Item: Pointer;
begin
  if CurIndex <> NewIndex then
  begin
    if (NewIndex < 0) or (NewIndex >= FCount) then Error;
    Item := Get(CurIndex);
    Delete(CurIndex);
    Insert(NewIndex, Item);
  end;
end;

procedure TsohoList.Put(Index: Integer; Item: Pointer);
var OldItem : pointer;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  OldItem := Items[Index];
  FList^[Index] := Item;
  if Assigned(FOnChangeItem) then FOnChangeItem(Self, OldItem, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoList.Remove(Item: Pointer): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then Delete(Result);
end;

procedure TsohoList.Pack;
var
  I: Integer;
begin
  for I := FCount - 1 downto 0 do if Items[I] = nil then Delete(I);
end;

procedure TsohoList.SetCapacity(NewCapacity: Integer);
var
  NewList: PPointerList;
begin
  if (NewCapacity < FCount) or (NewCapacity > MaxListSize) then Error;
  if NewCapacity <> FCapacity then
  begin
    if NewCapacity = 0 then NewList := nil else
    begin
      GetMem(NewList, NewCapacity * SizeOf(Pointer));
      if FCount <> 0 then
        System.Move(FList^, NewList^, FCount * SizeOf(Pointer));
    end;
    if FCapacity <> 0 then FreeMem(FList, FCapacity * SizeOf(Pointer));
    FList := NewList;
    FCapacity := NewCapacity;
  end;
end;

procedure TsohoList.SetCount(NewCount: Integer);
begin
  if (NewCount < 0) or (NewCount > MaxListSize) then Error;
  if NewCount > FCapacity then SetCapacity(NewCount);
  if NewCount > FCount then
    FillChar(FList^[FCount], (NewCount - FCount) * SizeOf(Pointer), 0);
  FCount := NewCount;
end;

{ TsohoIntegerList }
procedure TsohoIntegerList.Assign (Donor : TsohoIntegerList);
var Index : LongInt;
begin
  Clear;
  for Index := 0 to pred(Donor.Count) do Add(Donor[Index]);
end;

procedure TsohoIntegerList.SaveToFile (const FileName : string);
var OutFile: TextFile;
    Index: Longint;
begin
  try
    AssignFile(OutFile, FileName);
    try
      Rewrite(OutFile);
      for Index := 0 to pred(Count) do
        writeln(OutFile, IntToStr(Items[index]));
    except ErrorMsg('TohoIntegerList:     - ' + FileName);
    end;
  finally
    CloseFile(OutFile);
  end;
end;

procedure TsohoIntegerList.LoadFromFile (const FileName : string);
var InFile: TextFile;
  NewStr: string;
  Error : Boolean;
begin
  if not FileExists(FileName) then exit;
  Clear;
  try
    AssignFile(InFile, FileName);
    try
      Reset(InFile);
      while not EOF(InFile) do begin
        Readln(InFile, NewStr);
        Add(WStrToInt(NewStr, Error));
      end;
    except ErrorMsg('TsohoIntegerList:     - ' + FileName);
    end;
  finally
    CloseFile(InFile);
  end;
end;

destructor TsohoIntegerList.Destroy;
begin
  Clear;
end;

function TsohoIntegerList.Add(Item: LongInt): Integer;
begin
  Result := FCount;
  if Result = FCapacity then Grow;
  FList^[Result] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoIntegerList.Clear;
begin
  SetCount(0);
  SetCapacity(0);
  if Assigned(FOnClear) then FOnClear(Self);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoIntegerList.Delete(Index: Integer);
var Item : LongInt;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Item := Items[Index];
  Dec(FCount);
  if Index < FCount then
    System.Move(FList^[Index + 1], FList^[Index],
      (FCount - Index) * SizeOf(LongInt));
  if Assigned(FOnDeleteItem) then FOnDeleteItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoIntegerList.Error;
begin
  ListError(SListIndexError);
end;

procedure TsohoIntegerList.Exchange(Index1, Index2: Integer);
var
  Item: LongInt;
begin
  if (Index1 < 0) or (Index1 >= FCount) or
    (Index2 < 0) or (Index2 >= FCount) then Error;
  Item := FList^[Index1];
  FList^[Index1] := FList^[Index2];
  FList^[Index2] := Item;
end;

function TsohoIntegerList.Expand: TsohoIntegerList;
begin
  if FCount = FCapacity then Grow;
  Result := Self;
end;

function TsohoIntegerList.First: LongInt;
begin
  Result := Get(0);
end;

function TsohoIntegerList.Get(Index: Integer): LongInt;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Result := FList^[Index];
end;

procedure TsohoIntegerList.Grow;
var
  Delta: Integer;
begin
  if FCapacity > 8 then Delta := 16 else
    if FCapacity > 4 then Delta := 8 else
      Delta := 4;
  SetCapacity(FCapacity + Delta);
end;

function TsohoIntegerList.IndexOf(Item: LongInt): Integer;
begin
  Result := 0;
  while (Result < FCount) and (FList^[Result] <> Item) do Inc(Result);
  if Result = FCount then Result := -1;
end;

procedure TsohoIntegerList.Insert(Index: Integer; Item: LongInt);
begin
  if (Index < 0) or (Index > FCount) then Error;
  if FCount = FCapacity then Grow;
  if Index < FCount then
    System.Move(FList^[Index], FList^[Index + 1],
      (FCount - Index) * SizeOf(LongInt));
  FList^[Index] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoIntegerList.Last: LongInt;
begin
  Result := Get(FCount - 1);
end;

procedure TsohoIntegerList.Move(CurIndex, NewIndex: Integer);
var
  Item: LongInt;
begin
  if CurIndex <> NewIndex then
  begin
    if (NewIndex < 0) or (NewIndex >= FCount) then Error;
    Item := Get(CurIndex);
    Delete(CurIndex);
    Insert(NewIndex, Item);
  end;
end;

procedure TsohoIntegerList.Put(Index: Integer; Item: LongInt);
var OldItem : LongInt;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  OldItem := Items[Index];
  FList^[Index] := Item;
  if Assigned(FOnChangeItem) then FOnChangeItem(Self, OldItem, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoIntegerList.Remove(Item: LongInt): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then Delete(Result);
end;

procedure TsohoIntegerList.SetCapacity(NewCapacity: Integer);
var
  NewList: PIntegerList;
begin
  if (NewCapacity < FCount) or (NewCapacity > MaxListSize) then Error;
  if NewCapacity <> FCapacity then
  begin
    if NewCapacity = 0 then NewList := nil else
    begin
      GetMem(NewList, NewCapacity * SizeOf(LongInt));
      if FCount <> 0 then
        System.Move(FList^, NewList^, FCount * SizeOf(LongInt));
    end;
    if FCapacity <> 0 then FreeMem(FList, FCapacity * SizeOf(LongInt));
    FList := NewList;
    FCapacity := NewCapacity;
  end;
end;

procedure TsohoIntegerList.SetCount(NewCount: Integer);
begin
  if (NewCount < 0) or (NewCount > MaxListSize) then Error;
  if NewCount > FCapacity then SetCapacity(NewCount);
  if NewCount > FCount then
    FillChar(FList^[FCount], (NewCount - FCount) * SizeOf(LongInt), 0);
  FCount := NewCount;
end;

{ TsohoFloatList }
procedure TsohoFloatList.Assign (Donor : TsohoIntegerList);
var Index : LongInt;
begin
  Clear;
  for Index := 0 to pred(Donor.Count) do Add(Donor[Index]);
end;

procedure TsohoFloatList.SaveToFile (const FileName : string);
var OutFile: TextFile;
    Index: Longint;
begin
  try
    AssignFile(OutFile, FileName);
    try
      Rewrite(OutFile);
      for Index := 0 to pred(Count) do
        writeln(OutFile, SysUtils.FloatToStrF(Items[index], ffGeneral, 15, 4));
    except ErrorMsg('TohoIntegerList:     - ' + FileName);
    end;
  finally
    CloseFile(OutFile);
  end;
end;

procedure TsohoFloatList.LoadFromFile (const FileName : string);
var InFile: TextFile;
  NewStr: string;
  Error : Boolean;
begin
  if not FileExists(FileName) then exit;
  Clear;
  try
    AssignFile(InFile, FileName);
    try
      Reset(InFile);
      while not EOF(InFile) do begin
        Readln(InFile, NewStr);
        Add(WStrToFloat(NewStr, Error));
      end;
    except ErrorMsg('TsohoIntegerList:     - ' + FileName);
    end;
  finally
    CloseFile(InFile);
  end;
end;

destructor TsohoFloatList.Destroy;
begin
  Clear;
end;

function TsohoFloatList.Add(Item: Double): Integer;
begin
  Result := FCount;
  if Result = FCapacity then Grow;
  FList^[Result] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoFloatList.Clear;
begin
  SetCount(0);
  SetCapacity(0);
  if Assigned(FOnClear) then FOnClear(Self);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoFloatList.Delete(Index: Integer);
var Item : Double;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Item := Items[Index];
  Dec(FCount);
  if Index < FCount then
    System.Move(FList^[Index + 1], FList^[Index],
      (FCount - Index) * SizeOf(Double));
  if Assigned(FOnDeleteItem) then FOnDeleteItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TsohoFloatList.Error;
begin
  ListError(SListIndexError);
end;

procedure TsohoFloatList.Exchange(Index1, Index2: Integer);
var Item: Double;
begin
  if (Index1 < 0) or (Index1 >= FCount) or
    (Index2 < 0) or (Index2 >= FCount) then Error;
  Item := FList^[Index1];
  FList^[Index1] := FList^[Index2];
  FList^[Index2] := Item;
end;

function TsohoFloatList.Expand: TsohoFloatList;
begin
  if FCount = FCapacity then Grow;
  Result := Self;
end;

function TsohoFloatList.First: Double;
begin
  Result := Get(0);
end;

function TsohoFloatList.Get(Index: Integer): Double;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Result := FList^[Index];
end;

procedure TsohoFloatList.Grow;
var Delta: Integer;
begin
  if FCapacity > 8 then Delta := 16 else
    if FCapacity > 4 then Delta := 8 else
      Delta := 4;
  SetCapacity(FCapacity + Delta);
end;

function TsohoFloatList.IndexOf(Item: Double): Integer;
begin
  Result := 0;
  while (Result < FCount) and (FList^[Result] <> Item) do Inc(Result);
  if Result = FCount then Result := -1;
end;

procedure TsohoFloatList.Insert(Index: Integer; Item: Double);
begin
  if (Index < 0) or (Index > FCount) then Error;
  if FCount = FCapacity then Grow;
  if Index < FCount then
    System.Move(FList^[Index], FList^[Index + 1],
      (FCount - Index) * SizeOf(Double));
  FList^[Index] := Item;
  Inc(FCount);
  if Assigned(FOnNewItem) then FOnNewItem(Self, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoFloatList.Last: Double;
begin
  Result := Get(FCount - 1);
end;

procedure TsohoFloatList.Move(CurIndex, NewIndex: Integer);
var Item: Double;
begin
  if CurIndex <> NewIndex then begin
    if (NewIndex < 0) or (NewIndex >= FCount) then Error;
    Item := Get(CurIndex);
    Delete(CurIndex);
    Insert(NewIndex, Item);
  end;
end;

procedure TsohoFloatList.Put(Index: Integer; Item: Double);
var OldItem : Double;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  OldItem := Items[Index];
  FList^[Index] := Item;
  if Assigned(FOnChangeItem) then FOnChangeItem(Self, OldItem, Item);
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TsohoFloatList.Remove(Item: Double): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then Delete(Result);
end;

procedure TsohoFloatList.SetCapacity(NewCapacity: Integer);
var NewList: PFloatList;
begin
  if (NewCapacity < FCount) or (NewCapacity > MaxListSize) then Error;
  if NewCapacity <> FCapacity then
  begin
    if NewCapacity = 0 then NewList := nil else
    begin
      GetMem(NewList, NewCapacity * SizeOf(Double));
      if FCount <> 0 then
        System.Move(FList^, NewList^, FCount * SizeOf(Double));
    end;
    if FCapacity <> 0 then FreeMem(FList, FCapacity * SizeOf(Double));
    FList := NewList;
    FCapacity := NewCapacity;
  end;
end;

procedure TsohoFloatList.SetCount(NewCount: Integer);
begin
  if (NewCount < 0) or (NewCount > MaxListSize) then Error;
  if NewCount > FCapacity then SetCapacity(NewCount);
  if NewCount > FCount then
    FillChar(FList^[FCount], (NewCount - FCount) * SizeOf(Double), 0);
  FCount := NewCount;
end;

end.
