unit GridSizer;

interface

Uses Sizer, Grids, Classes, Controls, Messages;

Type
  TGridSizerHelper=Class(TSizerHelper)
  private
    FOldWindowProc:TWndMethod;
    FGrid:TDrawGrid;
    FDefWidth :Integer;
    FDefHeight:Integer;
    FColWidths:TList;
    FRowHeights:TList;
    FractionX :Real;
    FractionY :Real;
    procedure NewWindowProc (var Message: TMessage);
  public
    Constructor Create(AOwner:TComponent); Override;
    Destructor Destroy; Override;
    Procedure ForceColResize;
    Procedure ForceRowResize;
  protected
    Procedure DelInfo;
    Procedure MakeInfo;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    Procedure SizerResized(AFractionX,AFractionY:Real; Options:TSizerOptions); Override;
    Procedure SetGrid(AGrid:TDrawGrid);
  published
    Property  Grid:TDrawGrid read FGrid write SetGrid;
  End;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('ATSystems', [TGridSizerHelper]);
end;

{******************* Grid Sizer *********************}

Procedure TGridSizerHelper.DelInfo;
Begin
  While FColWidths.Count<>0 Do Begin
    Dispose(FColWidths[0]);
    FColWidths.Delete(0);
  End;
  While FRowHeights.Count<>0 Do Begin
    Dispose(FRowHeights[0]);
    FRowHeights.Delete(0);
  End;
  If (Assigned(FOldWindowProc)) And (Assigned(FGrid)) Then FGrid.WindowProc:=FOldWindowProc;
End;

Procedure TGridSizerHelper.MakeInfo;
Var TempInt:^Real;
    Loop   :Integer;
Begin
  FDefWidth :=FGrid.DefaultColWidth;
  FDefHeight:=FGrid.DefaultRowHeight;
  For Loop:=1 To FGrid.ColCount Do Begin
    New(TempInt);
    TempInt^:=FGrid.ColWidths[Pred(Loop)];
    FColWidths.Add(TempInt);
  End;
  For Loop:=1 To FGrid.RowCount Do Begin
    New(TempInt);
    TempInt^:=FGrid.RowHeights[Pred(Loop)];
    FRowHeights.Add(TempInt);
  End;
  FOldWindowProc:=FGrid.WindowProc;
  FGrid.WindowProc:=NewWindowProc;
End;

Procedure TGridSizerHelper.SetGrid(AGrid:TDrawGrid);
Begin
  If FGrid<>AGrid Then Begin
    FGrid:=AGrid;
    DelInfo;
    If AGrid<>Nil Then MakeInfo;
  End;
End;

Procedure TGridSizerHelper.SizerResized(AFractionX,AFractionY:Real; Options:TSizerOptions);
Var Loop:Integer;
Begin
  FractionX:=AFractionX;
  FractionY:=AFractionY;
  FGrid.DefaultColWidth :=Round(FDefWidth*FractionX);
  FGrid.DefaultRowHeight:=Round(FDefHeight*FractionY);
  If (FDefWidth<>0) And (FDefHeight<>0) And (FGrid<>Nil) Then Begin
    For Loop:=1 To FGrid.ColCount Do
      If Loop<=FColWidths.Count Then FGrid.ColWidths[Pred(Loop)]:=Round(Real(FColWidths[Pred(Loop)]^)*FractionX)
{      Else FGrid.ColWidths[Pred(Loop)]:=Round(FColWidth*FractionX)};
    For Loop:=1 To FGrid.RowCount Do
      If Loop<=FRowHeights.Count Then FGrid.RowHeights[Pred(Loop)]:=Round(Real(FRowHeights[Pred(Loop)]^)*FractionY)
{      Else FGrid.RowHeights[Pred(Loop)]:=Round(FRowHeight*FractionY)};
  End;
End;

procedure TGridSizerHelper.ForceRowResize;
Type PReal=^Real;
Var Loop,Temp:Integer;
begin
  For Loop:=1 To FGrid.RowCount Do
    If Loop<=FRowHeights.Count Then Begin
      Temp:=Round(Real(FRowHeights[Pred(Loop)]^)*FractionY);
      If Temp<>FGrid.RowHeights[Pred(Loop)] Then
        PReal(FRowHeights[Pred(Loop)])^:=FGrid.RowHeights[Pred(Loop)]/FractionY;
    End;
end;

procedure TGridSizerHelper.ForceColResize;
Type PReal=^Real;
Var Loop,Temp:Integer;
begin
  For Loop:=1 To FGrid.ColCount Do
    If Loop<=FColWidths.Count Then Begin
      Temp:=Round(Real(FColWidths[Pred(Loop)]^)*FractionX);
      If Temp<>FGrid.ColWidths[Pred(Loop)] Then
        PReal(FColWidths[Pred(Loop)])^:=FGrid.ColWidths[Pred(Loop)]/FractionX;
    End;
end;

Type
  ExposeGrid=Class(TCustomGrid)
    property GridState:TGridState read FGridState;
  End;

procedure TGridSizerHelper.NewWindowProc(var Message: TMessage);
Var Temp:TGridState;
Begin
  {If Left button then get grid state}
  If (Message.Msg=WM_LBUTTONUP) Then Temp:=ExposeGrid(FGrid).GridState;
  {Do normal messages, and allow grid to update}
  FOldWindowProc(Message);
  {Now process any resizing/moving}
  If (Message.Msg=WM_LBUTTONUP) Then
    Case Temp Of
      gsRowSizing,
      gsRowMoving:Begin
                    ForceRowResize;
                  End;
      gsColSizing,
      gsColMoving:Begin
                    ForceColResize;
                  End;
    End;
End;

Constructor TGridSizerHelper.Create(AOwner:TComponent);
Begin
  inherited Create(AOwner);
  FGrid:=Nil;
  FColWidths:=TList.Create;
  FRowHeights:=TList.Create;
  FDefWidth :=1;
  FDefHeight:=1;
  FractionX:=1;
  FractionY:=1;
End;

Destructor TGridSizerHelper.Destroy;
Begin
  DelInfo;
  FColWidths.Free;
  FRowHeights.Free;
  inherited Destroy;
End;

procedure TGridSizerHelper.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FGrid) then Begin
    DelInfo;
    FGrid := nil;
  End;
end;

end.
