unit uThreadPool;

interface

uses Windows;

type
  TThreadProc = procedure(Parameter: pointer) of object;

  TThreadPoolExecutable = class(TObject)
  protected
    procedure Execute(Parameter: pointer); virtual; abstract;
  end;

  TThreadPool = class(TObject)
  private
    FThreadCacheInitSize,FThreadCacheMaxSize,FWaitBeforeCreateNew: integer;
    FCount,FRunningThreadCount: integer;
    FStopping: boolean;
    Threads: pointer;
    FSuspendSemaphore: THandle;
    DataCritSect: TRTLCriticalSection;
    procedure Enter;
    procedure Leave;
    function GetFreeThreadIndex: integer;
    function CreateNewThread: integer;
    procedure PassParams(i: integer; AThreadProc: TThreadProc; AThreadProcParam: pointer);
    procedure InternalExecuteProc(AThreadProc: TThreadProc; AThreadProcParam: pointer);
  public
    constructor Create(AThreadCacheInitSize,AThreadCacheMaxSize,AWaitBeforeCreateNew: integer);
    destructor  Destroy; override;
    procedure CleanUp;
    procedure ExecuteProc(AThreadProc: TThreadProc; AThreadProcParam: pointer);
    procedure Execute(AThreadPoolExecutable: TThreadPoolExecutable; AParameter: pointer);
    property ThreadCacheSize: integer read FCount;
    property RunningThreadCount: integer read FRunningThreadCount;
  end;

const
  MaxTimeOut : integer = 5*60*1000; //5 min

implementation

type
  TThreadsArrayRec = class
    ThreadPool: TThreadPool;
    ThreadProc: TThreadProc;
    ThreadProcParam: pointer;
    ThreadHandle: THandle;
    ThreadID: DWord;
    DataAvailable: THandle;
    DataReceived: THandle;
    ThreadInUse: LongBool;
    procedure Execute;
  end;
  PThreadsArray = ^TThreadsArray;
  TThreadsArray = array[0..(MaxInt div sizeof(TThreadsArrayRec))-1] of TThreadsArrayRec;


constructor TThreadPool.Create(AThreadCacheInitSize,AThreadCacheMaxSize,AWaitBeforeCreateNew: integer);
var
  i: integer;
begin
  inherited Create;
  FCount:=0;
  FRunningThreadCount:=0;
  FThreadCacheInitSize:=AThreadCacheInitSize;
  FThreadCacheMaxSize:=AThreadCacheMaxSize;
  FWaitBeforeCreateNew:=AWaitBeforeCreateNew;
  GetMem(Threads,FThreadCacheMaxSize*sizeof(TThreadsArrayRec));
  for i:=0 to FThreadCacheMaxSize-1 do
    PThreadsArray(Threads)^[i]:=TThreadsArrayRec.Create;
  FSuspendSemaphore:=CreateSemaphore(nil,0,AThreadCacheMaxSize*2,nil); //not signaled
  assert(FSuspendSemaphore<>0);
  InitializeCriticalSection(DataCritSect);
end;

destructor  TThreadPool.Destroy;
var
  i: integer;
begin
  Enter;
  FStopping:=True;
  Leave;
  for i:=1 to FCount do
    InternalExecuteProc(nil,nil);
  for i:=0 to FThreadCacheMaxSize-1 do
    PThreadsArray(Threads)^[i].Free;
  FreeMem(Threads);
  CloseHandle(FSuspendSemaphore);
  DeleteCriticalSection(DataCritSect);
  inherited;
end;

procedure TThreadPool.Enter;
begin
  EnterCriticalSection(DataCritSect);
end;

procedure TThreadPool.Leave;
begin
  LeaveCriticalSection(DataCritSect);
end;

procedure TThreadsArrayRec.Execute;
begin
  while (WaitForSingleObject(DataAvailable,INFINITE)=WAIT_OBJECT_0) do begin
    ReleaseSemaphore(DataReceived,1,nil); //Parameters available
    if Assigned(ThreadProc) then
      try
        ThreadProc(ThreadProcParam);
      except
      end
    else
      break;
    InterlockedExchange(integer(ThreadInUse),ord(False)); //Release thread
    InterlockedDecrement(ThreadPool.FRunningThreadCount);
    ReleaseSemaphore(ThreadPool.FSuspendSemaphore,1,nil); //signal
  end;
  InterlockedDecrement(ThreadPool.FRunningThreadCount);
end;

function ThreadFuncWrapper(Parameter: Pointer): integer;
begin
  TThreadsArrayRec(Parameter).Execute;
  Result:=0;
  EndThread(Result);
end;

function TThreadPool.CreateNewThread: integer;
begin
  with PThreadsArray(Threads)^[FCount] do begin //create new thread
    ThreadPool:=self;
    ThreadInUse:=True; //Futni fog
    DataAvailable:=CreateSemaphore(nil,0,1,nil); //not signaled
    assert(DataAvailable<>0);
    DataReceived:=CreateSemaphore(nil,0,1,nil); //not signaled
    assert(DataReceived<>0);
    ThreadHandle:=BeginThread(nil,2048,@ThreadFuncWrapper,PThreadsArray(Threads)^[FCount],0,ThreadID);
    assert(ThreadHandle<>0);
  end;
  Result:=FCount;
  inc(FCount);
  InterlockedIncrement(FRunningThreadCount);
end;

function TThreadPool.GetFreeThreadIndex: integer;
begin
  Enter;
  try
    Result:=0;
    while (Result<FCount) and (PThreadsArray(Threads)^[Result].ThreadInUse) do inc(Result);
    Assert(Result<FCount);
    PThreadsArray(Threads)^[Result].ThreadInUse:=True;
    InterlockedIncrement(FRunningThreadCount);
  finally
    Leave;
  end;
end;

procedure TThreadPool.PassParams(i: integer; AThreadProc: TThreadProc; AThreadProcParam: pointer);
begin
  with PThreadsArray(Threads)^[i] do begin
    ThreadProc:=AThreadProc;
    ThreadProcParam:=AThreadProcParam;
    ReleaseSemaphore(DataAvailable,1,nil); //Data available
    if WaitForSingleObject(DataReceived,MaxTimeOut)<>WAIT_OBJECT_0 then //wait for Data received
      Assert(False);
    if not Assigned(AThreadProc) then begin
      if WaitForSingleObject(ThreadHandle,MaxTimeOut)<>WAIT_OBJECT_0 then //wait for terminated
        Assert(False);
      //close handles
      CloseHandle(ThreadHandle);
      CloseHandle(DataAvailable);
      CloseHandle(DataReceived);
    end;
  end;
end;


procedure TThreadPool.InternalExecuteProc(AThreadProc: TThreadProc; AThreadProcParam: pointer);
var
  i: integer;
begin
  //FCount=0 => create thread
  i:=-1;
  Enter; //Count, sensitive
  try
    if (FCount=0) and not FStopping then
      i:=CreateNewThread;
  finally
    Leave;
  end;
  if i=-1 then begin
    if WaitForSingleObject(FSuspendSemaphore,FWaitBeforeCreateNew)=WAIT_OBJECT_0 then
      i:=GetFreeThreadIndex 
    else begin
      Enter;
      try
        if (FCount<FThreadCacheMaxSize) and not FStopping then
          i:=CreateNewThread;
      finally
        Leave;
      end;
    end;
    if i=-1 then begin
      //Wait for free thread
      if WaitForSingleObject(FSuspendSemaphore,MaxTimeOut)<>WAIT_OBJECT_0 then
        Assert(False);
      i:=GetFreeThreadIndex; 
    end;
  end;
  PassParams(i,AThreadProc,AThreadProcParam);
end;

procedure TThreadPool.CleanUp;
var
  i: integer;
  Tmp: TThreadsArrayRec;
begin
  Enter; //Count, sensitive
  try
    if FCount>0 then begin
      //while exists free thread
      while WaitForSingleObject(FSuspendSemaphore,0)=WAIT_OBJECT_0 do begin
        i:=GetFreeThreadIndex; //Reserve
        PassParams(i,nil,nil); //Free thread
        dec(FCount);
        // move to the arrays end
        if FCount<>i then begin
          Tmp:=PThreadsArray(Threads)^[FCount];
          PThreadsArray(Threads)^[FCount]:=PThreadsArray(Threads)^[i];
          PThreadsArray(Threads)^[i]:=Tmp;
        end;
      end;
    end
  finally
    Leave;
  end;
end;

procedure TThreadPool.ExecuteProc(AThreadProc: TThreadProc; AThreadProcParam: pointer);
begin
  if Assigned(AThreadProc) then
    InternalExecuteProc(AThreadProc,AThreadProcParam);
end;

procedure TThreadPool.Execute(AThreadPoolExecutable: TThreadPoolExecutable; AParameter: pointer);
begin
  if Assigned(AThreadPoolExecutable) then
    InternalExecuteProc(AThreadPoolExecutable.Execute,AParameter);
end;

end.

