unit ShadowLabel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

type
  TLabelStyle = (lsNormal,lsShadow);
  TShadowDirection = (sdUpLeft,sdDownLeft,sdUpRight,sdDownRight);
  TShow = (sHide,sMaximize,sMinimize,sRestore,sShow,sShowDefault,sShowMaximize,
           sShowMinimize,sShowMinNoActive,sShowNA,sShowNoActive,sShowNormal);
  TClickAction = (acNone,acDblClick,acClick);

  TShadowCustomLabel = class;


  TShadow = class(TPersistent)
  private
    FOwner: TShadowCustomLabel;
    FDirection: TShadowDirection;
    FOffset:  Integer;
    FShadowColor: TColor;
    procedure SetDirection(AValue: TShadowDirection);
    procedure SetOffset(AValue : Integer);
    procedure SetShadowColor(AValue: TColor);
  public
    constructor Create(AOwner: TShadowCustomLabel);
    procedure Assign(Source: TPersistent); override;
    property Owner: TShadowCustomLabel  read FOwner;
  published
    property Direction :  TShadowDirection read FDirection write SetDirection default sdDownLeft;
    property Offset : Integer read FOffset write SetOffset default 2;
    property ShadowColor: TColor read FShadowColor write SetShadowColor default clBtnFace;
  end;

  {TShadowCustomLabel}
  TShadowCustomLabel = class(TGraphicControl)
  private
    FFocusControl: TWinControl;
    FAlignment: TAlignment;
    FAutoSize: Boolean;
    FLayout: TTextLayout;
    FWordWrap: Boolean;
    FShowAccelChar: Boolean;
    FLabelStyle : TLabelStyle;
    FShadow     : TShadow;
    FCommand    : String;
    FShow       : TShow;
    FClickAction : TClickAction;
    procedure SetLabelStyle(AValue : TLabelStyle);
    procedure SetShadow(AValue: TShadow);
    function GetTransparent: Boolean;
    procedure SetAlignment(Value: TAlignment);
    procedure SetFocusControl(Value: TWinControl);
    procedure SetShowAccelChar(Value: Boolean);
    procedure SetTransparent(Value: Boolean);
    procedure SetLayout(Value: TTextLayout);
    procedure SetWordWrap(Value: Boolean);
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure AdjustBounds; dynamic;
    procedure DoDrawText(var Rect: TRect; Flags: Longint); dynamic;
    function GetLabelText: string; virtual;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Paint; override;
    procedure Click; override;
    procedure DblClick; override;
    procedure SetAutoSize(Value: Boolean); virtual;
    property Alignment: TAlignment read FAlignment write SetAlignment
      default taLeftJustify;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property FocusControl: TWinControl read FFocusControl write SetFocusControl;
    property ShowAccelChar: Boolean read FShowAccelChar write SetShowAccelChar default True;
    property Transparent: Boolean read GetTransparent write SetTransparent default False;
    property Layout: TTextLayout read FLayout write SetLayout default tlTop;
    property WordWrap: Boolean read FWordWrap write SetWordWrap default False;
    property LabelStyle : TLabelStyle read FLabelStyle write SetLabelStyle default lsNormal;
    property Shadow : TShadow read FShadow write SetShadow;
    property ShowCommand : TShow read  FShow write FShow default sShowNormal;
    property Command : String read FCommand write FCommand;
    property ClickAction : TClickAction read FClickAction write FClickAction default acDblClick;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Execute;
    property Canvas;
  end;

  TShadowLabel = class(TShadowCustomLabel)
  published
    property Align;
    property Alignment;
    property AutoSize;
    property Caption;
    property ClickAction;
    property Color;
    property Command;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property LabelStyle;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property Shadow;
    property ShowCommand;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property Layout;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;


procedure Register;

implementation
uses ShellApi;

procedure Register;
begin
  RegisterComponents('WORK', [TShadowLabel]);
end;

{ TShadow }
procedure TShadow.Assign(Source: TPersistent);
var
  SrcAttrs: TShadow absolute Source;
begin
  if Source is TShadow then
  begin
    FOwner := SrcAttrs.FOwner;
    Direction := SrcAttrs.Direction ;
    Offset := SrcAttrs.Offset;
    ShadowColor := SrcAttrs.ShadowColor;
  end
  else
    inherited Assign(Source);
end;

constructor TShadow.Create(AOwner: TShadowCustomLabel);
begin
  inherited Create;
  FOwner := AOwner;
  FOffset := 2;
  FDirection := sdDownLeft;
  FShadowColor := clBtnFace;
end;

procedure TShadow.SetShadowColor(AValue: TColor);
begin
  if FShadowColor <> AValue then
  begin
    FShadowColor := AValue;
    FOwner.Invalidate;
  end;
end;

procedure TShadow.SetDirection(AValue: TShadowDirection);
begin
  if FDirection <> AValue then
  begin
    FDirection := AValue;
    FOwner.Invalidate;
  end;
end;

procedure TShadow.SetOffset(AValue: Integer);
begin
  if FOffset <> AValue  then
  begin
     FOffset := AValue;
     FOwner.Invalidate;
  end;
end;

{TShadowCustomLabel}
constructor TShadowCustomLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque, csReplicatable];
  Width := 65;
  Height := 17;
  FAutoSize := True;
  FShowAccelChar := True;
  FLabelStyle := lsNormal;
  FShadow := TShadow.Create(Self);
  FCommand :='';
  FShow := sShowNormal;
  FClickAction := acDblClick;
end;

destructor TShadowCustomLabel.Destroy;
begin
  FShadow.Free;
  inherited Destroy;
end;

procedure TShadowCustomLabel.Execute;
var
  CmdShow : Integer;
begin
   {Execute command}
   case FShow of
    sHide:  CmdShow := SW_HIDE;
    sMaximize: CmdShow := SW_MAXIMIZE;
    sMinimize: CmdShow := SW_MINIMIZE;
    sRestore: CmdShow := SW_RESTORE;
    sShow: CmdShow := SW_SHOW;
    sShowDefault: CmdShow := SW_SHOWDEFAULT;
    sShowMaximize: CmdShow := SW_SHOWMAXIMIZED;
    sShowMinimize: CmdShow := SW_SHOWMINIMIZED;
    sShowMinNoActive: CmdShow := SW_SHOWMINNOACTIVE;
    sShowNA: CmdShow := SW_SHOWNA;
    sShowNoActive: CmdShow := SW_SHOWNOACTIVATE;
    sShowNormal: CmdShow := SW_SHOWNORMAL;
   end;
   ShellExecute(0,'Open',PChar(FCommand),nil,nil,cmdShow);
end;

function TShadowCustomLabel.GetLabelText: string;
begin
  Result := Caption;
end;

procedure TShadowCustomLabel.DoDrawText(var Rect: TRect; Flags: Longint);
var
  Text: string;
  ShadowRect,TextRect : TRect;
begin
  Text := GetLabelText;
  ShadowRect := Rect;
  TextRect := Rect;
  if (Flags and DT_CALCRECT <> 0) and ((Text = '') or FShowAccelChar and
    (Text[1] = '&') and (Text[2] = #0)) then Text := Text + ' ';
  if not FShowAccelChar then Flags := Flags or DT_NOPREFIX;
  //Flags := DrawTextBiDiModeFlags(Flags);
  Canvas.Font := Font;
  if not Enabled then
  begin
    OffsetRect(Rect, 1, 1);
    Canvas.Font.Color := clBtnHighlight;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
    OffsetRect(Rect, -1, -1);
    Canvas.Font.Color := clBtnShadow;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
  end
  else
  begin
    if FLabelStyle = lsShadow then
    begin
       case FShadow.Direction of
          sdUpLeft    : OffsetRect(ShadowRect, -FShadow.Offset, -FShadow.Offset);
          sdDownRight : OffsetRect(ShadowRect, FShadow.Offset, FShadow.Offset);
          sdDownLeft  : OffsetRect(ShadowRect, -FShadow.Offset, FShadow.Offset);
          sdUpRight   : OffsetRect(ShadowRect, FShadow.Offset, -FShadow.Offset);
       end; { case }
       Canvas.Font.Color := FShadow.ShadowColor;
       DrawText(Canvas.Handle, PChar(Text), Length(Text), ShadowRect, Flags);
       Canvas.Font.Color := Font.Color;
       DrawText(Canvas.Handle, PChar(Text), Length(Text), TextRect, Flags);
       UnionRect(Rect, TextRect, ShadowRect);
    end
    else
    begin
       SetTextColor(Canvas.Handle, Color);
       DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
    end;
  end;
end;

procedure TShadowCustomLabel.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect, CalcRect: TRect;
  DrawStyle: Longint;
begin
  with Canvas do
  begin
    if not Transparent then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    { DoDrawText takes care of BiDi alignments }
    DrawStyle := DT_EXPANDTABS or WordWraps[FWordWrap] or Alignments[FAlignment];
    { Calculate vertical layout }
    if FLayout <> tlTop then
    begin
      CalcRect := Rect;
      DoDrawText(CalcRect, DrawStyle or DT_CALCRECT);
      if FLayout = tlBottom then OffsetRect(Rect, 0, Height - CalcRect.Bottom)
      else OffsetRect(Rect, 0, (Height - CalcRect.Bottom) div 2);
    end;
    DoDrawText(Rect, DrawStyle);
  end;
end;

procedure TShadowCustomLabel.Click;
begin
   if FClickAction = acClick then
      Execute;
  inherited Click;
end;

procedure TShadowCustomLabel.DblClick;
begin
   if FClickAction = acDblClick then
      Execute;
  inherited DblClick;
end;

procedure TShadowCustomLabel.Loaded;
begin
  inherited Loaded;
  AdjustBounds;
end;

procedure TShadowCustomLabel.AdjustBounds;
const
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  DC: HDC;
  X: Integer;
  Rect: TRect;
  AAlignment: TAlignment;
begin
  if not (csReading in ComponentState) and FAutoSize then
  begin
    Rect := ClientRect;
    DC := GetDC(0);
    Canvas.Handle := DC;
    DoDrawText(Rect, (DT_EXPANDTABS or DT_CALCRECT) or WordWraps[FWordWrap]);
    Canvas.Handle := 0;
    ReleaseDC(0, DC);
    X := Left;
    AAlignment := FAlignment;
    if AAlignment = taRightJustify then Inc(X, Width - Rect.Right);
    SetBounds(X, Top, Rect.Right, Rect.Bottom);
  end;
end;

procedure TShadowCustomLabel.SetAlignment(Value: TAlignment);
begin
  if FAlignment <> Value then
  begin
    FAlignment := Value;
    Invalidate;
  end;
end;

procedure TShadowCustomLabel.SetAutoSize(Value: Boolean);
begin
  if FAutoSize <> Value then
  begin
    FAutoSize := Value;
    AdjustBounds;
  end;
end;

function TShadowCustomLabel.GetTransparent: Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TShadowCustomLabel.SetFocusControl(Value: TWinControl);
begin
  FFocusControl := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

procedure TShadowCustomLabel.SetShowAccelChar(Value: Boolean);
begin
  if FShowAccelChar <> Value then
  begin
    FShowAccelChar := Value;
    Invalidate;
  end;
end;

procedure TShadowCustomLabel.SetTransparent(Value: Boolean);
begin
  if Transparent <> Value then
  begin
    if Value then
      ControlStyle := ControlStyle - [csOpaque] else
      ControlStyle := ControlStyle + [csOpaque];
    Invalidate;
  end;
end;

procedure TShadowCustomLabel.SetLayout(Value: TTextLayout);
begin
  if FLayout <> Value then
  begin
    FLayout := Value;
    Invalidate;
  end;
end;

procedure TShadowCustomLabel.SetWordWrap(Value: Boolean);
begin
  if FWordWrap <> Value then
  begin
    FWordWrap := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TShadowCustomLabel.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FFocusControl) then
    FFocusControl := nil;
end;

procedure TShadowCustomLabel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
  AdjustBounds;
end;

procedure TShadowCustomLabel.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustBounds;
end;

procedure TShadowCustomLabel.CMDialogChar(var Message: TCMDialogChar);
begin
  if (FFocusControl <> nil) and Enabled and ShowAccelChar and
    IsAccel(Message.CharCode, Caption) then
    with FFocusControl do
      if CanFocus then
      begin
        SetFocus;
        Message.Result := 1;
      end;
end;


procedure TShadowCustomLabel.SetShadow(AValue: TShadow);
begin
  FShadow := AValue;
end;

procedure TShadowCustomLabel.SetLabelStyle(AValue : TLabelStyle);
begin
    if FLabelStyle <> AValue then
    begin
       FLabelStyle := AValue;
       Invalidate;
    end;
end;


end.
