{*******************************************************************************
   Unit
      sEdits.pas
   Description:
      Flat style edit controls with a buttons, pictures and other extras.
   Version:
      2.1j
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   Comments:
      This is an almost total rewrite of sEditTools unit published a year ago on the
      Torry Delphi Pages. I sow the download count and compare it with a code...
      and understood that it worce to spend a time to improve it. Thanks to everyone
      who wrote me a notes or bug reports.
	History:
      2.1j  -  28/12/1998
               .Fixup of button refresh on button's properties changing (reported by Indy)
               .Several small fixups.
               .Work around for problem with right or center aligned edit controls. By
               default the carret is jumping to the next line when length of the text
               exeeds the edit bounds. Now sEdit with right or center alignment will
               simply refuse to assign any text bigger then can fit in the edit bounds.
      2.1i  -  23/12/1998 - 24/12/1998
               .Fixup of DoSetText: added Modified = TRUE. This fixes the problem
               with validation of the entered text in the edits which handle input
               themselvs.
               .Fixup of Reset in TsDateEdit
               .Fixed <Ctrl><Tab> handling
               .Key handling fixup in TsNumberEdit (with a help of Thomas von Stetten)
               .Added OnDateChanged event to TsDateEdit (proposed by Roman Yugov)
      2.1h  -  16/12/1998
               .Cashing of glyphs implemented. It is MUCH FASTER now.
      2.1g  -  16/12/1998
               .Reduced flicker of buttons and picture while editing
      2.1f  -  10/12/1998
               .Fixed painting of buttons. At last.
               .Fixed painting of Picture (sizing and stretch)
               .SetText is virtual now to meet the needs of TsMaskLinkEdit
               .Fixed bug in TsDateEdit.WMCut
               .Validate function can (optionally) give back the position of wrong
                  char in the editor text.
               .Time critical paint related functions changed from virtual to dynamic
      2.1e  -  3/12/1998
               .Fixed the range checking in assigning of AsInteger, AsDouble and
                  AsCurrency properties of TsNumberEdit.
               .Fixed VK_DELETE, VK_Back and VK_UP, VK_DOWN keys handling in TsNumberEdit
                  (suggested by Thomas von Stetten)
               .IsValidChar functions fixup.
               .Fixed VK_RETURN key handling
               .Fixed VK_DELETE key handling in TsNumberEdit
      2.1d  -  30/11/1998
               .Changes suggested by Thomas von Stetten.
                  They marked with TVS in the comment.
      2.1c  -  27/11/1998
               .Sending notifications about MouseEnter, MouseLeave events.
               .Small fixup of UpDown button painting.
      2.1b  -  25/11/1998
               .Change enabled notification message implementation
               .Moving strings to sConsts
      2.1a  -  22/11/1998
               .FixUp of the border sizing.
               .FixUp of handling of DlgKeys.
               .FixUp of borders redraw (there was a problem when text was assigned
                  before the edit control became visible, for example in OnCreate
                  event of the parent form)
      2.1   -  18/11/1998
               .TsGlyphList support
		2.0a	-	10/10/1998
      			.new property for TsDirectoryEdit DialogCaption (to meet the new
               	property in TsBrowseFolderDialog).
               .Bug fix in TsFileDirEdit - the edit now is redrawn correctly when
               	initializing or resizing.
		2.0*  - 	End of Sep. 1998
      			Initial release

*     I did not track the versions before, so let's consider it as 2.0
**    RX controls - the glyph policy, many lines of code in TsFilenameEdit
      and TsDirectoryEdit
*******************************************************************************}
unit sEdits;

interface

uses Windows, Classes, Messages, Controls, Menus, StdCtrls, Graphics, sDate,
   sPickDate, sGraphics, sNumPad, sBrowseFolder, Dialogs, sGlyphsList, sConsts;

{$I s.Inc}

type
   TsCustomEdit = class;
   TNumGlyphs = 1..4;
   TsButtonState = (bsUp, bsDisabled, bsDown);
   TsEditSubControlOptions = set of (scoNoStretch, scoChanging);

   TsEditSubControl = class(TsGraphicListControlChild)
   private
      FGlyphList: TGlyphList;
      FIndexs: array[TsButtonState] of Integer;
      FNumGlyphs: TNumGlyphs;
      FParent: TsCustomEdit;
      FPicture: TBitMap;
      FGlyphId: Integer;
      FState: TsButtonState;
      FOptions: TsEditSubControlOptions;
      FVisible: Boolean;
      FWidth: Integer;
      procedure SetNumGlyphs(Value: TNumGlyphs);
      procedure SetPicture(Value: TBitmap);
      procedure SetGlyphId(Value: Integer);
      procedure SetVisible(Value: Boolean);
      procedure SetWidth(Value: Integer);
      procedure InvalidateGlyphCash;
      procedure InvalidateParent;
   protected
      function GetPictureRect: TRect; virtual; abstract;
      function GetBackgroundColor: TColor; virtual; abstract;
      procedure SetState(Value: TsButtonState); virtual; abstract;
      function GetGlyphStored: Boolean; virtual;
      procedure PictureChanged(Sender: TObject); virtual;
      procedure Paint; virtual;
      function GetGlyph(Pict: TBitMap; State: TsButtonState): TBitMap; virtual;
      function GetGlyphList: TsGlyphList; override;
      procedure CheckGlyphId(oldId, newId: Integer);
      procedure RefreshGlyphId;
      property Visible: Boolean read FVisible write SetVisible;
      property Glyph: TBitmap read FPicture write SetPicture stored GetGlyphStored;
      property GlyphListId: Integer read FGlyphId write SetGlyphId default -1;
      property NumGlyphs: TNumGlyphs read FNumGlyphs write SetNumGlyphs default 1;
      property Width: Integer read FWidth write SetWidth default 0;
   public
      constructor Create( aParent: TsCustomEdit); virtual;
      destructor Destroy; override;
      property PictureRect: TRect read GetPictureRect;
      property State: TsButtonState read FState write SetState;
      property Parent: TsCustomEdit read FParent;
   end;

   TsEditPicture = class(TsEditSubControl)
   protected
      function GetPictureRect: TRect; override;
      function GetBackgroundColor: TColor; override;
      procedure SetState(Value: TsButtonState); override;
   published
      property Visible;
      property Glyph;
      property GlyphListId;
      property NumGlyphs;
      property Width;
   end;

   TsEditState = set of ( msReEnter, msMouseInControl, msChanging, msValidating,
         msKeyHandled, msNoRedraw, msFirstTimePainted,  msPopupOpen,
         msNoPopupCall, msCheckForBtn);
   TValidateEvent = procedure(Sender: TObject; var IsValid: Boolean) of Object;

   TsCustomEdit = class(TCustomEdit)
  	private
      FAlignment: TAlignment;
      FCanvas: TCanvas;
      FColor: TColor;
      FDisabledStyle: TsDisabledStyle;
      FDisabledFont: TFont;
      FFlat: Boolean;
      FGlyphList: TsGlyphList;
      FOldValue: String;
      FParentColor: Boolean;
      FPicture: TsEditPicture;
      FSolidBorder: Boolean;
      FText: String;
      FValidChars: String;
      FOnValidate: TValidateEvent;
      FOnValidateError: TNotifyEvent;
      procedure SetAlignment(Value: TAlignment);
      procedure SetDisabledStyle(Value: TsDisabledStyle);
      procedure SetDisabledFont(Value: TFont);
      procedure SetFlat(const Value: Boolean);
      procedure SetGlyphList(Value: TsGlyphList);
      procedure SetPicture(Value: TsEditPicture);
      procedure SetSolidBorder;
      function GetText: String;
      function GetPasswordText: String;
      function ValidCharsStored: Boolean;
      function GetBorderSize: Integer;
      procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
      procedure WMEraseBkgnd(var Message: TMessage); message WM_ERASEBKGND;
      procedure WMNCPaint(var Message: TMessage); message WM_NCPAINT;
      procedure WMSetFocus(var Message: TMessage); message WM_SETFOCUS;
      procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS;
      procedure WMGlyphIdChanged(var Message: TMessage); message STM_GLYPHIDCHANGED;
      procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure CMEnabledChanged(var Msg: TMessage); message CM_ENABLEDCHANGED;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
      procedure CMVisibleChanged(var Message: TMessage); message CM_VISIBLECHANGED;
   protected
      FEditState: TsEditState;
      procedure CreateParams(var Params: TCreateParams); override;
      procedure CreateWnd; override;
      procedure WndProc(var Message: TMessage); override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Change; override;
      procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
      procedure DoEnter; override;
      procedure DoExit; override;
      procedure DoSetText(Value: String);
      procedure SetText(Value: String); virtual;
      function GetEditRect: TRect; dynamic;
      procedure SetEditRect; virtual;
      function GetSolidBorder: Boolean; virtual;
      procedure SetupEnabled; virtual;
      procedure PaintDisabled(PS: TPaintStruct); dynamic;
      procedure InternalPaintWindow; dynamic;
      procedure RedrawBorders; virtual;
      function EditCanModify: Boolean; virtual;
      procedure Reset; virtual;
      procedure ChangeExpected; virtual;
      function IsValidChar(const Key: Char): Boolean; virtual;
      function Validate(var Pos: Integer): Boolean; virtual;
      procedure ValidateError; virtual;
      procedure CheckCursor(force: Boolean); virtual;
      procedure RefreshChildGlyphs; virtual;
{$IFDEF V1_COMP}      {Back compatibility}
      procedure ReadDefaultDraw(Reader: TReader);
      procedure WriteNothing(Writer: TWriter);
      procedure DefineProperties(Filer: TFiler); override;
{$ENDIF}
      property OldValue: String read FOldValue;
      property SolidBorder: Boolean read FSolidBorder;
      property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
      property DisabledStyle: TsDisabledStyle read FdisabledStyle write SetDisabledStyle default dsDefault;
      property DisabledFont: TFont read FDisabledFont write SetDisabledFont;
      property Flat: Boolean read FFlat write SetFlat default FALSE;
      property GlyphList: TsGlyphList read FGlyphList write SetGlyphList;
      property Picture: TsEditPicture read FPicture write SetPicture;
      property Text: String read GetText write SetText;
      property ValidChars: String read FValidChars write FValidChars stored ValidCharsStored;
      property OnValidate: TValidateEvent read FOnValidate write FOnValidate;
      property OnValidateError: TNotifyEvent read FOnValidateError write FOnValidateError;
   public
      constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
      procedure ValidateEdit;
      procedure GetSel(var SelStart: Integer; var SelStop: Integer);
      procedure SetSel(const SelStart, SelStop: Integer);
      // TVS: Moved from Protected to Public because it was already Public in Base-Class
      procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
   end;

   TsEdit = class( TsCustomEdit)
   published
      property Alignment;
      property AutoSelect;
    	property BorderStyle;
    	property CharCase;
    	property Color;
    	property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
    	property DragCursor;
    	property DragMode;
    	property Enabled;
      property Flat;
    	property Font;
    	property HideSelection;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
    	property ParentCtl3D;
    	property ParentFont;
    	property ParentShowHint;
    	property PasswordChar;
      property Picture;
      property GlyphList; // mast be after picture.
    	property PopupMenu;
    	property ReadOnly;
    	property ShowHint;
    	property TabOrder;
    	property TabStop;
      property Text;
      property ValidChars;
    	property Visible;
    	property OnChange;
    	property OnClick;
    	property OnDblClick;
    	property OnDragDrop;
    	property OnDragOver;
    	property OnEndDrag;
    	property OnEnter;
    	property OnExit;
    	property OnKeyDown;
    	property OnKeyPress;
    	property OnKeyUp;
    	property OnMouseDown;
    	property OnMouseMove;
    	property OnMouseUp;
    	property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TGlyphKind = (gkCustom, gkDefault, gkDropDown, gkEllipsis);
   TsButtonKind = (bkNormal, bkUpDown);
   TsEditButtonState = (bsNormal, bsUpButton, bsDownButton);
   TsEditButtons = (ebNone, ebButton1, ebButton2);

   TsEditButton = class(TsEditSubControl)
   private
      FAutoRepeat: Boolean;
      FButtonKind: TsButtonKind;
      FButtonState: TsEditButtonState;
      FClickKey: TShortCut;
      FClickKeyUp: TShortCut;
      FCursor: TCursor;
      FGlyphKind: TGlyphKind;
      FHint: String;
      FLinkedButton: TsEditButton;
      FShowHint: Boolean;
      procedure RedrawBorders;
      procedure SetButtonKind( Value: TsButtonKind);
      procedure SetGlyphKind( Value: TGlyphKind);
      procedure RecreateGlyph;
      procedure RecreateUpDownGlyph;
   protected
      function GetGlyphStored: Boolean; override;
      procedure Paint; override;
      procedure PictureChanged(Sender: TObject); override;
      function GetPictureRect: TRect; override;
      function GetBackgroundColor: TColor; override;
      function GetGlyph(Pict: TBitMap; State: TsButtonState): TBitMap; override;
      procedure SetState(Value: TsButtonState); override;
      function GetDisplayRect: TRect;
      property ClickKey: TShortCut read FClickKey write FClickKey;
      property ClickKeyUp: TShortCut read FClickKeyUp write FClickKeyUp;
   public
      constructor Create( aParent: TsCustomEdit); override;
      property ButtonState: TsEditButtonState read FButtonState;
   published
      property AutoRepeat: Boolean read FAutoRepeat write FAutoRepeat default TRUE;
      property Cursor: TCursor read FCursor write FCursor default crArrow;
      property Glyph stored GetGlyphStored;
      property GlyphListId;
      property GlyphKind: TGlyphKind read FGlyphKind write SetGlyphKind default gkCustom;
      property NumGlyphs;
      property ButtonKind: TsButtonKind read FButtonKind write SetButtonKind nodefault;
      property Hint: String read FHint write FHint;
      property ShowHint: Boolean read FShowHint write FShowHint default TRUE;
      property Width;
   end;

   TsRightEditButton = class(TsEditButton)
   published
      property ClickKey default scAlt + vk_Down;
      property ClickKeyUp default scAlt + vk_Up;
      property Visible default TRUE;
   end;

   TsLeftEditButton  = class(TsEditButton)
   public
      constructor Create( aParent: TsCustomEdit); override;
   published
      property ClickKey default scCtrl + vk_Down;
      property ClickKeyUp default scCtrl + vk_Up;
      property Visible default FALSE;
   end;

   TsCustomLinkEdit = class(TsCustomEdit)
   private
      FButton1: TsRightEditButton;
      FButton2: TsLeftEditButton;
      FHintForBtn: TsEditButtons;
      FTrackState: TsEditButtons;
      FOnButton1Click: TNotifyEvent;
      FOnButton2Click: TNotifyEvent;
      procedure SetLeftButton( Value: TsLeftEditButton);
      procedure SetRightButton( Value: TsRightEditButton);
      procedure TrackButton(const X,Y: Integer);
		procedure StopTracking;
      procedure OnTimer(Sender: TObject);
      procedure WMEraseBkgnd(var Message: TMessage); message WM_ERASEBKGND;
      procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
      procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
      procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
      procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
      procedure WMGlyphIdChanged(var Message: TMessage); message STM_GLYPHIDCHANGED;
      procedure CMHintShow(var Message: TMessage); message CM_HINTSHOW;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
   protected
      procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    	procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    	procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      function CheckShortCut(var Key: Word; Shift: TShiftState): Boolean;
      procedure ButtonClick(Button: TsEditButtons); virtual;
      function GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap; virtual;
      function GetEditRect: TRect; override;
      procedure SetupEnabled; override;
      procedure InternalPaintWindow; override;
      procedure RedrawBorders; override;
      procedure RefreshChildGlyphs; override;
{$IFDEF V1_COMP}      {Back compatibility}
      procedure ReadGlyphKind(Reader: TReader);
      procedure ReadClickKey(Reader: TReader);
      procedure DefineProperties(Filer: TFiler); override;
{$ENDIF}
      property Button1: TsRightEditButton read FButton1 write SetRightButton;
      property Button2: TsLeftEditButton read FButton2 write SetLeftButton;
      property OnButton1Click: TNotifyEvent read FOnButton1Click write FOnButton1Click;
      property OnButton2Click: TNotifyEvent read FOnButton2Click write FOnButton2Click;
   public
      constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
   end;

   TsLinkEdit = class(TsCustomLinkEdit)
   published
      property Alignment;
      property AutoSelect;
    	property BorderStyle;
      property Button1;
      property Button2;
    	property CharCase;
    	property Color;
    	property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
    	property DragCursor;
    	property DragMode;
    	property Enabled;
      property Flat;
    	property Font;
    	property HideSelection;
      property Hint;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
    	property ParentCtl3D;
    	property ParentFont;
    	property ParentShowHint;
    	property PasswordChar;
      property Picture;
      property GlyphList; // mast be after picture.
    	property PopupMenu;
    	property ReadOnly;
    	property ShowHint;
    	property TabOrder;
    	property TabStop;
      property Text;
      property ValidChars;
    	property Visible;
      property OnButton1Click;
      property OnButton2Click;
    	property OnChange;
    	property OnClick;
    	property OnDblClick;
    	property OnDragDrop;
    	property OnDragOver;
    	property OnEndDrag;
    	property OnEnter;
    	property OnExit;
    	property OnKeyDown;
    	property OnKeyPress;
    	property OnKeyUp;
    	property OnMouseDown;
    	property OnMouseMove;
    	property OnMouseUp;
    	property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsCustomPopupEdit = class( TsCustomLinkEdit)
   private
      procedure WMMouseActivate(var Message: TMessage); message WM_MOUSEACTIVATE;
      procedure WMNCHitTest(var Message: TMessage); message WM_NCHITTEST;
   protected
      function GetSolidBorder: Boolean; override;
      procedure OnAccept(Sender: TObject); virtual;
      procedure OnCancel(Sender: TObject); virtual;
   public
      procedure ShowPopup(doShow: Boolean); virtual; abstract;
   end;


   TDateFormat = set of (dfFullDay, dfFullMonth, dfFullYear, dfShowWeekDay,
                        dfStringDateView, dfStringDateEditing, dfMonthSelect);
   TDateEditCells = -1..3;

  	TsCustomDateEdit = class( TsCustomPopupEdit)
  	private
      FCalPop: TsPopupCalendar;
      FCalendarProps: TsPopupCalendarProps;
     	FDate: TsDate;
      FDateFormat: TDateFormat;
      FDateOrder: TDateOrder;
      FEmptyString: String;
      FSeparator: Char;
      FCaretPos: Integer;
      FShowToday: Boolean;
      FOnDateChange: TNotifyEvent;
      FOnDateChanged: TNotifyEvent;
      procedure DateChanged( Sender: TObject);
      procedure SetSeparator( Value: Char);
      procedure SetDateFormat(Value: TDateFormat);
      procedure SetDateOrder(Value: TDateOrder);
      procedure SetEmptyString(Value: String);
      procedure SetPopupCalendar(Value: TsPopupCalendarProps);
      function EmptyStringStored: Boolean;
      function GetEmptyString: String;
      procedure SetShowToday(value: Boolean);
      procedure CheckCursor(force: Boolean); override;
      procedure SetCursor(Pos: Integer);
      procedure NextCell(next: Boolean);
		procedure ArrowKeys(var CharCode: Word; Shift: TShiftState);
		procedure DeleteKeys(var CharCode: Word; Shift: TShiftState);
      procedure AssignDateValue;
      function GetSeparatorPosition(first: Boolean): ShortInt;
      function GetEditLength: Integer;
      procedure GetCellBounds(cell: TDateEditCells; var SelStart, SelStop: Integer);
      function GetCellFromPos(var SelStart, SelStop: Integer): TDateEditCells;
      function MapCell(cell: TDateEditCells): TDateEditCells;
      function GetCellValue(cell: TDateEditCells): Integer;
      procedure IncCellValue(doInc: Boolean);
      function GetFormatString(longFormat: Boolean): String;
		procedure FormatDateText;
      procedure WMPaste(var Message: TMessage); message WM_PASTE;
      procedure WMCut(var Message: TMessage); message WM_CUT;
      procedure WMContextMenu(var Message: TMessage); message WM_CONTEXTMENU;
  	protected
      function GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap; override;
      procedure Click; override;
      procedure ButtonClick(Button: TsEditButtons); override;
      procedure DoEnter; override;
      procedure DoExit; override;
		procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
      function IsValidChar(const Key: Char): Boolean; override;
      function Validate(var pos: Integer): Boolean; override;
      function EmptyText: String;
      procedure OnAccept(Sender: TObject); override;
      property EmptyString: String read FEmptyString write SetEmptyString stored EmptyStringStored;
      property DateSeparator: Char read FSeparator write SetSeparator {TVS: nodefault - see Create() default '/'};
      property DateFormat: TDateFormat read FDateFormat write SetDateFormat default [dfFullDay, dfFullMonth, dfFullYear];
      property DateOrder: TDateOrder read FDateOrder write SetDateOrder;
      property PopupCalendar: TsPopupCalendarProps read FCalendarProps write SetPopupCalendar;
      property ShowToday: Boolean read FShowToday write SetShowToday default FALSE;
      property OnDateChange: TNotifyEvent read FOnDateChange write FOnDateChange;
      property OnDateChanged: TNotifyEvent read FOnDateChanged write FOnDateChanged;
   public
   	constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
      procedure ShowPopup(doShow: Boolean); override;
     	property EditDate: TsDate read FDate;
   end;

   TsDateEdit = class( TsCustomDateEdit)
   published
//TVS: property Alignment was not published
      property Alignment;
      property ShowToday;
      property AutoSelect;
    	property BorderStyle;
      property Button1;
      property Button2;
    	property Color;
    	property Ctl3D;
      property DateSeparator;
      property DateFormat;
      property DateOrder;
      property DisabledStyle;
      property DisabledFont;
    	property DragCursor;
    	property DragMode;
      property EmptyString;
    	property Enabled;
      property Flat;
    	property Font;
    	property HideSelection;
      property ParentColor;
    	property ParentCtl3D;
    	property ParentFont;
    	property ParentShowHint;
    	property PasswordChar;
      property Picture;
      property GlyphList; // mast be after picture.
      property PopupCalendar;
   	property PopupMenu;
    	property ReadOnly;
    	property ShowHint;
    	property TabOrder;
    	property TabStop;
    	property Visible;
      property ImeMode;
      property ImeName;
      property OnButton1Click;
      property OnButton2Click;
    	property OnChange;
    	property OnClick;
      property OnDateChange;
      property OnDateChanged;
    	property OnDblClick;
    	property OnDragDrop;
    	property OnDragOver;
    	property OnEndDrag;
    	property OnEnter;
    	property OnExit;
    	property OnKeyDown;
    	property OnKeyPress;
    	property OnKeyUp;
    	property OnMouseDown;
    	property OnMouseMove;
    	property OnMouseUp;
    	property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsCustomNumberEdit = class(TsCustomPopupEdit)
   private
      FDecimals: ShortInt;
      FMinValue: Double;
      FMaxValue: Double;
      FKeptValue: Double;
		FormatMask: String;
      FIncrement: Double;
      FNumPad: TsNumPad;
      FNumPadProps: TsNumPadProps;
      function GetAsCurrency: Currency;
    	function GetAsInteger: Integer;
      function GetValue: Double;
      function DblValueStored(index: Integer): Boolean;
      procedure SetAsCurrency(Value: Currency);
    	procedure SetAsInteger(Value: Integer);
    	procedure SetDecimals(Value: ShortInt);
      // TVS: Parameter-Name changed for a better understanding of the implementation
    	procedure SetValue(dblValue: Double);
      procedure SetNumPad(Value: TsNumPadProps);
      procedure IncValue(doInc: Boolean);
      procedure DeleteKey(Key: Word);
    	procedure DeleteSelection;
    	procedure WMCut(var Message: TMessage); message WM_CUT;
    	procedure WMPaste(var Message: TMessage); message WM_PASTE;
   protected
   	procedure CreateParams(var Params: TCreateParams); override;
    	procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    	procedure KeyPress(var Key: Char); override;
      function IsValidChar(const Key: Char): Boolean; override;
      function Validate(var Pos: Integer): Boolean; override;
      function GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap; override;
      procedure ButtonClick(Button: TsEditButtons); override;
      procedure OnAccept(Sender: TObject); override;
      procedure OnCancel(Sender: TObject); override;
      procedure OnValChange(Sender: TObject);
      procedure SetFormatMask;
      property Decimals: ShortInt read FDecimals write SetDecimals default 2;
      property Increment: Double index 0 read FIncrement write FIncrement stored DblValueStored;
    	property MaxValue: Double index 1 read FMaxValue write FMaxValue stored DblValueStored;
    	property MinValue: Double index 2 read FMinValue write FMinValue stored DblValueStored;
    	property Value: Double read GetValue write SetValue;
      property AsCurrency: Currency read GetAsCurrency write SetAsCurrency;
   	property AsFloat: Double read GetValue write SetValue;
    	property AsInteger: Integer read GetAsInteger write SetAsInteger;
      property NumPad: TsNumPadProps read FNumPadProps write SetNumPad;
   public
   	constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
      procedure ShowPopup(doShow: Boolean); override;
	end;

   TsNumberEdit = class(TsCustomNumberEdit)
	public
      property AsCurrency;
    	property AsFloat;
    	property AsInteger;
   published
      property Alignment default taRightJustify;
      property AutoSelect;
    	property BorderStyle;
      property Button1;
      property Button2;
    	property Color;
    	property Ctl3D;
      property Decimals;
      property DisabledFont;
      property DisabledStyle;
    	property DragCursor;
    	property DragMode;
    	property Enabled;
      property Flat;
    	property Font;
    	property HideSelection;
      property ImeMode;
      property ImeName;
      property Increment;
      property MaxLength;
      property MaxValue;
    	property MinValue;
      property NumPad;
      property ParentColor;
    	property ParentCtl3D;
    	property ParentFont;
    	property ParentShowHint;
    	property PasswordChar;
      property Picture;
      property GlyphList; // mast be after picture.
    	property PopupMenu;
    	property ReadOnly;
    	property ShowHint;
    	property TabOrder;
    	property TabStop;
      property Value;
    	property Visible;
      property OnButton1Click;
      property OnButton2Click;
    	property OnChange;
    	property OnClick;
    	property OnDblClick;
    	property OnDragDrop;
    	property OnDragOver;
    	property OnEndDrag;
    	property OnEnter;
    	property OnExit;
    	property OnKeyDown;
    	property OnKeyPress;
    	property OnKeyUp;
    	property OnMouseDown;
    	property OnMouseMove;
    	property OnMouseUp;
    	property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TExecOpenDialogEvent = procedure(Sender: TObject; var Name: string; var Action: Boolean) of object;

   TsFileDirEdit = class(TsCustomLinkEdit)
   private
      FAcceptFiles: Boolean;
      FDirectoryEdit: Boolean;
      FDisplayedChars: Integer;
      FDisplayFullPath: Boolean;
      FDisplayRelativePath: Boolean;
      FNoPathValidate: Boolean;
      FPath: String;
      FOnBeforeDialog: TExecOpenDialogEvent;
      FOnAfterDialog: TExecOpenDialogEvent;
      FOnDropFiles: TNotifyEvent;
      FOnPathChange: TNotifyEvent;
      function GetPath: string;
      procedure SetPath(const Value: String);
      procedure SetDragAccept(Value: Boolean);
      procedure SetDisplayFullPath(Value: Boolean);
      procedure SetDisplayRelativePath(Value: Boolean);
      procedure SetAcceptFiles(Value: Boolean);
      procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
   protected
      procedure CreateHandle; override;
      procedure DestroyWindowHandle; override;
      procedure Change; override;
      procedure DoEnter; override;
      procedure DoExit; override;
      procedure PathChanged; virtual;
      procedure DoAfterDialog(var AFileName: string; var Action: Boolean); virtual;
      procedure DoBeforeDialog(var AFileName: string; var Action: Boolean); virtual;
      procedure ReceptFileDir(const AFileName: string); virtual; abstract;
      function GetDisplayText: String; virtual;
      function GetInitialDir: string; virtual; abstract;
      procedure SetInitialDir(const Value: string); virtual; abstract;
      property Path: String read GetPath write SetPath;
   public
      constructor Create(AOwner: TComponent); override;
      // TVS: Moved from Protected to Public because it was already Public in Base-Class
      procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
   published
      property AcceptFiles: Boolean read FAcceptFiles write SetAcceptFiles default FALSE;
      property DisplayFullPath: Boolean read FDisplayFullPath write SetDisplayFullPath default FALSE;
      property DisplayRelativePath: Boolean read FDisplayRelativePath write SetDisplayRelativePath default FALSE;
      property InitialDir: string read GetInitialDir write SetInitialDir;
      property OnBeforeDialog: TExecOpenDialogEvent read FOnBeforeDialog write FOnBeforeDialog;
      property OnAfterDialog: TExecOpenDialogEvent read FOnAfterDialog write FOnAfterDialog;
      property OnDropFiles: TNotifyEvent read FOnDropFiles write FOnDropFiles;
		property OnPathChange: TNotifyEvent read FOnPathChange write FOnPathChange;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TFileExt = string[3];
   TsFilenameEdit = class(TsFileDirEdit)
   private
      FDialog: TOpenDialog;
      function GetDefaultExt: TFileExt;
      function GetFileEditStyle: TFileEditStyle;
      function GetFilter: string;
      function GetFilterIndex: Integer;
      function GetHistoryList: TStrings;
      function GetOptions: TOpenOptions;
      function GetDialogTitle: string;
      function GetDialogFiles: TStrings;
      procedure SetDialogTitle(const Value: string);
      procedure SetDefaultExt(Value: TFileExt);
      procedure SetFileEditStyle(Value: TFileEditStyle);
      procedure SetFilter(const Value: string);
      procedure SetFilterIndex(Value: Integer);
      procedure SetHistoryList(Value: TStrings);
      procedure SetOptions(Value: TOpenOptions);
      function IsCustomTitle: Boolean;
      function IsCustomFilter: Boolean;
   protected
      procedure ButtonClick(Button: TsEditButtons); override;
      procedure ReceptFileDir(const AFileName: string); override;
      function GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap; override;
      function Validate(var Pos: Integer): Boolean; override;
      function GetInitialDir: string; override;
      procedure SetInitialDir(const Value: string); override;
   public
      constructor Create(AOwner: TComponent); override;
      property Dialog: TOpenDialog read FDialog;
      property DialogFiles: TStrings read GetDialogFiles;
   published
      property DefaultExt: TFileExt read GetDefaultExt write SetDefaultExt;
      property FileEditStyle: TFileEditStyle read GetFileEditStyle write SetFileEditStyle default fsEdit;
      property Path;
      property Filter: string read GetFilter write SetFilter stored IsCustomFilter;
      property FilterIndex: Integer read GetFilterIndex write SetFilterIndex default 1;
      property HistoryList: TStrings read GetHistoryList write SetHistoryList;
      property DialogOptions: TOpenOptions read GetOptions write SetOptions default [];
      property DialogTitle: string read GetDialogTitle write SetDialogTitle stored IsCustomTitle;
      property DisabledFont;
      property DisabledStyle;
      property AutoSelect;
    	property BorderStyle;
      property Button1;
      property Button2;
      property CharCase;
      property Color;
      property Ctl3D;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Flat;
      property Font;
      property ImeMode;
      property ImeName;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property Picture;
      property GlyphList; // mast be after picture.
      property PopupMenu;
      property ReadOnly;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDirectoryEdit = class(TsFileDirEdit)
   private
      FDialog: TsBrowseFolderDialog;
      FInitialDir: string;
      FOnInitialized: TBrowseDialogEvent;
      FOnSelectionChanged: TBrowseDialogEvent;
      function GetDialogCaption: string;
		procedure SetDialogCaption(Value: string);
		function IsCustomCaption: Boolean;
      function GetDialogTitle: string;
      procedure SetDialogTitle(Value: String);
      function IsCustomTitle: Boolean;
      procedure SetDialogOptions( Value: TBrowseOptions);
      function GetDialogOptions: TBrowseOptions;
      procedure SetFolder( Value: TSHFolders);
      function GetFolder: TSHFolders;
      procedure SetShowPath(Value: Boolean);
      function GetShowPath: Boolean;
   protected
      function GetInitialDir: string; override;
      procedure SetInitialDir(const Value: string); override;
      procedure ButtonClick(Button: TsEditButtons); override;
      procedure ReceptFileDir(const AFileName: string); override;
      function GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
   published
      property DialogOptions: TBrowseOptions read GetDialogOptions write SetDialogOptions default [];
      property DialogShowPath: Boolean read GetShowPath write SetShowPath;
      property DialogCaption: string read GetDialogCaption write SetDialogCaption stored IsCustomCaption;
      property DialogTitle: string read GetDialogTitle write SetDialogTitle stored IsCustomTitle;
      property Folder: TSHFolders read GetFolder write SetFolder default foCustom;
      property Path;
      property AutoSelect;
    	property BorderStyle;
      property Button1;
      property Button2;
      property CharCase;
      property Color;
      property Ctl3D;
      property DisabledFont;
      property DisabledStyle;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Flat;
      property Font;
      property ImeMode;
      property ImeName;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property Picture;
      property GlyphList; // mast be after picture.
      property PopupMenu;
      property ReadOnly;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnDialogInitialized: TBrowseDialogEvent read FOnInitialized write FOnInitialized;
      property OnDialogSelectionChanged: TBrowseDialogEvent read FOnSelectionChanged write FOnSelectionChanged;
      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

const
  InitRepeatPause = 400;  { UpDown button pause before repeat timer (ms) }
  RepeatPause     = 100;  { UpDown button repeat (ms)}

implementation
{$R SEDITS}

uses Forms, ExtCtrls, sFlat, sFileUtils, StdUtils, SysUtils, ShellAPI, Clipbrd,
   commctrl;


type
   ELinkEditError = class(Exception);

var
  GlyphCache: TGlyphCache = nil;

{******************** TsEditsubControl ************************}
constructor TsEditSubControl.Create( aParent: TsCustomEdit);
var
   ii: TsButtonState;
begin
   inherited Create;
   FPicture := TBitMap.Create;
   FPicture.OnChange := PictureChanged;
   FParent := aParent;
   FVisible := TRUE;
   FWidth := 0;
   FNumGlyphs := 1;
   FGlyphId := -1;
   FState := bsUp;
   for ii := Low(ii) to High(ii) do
      FIndexs[ii] := -1;
end;

destructor TsEditSubControl.Destroy;
begin
   FPicture.Free;
   FPicture := nil;
   if Assigned(GlyphCache) and GlyphCache.Empty then begin
      GlyphCache.Free;
      GlyphCache := nil;
   end;
   inherited;
end;

procedure TsEditSubControl.InvalidateGlyphCash;
var
   ii: TsButtonState;
begin
   for ii := Low(ii) to High(ii) do begin
      if FIndexs[ii] <> -1 then
         FGlyphList.Delete(FIndexs[ii]);
      FIndexs[ii] := -1;
   end;
   GlyphCache.ReturnList(FGlyphList);
   FGlyphList := nil;
end;

procedure TsEditSubControl.InvalidateParent;
var
   R: TRect;
begin
   if FParent.HandleAllocated then begin
      R := FParent.ClientRect;
      Exclude(FParent.FEditState, msNoRedraw);
      InvalidateRect( FParent.Handle, @R, TRUE);
      FParent.InternalPaintWindow;
   end;
end;

procedure TsEditSubControl.PictureChanged(Sender: TObject);
var
   Glyphs: Integer;
begin
   InvalidateGlyphCash;
   if not FPicture.Empty then begin
      FPicture.TransparentMode := tmAuto;
      if FPicture.Width mod FPicture.Height = 0 then begin
         Glyphs := FPicture.Width div FPicture.Height;
         if Glyphs > 4 then
            Glyphs := 1;
         FNumGlyphs := Glyphs;
      end else
         FNumGlyphs := 1;
   end;
   FParent.SetEditRect;
   if [csLoading, csReading] * FParent.ComponentState = [] then
      InvalidateParent;
end;

procedure TsEditSubControl.SetGlyphId(Value: Integer);
begin
	if FGlyphId <> Value then begin
		if csLoading in FParent.ComponentState then
   		FGlyphId := Value
		else begin
   		if FParent.FGlyphList = nil then
   			Exit;
      	if (Value > -1) then begin
            if FParent.FGlyphList[Value] = nil then
               Raise Exception.CreateFmt(SErrorGlyphNotFound, [Value]);
            if not FParent.FGlyphList[Value].InheritsFrom(TBitMap) then
      		   Raise Exception.Create(SErrorNeedsBmp);
         end;
   		FGlyphId := Value;
         RefreshGlyphId;
      end;
   end;
end;

procedure TsEditSubControl.RefreshGlyphId;
begin
   InvalidateGlyphCash;
   if FGlyphId = -1 then
      Exit;
   if (FParent.GlyphList = nil) or (FParent.FGlyphList[FGlyphId] = nil) then begin
      FPicture.Assign(nil);
   end else if FParent.FGlyphList <> nil then
      FPicture.Assign(FParent.FGlyphList[FGlyphId]);
end;

procedure TsEditSubControl.CheckGlyphId(oldId, newId: Integer);
begin
	if oldId = FGlyphId then begin
      if oldId = newId then
			RefreshGlyphId
      else
			GlyphListId := newId;
   end;
end;

procedure TsEditSubControl.SetVisible(Value: Boolean);
begin
   if FVisible <> value then begin
      FVisible := Value;
      if [csLoading, csreading] * FParent.ComponentState = [] then begin
         FParent.SetEditRect;
         InvalidateParent;
      end;
   end;
end;

procedure TsEditSubControl.SetWidth(Value: Integer);
begin
   if FWidth <> Value then begin
      FWidth := Value;
      if [csLoading, csreading] * FParent.ComponentState = [] then begin
         FParent.SetEditRect;
         InvalidateParent;
      end;
   end;
end;

procedure TsEditSubControl.SetPicture(Value: TBitMap);
begin
   InvalidateGlyphCash;
   FPicture.Assign(Value);
   FGlyphId := -1;
end;

procedure TsEditSubControl.SetNumGlyphs(Value: TNumGlyphs);
begin
   if (FNumGlyphs <> value) and (Value > 0) then begin
      InvalidateGlyphCash;
      FNumGlyphs := Value;
      if [csLoading, csreading] * FParent.ComponentState = [] then begin
         FParent.SetEditRect;
         FParent.Invalidate;
      end;
   end;
end;

function TsEditSubControl.GetGlyphStored: Boolean;
begin
   Result := FGlyphId = -1;
end;

function TsEditSubControl.GetGlyphList: TsGlyphList;
begin
   Result := FParent.FGlyphList;
end;

function TsEditSubControl.GetGlyph(Pict: TBitMap; State: TsButtonState): TBitMap;
var
   w, h: Integer;

   function GetBitmapFromIndex(index: Integer): TBitMap;
   begin
      Result := TBitMap.Create;
      Result.Width := w;
      Result.Height := h;
      ImageList_DrawEx(FGlyphList.Handle, Index, Result.Canvas.Handle, 0, 0, 0, 0,
            clNone, clNone, ILD_Transparent);
   end;

const
   ROP_DSPDxax = $00E20746;
var
   TmpBmp, DDB, MonoBmp: TBitmap;
   index: Integer;
   IRect, ORect: TRect;
   I: TsButtonState;
   DestDC: HDC;
begin
   Result := nil;
   I := State;
   if Ord(I) >= NumGlyphs then
      I := bsUp;

   w := Pict.Width div FNumGlyphs;
   h := Pict.Height;

   index := FIndexs[I];
   if index <> -1 then begin
      Result := GetBitmapFromIndex(index);
      Exit;
   end;

   if Pict.Empty or ((Pict.Width or Pict.Height) = 0) then
      Exit;

   ORect := Rect(Ord(I) * w, 0, (Ord(I) + 1) * w, h);
   IRect := Rect(0, 0, w, h);

   if FGlyphList = nil then begin
      if GlyphCache = nil then
         GlyphCache := TGlyphCache.Create;
      FGlyphList := GlyphCache.GetList(w, h);
   end;

   TmpBmp := TBitmap.Create;
   try
      TmpBmp.Width := w;
      TmpBmp.Height := h;
      TmpBmp.Canvas.Brush.Color := clBtnFace;
      TmpBmp.Palette := CopyPalette(Pict.Palette);
      case State of
         bsUp, bsDown: begin
            if (State = bsDown) and (State <> I) then begin
               TmpBmp.Canvas.Brush.Color := Pict.TransparentColor;
               TmpBmp.Canvas.FillRect( IRect);
               Inc(IRect.Top, 1);
               Inc(IRect.Left, 1);
               Dec(ORect.Bottom, 1);
               Dec(ORect.Right, 1);
            end;
            TmpBmp.Canvas.CopyRect(IRect, Pict.Canvas, ORect);
            TmpBmp.TransparentColor := Pict.TransparentColor;

            FIndexs[State] := FGlyphList.AddMasked(TmpBmp, TmpBmp.TransparentColor);
         end;
         bsDisabled: begin
            MonoBmp := nil;
            DDB := nil;
            try
               MonoBmp := TBitmap.Create;
               DDB := TBitmap.Create;
               DDB.Assign(Pict);
               DDB.HandleType := bmDDB;
               if NumGlyphs > 1 then with TmpBmp.Canvas do begin { Change white & gray to clBtnHighlight and clBtnShadow }
                  CopyRect(IRect, DDB.Canvas, ORect);
                  MonoBmp.Monochrome := True;
                  MonoBmp.Width := w;
                  MonoBmp.Height := h;
                  { Convert white to clBtnHighlight }
                  DDB.Canvas.Brush.Color := clWhite;
                  MonoBmp.Canvas.CopyRect(IRect, DDB.Canvas, ORect);
                  Brush.Color := clBtnHighlight;
                  DestDC := Handle;
                  SetTextColor(DestDC, clBlack);
                  SetBkColor(DestDC, clWhite);
                  BitBlt(DestDC, 0, 0, w, h,
                     MonoBmp.Canvas.Handle, 0, 0, ROP_DSPDxax);
                  { Convert gray to clBtnShadow }
                  DDB.Canvas.Brush.Color := clGray;
                  MonoBmp.Canvas.CopyRect(IRect, DDB.Canvas, ORect);
                  Brush.Color := clBtnShadow;
                  DestDC := Handle;
                  SetTextColor(DestDC, clBlack);
                  SetBkColor(DestDC, clWhite);
                  BitBlt(DestDC, 0, 0, w, h,
                     MonoBmp.Canvas.Handle, 0, 0, ROP_DSPDxax);
                  { Convert transparent color to parent.color }
                  DDB.Canvas.Brush.Color := ColorToRGB(Pict.TransparentColor);
                  MonoBmp.Canvas.CopyRect(IRect, DDB.Canvas, ORect);
                  Brush.Color := FParent.Color;//clBtnFace;
                  DestDC := Handle;
                  SetTextColor(DestDC, clBlack);
                  SetBkColor(DestDC, clWhite);
                  BitBlt(DestDC, 0, 0, w, h,
                     MonoBmp.Canvas.Handle, 0, 0, ROP_DSPDxax);
               end else begin { Create a disabled version }
                  with MonoBmp do begin
                     Assign(Pict);
                     HandleType := bmDDB;
                     Canvas.Brush.Color := clBlack;
                     Width := w;
                     if Monochrome then begin
                        Canvas.Font.Color := clWhite;
                        Monochrome := False;
                        Canvas.Brush.Color := clWhite;
                     end;
                     Monochrome := True;
                  end;
                  with TmpBmp.Canvas do begin
                     Brush.Color := clBtnFace;
                     FillRect(IRect);
                     Brush.Color := clBtnHighlight;
                     SetTextColor(Handle, clBlack);
                     SetBkColor(Handle, clWhite);
                     BitBlt(Handle, 1, 1, w, h,
                        MonoBmp.Canvas.Handle, 0, 0, ROP_DSPDxax);
                     Brush.Color := clBtnShadow;
                     SetTextColor(Handle, clBlack);
                     SetBkColor(Handle, clWhite);
                     BitBlt(Handle, 0, 0, w, h,
                        MonoBmp.Canvas.Handle, 0, 0, ROP_DSPDxax);
                  end;
               end;
            finally
               DDB.Free;
               MonoBmp.Free;
            end;
            FIndexs[State] := FGlyphList.AddMasked(TmpBmp, TmpBmp.TransparentColor);
         end;
      end;
   except
      TmpBmp.Free;
      raise;
   end;
   Result := GetBitmapFromIndex(FIndexs[State]);
   Pict.Dormant;
end;

procedure TsEditSubControl.Paint;
var
   R: TRect;
   Bmp: TBitMap;
begin
   if not FParent.HandleAllocated then
      Exit;
   if Visible then begin
      FParent.FCanvas.Brush.Color := GetBackgroundColor;
      R := PictureRect;
      FParent.FCanvas.FillRect( R);
      Bmp := GetGlyph(FPicture, FState);
      if Bmp <> nil then begin
         InternalDrawTransBmpRect(FParent.FCanvas.Handle, R, Bmp,
             not (scoNoStretch in FOptions));
         Bmp.Free;
      end;
   end;
end;
{******************** TsEditPicture ************************}
function TsEditPicture.GetPictureRect: TRect;
begin
   Result := Rect(0, 0, 0, 0);
   Exclude(FOptions, scoNoStretch);
   if Visible then begin
      Result := FParent.ClientRect;
      InflateRect(Result, -1, -1);

      if FWidth > 0 then
         Result.Right := Result.Left + FWidth
      else if not FPicture.Empty then begin
         if heightOf(Result) >= FPicture.Height then begin
            Result.Right := Result.Left + Trunc(FPicture.Width / FNumGlyphs);
            Include(FOptions, scoNoStretch);
         end else
            Result.Right := Result.Left + 2 +
               Trunc( FPicture.Width * (HeightOf(FParent.ClientRect)-2) / FPicture.Height) div FNumGlyphs + 1;
      end else
         Result.Right := Result.Left;
   end;
end;

function TsEditPicture.GetBackgroundColor: TColor;
begin
   Result := FParent.Color;
end;

procedure TsEditPicture.SetState(Value: TsButtonState);
begin
   if Ord(Value) >= NumGlyphs then
      Value := bsUp;
   if FState <> Value then begin
      FState := Value;
      Paint;
   end;
end;

{******************** TsCustomEdit ************************}
constructor TsCustomEdit.Create( AOwner: TComponent);
begin
   inherited Create( AOwner);
   ControlStyle := ControlStyle - [csSetCaption, csFramed];
   FCanvas := TControlCanvas.Create;
   TControlCanvas(FCanvas).Control := self;
   FPicture := TsEditPicture.Create(self);
   FDisabledFont := TFont.Create;
   FDisabledFont.Assign(Font);
   FDisabledStyle := dsDefault;
   AutoSize := FALSE;
   FColor := inherited Color;
   FParentColor := inherited ParentColor;
end;

destructor TsCustomEdit.Destroy;
begin
   FDisabledFont.Free;
   FPicture.Free;
   FCanvas.Free;
	inherited;
end;

procedure TsCustomEdit.CreateParams( var Params: TCreateParams );
const
   Alignments: array[TAlignment] of Word = (ES_LEFT, ES_RIGHT, ES_CENTER);
begin
   inherited CreateParams( Params);
   with Params do begin
      Style := Style or Alignments[FAlignment] or ES_MULTILINE;
		if FFlat then
     		ExStyle := ExStyle or WS_EX_CLIENTEDGE
   end;
end;

procedure TsCustomEdit.CreateWnd;
begin
   inherited;
   SetBounds(Left, Top, Width, Height);
   SetEditRect;
end;

procedure TsCustomEdit.WndProc(var Message: TMessage);
var
   ss, se: Integer;
begin
   if (PasswordChar = #0) or (msChanging in FEditState) then begin
      inherited;
      case  Message.Msg of
         WM_CHAR, WM_KEYDOWN, WM_CUT, WM_PASTE, WM_CLEAR:
            FText := inherited Text;
         EM_SETPASSWORDCHAR: begin
            Include( FEditState, msChanging);
            inherited Text := FText;
            Exclude( FEditState, msChanging);
         end;
      end;
   end else case Message.Msg of
      EM_SETPASSWORDCHAR: begin
         Include( FEditState, msChanging);
         FText := inherited Text;
         inherited Text := GetPasswordText;
         Exclude( FEditState, msChanging);
      end;
      WM_CHAR, WM_KEYDOWN, WM_PASTE, WM_UNDO:
         if (Message.Msg = WM_KEYDOWN) and not
            (TWMKeyDown(message).CharCode in [VK_DELETE, VK_BACK]) then
            inherited
         else begin
            Perform( WM_SETREDRAW, Integer(FALSE), 0);
            Perform( EM_GETSEL, Integer(@ss), Integer(@se));
            inherited Text := FText;
            Perform( EM_SETSEL, ss, se);
            inherited;
            if FText <> inherited Text then
               ChangeExpected;
            FText := inherited Text;
            Perform( EM_GETSEL, Integer(@ss), Integer(@se));
            inherited Text := GetPasswordText;
            Perform( EM_SETSEL, ss, se);
            Include(FEditState, msNoRedraw);
            Perform( WM_SETREDRAW, Integer(TRUE), 0);
            Exclude( FEditState, msChanging);
            Change;
         end;
      WM_CUT, WM_COPY, WM_CONTEXTMENU:
         ;
      else
         inherited;
   end;
end;

procedure TsCustomEdit.WMPaint(var Message: TWMPaint);
var
   DC: HDC;
   PS: TPaintStruct;
begin
   try
      DC := BeginPaint(Handle, PS);
      if Enabled or (FDisabledStyle <> dsEditTools) then begin
         PaintWindow(DC);
         InternalPaintWindow;
      end else
         PaintDisabled( PS);
      Exclude( FEditState, msNoRedraw);
   finally
      EndPaint(Handle, PS);
   end;
end;

procedure TsCustomEdit.WMEraseBkgnd(var Message: TMessage);
begin
   if Picture.Visible and (WidthOf(Picture.PictureRect) > 0) then
      ExcludeClipRect(Message.wParam, 0, 0, GetEditRect.Left, Height);
   inherited;
end;

procedure TsCustomEdit.WMNCPaint(var Message: TMessage);
begin
   inherited;
   InternalPaintWindow;
end;

procedure TsCustomEdit.WMSetFocus(var Message: TMessage);
begin
   inherited;
   if AutoSelect and not (csLButtonDown in ControlState) then
      SelectAll;
   SetSolidBorder;
   CheckCursor(FALSE);
end;

procedure TsCustomEdit.WMKillFocus(var Message: TMessage);
begin
   SetSolidBorder;
   inherited;
end;

procedure TsCustomEdit.WMGlyphIdChanged(var Message: TMessage);
begin
   Picture.CheckGlyphId(Message.WParam, Message.LParam);
end;

procedure TsCustomEdit.CMMouseEnter(var Message: TMessage);
begin
   inherited;
   if not (msMouseInControl in FEditState) and Enabled then begin
      Include(FEditState, msMouseInControl);
      SetSolidBorder;
      Notify( self, STM_MOUSEENTERNOTIFY, 0);
   end;
end;

procedure TsCustomEdit.CMMouseLeave(var Message: TMessage);
begin
   inherited;
   if (msMouseInControl in FEditState) and Enabled then begin
      Exclude(FEditState, msMouseInControl);
      SetSolidBorder;
      Notify( self, STM_MOUSELEAVENOTIFY, 0);
   end;
end;

procedure TsCustomEdit.CMEnabledChanged(var Msg: TMessage);
begin
   inherited;
   SetupEnabled;
   Notify( self, STM_ENABLEDCHANGENOTIFY, Ord(ENABLED));
end;

procedure TsCustomEdit.CMFontChanged( var Message: TMessage);
begin
   inherited;
   Height := GetFontHeight(Font) + GetBorderSize + 2;
end;

procedure TsCustomEdit.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
   inherited;
   Message.Result := Message.Result and not DLGC_WANTALLKEYS;
end;

{$IFDEF V1_COMP}
procedure TsCustomEdit.ReadDefaultDraw(Reader: TReader);
begin
   if Reader.ReadBoolean then
      DisabledStyle := dsDefault
   else
      DisabledStyle := dsOffice97;
end;

procedure TsCustomEdit.WriteNothing(Writer: TWriter);
begin
end;

procedure TsCustomEdit.DefineProperties(Filer: TFiler);
begin
   inherited;
   Filer.DefineProperty('DefaultDraw', ReadDefaultDraw, WriteNothing, FALSE);
end;
{$ENDIF}

function TsCustomEdit.GetBorderSize: Integer;
begin
   Result := 2;
end;

function TsCustomEdit.GetEditRect: TRect;
begin
   Result := GetClientRect;
   if FPicture.Visible then
      Inc(Result.Left, FPicture.PictureRect.Right)
end;

procedure TsCustomEdit.SetEditRect;
var
  	R: TRect;
begin
   if HandleAllocated then begin
      R := GetEditRect;
      SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@R));
   end;
end;

procedure TsCustomEdit.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
   inherited SetBounds(ALeft, ATop, AWidth, GetFontHeight(Font) + GetBorderSize+2);
   if (FPicture <> nil) and ([csreading, csLoading] * ComponentState = []) then
      SetEditRect;
end;

procedure TsCustomEdit.Change;
begin
   if not (msChanging in FEditState) then begin
      inherited;
      CheckCursor(FALSE);
   end;
end;

procedure TsCustomEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
	inherited;
   if (AComponent = FGlyphList) and (Operation = opRemove) then begin
      FGlyphList := nil;
      Invalidate;
   end;
end;

procedure TsCustomEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
   if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
      ChangeExpected;
   inherited;
end;

procedure TsCustomEdit.KeyPress(var Key: Char);
begin
   if Key = #27 then begin
      Reset;
      Key := #0;
   end else if Key = #13 then
      Key := #0
   else if not IsValidChar(Key) then begin
      ValidateError;
   	Key := #0
   end;
   ChangeExpected;
	inherited;
end;

function TsCustomEdit.IsValidChar(const Key: Char): Boolean;
begin
   Result := not (Key in [#0, #3, #9, #10, #22, #24]);
   if Result and (ValidChars > '') then
      Result := Pos(key, validChars) > 0;
end;

function TsCustomEdit.GetPasswordText: String;
var
   ii: Integer;
begin
   SetLength( Result, Length(FText));
   for ii := 1 to length(result) do
      result[ii] := PasswordChar;
end;

function TsCustomEdit.EditCanModify: Boolean;
begin
	Result := not ReadOnly;
end;

procedure TsCustomEdit.Reset;
begin
  	if Modified then begin
    	Text := FOldValue;
    	Modified := False;
  	end;
end;

procedure TsCustomEdit.ChangeExpected;
begin
end;

procedure TsCustomEdit.DoEnter;
begin
   if not (msReEnter in FEditState) then
   	FOldValue := Text;
   Exclude(FEditState, msReEnter);
   inherited;
end;

procedure TsCustomEdit.DoExit;
begin
	ValidateEdit;
   inherited;
end;

function TsCustomEdit.GetSolidBorder: Boolean;
begin
   Result := ((csDesigning in ComponentState) and Enabled) or
      (not(csDesigning in ComponentState) and
      (Focused or ((msMouseInControl in FEditState) and IsFlatApplied)));
end;

procedure TsCustomEdit.SetSolidBorder;
var
   sb: Boolean;
begin
   sb := GetSolidBorder;
   if sb <> FSolidBorder then begin
      FSolidBorder := sb;
      if ([csLoading, csReading] * ComponentState = []) then
         RedrawBorders;
   end;
end;

procedure TsCustomEdit.CMVisibleChanged(var Message: TMessage);
begin
   inherited;
   Exclude(FEditState, msFirstTimePainted);
end;

procedure TsCustomEdit.InternalPaintWindow;
var
   R: TRect;
begin
   if not (msFirstTimePainted in FEditState) and
      (HandleAllocated and IsWindowVisible(Handle)) then begin
      Exclude(FEditState, msNoRedraw);
      Include(FEditState, msFirstTimePainted);
   end;
   if not (msNoRedraw in FEditState) then begin
      if FFlat then
         InternalRedrawBorderEx( Handle, SolidBorder, GetEditRect.Right);
      if HandleAllocated then begin
         GetUpdateRect(Handle, R, False);
         if RectInRect(R, FPicture.PictureRect) then
            FPicture.Paint;
      end;
   end;
end;

procedure TsCustomEdit.PaintDisabled(PS: TPaintStruct);
const
   alignments: array[TAlignment] of Word = (0, DT_RIGHT, DT_CENTER);
var
   R: TRect;
   H: THandle;
begin
   FCanvas.Brush.Color := Color;
   FCanvas.FillRect(ClientRect);
   H := SelectObject(FCanvas.Handle, FDisabledFont.Handle);
   R := GetEditRect;
   DrawText(FCanvas.Handle, PChar(Text), Length(Text), R, DT_VCENTER or DT_NOCLIP or Alignments[Alignment]);
   SelectObject(FCanvas.Handle, H);
   Picture.Paint;
end;

procedure TsCustomEdit.RedrawBorders;
var
   st: TsButtonState;
begin
   if FFlat and not (csDesigning in ComponentState) then
      InternalRedrawBorder( Handle, SolidBorder);
      st := bsUp;
      if (msMouseInControl in FEditState) or Focused then
         st := bsDown;
   FPicture.State := st;
end;

procedure TsCustomEdit.GetSel(var SelStart: Integer; var SelStop: Integer);
begin
  	SendMessage(Handle, EM_GETSEL, Integer(@SelStart), Integer(@SelStop));
end;

procedure TsCustomEdit.SetSel(const SelStart, SelStop: Integer);
begin
  	SendMessage(Handle, EM_SETSEL, SelStart, SelStop);
end;

procedure TsCustomEdit.SetupEnabled;
const
   states: array[Boolean] of TsButtonState = (bsDisabled, bsUp);
begin
   if ([csLoading,csreading] * ComponentState = []) then begin
      if Flat and (FdisabledStyle = dsOffice97) then begin
         if Enabled then begin
            inherited ParentColor := FParentColor;
            inherited Color := FColor;
         end else begin
            FParentColor := inherited Parentcolor;
            FColor := inherited Color;
            inherited Parentcolor := TRUE;
         end;
      end;
      if Enabled then
         Font.Assign(FCanvas.Font)
      else begin
         FCanvas.Font.Assign(Font);
         Font.Assign(FDisabledFont);
      end;
      FPicture.FState := states[Enabled];
   end;
end;

procedure TsCustomEdit.ValidateEdit;
var
   Pos: Integer;
begin
  	if Modified and not Validate(pos) then begin
      if not (csDesigning in ComponentState) then begin
         Include(FEditState, msReEnter);
         if Pos = -1 then
            SelectAll
         else
            SetCursor(Pos);
         SetFocus;
      end;
      ValidateError;
  	end;
end;

function TsCustomEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := TRUE;
   if Assigned(FOnValidate) then
      FOnValidate(self, Result);
end;

procedure TsCustomEdit.ValidateError;
begin
   if Assigned(FOnValidateError) then
      FOnValidateError(self)
   else
	   MessageBeep( MB_ICONHAND);
end;

procedure TsCustomEdit.CheckCursor(force: Boolean);
begin
end;

{ properties handle}
procedure TsCustomEdit.SetAlignment(Value: TAlignment);
begin
   if FAlignment <> Value then begin
      FAlignment := Value;
      RecreateWnd;
   end;
end;

procedure TsCustomEdit.SetDisabledStyle(Value: TsDisabledStyle);
begin
   if FDisabledStyle <> Value then begin
      FDisabledStyle := Value;
      if not Enabled then
        Invalidate;
   end;
end;

procedure TsCustomEdit.SetDisabledFont(Value: TFont);
begin
   FDisabledFont.Assign(Value);
   if not Enabled then
      Invalidate;
end;

procedure TsCustomEdit.SetFlat(const Value: Boolean);
begin
   if Value <> FFlat then begin
      FFlat := Value;
      if FFlat then
         SetupEnabled;
//      InternalPaintWindow;
// TVS: Force new window because there is a other Flag set for the "Flat"-Property
      RecreateWnd;
   end;
end;

procedure TsCustomEdit.SetGlyphList(Value: TsGlyphList);
begin
	if FGlyphList <> Value then begin
      if FGlyphList <> nil then
			FGlyphList.ChangeNotification(self, FALSE);
   	FGlyphList := Value;
      if FGlyphList <> nil then
			FGlyphList.ChangeNotification(self, TRUE);
      RefreshChildGlyphs;
   end;
end;

procedure TsCustomEdit.RefreshChildGlyphs;
begin
   FPicture.RefreshGlyphId;
end;

procedure TsCustomEdit.SetPicture(Value: TsEditPicture);
begin
   FPicture.Assign( Value);
end;

function TsCustomEdit.GetText: String;
begin
   if PasswordChar = #0 then
      Result := inherited Text
   else
      Result := FText;
end;

procedure TsCustomEdit.DoSetText(Value: String);
begin
   if HandleAllocated and (Alignment <> taLeftJustify) and
      (((maxLength > 0) and (Length(Value) > maxLength)) or
      (FCanvas.TextWidth(Value) > WidthOf(GetEditRect))) then begin
      MessageBeep( MB_ICONHAND);
      Exit;
   end;

   FText := Value;
   if PasswordChar = #0 then
      inherited Text := FText
   else
      inherited Text := GetPasswordText;
   Modified := TRUE;
end;

procedure TsCustomEdit.SetText(Value: String);
begin
   DoSetText(Value);
   if HandleAllocated and IsWindowVisible(Handle) then
      Include( FEditState, msNoRedraw);
end;

function TsCustomEdit.ValidCharsStored: Boolean;
begin
   Result := FValidChars > '';
end;

{******************** TsEditButton ************************}
const
   UpDownBmp: TBitMap = nil;
   UpDownBmpName = 'SBTNUPDOWNSMALL';


constructor TsEditButton.Create( aParent: TsCustomEdit);
begin
   inherited;
   FAutoRepeat := TRUE;
   FCursor := crArrow;
   FShowHint := TRUE;
   FClickKey := scAlt + vk_Down;
   FClickKeyUp := scAlt + vk_Up;
   FButtonKind := bkNormal;
   FGlyphKind := gkCustom;
end;

procedure TsEditButton.PictureChanged(Sender: TObject);
begin
   inherited;
   if not (scoChanging in FOptions) then
      FGlyphKind := gkCustom;
{
   if (FButtonKind = bkUpDown) then begin
      if FPicture.Empty then begin
         if UpDownBmp = nil then begin
            UpDownBmp := TBitMap.Create;
            UpDownBmp.Handle := LoadBitmap(HInstance, UpDownBmpName);
         end;
         FPicture.Assign(UpDownBmp);
         FPicture.TransparentMode := tmAuto;
         FGlyphKind := gkDefault;
         FNumGlyphs := 1;
      end else if FPicture.Height mod 2 <> 0 then
         Raise ELinkEditError.Create(SErrorInvalidUpDownGlyph);
      FParent.Refresh;
      FParent.InternalPaintWindow;
   end;
}
end;

function TsEditButton.GetPictureRect: TRect;
var
   H: Integer;
begin
   Result := Rect(0, 0, 0, 0);
   Exclude(FOptions, scoNoStretch);
   if Visible then begin
      Result := FParent.ClientRect;
      if Fparent.FFlat then
         InflateRect(Result, -1, -1)
      else
         InflateRect(Result, -1, -2);
      H := heightOf( Result);
      if FButtonState = bsUpButton then
         Result.Bottom := Result.Top + HeightOf(result) div 2 - 1
      else if FButtonState = bsDownButton then
         Result.Top := Result.Bottom - HeightOf(result) div 2 + 1;
      if not Fparent.FFlat then
         Result.Top := Result.Top - 1;
      if FWidth > 0 then
         Result.Left := Result.Right - FWidth
      else if not FPicture.Empty then begin
         if H >= FPicture.Height then begin
            Result.Left := Result.Right -  FPicture.Width;
            Include(FOptions, scoNoStretch);
         end else
         Result.Left := Result.Right -
               Trunc( FPicture.Width * (HeightOf(FParent.ClientRect)-2) / FPicture.Height) div FNumGlyphs+1;
      end else
         Result.Left := Result.Right - 20;
      if self is TsLeftEditButton then begin
         OffsetRect( Result, - WidthOf(FLinkedButton.GetDisplayRect), 0);
         if Fparent.FFlat and FLinkedButton.Visible then
            OffsetRect( Result, -1, 0);
      end;
   end;
end;

function TsEditButton.GetDisplayRect: TRect;
begin
   Result := GetPictureRect;
   if WidthOf(Result) > 0 then begin
      if Fparent.FFlat then
         InflateRect(Result, 1, 1)
      else begin
         InflateRect(Result, 1, 2);
         case FButtonState of
            bsDownButton: Inc(Result.Top);
            bsUpButton: Dec(Result.Bottom);
         end;
      end;
   end;
end;

function TsEditButton.GetGlyph(Pict: TBitMap; State: TsButtonState): TBitMap;
var
   IRect, ORect: TRect;
   Bmp: TBitMap;
begin
   Result := inherited GetGlyph(Pict, State);
   if (FButtonState = bsNormal) then
      Exit;

   Bmp := TBitMap.Create;
   try
      if FButtonState = bsUpButton then
         ORect := Rect(0, 0, Result.Width, Result.Height div 2)
      else
         ORect := Rect(0, Result.Height div 2, Result.Width, Result.Height);
      IRect := Rect(0, 0, Result.Width, Result.Height div 2);

      Bmp.Width := Result.Width;
      Bmp.Height := Result.Height div 2;
      Bmp.Canvas.CopyRect(IRect, Result.Canvas, ORect);
      Bmp.TransparentColor := Result.TransparentColor;
      Result.Assign( Bmp);
   finally
      Bmp.Free;
   end;
end;

procedure TsEditButton.RecreateUpDownGlyph;
begin
   if (FGlyphKind <> gkCustom) or FPicture.Empty then begin
      if UpDownBmp = nil then begin
         UpDownBmp := TBitMap.Create;
         UpDownBmp.Handle := LoadBitmap(HInstance, UpDownBmpName);
      end;
      Include( FOptions, scoChanging); // to preserve original GlyphKind
      FNumGlyphs := 1;
      FPicture.Assign(UpDownBmp);
      Exclude( FOptions, scoChanging);
   end else if FPicture.Height mod 2 <> 0 then
      Raise ELinkEditError.Create(SErrorInvalidUpDownGlyph)
   else
      InvalidateParent;
end;

type
   TGraphicTrash = class(TGraphic)
   protected
      function Equals(Graphic: TGraphic): Boolean; override;
   end;

function TGraphicTrash.Equals(Graphic: TGraphic): Boolean;
begin
   Result := inherited Equals(Graphic);
end;

procedure TsEditButton.SetButtonKind( Value: TsButtonKind);
begin
   if FButtonKind <> Value then begin
      FButtonKind := Value;
      FButtonState := bsNormal;
      if (FButtonKind = bkUpDown) then
         RecreateUpDownGlyph
      else if FGlyphKind <> gkCustom then
         RecreateGlyph
      else if (UpDownBmp <> nil) and TGraphicTrash(FPicture).Equals(UpDownBmp) then
         Glyph := nil
      else
         InvalidateParent;
   end;
end;

procedure TsEditButton.SetGlyphKind( Value: TGlyphKind);
begin
   if (FGlyphKind <> Value) then begin
      FGlyphKind := Value;
      if (FGlyphKind = gkCustom) and ([csLoading, csReading] * FParent.ComponentState = []) then
         Glyph := nil;
      if FButtonKind = bkNormal then
         RecreateGlyph
      else
         RecreateUpDownGlyph;
   end;
end;

procedure TsEditButton.RecreateGlyph;
   function CreateEllipsisGlyph: TBitmap;
   var
      W, G, I, M: Integer;
      R: TRect;
   begin
      Result := TBitmap.Create;
      try
         R := GetPictureRect;
         Result.Monochrome := True;
         Result.Width := WidthOf(R);
         Result.Height := HeightOf(R);
         M := Result.Height div 2;
         W := 2;
         G := (Result.Width - 4 * W) div 2;
         if G = 0 then G := 1;
         if G > 3 then G := 3;
         I := (Result.Width - 3 * W - 2 * G) div 2;
         PatBlt(Result.Canvas.Handle, I, M, W, W, BLACKNESS);
         PatBlt(Result.Canvas.Handle, I + G + W, M, W, W, BLACKNESS);
         PatBlt(Result.Canvas.Handle, I + 2 * G + 2 * W, M, W, W, BLACKNESS);
      except
         Result.Free;
         raise;
      end;
   end;
var
   NewGlyph: TBitmap;
   DestroyNeeded: Boolean;
begin
   try
      Include( FOptions, scoChanging);
      case FGlyphKind of
         gkDefault: begin
            DestroyNeeded := False;
            NewGlyph := TsLinkEdit(FParent).GetDefaultBitmap(DestroyNeeded);
            try
               FPicture.Assign(NewGlyph);
               FNumGlyphs := 1;
            finally
               if DestroyNeeded then
                  NewGlyph.Destroy;
            end;
         end;
         gkDropDown: begin
            FPicture.Handle := LoadBitmap(0, PChar(32738));
            FNumGlyphs := 1;
         end;
         gkEllipsis: begin
            NewGlyph := CreateEllipsisGlyph;
            try
               FPicture.Assign(NewGlyph);
               FNumGlyphs := 1;
            finally
               NewGlyph.Destroy;
            end;
         end;
      end;
   finally
      Exclude( FOptions, scoChanging);
   end;
end;

function TsEditButton.GetBackgroundColor: TColor;
begin
   Result := clBtnFace;
end;

procedure TsEditButton.SetState(Value: TsButtonState);
begin
   if FState <> Value then begin
      FState := Value;
      Paint;
   end;
end;

function TsEditButton.GetGlyphStored: Boolean;
begin
   Result := (FGlyphKind = gkCustom) and (FGlyphId = -1);
end;

procedure TsEditButton.Paint;
   procedure DoPaint;
   begin
      inherited;
      RedrawBorders
   end;
begin
   if (FButtonkind = bkNormal) or (FButtonState <> bsNormal) then
      DoPaint
   else begin
      FButtonState := bsUpButton;
      DoPaint;
      FButtonState := bsDownButton;
      DoPaint;
      FButtonState := bsNormal;
   end;
end;

procedure TsEditButton.RedrawBorders;
   procedure PaintBorders;
   const
      flags: array[Boolean] of Integer = ( EDGE_RAISED, EDGE_SUNKEN);
      Colors: array[Boolean] of TColor = (clBtnShadow, clWindow);
   var
      R: TRect;
      t, b: Integer;
   begin
      if not Visible then
         Exit;
      R := GetDisplayRect;

      if FParent.Flat then with FParent.FCanvas do begin
         if FParent.SolidBorder then begin
            Brush.Color := Colors[FState = bsDown];
            FrameRect(R);
            Pen.Color := Colors[FState <> bsDown];
            Polyline([Point(R.Right-2, R.Top), Point(R.Left, R.Top),
                     Point(R.Left, R.Bottom-1)]);
            t := R.Top;
            b := R.Bottom;
            if FButtonState = bsUpButton then
               Inc( b, 2)
            else if FButtonState = bsDownButton then
               Dec( t, 2);

            if (self is TsLeftEditButton) or not FLinkedButton.Visible then begin
               Pen.Color := clWindow;
               PolyLine([Point(R.Left - 2, t), Point(R.Left - 2, b)]);
            end;
            Pen.Color := clBtnFace;
            Polyline([Point(R.Left-1, t), Point(R.Left-1, b)]);
         end else begin
            Brush.Color := clWindow;
            FrameRect(R);
            InflateRect(R, 0, -1);

            if (self is TsLeftEditButton) or not FLinkedButton.Visible then begin
               Pen.Color := FParent.Color;
               t := R.Top;
               b := R.Bottom;
               if FButtonState = bsUpButton then
                  Inc( b, 2)
               else if FButtonState = bsDownButton then
                  Dec( t, 2);
               Polyline([Point(R.Left-2, t), Point(R.Left-2, b)]);
               Polyline([Point(R.Left-1, t), Point(R.Left-1, b)]);
            end;

            if FLinkedButton.Visible then begin
               if (self is TsRightEditButton) then begin
                  Pen.Color := clBtnFace;
                  Polyline([Point(R.Left, R.Top), Point(R.Left, R.Bottom)]);
                  Pen.Color := clWindow;
                  Polyline([Point(R.Left-1, R.Top), Point(R.Left-1, R.Bottom)]);
               end else begin
                  Pen.Color := clBtnFace;
                  Polyline([Point(R.Right-1, R.Top), Point(R.Right-1, R.Bottom)]);
                  Pen.Color := clWindow;
                  Polyline([Point(R.Right, R.Top), Point(R.Right, R.Bottom)]);
               end;
            end;

            InflateRect(R, 0, 1);
            if (FButtonKind = bkUpDown) and (HeightOf( FParent.ClientRect) mod 2 = 1) then begin
               Pen.Color := clBtnFace;
               if (self is TsLeftEditButton) or not FLinkedButton.Visible then begin
                  Inc(R.Left);
                  if not (self is TsRightEditButton) then
                     Inc(R.Right);
               end;
               if FButtonState = bsUpButton then
                  Polyline([Point(R.Left, R.Bottom-1), Point(R.Right-1, R.Bottom-1)])
               else if FButtonState = bsDownButton then
                  Polyline([Point(R.Left, R.Top), Point(R.Right-1, R.Top)]);
               if (self is TsLeftEditButton) or not FLinkedButton.Visible then begin
                  Dec(R.Left);
                  if (self is TsRightEditButton) then
                     Dec(R.Right);
               end;
            end;
         end;

         if (FButtonKind = bkUpDown) and (HeightOf( FParent.ClientRect) mod 2 = 1) then begin
            if not FParent.Flat or FParent.SolidBorder then
               Pen.Color := clBtnFace
            else
               Pen.Color := clWindow;
            if (self is TsRightEditButton) and FLinkedButton.Visible then
               Dec(R.Left);

            if FButtonState = bsUpButton then
               Polyline([Point(R.Left, R.Bottom), Point(R.Right, R.Bottom)])
            else if FButtonState = bsDownButton then
               Polyline([Point(R.Left, R.Top-1), Point(R.Right, R.Top-1)]);
            if (self is TsRightEditButton) and FLinkedButton.Visible then
               Inc(R.Left);
         end;
      end else
         DrawEdge (FParent.FCanvas.Handle, R, Flags[FState = bsDown], BF_RECT);
   end;
begin
   if (FButtonkind = bkNormal) or (FButtonState <> bsNormal) then
      PaintBorders
   else begin
      FButtonState := bsUpButton;
      PaintBorders;
      FButtonState := bsDownButton;
      PaintBorders;
      FButtonState := bsNormal;
   end;
end;

{******************** TsLeftEditButton ************************}
constructor TsLeftEditButton.Create( aParent: TsCustomEdit);
begin
   inherited;
   FClickKey := scCtrl + vk_Down;
   FClickKeyUp := scCtrl + vk_Up;
   Visible := FALSE;
end;

{******************** TsCustomLinkEdit ************************}
constructor TsCustomLinkEdit.Create( AOwner: TComponent );
begin
   inherited;
   FButton1 := TsRightEditButton.Create(self);
   FButton2 := TsLeftEditButton.Create(self);
   FButton1.FLinkedButton := FButton2;
   FButton2.FLinkedButton := FButton1;
end;

destructor TsCustomLinkEdit.Destroy;
begin
   FButton1.Free;
   FButton2.Free;
   inherited;
end;

procedure TsCustomLinkEdit.WMGlyphIdChanged(var Message: TMessage);
begin
   inherited;
   FButton1.CheckGlyphId(Message.WParam, Message.LParam);
   FButton2.CheckGlyphId(Message.WParam, Message.LParam);
end;

procedure TsCustomLinkEdit.WMEraseBkgnd(var Message: TMessage);
var
   r: Integer;
begin
   if FButton1.Visible or FButton2.Visible then begin
      r := GetEditRect.Right;
      if not Flat then
         Inc(r, 2);
      ExcludeClipRect(Message.wParam, r, 0, Width, Height);
   end;
   inherited;
end;

procedure TsCustomLinkEdit.WMLButtonDown(var Message: TWMLButtonDown);
begin
   FTrackState := ebNone;
   if EditCanModify then begin
      if PtInRect( FButton1.GetDisplayRect, Point(message.XPos, message.YPos)) then
         FTrackState := ebButton1
      else if PtInRect( FButton2.GetDisplayRect, Point(message.XPos, message.YPos)) then
         FTrackState := ebButton2;
   end;

   if FTrackState <> ebNone then begin
      SendCancelMode(Self);
      SetFocus;
      MouseCapture := True;
      TrackButton(message.XPos, message.YPos);
   end else begin
   	CheckCursor(FALSE);
  		inherited;
   end;
end;

procedure TsCustomLinkEdit.WMRButtonDown(var Message: TWMLButtonDown);
begin
   if not PtInRect( FButton1.GetDisplayRect, Point(message.XPos, message.YPos)) and
      not PtInRect( FButton2.GetDisplayRect, Point(message.XPos, message.YPos)) then
      inherited;
end;

procedure TsCustomLinkEdit.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
   if PtInRect( FButton1.GetDisplayRect, Point(message.XPos, message.YPos)) or
      PtInRect( FButton2.GetDisplayRect, Point(message.XPos, message.YPos)) then
      WMLButtonDown(Message)
   else
      inherited;
end;

procedure TsCustomLinkEdit.WMSetCursor(var Message: TWMSetCursor);
var
  	P: TPoint;
   function DoCheckCursor(button: TsEditButton): Boolean;
   begin
      Result := PtInRect( Button.GetDisplayRect, P);
      if Result then
         Windows.SetCursor(Screen.Cursors[Button.Cursor])
   end;
begin
	GetCursorPos(P);
   P := ScreenToClient(P);
   if not DoCheckCursor(FButton1) then
      if not DoCheckCursor(FButton2) then
         inherited;
end;

procedure TsCustomLinkEdit.CMHintShow(var Message: TMessage);
var
   P: TPoint;
   function GetButtonHint( Button: TsEditButton): Boolean;
   begin
      Result := PtInRect( Button.GetDisplayRect, P);
      if Result and Button.ShowHint then
         (PHintInfo(Message.LParam))^.HintStr := Button.Hint;
   end;
begin
   if FTrackState <> ebNone then
      Message.Result := 1
   else begin
	   GetCursorPos(P);
      P := ScreenToClient(P);
      if not GetButtonHint( Button1) then
         GetButtonHint( Button2);
   end;
end;

procedure TsCustomLinkEdit.CMMouseLeave(var Message: TMessage);
begin
{
   There is a small problemm with CM_MOUSELEAVE message: it is generated by
   Delphi when some dialog is called (ShowMessage for example) no metter if
   mouse actually leaving the control or not. This fixes this issue (especially
   important for UpDown button).
}
   if FTrackState = ebNone then  
      inherited;
end;

{$IFDEF V1_COMP}
procedure TsCustomLinkEdit.ReadGlyphKind(Reader: TReader);
const
   sGlyphKinds: array[TGlyphKind] of String = ('gkCustom', 'gkDefault', 'gkDropDown', 'gkEllipsis');
var
   S: String;
   ii: TGlyphKind;
begin
   S := Reader.ReadIdent;
   for ii := Low(TGlyphKind) to High(TGlyphKind) do
      if S = sGlyphKinds[ii] then begin
         Button1.GlyphKind := ii;
         Break;
      end;
end;

procedure TsCustomLinkEdit.ReadClickKey(Reader: TReader);
begin
   Button1.ClickKey := Reader.ReadInteger;
end;

procedure TsCustomLinkEdit.DefineProperties(Filer: TFiler);
begin
  inherited;
  Filer.DefineProperty('GlyphKind', ReadGlyphKind, WriteNothing, FALSE);
  Filer.DefineProperty('ClickKey', ReadClickKey, WriteNothing, FALSE);
end;
{$ENDIF}

function TsCustomLinkEdit.GetEditRect: TRect;
begin
   Result := inherited GetEditRect;
   if Enabled or (DisabledStyle <> dsEditTools) then begin
      if FButton1.Visible then
         Dec(Result.Right, WidthOf(FButton1.GetDisplayRect));
      if FButton2.Visible then
         Dec(Result.Right, WidthOf(FButton2.GetDisplayRect));
      if FButton1.Visible or FButton2.Visible then
         Dec(result.Right, 2);
   end;
end;

procedure TsCustomLinkEdit.InternalPaintWindow;
var
   R: TRect;
begin
   inherited;
   if not (msNoRedraw in FEditState) and HandleAllocated then begin
      GetUpdateRect(Handle, R, False);
      if RectInRect(R, FButton1.PictureRect) then
         FButton1.Paint;
      if RectInRect(R, FButton2.PictureRect) then
         FButton2.Paint;
   end;
end;

procedure TsCustomLinkEdit.RedrawBorders;
begin
   inherited;
   if ([csLoading, csReading] * ComponentState = []) then begin
      FButton1.RedrawBorders;
      FButton2.RedrawBorders;
   end;
end;

procedure TsCustomLinkEdit.SetupEnabled;
const
   states: array[Boolean] of TsButtonState = (bsDisabled, bsUp);
begin
   inherited;
   FButton1.FState :=  states[enabled];
   FButton2.FState :=  states[enabled];
end;

function TsCustomLinkEdit.CheckShortCut(var Key: Word; Shift: TShiftState): Boolean;
const
   buttons: array[Boolean] of TsEditButtons = (ebButton2, ebButton1);
var
	Btn: TsEditButton;
   sc: TShortCut;
begin
   Result := FALSE;
   if not EditCanModify then
      Exit;
   Btn := nil;
   sc := ShortCut(Key, Shift);
   if (sc = Button1.FClickKey) or
      ((Button1.ButtonKind = bkUpDown) and (sc = Button1.FClickKeyUp)) then
      Btn := Button1
   else if (sc = Button2.FClickKey) or
      ((Button2.ButtonKind = bkUpDown) and (sc = Button2.FClickKeyUp)) then
      Btn := Button2;
   if Btn <> nil then begin
      if Btn.ButtonKind = bkUpDown then begin
         if sc = btn.FClickKey then
            Btn.FButtonState := bsDownButton
         else
            Btn.FButtonState := bsUpButton
      end;
      ButtonClick( buttons[Btn = FButton1]);
      Btn.FButtonState := bsNormal;
      Key := 0;
      Result := TRUE;
   end;
end;

procedure TsCustomLinkEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
   if not CheckShortCut(Key, Shift) then
      inherited KeyDown(Key, Shift);
end;

procedure TsCustomLinkEdit.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
	Pressed: TsEditButtons;
begin
   if FTrackState = ebButton1 then
	   Pressed := TsEditButtons(Ord(FButton1.State = bsDown))
   else
      Pressed := TsEditButtons(2 * Ord(FButton2.State = bsDown));
  	if (FTrackState <> ebNone) then begin
      if Pressed <> ebNone then
         ButtonClick(Pressed);
      StopTracking;
   end else
      inherited MouseUp(Button, Shift, X, Y);
end;

procedure TsCustomLinkEdit.MouseMove(Shift: TShiftState; X, Y: Integer);
var
   oldHintForBtn: TsEditButtons;
begin
  	if FTrackState <> ebNone then
      TrackButton(X, Y)
   else begin
      // hint support.
      oldHintForBtn := FHintForBtn;
      FHintForBtn := TsEditButtons(Ord(PtInRect( Button1.GetDisplayRect, Point( X, Y))));
      if FHintForBtn = ebNone then
         FHintForBtn := TsEditButtons(2 * Ord(PtInRect( Button2.GetDisplayRect, Point( X, Y))));
      if FHintForBtn <> oldHintForBtn then
         Application.CancelHint;
   end;
  	inherited MouseMove(Shift, X, Y);
end;

var
   RepeatTimer: TTimer;
   RepeatStarted: Boolean;

procedure TsCustomLinkEdit.TrackButton(const X, Y: Integer);
var
   Button: TsEditButton;
   Pressed: Boolean;
   R: TRect;
begin
   if FTrackState = ebButton1 then
      Button := Button1
   else
      Button := Button2;
   if (Button.ButtonKind = bkNormal) or (Button.FButtonState <> bsNormal) then begin
      R := Button.GetDisplayRect;
      Pressed := PtInRect(R, Point(X, Y));
   end else begin
      Button.FButtonState := bsUpButton;
      R := Button.GetDisplayRect;
      Pressed := PtInRect(R, Point(X, Y));
      if not Pressed then begin
         Button.FButtonState := bsDownButton;
         R := Button.GetDisplayRect;
         Pressed := PtInRect(R, Point(X, Y));
      end;
      if not Pressed then begin
         Button.FButtonState := bsNormal;
         Exit;
      end;
   end;
   if (Button.ButtonKind <> bkNormal) and Button.FAutoRepeat then begin
      if RepeatTimer = nil then
         RepeatTimer := TTimer.Create(parent);
      if not Assigned(RepeatTimer.OnTimer) then
         RepeatTimer.OnTimer := OnTimer;
      if not RepeatStarted then
         RepeatTimer.Interval := InitRepeatPause;
   end;
  	if (Button.State = bsDown) <> Pressed then begin
      if Pressed then
         Button.State := bsDown
      else
         Button.State := bsUp;
      if (Button.ButtonKind <> bkNormal) and Button.FAutoRepeat then
         RepeatTimer.Enabled := Pressed;
  	end;
end;

procedure TsCustomLinkEdit.StopTracking;
begin
  	if FTrackState <> ebNone then begin
    	TrackButton(-1, -1);
      if RepeatTimer <> nil then
         RepeatTimer.OnTimer := nil;
      RepeatStarted := FALSE;
    	FTrackState := ebNone;
      Button1.FButtonState := bsNormal;
      Button2.FButtonState := bsNormal;
    	MouseCapture := False;
  	end;
   Exclude( FEditState, msNoPopupCall);
end;

procedure TsCustomLinkEdit.OnTimer(Sender: TObject);
begin
   if RepeatStarted then begin
      if FTrackState <> ebNone then try
         ButtonClick(FTrackState);
      except
         StopTracking;
         Raise;
      end;
   end else begin
      RepeatTimer.Interval := RepeatPause;
      RepeatStarted := TRUE;
   end;
end;

procedure TsCustomLinkEdit.ButtonClick(Button: TsEditButtons);
begin
   if (button = ebButton1) and Assigned(FOnButton1Click) then
      FOnButton1Click(self);
   if (button = ebButton2) and Assigned(FOnButton2Click) then
      FOnButton2Click(self);
end;

function TsCustomLinkEdit.GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap;
begin
   Result := nil;
end;

procedure TsCustomLinkEdit.SetLeftButton( Value: TsLeftEditButton);
begin
   FButton2.Assign(Value);
end;

procedure TsCustomLinkEdit.SetRightButton( Value: TsRightEditButton);
begin
   FButton1.Assign(Value)
end;

procedure TsCustomLinkEdit.RefreshChildGlyphs;
begin
   inherited;
   FButton1.RefreshGlyphId;
   FButton2.RefreshGlyphId;
end;

{******************** TsCustomPopupEdit ************************}
procedure TsCustomPopupEdit.WMMouseActivate(var Message: TMessage);
begin
   if msPopupOpen in FEditState then
      Include(FEditState, msCheckForBtn);
end;

procedure TsCustomPopupEdit.WMNCHitTest(var Message: TMessage);
var
   btn: TsEditButton;
begin
   if msCheckForBtn in FEditState then begin
      btn := nil;
      if FButton1.Visible and (FButton1.FButtonKind = bkNormal) and
         not Assigned(FOnButton1Click) then
         btn := Button1
      else if FButton2.Visible and (FButton2.FButtonKind = bkNormal) and
         not Assigned(FOnButton2Click) then
         btn := Button2;
      if (btn <> nil) and PtInRect( btn.GetDisplayRect,
         ScreenToClient(Point(loWord(Message.lParam), hiWord(Message.lParam)))) then
         Include(FEditState, msNoPopupCall);
      Exclude(FEditState, msCheckForBtn);
   end;
   inherited;
end;

function TsCustomPopupEdit.GetSolidBorder: Boolean;
begin
   Result := inherited GetSolidBorder or (msPopupOpen in FEditState);
end;

procedure TsCustomPopupEdit.OnAccept(Sender: TObject);
begin
   Exclude( FEditState, msPopupOpen);
end;

procedure TsCustomPopupEdit.OnCancel(Sender: TObject);
begin
   Exclude( FEditState, msPopupOpen);
end;
{******************** Bitmaps ************************}
const
   FileBitmap: TBitmap = nil;
   DateBitmap: TBitmap = nil;
   DirBitmap: TBitMap = nil;
   NumBitmap: TBitMap = nil;
   sFileBmp = 'FILEBMP'; { Filename and directory editor button glyph }
   sDirBmp = 'DIRBMP';
   sDateBmp = 'CALBMP'; { Date editor button glyph }
   sNumBmp = 'CALCBMP'; { calculator}

type
   TsPopupCalendarCrack = class(TsPopupCalendar)
   end;

{******************** TsCustomDateEdit ************************}
constructor TsCustomDateEdit.Create( AOwner: TComponent );
begin
   inherited Create( AOwner);
   FDate := TsDate.Create;
   FDate.OnDateChange := DateChanged;
   FDate.DateOrder := FDateOrder;
   FDate.SetNull;
   FCalPop := TsPopupCalendar.Create(self);
   FCalPop.OnAccept := OnAccept;
   FCalPop.OnCancel := OnCancel;
   FCalPop.Caller := self;
   FCalendarProps := TsPopupCalendarProps.Create(FCalPop);
   FDateFormat := [dfFullDay, dfFullMonth, dfFullYear];
   MaxLength := 10;
//TVS: Default is the current Dateseparator from Windows
//   FSeparator := '/';
//   Text := '  /  /    ';
   FSeparator := SysUtils.DateSeparator;
   Text := Format('  %s  %s    ', [FSeparator, FSeparator]);
   FEmptyString := GetEmptyString;
   FButton1.GlyphKind := gkDefault;
//   FButton2.ButtonKind := bkUpDown;
end;

destructor TsCustomDateEdit.Destroy;
begin
   FDate.Free;
   FCalendarProps.Free;
   FCalPop.Free;
	inherited;
end;

procedure TsCustomDateEdit.WMPaste(var Message: TMessage);
var
  	Value: string;
	LastDate: TDateTime;
  	SelStart, SelStop : Integer;
begin
  	if not EditCanModify then
    	inherited
  	else begin
    	Clipboard.Open;
    	Value := Clipboard.AsText;
    	Clipboard.Close;
    	GetSel(SelStart, SelStop);
		LastDate := FDate.AsDateTime;
      FDate.AsString := Value;
      if FDate.AsDateTime = LastDate then
      	MessageBeep( MB_ICONHAND);
      GetSel(SelStop, SelStop);
      Clipboard.Close;
  end;
end;

procedure TsCustomDateEdit.WMCut(var Message: TMessage);
var
 C: Word;
begin
   CopyToClipboard;
   C := VK_DELETE;
   DeleteKeys( C, [ssShift]);
end;

procedure TsCustomDateEdit.WMContextMenu(var Message: TMessage);
   function MouseToCell(var P: TPoint): TDateEditCells;
   var
      S, cellS: String;
      ii, w, sw: Integer;
   begin
      Result := -1;
      FCanvas.Font := Font;
      S := Text;
      sw := 0;
      w := 0;
      for ii := 0 to 3 do begin
         Inc(sw, w + FCanvas.TextWidth(' '));
         cellS := GetToken( S, ' ', FALSE);
         if cellS  = '' then
            Break;
         w := FCanvas.TextWidth(cellS);
         if P.X in [sw..sw + w] then begin
            P.X := sw;
            Result := ii;
            Break;
         end;
      end;
      if not (dfShowWeekDay in FDateFormat) then
         Inc(Result);
      Result := MapCell(Result);
  end;
var
   Msg: TMsg;
   cell: TDateEditCells;
   P: TPoint;
   SelStart, SelStop: Integer;
begin
   P := ScreenToClient(Point(LoWord(message.lParam), HiWord(message.lParam)));
   cell := MouseToCell(P);
   if (dfMonthSelect in FDateFormat) then begin
      if cell = 2 then begin
         SetFocus;
         GetCellBounds(cell, SelStart, SelStop);
         SetSel(SelStart, SelStop);
         MonthPopup.PopupOwner := self;
         MonthPopup.Date := FDate;
         P.Y := Height;
         MonthPopup.Alignment := paLeft;
         MonthPopup.Popup(P.X, P.Y);
         while PeekMessage(Msg, 0, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do;
      end;
   end else
      inherited;
end;

procedure TsCustomDateEdit.Click;
var
   SelStart, SelStop: Integer;
   hasFocus: Boolean;
   cell: TDateEditCells;
begin
   hasFocus := Focused;
   inherited;
   if hasFocus or not AutoSelect then begin
      GetSel(SelStart, SelStop);
      if (dfStringDateEditing in FDateFormat) or (SelStart = SelStop) then begin
         cell := GetCellFromPos(SelStart, SelStop);
         if (cell = 0) or (dfStringDateEditing in FDateFormat) then
            SetSel(SelStart, SelStop);
      end;
   end;
end;

procedure TsCustomDateEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
   SelStart, SelStop: Integer;
begin
   if not CheckShortCut(Key, Shift) and EditCanModify then begin
      case Key of
         VK_UP, VK_DOWN: if EditCanModify then begin
            IncCellValue(key = VK_UP);
            key := 0;
         end;
         VK_LEFT, VK_RIGHT: if not (ssShift in Shift) then
            ArrowKeys(Key, Shift);
         VK_DELETE, VK_BACK:
            DeleteKeys( Key, Shift);
         VK_HOME: begin
            if (dfStringDateEditing in FDateFormat) and (Text = EmptyText) or
               (ssShift in Shift) then
               Exit;
            SelStart := 0;
            SelStop := 0;
            if dfShowWeekday in FDateFormat then begin
               if Text = EmptyText then begin
                  SelStart := 4;
                  SelStop := 4;
               end else
                  Inc(SelStop, 3);
            end else if dfStringDateEditing in FDateFormat then
               GetCellBounds(1 + Ord(dfMonthSelect in FDateFormat), SelStart, SelStop);

            SetSel(SelStart, SelStop);
            key := 0;
         end;
         VK_END: if (dfStringDateEditing in FDateFormat) and not (ssShift in Shift) then begin
            GetCellBounds(3, SelStart, SelStop);
            SetSel(SelStart, SelStop);
            key := 0;
         end;
         9:
            Key := 0;
   	end;
	   inherited KeyDown(Key, Shift);
   end;
end;

procedure TsCustomDateEdit.KeyPress(var Key: Char);
   procedure InsertChar;
   begin
      try
   	   Include(FEditState, msChanging);
         SelText := String(Key);
         SetCursor( FCaretPos+1);
         Key := #0;
      finally
   	   Exclude(FEditState, msChanging);
      end;
   end;
var
   C: Word;
begin
   if Key = FSeparator then begin
      NextCell(TRUE);
      Key := #0;
   end else if Key = #8 then // remove backspace
      Key := #0
   else if (dfStringDateEditing in FDateFormat) then
      Key := #0
   else begin
      CheckCursor(TRUE);
      inherited KeyPress(Key);
      if (key <> #0) and EditCanModify then begin
         if SelLength > 1 then begin
            C := VK_DELETE;
            DeleteKeys(C, []);
         end;
         SelLength := 1;
         if FCaretPos in [GetSeparatorPosition(TRUE) - 2, GetSeparatorPosition(FALSE) - 2] then
            InsertChar;
      end;
   end;
end;

function TsCustomDateEdit.IsValidChar(const Key: Char): Boolean;
begin
   Result := inherited IsValidChar(key) or (Key in ['0'..'9', ' ', FSeparator]);
end;

procedure TsCustomDateEdit.AssignDateValue;
var
   s: String;
begin
   S := Text;
   if dfShowWeekDay in FDateFormat then
      S := Copy(S, 5, length(S) - 4);
   FDate.AsString := S;
end;

procedure TsCustomDateEdit.NextCell(next: Boolean);
var
  	cell, SelStart, SelStop : Integer;
begin
	GetSel(SelStart, SelStop);
   cell := GetCellFromPos(SelStart, SelStop);
   if cell = -1 then begin
      SelStop := SelStart;
      cell := GetCellFromPos(SelStart, SelStop);
   end else begin
      if next then begin
         Inc(cell);
         if cell > 3 then
            cell := 0;
      end else begin
         Dec(cell);
         if cell < Ord(not(dfShowWeekDay in FDateFormat)) then
            cell := 3;
      end;
      if not (dfShowWeekDay in FDateFormat) and (cell = 0) then
         Inc(cell);
   end;

   GetCellBounds(cell, SelStart, SelStop);
   if (cell > 0) and not (dfStringDateEditing in FDateFormat) then
      SelStop := SelStart;
   SetSel( SelStart, SelStop);
   FCaretPos := SelStart;
end;

procedure TsCustomDateEdit.ArrowKeys(var CharCode: Word; Shift: TShiftState);
var
  	SelStart, SelStop : Integer;
begin
  	if (ssCtrl in Shift) or (dfStringDateEditing in FDateFormat) then begin
      NextCell(CharCode = VK_RIGHT);
      CharCode := 0;
      Exit;
   end;
   if (ssShift in Shift) then
      Exit;
   GetSel(SelStart, SelStop);
   if ((SelStop - SelStart) > 1) and not (dfShowWeekDay in FDateFormat) then begin
      if SelStop = FCaretPos then
         Dec(FCaretPos);
      SetCursor(FCaretPos);
   end else if (CharCode = VK_LEFT) then begin
      if (dfShowWeekDay in FDateFormat) and not FDate.IsNull and (SelStart = 4) then
         SetCursor(0)
      else
         SetCursor(SelStart - 1)
   end else begin
      if (dfShowWeekDay in FDateFormat) and (SelStart < 4) then
         NextCell(TRUE)
      else
         SetCursor(SelStart+1);
   end;
   CharCode := 0;
end;

procedure TsCustomDateEdit.DeleteKeys(var CharCode: Word; Shift: TShiftState);
var
  	ii, SelStart, SelStop : Integer;
	cBuffer: array[0..10] of Char;
   Str: String;
begin
   GetSel(SelStart, SelStop);
   if ((SelStart > 0) or (SelStop < Length(Text))) and
      ((dfStringDateEditing in FDateFormat) or
      (GetCellFromPos(SelStart, SelStop) = 0)) then begin
      CharCode := 0;
      Exit;
   end;

	if (CharCode = VK_DELETE) and ( ssCtrl in Shift) then
       CopyToClipboard;

  	if not EditCanModify then
      Exit;

   ChangeExpected;
   if (dfStringDateEditing in FDateFormat) then try
      Include(FEditState, msChanging);
      Text := EmptyText;
      FDate.SetNull;
      CharCode := 0;
      Exit;
   finally
      Exclude(FEditState, msChanging);
   end;

   GetSel(SelStart, SelStop);
   if (SelStop = SelStart) and (CharCode = VK_BACK) then begin
   	Dec(SelStart);
      SetCursor(SelStart);
      GetSel( SelStop, SelStart);
  	end;

	if SelStart = SelStop then
      Inc(selStop);
	for ii := 0 to SelStop - SelStart do
      cBuffer[ii] := ' ';
   cBuffer[ SelStop - SelStart] := #0;
  	SetSel( SelStart, SelStop);
   Include(FEditState, msChanging);
   try
  		SendMessage(Handle, EM_REPLACESEL, 0, LongInt(PChar(@cBuffer)));
		Str := Text;
      Str[GetSeparatorPosition(TRUE)] := FSeparator;
      Str[GetSeparatorPosition(FALSE)] := FSeparator;
      Text := Str;
   finally
   	Exclude(FEditState, msChanging);
   end;
   SetCursor( SelStart);
   CharCode := 0;
   Modified := TRUE;
end;

procedure TsCustomDateEdit.CheckCursor(force: Boolean);
var
  	SelStart, SelStop: Integer;
begin
  	if HandleAllocated then  begin
      GetSel(SelStart, SelStop);
      if (SelStart = SelStop) or Force then begin
         if Force and (dfShowWeekday in FDateFormat) and (SelStart < 4) then
            SelStart := 4;
   	   SetCursor(SelStart);
      end;
   end;
end;

procedure TsCustomDateEdit.SetCursor(Pos: Integer);
begin
   if Pos < 0 then
      Pos := 0;
   if (MaxLength > 0) and (Pos >= MaxLength) then
      Pos := MaxLength;
   if (dfShowWeekDay in FDateFormat) and (Pos < 4) then begin
      if (Text <> EmptyText) then begin
         pos := 0;
         SetSel( pos, pos+3);
      end else begin
         pos := 4;
         SetSel( 4, 4);
      end;
   end else begin
      if Pos in [GetSeparatorPosition(TRUE)-1, GetSeparatorPosition(FALSE)-1] then begin
   	   if Pos > FCaretPos then
            Inc(Pos)
   	   else
            Dec(Pos);
	   end;
      SetSel( Pos, Pos);
   end;
   FCaretPos := Pos;
end;

function TsCustomDateEdit.GetSeparatorPosition(first: Boolean): ShortInt;
const
   sepOffset: array[Boolean] of ShortInt = (6, 3);
   SpaceChar = #32;
var
   S, cellS: String;
   p: Integer;
begin
   if dfStringDateEditing in FDateFormat then begin
      s := Text;
      if (dfMonthSelect in FDateFormat) then begin
         if First then
            Result := 0
         else
            Result := Pos(SpaceChar, S);
      end else begin
         p := 0;
         if (dfShowWeekDay in FDateFormat) then
            S := Copy(S, 5, length(S) - 4);
         if not first then begin
            cellS := GetToken(S, SpaceChar, TRUE);
            Inc(p, Length(cellS));
         end;
         Result := p + Pos(SpaceChar, S);
      end;
   end else begin
      Result := sepOffset[first];
      if (FDateOrder = doYMD) and (dfFullYear in  FDateFormat) then
         Inc(Result, 2);
   end;
   if dfShowWeekDay in FDateFormat then
      Inc(Result, 4);
end;

function TsCustomDateEdit.Validate(var Pos: Integer): Boolean;
var
   dt: TDateTime;
begin
   dt := FDate.AsDateTime;
   Include(FEditState, msChanging);
   AssignDateValue;
   Exclude(FEditState, msChanging);
   Include(FEditState, msValidating);
   with FDate do
      Result := not FDate.IsNull or (Text = EmptyText);
   Exclude(FEditState, msValidating);
   Pos := -1;
   if Result then
      Result := inherited Validate(Pos);
   if Result and (dt <> FDate.AsDateTime) then
      DateChanged(nil);
end;

function TsCustomDateEdit.GetEditLength: Integer;
begin
   Result := 8;
   if dfFullYear in FDateFormat then
      Inc(Result, 2);
   if dfShowWeekDay in FDateFormat then
      Inc(Result, 4);
end;

function TsCustomDateEdit.EmptyText: String;
begin
   if (dfStringDateEditing in FDateFormat) or
      ((dfStringDateView in FDateFormat) and not Focused and not (msValidating in FEditState)) then
      Result := FEmptyString
   else begin
      Result := '              ';
      Result[ GetSeparatorPosition(TRUE)] := FSeparator;
      Result[ GetSeparatorPosition(FALSE)] := FSeparator;
      SetLength(Result, GetEditLength);
   end;
end;

procedure TsCustomDateEdit.GetCellBounds(cell: TDateEditCells; var SelStart, SelStop: Integer);
begin
   SelStart := -1;
   SelStop := -1;
   if (dfStringDateEditing in FDateFormat) and (Text = EmptyText) then
      Exit;
   case cell of
      0: begin
         SelStart := 0;
         SelStop := 3;
      end;
      1: begin
         SelStart := 0;
         SelStop := GetSeparatorPosition(TRUE) - 1;
         if (dfShowWeekDay in FDateFormat) then
            Inc(SelStart, 4);
      end;
      2: begin
         SelStart := GetSeparatorPosition(TRUE);
         SelStop := GetSeparatorPosition(FALSE) - 1;
      end;
      3: begin
         SelStart := GetSeparatorPosition(FALSE);
         if dfStringDateEditing in FDateFormat then
            SelStop := Length(text)
         else
            SelStop := GetEditLength;
      end;
   end;
end;

function TsCustomDateEdit.GetCellFromPos(var SelStart, SelStop: Integer): TDateEditCells;
begin
   if (dfShowWeekDay in FDateFormat) and (SelStart in [0..3]) then
      Result := 0
   else if SelStart < GetSeparatorPosition(TRUE) then
      Result := 1
   else if SelStart in [GetSeparatorPosition(TRUE)..GetSeparatorPosition(FALSE)-2] then
      Result := 2
   else
      Result := 3;
   if Result > -1 then
      GetCellBounds(Result, SelStart, SelStop);
end;

function TsCustomDateEdit.MapCell(cell: TDateEditCells): TDateEditCells;
begin
   Result := Cell;
   if dfMonthSelect in FDateFormat then begin
      if ((Cell = 1) and (FDateOrder in [doDMY, doMDY])) or (cell = 2) then
         Result := 2
      else
         Result := 3;
   end else begin
      if cell = Ord(FDateOrder)+1 then
         Result := 1
      else if ((cell = 2) and (FDateOrder in [doDMY, doYMD])) or
         ((cell = 1) and (FDateOrder = doMDY)) then
         Result := 2
      else if ((cell = 3) and (FDateOrder in [doDMY, doMDY])) or
         ((cell = 1) and (FDateOrder = doYMD)) then
         Result := 3;
   end;
end;

function TsCustomDateEdit.GetCellValue(cell: TDateEditCells): Integer;
var
   pCell: TDateEditCells;
   SelStart, SelStop: Integer;
   S: String;
begin
   Result := 0;
   case cell of
      1: // date
         pCell := Ord(FDateOrder)+1;
      2: begin
         pCell := 2;
         if FDateOrder = doMDY then
            pCell := 1;
      end;
      3: begin
         pCell := 3;
         if FDateOrder = doYMD then
            pCell := 1;
      end;
      else
         Exit;
   end;
   GetCellBounds(pCell, SelStart, SelStop);
   S := Text;
   S := Copy(S, SelStart + 1, SelStop - SelStart);
   Result := S2I(S);
end;

procedure TsCustomDateEdit.IncCellValue(doInc: Boolean);
var
   SelStart, SelStop: Integer;
   cell: TDateEditCells;

   val, m: Integer;
   S: String;
begin
   if not (dfStringDateEditing in FDateFormat) then begin
      Include( FEditState, msChanging);
      AssignDateValue;
      Exclude( FEditState, msChanging);
   end else if FDate.IsNull then
      Exit;

   ChangeExpected;
  	GetSel(SelStart, SelStop);
   cell := GetCellFromPos(SelStart, SelStop);
   SetSel(SelStart, SelStop);

   if FDate.IsNull then begin
      if Cell < 1 then
         Exit;
      S := Trim(SelText);
      if S > '' then begin
         Val := StrToInt(S);
         if doInc then
            Inc(Val)
         else
            dec(Val);
         if (Val <= 0) then
            Exit;
         case MapCell(Cell) of
            1: begin
                  if Val > 31 then
                     Exit;
                  m := GetCellValue(2);
                  if ((m in [4,6, 9,11]) and (Val > 30)) or
                     ((m = 2) and (Val > 29)) then
                     Exit;
                  if m = 2 then begin
                     m := GetCellValue(3);
                     if not IsLeapYear(m) and (Val > 28) then
                        Exit;
                  end;
            end;
            2: if Val > 12 then
               Exit;
         end;
         S := IntToStr(val);
         if length(S) < 2 then
            S := ' ' + S;
         try
   	      Include(FEditState, msChanging);
            SelText := S;
            SetSel(SelStart, SelStop);
         finally
   	      Exclude(FEditState, msChanging);
         end;

      end;
   end else case MapCell(Cell) of
      0, 1:
         if DoInc then
            FDate.NextDay
         else
            FDate.PrevDay;
      2:
         if DoInc then
            FDate.NextMonth
         else
            FDate.PrevMonth;
      3:
         if DoInc then
            FDate.NextYear
         else
            FDate.PrevYear;
   end;
   if dfStringDateEditing in FDateFormat then
      GetCellBounds(Cell, SelStart, SelStop);
   SetSel(SelStart, SelStop);
end;

function TsCustomDateEdit.GetFormatString(longFormat: Boolean): String;
var
   d, m, y: String[4];
   sep: Char;
begin
   d := 'd';
   if (dfFullDay in FDateFormat) or (not LongFormat and Focused) then
      d := 'dd';
   m := 'm';
   if (dfFullMonth in FDateFormat) or (not LongFormat and Focused) then
      m := m + 'm';
   if longFormat then
      m := m + 'mm';
   y := 'yy';
   if (dfFullYear in FDateFormat) or (not LongFormat and Focused) then
      y := y + 'yy';
   if longFormat then
      sep := #32
   else
      sep := FSeparator;
   if longFormat and (dfMonthSelect in FDateFormat) then case FDateOrder of
      doDMY, doMDY:
         Result := Format('%s %s', [m, y]);
      doYMD:
         Result := Format('%s %s', [y, m]);
   end else case FDateOrder of
      doDMY:
         Result := Format('%s%s%s%s%s', [d, sep, m, sep, y]);
      doMDY:
         Result := Format('%s%s%s%s%s', [m, sep, d, sep, y]);
      doYMD:
         Result := Format('%s%s%s%s%s', [y, sep, m, sep, d]);
   end;
end;

procedure TsCustomDateEdit.FormatDateText;
var
	Str: String;
begin
	with FDate do begin
      if IsNull then
         Str := EmptyText
      else begin
         Str := FormatDateTime( GetFormatString(
            ((dfStringDateView in FDateFormat) and not Focused and not (msPopupOpen in FEditState)) or
            (dfStringDateEditing in FDateFormat)), FDate.AsDateTime);
         if dfShowWeekDay in FDateFormat then
               Str := Format('%3s ', [WeekDays[SysUtils.DayOfWeek(AsDateTime)]]) + Str;
      end;
   end;
   try
   	Include(FEditState, msChanging);
		Text := Str;
   finally
   	Exclude(FEditState, msChanging);
   end;
end;

procedure TsCustomDateEdit.DateChanged( Sender: TObject);
begin
   if not (msChanging in FEditState) then begin
   	if Assigned( FOnDateChange) then
         FOnDateChange( self);
      if (not FDate.IsNull or (Text = EmptyText)) then begin
		   Modified := TRUE;
         ChangeExpected;
         FormatDateText;
   	   if Assigned( FOnDateChanged) then
            FOnDateChanged( self);
      end;
   end;
end;

procedure TsCustomDateEdit.SetSeparator(Value: Char);
var
	Str: String;
begin
	if FSeparator <> Value then begin
   	FSeparator := Value;
		Include(FEditState, msChanging);
      try
			Str := Text;
         Str[GetSeparatorPosition(TRUE)] := FSeparator;
         Str[GetSeparatorPosition(FALSE)] := FSeparator;
         Text := Str;
      finally
      	Exclude(FEditState, msChanging);
      end;
   end;
end;

procedure TsCustomDateEdit.SetDateFormat(Value: TDateFormat);
begin
	if Value <> FDateFormat then begin
   	FDateFormat := Value;
      if dfMonthSelect in FDateFormat then begin
         Include(FDateFormat, dfStringDateEditing);
         Exclude(FDateFormat, dfShowWeekDay);
      end;
		FormatDateText;
      if dfStringDateEditing in FDateFormat then
         maxLength := 0
      else
         maxLength := GetEditLength;
   end;
end;

procedure TsCustomDateEdit.SetDateOrder(Value: TDateOrder);
begin
	if Value <> FDateOrder then begin
   	FDateOrder := Value;
      FDate.DateOrder := Value;
		FormatDateText;
   end;
end;

procedure TsCustomDateEdit.SetEmptyString(Value: String);
begin
   if FEmptyString <> Value then begin
      FEmptyString := Value;
      if (dfStringDateEditing in FDateFormat) or ((dfStringDateView in FDateFormat) and not Focused) then
         Text := EmptyString;
   end;
end;

procedure TsCustomDateEdit.SetPopupCalendar(Value: TsPopupCalendarProps);
begin
   FCalendarProps.Assign(Value);
end;

function TsCustomDateEdit.EmptyStringStored: Boolean;
begin
   Result := EmptyString <> GetEmptyString;
end;

function TsCustomDateEdit.GetEmptyString: String;
begin
   Result := SNoDateCaption;
end;

procedure TsCustomDateEdit.SetShowToday(value: Boolean);
begin
   FShowToday := Value;
   if Value then
      FDate.AsDateTime := Date;
end;

function TsCustomDateEdit.GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap;
begin
   DestroyNeeded := FALSE;
   if DateBitmap = nil then begin
      DateBitmap := TBitmap.Create;
      DateBitmap.Handle := LoadBitmap(hInstance, sDateBmp);
   end;
   Result := DateBitmap;
end;

procedure TsCustomDateEdit.ButtonClick(Button: TsEditButtons);
var
   btn: TsEditButton;
   aButtonClick: TNotifyEvent;
begin
   if button = ebButton1 then begin
      btn := FButton1;
      aButtonClick := FOnButton1Click;
   end else begin
      btn := FButton2;
      aButtonClick := FOnButton2Click;
   end;
   if Assigned(aButtonClick) then
      inherited
   else if btn.ButtonKind = bkUpDown then
      IncCellValue(btn.FButtonState = bsUpButton)
   else if EditCanModify then
      ShowPopup( not FCalPop.Visible)
end;

procedure TsCustomDateEdit.DoEnter;
var
   pos, selStop: Integer;
begin
   if not (msReEnter in FEditState) then begin
      FormatDateText;
      Exclude( FEditState, msNoRedraw);
   end;
   if Text <> EmptyText then begin
      if not (dfStringDateEditing in FDateFormat) then begin
         GetSel(pos, SelStop);
         if (pos > MaxLength)then
            pos := MaxLength;
         while (pos > 0) and (Text[pos] = ' ') or (Text[pos] = FSeparator) do
   	      Dec(pos);
         SetCursor(pos);
      end;
   end;
   inherited;
end;

procedure TsCustomDateEdit.DoExit;
begin
   inherited;
   if not Focused then begin
      FormatDateText;
      Exclude( FEditState, msNoRedraw);
   end;
end;

procedure TsCustomDateEdit.ShowPopup(doShow: Boolean);
begin
   if doShow then begin
      if msNoPopupCall in FEditState then
         OnAccept(self)
      else begin
         if not FDate.IsNull then
            FCalPop.Date.AsDateTime := FDate.AsDateTime;
         FCalPop.Popup(-3, ClientHeight);
         Include( FEditState, msPopupOpen);
      end;
   end else begin
      OnCancel(self);
      FCalPop.Hide;
      SetFocus;
   end;
end;

procedure TsCustomDateEdit.OnAccept(Sender: TObject);
begin
	FDate.AsDateTime := FCalPop.Date.AsDateTime;
   inherited;
end;

{******************** TsCustomNumberEdit ************************}
constructor TsCustomNumberEdit.Create( AOwner: TComponent );
begin
   inherited;
   FAlignment := taRightJustify;
   FDecimals := 2;
   FIncrement := 1;
  	FMaxValue := 0;
  	FMinValue := 0;
   FNumPad := TsNumPad.Create(self);
   FNumPad.OnAccept := OnAccept;
   FNumPad.OnCancel := OnCancel;
   FNumPad.OnValueChange := OnValChange;
   FNumPadProps := TsNumPadProps.Create(FNumPad);
   FButton1.GlyphKind := gkDefault;
   //FButton2.ButtonKind := bkUpDown;
   Value := 0;
  	SetFormatMask;
end;

destructor TsCustomNumberEdit.Destroy;
begin
   FNumPadProps.Free;
   FNumPad.Free;
   inherited;
end;

procedure TsCustomNumberEdit.CreateParams(var Params: TCreateParams);
begin
  	inherited CreateParams(Params);
  	Params.Style:= Params.Style or ES_NUMBER;
end;

function TsCustomNumberEdit.Validate(var Pos: Integer): Boolean;
var
	val: Extended;
begin
   val := AsFloat;
   {Suggested by TVS}
   if Val = 0 then
      Value := 0;

   Result := ((FMaxValue = FMinValue) and (FMaxValue =0)) or
               ((val >= FMinValue) and (val <= FMaxValue));
   Pos := -1;
   if Result then
      Result := inherited Validate(Pos);
end;

procedure TsCustomNumberEdit.IncValue(doInc: Boolean);
begin
   if doInc then
      Value := Value + FIncrement
   else
      Value := Value - FIncrement;
end;

procedure TsCustomNumberEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
   if EditCanModify then
    	if Key in [VK_DELETE, VK_BACK] then begin
         ChangeExpected;
      	if SelLength > 0 then begin
        		DeleteSelection;
        		Change;
      	end else
        		DeleteKey(Key);
      	Key := 0;
      end else if key in [VK_UP, VK_DOWN] then begin
         ChangeExpected;
         IncValue(key = VK_UP);
    	end else if Key in [9] then
         Key := 0;
  	inherited;
end;

procedure TsCustomNumberEdit.KeyPress(var Key: Char);
var
  	X: Integer;
  	N: Boolean;

   S: String;
   p: Integer;
begin
   if EditCanModify then begin
      inherited KeyPress(Key);
      if Key = #8 then
         Key := #0;

      if Key = #0 then
         Exit
      else if Key = DecimalSeparator then // Decimal separator is pressed
        SelStart := Length(Text) - FDecimals
      else begin
         if (SelLength > 0) then
            DeleteSelection;

         S := Text;
         p := SelStart;

         N := (Pos('-', Text) > 0);
         X := SelStart;
         if (Key = '-') then begin
            if Pos('-', S) = 0 then begin
               S := '-' + S;
               p := X + 1;
            end else begin
               S := Copy(S, 2, Length(S) - 1);
               p := X - 1;
            end;
         end else if not ((X = 0) and (Key = '0')) then begin
            if N and (X = 0) then // Don't overwrite '-'
               X := 1;
            if FDecimals = 0 then begin // No decimals
               if (X <= 2) and (Copy(S, 2, 1) = '0') and N then begin// Overwrite zero when negative
                  S := '-' + Key;
                  p:= 1;
               end else if (X <= 1) and (Copy(S, 1, 1) = '0') then begin// Overwrite zero when not negative
                  { to Thomas: change here..}
                  if S = '0' then // Override zero when '0'
                     S := Key
                  else
                     S := Key + S;
                  p := 1;
               end else begin
                  S := Copy(S, 1, X) + Key + Copy(S, X + 1, Length(S));
                  p := X + 1;
               end;
            end else begin // with Decimals
               if (x = Length(S)) then     // If behind the last decimal override this
                  Dec(x);
               if (x <= 2) and (Copy(S, 2, 1) = '0') and N then begin // Overwrite zero when negative
                  S := '-' + Key + Copy(S, 3, Length(S) - 2);
                  p := 2;
               end else if (x <= 1) and (Copy(S, 1, 1) = '0') then begin // Overwrite zero when not negative
                  { to Thomas: .. and here}
                  if Copy(S, 2, 1) = DecimalSeparator then // Override zero when '0.XX'
                     S := Key + Copy(S, 2, Length(S) - 1)
                  else
                     S := Key + S;
                  p := 1;
               end else if x >= (Length(S) - FDecimals) then begin // Overwrite decimals
                  S := Copy(S, 1, x) + Key + Copy(S, x + 2, Length(S));
                  p := X + 1;
               end else begin    // Overwrite digit if left of decimal point
                  S := Copy(S, 1, x) + Key + Copy(S, x + 1, Length(S));
                  P := X + 1;
               end;
            end;
         end;
         if (Alignment = taLeftJustify) or
            (((maxLength = 0) or (Length(S) <= maxLength)) and
            (FCanvas.TextWidth(S) <= WidthOf(GetEditRect))) then begin
            Text := S;
            SelStart := P;
         end;
      end;

      SelLength:= 0;
      Key:= #0;
   end;
end;

procedure TsCustomNumberEdit.DeleteKey(Key: Word);
var
  	P: Integer;
  	N: Boolean;
begin
   P := SelStart;
  	N := (Pos('-', Text) > 0);

   if (Key = VK_DELETE) and (P <> Length(Text)-FDecimals-Ord(FDecimals<>0)) then //delete character to the left of P
      Inc(P);

 	if (P = 0) or (P = Length(Text) + 1) then
    	Exit; // Can't delete non-existent character

  	if FDecimals = 0 then begin // No decimals
    	if (P = 2) and (Length(Text) = 2) and N then begin // Reset only digit to 0 when negative - no decimals
      	Text := '-0' + Copy(Text, 3, Length(Text) - 2);
      	SelStart := 1;
    	end else if (P = 1) and (Length(Text) = 1) then begin // Reset only digit to 0 - no decimals
        	Text := '0' + Copy(Text, 2, Length(Text) - 1);
        	SelStart := 1;
      end else begin
      	Text := Copy(Text, 1, P - 1) + Copy(Text, P + 1, Length(Text) - P);
        	SelStart := P - 1;
      end;
   end else begin // Decimals
    	if P > (Length(Text) - FDecimals) then begin // Delete decimal - reset to 0
      	Text := Copy(Text, 1, P - 1) + Copy(Text, P + 1, Length(Text) - P) + '0';
         SelStart := P - 1;
    	end else if (P = (Length(Text) - FDecimals)) then begin// Not possible to delete decimal point
      	if Key = VK_DELETE then
            SelStart := (Length(Text) - FDecimals)
        	else
          	SelStart := (Length(Text) - FDecimals) - 1
      end else if (P = 2) and (P = (Length(Text) - FDecimals) - 1) and N then begin // Reset only digit to 0 when negative
      	Text := '-0' + Copy(Text, 3, Length(Text) - 2);
         SelStart := 1;
      end else if (P = 1) and (P = (Length(Text) - FDecimals) - 1) then begin // Reset only digit to 0
         Text := '0' + Copy(Text, 2, Length(Text) - 1);
         SelStart := 1;
      end else if P > 0 then begin // Delete digit left of decimal point
         Text := Copy(Text, 1, P - 1) + Copy(Text, P + 1, Length(Text) - P);
         SelStart := P - 1;
      end;
   end;
end;


procedure TsCustomNumberEdit.DeleteSelection;
var
  	X: Integer;
  	Y: Integer;
begin
	if SelLength = 0 then
   	Exit;
   Include(FEditState, msChanging);
   try
  		Y := Length(SelText);
  		SelStart := SelStart + Y;
  		for X := 1 to Y do
    		DeleteKey(VK_BACK);
   finally
		Exclude(FEditState, msChanging);
	end;
end;

function TsCustomNumberEdit.IsValidChar(const Key: Char): Boolean;
var
   ValidChars: set of Char;
begin
   ValidChars := [ '0'..'9', '-', Char(VK_Up), Char(VK_Down)];
   if ((FMaxValue = 0) and (FMaxValue = FMinValue)) or (FMinValue < 0) then
      validChars := ValidChars + ['-'];
   if Fdecimals > 0 then
      ValidChars := ValidChars + [DecimalSeparator];
   Result := inherited IsValidChar(key) or (Key in ValidChars);
end;

function TsCustomNumberEdit.GetAsCurrency: Currency;
begin
  	Result := StrToCurr(Text);
end;

procedure TsCustomNumberEdit.SetAsCurrency(Value: Currency);
begin
   SetValue( Value);
end;

function TsCustomNumberEdit.GetAsInteger: Integer;
begin
  	Result := trunc(StrToFloat(Text));
end;

function TsCustomNumberEdit.GetValue: Double;
begin
  	Result := StrToFloat(Text);
end;

procedure TsCustomNumberEdit.SetAsInteger(Value: Integer);
begin
   SetValue(Value);
end;

procedure TsCustomNumberEdit.SetDecimals(Value: ShortInt);
begin
	if FDecimals <> Value then begin
    	if Value < 0 then
      	Value := 0;
    	if Value > 9 then
      	Value := 9;
   	FDecimals := Value;
      SetFormatMask;
   end;
end;

// TVS: Parameter-Name changed for a better understanding of the implementation
procedure TsCustomNumberEdit.SetValue(dblValue: Double);
var
  	NewText: string;
begin
//TVS: Added Range-Check, if one of them <> 0
    IF ((MinValue <> 0) OR (MaxValue <> 0)) then begin
       IF (dblValue > MaxValue) then
          dblValue := MaxValue
       ELSE IF (dblValue < MinValue) then
          dblValue := MinValue;
    end;
//TVS: End of changes

  	NewText:= FormatFloat(FormatMask, dblValue);
  	if Text <> NewText then
    	Inherited Text:= NewText;
end;

function TsCustomNumberEdit.DblValueStored(index: Integer): Boolean;
begin
   Result := FALSE;
//TVS: All compares changed from '=' to '<>'
   case index of
      0: Result := FIncrement <> 0.0;
      1: Result := FMaxValue <> 0.0;
      2: Result := FMinValue <> 0.0;
   end;
end;

procedure TsCustomNumberEdit.SetNumPad(Value: TsNumPadProps);
begin
   FNumPadProps.Assign(Value);
end;

procedure TsCustomNumberEdit.SetFormatMask;
begin
	if FDecimals = 0 then
   	FormatMask := '0'
  	else
    	FormatMask := '0.' + StringOfChar('0', FDecimals);
   Include(FEditState, msChanging);
  	try
   	Text := FormatFloat(FormatMask, AsFloat);
  	finally
   	Exclude(FEditState, msChanging);
   end;
end;

procedure TsCustomNumberEdit.WMCut(var Message: TMessage);
begin
  	CopyToClipboard;
   ChangeExpected;
  	DeleteSelection;
end;

procedure TsCustomNumberEdit.WMPaste(var Message: TMessage);
var
  	ii: integer;
  	S: String;
  	W: Word;
begin
  	DeleteSelection;
   Clipboard.Open;
  	S := Trim(Clipboard.AsText);
   Clipboard.Close;
   if Length(S) > 0 then
      ChangeExpected;
  	for ii := 1 to Length(S) do begin
    	W := Ord(S[ii]);
    	Perform(WM_CHAR, W, 0);
  	end;
end;

function TsCustomNumberEdit.GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap;
begin
   DestroyNeeded := FALSE;
   if NumBitmap = nil then begin
      NumBitmap := TBitmap.Create;
      NumBitmap.Handle := LoadBitmap(hInstance, sNumBmp);
   end;
   Result := NumBitmap;
end;

procedure TsCustomNumberEdit.ButtonClick(Button: TsEditButtons);
var
   btn: TsEditButton;
   aButtonClick: TNotifyEvent;
begin
   if button = ebButton1 then begin
      btn := FButton1;
      aButtonClick := FOnButton1Click;
   end else begin
      btn := FButton2;
      aButtonClick := FOnButton2Click;
   end;
   if Assigned(aButtonClick) then
      inherited
   else if btn.ButtonKind = bkUpDown then
      IncValue(btn.FButtonState = bsUpButton)
   else if EditCanModify then
      ShowPopup( not FNumPad.Visible)
end;

procedure TsCustomNumberEdit.ShowPopup(doShow: Boolean);
begin
   if doShow then begin
      if msNoPopupCall in FEditState then begin
         OnAccept(self);
      end else begin
         FKeptValue := FNumPad.Value;
         FNumPad.StrValue := Text;
         FNumPad.Popup( Width - 3, ClientHeight);
         FEditState := FEditState + [msPopupOpen, msChanging];
      end;
   end else begin
      OnCancel(self);
      FNumPad.Hide;
      SetFocus;
   end;
end;

procedure TsCustomNumberEdit.OnAccept(Sender: TObject);
begin
   ChangeExpected;
   Value := FNumPad.Value;
   Exclude( FEditState, msChanging);
   Change;
   RedrawBorders;
   inherited;
end;

procedure TsCustomNumberEdit.OnCancel(Sender: TObject);
begin
   Value := FKeptValue;
   Exclude( FEditState, msChanging);
   RedrawBorders;
   inherited;
end;

procedure TsCustomNumberEdit.OnValChange(Sender: TObject);
begin
   Text := FNumPad.StrValue;
end;

{******************** TsFileDirEdit *******************************}
function ClipFilename(const AFileName: string): string;
var
   Params: string;
begin
   SplitCommandLine(AFileName, Result, Params);
end;

function ExtFilename(const AFileName: string): string;
begin
   if (Pos(' ', AFileName) > 0) and (AFileName[1] <> '"') then
   	Result := Format('"%s"', [AFileName])
   else
		Result := AFileName;
end;

constructor TsFileDirEdit.Create(AOwner: TComponent);
begin
   inherited;
   OEMConvert := TRUE;
   LoadSystemStrings;
   FButton1.GlyphKind := gkDefault;
end;

procedure TsFileDirEdit.CreateHandle;
begin
   inherited CreateHandle;
   if FAcceptFiles then
   	SetDragAccept(True);
   FDisplayedChars := Round(WidthOf(GetEditRect) div GetFontMetrics(Font).tmAveCharWidth);
   try
      Include( FEditState, msChanging);
      DoSetText(GetDisplayText);
   finally
      Exclude(FEditState, msChanging);
   end;
end;

procedure TsFileDirEdit.DestroyWindowHandle;
begin
   SetDragAccept(False);
   inherited DestroyWindowHandle;
end;

procedure TsFileDirEdit.Change;
var
   S: String;
begin
   if not (msChanging in FEditState) and (Trim(Text) <> '') then begin
      S := ClipFileName(ExtFilename(Text));
      if S[Length(S)] = '\' then
         SetLength(S, Length(S) - 1);
      if DirExists(S) and (FPath <> ExtFilename(S)) then begin
         FPath := ExtFilename(S);
         PathChanged;
      end;
   end;
   inherited;
end;

procedure TsFileDirEdit.DoEnter;
begin
   if not (msReEnter in FEditState) then try
      Include( FEditState, msChanging);
      DoSetText(GetDisplayText);
   finally
		Exclude( FEditState, msChanging);
   end;
   inherited
end;

procedure TsFileDirEdit.DoExit;
begin
   inherited;
   if not Focused then try
   	Include( FEditState, msChanging);
   	DoSetText(GetDisplayText);
   finally
      Exclude(FEditState, msChanging);
   end;
end;

procedure TsFileDirEdit.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
   if [csLoading,csReading] * ComponentState = [] then begin
      if HandleAllocated then
         FDisplayedChars := Round(WidthOf(GetEditRect) div GetFontMetrics(Font).tmAveCharWidth);
      try
         Include( FEditState, msChanging);
         DoSetText(GetDisplayText);
      finally
         Exclude( FEditState, msChanging);
      end;
   end;
   inherited;
end;

function TsFileDirEdit.GetPath: string;
begin
   Result := ClipFilename(FPath);
end;

procedure TsFileDirEdit.SetPath(const Value: string);
var
   S: String;
begin
   if FNoPathValidate or ValidFileName(Value) then begin
      S := ClipFileName(ExtFilename(Value));
      if not FDirectoryEdit then
         S := ExtFilename(UnTerminateDir(S))
       else
         S := ExtFilename(TerminateDir(S));
      if FPath <> S then begin
         FPath := S;
         try
            Include( FEditState, msChanging);
      	   inherited Text := GetDisplayText;
         finally
      	   Exclude( FEditState, msChanging);
         end;
         if not (csLoading in ComponentState) then
      	   PathChanged;
      end;
   end else if (Value <> '') and not FNoPathValidate then
      raise ELinkEditError.CreateFmt(MsgFilenameInvalid, [Value]);
end;

procedure TsFileDirEdit.SetDragAccept(Value: Boolean);
begin
   if not (csDesigning in ComponentState) and (Handle <> 0) then
      DragAcceptFiles(Handle, Value);
end;

procedure TsFileDirEdit.SetDisplayFullPath(Value: Boolean);
begin
   if Value <> FDisplayFullPath then begin
      FDisplayFullPath := Value;
      if [csLoading, csreading] * ComponentState = [] then try
         Include( FEditState, msChanging);
      	inherited Text := GetDisplayText;
      finally
      	Exclude( FEditState, msChanging);
      end;
   end;
end;

procedure TsFileDirEdit.SetDisplayRelativePath(Value: Boolean);
begin
   if Value <> FDisplayRelativePath then begin
      FDisplayRelativePath := Value;
      if [csLoading, csreading] * ComponentState = [] then try
         Include( FEditState, msChanging);
      	inherited Text := GetDisplayText;
      finally
      	Exclude( FEditState, msChanging);
      end;
   end;
end;

function TsFileDirEdit.GetDisplayText: String;
begin
   if DisplayFullPath or Focused then
      Result := FPath
   else begin
      Result := ClipFilename(FPath);
      if not FDisplayFullPath then
         Result := CompressPath( Result, FDisplayedChars);
      if (Result = ClipFilename(FPath)) and FDisplayRelativePath then
         Result := MakeRelativePath( InitialDir, Result);
   end;
end;

procedure TsFileDirEdit.SetAcceptFiles(Value: Boolean);
begin
   if FAcceptFiles <> Value then begin
      SetDragAccept(Value);
      FAcceptFiles := Value;
   end;
end;

procedure TsFileDirEdit.WMDropFiles(var Msg: TWMDropFiles);
var
   AFileName: array[0..255] of Char;
begin
   Msg.Result := 0;
   try
      if DragQueryFile(Msg.Drop, 0, @AFileName, Pred(SizeOf(AFileName))) > 0 then begin
         ReceptFileDir(StrPas(AFileName));
         if Assigned(FOnDropFiles) then FOnDropFiles(Self);
      end;
   finally
      DragFinish(Msg.Drop);
   end;
end;

procedure TsFileDirEdit.DoBeforeDialog(var AFileName: string; var Action: Boolean);
begin
   if Assigned(FOnBeforeDialog) then
      FOnBeforeDialog(Self, AFileName, Action);
end;

procedure TsFileDirEdit.DoAfterDialog(var AFileName: string; var Action: Boolean);
begin
   if Assigned(FOnAfterDialog) then
      FOnAfterDialog(Self, AFileName, Action);
end;

procedure TsFileDirEdit.PathChanged;
begin
	if Assigned(FOnPathChange) then
      FOnPathChange(self);
end;

{******************** TsFilenameEdit *******************************}
constructor TsFilenameEdit.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FDialog := TOpenDialog.Create(Self);
   FDialog.Title := SBrowseCaption;
   FDialog.Filter := SDefaultFilesFilter;
end;

procedure TsFilenameEdit.ButtonClick;
var
   aButtonClick: TNotifyEvent;
   TempPath: string;
   Action: Boolean;
begin
   if button = ebButton1 then begin
      aButtonClick := FOnButton1Click;
   end else begin
      aButtonClick := FOnButton2Click;
   end;
   if Assigned(aButtonClick) then
      inherited
   else if EditCanModify then begin
      TempPath := Path;
      Action := TRUE;
      DoBeforeDialog(TempPath, Action);
      if not Action then
         Exit;
      if (ofNoValidate in DialogOptions) or ValidFileName(TempPath) then try
         if DirExists(TempPath) then
            SetInitialDir(TempPath)
         else begin
            FDialog.FileName := TempPath;
            TempPath := ExtractFilePath(TempPath);
            if DirExists(TempPath) then
               SetInitialDir(TempPath);
         end;
      except
      end;
      FDialog.HelpContext := Self.HelpContext;
      Action := FDialog.Execute;
      if TempPath = FDialog.FileName then
         Action := FALSE
      else
         TempPath := FDialog.FileName;
      SetFocus;
      DoAfterDialog(TempPath, Action);
      if Action then begin
         ChangeExpected;
         if not FDisplayRelativePath then
            SetInitialDir(ExtractFilePath(Path));
         Path := ClipFilename(ExtFilename(TempPath));
      end;
   end;
end;

function TsFilenameEdit.Validate(var Pos: Integer): Boolean;
var
   options: TOpenOptions;
begin
   Result := TRUE;
   if Trim(Text) <> '' then begin
      options := FDialog.Options;
      Result := ValidateFile(ClipFilename(Text), DefaultExt, Options, TRUE);
      FDialog.Options := options;
   end;
   Pos := -1;
   if Result then
      Result := inherited Validate(Pos);
end;

procedure TsFilenameEdit.ReceptFileDir(const AFileName: string);
begin
   SetPath(AFileName);
end;

function TsFilenameEdit.IsCustomTitle: Boolean;
begin
   Result := CompareStr(SBrowseCaption, FDialog.Title) <> 0;
end;

function TsFilenameEdit.IsCustomFilter: Boolean;
begin
   Result := CompareStr(SDefaultFilesFilter, FDialog.Filter) <> 0;
end;

function TsFilenameEdit.GetDialogFiles: TStrings;
begin
   Result := FDialog.Files;
end;

function TsFilenameEdit.GetDefaultExt: TFileExt;
begin
   Result := FDialog.DefaultExt;
end;

function TsFilenameEdit.GetFileEditStyle: TFileEditStyle;
begin
   Result := FDialog.FileEditStyle;
end;

function TsFilenameEdit.GetFilter: string;
begin
   Result := FDialog.Filter;
end;

function TsFilenameEdit.GetFilterIndex: Integer;
begin
   Result := FDialog.FilterIndex;
end;

function TsFilenameEdit.GetInitialDir: string;
begin
   Result := FDialog.InitialDir;
end;

function TsFilenameEdit.GetHistoryList: TStrings;
begin
   Result := FDialog.HistoryList;
end;

function TsFilenameEdit.GetOptions: TOpenOptions;
begin
   Result := FDialog.Options;
end;

function TsFilenameEdit.GetDialogTitle: string;
begin
   Result := FDialog.Title;
end;

function TsFilenameEdit.GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap;
begin
   DestroyNeeded := FALSE;
   if FileBitmap = nil then begin
      FileBitmap := TBitmap.Create;
      FileBitmap.Handle := LoadBitmap(hInstance, sFileBmp);
   end;
   Result := FileBitmap;
end;

procedure TsFilenameEdit.SetDefaultExt(Value: TFileExt);
begin
   FDialog.DefaultExt := Value;
end;

procedure TsFilenameEdit.SetFileEditStyle(Value: TFileEditStyle);
begin
   FDialog.FileEditStyle := Value;
end;

procedure TsFilenameEdit.SetFilter(const Value: string);
begin
   FDialog.Filter := Value;
end;

procedure TsFilenameEdit.SetFilterIndex(Value: Integer);
begin
   FDialog.FilterIndex := Value;
end;

procedure TsFilenameEdit.SetInitialDir(const Value: string);
begin
   FDialog.InitialDir := Value;
end;

procedure TsFilenameEdit.SetHistoryList(Value: TStrings);
begin
   FDialog.HistoryList := Value;
end;

procedure TsFilenameEdit.SetOptions(Value: TOpenOptions);
begin
   FDialog.Options := Value;
   FNoPathValidate := ofNoValidate in Value;
end;

procedure TsFilenameEdit.SetDialogTitle(const Value: string);
begin
   FDialog.Title := Value;
end;

{******************** TsDirectoryEdit *******************************}
constructor TsDirectoryEdit.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FDirectoryEdit := TRUE;
   FDialog := TsBrowseFolderDialog.Create(self);
   FDialog.Title := SBrowseCaption;
   FDialog.Folder := foCustom;
   FDialog.ShowPath := TRUE;
end;

destructor TsDirectoryEdit.Destroy;
begin
   FDialog.Free;
   inherited;
end;

function TsDirectoryEdit.GetDialogCaption: string;
begin
   Result := FDialog.Caption;
end;

procedure TsDirectoryEdit.SetDialogCaption(Value: string);
begin
   FDialog.Caption := Value;
end;

function TsDirectoryEdit.IsCustomCaption: Boolean;
begin
   Result := FDialog.Caption > '';
end;

function TsDirectoryEdit.GetDialogTitle: string;
begin
   Result := FDialog.Title;
end;

procedure TsDirectoryEdit.SetDialogTitle(Value: string);
begin
   FDialog.Title := Value;
end;

function TsDirectoryEdit.IsCustomTitle: Boolean;
begin
   Result := CompareStr( SBrowseCaption, FDialog.Title) <> 0;
end;

procedure TsDirectoryEdit.SetDialogOptions( Value: TBrowseOptions);
begin
   FDialog.options := Value;
end;

function TsDirectoryEdit.GetDialogOptions: TBrowseOptions;
begin
   Result := FDialog.Options;
end;

procedure TsDirectoryEdit.SetFolder( Value: TSHFolders);
begin
   FDialog.Folder := Value;
end;

function TsDirectoryEdit.GetFolder: TSHFolders;
begin
   Result := FDialog.Folder;
end;

procedure TsDirectoryEdit.SetShowPath(Value: Boolean);
begin
   FDialog.ShowPath := Value;
end;

function TsDirectoryEdit.GetShowPath: Boolean;
begin
   Result := FDialog.ShowPath;
end;

function TsDirectoryEdit.GetInitialDir: string;
begin
   Result := FInitialDir;
end;

procedure TsDirectoryEdit.SetInitialDir(const Value: string);
begin
   FInitialDir := Value;
end;

procedure TsDirectoryEdit.ButtonClick;
var
   aButtonClick: TNotifyEvent;
   Action: Boolean;
   tempPath: String;
begin
   if button = ebButton1 then begin
      aButtonClick := FOnButton1Click;
   end else begin
      aButtonClick := FOnButton2Click;
   end;
   if Assigned(aButtonClick) then
      inherited
   else if EditCanModify then begin
      Action := TRUE;
      tempPath := Path;
      DoBeforeDialog(tempPath, Action);
      if not Action then
      	Exit;
      if not DirExists(tempPath) then
      	tempPath := '';
      if (tempPath = '') then
         if (InitialDir <> '') and DirExists(InitialDir) then
         	tempPath := InitialDir
         else
         	tempPath := '\';
      FDialog.Directory := tempPath;
      FDialog.OnInitialized := FOnInitialized;
      FDialog.OnSelectionChanged := FOnSelectionChanged;
      Action := FDialog.Execute;
      SetFocus;
      DoAfterDialog(tempPath, Action);
      if Action then begin
      	if tempPath = FDialog.Directory then
      		Action := FALSE
      	else
      		tempPath := FDialog.Directory;
         ChangeExpected;
         Path := tempPath;
      end;
   end;
end;

procedure TsDirectoryEdit.ReceptFileDir(const AFileName: string);
begin
   if FileExists(AFileName) then
      Text := ExtractFilePath(AFileName)
   else
      Text := AFileName;
end;

function TsDirectoryEdit.GetDefaultBitmap(var DestroyNeeded: Boolean): TBitmap;
begin
   DestroyNeeded := FALSE;
   if DirBitmap = nil then begin
      DirBitmap := TBitmap.Create;
      DirBitmap.Handle := LoadBitmap(hInstance, sDirBmp);
   end;
   Result := DirBitmap;
end;

initialization
   RegisterFlatControl(TsCustomEdit);

finalization
   UpDownBmp.Free;
   DateBitmap.Free;
   NumBitmap.Free;
   FileBitmap.Free;
   DirBitmap.Free;


end.
