{*******************************************************************************
   Unit
      sDBCtrls.pas
   Description:
      DB aware implementation of sEditTools
   Versions:
      2.2
 History:
      2.2   -  10/12/1998
               .TsDBMaskLinkEdit included
               .TsDBMemo included
      2.1a  -  30/11/1998
               .Changes suggested by Thomas von Stetten.
                  They marked with TVS in the comment.
      2.1	- 	3/11/1998
               .TsDBText included.
      2.0a	- 	1/11/1998
               .Fixed the bug reported by Guy from First Creative Group
      2.0*	- 	End of September
               .Initial release
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com
   Comments:
*     I did not track the versions before, so let's consider it as 2.0
*******************************************************************************}
unit sDBCtrls;

{$R-,B-}
{$I S.INC}

interface

uses
   Windows, Registry, Bde, Messages, Classes, Controls, Forms, Grids, Graphics, Buttons,
   Menus, StdCtrls, Mask, IniFiles, sEdits, sCtrls, DB, DBTables, DBCtrls, sMask, sMemo;

type
   TsFieldDataLink = class(TFieldDataLink)
   private
      FModified: Boolean; // <- Why, why Borland did not make any interface to this member?
   protected
      procedure EditingChanged; override;
      procedure RecordChanged(Field: TField); override;
      procedure UpdateData; override;
      procedure SetModified;
   public
      property IsModified: Boolean read FModified;
   end;

   TsDBEdit = class(TsCustomEdit)
   private
      FDataLink: TsFieldDataLink;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      procedure UpdateData(Sender: TObject);
      function GetDataField: string;
      procedure SetDataField(const Value: string);
      function GetDataSource: TDataSource;
      procedure SetDataSource(Value: TDataSource);
      function GetField: TField;
      function GetReadOnly: Boolean;
      procedure SetReadOnly(Value: Boolean);
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      function EditCanModify: Boolean; override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Reset; override;
      function IsValidChar(const Key: Char): Boolean; override;
      procedure Change; override;
      function Validate(var Pos: Integer): Boolean; override;
      procedure DoEnter; override;
      procedure ChangeExpected; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property DataLink: TsFieldDataLink read FDataLink;
   published
      property Alignment;
      property AutoSelect;
      property BorderStyle;
      property CharCase;
      property Color;
      property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Flat;
      property Font;
      property HideSelection;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PasswordChar;
      property Picture;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property ValidChars;
      property Visible;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
      //TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDBLinkEdit = class(TsCustomLinkEdit)
   private
      FDataLink: TsFieldDataLink;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      procedure UpdateData(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      function GetReadOnly: Boolean;
      procedure SetReadOnly(Value: Boolean);
      function GetField: TField;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      function EditCanModify: Boolean; override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Reset; override;
      function IsValidChar(const Key: Char): Boolean; override;
      procedure Change; override;
      function Validate(var Pos: Integer): Boolean; override;
      procedure DoEnter; override;
      procedure ChangeExpected; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property DataLink: TsFieldDataLink read FDataLink;
   published
      property Alignment;
      property AutoSelect;
      property BorderStyle;
      property Button1;
      property Button2;
      property CharCase;
      property Color;
      property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Flat;
      property Font;
      property HideSelection;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PasswordChar;
      property Picture;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property ValidChars;
      property Visible;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;

      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
      //TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   {TsDBDateEdit}
   TsDBDateEdit = class(TsCustomDateEdit)
   private
      FDataLink: TsFieldDataLink;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      procedure UpdateData(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      function GetReadOnly: Boolean;
      procedure SetReadOnly(Value: Boolean);
      function GetField: TField;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      procedure Loaded; override;
      function EditCanModify: Boolean; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Reset; override;
      procedure Change; override;
      function Validate(var Pos: Integer): Boolean; override;
      procedure DoEnter; override;
      procedure ChangeExpected; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property DataLink: TsFieldDataLink read FDataLink;
   published
      property Alignment;
      property AutoSelect;
      property BorderStyle;
      property Button1;
      property Button2;
      property Color;
      property Ctl3D;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property DateSeparator;
      property DateFormat;
      property DateOrder;
      property DisabledStyle;
      property DisabledFont;
      property DragCursor;
      property DragMode;
      property EmptyString;
      property Enabled;
      property Flat;
      property Font;
      property HideSelection;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PasswordChar;
      property Picture;
      property PopupCalendar;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
      //TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDBNumberEdit = class(TsCustomNumberEdit)
   private
      FDataLink: TsFieldDataLink;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      procedure UpdateData(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      function GetReadOnly: Boolean;
      procedure SetReadOnly(Value: Boolean);
      function GetField: TField;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      function EditCanModify: Boolean; override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Reset; override;
      procedure Change; override;
      function Validate(var Pos: Integer): Boolean; override;
      procedure DoEnter; override;
      procedure ChangeExpected; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property DataLink: TsFieldDataLink read FDataLink;
      property Value;
   published
      property Alignment;
      property AutoSelect;
      property BorderStyle;
      property Button1;
      property Button2;
      property Color;
      property Ctl3D;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property Decimals;
      property DisabledStyle;
      property DisabledFont;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Flat;
      property Font;
      property HideSelection;
      property ImeMode;
      property ImeName;
      property Increment;
      property MaxLength;
      property MaxValue;
      property MinValue;
      property NumPad;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PasswordChar;
      property Picture;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
      //TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDBMaskLinkEdit = class(TsCustomMaskEdit)
   private
      FDataLink: TsFieldDataLink;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      procedure UpdateData(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      function GetReadOnly: Boolean;
      procedure SetReadOnly(Value: Boolean);
      function GetField: TField;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      function EditCanModify: Boolean; override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Reset; override;
      procedure Change; override;
      function IsValidChar(const Key: Char): Boolean; override;
      function Validate(var Pos: Integer): Boolean; override;
      procedure DoEnter; override;
      procedure ChangeExpected; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property DataLink: TsFieldDataLink read FDataLink;
   published
      property AutoSelect;
      property BorderStyle;
      property Button1;
      property Button2;
      property CharCase;
      property Color;
      property Ctl3D;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property DisabledStyle;
      property DisabledFont;
      property DragCursor;
      property DragMode;
      property Enabled;
      property EditMask;
      property Flat;
      property Font;
      property HideSelection;
      property Hint;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PasswordChar;
      property Picture;
      property GlyphList; // mast be after picture.
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnButton1Click;
      property OnButton2Click;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
      property OnValidate;
      property OnValidateError;
{$IFDEF VER120}
      //Standard Delphi 4 Properties
   published
      property Anchors;
      property AutoSize;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDBCheckBox = class(TsCheckBox)
   private
      FDataLink: TFieldDataLink;
      FUserChange: Boolean;
      FValueCheck: string;
      FValueUncheck: string;
      procedure DataChange(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      function GetField: TField;
      function GetFieldState: TCheckBoxState;
      function GetReadOnly: Boolean;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      procedure SetReadOnly(Value: Boolean);
      procedure SetValueCheck(const Value: string);
      procedure SetValueUncheck(const Value: string);
      procedure UpdateData(Sender: TObject);
      function ValueMatch(const ValueList, Value: string): Boolean;
      procedure CMExit(var Message: TCMExit); message CM_EXIT;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      procedure Click; override;
      procedure KeyPress(var Key: Char); override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      function EditCanModify: Boolean; override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
      property State: TCheckBoxState read FState;
      property Checked: Boolean read GetChecked;
   published
      property Alignment;
      property Caption;
      property CheckColor;
      property Color;
      property Ctl3D;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Font;
      property Font3d;
      property HilightStyle;
      property Hilight3d;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property TextAlignment;
      property TransparentMode;
      property ValueChecked: string read FValueCheck write SetValueCheck;
      property ValueUnchecked: string read FValueUncheck write SetValueUncheck;
      property VerticalAlignment;
      property Visible;
      property WordWrap;
      property OnClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
   end;

   TsDBText = class(TsCustomLabel)
   private
      FDataLink: TsFieldDataLink;
      FEmptyString: string;
      procedure DataChange(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      function GetField: TField;
      function GetFieldText: string;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      procedure SetEmptyString(const Value: string);
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      function GetLabelText: string; override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure SetAutoSize(Value: Boolean); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Field: TField read GetField;
   published
      property Alignment;
      property Align;
      property AutoSize;
      property ButtonLike;
      property Color;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property DragCursor;
      property DragMode;
      property EmptyString: string read FEmptyString write SetEmptyString;
      property Enabled;
      property Font;
      property Font3D;
      property FocusControl;
      property Hilight3d;
      property HilightStyle;
      property Layout;
      property ParentColor;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowAccelChar;
      property ShowFocus;
      property ShowHint;
      property Transparent;
      property WordWrap;
      property Visible;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnMouseEnter;
      property OnMouseLeave;
      property OnMouseClick;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
{$IFDEF VER120}
      //TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsDBMemo = class(TsMemo)
   private
      FDataLink: TFieldDataLink;
      FAutoDisplay: Boolean;
      FFocused: Boolean;
      FMemoLoaded: Boolean;
      FPaintControl: TPaintControl;
      procedure DataChange(Sender: TObject);
      procedure EditingChange(Sender: TObject);
      function GetDataField: string;
      function GetDataSource: TDataSource;
      function GetField: TField;
      function GetReadOnly: Boolean;
      procedure SetDataField(const Value: string);
      procedure SetDataSource(Value: TDataSource);
      procedure SetReadOnly(Value: Boolean);
      procedure SetAutoDisplay(Value: Boolean);
      procedure SetFocused(Value: Boolean);
      procedure UpdateData(Sender: TObject);
      procedure WMCut(var Message: TMessage); message WM_CUT;
      procedure WMPaste(var Message: TMessage); message WM_PASTE;
      procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
      procedure CMExit(var Message: TCMExit); message CM_EXIT;
      procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
      procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
      procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
   protected
      procedure Change; override;
      procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent;
         Operation: TOperation); override;
      procedure WndProc(var Message: TMessage); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure LoadMemo;
      property Field: TField read GetField;
   published
      property Align;
      property Alignment;
      property AutoDisplay: Boolean read FAutoDisplay write SetAutoDisplay default True;
      property BorderStyle;
      property Color;
      property Ctl3D;
      property DataField: string read GetDataField write SetDataField;
      property DataSource: TDataSource read GetDataSource write SetDataSource;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Font;
      property ImeMode;
      property ImeName;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
      property ScrollBars;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property WantTabs;
      property WordWrap;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
{$IFDEF VER120}
   published
      property Anchors;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

implementation

uses SysUtils, Dialogs, ExtCtrls, sDate, consts, DBconsts;

procedure TsFieldDataLink.EditingChanged;
begin
   inherited;
   if Editing and CanModify then
      FModified := TRUE;
end;

procedure TsFieldDataLink.RecordChanged(Field: TField);
begin
   inherited;
   if (Field = nil) or (Field = inherited Field) then
      FModified := False;
end;

procedure TsFieldDataLink.UpdateData;
begin
   inherited;
   FModified := False;
end;

procedure TsFieldDataLink.SetModified;
begin
   Modified;
   FModified := TRUE;
end;

{ sDBEdit}

constructor TsDBEdit.Create(AOwner: TComponent);
begin
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TsFieldDataLink.Create;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnEditingChange := EditingChange;
   FDataLink.OnUpdateData := UpdateData;
   inherited Create(AOwner);
   inherited ReadOnly := TRUE;
end;

destructor TsDBEdit.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited;
end;

procedure TsDBEdit.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
end;

procedure TsDBEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

function TsDBEdit.IsValidChar(const Key: Char): Boolean;
begin
   Result := inherited IsValidChar(Key);
   if Result and (Key in [#32..#255]) and (FDataLink.Field <> nil) then
      Result := FDataLink.Field.IsValidChar(Key);
end;

procedure TsDBEdit.ChangeExpected;
begin
   FDataLink.Edit;
end;

function TsDBEdit.EditCanModify: Boolean;
begin
   Result := FDataLink.Editing;
end;

procedure TsDBEdit.Reset;
begin
   FDataLink.Reset;
   SelectAll;
end;

procedure TsDBEdit.DoEnter;
begin
   Reset;
   Exclude(FEditState, msNoRedraw);
   inherited;
end;

function TsDBEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := inherited Validate(Pos);
   if Result then try
      FDataLink.UpdateRecord;
   except
      Pos := -1;
      Result := FALSE;
   end;
end;

procedure TsDBEdit.Change;
begin
   FDataLink.SetModified;
   inherited Change;
end;

function TsDBEdit.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBEdit.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBEdit.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBEdit.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBEdit.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBEdit.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBEdit.DataChange(Sender: TObject);
begin
   if FDataLink.Field <> nil then begin
      if FDataLink.Field.DataType = ftString then
         MaxLength := FDataLink.Field.Size
      else
         MaxLength := 0;
      if Focused and FDataLink.CanModify then
         Text := FDataLink.Field.Text
      else begin
         Text := FDataLink.Field.DisplayText;
         if FDataLink.Editing and FDataLink.FModified then
            Modified := TRUE;
      end;
   end else begin
      if csDesigning in ComponentState then
         Text := Name
      else
         Text := '';
   end;
end;

procedure TsDBEdit.EditingChange(Sender: TObject);
begin
   inherited ReadOnly := not FDataLink.Editing;
end;

procedure TsDBEdit.UpdateData(Sender: TObject);
begin
   FDataLink.Field.Text := Text;
end;

procedure TsDBEdit.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{ DBLinkEdit}

constructor TsDBLinkEdit.Create(AOwner: TComponent);
begin
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TsFieldDataLink.Create;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnEditingChange := EditingChange;
   FDataLink.OnUpdateData := UpdateData;
   inherited Create(AOwner);
   inherited ReadOnly := True;
end;

destructor TsDBLinkEdit.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited;
end;

procedure TsDBLinkEdit.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
   Exclude(FEditState, msFirstTimePainted);
end;

procedure TsDBLinkEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

function TsDBLinkEdit.IsValidChar(const Key: Char): Boolean;
begin
   Result := inherited IsValidChar(Key);
   if Result and (Key in [#32..#255]) and (FDataLink.Field <> nil) then
      Result := FDataLink.Field.IsValidChar(Key);
end;

procedure TsDBLinkEdit.ChangeExpected;
begin
   FDataLink.Edit;
end;

function TsDBLinkEdit.EditCanModify: Boolean;
begin
   Result := not FDataLink.ReadOnly;
end;

procedure TsDBLinkEdit.Reset;
begin
   FDataLink.Reset;
   SelectAll;
end;

procedure TsDBLinkEdit.DoEnter;
begin
   Reset;
   Exclude(FEditState, msNoRedraw);
   inherited;
end;

function TsDBLinkEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := inherited Validate(pos);
   if Result then try
      FDataLink.UpdateRecord;
   except
      Pos := -1;
      Result := FALSE;
   end;
end;

procedure TsDBLinkEdit.Change;
begin
   FDataLink.SetModified;
   inherited Change;
end;

function TsDBLinkEdit.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBLinkEdit.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBLinkEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBLinkEdit.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBLinkEdit.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBLinkEdit.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBLinkEdit.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBLinkEdit.DataChange(Sender: TObject);
begin
   if FDataLink.Field <> nil then begin
      if FDataLink.Field.DataType = ftString then
         MaxLength := FDataLink.Field.Size
      else
         MaxLength := 0;
      if Focused and FDataLink.CanModify then
         Text := FDataLink.Field.Text
      else begin
         Text := FDataLink.Field.DisplayText;
         if FDataLink.Editing and FDataLink.FModified then
            Modified := TRUE;
      end;
   end else begin
      if csDesigning in ComponentState then
         Text := Name
      else
         Text := '';
   end;
end;

procedure TsDBLinkEdit.EditingChange(Sender: TObject);
begin
   inherited ReadOnly := not FDataLink.Editing;
end;

procedure TsDBLinkEdit.UpdateData(Sender: TObject);
begin
   FDataLink.Field.Text := Text;
end;

procedure TsDBLinkEdit.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{TsDateEdit}

constructor TsDBDateEdit.Create(AOwner: TComponent);
begin
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TsFieldDataLink.Create;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnEditingChange := EditingChange;
   FDataLink.OnUpdateData := UpdateData;
   inherited Create(AOwner);
   inherited ReadOnly := True;
end;

destructor TsDBDateEdit.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited;
end;

procedure TsDBDateEdit.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
   Exclude(FEditState, msFirstTimePainted);
end;

procedure TsDBDateEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

procedure TsDBDateEdit.ChangeExpected;
begin
   FDataLink.Edit;
end;

function TsDBDateEdit.EditCanModify: Boolean;
begin
   Result := not FDataLink.ReadOnly;
end;

procedure TsDBDateEdit.Reset;
begin
   FDataLink.Reset;
   SelectAll;
end;

procedure TsDBDateEdit.DoEnter;
begin
   Reset;
   Exclude(FEditState, msNoRedraw);
   inherited;
end;

function TsDBDateEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := inherited Validate(pos);
   if Result then begin
      if FDataLink.IsModified and EditDate.IsNull then begin
         if FDataLink.Field <> nil then
            FDataLink.Field.Clear;
      end else try
         FDataLink.UpdateRecord;
      except
         Pos := -1;
         Result := FALSE;
      end;
   end;
end;

procedure TsDBDateEdit.Change;
begin
   FDataLink.SetModified;
   inherited Change;
end;

function TsDBDateEdit.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBDateEdit.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBDateEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBDateEdit.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBDateEdit.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBDateEdit.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBDateEdit.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBDateEdit.DataChange(Sender: TObject);
begin
   if FDataLink.Field <> nil then begin
      try
         EditDate.AsDateTime := FDataLink.Field.AsDateTime;
      except
         Text := EmptyText;
      end;
      if not Focused or not FDataLink.CanModify then begin
         if FDataLink.Editing and FDataLink.FModified then
            Modified := TRUE;
      end;
   end else
      Text := EmptyText;
end;

procedure TsDBDateEdit.EditingChange(Sender: TObject);
begin
   inherited ReadOnly := not FDataLink.Editing;
end;

procedure TsDBDateEdit.UpdateData(Sender: TObject);
begin
   FDataLink.Field.AsDateTime := EditDate.AsDateTime;
end;

procedure TsDBDateEdit.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{TsDBNumberEdit}

constructor TsDBNumberEdit.Create(AOwner: TComponent);
begin
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TsFieldDataLink.Create;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnEditingChange := EditingChange;
   FDataLink.OnUpdateData := UpdateData;
   inherited Create(AOwner);
   inherited ReadOnly := True;
end;

destructor TsDBNumberEdit.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited;
end;

procedure TsDBNumberEdit.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
   Exclude(FEditState, msFirstTimePainted);
end;

procedure TsDBNumberEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

procedure TsDBNumberEdit.ChangeExpected;
begin
   FDataLink.Edit;
end;

function TsDBNumberEdit.EditCanModify: Boolean;
begin
   Result := not FDataLink.ReadOnly;
end;

procedure TsDBNumberEdit.Reset;
begin
   FDataLink.Reset;
   SelectAll;
end;

procedure TsDBNumberEdit.DoEnter;
begin
   Reset;
   Exclude(FEditState, msNoRedraw);
   inherited;
end;

function TsDBNumberEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := inherited Validate(Pos);
   if Result then try
      FDataLink.UpdateRecord;
   except
      pos := -1;
      Result := FALSE;
   end;
end;

procedure TsDBNumberEdit.Change;
begin
   FDataLink.SetModified;
   inherited Change;
end;

function TsDBNumberEdit.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBNumberEdit.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBNumberEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBNumberEdit.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBNumberEdit.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBNumberEdit.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBNumberEdit.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBNumberEdit.DataChange(Sender: TObject);
begin
   if FDataLink.Field <> nil then begin
      try
         Value := FDataLink.Field.AsFloat;
      except
         Value := 0;
      end;
      if not Focused or not FDataLink.CanModify then begin
         if FDataLink.Editing and FDataLink.FModified then
            Modified := TRUE;
      end;
   end else
      Value := 0;
end;

procedure TsDBNumberEdit.EditingChange(Sender: TObject);
begin
   inherited ReadOnly := not FDataLink.Editing;
end;

procedure TsDBNumberEdit.UpdateData(Sender: TObject);
begin
   FDataLink.Field.Text := Text;
end;

procedure TsDBNumberEdit.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{ DBMaskLinkEdit}

constructor TsDBMaskLinkEdit.Create(AOwner: TComponent);
begin
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TsFieldDataLink.Create;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnEditingChange := EditingChange;
   FDataLink.OnUpdateData := UpdateData;
   inherited Create(AOwner);
   inherited ReadOnly := True;
end;

destructor TsDBMaskLinkEdit.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited;
end;

procedure TsDBMaskLinkEdit.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
   Exclude(FEditState, msFirstTimePainted);
end;

procedure TsDBMaskLinkEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

function TsDBMaskLinkEdit.IsValidChar(const Key: Char): Boolean;
begin
   Result := inherited IsValidChar(Key);
   if Result and (Key in [#32..#255]) and (FDataLink.Field <> nil) then
      Result := FDataLink.Field.IsValidChar(Key);
end;

procedure TsDBMaskLinkEdit.ChangeExpected;
begin
   FDataLink.Edit;
end;

function TsDBMaskLinkEdit.EditCanModify: Boolean;
begin
   Result := not FDataLink.ReadOnly;
end;

procedure TsDBMaskLinkEdit.Reset;
begin
   FDataLink.Reset;
   SelectAll;
end;

procedure TsDBMaskLinkEdit.DoEnter;
begin
   Reset;
   Exclude(FEditState, msNoRedraw);
   inherited;
end;

function TsDBMaskLinkEdit.Validate(var Pos: Integer): Boolean;
begin
   Result := inherited Validate(pos);
   if Result then try
      FDataLink.UpdateRecord;
   except
      Pos := -1;
      Result := FALSE;
   end;
end;

procedure TsDBMaskLinkEdit.Change;
begin
   FDataLink.SetModified;
   inherited Change;
end;

function TsDBMaskLinkEdit.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBMaskLinkEdit.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBMaskLinkEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBMaskLinkEdit.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBMaskLinkEdit.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBMaskLinkEdit.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBMaskLinkEdit.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBMaskLinkEdit.DataChange(Sender: TObject);
begin
   if FDataLink.Field <> nil then begin
      if FDataLink.Field.DataType = ftString then
         MaxLength := FDataLink.Field.Size
      else
         MaxLength := 0;
      if Focused and FDataLink.CanModify then
         Text := FDataLink.Field.Text
      else begin
         Text := FDataLink.Field.DisplayText;
         if FDataLink.Editing and FDataLink.FModified then
            Modified := TRUE;
      end;
   end else begin
      if csDesigning in ComponentState then
         Text := Name
      else
         Text := '';
   end;
end;

procedure TsDBMaskLinkEdit.EditingChange(Sender: TObject);
begin
   inherited ReadOnly := not FDataLink.Editing;
end;

procedure TsDBMaskLinkEdit.UpdateData(Sender: TObject);
begin
   FDataLink.Field.Text := Text;
end;

procedure TsDBMaskLinkEdit.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{TsDBCheckBox}

constructor TsDBCheckBox.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csReplicatable];
   FDataLink := TFieldDataLink.Create;
   FValueCheck := STextTrue;
   FValueUncheck := STextFalse;
   FDataLink.Control := Self;
   FDataLink.OnDataChange := DataChange;
   FDataLink.OnUpdateData := UpdateData;
end;

destructor TsDBCheckBox.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited Destroy;
end;

procedure TsDBCheckBox.CMExit(var Message: TCMExit);
begin
   try
      FDataLink.UpdateRecord;
   except
      SetFocus;
      raise;
   end;
   inherited;
end;

procedure TsDBCheckBox.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

procedure TsDBCheckBox.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

function TsDBCheckBox.GetFieldState: TCheckBoxState;
var
   Text: string;
begin
   if FDatalink.Field <> nil then
      if FDataLink.Field.IsNull then
         Result := cbUnchecked
      else if FDataLink.Field.DataType = ftBoolean then
         if FDataLink.Field.AsBoolean then
            Result := cbChecked
         else
            Result := cbUnchecked
      else begin
         Result := cbGrayed;
         Text := FDataLink.Field.Text;
         if ValueMatch(FValueCheck, Text) then
            Result := cbChecked
         else if ValueMatch(FValueUncheck, Text) then
            Result := cbUnchecked;
      end else
      Result := cbUnchecked;
end;

procedure TsDBCheckBox.DataChange(Sender: TObject);
begin
   if not FUserChange then
      inherited State := GetFieldState;
end;

procedure TsDBCheckBox.UpdateData(Sender: TObject);
var
   Pos: Integer;
   S: string;
begin
   if State = cbGrayed then
      FDataLink.Field.Clear
   else if FDataLink.Field.DataType = ftBoolean then
      FDataLink.Field.AsBoolean := Checked
   else begin
      if Checked then
         S := FValueCheck
      else
         S := FValueUncheck;
      Pos := 1;
      FDataLink.Field.Text := ExtractFieldName(S, Pos);
   end;
end;

function TsDBCheckBox.ValueMatch(const ValueList, Value: string): Boolean;
var
   Pos: Integer;
begin
   Result := False;
   Pos := 1;
   while Pos <= Length(ValueList) do
      if AnsiCompareText(ExtractFieldName(ValueList, Pos), Value) = 0 then begin
         Result := True;
         Break;
      end;
end;

procedure TsDBCheckBox.Click;
begin
   try
      FUserChange := TRUE;
      if FDataLink.Edit then begin
         inherited Click;
         FDataLink.Modified;
      end;
   finally
      FUserChange := FALSE;
   end;
end;

function TsDBCheckBox.EditCanModify: Boolean;
begin
   Result := FDataLink.Editing;
end;

function TsDBCheckBox.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBCheckBox.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then
      Value.FreeNotification(Self);
end;

function TsDBCheckBox.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBCheckBox.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBCheckBox.GetReadOnly: Boolean;
begin
   Result := FDataLink.ReadOnly;
end;

procedure TsDBCheckBox.SetReadOnly(Value: Boolean);
begin
   FDataLink.ReadOnly := Value;
end;

function TsDBCheckBox.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBCheckBox.KeyPress(var Key: Char);
begin
   inherited KeyPress(Key);
   case Key of
      #8, ' ':
         FDataLink.Edit;
      #27:
         FDataLink.Reset;
   end;
end;

procedure TsDBCheckBox.SetValueCheck(const Value: string);
begin
   FValueCheck := Value;
   DataChange(Self);
end;

procedure TsDBCheckBox.SetValueUncheck(const Value: string);
begin
   FValueUncheck := Value;
   DataChange(Self);
end;

{ Ts3DDBText }

constructor TsDBText.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csReplicatable];
   AutoSize := False;
   ShowAccelChar := False;
   FDataLink := TsFieldDataLink.Create;
   FDataLink.OnDataChange := DataChange;
   Caption := GetFieldText;
end;

destructor TsDBText.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   inherited Destroy;
end;

procedure TsDBText.Loaded;
begin
   inherited Loaded;
   if (csDesigning in ComponentState) then
      DataChange(Self);
end;

procedure TsDBText.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (FDataLink <> nil) and (AComponent = DataSource) then
      DataSource := nil;
end;

procedure TsDBText.SetAutoSize(Value: Boolean);
begin
   if AutoSize <> Value then begin
      if Value and FDataLink.DataSourceFixed then
         raise Exception.Create(sDataSourceFixed);
      inherited SetAutoSize(Value);
   end;
end;

function TsDBText.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TsDBText.SetDataSource(Value: TDataSource);
begin
   FDataLink.DataSource := Value;
   if Value <> nil then Value.FreeNotification(Self);
end;

function TsDBText.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TsDBText.SetDataField(const Value: string);
begin
   FDataLink.FieldName := Value;
end;

function TsDBText.GetField: TField;
begin
   Result := FDataLink.Field;
end;

procedure TsDBText.SetEmptyString(const Value: string);
begin
   if Value <> FEmptyString then begin
      FEmptyString := Value;
      Caption := GetFieldText;
   end;
end;

function TsDBText.GetFieldText: string;
begin
   if FDataLink.Active and (FDataLink.Field <> nil) then
      if FDataLink.Field.IsNull then
         Result := FEmptyString
      else if FDataLink.Field.DataType = ftMemo then
         Result := FDataLink.Field.AsString
      else
         Result := FDataLink.Field.DisplayText
   else
      if csDesigning in ComponentState then
      Result := Name
   else
      Result := FEmptyString;
end;

procedure TsDBText.DataChange(Sender: TObject);
begin
   Caption := GetFieldText;
end;

function TsDBText.GetLabelText: string;
begin
   if csPaintCopy in ControlState then
      Result := GetFieldText
   else
      Result := Caption;
end;

procedure TsDBText.CMGetDataLink(var Message: TMessage);
begin
   Message.Result := Integer(FDataLink);
end;

{...........}

constructor TsDBMemo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  ControlStyle := ControlStyle + [csReplicatable];
  FAutoDisplay := True;
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  FPaintControl := TPaintControl.Create(Self, 'EDIT');
end;

destructor TsDBMemo.Destroy;
begin
  FPaintControl.Free;
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TsDBMemo.Loaded;
begin
  inherited Loaded;
  if (csDesigning in ComponentState) then DataChange(Self);
end;

procedure TsDBMemo.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TsDBMemo.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if FMemoLoaded then
  begin
    if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
      FDataLink.Edit;
  end;
end;

procedure TsDBMemo.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if FMemoLoaded then
  begin
    if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
      not FDataLink.Field.IsValidChar(Key) then
    begin
      MessageBeep(0);
      Key := #0;
    end;
    case Key of
      ^H, ^I, ^J, ^M, ^V, ^X, #32..#255:
        FDataLink.Edit;
      #27:
        FDataLink.Reset;
    end;
  end else
  begin
    if Key = #13 then LoadMemo;
    Key := #0;
  end;
end;

procedure TsDBMemo.Change;
begin
   if FMemoLoaded then
      FDataLink.Modified;
   FMemoLoaded := True;
   inherited Change;
end;

function TsDBMemo.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TsDBMemo.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TsDBMemo.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TsDBMemo.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function TsDBMemo.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TsDBMemo.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TsDBMemo.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TsDBMemo.LoadMemo;
begin
  if not FMemoLoaded and Assigned(FDataLink.Field) and FDataLink.Field.IsBlob then
  begin
    try
      Lines.Text := FDataLink.Field.AsString;
      FMemoLoaded := True;
    except
      { Memo too large }
      on E:EInvalidOperation do
        Lines.Text := Format('(%s)', [E.Message]);
    end;
    EditingChange(Self);
  end;
end;

procedure TsDBMemo.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
    if FDataLink.Field.IsBlob then
    begin
      if FAutoDisplay or (FDataLink.Editing and FMemoLoaded) then
      begin
        FMemoLoaded := False;
        LoadMemo;
      end else
      begin
        Text := Format('(%s)', [FDataLink.Field.DisplayLabel]);
        FMemoLoaded := False;
      end;
    end else
    begin
      if FFocused and FDataLink.CanModify then
        Text := FDataLink.Field.Text
      else
        Text := FDataLink.Field.DisplayText;
      FMemoLoaded := True;
    end
  else
  begin
    if csDesigning in ComponentState then Text := Name else Text := '';
    FMemoLoaded := False;
  end;
  if HandleAllocated then
    RedrawWindow(Handle, nil, 0, RDW_INVALIDATE or RDW_ERASE or RDW_FRAME);
end;

procedure TsDBMemo.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not (FDataLink.Editing and FMemoLoaded);
end;

procedure TsDBMemo.UpdateData(Sender: TObject);
begin
  FDataLink.Field.AsString := Text;
end;

procedure TsDBMemo.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if not Assigned(FDataLink.Field) or not FDataLink.Field.IsBlob then
      FDataLink.Reset;
  end;
end;

procedure TsDBMemo.WndProc(var Message: TMessage);
begin
  with Message do
    if (Msg = WM_CREATE) or (Msg = WM_WINDOWPOSCHANGED) or
      (Msg = CM_FONTCHANGED) then FPaintControl.DestroyHandle;
  inherited;
end;

procedure TsDBMemo.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
  if SysLocale.FarEast and FDataLink.CanModify then
    inherited ReadOnly := False;
end;

procedure TsDBMemo.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SetFocus;
    raise;
  end;
  SetFocused(False);
  inherited;
end;

procedure TsDBMemo.SetAutoDisplay(Value: Boolean);
begin
  if FAutoDisplay <> Value then
  begin
    FAutoDisplay := Value;
    if Value then LoadMemo;
  end;
end;

procedure TsDBMemo.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  if not FMemoLoaded then LoadMemo else inherited;
end;

procedure TsDBMemo.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TsDBMemo.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TsDBMemo.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

procedure TsDBMemo.WMPaint(var Message: TWMPaint);
var
  S: string;
begin
  if not (csPaintCopy in ControlState) then inherited else
  begin
    if FDataLink.Field <> nil then
      if FDataLink.Field.IsBlob then
      begin
        if FAutoDisplay then
          S := AdjustLineBreaks(FDataLink.Field.AsString) else
          S := Format('(%s)', [FDataLink.Field.DisplayLabel]);
      end else
        S := FDataLink.Field.DisplayText;
    SendMessage(FPaintControl.Handle, WM_SETTEXT, 0, Integer(PChar(S)));
    SendMessage(FPaintControl.Handle, WM_ERASEBKGND, Message.DC, 0);
    SendMessage(FPaintControl.Handle, WM_PAINT, Message.DC, 0);
  end;
end;



end.

