{*******************************************************************************
   Unit
      sCtrls.pas
   Description:
      Transparent, BWCC style CheckBox and RadioButton,
      3d label
      TsSpeedButton with GlyphList support and TBitBtn properties
      TsStatusBar
   Versions:
      2.2e
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   History:
      2.2e        -  14/12/1998
                     .New property ButtonLike for TBWCC controls - it is not always
                     suitable to have button like behavior, so now you can switch it off.
                     .New property AutoSize for TBWSS controls.
                     .New property FocusFrame for TBWCC controls and labels - to be able
                     to switch off the focus frame.
                     .Fixed focus frame painting in TsLabel
                     .Fixed improper beh. of TsLabel when ShowFocus = FALSE and
                        FocusControl <> nil
                     .Fixed size calculation when hilightStyle = hsFlushing
      2.2d        -  3/12/1998
                     .check invalidate problem fixed (I was sick to make such kind of bug!)
                     .Flicker redused for sLabel
      2.2c        -  30/11/1998
                     .Changes suggested by Thomas von Stetten.
                        They marked with TVS in the comment.
      2.2b        -  27/11/1998
                     ShortCut handling for TsCheckBox and TsRadioButton.
                     TsLabel's connection to LinkedControl is improved:
                        -MouseEnter, MouseLeave messages are resend to linked
                        control (looks better)
                        -If it is linked to sXXXEdit or sXXXComboBox then
                        notifications for MouseEnter, MouseLeave messages of this
                        control are handled too.
                     It gives the fill that the label and linked control behave
                     as one logical "unit".
      2.2a        -  25/11/1998
                     handling of "enabled changed" message from the linked control
                     for TsLabel. 
      2.2         -  21/11/1998
                     -TsSpeedButton with support for TsGlyphList. Initial realease.
                     -TsStstusBar. Initial realease.
      2.1         -  08/11/1998
                     -TsLabel. Initial release.
                     -Improved painting of disabled control
      2.0*        -  End of Sep. 1998
      			      Initial release
   Comments:
      This components are part of sEditTools. Why BWCC ? Whell, I believe that
      it's style is more "flat" than Microsoft's check box and radio button. If you
      are not agree - use transparent controls from Dream Company, they are great!

*     I did not track the versions before, so let's consider it as 2.0
**    Enrico Lodolo - ldlc18k1@bo.nettuno.it -
         Idea and some code in the box drawing part was taken from his
         TBorRadio and TBorCheck. Thanks!
*******************************************************************************}
unit sCtrls;

{$S-,W-,R-}
{$C PRELOAD}

interface

uses
   SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
   Forms, Dialogs, StdCtrls, ComCtrls, Buttons, sTrans, sGraphics, sGlyphsList,
   sConsts;

type
   TTransparentMode = (tmOpaque, tmSemiTransparent, tmTransparent);

   TsCtrlState = (scsMouseInControl, scsFocused, scsInvalidateCaption, scsInvalidateBox);
   TsCtrlStates = set of TsCtrlState;

   TVerticalAlignment = (vaTop, vaBottom, vaCenter);
   TsBWCCBoxStyle = (bwsRaised, bwsSunken);

   TsBWCCControl = class(TsTransControl)
   private
      FAlignment: TLeftRight;
      FAutoSize: Boolean;
      FButtonLike: Boolean;
      FCheckColor: TColor;
      FCtrlState: TsCtrlStates;
      FDown: Boolean;
      FPaint3D: TFont3d;
      FFont3d: TFont3d;
      FHilightStyle: TTextHilightStyle;
      FHilight3d: TFont3d;
      FFocusFrame: Boolean;
      FTextAlignment: TAlignment;
      FTransparentMode: TTransparentMode;
      FVerticalAlignment: TVerticalAlignment;
      FWordWrap: Boolean;
      FOnMouseEnter: TNotifyEvent;
      FOnMouseLeave: TNotifyEvent;
      procedure Font3dChanged(Sender: Tobject);
      procedure AdjustBounds;
      function GetCaption: TCaption;
      procedure SetAlignment(Value: TLeftRight);
      procedure SetAutoSize(Value: Boolean);
      procedure SetButtonLike(Value: Boolean);
      procedure SetCaption(const Value: TCaption);
      procedure SetCheckColor(Value: TColor);
      procedure SetFont3d(Value: TFont3d);
      procedure SetHilight3d(Value: TFont3d);
      procedure SetHilightStyle(Value: TTextHilightStyle);
      procedure SetFocusFrame(Value: Boolean);
      procedure SetTextAlignment(Value: TAlignment);
      procedure SetTransparentMode(Value: TTransparentMode);
      procedure SetVerticalAlignment(Value: TVerticalAlignment);
      procedure SetWordWrap(Value: Boolean);
      procedure PaintCaption(aCanvas: TCanvas);
      procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
      procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
      procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
   protected
      FState: TCheckBoxState;
      procedure Paint; override;
      procedure PaintBox(aCanvas: TCanvas; R: TRect); virtual; abstract;
      procedure Toggle; virtual; abstract;
      procedure SetChecked(Value: Boolean); virtual;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
      procedure KeyDown(var Key: Word; Shift: TShiftSTate); override;
      procedure KeyUp(var Key: Word; Shift: TShiftSTate); override;
      procedure DoEnter; override;
      procedure DoExit; override;
      procedure DoMouseEnter; virtual;
      procedure DoMouseLeave; virtual;
      function EditCanModify: Boolean; virtual;
      function GetChecked: Boolean;
      property Alignment: TLeftRight read FAlignment write SetAlignment default taLeftJustify;
      property AutoSize: Boolean read FAutoSize write SetAutoSize default FALSE;
      property ButtonLike: Boolean read FButtonLike write SetButtonLike default TRUE;
      property Caption: TCaption read GetCaption write SetCaption;
      property CheckColor: TColor read FCheckColor write SetCheckColor default clBlack;
      property Checked: Boolean read GetChecked write SetChecked default False;
      property FocusFrame: Boolean read FFocusFrame write SetFocusFrame default FALSE;
      property Font3d: TFont3d read FFont3d write SetFont3d;
      property HilightStyle: TTextHilightStyle read FHilightStyle write SetHilightStyle default hsFlushing;
      property Hilight3d: TFont3d read FHilight3d write SetHilight3d;
      property TextAlignment: TAlignment read FTextAlignment write SetTextAlignment default taLeftJustify;
      property TransparentMode: TTransparentMode read FTransparentMode write SetTransparentMode default tmOpaque;
      property VerticalAlignment: TVerticalAlignment read FVerticalAlignment write SetVerticalAlignment default vaTop;
      property WordWrap: Boolean read FWordWrap write SetWordWrap default FALSE;
      property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
      property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
   end;

   TsCustomCheckBox = class(TsBWCCControl)
   private
      FAllowGrayed: Boolean;
      FStyle: TsBWCCBoxStyle;
      procedure SetStyle( Value: TsBWCCBoxStyle);
      procedure SetState(Value: TCheckBoxState);
   protected
      procedure PaintBox(aCanvas: TCanvas; R: TRect); override;
      property AllowGrayed: Boolean read FAllowGrayed write FAllowGrayed default FALSE;
      property State: TCheckBoxState read FState write SetState default cbUnchecked;
      property Style: TsBWCCBoxStyle read FStyle write SetStyle default bwsRaised;
   public
      procedure Toggle; override;
   end;

   TsCheckBox = class(TsCustomCheckBox)
   published
      property Alignment;
      property AllowGrayed;
      property AutoSize;
      property ButtonLike;
      property Caption;
      property CheckColor;
      property Checked;
      property Color;
      property DragCursor;
      property DragMode;
      property Enabled;
      property FocusFrame;
      property Font;
      property Font3d;
      property Hilight3d;
      property HilightStyle;
      property ParentFont;
      property PopupMenu;
      property ShowHint;
      property State;
      property Style;
      property TabOrder;
      property TabStop;
      property TextAlignment;
      property TransparentMode;
      property VerticalAlignment;
      property Visible;
      property WordWrap;
      property OnClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnMouseEnter;
      property OnMouseLeave;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsCustomRadioButton = class(TsBWCCControl)
   private
      FGroupIndex: Byte;
   protected
      procedure PaintBox(aCanvas: TCanvas; R: TRect); override;
      procedure SetChecked(Value: Boolean); override;
      property GroupIndex: Byte read FGroupIndex write FGroupIndex default 0;
   public
      procedure Toggle; override;
   end;

   TsRadioButton = class(TsCustomRadioButton)
   published
      property Alignment;
      property AutoSize;
      property ButtonLike;
      property Caption;
      property CheckColor;
      property Checked;
      property Color;
      property DragCursor;
      property DragMode;
      property Enabled;
      property FocusFrame;
      property Font;
      property Font3d;
      property GroupIndex;
      property Hilight3d;
      property HilightStyle;
      property ParentFont;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property TextAlignment;
      property TransparentMode;
      property VerticalAlignment;
      property Visible;
      property WordWrap;
      property OnClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnMouseEnter;
      property OnMouseLeave;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

   TsCustomLabel = class(TGraphicControl)
   private
      FAlignment: TAlignment;
      FAutoSize: Boolean;
      FButtonLike: Boolean;
      FCtrlState: TsCtrlStates;
      FDown: Boolean;
      FFocusControl: TWinControl;
      FPaint3D: TFont3d;
      FFont3D: TFont3D;
      FHilightStyle: TTextHilightStyle;
      FHilight3d: TFont3d;
      FFocusFrame: Boolean;
      FLayout: TTextLayout;
      FShowAccelChar: Boolean;
      FShowFocus: Boolean;
      FWordWrap: Boolean;
      FOnMouseEnter: TNotifyEvent;
      FOnMouseLeave: TNotifyEvent;
      FOnMouseClick: TNotifyEvent;
      procedure Font3dChanged(Sender: Tobject);
      procedure AdjustBounds;
      procedure DoDrawText(aCanvas: TCanvas; var Rect: TRect; Flags: Word);
      function GetTransparent: Boolean;
      procedure SetAlignment(Value: TAlignment);
      procedure SetFocusControl(Value: TWinControl);
      procedure SetLayout(Value: TTextLayout);
      procedure SetButtonLike(Value: Boolean);
      procedure SetFont3D(Value: TFont3D);
      procedure SetHilight3d(Value: TFont3d);
      procedure SetHilightStyle(Value: TTextHilightStyle);
      procedure SetFocusFrame(Value: Boolean);
      procedure SetShowAccelChar(Value: Boolean);
      procedure SetTransparent(Value: Boolean);
      procedure SetWordWrap(Value: Boolean);
      procedure SetShowFocus(Value: Boolean);
      procedure CheckInvalidate(hilight, mouseEvent: Boolean);
      procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
      procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
      procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure STMEnabledChangeNotify(var Message: TMessage); message STM_ENABLEDCHANGENOTIFY;
      procedure STMMouseEnterNotify(var Message: TMessage); message STM_MOUSEENTERNOTIFY;
      procedure STMMouseLeaveNotify(var Message: TMessage); message STM_MOUSELEAVENOTIFY;
   protected
      procedure DoMouseEnter; virtual;
      procedure DoMouseLeave; virtual;
      procedure DoMouseClick; virtual;
      procedure SetAutoSize(Value: Boolean); virtual;
      function GetLabelText: String; virtual;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure Paint; override;
      property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
      property AutoSize: Boolean read FAutoSize write SetAutoSize default TRUE;
      property ButtonLike: Boolean read FButtonLike write SetButtonLike default FALSE;
      property FocusControl: TWinControl read FFocusControl write SetFocusControl;
      property FocusFrame: Boolean read FFocusFrame write SetFocusFrame default FALSE;
      property Font;
		property Font3D: TFont3D read FFont3D write SetFont3D;
      property HilightStyle: TTextHilightStyle read FHilightStyle write SetHilightStyle default hsFlushing;
      property Hilight3d: TFont3d read FHilight3d write SetHilight3d;
      property Layout: TTextLayout read FLayout write SetLayout default tlTop;
      property ShowAccelChar: Boolean read FShowAccelChar write SetShowAccelChar default True;
      property ShowFocus: Boolean read FShowFocus write SetShowFocus default TRUE;
      property Transparent: Boolean read GetTransparent write SetTransparent default False;
      property WordWrap: Boolean read FWordWrap write SetWordWrap default False;
      property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
      property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
      property OnMouseClick: TNotifyEvent read FOnMouseClick write FOnMouseClick;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Canvas;
	end;

   TsLabel = class(TsCustomLabel)
   published
      property AutoSize;
      property Align;
      property Alignment;
      property ButtonLike;
      property Caption;
      property Color;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Font;
      property Font3D;
      property FocusControl;
      property Hilight3d;
      property HilightStyle;
      property FocusFrame;
      property Layout;
      property ShowAccelChar;
      property ShowFocus;
      property Transparent;
      property WordWrap;
      property ParentColor;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property Visible;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnMouseEnter;
      property OnMouseLeave;
      property OnMouseClick;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnStartDrag;
{$IFDEF VER120}
//TVS: Standard Delphi 4 Properties
   published
      property Anchors;
      property BiDiMode;
      property Constraints;
      property DragKind;
      property ParentBiDiMode;
      property OnEndDock;
      property OnStartDock;
{$ENDIF VER120}
   end;

	TsStatusBar=class(TStatusBar)
	public
	 constructor Create(AOwner: TComponent); override;
   end;

   TsSpeedButton = class(TSpeedButton)
   private
      FCancel: Boolean;
      FDefault: Boolean;
      FGlyphList: TsGlyphList;
      FGlyphId: Integer;
      FKind: TBitBtnKind;
      FModalResult: TModalResult;
      FModifiedGlyph: Boolean;
      procedure CMDialogKey(var Message: TCMDialogKey); message CM_DIALOGKEY;
      procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
      procedure CNCommand(var Message: TWMCommand); message CN_COMMAND;
      procedure SetGlyphList(Value: TsGlyphList);
      procedure SetGlyphId(Value: Integer);
      function GlyphStored: Boolean;
      procedure SetKind(Value: TBitBtnKind);
      function GetKind: TBitBtnKind;
      function IsCustom: Boolean;
      procedure WMGlyphIdChanged(var Message: TMessage); message STM_GLYPHIDCHANGED;
      function GetGlyph: TBitMap;
      procedure SetGlyph(Value: TBitmap);
   protected
      procedure Loaded; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
   public
      constructor Create(AOwner: TComponent); override;
      // TVS: Moved from Protected to Public because it was already Public in Base-Class
      procedure Click; override;
   published
      property Cancel: Boolean read FCancel write FCancel stored IsCustom;
      property Default: Boolean read FDefault write FDefault stored IsCustom;
      property GlyphList: TsGlyphList read FGlyphList write SetGlyphList;
      property GlyphListId: Integer read FGlyphId write SetGlyphId default -1;
      property Glyph: TBitMap read GetGlyph write SetGlyph stored GlyphStored;
      property Kind: TBitBtnKind read GetKind write SetKind default bkCustom;
      property ModalResult: TModalResult read FModalResult write FModalResult stored IsCustom;
   end;

implementation

uses Consts, sFlat, stdUtils;

const
   BoxWidth = 12;

constructor TsBWCCControl.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   Color := clBtnFace;
   FButtonLike := TRUE;
   FFont3d := TFont3d.Create(Font3dChanged);
   FHilight3d := TFont3d.Create(Font3dChanged);
   FPaint3D := TFont3d.Create(nil);
   Width := 98;
   Height := 20;
   ParentColor := FALSE;
   FTransparentMode := tmOpaque;
   FAlignment := taLeftJustify;
   FVerticalAlignment := vaTop;
   FTextAlignment := taLeftJustify;
   FHilightStyle := hsFlushing;
   FFocusFrame := FALSE;
end;

destructor TsBWCCControl.Destroy;
begin
   FHilight3d.Free;
   FFont3d.Free;
   FPaint3D.Free;
   inherited;
end;

procedure TsBWCCControl.CMMouseEnter(var Message: TMessage);
begin
   Include(FCtrlState, scsMouseInControl);
   DoMouseEnter;
   if Enabled and (HilightStyle <> hsNone) and (FHilight3d.Active) and IsFlatApplied and
      not (scsFocused in FCtrlState) then begin
      Include( FCtrlState, scsInvalidateCaption);
      Paint;
   end;
end;

procedure TsBWCCControl.CMMouseLeave(var Message: TMessage);
begin
   Exclude(FCtrlState, scsMouseInControl);
   DoMouseLeave;
   if Enabled and (HilightStyle <> hsNone) and (FHilight3d.Active) and IsFlatApplied and
      not (scsFocused in FCtrlState) then begin
      Include( FCtrlState, scsInvalidateCaption);
      Paint;
   end;
end;

procedure TsBWCCControl.CMEnabledChanged(var Message: TMessage);
begin
   inherited;
   Invalidate;
end;

procedure TsBWCCControl.CMTextChanged(var Message: TMessage);
begin
   Invalidate;
   AdjustBounds;
end;

procedure TsBWCCControl.CMFontChanged(var Message: TMessage);
begin
   inherited;
   AdjustBounds;
end;

procedure TsBWCCControl.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
      if Enabled and IsAccel(CharCode, Caption) then begin
         Toggle;
         Result := 1;
      end else
         inherited;
end;

procedure TsBWCCControl.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
   if Transparent then
      inherited
   else
      Msg.Result := 1;
end;

procedure TsBWCCControl.Paint;
var
   iR, R: TRect;
   bmp: TBitMap;
begin
   if BackgroundChanged {or FDown }or (csDesigning in ComponentState) then
      FCtrlState := FCtrlState - [scsInvalidateCaption, scsInvalidateBox];
   inherited;
   with Canvas do begin
      case FVerticalAlignment of
         vaTop: begin
               R.Top := 1;
               R.Bottom := BoxWidth + 1;
            end;
         vaBottom: begin
               R.Top := Height - BoxWidth - 1;
               R.Bottom := Height - 1;
            end;
         else begin // vaCenter
            R.Top := (Height div 2) - (BoxWidth div 2);
            R.Bottom := R.Top + BoxWidth;
         end;
      end;

      if FAlignment = taRightJustify then begin
         R.Left := Width - 1 - BoxWidth;
         R.Right := Width - 1;
      end else begin
         R.Left := 1;
         R.Right := BoxWidth + 1;
      end;
   end;
   if scsInvalidateCaption in FCtrlState then begin
      iR := Bounds(0, 0, Width - BoxWidth - 4, Height - 2);
      if FAlignment = taLeftJustify then
         OffsetRect( iR, BoxWidth + 4, 0);
   end else if scsInvalidateBox in FCtrlState then begin
      if FAlignment = taLeftJustify then
         iR := Bounds(0, 0, BoxWidth + 4, Height - 2)
      else
         iR := Bounds(Width - BoxWidth - 4, 0, Width, Height - 2);
   end else
      iR := ClientRect;

   bmp := TBitMap.CReate;
   try
      bmp.height := Height;
      bmp.Width := Width;

      if Transparent then
         PaintBackgroundRect(Bmp.Canvas, iR)
      else with Bmp.Canvas do begin
         Brush.Style := bsSolid;
         Brush.Color := Color;
         FillRect(iR);
      end;

      Bmp.Canvas.Font := Font;
      if not (scsInvalidateCaption in FCtrlState) then begin
         PaintBox(Bmp.Canvas, R);
         BitBlt( Canvas.handle, R.Left, R.Top, widthOf(R), heightOf(R),
            bmp.canvas.handle, R.Left, R.Top, srccopy);
      end;
      if not (scsInvalidateBox in FCtrlState) then begin
         PaintCaption(Bmp.Canvas);
         BitBlt( Canvas.handle, iR.Left, iR.Top, widthOf(iR), heightOf(iR),
            bmp.canvas.handle, iR.Left, iR.Top, srccopy);
      end;
      FCtrlState := FCtrlState - [scsInvalidateCaption, scsInvalidateBox];
   finally
      bmp.Free;
   end;
end;

procedure TsBWCCControl.PaintCaption( aCanvas: TCanvas);
const
   Alignments: array [TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
   WordWraps: array [Boolean] of Word = (0, DT_WORDBREAK);
var
   DrawStyle: Integer;
   R: TRect;
   hilightStyle: TTextHilightStyle;
   X: Integer;
begin
   DrawStyle := DT_EXPANDTABS or WordWraps [FWordWrap] or Alignments [FTextAlignment];
   with aCanvas do begin
      Font := Font;

      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         hilightStyle := FHilightStyle
      else
         hilightStyle := hsNone;

      FPaint3D.Assign(FFont3d);
      if not Enabled then begin
         if NewStyleControls then begin
            FPaint3D.Position := spRightBottom;
            FPaint3D.Spacing := 1;
            FPaint3D.Active := TRUE;
         end;
         Font.Color := clGrayText;
         FPaint3D.Color := clBtnHighlight;
         hilightStyle := hsNone;
      end;

      if FAlignment = taLeftJustify then
         X := BoxWidth + 7
      else
         X := 1;
      R := Bounds(X, 1, Width - BoxWidth - 9, Height - 2);

      PaintText( aCanvas, PChar(Caption), Length(Caption), R,
         DrawStyle or DT_CALCRECT, FPaint3D, FHilight3d, hilightStyle);


      R.left := X;
      R.Right := X + Width - BoxWidth - 9;

      if FVerticalAlignment = vaBottom then
         OffsetRect(R, 0, Height - R.Bottom)
      else if FVerticalAlignment = vaCenter then
         OffsetRect(R, 0, (Height - R.Bottom) div 2);

      Brush.Style := bsClear;
      if FDown and FButtonLike then
         OffsetRect(R, 1, 1);
      PaintText( aCanvas, PChar(Caption), Length(Caption), R, DrawStyle,
         FPaint3D, FHilight3d, hilightStyle);

      if (scsFocused in FCtrlState) and FFocusFrame then begin
         PaintText( aCanvas, PChar(Caption), Length(Caption), R, DrawStyle or DT_CALCRECT,
            FPaint3D, FHilight3d, hilightStyle);
         case FTextAlignment of
            taCenter:
               OffsetRect(R, (Width - BoxWidth - 9 - R.Right + R.Left) div 2, 0);
            taRightJustify:
               OffsetRect(R, (Width - BoxWidth - 9 - R.Right + R.Left), 0);
         end;
         InflateRect(R, 1, 0);
         Brush.Color := clBtnFace;
         FrameRect(R);
         DrawFocusRect(R);
      end;
   end;
end;

procedure TsBWCCControl.AdjustBounds;
const
   WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
   DC: HDC;
   dX, X: Integer;
   Rect: TRect;
begin
   if FAutoSize then begin
      Rect := ClientRect;

      dX := 1;
      if hilightStyle = hsFlushing then
         dX := FHilight3D.Spacing;

      InflateRect(Rect, -dX, 0);
      DC := GetDC(0);
      Canvas.Handle := DC;

      PaintText( Canvas, PChar(Caption), Length(Caption), Rect, (DT_EXPANDTABS or DT_CALCRECT) or WordWraps[FWordWrap],
         FFont3d, FHilight3D, hsNone);//hilightStyle);

      Canvas.Handle := 0;
      ReleaseDC(0, DC);

      Inc(Rect.Right, BoxWidth + 6);
      Inc( Rect.Bottom, 2);

      X := Left;
      if FAlignment = taRightJustify then
         Inc(X, Width - Rect.Right - 1);
      InflateRect(Rect, dX, 0);

      if FButtonLike then begin
         Inc( Rect.Right, 1);
         Inc( Rect.Bottom, 1);
      end;

      SetBounds(X, Top, Rect.Right, Rect.Bottom);
   end;
end;

procedure TsBWCCControl.SetTransparentMode(Value: TTransparentMode);
begin
   if FTransparentMode <> Value then begin
      FTransparentMode := Value;
      Transparent := Value <> tmOpaque;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetAlignment(Value: TLeftRight);
begin
   if Value <> FAlignment then begin
      FAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetAutoSize(Value: Boolean);
begin
   if FAutoSize <> Value then begin
      FAutoSize := Value;
      AdjustBounds;
   end;
end;

procedure TsBWCCControl.SetButtonLike(Value: Boolean);
begin
   if FButtonLike <> Value then begin
      FButtonLike := Value;
      if FAutoSize then
         AdjustBounds;
   end;
end;

procedure TsBWCCControl.SetTextAlignment(Value: TAlignment);
begin
   if FTextAlignment <> Value then begin
      FTextAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetVerticalAlignment(Value: TVerticalAlignment);
begin
   if FVerticalAlignment <> Value then begin
      FVerticalAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetWordWrap(Value: Boolean);
begin
   if FWordWrap <> Value then begin
      FWordWrap := Value;
      AdjustBounds;
      Include(FCtrlState, scsInvalidateCaption);
      Invalidate;
   end;
end;

function TsBWCCControl.GetCaption: TCaption;
var
   Buf: array [0..256] of Char;
begin
   GetTextBuf(Buf, 256);
   Result := StrPas(Buf);
end;

procedure TsBWCCControl.SetCaption(const Value: TCaption);
var
   Buffer: array [0..255] of Char;
begin
   if GetCaption <> Value then
      SetTextBuf(StrPCopy(Buffer, Value));
   Include(FCtrlState, scsInvalidateCaption);
   Invalidate;
end;

procedure TsBWCCControl.SetFont3d(Value: TFont3d);
begin
   if FFont3d <> Value then
      FFont3d.Assign(Value);
end;

procedure TsBWCCControl.Font3dChanged(Sender: Tobject);
begin
   AdjustBounds;
   Invalidate;
end;

procedure TsBWCCControl.SetHilight3d(Value: TFont3d);
begin
   if FHilight3d <> Value then
      FHilight3d.Assign( Value);
end;

procedure TsBWCCControl.SetHilightStyle(Value: TTextHilightStyle);
begin
   if Value <> FHilightStyle then begin
      FHilightStyle := Value;
      if FHilightStyle = hsFlushing then
         FFocusFrame := FALSE;
      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         Invalidate;
   end;
end;

procedure TsBWCCControl.SetFocusFrame(Value: Boolean);
begin
   if Value <> FFocusFrame then begin
      FFocusFrame := Value;
      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         Invalidate;
   end;
end;

function TsBWCCControl.GetChecked: Boolean;
begin
   Result := FState = cbChecked;
end;

function TsBWCCControl.EditCanModify: Boolean;
begin
   Result := TRUE;
end;

procedure TsBWCCControl.SetChecked(Value: Boolean);
const
   checkStates: array [Boolean] of TCheckBoxState = (cbUnchecked, cbChecked);
begin
   if EditCanModify and (Value <> Checked) then begin
      FState := checkStates [Value];
      if not FButtonLike then
         Include( FCtrlState, scsInvalidateBox);
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetCheckColor(Value: TColor);
begin
   FCheckColor := Value;
   Invalidate;
end;

procedure TsBWCCControl.DoEnter;
begin
   inherited DoEnter;
   Include( FCtrlState, scsFocused);
   Invalidate;
end;

procedure TsBWCCControl.DoExit;
begin
   inherited DoExit;
   Exclude( FCtrlState, scsFocused);
   Invalidate;
end;

procedure TsBWCCControl.DoMouseEnter;
begin
	if Assigned( FOnMouseEnter) then
      FOnMouseEnter(self);
end;

procedure TsBWCCControl.DoMouseLeave;
begin
	if Assigned( FOnMouseLeave) then
      FOnMouseLeave(self);
end;

procedure TsBWCCControl.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      SetFocus;
      inherited;
      MouseCapture := True;
      FDown := TRUE;
      if not FButtonLike then
         Include( FCtrlState, scsInvalidateBox);
      Invalidate;
   end else
      inherited;
end;

procedure TsBWCCControl.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      MouseCapture := False;
      FDown := FALSE;
      if PtInRect(ClientRect, Point(X, Y)) then
         Toggle;
   end;
   inherited;
end;

procedure TsBWCCControl.MouseMove(Shift: TShiftState; X, Y: Integer);
var
   oldDown: Boolean;
begin
   if MouseCapture then begin
      oldDown := FDown;
      FDown := (X >= 0) and (X <= Width) and (Y >= 0) and (Y <= Height);
      if oldDown <> FDown then begin
         if not FButtonLike then
            Include( FCtrlState, scsInvalidateBox);
         Invalidate;
      end;
   end;
   inherited;
end;

procedure TsBWCCControl.KeyDown(var Key: Word; Shift: TShiftSTate);
begin
   if (Key = vk_Space) and not FDown then begin
      FDown := TRUE;
      if not FButtonLike then
         Include( FCtrlState, scsInvalidateBox);
      Invalidate;
   end;
   inherited KeyDown(Key, Shift);
end;

procedure TsBWCCControl.KeyUp(var Key: Word; Shift: TShiftSTate);
begin
   if Key = vk_Space then begin
      FDown := False;
      Toggle;
   end;
end;

{*******************************************************************************
   TsCheckBox
*******************************************************************************}

procedure TsCustomCheckBox.SetStyle( Value: TsBWCCBoxStyle);
begin
   if FStyle <> Value then begin
      FStyle := Value;
      Invalidate;
   end;
end;

procedure TsCustomCheckBox.SetState(Value: TCheckBoxState);
begin
   if EditCanModify and (FState <> Value) then begin
      FState := Value;
      if not FButtonLike then
         Include( FCtrlState, scsInvalidateBox);
      Invalidate;
   end;
end;

procedure TsCustomCheckBox.Toggle;
begin
   if EditCanModify then begin
      if FAllowGrayed and (FState = cbUnchecked) then
         State := cbGrayed
      else
         Checked := not Checked;
      Click;
   end else begin
      if not FButtonLike then
         Include( FCtrlState, scsInvalidateBox);
      Invalidate;
   end;
end;

const
   ColorMap: array[1..2, Boolean] of TColor =
      ((clBtnHighLight, clBtnShadow), (clBtnShadow, clBtnHighLight));

procedure TsCustomCheckBox.PaintBox(aCanvas: TCanvas; R: TRect);
begin
   with aCanvas do begin
      Brush.Color := Color;
      if transparentMode = tmSemiTransparent then
         FillRect(R);
      if FDown then begin
         Pen.Color := clBlack;
         Pen.Width := 2;
         Rectangle( R.Left + 1, R.Top + 1, R.Right + 1, R.Bottom + 1);
         Pen.Width := 1;
      end else begin
         Brush.Color  := Color;
         FrameRect(R);
         Pen.Color := ColorMap[1, Boolean(FStyle)];
         PolyLine([Point(R.Left, R.Bottom), R.TopLeft, Point(R.Right, R.Top)]);
         Pen.Color := ColorMap[2, Boolean(FStyle)];
         PolyLine([Point(R.Right, R.Top), R.BottomRight, Point(R.Left, R.Bottom)]);
      end;

      case State of
         cbChecked: begin
               if FStyle = bwsSunken then
                  OffsetRect(R, 0, -1);

               Pen.Color := FCheckColor;
               Pen.Width := 1;
               Dec(R.Top);
               Dec(R.Bottom);
               Polyline([Point(R.Left + 2, R.Top + BoxWidth div 2 + 1), Point(R.Left + 2, R.Bottom - 1)]);
               Polyline([Point(R.Left + 3, R.Top + BoxWidth div 2), Point(R.Left + 3, R.Bottom - 2)]);
               Polyline([Point(R.Left + 2, R.Bottom - 1), Point(R.Right - 2, R.Top + 3)]);
               Polyline([Point(R.Left + 3, R.Bottom - 1), Point(R.Right - 1, R.Top + 3)]);
            end;
         cbGrayed: begin
               OffsetRect( R, 2, 2);
               Dec(R.Right, 3);
               Dec(R.Bottom, 3);
               if FDown then begin
                  Pen.Color := clBtnFace;
                  FrameRect(R);
               end;
               Pen.Color := ColorMap[1, Boolean(FStyle)];
               Brush.Color := FCheckColor;
               Rectangle(R.Left, R.Top, R.Right, R.Bottom);
         end;
      end;
   end;
end;

{*******************************************************************************
   TsRadioButton
*******************************************************************************}

procedure TsCustomRadioButton.Toggle;
begin
   if EditCanModify then begin
      if not Checked then begin
         Checked := TRUE;
         Click;
      end else begin
         if not FButtonLike then
            Include( FCtrlState, scsInvalidateBox);
         Invalidate;
      end;
   end;
end;

procedure TsCustomRadioButton.SetChecked(Value: Boolean);
var
   ii: Integer;
   ctrl: TsCustomRadioButton;
begin
   inherited;
   if Value and (Parent <> nil) then for ii := 0 to Parent.ControlCount - 1 do
      if Parent.Controls[ii] is TsCustomRadioButton then begin
         ctrl := TsCustomRadioButton(Parent.Controls[ii]);
         if( ctrl <> Self) and ( ctrl.FGroupIndex = FGroupIndex) then
            Ctrl.SetChecked(FALSE);
      end;
end;

procedure TsCustomRadioButton.PaintBox(aCanvas: TCanvas; R: TRect);
var
   HalfBoxWidth: Integer;
begin
   HalfBoxWidth := BoxWidth div 2;
   with aCanvas do begin
      if FDown then begin
         Pen.Color := clBlack;
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top)]);
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top + 1), Point(R.Left + 1, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom - 1), Point(R.Right - 1, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top + 1)]);
      end else begin
         Pen.Color := ColorMap[1, Checked];
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom)]);
         Pen.Color := ColorMap[2, Checked];
         PolyLine([Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top)]);
         if FTransparentMode = tmSemiTransparent then begin
            Pen.Color := Color;
            Brush.Color := Color;
            InflateRect(R, -1, -1);
            dec(HalfBoxWidth);
            Polygon([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
                     Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
                     Point(R.Left + HalfBoxWidth, R.Top)]);
            InflateRect(R, 1, 1);
            Inc(HalfBoxWidth);
         end;
      end;
      if Checked then begin
         Pen.Color := FCheckColor;
         Inc(R.Left, HalfBoxWidth);
         Inc(R.Top, HalfBoxWidth);

         PolyLine([Point(R.Left - 1, R.Top - 1), Point(R.Left + 2, R.Top - 1)]);
         PolyLine([Point(R.Left - 2, R.Top), Point(R.Left + 3, R.Top)]);
         PolyLine([Point(R.Left - 1, R.Top + 1), Point(R.Left + 2, R.Top + 1)]);
         PolyLine([Point(R.Left, R.Top - 2), Point(R.Left, R.Top + 3)]);
      end;
   end;
end;

{*******************************************************************************
   TsLabel
*******************************************************************************}
constructor TsCustomLabel.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csOpaque];
   FFont3d := TFont3d.Create(Font3dChanged);
   FHilight3d := TFont3d.Create(Font3dChanged);
   FPaint3D := TFont3d.Create(nil);
   Width := 65;
   Height := 17;
   FAutoSize := TRUE;
   FShowAccelChar := TRUE;
   FShowFocus := TRUE;
   FHilightStyle := hsFlushing;
   FFocusFrame := FALSE;
end;

destructor TsCustomLabel.Destroy;
begin
   FFont3d.Free;
   FHilight3d.Free;
   FPaint3D.Free;
   inherited;
end;

function TsCustomLabel.GetLabelText: String;
begin
   Result := Caption;
end;

procedure TsCustomLabel.DoDrawText(aCanvas: TCanvas; var Rect: TRect; Flags: Word);
var
   Text: string;
   hilightStyle: TTextHilightStyle;
begin
   Text := GetLabelText;
   if (Flags and DT_CALCRECT <> 0) and ((Text = '') or FShowAccelChar and
      (Text[1] = '&') and (Text[2] = #0)) then
      Text := Text + ' ';
   if not FShowAccelChar then
      Flags := Flags or DT_NOPREFIX;
   aCanvas.Font := Font;

   if ([scsMouseInControl, scsFocused] * FCtrlState <> []) or
      (FShowFocus and Assigned(FFocusControl) and FFocusControl.Focused and
      not (csDesigning in ComponentState)) then
      hilightStyle := FHilightStyle
   else
      hilightStyle := hsNone;

   FPaint3D.Assign(FFont3D);
   if not Enabled then begin
      if NewStyleControls then begin
         FPaint3D.Position := spRightBottom;
         FPaint3D.Spacing := 1;
         FPaint3D.Active := TRUE;
      end;
      aCanvas.Font.Color := clGrayText;
      FPaint3D.Color := clBtnHighlight;
      hilightStyle := hsNone;
   end;
   PaintText( aCanvas, PChar(Text), Length(Text), Rect, Flags,
      FPaint3d, FHilight3d, hilightStyle);
end;

procedure TsCustomLabel.Paint;
const
   Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
   WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
   Rect: TRect;
   DrawStyle: Integer;
   Bmp: TBitMap;
   H, W: Integer;
begin
   Bmp := TBitmap.Create;
   W := Width;
   H := Height;
   if FDown then begin
      Inc(W, 2);
      Inc(H, 2);
   end;
   BMP.Width := W;
   BMP.Height := H;

   with BMP.Canvas do try
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
      Brush.Style := bsClear;
      Rect := ClientRect;
      InflateRect(Rect, -2, -1);
      DrawStyle := DT_EXPANDTABS or WordWraps[FWordWrap] or Alignments[FAlignment];
      { Calculate vertical layout }
      if FLayout <> tlTop then begin
         DoDrawText(BMP.Canvas, Rect, DrawStyle or DT_CALCRECT);
         Rect.Left := ClientRect.Left;
         Rect.Right := ClientRect.Right;
         if FLayout = tlBottom then
            OffsetRect(Rect, 0, Height - Rect.Bottom)
         else
            OffsetRect(Rect, 0, (Height - Rect.Bottom) div 2);
      end;
      if FDown then
         OffsetRect(Rect, 1, 1);

      DoDrawText( BMP.Canvas, Rect, DrawStyle);
      if FShowFocus and Assigned(FFocusControl) and FFocusControl.Focused and
         FFocusFrame and not (csDesigning in ComponentState) then begin
         InflateRect(Rect, 1, 0);
         Brush.Color := clBtnFace;
         FrameRect(Rect);
         DrawFocusRect(Rect);
      end;
      if Transparent then
         DrawBitmapTransparent( Canvas, 0, 0, Bmp, Self.Color)
      else
         BitBlt( Canvas.handle, 0, 0, width, height, bmp.canvas.handle, 0, 0, srccopy);
   finally
      BMP.Free;
   end;
end;

procedure TsCustomLabel.AdjustBounds;
const
   WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
   DC: HDC;
   dX, X: Integer;
   Rect: TRect;
begin
   if FAutoSize then begin
      Rect := ClientRect;

      dX := 1;
      if hilightStyle = hsFlushing then
         dX := FHilight3D.Spacing;

      InflateRect(Rect, -dX, 0);
      DC := GetDC(0);
      Canvas.Handle := DC;
      DoDrawText( Canvas, Rect, (DT_EXPANDTABS or DT_CALCRECT) or WordWraps[FWordWrap]);
      Canvas.Handle := 0;
      ReleaseDC(0, DC);
      X := Left;
      if FAlignment = taRightJustify then
         Inc(X, Width - Rect.Right);
      InflateRect(Rect, dX, 0);
      Inc( Rect.Right, 1);
      if FButtonLike then begin
         Inc( Rect.Right, 1);
         Inc( Rect.Bottom, 2);
      end;
      SetBounds(X, Top, Rect.Right, Rect.Bottom);
   end;
end;

procedure TsCustomLabel.SetAlignment(Value: TAlignment);
begin
   if FAlignment <> Value then begin
      FAlignment := Value;
      Invalidate;
   end;
end;

procedure TsCustomLabel.SetAutoSize(Value: Boolean);
begin
   if FAutoSize <> Value then begin
      FAutoSize := Value;
      AdjustBounds;
   end;
end;

procedure TsCustomLabel.SetLayout(Value: TTextLayout);
begin
   if FLayout <> Value then begin
      FLayout := Value;
      Invalidate;
   end;
end;

procedure TsCustomLabel.SetButtonLike(Value: Boolean);
begin
   if FButtonLike <> Value then begin
      FButtonLike := Value;
      if FAutoSize then
         AdjustBounds;
   end;
end;

procedure TsCustomLabel.Font3dChanged(Sender: Tobject);
begin
   AdjustBounds;
   Invalidate;
end;

procedure TsCustomLabel.SetFont3D(Value: TFont3D);
begin
   if FFont3d <> Value then
      FFont3d.Assign(Value);
end;

procedure TsCustomLabel.SetHilight3d(Value: TFont3d);
begin
   if FHilight3d <> Value then
      FHilight3d.Assign( Value);
end;

procedure TsCustomLabel.SetHilightStyle(Value: TTextHilightStyle);
begin
   if Value <> FHilightStyle then begin
      FHilightStyle := Value;
      if FHilightStyle = hsFlushing then
         FFocusFrame := FALSE;
      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         Invalidate;
   end;
end;

procedure TsCustomLabel.SetFocusFrame(Value: Boolean);
begin
   if Value <> FFocusFrame then begin
      FFocusFrame := Value;
      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         Invalidate;
   end;
end;

function TsCustomLabel.GetTransparent: Boolean;
begin
   Result := not (csOpaque in ControlStyle);
end;

procedure TsCustomLabel.SetFocusControl(Value: TWinControl);
begin
   FFocusControl := Value;
   if Value <> nil then begin
      Value.FreeNotification(Self);
      Enabled := FFocusControl.Enabled;
   end;
   if FShowFocus then
      Invalidate;
end;

procedure TsCustomLabel.SetShowAccelChar(Value: Boolean);
begin
   if FShowAccelChar <> Value then begin
      FShowAccelChar := Value;
      Invalidate;
   end;
end;

procedure TsCustomLabel.SetTransparent(Value: Boolean);
begin
   if Transparent <> Value then begin
      if Value then
         ControlStyle := ControlStyle - [csOpaque]
      else
         ControlStyle := ControlStyle + [csOpaque];
      Invalidate;
   end;
end;

procedure TsCustomLabel.SetShowFocus(Value: Boolean);
begin
   if FShowFocus <> Value then begin
      FShowFocus := Value;
      Invalidate;
   end;
end;

procedure TsCustomLabel.SetWordWrap(Value: Boolean);
begin
   if FWordWrap <> Value then begin
      FWordWrap := Value;
      AdjustBounds;
   end;
end;

procedure TsCustomLabel.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and (AComponent = FFocusControl) then
      FocusControl := nil;
end;

procedure TsCustomLabel.DoMouseEnter;
begin
	if Assigned( FOnMouseEnter) then
      FOnMouseEnter(self);
end;

procedure TsCustomLabel.DoMouseLeave;
begin
	if Assigned( FOnMouseLeave) then
      FOnMouseLeave(self);
end;

procedure TsCustomLabel.DoMouseClick;
begin
	if Assigned( FOnMouseClick) then
      FOnMouseClick(self);
   if FFocusControl <> nil then
      with FFocusControl do if CanFocus then
         SetFocus;
end;

procedure TsCustomLabel.CMFocusChanged(var Message: TCMFocusChanged);
var
   Active: Boolean;
begin
   Active := Assigned(FFocusControl) and (Message.Sender = FFocusControl);
   if (scsFocused in FCtrlState) <> Active then begin
      if FShowFocus then
         CheckInvalidate(Active, FALSE);
   end;
   inherited;
end;

procedure TsCustomLabel.CMTextChanged(var Message: TMessage);
begin
   Invalidate;
   AdjustBounds;
end;

procedure TsCustomLabel.CMFontChanged(var Message: TMessage);
begin
   inherited;
   AdjustBounds;
end;

procedure TsCustomLabel.CMDialogChar(var Message: TCMDialogChar);
begin
   if (FFocusControl <> nil) and Enabled and ShowAccelChar and
      IsAccel(Message.CharCode, Caption) then with FFocusControl do
      if CanFocus then begin
         SetFocus;
         Message.Result := 1;
      end;
end;

procedure TsCustomLabel.CMMouseEnter(var Message: TMessage);
begin
   CheckInvalidate(TRUE, TRUE);
   if (FFocusControl <> nil) and Enabled then
      FFocusControl.Perform(CM_MouseEnter, 0, 0);
   DoMouseEnter;
end;

procedure TsCustomLabel.CMMouseLeave(var Message: TMessage);
begin
   CheckInvalidate(FALSE, TRUE);
   if (FFocusControl <> nil) and Enabled then
      FFocusControl.Perform(CM_MouseLeave, 0, 0);
   DoMouseLeave;
end;

procedure TsCustomLabel.STMEnabledChangeNotify(var Message: TMessage);
begin
   if (FFocusControl <> nil) and (Message.Result = 0) and
      (Message.WParam = LongInt(FFocusControl)) then begin
      Enabled := Boolean(Message.LParam);
      Message.Result := 1;
   end;
end;

procedure TsCustomLabel.STMMouseEnterNotify(var Message: TMessage);
begin
   if (FFocusControl <> nil) and (Message.Result = 0) and
      (Message.WParam = LongInt(FFocusControl)) then begin
      CheckInvalidate(TRUE, TRUE);
      Message.Result := 1;
   end;
end;

procedure TsCustomLabel.STMMouseLeaveNotify(var Message: TMessage);
begin
   if (FFocusControl <> nil) and (Message.Result = 0) and
      (Message.WParam = LongInt(FFocusControl)) then begin
      CheckInvalidate(FALSE, TRUE);
      Message.Result := 1;
   end;
end;

procedure TsCustomLabel.CheckInvalidate(hilight, mouseEvent: Boolean);
var
   needRefresh: Boolean;
begin
   needRefresh := ((not (scsMouseInControl in FCtrlState) and not MouseEvent) or
                  (not (scsFocused in FCtrlState) and MouseEvent) or
                  (FFocusFrame and  not (scsFocused in FCtrlState) and not mouseEvent)) and
                  Enabled and (HilightStyle <> hsNone) and (FHilight3d.Active) and
                  IsFlatApplied;
   if mouseEvent then begin
      if hilight then
         Include(FCtrlState, scsMouseInControl)
      else
         Exclude(FCtrlState, scsMouseInControl);
   end else begin
      if hilight then
         Include(FCtrlState, scsFocused)
      else
         Exclude(FCtrlState, scsFocused);
   end;
   if needRefresh then
      Refresh;//Invalidate;
end;

procedure TsCustomLabel.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Enabled and FButtonLike and (Button = mbLeft) and
      (FFocusControl <> nil) and (FFocusControl.CanFocus) then begin
      inherited;
      MouseCapture := True;
      FDown := TRUE;
      Invalidate;
   end else
      inherited;
end;

procedure TsCustomLabel.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Enabled and FButtonLike and (Button = mbLeft) and
      (FFocusControl <> nil) and (FFocusControl.CanFocus) then begin
      MouseCapture := False;
      if PtInRect(ClientRect, Point(X, Y)) then
         DoMouseClick;
      FDown := FALSE;
      Invalidate;
   end;
   inherited;
end;

procedure TsCustomLabel.MouseMove(Shift: TShiftState; X, Y: Integer);
var
   oldDown: Boolean;
begin
   if MouseCapture then begin
      oldDown := FDown;
      FDown := (X >= 0) and (X <= Width) and (Y >= 0) and (Y <= Height);
      if oldDown <> FDown then
         Invalidate;
   end;
   inherited;
end;

{*******************************************************************************
   TsSpeedButton
*******************************************************************************}

constructor TsSpeedButton.Create(AOwner: TComponent);
begin
   inherited;
   FGlyphList := nil;
   FGlyphId := -1;
   FModifiedGlyph := FALSE;
end;

procedure TsSpeedButton.SetGlyphList(Value: TsGlyphList);
begin
	if FGlyphList <> Value then begin
      if FGlyphList <> nil then
			FGlyphList.ChangeNotification(self, FALSE);
   	FGlyphList := Value;
      if FGlyphList <> nil then begin
			FGlyphList.ChangeNotification(self, TRUE);
         inherited Glyph := TBitMap(FGlyphList[FGlyphId]);
      end else
         inherited Glyph := nil;
   end;
end;

procedure TsSpeedButton.SetGlyphId(Value: Integer);
begin
	if FGlyphId <> Value then begin
		if csLoading in ComponentState then
   		FGlyphId := Value
		else begin
   		if FGlyphList = nil then
   			Exit;
      	if (Value > -1) then begin
            if FGlyphList[Value] = nil then
               Raise Exception.CreateFmt(SErrorGlyphNotFound, [Value]);
            if not FGlyphList[Value].InheritsFrom(TBitMap) then
      		   Raise Exception.Create(SErrorNeedsBmp);
         end;
   		FGlyphId := Value;
         inherited Glyph := TBitMap(FGlyphList[FGlyphId]);
      end;
   end;
end;

procedure TsSpeedButton.Loaded;
begin
   inherited;
   if FGlyphList <> nil then
      inherited Glyph := TBitMap(FGlyphList[FGlyphId])
end;

procedure TsSpeedButton.WMGlyphIdChanged(var Message: TMessage);
begin
	if Message.WParam = FGlyphId then begin
      if Message.lParam = Message.wParam then
         inherited Glyph := TBitMap(FGlyphList[FGlyphId])
      else
			GlyphListId := Message.LParam;
      FModifiedGlyph := FALSE;
   end;
end;

procedure TsSpeedButton.Notification(AComponent: TComponent; Operation: TOperation);
begin
	inherited;
   if (AComponent = FGlyphList) and (Operation = opRemove) then begin
   	FGlyphList := nil;
      inherited Glyph := nil;
   end;
end;

function TsSpeedButton.GetGlyph: TBitMap;
begin
	Result := nil;
   if (FGlyphList <> nil) and (FGlyphId > -1) then
   	Result := TBitMap(FGlyphList[FGlyphId]);
   if Result = nil then
      Result := inherited Glyph;
end;

procedure TsSpeedButton.SetGlyph(Value: TBitmap);
begin
   inherited Glyph := Value;
   FModifiedGlyph := TRUE;
end;

function TsSpeedButton.GlyphStored: Boolean;
begin
   Result := (Kind = bkCustom) and ((GlyphList = nil) or (GlyphListId = -1));
end;

{ TBitBtn data }
var
   BitBtnResNames: array[TBitBtnKind] of PChar = (
      nil, 'BBOK', 'BBCANCEL', 'BBHELP', 'BBYES', 'BBNO', 'BBCLOSE',
      'BBABORT', 'BBRETRY', 'BBIGNORE', 'BBALL');
   BitBtnModalResults: array[TBitBtnKind] of TModalResult = (
      0, mrOk, mrCancel, 0, mrYes, mrNo, 0, mrAbort, mrRetry, mrIgnore, mrAll);

var
  BitBtnGlyphs: array[TBitBtnKind] of TBitmap;

function GetBitBtnGlyph(Kind: TBitBtnKind): TBitmap;
begin
   if BitBtnGlyphs[Kind] = nil then begin
      BitBtnGlyphs[Kind] := TBitmap.Create;
      BitBtnGlyphs[Kind].LoadFromResourceName(HInstance, BitBtnResNames[Kind]);
   end;
   Result := BitBtnGlyphs[Kind];
end;

procedure TsSpeedButton.SetKind(Value: TBitBtnKind);
begin
   if Value <> FKind then begin
      if Value <> bkCustom then begin
         Default := Value in [bkOK, bkYes];
         Cancel := Value in [bkCancel, bkNo];
         ModalResult := BitBtnModalResults[Value];
         inherited Glyph := GetBitBtnGlyph(Value);
         NumGlyphs := 2;
         FModifiedGlyph := FALSE;
      end;
      FKind := Value;
      Invalidate;
   end;
end;

function TsSpeedButton.GetKind: TBitBtnKind;
begin
   if FKind <> bkCustom then
      if ((FKind in [bkOK, bkYes]) xor Default) or
         ((FKind in [bkCancel, bkNo]) xor Cancel) or
         (ModalResult <> BitBtnModalResults[FKind]) or
         FModifiedGlyph then
         FKind := bkCustom;
   Result := FKind;
end;

function TsSpeedButton.IsCustom: Boolean;
begin
   Result := Kind = bkCustom;
end;

procedure TsSpeedButton.Click;
var
   Form: TCustomForm;
begin
   Form := GetParentForm(Self);
   if Form <> nil then
      Form.ModalResult := ModalResult;
   inherited Click;
end;

procedure TsSpeedButton.CNCommand(var Message: TWMCommand);
begin
   if Message.NotifyCode = BN_CLICKED then
      Click;
end;

procedure TsSpeedButton.CMDialogKey(var Message: TCMDialogKey);
begin
   with Message do
      if Enabled and (((CharCode = VK_RETURN) and Default) or
         ((CharCode = VK_ESCAPE) and FCancel)) and
         (KeyDataToShiftState(Message.KeyData) = []) then begin
         Click;
         Result := 1;
      end else
         inherited;
end;

procedure TsSpeedButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
      if Enabled and IsAccel(CharCode, Caption) then begin
         Click;
         Result := 1;
      end else
         inherited;
end;

{*******************************************************************************
   TsStatusBar
*******************************************************************************}

constructor TsStatusBar.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
   ControlStyle := [csAcceptsControls, csCaptureMouse, csClickEvents,
      csSetCaption, csOpaque, csDoubleClicks, csReplicatable];
end;


procedure DestroyLocals; far;
var
   ii: TBitBtnKind;
begin
   for ii := Low(TBitBtnKind) to High(TBitBtnKind) do
      BitBtnGlyphs[ii].Free;
end;

initialization
   FillChar(BitBtnGlyphs, SizeOf(BitBtnGlyphs), 0);
   RegisterFlatControl(TsBWCCControl);

finalization
   DestroyLocals;
end.
