{*******************************************************************************
   Unit
      TiledBmp.pas
   Description:
      TTiledBmp - the very stripped out revrite of TImage with ability to tile bitmap
   Versions:
   	1.1
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com.*
   Comments:
	History:
      1.1	-	18/11/1998
               TsGlyphList support
      1.0 	- 	End of September 1998
      			Initial release
*******************************************************************************}
unit TiledBmp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  sGlyphsList, sConsts;

type
	TWallpaperMode = (wpTile, wpCenter, wpTopLeft);

  	TTiledBmp = class(TGraphicControl)
  	private
    	FImage: TBitmap;
    	FMode: TWallpaperMode;
      FGlyphList: TsGlyphList;
      FGlyphId: Integer;
    	procedure PictureChanged(Sender: TObject);
      procedure SetGlyphList(Value: TsGlyphList);
      procedure SetGlyphId(Value: Integer);
      procedure SetImage(Value: TBitMap);
      function IsGlyphStored: Boolean;
      procedure SetMode(value: TWallpaperMode);
      procedure STMGlyphIdChanged(var Message: TMessage); message STM_GLYPHIDCHANGED;
	protected
    	procedure Paint; override;
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  	public
   	constructor Create(AOwner: TComponent); override;
    	destructor Destroy; override;
   published
    	property Align default alClient;
      property GlyphList: TsGlyphList read FGlyphList write SetGlyphList;
      property GlyphListId: Integer read FGlyphId write SetGlyphId default -1;
    	property Mode: TWallpaperMode read FMode write SetMode;
    	property Picture: TBitMap read FImage write SetImage stored IsGlyphStored;
  	end;

procedure Register;

implementation

constructor TTiledBmp.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle + [csReplicatable];
   FGlyphList := nil;
   FGlyphId := -1;
   FImage := TBitmap.Create;
   FImage.OnChange := PictureChanged;
   Align := alClient;
end;

destructor TTiledBmp.Destroy;
begin
  	FImage.Free;
  	inherited;
end;

procedure TileBitmap(Bitmap: TBitmap; Canvas: TCanvas; Width, Height: Integer);
var
	ii, jj: Integer;
begin
  	for ii := 0 to (Width-1) div Bitmap.Width do
  		for jj := 0 to (Height-1) div Bitmap.Height do
    		Canvas.Draw(ii*Bitmap.Width, jj*Bitmap.Height, Bitmap);
end;

procedure TTiledBmp.Paint;
var
	Dest: TRect;
begin
   if not FImage.Empty then begin
 	  if FMode = wpTile then
   		TileBitmap( FImage, inherited Canvas, Width, Height)
   	else begin
   		if FMode = wpCenter then
   			Dest := Bounds((Width - FImage.Width) div 2, (Height - FImage.Height) div 2,
      			FImage.Width, FImage.Height)
      	else
				Dest := Rect(0, 0, FImage.Width, FImage.Height);
      	with inherited Canvas do
    			StretchDraw(Dest, FImage);
   	end;
   end;
   if csDesigning in ComponentState then with inherited Canvas do begin
   	Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
   end;
end;

procedure TTiledBmp.Notification(AComponent: TComponent; Operation: TOperation);
begin
	inherited;
   if (AComponent = FGlyphList) and (Operation = opRemove) then begin
   	FGlyphList := nil;
      //FGlyphId := -1;
      Invalidate;
   end;
end;

procedure TTiledBmp.STMGlyphIdChanged(var Message: TMessage);
begin
	if Message.WParam = FGlyphId then begin
      if Message.lParam = Message.wParam then
			PictureChanged(self)
      else
			GlyphListId := Message.LParam;
   end;
end;

procedure TTiledBmp.SetImage(Value: TBitmap);
begin
  	FImage.Assign(Value);
   FGlyphId := -1;
  	Invalidate;
end;

function TTiledBmp.IsGlyphStored: Boolean;
begin
   Result := FGlyphId = -1;
end;

procedure TTiledBmp.SetGlyphList(Value: TsGlyphList);
begin
	if FGlyphList <> Value then begin
      if FGlyphList <> nil then
			FGlyphList.ChangeNotification(self, FALSE);
   	FGlyphList := Value;
      if FGlyphList <> nil then begin
			FGlyphList.ChangeNotification(self, TRUE);
         FImage.Assign(FGlyphList[FGlyphId]);
      end else
         FImage.Assign(nil);
   end;
end;

procedure TTiledBmp.SetGlyphId(Value: Integer);
begin
	if FGlyphId <> Value then begin
		if csLoading in ComponentState then
   		FGlyphId := Value
		else begin
   		if FGlyphList = nil then
   			Exit;
      	if (Value > -1) then begin
            if FGlyphList[Value] = nil then
               Raise Exception.CreateFmt(SErrorGlyphNotFound, [Value]);
            if not FGlyphList[Value].InheritsFrom(TBitMap) then
      		   Raise Exception.Create(SErrorNeedsBmp);
         end;
   		FGlyphId := Value;
      	FImage.Assign(FGlyphList[FGlyphId]);
      end;
   end;
end;

procedure TTiledBmp.PictureChanged( Sender: TObject);
begin
   if ((FMode = wpTile) and not FImage.Empty)  or ((FImage.width >= Width) and (FImage.Height >= Height)) then
      ControlStyle := ControlStyle + [csOpaque]
   else
      ControlStyle := ControlStyle - [csOpaque];
  	Invalidate;
end;

procedure TTiledBmp.SetMode(value: TWallpaperMode);
begin
	if Value <> FMode then begin
   	FMode := Value;
      Invalidate;
   end;
end;


procedure Register;
begin
   RegisterComponents(SEditToolsPageCaption, [TTiledBmp]);
end;

end.
