{*******************************************************************************
   Unit
      sTrans.pas
   Description:
      TCustomControl which provide the transparency mechanizme for its child controls
   Versions:
   	1.0a
	History:
		1.0a  - 	01/11/1998.
      			.Small fix suggested by Jim Reid.
					.DrawFlatEDGE is rewritten to fix a problem with painting of EDGE
                when control is non-transparent.
      			.Implemented cashing of background (to improve paint speed)
            	.RefreshTransControls method to manualy refresh transparent
                controls after background was changed.
      1.0 	- 	End of September 1998
      			Initial release
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com.
   Comments:
      TMDIWallpeper is not fully supported.
*******************************************************************************}

unit sTrans;

interface

uses
   Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
   StdCtrls, ExtCtrls;

type
   TsTransControl = class(TCustomControl)
   private
      FTransparent: Boolean;

      FBackground: HDC;
      FTempBitmap: THandle;
      FOldBitmap: THandle;

      FBackgroundChanged: Boolean;
      procedure SetTransparent(Value: Boolean);
      function GetLeftTopPoint: TPoint;

      procedure KillTempDC;

      procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
      procedure WMPaint(var Msg: TWMPaint); message WM_PAINT;
      procedure WMNCPaint(var Msg: TWMPaint); message WM_NCPAINT;
      procedure WMMove(var Msg: TMessage); message WM_MOVE;
      procedure WMSize(var Msg: TMessage); message WM_SIZE;
   protected
      procedure Paint; override;
      procedure SaveBackground;
      property Transparent: Boolean read FTransparent write SetTransparent default FALSE;
      property BackgroundChanged: Boolean read FBackgroundChanged;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure PaintBackground;
      procedure PaintBackgroundRect(R: TRect);
      procedure FrameRect(R: TRect; penWidth: Integer);
      procedure DrawFlatEDGE(R: TRect; Down: Boolean);
      procedure ForceRefresh;
   end;

procedure RefreshTransControls(Parent: TWinControl);

implementation

{$I S.Inc}
uses sGraphics, sTdUtils
{$IFDEF MDIWALLPAPER_SUPPORT}
   ,MDIWallp
{$ENDIF}
;

var
   DCList: TList;
   BmpList: TList;
   CtrlsList: TLIst;
   FForceRefresh: Boolean;

procedure RefreshTransControls(Parent: TWinControl);
var
   ii: Integer;
begin
   FForceRefresh := TRUE;
{
	Can anyone tell me how to enumerate the child controls at the design time?
}
   for ii := 0 to Parent.ComponentCount - 1 do begin
      if Parent.Components[ii].InheritsFrom(TsTransControl) then
         TsTransControl(Parent.Components[ii]).ForceRefresh;
   end;
end;

constructor TsTransControl.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle - [csOpaque];
   FTransparent := FALSE;
   FBackgroundChanged := TRUE;
end;

destructor TsTransControl.Destroy;
begin
   KillTempDC;
   inherited;
end;

procedure TsTransControl.KillTempDC;
begin
   if FBackground <> 0 then begin
      SelectObject(FBackground, foldbitmap);
      DeleteObject(ftempbitmap);
      DeleteDC(FBackground);
      FBackground := 0;
   end;
end;

procedure TsTransControl.SetTransparent(Value: Boolean);
begin
   if Value <> FTransparent then begin
      FTransparent := Value;
      Invalidate;
   end;
end;

{$IFDEF MDIWALLPAPER_SUPPORT}
type
   TComponentClass = class of TComponent;

function FindComponentClass(aOwner: TComponent; CompClass: TComponentClass): TComponent;
var
   ii: Integer;
begin
   Result := nil;
   for ii := 0 to aOwner.ComponentCount - 1 do
      if aOwner.Components[ii] is CompClass then begin
         Result := aOwner.Components[ii];
         Exit;
      end;
end;
{$ENDIF}


(*
procedure TsTransControl.SaveBackground;
var
   paintDC, parentDC: HDC;
   {oldBitmap, }bitmap: HBITMAP;
   P: TPoint;
   index: Integer;
{$IFDEF MDIWALLPAPER_SUPPORT}
   cmp: TComponent;
{$ENDIF}
begin
   if ([csReading, csLoading] * ComponentState <> []) or (Parent = nil)
      or ([csReading, csLoading] * Parent.ComponentState <> [])
      or not HandleAllocated or not Visible then
      Exit;

   FBackground.Width := Width;
   FBackground.Height := Height;

   index := CtrlsList.IndexOf(Parent);
   if (index = -1) or FForceRefresh then begin

      if (index > -1) and FForceRefresh then begin
         DeleteObject( Integer(BmpList[index]));
         BmpList.Delete(index);
         DeleteDC( Integer(DCList[index]));
         DCList.Delete(index);
      end;

      {$IFDEF MDIWALLPAPER_SUPPORT}
      cmp := FindComponentClass(parent, TMDIWallpaper);
      if cmp <> nil then
         ParentDC := GetDC( TMDIWallpaper(cmp).ClientHandle)
      else
      {$ENDIF}
         ParentDC := GetDC( Parent.Handle);

      paintDC := CreateCompatibleDC( ParentDC);
      bitmap := CreateCompatibleBitmap( parentDC, Parent.width, Parent.height);
      ReleaseDC( Parent.Handle, parentDC);
      {oldBitmap := }SelectObject( paintDC, bitmap);

      P := GetLeftTopPoint;
      {$IFDEF MDIWALLPAPER_SUPPORT}
      if cmp <> nil then
         SendMessage( TMDIWallpaper(cmp).ClientHandle, WM_ERASEBKGND, paintDC, 0)
      else
      {$ENDIF}
         Parent.PaintTo(paintDC, 0, 0);

      CtrlsList.Add(Parent);
      DCList.Add(Pointer(paintDC));
      BmpList.Add(Pointer(bitmap));
      FForceRefresh := FALSE;
   end else
      paintDC := Integer(DCList[index]);

   P := GetLeftTopPoint;
   BitBlt( FBackground.canvas.handle, 0, 0, width, height, paintDC, P.X, P.Y, SRCCOPY);

   FBackgroundChanged := FALSE;
end;

procedure ParentPaintTo(P: TWinControl; DC: HDC; R: TRect);
var
   SaveIndex: Integer;
begin
   P.ControlState := P.ControlState + [csPaintCopy];
   SaveIndex := SaveDC(DC);
   IntersectClipRect( DC, R.Left, R.Top, R.Right + 1, R.Bottom + 1);
//   P.Perform(WM_ERASEBKGND, DC, 0);
   P.Perform(WM_PAINT, DC, 0);
   RestoreDC(DC, SaveIndex);
   P.ControlState := P.ControlState - [csPaintCopy];
end;

procedure TsTransControl.SaveBackground;
var
   paintDC, parentDC: HDC;
   oldBitmap, bitmap: HBITMAP;
   P: TPoint;
   index: Integer;
{$IFDEF MDIWALLPAPER_SUPPORT}
   cmp: TComponent;
{$ENDIF}
begin
   if ([csReading, csLoading] * ComponentState <> []) or (Parent = nil)
      or ([csReading, csLoading] * Parent.ComponentState <> [])
      or not HandleAllocated or not Visible then
      Exit;

   FBackground.Width := Width;
   FBackground.Height := Height;

   {$IFDEF MDIWALLPAPER_SUPPORT}
   cmp := FindComponentClass(parent, TMDIWallpaper);
   if cmp <> nil then
      ParentDC := GetDC( TMDIWallpaper(cmp).ClientHandle)
   else
   {$ENDIF}
      ParentDC := GetDC( Parent.Handle);

   paintDC := CreateCompatibleDC( ParentDC);
   bitmap := CreateCompatibleBitmap( parentDC, Parent.width, Parent.height);
   ReleaseDC( Parent.Handle, parentDC);
   oldBitmap := SelectObject( paintDC, bitmap);


   {$IFDEF MDIWALLPAPER_SUPPORT}
   if cmp <> nil then begin
      SendMessage( TMDIWallpaper(cmp).ClientHandle, WM_ERASEBKGND, paintDC, 0);
      P := GetLeftTopPoint;
      BitBlt( FBackground.canvas.handle, 0, 0, width, height, paintDC, P.X, P.Y, SRCCOPY);
   end else begin
   {$ENDIF}
      ParentPaintTo(Parent, PaintDC, Bounds(Left, Top, Width, Height));
      BitBlt( FBackground.canvas.handle, 0, 0, width, height, PaintDC, left + 1, top + 1, SRCCOPY);
   {$IFDEF MDIWALLPAPER_SUPPORT}
   end;
   {$ENDIF}


   ReleaseDC( Parent.Handle, parentDC);
   SelectObject( paintDC, oldBitmap);
   DeleteObject( bitmap);
   DeleteDC( paintDC);

   FForceRefresh := FALSE;
   FBackgroundChanged := FALSE;
end;
*)
procedure TsTransControl.SaveBackground;
var
   dc: THandle;
   formdc: THandle;
   oldfbitmap: THandle;
   fbitmap: THandle;
   fdc: THandle;
begin
   if Parent = nil then
      Exit;
   if FBackgroundChanged then
      KillTempDC;

    dc := GetDC(handle);
    fdc := GetDC(parent.handle);
    formdc := CreateCompatibleDC(fdc);
    fbitmap := CreateCompatibleBitmap(fdc, parent.width, parent.height);
    oldfbitmap := SelectObject(formdc, fbitmap);

    if FBackground = 0 then begin
      FBackground := CreateCompatibleDC(dc);
      FTempBitmap := CreateCompatibleBitmap(dc, width, height);
      FOldBitmap := SelectObject(FBackground, FTempBitmap);
    end;

    IntersectClipRect(formdc, left, top, left + width + 1, top + height + 1);
    parent.PaintTo(formdc, 0, 0);
    BitBlt(FBackground, 0, 0, width, height, formdc, left, top, SRCCOPY);
    SelectObject(formdc, oldfbitmap);
    DeleteObject(fbitmap);
    DeleteDC(formdc);
    ReleaseDC(Parent.Handle, fdc);
    ReleaseDC(handle, dc);

    FBackgroundChanged := FALSE;
end;

function TsTransControl.GetLeftTopPoint: TPoint;
begin
   Result := ClientToScreen(Point(0, 0));
   Result := Parent.ScreenToClient(Result);
end;

procedure TsTransControl.PaintBackground;
begin
   PaintBackgroundRect( Rect(0, 0, Width, Height));
end;

procedure TsTransControl.PaintBackgroundRect(R: TRect);
begin
   if FTransparent then begin
      BitBlt( Canvas.Handle, R.Left, R.Top, WidthOf(R), HeightOf(R),
         FBackground{.Canvas.handle}, R.Left, R.Top, srccopy);
   end else with Canvas do begin
      Brush.Style := bsSolid;
      Brush.Color := Color;
      FillRect(R);
   end;
end;

procedure TsTransControl.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
   if FTransparent then begin
      Msg.Result := 1
   end else
      inherited;
end;

procedure TsTransControl.WMNCPaint(var Msg: TWMPaint);
begin
   inherited;
end;

procedure TsTransControl.WMPaint(var Msg: TWMPaint);
var
   R: TRect;
   ps: TPaintStruct;
begin
   if csPaintCopy in ControlState then
      Exit;
   if FTransparent then begin
      Msg.Result := 0;
      GetUpdateRect( Handle, R, False);
      if IsRectEmpty(R) then
         Exit;
      BeginPaint( Handle, ps);
      Paint;
      EndPaint( Handle, ps);
   end else
      inherited;
end;

procedure TsTransControl.WMMove(var Msg: TMessage);
begin
   FBackgroundChanged := TRUE;
   inherited;
   Invalidate;
end;

procedure TsTransControl.WMSize(var Msg: TMessage);
begin
   FBackgroundChanged := TRUE;
   inherited;
   Invalidate;
end;

procedure TsTransControl.Paint;
begin
   if FTransparent and FBackgroundChanged then
      SaveBackground;
   inherited;
end;

procedure TsTransControl.ForceRefresh;
begin
   FBackgroundChanged := TRUE;
   FForceRefresh := TRUE;
   Invalidate;
end;

procedure TsTransControl.FrameRect(R: TRect; penWidth: Integer);
var
   MemDC: HDC;
   aBitmap, oldObject: HBITMAP;
   ii: Integer;
begin
   if Transparent then begin
      InflateRect(R, -penWidth, -PenWidth);
      MemDC := CreateCompatibleDC( Canvas.Handle);
      aBitmap := CreateCompatibleBitmap( Canvas.Handle, R.Right - R.Left, R.Bottom - R.Top);
      oldObject := SelectObject( MemDC, aBitmap);
      BitBlt( MemDC, 0, 0, R.Right - R.Left, R.Bottom - R.Top, Canvas.Handle,
         R.Left, R.Top, SRCCOPY);
      InflateRect(R, penWidth, PenWidth);
      BitBlt( Canvas.Handle, R.left, R.Top, R.Right - R.Left, R.Bottom - R.Top, FBackground{.Canvas.Handle},
         R.Left, R.Top, SRCCOPY);
      InflateRect(R, -penWidth, -PenWidth);
      BitBlt( Canvas.Handle, R.left, R.Top, R.Right - R.Left, R.Bottom - R.Top, MemDC, 0, 0, SRCCOPY);
      SelectObject( MemDC, oldObject);
      DeleteObject( aBitmap);
      DeleteDC( MemDC);
   end else for ii := 0 to penWidth do begin
      Canvas.Brush.Color := Color;
      Canvas.FrameRect(R);
      InflateRect(R, -1, -1);
   end;
end;

procedure TsTransControl.DrawFlatEDGE(R: TRect; Down: Boolean);
begin
	Canvas.Brush.Color := clBtnShadow;
	Canvas.FrameRect(R);
   Canvas.Pen.Color := clHighlightText;
   if Down then
		Canvas.PolyLine( [Point(R.Left, R.Bottom-1), Point(R.Right-1, R.Bottom-1), Point(R.Right-1, R.Top-1)])
	else
		Canvas.PolyLine( [Point(R.Left, R.Bottom-1), Point(R.Left, R.Top), Point(R.Right-1, R.Top)])
end;

{
procedure DestroyLocals;
var
   ii: Integer;
begin
   for ii := 0 to BmpList.Count - 1 do begin
      DeleteObject( Integer(BmpList[ii]));
      DeleteDC( Integer(DCList[ii]));
   end;
   BmpList.Free;
   CtrlsList.Free;
   DCList.Free;
end;
}
initialization
{
   DCList := TList.Create;
   BmpList := TList.Create;
   CtrlsList := TList.Create;
}
finalization
//   DestroyLocals;

end.

