{*******************************************************************************
   Utit:
      sNumPad.pas
   Description:
      This unit implements simple popup NumPad.
   Versions:
      2.0a
   Author(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com*
   History:
		2.0a	- 	12/11/1998
      			-Bug fix for popup calendar: VK_RETURN and VK_ESCAPE handling when control, which
					owned the calendar is sitted on the form with buttons with Cancel or Default
               property set to TRUE.  (declared by Roman Yugov).
               - TsNumPad now ignores parent forms's KeyPreview property and does
               not notify it about key events.
		2.0*  - 	End of Sep. 1998
      			Initial release
*     I did not track the versions before, so let's consider it as 2.0
**    The calculations mechanizme was taken from RX's controls.
*******************************************************************************}

unit sNumPad;

interface

uses Classes, Controls, Graphics, sGraphics, Windows, Messages;


type
   TnpBtn = (  np7,     np8,     np9,     npDiv,
               np4,     np5,     np6,     npMul,
               np1,     np2,     np3,     npMinus,
               np0,     npDot,   npSign,  npPlus,
               npClear, npBack,  npResult, npNone);
   TnpBtnState = (npsNormal, npsUp, npsDown);
   TsNumPadState = (csFirst, csValid, csError);

  	TsNumPad = class( TCustomControl)
   private
      FFont3d: TFont3d;
      FLastHilightBtn: TnpBtn;
      FTrackState: TnpBtn;
      FPressed: Boolean;
      FStatus: TsNumPadState;
      FStrValue: String;
      FOperator: Char;
      FOperand: Double;
      FOnAccept: TNotifyEvent;
      FOnCancel: TNotifyEvent;
      FOnValueChange: TNotifyEvent;
      procedure SetFont3d(Value: TFont3d);
      procedure SetStrValue(Value: String);
      procedure SetValue(Value: Double);
      function GetValue: Double;
      procedure Track( X, Y: Integer);
      procedure StopTracking(X, Y: Integer);
      procedure CalculatePosition( PopupPoint: TPoint);
      procedure PaintButton(btn: TnpBtn; down: Boolean);
      procedure PaintHilightFrame( btn: TnpBtn; btnStates: TnpBtnState);
      function GetBtnRect(btn: TnpBtn): TRect;
      function GetBtnFromPoint(P: TPoint): TnpBtn;
      procedure CalcKey(Key: Char);
      procedure CheckFirst;
      procedure Clear;
      procedure Error;
    	procedure CNKeyDown(var Message: TWMKeyDown); message CN_KEYDOWN;
      procedure WMKeyDown(var Message: TWMKeyDown); message WM_KEYDOWN;
      procedure WMKeyUp(var Message: TWMKeyDown); message WM_KEYUP;
      procedure WMChar(var Message: TWMChar); message WM_CHAR;
      procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
   protected
      procedure CreateParams(var Params: TCreateParams); override;
  		procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    	procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    	procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    	procedure Paint; override;
      procedure KeyUp(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
      procedure Cancel;
      procedure Accept;
  	public
    	constructor Create( AOwner: TComponent); override;
      destructor Destroy; override;
      procedure Popup(X, Y: Integer);
      property StrValue: String read FStrValue write SetStrValue;
      property Value: Double read GetValue write SetValue;
   	property OnAccept: TNotifyEvent read FOnAccept write FOnAccept;
      property OnCancel: TNotifyEvent read FOnCancel write FOnCancel;
      property OnValueChange: TNotifyEvent read FOnValueChange write FOnValueChange;
   published
      property Color default clBtnFace;
      property Font;
      property Font3d: TFont3d read FFont3d write SetFont3d;
  	end;

   TsNumPadProps = class(TPersistent)
   private
      FNumPad: TsNumPad;
      procedure SetColor(Value: TColor);
      function GetColor: TColor;
      procedure SetFont(Value: TFont);
      function GetFont: TFont;
      procedure SetFont3d(Value: TFont3d);
      function GetFont3d: TFont3d;
   public
      constructor Create(numPad: TsNumPad);
   published
      property Color: TColor read GetColor write SetColor;
      property Font: TFont read GetFont write SetFont;
      property Font3d: TFont3d read GetFont3d write SetFont3d;
   end;

implementation

uses Forms, SysUtils;

{******************** TsNumPad ******************************************}

constructor TsNumPad.Create( AOwner: TComponent);
begin
   inherited;
   Visible := FALSE;
   ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable];
   Width := 84;
   Height := 104;
   FLastHilightBtn := npNone;
   FTrackState := npNone;
   FFont3d := TFont3d.Create(nil);
   Color := clBtnFace;
   Canvas.Font.Style := [fsBold];
   Parent := AOwner as TWinControl;
end;

destructor TsNumPad.Destroy;
begin
   FFont3d.Free;
   inherited;
end;

procedure TsNumPad.CreateParams(var Params: TCreateParams);
begin
	inherited CreateParams( Params);
  	with Params do begin
      Style := WS_POPUP or WS_CLIPCHILDREN;
      ExStyle := WS_EX_TOOLWINDOW;
    	WindowClass.Style := WindowClass.Style or CS_SAVEBITS;
	end;
end;

procedure TsNumPad.CNKeyDown(var Message: TWMKeyDown);
begin
  	Message.Result := 0;
end;

procedure TsNumPad.WMKeyDown(var Message: TWMKeyDown);
begin
  	with Message do
     	KeyDown(CharCode, KeyDataToShiftState(KeyData));
end;

procedure TsNumPad.WMKeyUp(var Message: TWMKeyDown);
begin
	with Message do
      KeyUp(CharCode, KeyDataToShiftState(KeyData));
end;

procedure TsNumPad.WMChar(var Message: TWMChar);
var
	Ch: Char;
begin
	with Message do begin
      Ch := Char(CharCode);
      KeyPress(Ch);
      CharCode := Word(Ch);
   end;
end;

procedure TsNumPad.WMKillFocus(var Message: TMessage);
begin
   if Visible and (Message.WParam <> Handle) then begin
      Cancel;
   end;
   inherited;
end;

procedure TsNumPad.CMMouseLeave(var Message: TMessage);
begin
   if FTrackState = npNone then begin
      PaintHilightFrame( FTrackState, npsNormal);
      PaintHilightFrame( FLastHilightBtn, npsNormal);
   end;
   inherited;
end;

procedure TsNumPad.Paint;
var
  	R: TRect;
   ii: TnpBtn;
begin
   R := ClientRect;
   Canvas.Brush.Color := Color;
   DrawEdge( Canvas.Handle, R, EDGE_RAISED, BF_RECT);
   for ii := Low(TnpBtn) to TnpBtn(Ord(High(TnpBtn))-1) do
      PaintButton(ii, FALSE);
end;

procedure TsNumPad.PaintButton(btn: TnpBtn; down: Boolean);
const
   FaceVal: array[TnpBtn] of Char = ('7', '8', '9', '/',
                                     '4', '5', '6', '*',
                                     '1', '2', '3', '-',
                                     '0', '.', '', '+',
                                     'C', '<', '=', #0);
var
   R: TRect;
   S: String;
begin
   R := GetBtnRect(btn);
   S := FaceVal[btn];
   if Btn = npBack then
      S := '<-';
   if down then begin
      Inc(R.Top);
      Inc(R.Left);
   end;
   with Canvas do begin
      Brush.Style := bsSolid;
      Brush.Color := Color;
      FillRect(R);
      Brush.Style := bsClear;
      Font := self.Font;
   end;
   PaintText( Canvas, PChar(S), Length( S), R,
         DT_CENTER or DT_VCENTER or DT_TOP or DT_SINGLELINE, FFont3d, nil, hsNone);
end;

procedure TsNumPad.PaintHilightFrame( btn: TnpBtn; btnStates: TnpBtnState);
const
   edges: array[TnpBtnState] of Word = ( 0, EDGE_RAISED, EDGE_SUNKEN);
var
   R: TRect;
   procedure PaintFrame;
   begin
      Canvas.Brush.Style := bsSolid;
      R := GetBtnRect(FLastHilightBtn);
      Canvas.FrameRect(R);
      InflateRect(R, -1, -1);
      Canvas.FrameRect(R);
   end;
begin
   if (btn <> FLastHilightBtn) or (FTrackState <> npNone) then begin
      Canvas.Brush.Color := Color;
      if FLastHilightBtn <> npNone then
         PaintFrame;
      FLastHilightBtn := btn;
      if btn <> npNone then begin
         if btnStates = npsNormal then
            PaintFrame
         else begin
            R := GetBtnRect(FLastHilightBtn);
            Canvas.Brush.Style := bsSolid;
            PaintButton(FLastHilightBtn, btnStates = npsDown);
            DrawEdge( Canvas.Handle, R, edges[btnStates], BF_RECT);
            InflateRect(R, -1, -1);
            Canvas.FrameRect(R);
         end;
      end;
   end;
end;

function TsNumPad.GetBtnRect(btn: TnpBtn): TRect;
var
   r, c: Integer;
begin
   c := Ord(btn) mod 4;
   r := Ord(btn) div 4;
   Result := Bounds( 20*c, 20*r, 20, 20);
   OffsetRect(Result, 2, 2);
   if btn = npResult then
      Inc(Result.Right, 20);
end;

function TsNumPad.GetBtnFromPoint(P: TPoint): TnpBtn;
var
   btn: Integer;
begin
   Result := npNone;
   if PtInRect(Rect(2, 2, 80, 100), P) then begin
      btn := p.y div 20 * 4 + p.x div 20;
      if btn in [Ord(Low(TnpBtn))..Ord(High(TnpBtn))] then begin
         Result := TnpBtn(btn);
         if Result = npNone then
            Result := npResult;
      end;
   end;
end;

procedure TsNumPad.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      FTrackState := GetBtnFromPoint(Point(X, Y));
      Track(X, Y);
      MouseCapture := FTrackState <> npNone;
   end;
   inherited;
end;

procedure TsNumPad.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
   if ([csLoading, csReading] * ComponentState = []) and (FTrackState = npNone) then
      PaintHilightframe( GetBtnFromPoint(Point(X, Y)), npsUp)
   else
      Track(X, Y);
   inherited;
end;

function npKey2Char(btn: TnpBtn): Char;
const
   charmap: array[TnpBtn] of Char = ( '7', '8', '9', '/', '4', '5', '6', '*',
            '1', '2', '3', '-', '0', '.', '_', '+', 'C', #8, #13, #0);
begin
   Result := charmap[btn];
   if Result = '.' then
      Result := DecimalSeparator;
end;

function char2npKey(Key: Char): TnpBtn;
const
   digmap: array['0'..'9'] of TnpBtn = ( np0, np1, np2, np3, np4, np5, np6, np7, np8, np9);
   opmap: array['*'..'/'] of TnpBtn = (npMul, npPlus, npDot, npMinus, npDot, npDiv);
begin
   Result := npNone;
   case key of
      '0'..'9': Result := digmap[key];
      '*'..'/': Result := opmap[key];
      '_': Result := npSign;
      'c', 'C': Result := npClear;
      Char(VK_BACK): Result := npBack;
   end;
end;

procedure TsNumPad.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer );
begin
   MouseCapture := FALSE;
  	if (Button = mbLeft) then begin
      if PtInRect( GetBtnRect(FTrackState), Point(X, Y)) then
         CalcKey( npKey2Char(FTrackState));
      StopTracking(X, Y);
   end else
   	inherited;
end;

procedure TsNumPad.Track(X,Y: Integer);
const
   btnStates: array[Boolean] of TnpBtnState = (npsUp, npsDown);
var
  	NewState: Boolean;
  	R: TRect;
begin
   R := GetBtnRect( FTrackState);
   NewState := PtInRect(R, Point(X, Y));
   if FPressed <> NewState then begin
      FPressed := NewState;
      PaintHilightFrame( FTrackState, btnStates[newState]);
   end;
end;

procedure TsNumPad.StopTracking(X, Y: Integer);
const
   btnStates: array[Boolean] of TnpBtnState = (npsNormal, npsUp);
begin
   if FTrackState <> npNone then begin
    	Track(-1, -1);
    	MouseCapture := False;
      PaintHilightFrame( FLastHilightBtn,
         btnStates[PtInRect(GetBtnRect( FTrackState), Point(X, Y))]);
      FTrackState := npNone;
  	end;
end;

procedure TsNumPad.KeyUp(var Key: Word; Shift: TShiftState);
begin
   PaintHilightFrame( FTrackState, npsNormal);
   FTrackState := npNone;
   inherited;
end;

procedure TsNumPad.KeyPress(var Key: Char);
begin
   FTrackState := char2npKey(key);
   PaintHilightFrame( FTrackState, npsDown);
   case Key of
      Char(VK_ESCAPE): begin
         if FStatus = csFirst then
            Cancel
         else
            CalcKey('C');
      end;
      '0'..'9', '+', '-', '*', '/', '=', '%', '_', #13, #187, 'C', 'c':
         CalcKey(Key);
      '.', ',':
         CalcKey(decimalSeparator);
   end;
end;

procedure TsNumPad.CalculatePosition( PopupPoint: TPoint);
var
   pt: TPoint;
   ALeft, ATop: Integer;
   d: Integer;
begin
  	pt := Parent.ClientToScreen( PopupPoint);
   ATop := pt.Y + 1;
   ALeft := pt.X - Width + 1;
   SetBounds(ALeft, ATop, 84, 104);

   d := Screen.Height - (Top + Height);
   if d < 0 then
      Top := Top + d;
   d := Screen.Width - (Left + Width);
   if d < 0 then
      left := Left + d;
end;

procedure TsNumPad.SetStrValue(Value: String);
begin
   if FStrValue <> Value then begin
      FStrValue := Value;
      if assigned(FOnValueChange) then
         FOnValueChange(self);
   end;
end;

function TsNumPad.GetValue: Double;
begin
   if (StrValue = '') or (FStatus = csError) then
      Result := 0
   else try
      Result := StrToFloat(Trim(FStrValue));
   except
      Result := 0;
   end;
end;

procedure TsNumPad.SetValue(Value: Double);
var
   S: string;
begin
   S := Format('%31.16f', [Value]);
   while S[Length(S)] = '0' do
      Delete(S, Length(S), 1);
   if S[Length(S)] = DecimalSeparator then
      Delete(S, Length(S), 1);
   StrValue := Trim(S);
end;

procedure TsNumPad.SetFont3d(Value: TFont3d);
begin
   if FFont3d <> Value then
      FFont3d.Assign(Value);
end;

procedure TsNumPad.Popup(X, Y: Integer);
begin
   if Value <> 0 then begin
      FStatus := csFirst;
      FOperator := '=';
   end;
   CalculatePosition(Point(X, Y));
   Visible := True;
   SetFocus;
end;

procedure TsNumPad.Cancel;
begin
   Visible := FALSE;
   Parent.SetFocus;
   if Assigned( OnCancel) then
      OnCancel(self);
end;

procedure TsNumPad.Accept;
begin
   Visible := FALSE;
   Parent.SetFocus;
   if Assigned( OnAccept) then
      OnAccept(self);
end;


procedure TsNumPad.CheckFirst;
begin
   if FStatus = csFirst then begin
      FStatus := csValid;
      FStrValue := '0';
   end;
end;

procedure TsNumPad.CalcKey(Key: Char);
var
   R: Double;
begin
   Key := UpCase(Key);
   if (FStatus = csError) and (Key <> 'C') then
      Key := ' ';
   if Key = DecimalSeparator then begin
      CheckFirst;
      if Pos(DecimalSeparator, FStrValue) = 0 then
         StrValue := FStrValue + DecimalSeparator;
   end else
      case Key of
         '0'.. '9': begin
               CheckFirst;
               if FStrValue = '0' then
                  StrValue := '';
               if Pos('E', FStrValue) = 0 then
                  StrValue := FStrValue + Key;
            end;
         #8: begin
               CheckFirst;
               if (Length(FStrValue) = 1) or
                  ((Length(FStrValue) = 2) and (FStrValue[1] = '-')) then
                  StrValue := '0'
               else
                  StrValue := Copy(FStrValue, 1, Length(FStrValue) - 1);
            end;
         '_': begin
            //CheckFirst;
            SetValue(-GetValue);
         end;
         '+', '-', '*', '/', '=', '%', #13: begin
               if FStatus = csValid then begin
                  FStatus := csFirst;
                  R := GetValue;
                  if Key = '%' then
                     case FOperator of
                        '+', '-':
                           R := FOperand * R / 100;
                        '*', '/':
                           R := R / 100;
                     end;
                  case FOperator of
                     '+':
                        SetValue(FOperand + R);
                     '-':
                        SetValue(FOperand - R);
                     '*':
                        SetValue(FOperand * R);
                     '/':
                        if R = 0 then
                           Error
                        else
                           SetValue(FOperand / R);
                  end;
               end;
               FOperator := Key;
               FOperand := GetValue;
               if (Key = #13) and (FStatus = csFirst) then
                  Accept
            end;
         'C':
            Clear;
      end;
end;

procedure TsNumPad.Clear;
begin
   FStatus := csFirst;
   StrValue := '0';
   FOperator := '=';
end;

procedure TsNumPad.Error;
begin
   FStatus := csError;
   StrValue := 'Error';
end;

{******************** TsNumPadProps *****************************}

constructor TsNumPadProps.Create(numPad: TsNumPad);
begin
   FNumPad := numPad;
end;

procedure TsNumPadProps.SetColor(Value: TColor);
begin
   FNumPad.Color := Value;
end;

function TsNumPadProps.GetColor: TColor;
begin
   Result := FNumPad.Color;
end;

procedure TsNumPadProps.SetFont(Value: TFont);
begin
   FNumPad.Font.Assign(Value);
end;

function TsNumPadProps.GetFont: TFont;
begin
   Result := FNumPad.Font;
end;

procedure TsNumPadProps.SetFont3d(Value: TFont3d);
begin
   FNumPad.Font3d.Assign(Value);
end;

function TsNumPadProps.GetFont3d: TFont3d;
begin
   Result := FNumPad.Font3d;
end;




end.
