{*******************************************************************************
   Utit:
      sGraphics.pas
   Description:
      Commonly used graphic and text-drawing routines
   Versions:
      2.0b
   Author(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   History:
      2.0b  -  25/11/1998
               1. Two more selection styles - underline and color
               2. PaintText's first parameter changed from DC to TCanvas.
               3. Some fixes.
      2.0a  -  18/11/1998
               Small change for the TsGlyphList
		2.0*  - 	End of Sep. 1998
      			Initial release
*     I did not track the versions before, so let's consider it as 2.0
**    Some functions here were copied (some of them modified) from RX's VCLUtils
      unit. 
*******************************************************************************}

unit sGraphics;

interface

uses Windows, Classes, Graphics;

type
   TShadowPosition = (spLeftTop, spLeftBottom, spRightBottom, spRightTop);
   THilightStyle = (hsNone, hsMoving, hsRaising, hsFlushing, hsUnderline, hsColor, hsFrame);
   TTextHilightStyle = hsNone..hsColor;
   TsDisabledStyle = (dsDefault, dsOffice97, dsEditTools);

   TFont3d = class(TPersistent)
   private
      FActive: Boolean;
      FColor: TColor;
      FPosition: TShadowPosition;
      FSpacing: ShortInt;
      FOnChange: TNotifyEvent;
      procedure SetActive(Value: Boolean);
      procedure SetPosition(Value: TShadowPosition);
      procedure SetColor(Value: TColor);
      procedure SetSpacing(Value: ShortInt);
      procedure Changed;
   public
      constructor Create( aOnChange: TNotifyEvent);
      procedure Assign(Source: TPersistent); override;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
   published
      property Active: Boolean read FActive write SetActive default TRUE;
      property Color: TColor read FColor write SetColor default clBtnHighlight;
      property Position: TShadowPosition read FPosition write SetPosition default spRightBottom;
      property Spacing: ShortInt read FSpacing write SetSpacing default 1;
   end;


function PaintText( Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; Font3d, Hilight3d: TFont3d; hilightStyle: TTextHilightStyle): Integer;

function DrawShadowText(Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; ShadowSize: Byte; ShadowColor: TColorRef;
   ShadowPos: TShadowPosition): Integer;

function DrawFlashText( Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; ShadowSize: Byte; ShadowColor: TColorRef): Integer;

{
   from RX
}
procedure StretchBitmapRectTransparent(Dest: TCanvas; DstX, DstY, DstW, DstH: Integer;
   SrcRect: TRect; Bitmap: TBitmap; TransparentColor: TColor);

procedure StretchBltTransparent(DstDC: HDC; DstX, DstY, DstW, DstH: Integer;
  SrcDC: HDC; SrcX, SrcY, SrcW, SrcH: Integer; Palette: HPalette;
  TransparentColor: TColorRef);

procedure DrawTransparentBitmapRect(DC: HDC; Bitmap: HBitmap; DstX, DstY,
   DstW, DstH: Integer; SrcRect: TRect; TransparentColor: TColorRef);

procedure DrawBitmapTransparent(Dest: TCanvas; DstX, DstY: Integer;
  Bitmap: TBitmap; TransparentColor: TColor);

const
   PaletteMask = $02000000;
   TransparentMask = $02000000;

implementation

uses Consts, SysUtils;

constructor TFont3d.Create( aOnChange: TNotifyEvent);
begin
   FOnChange := aOnChange;
   FActive := TRUE;
   FPosition := spRightBottom;
   FColor := clBtnHighlight;
   FSpacing := 1;
end;

procedure TFont3d.SetActive(Value: Boolean);
begin
   if FActive <> Value then begin
      FActive := Value;
      Changed;
   end;
end;

procedure TFont3d.SetPosition(Value: TShadowPosition);
begin
   if FPosition <> Value then begin
      FPosition := Value;
      Changed;
   end;
end;

procedure TFont3d.SetColor(Value: TColor);
begin
   if FColor <> Value then begin
      FColor := Value;
      Changed;
   end;
end;

procedure TFont3d.SetSpacing(Value: ShortInt);
begin
   if FSpacing <> Value then begin
      FSpacing := Value;
      Changed;
   end;
end;

procedure TFont3d.Changed;
begin
   if Assigned(FOnChange) then
      FOnChange(self);
end;

procedure TFont3d.Assign(Source: TPersistent);
var
   SourceName: string;
begin
   if Source = nil then
      SourceName := 'nil'
   else
      SourceName := Source.ClassName;
   if (Source = nil) or not (Source is TFont3d) then
      raise EConvertError.CreateFmt( SAssignError, [SourceName, ClassName]);
   with TFont3d(Source) do begin
      self.FActive := FActive;
      self.FColor := FColor;
      self.FPosition := FPosition;
      self.FSpacing := FSpacing;
   end;
   Changed;
end;

function PaintText( Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; Font3d, Hilight3d: TFont3d; hilightStyle: TTextHilightStyle): Integer;
var
   saveStyle: TFontStyles;
   saveColor: TColor;
   font3Dcreated: Boolean;
begin
   Result := 0;
   font3Dcreated := FALSE;
   if Font3D = nil then begin
      Font3D := TFont3D.Create(nil);
      Font3D.Spacing := 0;
      font3Dcreated := TRUE;
   end;
   if (hilightStyle in [hsMoving, hsRaising]) and ((Format and (0 xor DT_CALCRECT)) = 0) then begin
      if (hilight3d <> nil) and hilight3d.Active then begin
         if not Font3d.Active then
            Font3d.Spacing := 0;   
         case Hilight3d.Position of
            spLeftTop:
               if hilightStyle = hsMoving then
                  OffsetRect(Rect, -Hilight3d.Spacing, -Hilight3d.Spacing);
            spRightBottom:
               if hilightStyle = hsMoving then
                  OffsetRect(Rect, Hilight3d.Spacing, Hilight3d.Spacing)
               else
                  OffsetRect( Rect, -Hilight3d.Spacing + Font3d.Spacing,
                                    -Hilight3d.Spacing + Font3d.Spacing);
            spLeftBottom: begin
               if hilightStyle = hsMoving then
                  OffsetRect(Rect, -Hilight3d.Spacing, Hilight3d.Spacing)
               else
                  OffsetRect(Rect, 0, -Hilight3d.Spacing + Font3d.Spacing);
            end;
            spRightTop: begin
               if hilightStyle = hsMoving then
                  OffsetRect(Rect, Hilight3d.Spacing, -Hilight3d.Spacing)
               else
                  OffsetRect(Rect, -Hilight3d.Spacing + Font3d.Spacing, 0);
            end;
         end;
      end;
   end;
   if (HilightStyle <> hsNone) and (Hilight3d <> nil) and (Hilight3d.Active) then begin
      case hilightStyle of
         hsRaising:
            Result := DrawShadowText( Canvas, Str, Count, Rect, Format,
               Hilight3d.Spacing, ColorToRGB(Hilight3d.Color), Hilight3d.Position);//Font3d.Position);
         hsFlushing:
            Result := DrawFlashText( Canvas, Str, Count, Rect, Format,
               Hilight3d.Spacing, ColorToRGB(Hilight3d.Color));
         hsMoving:
            Result := DrawShadowText( Canvas, Str, Count, Rect, Format,
               Font3D.Spacing, ColorToRGB(Font3d.Color), Font3D.Position);
         hsUnderline: begin
            saveStyle := Canvas.Font.Style;
            Canvas.Font.Style := Canvas.Font.Style + [fsUnderline];
            Result := DrawShadowText( Canvas, Str, Count, Rect, Format,
               Font3d.Spacing, ColorToRGB(Hilight3d.Color), Font3d.Position);
            Canvas.Font.Style := saveStyle;
         end;
         hsColor: begin
            saveColor := Canvas.Font.Color;
            Canvas.Font.Color := Hilight3d.Color;
            Result := DrawShadowText( Canvas, Str, Count, Rect, Format,
               Font3d.Spacing, ColorToRGB(Font3d.Color), Font3d.Position);
            Canvas.Font.Color := saveColor;
         end;
      end;
   end else if (Font3d <> nil) and Font3d.Active then
      Result := DrawShadowText( Canvas, Str, Count, Rect, Format,
         Font3d.Spacing, ColorToRGB(Font3d.Color), Font3d.Position);
   if font3Dcreated then
      font3D.Free;
end;

function DrawShadowText(Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; ShadowSize: Byte; ShadowColor: TColorRef;
   ShadowPos: TShadowPosition): Integer;
var
   RText, RShadow: TRect;
   Color: TColorRef;
begin
   RText := Rect;
   if ShadowSize > 0 then begin
      RShadow := Rect;
      case ShadowPos of
         spLeftTop:
            OffsetRect(RText, ShadowSize, ShadowSize);
         spRightBottom:
            OffsetRect(RShadow, ShadowSize, ShadowSize);
         spLeftBottom: begin
            OffsetRect(RText, ShadowSize, 0);
            OffsetRect(RShadow, 0, ShadowSize);
         end;
         spRightTop: begin
            OffsetRect(RText, 0, ShadowSize);
            OffsetRect(RShadow, ShadowSize, 0);
         end;
      end; { case }
      Color := SetTextColor(Canvas.Handle, ShadowColor);
      DrawText(Canvas.Handle, Str, Count, RShadow, Format);
      SetTextColor(Canvas.Handle, Color);
   end;
   Result := DrawText(Canvas.Handle, Str, Count, RText, Format);
   if Result > 0 then
      Inc(Result, ShadowSize);
   if (Format and (0 xor DT_CALCRECT)) > 0 then
      UnionRect(Rect, RText, RShadow);
end;

function DrawFlashText( Canvas: TCanvas; Str: PChar; Count: Integer; var Rect: TRect;
   Format: Word; ShadowSize: Byte; ShadowColor: TColorRef): Integer;
var
   Color: TColorRef;
   RShadow: TRect;
   x, y: Integer;
begin
   Result := 0;
   Color := SetTextColor(Canvas.Handle, ShadowColor);
   for x := 1 to ShadowSize do begin
      for y := 1 to ShadowSize do begin
         RShadow := Rect;
         OffsetRect( RShadow, x, y);
         Result := DrawText( Canvas.Handle, Str, Count, RShadow, Format);
         if Result > 0 then begin
            RShadow := Rect;
            OffsetRect( RShadow, x, -y);
            Result := DrawText( Canvas.Handle, Str, Count, RShadow, Format);
         end;
         if Result > 0 then begin
            RShadow := Rect;
            OffsetRect( RShadow, -x, y);
            Result := DrawText( Canvas.Handle, Str, Count, RShadow, Format);
         end;
         if Result > 0 then begin
            RShadow := Rect;
            OffsetRect( RShadow, -x, -y);
            Result := DrawText( Canvas.Handle, Str, Count, RShadow, Format);
         end;
      end;
   end;

   if Result > 0 then
      Inc(Result, 2 * ShadowSize);

   SetTextColor(Canvas.Handle, Color);
   DrawText(Canvas.Handle, Str, Count, Rect, Format);
   if (Format and (0 xor DT_CALCRECT)) > 0 then
      InflateRect(Rect, ShadowSize, ShadowSize);
end;

{
   From RX.
}

procedure StretchBitmapTransparent(Dest: TCanvas; Bitmap: TBitmap;
  TransparentColor: TColor; DstX, DstY, DstW, DstH, SrcX, SrcY,
  SrcW, SrcH: Integer);
var
  CanvasChanging: TNotifyEvent;
  Temp: TBitmap;
begin
  if DstW <= 0 then DstW := Bitmap.Width;
  if DstH <= 0 then DstH := Bitmap.Height;
  if (SrcW <= 0) or (SrcH <= 0) then begin
    SrcX := 0; SrcY := 0;
    SrcW := Bitmap.Width;
    SrcH := Bitmap.Height;
  end;
  if not Bitmap.Monochrome then
    SetStretchBltMode(Dest.Handle, STRETCH_DELETESCANS);
  CanvasChanging := Bitmap.Canvas.OnChanging;
  try
    Bitmap.Canvas.OnChanging := nil;
{$IFDEF RX_D3}
    {if Bitmap.HandleType = bmDIB then begin
      Temp := TBitmap.Create;
      Temp.Assign(Bitmap);
      Temp.HandleType := bmDDB;
    end
    else} Temp := Bitmap;
{$ELSE}
    Temp := Bitmap;
{$ENDIF}
    try
      if TransparentColor = clNone then begin
        StretchBlt(Dest.Handle, DstX, DstY, DstW, DstH, Temp.Canvas.Handle,
          SrcX, SrcY, SrcW, SrcH, Dest.CopyMode);
      end
      else begin
{$IFDEF RX_D3}
        if TransparentColor = clDefault then
          TransparentColor := Temp.Canvas.Pixels[0, Temp.Height - 1];
{$ENDIF}
        if Temp.Monochrome then TransparentColor := clWhite
        else TransparentColor := ColorToRGB(TransparentColor);
        StretchBltTransparent(Dest.Handle, DstX, DstY, DstW, DstH,
          Temp.Canvas.Handle, SrcX, SrcY, SrcW, SrcH, Temp.Palette,
          TransparentColor);
      end;
    finally
{$IFDEF RX_D3}
      {if Bitmap.HandleType = bmDIB then Temp.Free;}
{$ENDIF}
    end;
  finally
    Bitmap.Canvas.OnChanging := CanvasChanging;
  end;
end;

procedure StretchBltTransparent(DstDC: HDC; DstX, DstY, DstW, DstH: Integer;
  SrcDC: HDC; SrcX, SrcY, SrcW, SrcH: Integer; Palette: HPalette;
  TransparentColor: TColorRef);
var
  Color: TColorRef;
  bmAndBack, bmAndObject, bmAndMem, bmSave: HBitmap;
  bmBackOld, bmObjectOld, bmMemOld, bmSaveOld: HBitmap;
  MemDC, BackDC, ObjectDC, SaveDC: HDC;
  palDst, palMem, palSave, palObj: HPalette;
begin
  { Create some DCs to hold temporary data }
  BackDC := CreateCompatibleDC(DstDC);
  ObjectDC := CreateCompatibleDC(DstDC);
  MemDC := CreateCompatibleDC(DstDC);
  SaveDC := CreateCompatibleDC(DstDC);
  { Create a bitmap for each DC }
  bmAndObject := CreateBitmap(SrcW, SrcH, 1, 1, nil);
  bmAndBack := CreateBitmap(SrcW, SrcH, 1, 1, nil);
  bmAndMem := CreateCompatibleBitmap(DstDC, DstW, DstH);
  bmSave := CreateCompatibleBitmap(DstDC, SrcW, SrcH);
  { Each DC must select a bitmap object to store pixel data }
  bmBackOld := SelectObject(BackDC, bmAndBack);
  bmObjectOld := SelectObject(ObjectDC, bmAndObject);
  bmMemOld := SelectObject(MemDC, bmAndMem);
  bmSaveOld := SelectObject(SaveDC, bmSave);
  { Select palette }
  palDst := 0; palMem := 0; palSave := 0; palObj := 0;
  if Palette <> 0 then begin
    palDst := SelectPalette(DstDC, Palette, True);
    RealizePalette(DstDC);
    palSave := SelectPalette(SaveDC, Palette, False);
    RealizePalette(SaveDC);
    palObj := SelectPalette(ObjectDC, Palette, False);
    RealizePalette(ObjectDC);
    palMem := SelectPalette(MemDC, Palette, True);
    RealizePalette(MemDC);
  end;
  { Set proper mapping mode }
  SetMapMode(SrcDC, GetMapMode(DstDC));
  SetMapMode(SaveDC, GetMapMode(DstDC));
  { Save the bitmap sent here }
  BitBlt(SaveDC, 0, 0, SrcW, SrcH, SrcDC, SrcX, SrcY, SRCCOPY);
  { Set the background color of the source DC to the color,         }
  { contained in the parts of the bitmap that should be transparent }
  Color := SetBkColor(SaveDC, ColorToRGB(TransparentColor) or PaletteMask);
  { Create the object mask for the bitmap by performing a BitBlt()  }
  { from the source bitmap to a monochrome bitmap                   }
  BitBlt(ObjectDC, 0, 0, SrcW, SrcH, SaveDC, 0, 0, SRCCOPY);
  { Set the background color of the source DC back to the original  }
  SetBkColor(SaveDC, Color);
  { Create the inverse of the object mask }
  BitBlt(BackDC, 0, 0, SrcW, SrcH, ObjectDC, 0, 0, NOTSRCCOPY);
  { Copy the background of the main DC to the destination }
  BitBlt(MemDC, 0, 0, DstW, DstH, DstDC, DstX, DstY, SRCCOPY);
  { Mask out the places where the bitmap will be placed }
  StretchBlt(MemDC, 0, 0, DstW, DstH, ObjectDC, 0, 0, SrcW, SrcH, SRCAND);
  { Mask out the transparent colored pixels on the bitmap }
  BitBlt(SaveDC, 0, 0, SrcW, SrcH, BackDC, 0, 0, SRCAND);
  { XOR the bitmap with the background on the destination DC }
  StretchBlt(MemDC, 0, 0, DstW, DstH, SaveDC, 0, 0, SrcW, SrcH, SRCPAINT);
  { Copy the destination to the screen }
  BitBlt(DstDC, DstX, DstY, DstW, DstH, MemDC, 0, 0,
    SRCCOPY);
  { Restore palette }
  if Palette <> 0 then begin
    SelectPalette(MemDC, palMem, False);
    SelectPalette(ObjectDC, palObj, False);
    SelectPalette(SaveDC, palSave, False);
    SelectPalette(DstDC, palDst, True);
  end;
  { Delete the memory bitmaps }
  DeleteObject(SelectObject(BackDC, bmBackOld));
  DeleteObject(SelectObject(ObjectDC, bmObjectOld));
  DeleteObject(SelectObject(MemDC, bmMemOld));
  DeleteObject(SelectObject(SaveDC, bmSaveOld));
  { Delete the memory DCs }
  DeleteDC(MemDC);
  DeleteDC(BackDC);
  DeleteDC(ObjectDC);
  DeleteDC(SaveDC);
end;


procedure StretchBitmapRectTransparent(Dest: TCanvas; DstX, DstY,
  DstW, DstH: Integer; SrcRect: TRect; Bitmap: TBitmap;
  TransparentColor: TColor);
begin
  with SrcRect do
    StretchBitmapTransparent(Dest, Bitmap, TransparentColor,
    DstX, DstY, DstW, DstH, Left, Top, Right - Left, Bottom - Top);
end;

procedure DrawTransparentBitmapRect(DC: HDC; Bitmap: HBitmap; DstX, DstY,
   DstW, DstH: Integer; SrcRect: TRect; TransparentColor: TColorRef);
var
   hdcTemp: HDC;
begin
   hdcTemp := CreateCompatibleDC(DC);
   try
      SelectObject(hdcTemp, Bitmap);
      with SrcRect do
         StretchBltTransparent(DC, DstX, DstY, DstW, DstH, hdcTemp,
            Left, Top, Right - Left, Bottom - Top, 0, TransparentColor);
   finally
      DeleteDC(hdcTemp);
   end;
end;

procedure DrawBitmapTransparent(Dest: TCanvas; DstX, DstY: Integer;
  Bitmap: TBitmap; TransparentColor: TColor);
begin
   StretchBitmapTransparent(Dest, Bitmap, TransparentColor, DstX, DstY,
      Bitmap.Width, Bitmap.Height, 0, 0, Bitmap.Width, Bitmap.Height);
end;

end.
