{*******************************************************************************
   Unit
      sGlyphsList.pas
   Description:
      Glyphs container component
   Versions:
   	1.0
   History:
      1.0	- 	16/11/1998 - 18/11/1998.
      			Initial release
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   Comments:
      This class was designed in oredr to provide the simple holder for
      the glyphs used by applications. It can hold any type of the graphic.
*******************************************************************************}

unit sGlyphsList;

interface

uses
   Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
   StdCtrls, Buttons, ExtDlgs, ExtCtrls, ComCtrls, StdUtils, ToolWin, DsgnIntf,
   Menus, sConsts;

type
   TsGlyphCollection = class;
   TsGlyphList = class;
   EGlyphCollectionError = class(Exception);

	TsGlyph = class(TCollectionItem)
   private
   	FPicture: TPicture;
      FGlyphId: Integer;
      procedure SetPicture(Value: TPicture);
      procedure SetGlyphId(Value: Integer);
	protected
      function GetDisplayName: string; override;
      procedure OnChange(Sender: TObject);
   public
   	constructor Create(Collection: TCollection); override;
      destructor Destroy; override;
      procedure Assign(Source: TPersistent); override;
   published
      property GlyphId: Integer read FGlyphId write SetGlyphId;
      property Picture: TPicture read FPicture write SetPicture;
   end;

   TsGlyphCollection = class(TCollection)
   private
      FGlyphList: TsGlyphList;
      function GetItem(Index: Integer): TsGlyph;
      procedure SetItem(Index: Integer; Value: TsGlyph);
   protected
	   function GetOwner: TPersistent; override;
    	procedure Update(Item: TCollectionItem); override;
      function FindItem(Id: Integer): Integer;
      procedure CheckGlyphId(Value: Integer);
      function NewId: Integer;
      procedure Exchange(i1, i2: Integer);
      procedure SendNotification(oldGlyphId, newGlyphId: Integer);
   public
      constructor Create(AOwner: TsGlyphList);
      property GlyphList: TsGlyphList read FGlyphList;
	   property Items[Index: Integer]: TsGlyph read GetItem write SetItem; default;
   end;

   TsGlyphList = class(TComponent)
   private
      FItems: TsGlyphCollection;
      FNotifyControls: TList;
      FChanging: Boolean;
      function GetGlyph(id: Integer): TGraphic;
      procedure SetGlyph(id: Integer; Value: TGraphic);
      procedure SetList(Value: TsGlyphCollection);
   protected
    	procedure Notification(AComponent: TComponent; Operation: TOperation); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure Assign(Source: TPersistent); override;
      procedure ChangeNotification(Component: TComponent; add: Boolean);
      property Glyph[id: Integer]: TGraphic read GetGlyph write SetGlyph; default;
   published
      property Items: TsGlyphCollection read FItems write SetList;
   end;


   TsGraphicListControlChild = class(TPersistent)
   protected
      function GetGlyphList: TsGlyphList; virtual; abstract;
   end;

   TGlyphsListForm = class(TForm)
      StatusBar1: TStatusBar;
      ToolBar1: TToolBar;
      AddBtn: TToolButton;
      RemoveBtn: TToolButton;
      ImageList1: TImageList;
      GlyphList: TListBox;
      OkBtn: TToolButton;
      CancelBtn: TToolButton;
      ToolButton1: TToolButton;
      ToolButton2: TToolButton;
      ChangeBtn: TToolButton;
    	EditGlyphIdBtn: TToolButton;
      PopupMenu: TPopupMenu;
      Editpicture1: TMenuItem;
    	EditGlyphId1: TMenuItem;
      N1: TMenuItem;
      Newglyph1: TMenuItem;
      Deleteglyph1: TMenuItem;
      procedure AddBtnClick(Sender: TObject);
      procedure RemoveBtnClick(Sender: TObject);
      procedure GlyphListMeasureItem(Control: TWinControl; Index: Integer; var Height: Integer);
      procedure GlyphListDrawItem(Control: TWinControl; Index: Integer; Rect: TRect; State: TOwnerDrawState);
      procedure FormCreate(Sender: TObject);
      procedure FormDestroy(Sender: TObject);
      procedure GlyphListClick(Sender: TObject);
      procedure FormResize(Sender: TObject);
      procedure OkBtnClick(Sender: TObject);
      procedure CancelBtnClick(Sender: TObject);
      procedure GlyphListDblClick(Sender: TObject);
      procedure ChangeBtnClick(Sender: TObject);
      procedure EditGlyphIdBtnClick(Sender: TObject);
      procedure GlyphListDragOver(Sender, Source: TObject; X, Y: Integer;
         State: TDragState; var Accept: Boolean);
      procedure GlyphListDragDrop(Sender, Source: TObject; X, Y: Integer);
      procedure GlyphListMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
   private
      FSelect: Boolean;
      FImage: TImage;
      FGlList: TsGlyphList;
      FDeletedGlyphs: TList;
      FGlyphEditor: TPropertyEditor;
      procedure AssignGlyphList(GlList: TsGlyphList);
      procedure ApplyGlyphList(GlList: TsGlyphList);
      procedure LocateItem(id: Integer);
      procedure CheckGlyphEditor;
      procedure GetPropEditProc(Prop: TPropertyEditor);
      procedure ReadFormSettings;
      procedure WriteFormSettings;
   end;

   TGlyphsListEditor = class(TComponentEditor)
   public
      procedure ExecuteVerb(Index: Integer); override;
      function GetVerb(Index: Integer): string; override;
      function GetVerbCount: Integer; override;
      procedure Edit; override;
   end;

   TGlyphListProperty = class(TPropertyEditor)
   public
      procedure Edit; override;
      function GetAttributes: TPropertyAttributes; override;
   end;

   TGlyphIdProperty = class(TIntegerProperty)
   public
      procedure Edit; override;
      function GetAttributes: TPropertyAttributes; override;
   end;

procedure Register;

implementation
{$R *.DFM}

uses TypInfo, Consts, registry;

//******************* TsGlyph *****************************

constructor TsGlyph.Create(Collection: TCollection);
begin
	FPicture := TPicture.Create;
   FGlyphId := TsGlyphCollection(Collection).NewId;
   inherited;
end;

destructor TsGlyph.Destroy;
begin
   TsGlyphCollection(Collection).SendNotification(GlyphId, -1);
   FPicture.OnChange := nil;
	FPicture.Free;
	inherited;
end;

procedure TsGlyph.Assign(Source: TPersistent);
begin
   if not (Source is TsGlyph) then
      Raise EConvertError.CreateFmt( SAssignError, [Source.ClassName, ClassName]);
   FPicture.Assign(TsGlyph(Source).FPicture);
   FGlyphId := TsGlyph(Source).FGlyphId;
end;

function TsGlyph.GetDisplayName: string;
begin
 	if FGlyphId > -1 then
  		Result := IntToStr(FGlyphId)
  	else
  		Result := inherited GetDisplayName;
end;

procedure TsGlyph.SetPicture(Value: TPicture);
begin
	FPicture.Assign(Value);
end;

procedure TsGlyph.OnChange(Sender: TObject);
begin
   TsGlyphCollection(Collection).SendNotification(FGlyphId, FGlyphId);
end;

procedure TsGlyph.SetGlyphId(Value: Integer);
var
	oldGlyphId: Integer;
begin
	if FGlyphId <> Value then begin
		TsGlyphCollection(Collection).CheckGlyphId(Value);
      oldGlyphId := FGlyphId;
		FGlyphId := Value;
      TsGlyphCollection(Collection).SendNotification(oldGlyphId, FGlyphId);
   end;
end;

//******************* TsGlypCollection ********************
constructor TsGlyphCollection.Create(AOwner: TsGlyphList);
begin
   inherited Create(TsGlyph);
  	FGlyphList := AOwner;
end;

function TsGlyphCollection.GetItem(Index: Integer): TsGlyph;
begin
	Result := TsGlyph(inherited GetItem(Index));
end;

procedure TsGlyphCollection.SetItem(Index: Integer; Value: TsGlyph);
begin
	inherited SetItem(Index, Value);
end;

function TsGlyphCollection.GetOwner: TPersistent;
begin
  Result := FGlyphList;
end;

procedure TsGlyphCollection.Update(Item: TCollectionItem);
begin
end;

function TsGlyphCollection.FindItem(Id: Integer): Integer;
var
	ii: Integer;
begin
	Result := -1;
	for ii := 0 to Count - 1 do
   	if Items[ii].GlyphId = Id then begin
      	Result := ii;
			Exit;
      end;
end;

procedure TsGlyphCollection.CheckGlyphId(Value: Integer);
begin
	if FindItem(Value) > - 1 then
   	Raise EGlyphCollectionError.CreateFmt(SErrorDuplicateId, [Value]);
end;

procedure TsGlyphCollection.Exchange(i1, i2: Integer);
var
	glyph: TPicture;
   gid: Integer;
begin
	glyph := TPicture.Create;
   try
   	glyph.Assign(Items[i1].Picture);
   	gid := Items[i1].FGlyphId;
		Items[i1].Picture.Assign(Items[i2].Picture);
     	Items[i1].FGlyphId := Items[i2].FGlyphId;
		Items[i2].Picture.Assign(Glyph);
     	Items[i2].FGlyphId := gid;
   finally
   	Glyph.Free;
   end;
end;

function TsGlyphCollection.NewId: Integer;
begin
	Result := 1;
   while FindItem(Result) <> -1 do
   	Inc(Result);
end;

procedure TsGlyphCollection.SendNotification(oldGlyphId, newGlyphId: Integer);
var
	ii: Integer;
begin
   with FGlyphList do
      if not FChanging and (FNotifyControls <> nil) then begin
		   for ii := 0 to FNotifyControls.Count - 1 do
            if Assigned(FNotifyControls[ii]) then
			      TControl(FNotifyControls[ii]).Perform(
                  STM_GLYPHIDCHANGED, oldGlyphId, newGlyphId);
      end;
end;

//******************* TsGlyphList *************************

constructor TsGlyphList.Create(AOwner: TComponent);
begin
   inherited;
   FItems := TsGlyphCollection.Create(self);
end;

destructor TsGlyphList.Destroy;
begin
   FNotifyControls.Free;
   FNotifyControls := nil;
   FItems.Free;
   inherited;
end;

procedure TsGlyphList.Assign(Source: TPersistent);
begin
   if not (Source is TsGlyphList) then
      Raise EConvertError.CreateFmt( SAssignError, [Source.ClassName, ClassName]);
   FChanging := TRUE;
	FItems.Assign(TsGlyphList(Source).Items);
   FChanging := FALSE;
end;

function TsGlyphList.GetGlyph(id: Integer): TGraphic;
var
	index: Integer;
begin
   Result := nil;
	index := FItems.FindItem(id);
   if index > -1 then
		Result := FItems[index].Picture.Graphic
//   else
//   	raise EGlyphCollectionError.CreateFmt(SErrorGlyphNotFound, [id]);
end;

procedure TsGlyphList.SetGlyph(id: Integer; Value: TGraphic);
var
	index: Integer;
begin
	index := FItems.FindItem(id);
   if index > -1 then
		FItems[index].Picture.Assign(Value)
   else
   	raise EGlyphCollectionError.CreateFmt(SErrorGlyphNotFound, [id]);
end;

procedure TsGlyphList.SetList(Value: TsGlyphCollection);
begin
	FItems.Assign(Value);
end;

procedure TsGlyphList.Notification(AComponent: TComponent; Operation: TOperation);
var
	index: integer;
begin
	inherited;
   if (FNotifyControls <> nil) and (Operation = opRemove) then begin
   	index := FNotifyControls.IndexOf(AComponent);
      if index > -1 then
			FNotifyControls.Delete(index);
   end;
end;

procedure TsGlyphList.ChangeNotification(Component: TComponent; add: Boolean);
var
   index: Integer;
begin
   if FNotifyControls = nil then begin
      if Add then
		   FNotifyControls := TList.Create
      else
         Exit;
   end;
   index := FNotifyControls.IndexOf(Component);
   if (index = -1) and add then
      FNotifyControls.Add(Component)
   else if (index <> -1) and not add then
      FNotifyControls.Delete(index);
end;

//******************* TGlyphsListForm *************************

procedure TGlyphsListForm.FormCreate(Sender: TObject);
begin
   FGlList := TsGlyphList.Create(self);
   FDeletedGlyphs := TList.Create;
   ReadFormSettings;
end;

procedure TGlyphsListForm.FormDestroy(Sender: TObject);
begin
   WriteFormSettings;
   FGlList.Free;
   FDeletedGlyphs.Free;
   FImage.Free;
end;

procedure TGlyphsListForm.AssignGlyphList(GlList: TsGlyphList);
var
	ii: Integer;
begin
   FGlList.Assign(GlList);
   GlyphList.Items.Clear;
	for ii := 0 to GlList.Items.Count - 1 do with GlyphList do
      Items.AddObject(IntToStr(GlList.Items[ii].GlyphId), TObject(GlList.Items[ii].GlyphId));
end;

procedure TGlyphsListForm.ApplyGlyphList(GlList: TsGlyphList);
var
	ii, id: Integer;
begin
   GlList.Assign(FGlList);
   // notify controls about the changes
   for ii := 0 to GlyphList.Items.Count - 1 do begin
      id := Integer(GlyphList.Items.Objects[ii]);
      if id <> 0 then begin// do not notify about the new items...
         if Id <> FGlList.Items[ii].GlyphId then begin// Glyph id was changed
            GlList.Items.SendNotification(Id, FGlList.Items[ii].GlyphId)
         end else if GlyphList.Items[ii][1] = '*' then
            GlList.Items.SendNotification(Id, Id)
      end;
   end;
   for ii := 0 to FDeletedGlyphs.Count - 1 do
      GlList.Items.SendNotification(Integer(FDeletedGlyphs[ii]), -1);
end;

procedure TGlyphsListForm.LocateItem(id: Integer);
var
	ii: Integer;
begin
	for ii := 0 to FGlList.Items.Count - 1 do
     if FGlList.Items[ii].GlyphId = id then begin
			GlyphList.ItemIndex := ii;
         Exit;
     end;
end;

procedure TGlyphsListForm.CheckGlyphEditor;
var
   comps: TComponentList;
begin
   if FGlyphEditor = nil then begin
      FImage := TImage.Create(self);
      FImage.Parent := self;
      comps := TComponentList.Create;
      try
         comps.Add(FImage);
         GetComponentProperties(comps, [tkClass], TFormDesigner(Designer), GetPropEditProc);
      finally
         comps.Free;
      end;
   end;
end;

procedure TGlyphsListForm.GetPropEditProc(Prop: TPropertyEditor);
begin
   if Prop.GetName = 'Picture' then
      FGlyphEditor := Prop;
end;

procedure TGlyphsListForm.OkBtnClick(Sender: TObject);
begin
   ModalResult := mrOk;
end;

procedure TGlyphsListForm.CancelBtnClick(Sender: TObject);
begin
   ModalResult := mrCancel;
end;

procedure TGlyphsListForm.AddBtnClick(Sender: TObject);
var
   i: Integer;
begin
   CheckGlyphEditor;
   FImage.Picture.Graphic := nil;
   try
      FGlyphEditor.Edit;
   except
   end;
   with FImage do if (Picture <> nil) and (Picture.Graphic <> nil) and
      (not Picture.Graphic.Empty) then begin
		with TsGlyph(FGlList.Items.Add) do begin
         SetPicture(FImage.Picture);
         i := GlyphId;
      end;
      GlyphList.Items.Add(IntToStr(i));
  	end;
end;

procedure TGlyphsListForm.RemoveBtnClick(Sender: TObject);
begin
   if Integer(GlyphList.Items.Objects[GlyphList.ItemIndex]) <> 0 then
      FDeletedGlyphs.Add(TObject(FGlList.Items[GlyphList.ItemIndex].GlyphId));
   FGlList.Items[GlyphList.ItemIndex].Free;
   GlyphList.Items.Delete(GlyphList.ItemIndex);
end;

procedure TGlyphsListForm.ChangeBtnClick(Sender: TObject);
begin
   if GlyphList.ItemIndex > -1 then begin
      CheckGlyphEditor;
      FImage.Picture.Assign(FGlList.Items[GlyphList.ItemIndex].Picture);
      FGlyphEditor.Edit;
      FGlList.Items[GlyphList.ItemIndex].Picture.Assign(FImage.Picture.Graphic);
      // mark modified with *
      with GlyphList do if (Integer(Items.Objects[ItemIndex]) <> 0) and
         (Items[ItemIndex][1] <> '*') then
         Items[ItemIndex] := '*' + Items[ItemIndex];
   end;
end;

procedure TGlyphsListForm.EditGlyphIdBtnClick(Sender: TObject);
var
   S: string;
   id: Integer;
begin
   if GlyphList.ItemIndex > -1 then begin
      S := GlyphList.Items[GlyphList.ItemIndex];
      if not InputQuery( SChangeGlyphIdTitles, SChangeGlyphIdCaption, S) then
         Exit;
      id := StrToInt(S);
      if id < 1 then
         Raise Exception.Create(SErrorInvalidGlyphId);
      FGlList.Items[GlyphList.ItemIndex].GlyphId := id;
      GlyphList.Items[GlyphList.ItemIndex] := S;
   end;
end;

procedure TGlyphsListForm.GlyphListMeasureItem(Control: TWinControl;
   Index: Integer; var Height: Integer);
begin
   Height := GetFontHeight(Canvas.Font) + FGlList.Items[Index].Picture.Height + 4;
end;

procedure TGlyphsListForm.GlyphListDrawItem(Control: TWinControl;
   Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
   graphic: TGraphic;
   aLeft: Integer;
begin
   graphic := FGlList.Items[Index].Picture.Graphic;
   GlyphList.Canvas.FillRect(Rect);
   aLeft := Rect.Left + 25;
   if WidthOf(Rect) - 25 > graphic.Width then
      aLeft := (WidthOf(Rect) - graphic.Width) div 2;
   GlyphList.Canvas.TextOut(Rect.Left + 2,
      Rect.Top, Format( '%s, [%s]', [GlyphList.Items[Index], graphic.ClassName]));
   OffsetRect(Rect, 0, GetFontHeight(Canvas.Font));
   if Graphic is TBitMap then
      GlyphList.Canvas.BrushCopy( Bounds(aLeft, Rect.Top + 1, graphic.Width, graphic.Height),
   		TBitMap(graphic), Bounds( 0, 0, graphic.Width, graphic.Height),
         TBitMap(graphic).TransparentColor)
   else
      GlyphList.Canvas.Draw(aLeft, Rect.Top + 2, graphic);
end;

procedure TGlyphsListForm.GlyphListClick(Sender: TObject);
begin
   RemoveBtn.Enabled := GlyphList.ItemIndex > -1;
   ChangeBtn.Enabled := RemoveBtn.Enabled;
   EditGlyphIdBtn.Enabled := RemoveBtn.Enabled;
   Editpicture1.Enabled := RemoveBtn.Enabled;
   EditGlyphId1.Enabled := RemoveBtn.Enabled;
   Deleteglyph1.Enabled := RemoveBtn.Enabled;
end;

procedure TGlyphsListForm.GlyphListDblClick(Sender: TObject);
begin
   if FSelect then
   	ModalResult := mrOk
   else
   	ChangeBtnClick(Sender);
end;

procedure TGlyphsListForm.FormResize(Sender: TObject);
begin
   GlyphList.Invalidate;
end;

procedure TGlyphsListForm.GlyphListMouseMove(Sender: TObject;
   Shift: TShiftState; X, Y: Integer);
begin
   if (ssLeft in Shift) and not GlyphList.Dragging and (GlyphList.ItemIndex > -1) then
      GlyphList.BeginDrag(FALSE);
end;

procedure TGlyphsListForm.GlyphListDragOver(Sender, Source: TObject; X,
   Y: Integer; State: TDragState; var Accept: Boolean);
begin
   if Source = self then
      Accept := TRUE;
end;

procedure TGlyphsListForm.GlyphListDragDrop(Sender, Source: TObject; X, Y: Integer);
var
   index: Integer;
   begin
   with GlyphList do begin
      index := ItemAtPos(Point(X, Y), FALSE);
      if index >= Items.Count then
         Dec(index);
      FGlList.Items.Exchange(ItemIndex, index);
      Items.Exchange(ItemIndex, index);
      ItemIndex := Index;
   end;
end;

procedure TGlyphsListForm.ReadFormSettings;
begin
   with TRegIniFile.Create(RegistryDelphiPath + SGlyphListEditorRegistryEntry) do try
      WindowState  := TWindowState( ReadInteger('', RegEntryFormState, Ord(wsNormal)));
      if WindowState = wsNormal then begin
         Left := ReadInteger('', RegEntryFormLeft, Left);
         Top := ReadInteger('', RegEntryFormTop, Top);
         Width := ReadInteger('', RegEntryFormWidth, Width);
         Height := ReadInteger('', RegEntryFormHeight, Height)
      end;
   finally
      Free;
   end;
end;

procedure TGlyphsListForm.WriteFormSettings;
begin
   with TRegIniFile.Create(RegistryDelphiPath + SGlyphListEditorRegistryEntry) do try
   	WriteInteger('', RegEntryFormState, Ord(WindowState));
      if WindowState = wsNormal then begin
         WriteInteger('', RegEntryFormLeft, Left);
         WriteInteger('', RegEntryFormTop, Top);
         WriteInteger('', RegEntryFormWidth, Width);
         WriteInteger('', RegEntryFormHeight, Height)
      end;
   finally
      Free;
   end;
end;

//******************* TGlyphsListEditor *************************

procedure TGlyphsListEditor.ExecuteVerb(Index: Integer);
begin
   if Index <> 0 then
      Exit;
   Edit;
end;

function TGlyphsListEditor.GetVerb(Index: Integer): AnsiString;
begin
   Result := SGlyphEditorCaption;
end;

function TGlyphsListEditor.GetVerbCount: Integer;
begin
   Result := 1;
end;

procedure TGlyphsListEditor.Edit;
begin
   with TGlyphsListForm.Create(nil) do try
      AssignGlyphList(TsGlyphList(Component));
      if ShowModal = mrOk then
         ApplyGlyphList(TsGlyphList(Component));
   finally
      Free;
   end;
end;

//******************* TGlyphsListProperty ***********************
procedure TGlyphListProperty.Edit;
var
	cmp: TsGlyphList;
begin
	cmp := GetComponent(0) as TsGlyphList;
   with TGlyphsListForm.Create(nil) do try
      AssignGlyphList(cmp);
      if ShowModal = mrOk then
         ApplyGlyphList(cmp);
   finally
      Free;
   end;
end;

function TGlyphListProperty.GetAttributes: TPropertyAttributes;
begin
	Result := [paDialog, paRevertable];
end;

//******************* TGlyphsIdProperty ************************
procedure TGlyphIdProperty.Edit;
var
	cmp: TPersistent;
   glList: TsGlyphList;
   PropInfo: PPropInfo;
   id: Integer;
begin
   glList := nil;
   cmp := GetComponent(0) as TPersistent;
   if cmp.InheritsFrom(TsGraphicListControlChild) then
      glList := TsGraphicListControlChild(cmp).GetGlyphList
   else begin
      PropInfo := TypInfo.GetPropInfo( cmp.ClassInfo, 'GlyphList');
      if PropInfo <> nil then
	      glList := TsGlyphList(TypInfo.GetOrdProp(cmp, PropInfo));
   end;
   if glList <> nil then with TGlyphsListForm.Create(nil) do try
      AssignGlyphList(glList);
      id := GetOrdValue;
      if id > -1 then
         LocateItem(id);
      FSelect := TRUE;
      if ShowModal = mrOk then begin
         ApplyGlyphList(GlList);
         id := FGlList.Items[GlyphList.ItemIndex].GlyphId;
         SetOrdValue(id);
      end;
   finally
      Free;
   end;
end;

function TGlyphIdProperty.GetAttributes: TPropertyAttributes;
begin
	Result := [paDialog, paRevertable];
end;


procedure Register;
begin
   RegisterComponents('TsEditTools', [TsGlyphList]);
   RegisterComponentEditor(TsGlyphList, TGlyphsListEditor);
	RegisterPropertyEditor( TypeInfo(Integer), TPersistent, 'GlyphListId', TGlyphIdProperty);
//   RegisterPropertyEditor( TypeInfo(TsGlyphCollection), TsGlyphList, 'List', TGlyphListProperty);   
end;

initialization
finalization

end.

