{*******************************************************************************
   Unit
      sTrans.pas
   Description:
      TCustomControl which provide the transparency mechanizme for its child controls
   Versions:
      1.0
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com.
   Comments:
      TMDIWallpeper is not fully supported.
*******************************************************************************}

unit sTrans;

interface

uses
   Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
   StdCtrls, ExtCtrls;

type
   TsTransControl = class(TCustomControl)
   private
      FTransparent: Boolean;
      FBackground: TBitmap;
      FBackgroundChanged: Boolean;
      FSelfCall: Boolean;
      procedure SetTransparent(Value: Boolean);
      function GetLeftTopPoint: TPoint;
      procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
      procedure WMPaint(var Msg: TWMPaint); message WM_PAINT;
      procedure WMNCPaint(var Msg: TWMPaint); message WM_NCPAINT;
      procedure WMMove(var Msg: TMessage); message WM_MOVE;
      procedure WMSize(var Msg: TMessage); message WM_SIZE;
   protected
      procedure Paint; override;
      procedure SaveBackground;
      property Transparent: Boolean read FTransparent write SetTransparent default FALSE;
      property BackgroundChanged: Boolean read FBackgroundChanged;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure PaintBackground;
      procedure PaintBackgroundRect(R: TRect);
      procedure FrameRect(R: TRect; penWidth: Integer);
      procedure DrawFlatEDGE(R: TRect; Down: Boolean);
   end;

implementation

{$I S.Inc}
uses sGraphics, sTdUtils
{$IFDEF MDIWALLPAPER_SUPPORT}
   ,MDIWallp
{$ENDIF}
;

constructor TsTransControl.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle - [csOpaque];
   FTransparent := FALSE;
   FBackground := TBitmap.Create;
   FBackgroundChanged := TRUE;
end;

destructor TsTransControl.Destroy;
begin
   FBackground.Free;
   inherited;
end;

procedure TsTransControl.SetTransparent(Value: Boolean);
begin
   if Value <> FTransparent then begin
      FTransparent := Value;
      Invalidate;
   end;
end;

{$IFDEF MDIWALLPAPER_SUPPORT}
type
   TComponentClass = class of TComponent;

function FindComponentClass(aOwner: TComponent; CompClass: TComponentClass): TComponent;
var
   ii: Integer;
begin
   Result := nil;
   for ii := 0 to aOwner.ComponentCount - 1 do
      if aOwner.Components[ii] is CompClass then begin
         Result := aOwner.Components[ii];
         Exit;
      end;
end;
{$ENDIF}


procedure TsTransControl.SaveBackground;
var
   paintDC, parentDC: HDC;
   oldBitmap, bitmap: HBITMAP;
   P: TPoint;
{$IFDEF MDIWALLPAPER_SUPPORT}
   cmp: TComponent;
{$ENDIF}
begin
   if ([csReading, csLoading] * ComponentState <> []) or (Parent = nil)
      or ([csReading, csLoading] * Parent.ComponentState <> [])
      or not HandleAllocated or not Visible then
      Exit;

   FBackground.Width := Width;
   FBackground.Height := Height;

{$IFDEF MDIWALLPAPER_SUPPORT}
   cmp := FindComponentClass(parent, TMDIWallpaper);
   if cmp <> nil then
      ParentDC := GetDC( TMDIWallpaper(cmp).ClientHandle)
   else
{$ENDIF}
      ParentDC := GetDC( Parent.Handle);
   paintDC := CreateCompatibleDC( ParentDC);
   bitmap := CreateCompatibleBitmap( parentDC, parent.width, parent.height);
   oldBitmap := SelectObject( paintDC, bitmap);

{$IFDEF MDIWALLPAPER_SUPPORT}
   if cmp <> nil then
      SendMessage( TMDIWallpaper(cmp).ClientHandle, WM_ERASEBKGND, paintDC, 0)
   else
{$ENDIF}
      Parent.PaintTo( paintDC, 0, 0);
   P := GetLeftTopPoint;
   BitBlt( FBackground.canvas.handle, 0, 0, width, height, paintDC, P.X, P.Y, SRCCOPY);

   ReleaseDC( Parent.Handle, parentDC);
   SelectObject( paintDC, oldBitmap);
   DeleteObject( bitmap);
   DeleteDC( paintDC);

   Dec(FBackgroundChanged);
end;

function TsTransControl.GetLeftTopPoint: TPoint;
begin
   Result := ClientToScreen(Point(0, 0));
   Result := Parent.ScreenToClient(Result);
end;

procedure TsTransControl.PaintBackground;
begin
   PaintBackgroundRect( Rect(0, 0, Width, Height));
end;

procedure TsTransControl.PaintBackgroundRect(R: TRect);
begin
   if FTransparent then begin
      BitBlt( Canvas.Handle, R.Left, R.Top, WidthOf(R), HeightOf(R),
         FBackground.Canvas.handle, R.Left, R.Top, srccopy);
   end else with Canvas do begin
      Brush.Style := bsSolid;
      Brush.Color := Color;
      FillRect(R);
   end;
end;

procedure TsTransControl.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
   if FTransparent then begin
//      if not FSelfCall and not (csPaintCopy in ControlState) then
//         FBackgroundChanged := TRUE;
      Msg.Result := 1
   end else
      inherited;
end;

procedure TsTransControl.WMNCPaint(var Msg: TWMPaint);
begin
   inherited;
end;

procedure TsTransControl.WMPaint(var Msg: TWMPaint);
var
   R: TRect;
   ps: TPaintStruct;
begin
   if csPaintCopy in ControlState then
      Exit;
   if FTransparent then begin
      Msg.Result := 0;
      GetUpdateRect( Handle, R, False);
      if IsRectEmpty(R) then
         Exit;
      BeginPaint( Handle, ps);
      Inc(FSelfCall);
      Paint;
      Dec(FSelfCall);
      EndPaint( Handle, ps);
   end else
      inherited;
end;

procedure TsTransControl.WMMove(var Msg: TMessage);
begin
   Inc(FBackgroundChanged);
   inherited;
   Invalidate;
end;

procedure TsTransControl.WMSize(var Msg: TMessage);
begin
   Inc(FBackgroundChanged);
   inherited;
   Invalidate;
end;

procedure TsTransControl.Paint;
begin
   if FTransparent and FBackgroundChanged then
      SaveBackground;
   inherited;
end;

procedure TsTransControl.FrameRect(R: TRect; penWidth: Integer);
var
   MemDC: HDC;
   aBitmap, oldObject: HBITMAP;
   ii: Integer;
begin
   if Transparent then begin
      InflateRect(R, -penWidth, -PenWidth);
      MemDC := CreateCompatibleDC( Canvas.Handle);
      aBitmap := CreateCompatibleBitmap( Canvas.Handle, R.Right - R.Left, R.Bottom - R.Top);
      oldObject := SelectObject( MemDC, aBitmap);
      BitBlt( MemDC, 0, 0, R.Right - R.Left, R.Bottom - R.Top, Canvas.Handle,
         R.Left, R.Top, SRCCOPY);
      InflateRect(R, penWidth, PenWidth);
      BitBlt( Canvas.Handle, R.left, R.Top, R.Right - R.Left, R.Bottom - R.Top, FBackground.Canvas.Handle,
         R.Left, R.Top, SRCCOPY);
      InflateRect(R, -penWidth, -PenWidth);
      BitBlt( Canvas.Handle, R.left, R.Top, R.Right - R.Left, R.Bottom - R.Top, MemDC, 0, 0, SRCCOPY);
      SelectObject( MemDC, oldObject);
      DeleteObject( aBitmap);
      DeleteDC( MemDC);
   end else for ii := 0 to penWidth do begin
      Canvas.Brush.Color := Color;
      Canvas.FrameRect(R);
      InflateRect(R, -1, -1);
   end;
end;

procedure TsTransControl.DrawFlatEDGE(R: TRect; Down: Boolean);
const
   Edge: array [Boolean] of Word = (EDGE_RAISED, EDGE_SUNKEN);
begin
   DrawEdge( Canvas.Handle, R, edge[Down], BF_RECT);
   InflateRect(R, -1, -1);
   FrameRect(R, 1);
end;

end.

