{*******************************************************************************
   Utit:
      sPickDate.pas
   Description:
      This unit implements 2 calendar controls - TsPopupCalendar and TsCalendar.
   Versions:
      2.0*
   Author(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com

*     I did not track the versions before, so let's consider it as 2.0
*******************************************************************************
Known problems:
   1. Slight Problem with appearance when transparent and borderstyle = bsSingle
Things to do:
   1. Multi select
   2. To support yaer calendar:
      a. Ability to choose visible buttons (at least left, right and year)
      b. BeforeReload event;
*******************************************************************************}

unit sPickDate;

interface

uses Windows, Messages, Classes, Controls, Graphics, Forms, StdCtrls, Menus,
   sDate, sGraphics, sTrans;

type

   TCalendarBevelStyle = (bsNone, bsLowered, bsRaised);

   TCalendarOptions = set of (coFullDisplay, coShowPopups, coMonthSwitch,
      coCancelIfOut, coShowWeeks, coShowToday, coShowTodayBtn);

   TsCalendarState = set of (cstPressed);

   TscBtnState = (cbsNormal, cbsUp, cbsDown);
   TscPopupBtn = (cpbNone, cpbLeft, cpbRight, cpbToday, cpbMonth, cpbYear);
   TscTrackState = (tsNone, tsLeft, tsRight, tsToday, tsDay);
   TSelectStyle = (ssNone, ssFrame, ssOval);

const
   CM_CloseEditor = WM_USER + 3;

type
   TsCustomCalendar = class;

   TsCalendarColors = class(TPersistent)
   private
      FOwner: TsCustomCalendar;
      FTitles: TColor;
      FWeekEnd: TColor;
      FInactive: TColor;
      FInactiveWeekEnd: TColor;
      FSelect: TColor;
      procedure SetColor(Index: Integer; Value: TColor);
   public
      constructor Create(AOwner: TsCustomCalendar);
      procedure Assign(Source: TPersistent); override;
   published
      property Titles: TColor index 0 read FTitles write SetColor default clWindowText;
      property WeekEnd: TColor index 1 read FWeekEnd write SetColor default clMaroon;
      property Inactive: TColor index 2 read FInactive write SetColor default cl3DLight;
      property InactiveWeekEnd: TColor index 3 read FInactiveWeekEnd write SetColor default clInfoBk;
      property Select: TColor index 4 read FSelect write SetColor default clHighlight;
   end;

  	TsCustomCalendar = class( TsTransControl)
   private
      FBevel: TCalendarBevelStyle;
   	FBorderStyle: TBorderStyle;
      FCaption: String;
    	FCellHeight: Integer;
  	   FCellWidth: Integer;
      FColors: TsCalendarColors;
      FDate: TsCalendarDate;
      FFont3d: TFont3d;
      FFontColor: TColor;
      FHilight3d: TFont3d;
      FHilightStyle: THilightStyle;
      FLastPopupBtn: TscPopupBtn;
      FPrevIndex: Integer;
      FPrevFocusIndex: Integer;
      FOptions: TCalendarOptions;
      FPrevHilightIndex: Integer;
      FSelectStyle: TSelectStyle;
      FState: TsCalendarState;
      FTodayCaption: String;
      FTrackState: TscTrackState;
      FWeekEnds: TWeekEnds;
      FYearPopup: TEdit;
      FhOffset: Integer;
      FwOffset: Integer;
      FvOffset: Integer;
      FOnAccept: TNotifyEvent;
      FOnCancel: TNotifyEvent;
      FOnMouseEnter: TNotifyEvent;
      FOnMouseLeave: TNotifyEvent;
      FBeforeDateChange: TDateChangeEvent;
      FOnDateChange: TNotifyEvent;
      procedure SetBevel(Value: TCalendarBevelStyle);
      procedure SetBorderStyle(Value: TBorderStyle);
      function GetBeginingOfWeek: TWeekDay;
      procedure SetBeginingOfWeek(Value: TWeekDay);
      procedure SetColors( Value: TscalendarColors);
      procedure SetFont3d(Value: TFont3d);
      procedure Font3dChanged(Sender: Tobject);
      procedure SetHilight3d(Value: TFont3d);
      procedure SetHilightStyle(Value: THilightStyle);
      procedure SetSelectStyle(Value: TSelectStyle);
      procedure SetOptions(Value: TCalendarOptions);
      procedure SetTodayCaption(Value: String);
      function TodayCaptionStored: Boolean;
      procedure SetWeekEnds(Value: TWeekEnds);
      procedure Track( X, Y: Integer);
      procedure StopTracking(X, Y: Integer);
      procedure SetCaption;
      procedure CallPopupYear(R: TRect);
      procedure CallPopupMonth(R: TRect);
    	function GetCalendarRect: TRect;
  	   function GetButtonRect( btn: TscPopupBtn): TRect;
    	function GetRectFromIndex(index: Integer): TRect;
    	function GetIndexFromPoint(nLeft: Integer ; nTop : Integer) : Integer;
      procedure DateChanged(Sender: TObject);
      function GetFontColor(index: Integer): TColor;
      procedure PaintBevel;
      procedure PaintHeader;
      procedure PaintButton( btn: TscPopupBtn; Down: Boolean);
      procedure PaintFooter;
      procedure PaintDates;
      procedure PaintSingleDay(R: TRect; index: integer; hilight: TTextHilightStyle);
      procedure PaintToday;
      procedure PaintFocusFrame(index: Integer; Force: Boolean);
      procedure PaintHilightFrame(index: Integer);
      procedure PaintHilightButton(P: TPoint);
      procedure PaintPopupBtn(R: TRect; state: TscBtnState);
      procedure InvalidateDates;
      procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
      procedure CMCloseEditor(var Message: TMessage); message CM_CloseEditor;
   protected
      procedure CreateParams(var Params: TCreateParams); override;
      procedure CreateWnd; override;
  		procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    	procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    	procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    	procedure Paint; override;
      procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    	procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
      procedure CalculateValues( var AHeight, AWidth: Integer); virtual;
      function GetTodayCaptionDefault: String;
      procedure Cancel; virtual;
      procedure Accept; virtual;
      procedure DoMouseEnter; virtual;
      procedure DoMouseLeave; virtual;
      property Bevel: TCalendarBevelStyle read FBevel write SetBevel default bsRaised;
      property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default TBorderStyle(bsNone);
      property BeginingOfWeek: TWeekDay read GetBeginingOfWeek write SetBeginingOfWeek default Mon;
      property Color default clBtnFace;
      property Colors: TscalendarColors read FColors write SetColors;
      property Font3d: TFont3d read FFont3d write SetFont3d;
      property Hilight3d: TFont3d read FHilight3d write SetHilight3d;
      property HilightStyle: THilightStyle read FHilightStyle write SetHilightStyle default hsFrame;
      property SelectStyle: TSelectStyle read FSelectStyle write SetSelectStyle default ssFrame;
      property Options: TCalendarOptions read FOptions write SetOptions default [coShowPopups];
      property TodayCaption: String read FTodayCaption write SetTodayCaption stored TodayCaptionStored;
      property Transparent;
      property WeekEnds: TWeekEnds read FWeekEnds write SetWeekEnds default [Sat, Sun];
   	property OnAccept: TNotifyEvent read FOnAccept write FOnAccept;
      property OnCancel: TNotifyEvent read FOnCancel write FOnCancel;
      property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
      property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
      property BeforeDateChange: TDateChangeEvent read FBeforeDateChange write FBeforeDateChange;
      property OnDateChange: TNotifyEvent read FOnDateChange write FOnDateChange;
  	public
    	constructor Create( AOwner: TComponent ); override;
      destructor Destroy; override;
      property Date: TsCalendarDate read FDate;
  	end;

   TsCalendar = class( TsCustomCalendar)
   protected
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
   public
      constructor Create( AOwner: TComponent ); override;
   published
      property BeginingOfWeek;
   	property Bevel;
      property BorderStyle;
      property Color;
      property Colors;
      property Enabled;
      property Font;
      property Font3d;
      property Hilight3d;
      property HilightStyle;
      property SelectStyle;
      property Options;
      property ParentShowHint;
    	property PopupMenu;
      property ShowHint;
    	property TabOrder;
    	property TabStop;
      property TodayCaption;
      property Transparent default FALSE;
    	property Visible;
      property WeekEnds;
      property OnEnter;
      property OnExit;
      property OnAccept;
      property OnCancel;
      property OnMouseEnter;
      property OnMouseLeave;
      property BeforeDateChange;
      property OnDateChange;
   end;

	TsPopupCalendar = class( TsCustomCalendar)
   private
      FCaller: TWinControl;
      FPopupPoint: TPoint;
      procedure CalculatePosition;
      procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS;
	protected
      procedure CalculateValues( var AHeight, AWidth: Integer); override;
      procedure Cancel; override;
      procedure Accept; override;
      procedure CreateParams(var Params: TCreateParams); override;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
   public
		constructor Create( AOwner: TComponent ); override;
      procedure Popup(X, Y: Integer);
   published
      property BeginingOfWeek;
   	property Bevel;
      property Caller: TWinControl read FCaller write FCaller;
      property Color;
      property Colors;
      property Enabled;
      property Font;
      property Font3d;
      property Hilight3d;
      property HilightStyle;
      property SelectStyle;
      property Options;
      property ParentShowHint;
    	property PopupMenu;
      property ShowHint;
    	property TabOrder;
    	property TabStop;
      property TodayCaption;
      property Transparent default FALSE;
    	property Visible;
      property WeekEnds;
   	property OnAccept;
      property OnCancel;
   end;

   TsPopupCalendarProps = class(TPersistent)
   private
      FCalendar: TsPopupCalendar;
      procedure SetBeginingOfWeek(Value: TWeekDay);
      function getBeginingOfWeek: TWeekDay;
      procedure SetColor( Value: TColor);
      function GetColor: TColor;
      procedure SetColors( Value: TsCalendarColors);
      function GetColors: TsCalendarColors;
      procedure Set3d(index: Integer; Value: TFont3D);
      function Get3D(index: Integer): TFont3D;
      procedure SetHilightStyle(Value: THilightStyle);
      function GetHilightStyle: THilightStyle;
      procedure SetSelectStyle(Value: TSelectStyle);
      function GetSelectStyle: TSelectStyle;
      procedure SetOptions(Value: TCalendarOptions);
      function GetOptions: TCalendarOptions;
      procedure SetTodayCaption(Value: String);
      function GetTodayCaption: String;
      function TodayCaptionStored: Boolean;
      procedure SetWeekEnds(Value: TWeekEnds);
      function GetWeekEnds: TWeekEnds;
   public
      constructor Create(calendar: TsPopupCalendar);
   published
      property BeginingOfWeek: TWeekday read GetBeginingOfWeek write SetBeginingofWeek default Mon;
      property Color: TColor read GetColor write SetColor default clBtnFace;
      property Colors: TsCalendarColors read GetColors write SetColors;
      property Font3d: TFont3d index 1 read Get3D write Set3D;
      property Hilight3d: TFont3d index 1 read Get3D write Set3D;
      property HilightStyle: THilightStyle read GetHilightStyle write SetHilightStyle default hsFrame;
      property Options: TCalendarOptions read GetOptions write SetOptions default [coShowPopups];
      property SelectStyle: TSelectStyle read GetSelectStyle write SetSelectStyle default ssFrame;
      property TodayCaption: String read GetTodayCaption write SetTodayCaption stored TodayCaptionStored;
      property WeekEnds: TWeekEnds read GetWeekEnds write SetWeekEnds default [Sat, Sun];
   end;

   TsMonthPopup = class(TPopupMenu)
   private
      FPopupOwner: TControl;
      FDate: TsDate;
      procedure ItemClick(Sender: TObject);
   protected
      procedure Popup(X, Y: Integer); override;
   public
      constructor Create(AOWner: TComponent); override;
      property PopupOwner: TControl write FPopupOwner;
      property Date: TsDate write FDate;
   end;

var
   MonthPopup: TsMonthPopup;


implementation
{$R scalendar.res}

uses SysUtils, Consts, ComCtrls, sTdUtils;

{*******************************************************************************
   TsCalendarColors
*******************************************************************************}
constructor TsCalendarColors.Create(AOwner: TsCustomCalendar);
begin
   FOwner := AOwner;
   FTitles := clWindowText;
   FWeekEnd := clMaroon;
   FInactive := cl3DLight;
   FInactiveWeekEnd := clInfoBk;
   FSelect := clHighlight;
end;

procedure TsCalendarColors.Assign(Source: TPersistent);
var
   SourceName: string;
begin
   if Source = nil then
      SourceName := 'nil'
   else
      SourceName := Source.ClassName;
   if (Source = nil) or not (Source is TsCalendarColors) then
      raise EConvertError.CreateFmt( SAssignError, [SourceName, ClassName]);
   with TsCalendarColors(Source) do begin
      self.FTitles := FTitles;
      self.FWeekEnd := FWeekEnd;
      self.FInactive := FInactive;
      self.FInactiveWeekEnd := FInactiveWeekEnd;
   end;
   FOwner.Invalidate;
end;

procedure TsCalendarColors.SetColor(Index: Integer; Value: TColor);
begin
   case Index of
      0: FTitles := Value;
      1: FWeekEnd := Value;
      2: FInactive := Value;
      3: FInactiveWeekEnd := Value;
      4: FSelect := Value;
   end;
   FOwner.Invalidate;
end;

{*******************************************************************************
   TsMonthPopup
*******************************************************************************}
constructor TsMonthPopup.Create(AOwner: TComponent);
var
   ii: Integer;
   item: TMenuItem;
begin
   inherited Create(AOwner);
   Alignment := paRight;
   FPopupOwner := nil;
   FDate := nil;
   for ii := 1 to 12 do begin
      item := TMenuItem.Create(self);
      item.Caption := FormatDateTime( 'mmmm', Encodedate( 1900, ii, 1));
      item.OnClick := ItemClick;
      item.Tag := ii;
      Items.Add( item);
   end;
end;

procedure TsMonthPopup.Popup(X, Y: Integer);
var
   P: TPoint;
begin
   if FPopupOwner <> nil then
      P := FPopupOwner.ClientToScreen(Point(X, Y));
   inherited Popup(P.X, P.Y);
end;

procedure TsMonthPopup.ItemClick(Sender: TObject);
begin
   if FDate <> nil then
      FDate.Month := (Sender as TMenuItem).Tag;
   FPopupOwner := nil;
   FDate := nil;
end;

{*******************************************************************************
   TsYearSpin
*******************************************************************************}
type
   TsYearSpin = class(TEdit)
   private
      FSpin: TUpDown;
      procedure CMCancelMode(var Message: TCMCancelMode); message CM_CANCELMODE;
      procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS;
      procedure CloseUp;
   protected
      procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure KeyPress(var Key: Char); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure Popup;
   end;

constructor TsYearSpin.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   Visible := FALSE;
   ReadOnly := TRUE;
   Ctl3d := FALSE;
   ParentCtl3D := FALSE;
   Parent := TsCustomCalendar(AOwner);
   Font.Assign(TsCustomCalendar(AOwner).Canvas.Font);
   Font.Color := TsCustomCalendar(AOwner).Colors.Titles;
   FSpin := TUpDown.Create(AOwner);
   FSpin.Min := 0;
   FSpin.Max := 3000;
   FSpin.Thousands := FALSE;
   FSpin.Parent := TsCustomCalendar(AOwner);
   FSpin.Associate := self;
   FSpin.visible := FALSE;
end;

destructor TsYearSpin.Destroy;
begin
   FSpin.Free;
   inherited;
end;

procedure TsYearSpin.Popup;
var
   R: TRect;
begin
   Color := TsCustomCalendar(parent).Color;
   if (parent is TsCalendar) and TsCalendar(parent).Transparent then
      Color := clBtnFace;
   R := TsCustomCalendar(parent).GetButtonRect( cpbYear);
   if HeightOf( R) > GetFontHeight(Font) then
      InflateRect( R, 0, -(HeightOf( R) - GetFontHeight(Font)) div 2);
   SetBounds( R.Left+2, R.Top, 5 * GetFontWidth(Font) + 6, HeightOf(R));
   Visible := TRUE;
   FSpin.Brush.Color := Color;
   FSpin.Visible := TRUE;
   FSpin.SetBounds( R.Left + Width + 3, Top, height, height);
   FSpin.position := TsCustomCalendar(parent).Date.Year;
   Windows.SetFocus(Handle);
end;

procedure TsYearSpin.CloseUp;
begin
   Visible := FALSE;
   FSpin.Visible := FALSE;
end;

procedure TsYearSpin.CMCancelMode(var Message: TCMCancelMode);
begin
   if (Message.Sender <> self) and (Message.Sender <> FSpin) then
      SendMessage( Parent.Handle, CM_CloseEditor, Integer(Message.Sender = Parent), 0);
   inherited;
end;

procedure TsYearSpin.WMKillFocus(var Message: TMessage);
begin
   if Message.WParam <> Parent.Handle then
      SendMessage( Parent.Handle, CM_CloseEditor, Integer(FALSE), 0);
   inherited;
end;

procedure TsYearSpin.KeyDown(var Key: Word; Shift: TShiftState);
begin
   case Key of
      33..40, 45:
         inherited KeyDown(Key, Shift);
      VK_RETURN, VK_ESCAPE:
         SendMessage( Parent.Handle, CM_CloseEditor, Integer(Key = VK_RETURN), 0);
      else
         Key := 0;
   end;
end;

procedure TsYearSpin.KeyPress(var Key: Char);
begin
   Key := #0;
end;

{*******************************************************************************
   TsCustomCalendar
*******************************************************************************}
const
   BORDER = 2;
   TEXT_INDENT = 2;
   TodayBitmap: TBitmap = nil;
   sTodayBmp = 'TodayBmp';

constructor TsCustomCalendar.Create(AOwner: TComponent);
begin
  	inherited Create(AOwner);
   FDate := TsCalendarDate.Create;
   FDate.BeginingOfWeek := Mon;
   FDate.OnDateChange := DateChanged;
   FFont3d := TFont3d.Create(Font3dChanged);
   FHilight3d := TFont3d.Create(Font3dChanged);
   FColors := TsCalendarColors.Create(self);
   FFontColor := clBlack;
   FBorderStyle := TBorderStyle(bsNone);
   FBevel := bsRaised;
   FOptions := [coShowPopups];
   FWeekEnds := [Sun,Sat];
   FHilightStyle := hsFrame;
   FSelectStyle := ssFrame;
   FYearPopup := nil;
   FTodayCaption := GetTodayCaptionDefault;
   Color := clBtnFace;
   Transparent := FALSE;
end;

destructor TsCustomCalendar.Destroy;
begin
   FColors.Free;
	FDate.Free;
   FFont3d.Free;
   FHilight3d.Free;
   FYearPopup.Free;
   FYearPopup := nil;
   inherited;
end;

procedure TsCustomCalendar.CreateParams(var Params: TCreateParams);
const
   BorderStyles: array[TBorderStyle] of Longint = (0, WS_BORDER);
begin
	inherited CreateParams( Params);
  	with Params do begin
    	WindowClass.Style := WindowClass.Style or CS_SAVEBITS;
      Style := Style or BorderStyles[FBorderStyle];
	end;
end;

procedure TsCustomCalendar.CreateWnd;
begin
   inherited;
   SetBounds(Left, Top, Width, Height);
end;

procedure TsCustomCalendar.CMMouseEnter(var Message: TMessage);
begin
   DoMouseEnter;
   inherited;
end;

procedure TsCustomCalendar.CMMouseLeave(var Message: TMessage);
begin
   if FTrackState = tsnone then
      PaintHilightButton(point(-1, -1));
   PaintHilightFrame(-1);
   DoMouseLeave;
   inherited;
end;

procedure TsCustomCalendar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TsCustomCalendar.CMFontChanged(var Message: TMessage);
begin
   Canvas.Font.Assign(inherited Font);
   FFontColor := Canvas.Font.Color;
   SetBounds(Left, Top, Width, Height);
   if FYearPopup <> nil then begin
      FYearPopup.Font.Assign(Canvas.Font);
      FYearPopup.Font.Color := Colors.Titles
   end;
   Invalidate;
end;

procedure TsCustomCalendar.CMCloseEditor(var Message: TMessage);
begin
   if Message.WParam = 1 then begin
      Date.Year := TsYearSpin(FYearPopup).FSpin.Position;
      SetFocus;
   end else
      Cancel;
   TsYearSpin(FYearPopup).CloseUp;
end;

procedure TsCustomCalendar.SetCaption;
var
   W: Integer;
begin
   W := Width - 2 * BORDER - 2 * TEXT_INDENT -
      TEXT_INDENT * Ord(FBorderStyle = bsSingle) - 2 * FCellHeight;

   FCaption := FormatDateTime( ' mmmm, yyyy ', Fdate.AsDateTime);
   Canvas.Font.Style := [fsBold];
   if Canvas.TextWidth(FCaption) >= W then
      FCaption := FormatDateTime( 'mmm, yyyy', Fdate.AsDateTime);
end;


function TsCustomCalendar.GetFontColor(index: Integer): TColor;
begin
   if FDate.MajorMonth(index) = 0 then begin
      if FDate.DayOfWeek(index) in FWeekEnds then
         Result := FColors.WeekEnd
      else
         Result := FFontColor;
   end else begin
      if FDate.DayOfWeek(index) in FWeekEnds then
         Result := FColors.InactiveWeekend
      else
         Result := FColors.Inactive;
   end;
end;

procedure TsCustomCalendar.PaintBevel;
const
   BevelState: array [TCalendarBevelStyle] of Word = (0, EDGE_SUNKEN, EDGE_RAISED);
var
  	R: TRect;
begin
   R := ClientRect;
   if FBevel <> bsNone then
   	DrawEdge( Canvas.Handle, R, BevelState[FBevel], BF_RECT)
   else
      FrameRect(R, 2);
end;

procedure TsCustomCalendar.PaintHeader;
var
	cBuffer: array[0..3] of Char;
   R: TRect;
   ii, index: Integer;
begin
   with Canvas do begin
      Font.Style := [fsBold];
      Font.Color := FColors.Titles;
      Brush.Style := bsClear;
   end;

   // week days
   with R do begin
  		Top := FvOffset + FCellHeight + BORDER - 4;
  		Left := FwOffset + FhOffset + BORDER + TEXT_INDENT;
  		Bottom := Top + FCellHeight;
  		Right := Left + FCellWidth * 3;
   end;

   for ii := 0 to 6 do begin
      index := ii + Ord(BeginingOfWeek) + 1;
      if index > 6 then
         index := index - 7;
      StrPCopy( @cBuffer, Format('%2s', [WeekDays[index+1]]));
      PaintText(Canvas.Handle, PChar(@cBuffer), 2, R, DT_CENTER or DT_BOTTOM or DT_SINGLELINE,
         FFont3d, nil, hsNone);
      with R do begin
         Left := Right;
         Right := Right + FCellWidth * 3;
      end;
   end;

   with R do begin
      Left := BORDER + TEXT_INDENT;
      Top := FvOffset + BORDER + 2 * FCellHeight - 4;
      Right := Width - BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle);
   end;
   DrawEdge( Canvas.Handle, R, EDGE_ETCHED, BF_TOP);

   with R do begin
      Left := BORDER + TEXT_INDENT + FCellWidth + 10;
   	Top := BORDER;
      Right := Right - BORDER - TEXT_INDENT - FCellWidth - 10;
      Bottom := Top + FCellHeight;
   end;

   PaintText(Canvas.Handle, PChar(FCaption), length(FCaption), R,
         DT_CENTER or DT_VCENTER or DT_SINGLELINE, FFont3d, nil, hsNone);

   PaintButton( cpbLeft, FALSE);
   PaintButton( cpbRight, FALSE);
end;

procedure TsCustomCalendar.PaintFooter;
var
   R: TRect;
begin
   if not (coShowTodayBtn in FOptions) then
      Exit;
   PaintButton(cpbToday, FALSE);
   with R do begin
      Left := BORDER + TEXT_INDENT;
      Right := Width - BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle);
      Top := BORDER + FCellHeight * 8 + 1;
   end;
   DrawEdge( Canvas.Handle, R, EDGE_ETCHED, BF_TOP);
end;

procedure TsCustomCalendar.PaintButton( btn: TscPopupBtn; Down: Boolean);
var
   sR, R: TRect;
   d: Integer;
   S: String;
begin
   if btn in [cpbLeft, cpbRight, cpbToday] then begin
      R := GetButtonRect( btn);
      PaintBackgroundRect(R);
   end;

   if btn in [cpbLeft, cpbRight] then begin
      if Down then begin
         Inc(R.Left);
         Inc(R.Top);
      end;
      with Canvas do begin
         pen.Color := FColors.Titles;
         Brush.Color := FColors.Titles;
         Brush.Style := bsSolid;
      end;
   end;

   case btn of
      cpbLeft: with Canvas do begin
         d := WidthOf(R) div 2;
         if d > 10 then begin
            InflateRect(R, 10 - d, 10 - d);
            d := 10;
         end;

         Polygon([ Point( R.Right - 5, R.Top + 3),
             Point( R.Right - 5, R.Bottom - 4),
             Point( R.Left + 3, R.Top + d)]);
         Pen.Color := clWhite;
         PolyLine( [Point( R.Right - 5, R.Top + 3),
            Point(R.Left + 2, R.Top + d)]);
      end;
      cpbRight: with Canvas do begin
         d := WidthOf(R) div 2;
         if d > 10 then begin
            InflateRect(R, 10 - d, 10 - d);
            d := 10;
         end;
         Polygon([Point(R.Left + 4, R.Top + 3),
            Point(R.Left + 4, R.Bottom - 5),
            Point(R.Right - 4, R.Top + d)]);
            Pen.Color := clWhite;
         PolyLine([Point(R.Left + 3, R.Bottom - 6),
            Point(R.Left + 3, R.Top + 2),
            Point(R.Right - 3, R.Top + d)]);
      end;
      cpbToday: with Canvas do begin
         Font.Color := FColors.Titles;
         Brush.Style := bsClear;
         if coShowToday in FOptions then begin
            R := GetButtonRect(cpbToday);
            if Down then
               OffsetRect(R, 1, 1);
            InflateRect(R, 0, -2);
            d := Trunc( HeightOf(R) / TodayBitmap.Height * TodayBitmap.Width);
            R.Right := R.left + d;
            sR := Rect(0, 0, TodayBitmap.Width, TodayBitmap.Height);
            StretchBitmapRectTransparent( Canvas,
                  R.Left + 2, R.Top, d, HeightOf(R), sR, TodayBitmap, clWhite);
         end else
            d := 0;
         R := GetButtonRect(cpbToday);
         if Down then begin
            Inc(R.Left);
            Inc(R.Top);
         end;
         R.Left := R.Left + d + 10;
         S := Format('%s %s', [ FTodayCaption, FormatDateTime( ShortDateFormat, SysUtils.Date)]);
         PaintText(Canvas.Handle, PChar(S), length(S),
            R, ( DT_VCENTER or DT_SINGLELINE), FFont3d, nil, hsNone);
      end;
   end;
end;

procedure TsCustomCalendar.PaintDates;
var
	cBuffer: array[0..3] of Char;
   nWeek, nDay, ii: Integer;
   R: TRect;
   f3d: TFont3d;
begin
   Canvas.Font.Style := [];
   Canvas.Brush.Style := bsClear;
   with R do begin
      Left := FwOffset + FhOffset + BORDER + TEXT_INDENT;
      Top := 2 * FCellHeight + Border;
      Right := Left + FCellWidth * 3;
      Bottom := Top +  FCellHeight ;
   end;

   for nWeek := 1 to 6 do begin
      for nDay := 1 to 7 do begin
         ii := nDay + ((nWeek - 1) * 7);
         Canvas.Font.Color := GetFontColor(ii);
         if (FDate.MajorMonth(ii) <> 0) then begin
            if coFullDisplay in FOptions then
               StrPCopy( @cBuffer, intToStr(FDate.DateArray[ii]))
            else
               cBuffer[0] := #0;
            f3d := nil;
         end else begin
            StrPCopy( @cBuffer, intToStr(FDate.DateArray[ii]));
            f3d := FFont3d;
         end;

         PaintText(Canvas.Handle, @cBuffer, Strlen(@cBuffer), R,
            (DT_CENTER or DT_VCENTER or DT_TOP or DT_SINGLELINE), f3d, nil, hsNone);
         OffsetRect(R, (FCellWidth * 3), 0);
      end;
      OffsetRect(R, -21 * FCellWidth, FCellHeight);
   end;
   if coShowWeeks in FOptions then begin
      with R do begin
         Left := FhOffset + BORDER + TEXT_INDENT;
         Top := 2 * FCellHeight + Border;
         Right := Left + FCellWidth * 3;
         Bottom := Top +  FCellHeight ;
      end;

      for nWeek := 1 to 6 do begin
         ii := FDate.WeeksArray[nWeek];
         if ((nWeek = 1) and (FDate.MajorMonth(7) < 0)) or
            ((nWeek = 6) and (FDate.MajorMonth(36) > 0)) then begin
            if coFullDisplay in FOptions then begin
               Canvas.Font.Color := FColors.Inactive;
               StrPCopy( @cBuffer, intToStr(ii));
            end else
               cBuffer[0] := #0;
            f3d := nil;
         end else begin
            Canvas.Font.Color := FFontColor;
            f3d := FFont3d;
            StrPCopy( @cBuffer, intToStr(ii));
         end;
         PaintText(Canvas.Handle, @cBuffer, Strlen(@cBuffer), R,
            (DT_CENTER or DT_VCENTER or DT_TOP or DT_SINGLELINE), f3d, nil, hsNone);
         OffsetRect(R, 0, FCellHeight);
      end;

      with R do begin
         Left := FhOffset + BORDER + TEXT_INDENT + 3 * FCellWidth;
   	   Top := FCellHeight * 2 + BORDER;
         Bottom := Top + FCellHeight * 6;
      end;
      DrawEdge( Canvas.Handle, R, EDGE_ETCHED, BF_LEFT);
   end;
   Canvas.Brush.Style := bsSolid;
end;

procedure TsCustomCalendar.PaintSingleDay(R: TRect; index: integer; hilight: TTextHilightStyle);
var
	cBuffer: array[0..3] of Char;
   f3d: TFont3d;
begin
   Canvas.Brush.Color := Color;
   PaintBackgroundRect(R);
   if (coFullDisplay in FOptions) or (FDate.MajorMonth(FPrevFocusIndex) = 0) then begin
      Canvas.Font.Style := [];
      Canvas.Font.Color := GetFontColor(index);
      Canvas.Brush.Style := bsClear;
      if FDate.MajorMonth(index) <> 0 then
         f3d := nil
      else
         f3d := FFont3d;
      StrPCopy( @cBuffer, IntToStr(FDate.DateArray[index]));
      PaintText(Canvas.Handle, @cBuffer, Strlen(@cBuffer), R,
         (DT_CENTER or DT_VCENTER or DT_TOP or DT_SINGLELINE), f3d, FHilight3d, hilight);
      Canvas.Brush.Style := bsSolid;
      Canvas.Brush.Color := Color;
   end;
end;

procedure TsCustomCalendar.PaintToday;
var
   R, sR: TRect;
   W, X: Integer;
begin
   if (FDate.TodayIndex in [1..42]) and
      (coFullDisplay in FOptions) or (FDate.MajorMonth(FDate.TodayIndex) = 0) then begin
      R := GetRectFromIndex(FDate.TodayIndex);
      X := Trunc( HeightOf(R) / TodayBitmap.Height * TodayBitmap.Width);
      W := R.Right - R.left;
      R.Left := R.left - (X - W) div 2;
      R.Right := R.left + X;
      sR := Rect(0, 0, TodayBitmap.Width, TodayBitmap.Height);
      StretchBitmapRectTransparent( Canvas,
         R.Left, R.Top, X, HeightOf(R), sR, TodayBitmap, clWhite);
   end;
end;

procedure TsCustomCalendar.PaintFocusFrame(index : Integer; force: Boolean);
var
	cBuffer: array[0..3] of Char;
  	R : TRect;
begin
  	if (index in [1..42]) and ((index <> FPrevFocusIndex) or force) and
      ((coFullDisplay in FOptions) or (FDate.MajorMonth(index) = 0)) then begin
      // Erase Previous Date Focus
      if FPrevFocusIndex > 0 then begin
         R := GetRectFromIndex(FPrevFocusIndex);
         InflateRect(R, 1, 1);
         PaintSingleDay(R, FPrevFocusIndex, hsNone);
         if (coShowToday in FOptions) and
            ((FPrevFocusIndex >= FDate.TodayIndex-1) or (FPrevFocusIndex <= FDate.TodayIndex+1)) then
            PaintToday;
      end;
      if FSelectStyle <> ssNone then begin
         // Draw the Date in Bold font
         Canvas.Font.Style := [fsBold];
         Canvas.Font.Color := GetFontColor(index);
         Canvas.Brush.Color := Color;
         R := GetRectFromIndex(index);
         StrPCopy( @cBuffer, IntToStr(FDate.DateArray[index]));
         if FSelectStyle = ssFrame then begin
            paintBackgroundRect(R);
            Canvas.Brush.Style := bsClear;
            DrawText(Canvas.Handle, @cBuffer, Strlen(@cBuffer),
               R, ( DT_CENTER or DT_VCENTER or DT_SINGLELINE));
            DrawFlatEDGE(R, TRUE);
         end else begin
            Canvas.Brush.Color := Colors.Select;
            Canvas.Pen.Color := Colors.Select;
            InflateRect(R, 0, -1);
            Canvas.Ellipse(R.Left, R.Top, R.Right, R.Bottom);
            Canvas.Brush.Style := bsClear;
            Canvas.Font.Color := clWhite;
            InflateRect(R, 0, 2);
            DrawText(Canvas.Handle, @cBuffer, Strlen(@cBuffer),
               R, ( DT_CENTER or DT_VCENTER or DT_SINGLELINE));
            Canvas.Brush.Color := Color;
         end;
         Canvas.Brush.Style := bsSolid;
         FPrevFocusIndex := index;
         if (coShowToday in FOptions) and
            ((index >= FDate.TodayIndex-1) or (index <= FDate.TodayIndex+1)) then
            PaintToday;
      end else
         FPrevFocusIndex := -1;
   end;
end;

procedure TsCustomCalendar.PaintHilightFrame(index: Integer);
var
   R: TRect;
begin
   if (FHilightStyle = hsNone) or ((FHilightStyle <> hsFrame) and not FHilight3d.Active) then
      Exit;
  	if (index <> FPrevHilightIndex) then begin
      if (FPrevHilightIndex > 0) and (FPrevHilightIndex <> FPrevFocusIndex) then begin
         R := GetRectFromIndex(FPrevHilightIndex);
         if FHilightStyle = hsFrame then
            FrameRect(R, 2)
         else
            PaintSingleDay(R, FPrevHilightIndex, hsNone);
         if (coShowToday in FOptions) and
            ((FPrevHilightIndex >= FDate.TodayIndex-1) or (FPrevHilightIndex <= FDate.TodayIndex+1)) then
            PaintToday;
      end;
      if (index in [1..42]) and (index <> FPrevFocusIndex) and
         ((coFullDisplay in FOptions) or (FDate.MajorMonth(index) = 0)) then begin
         R := GetRectFromIndex(index);
         if FHilightStyle = hsFrame then begin
            DrawFlatEDGE(R, FALSE);
         end else
            PaintSingleDay(R, index, FHilightStyle);
         if (coShowToday in FOptions) and
            ((index >= FDate.TodayIndex-1) or (index <= FDate.TodayIndex+1)) then
            PaintToday;
         FPrevHilightIndex := index;
      end else
         FPrevHilightIndex := -1;
   end;
end;

procedure TsCustomCalendar.PaintHilightButton(P: TPoint);
var
   ii: Integer;
   aPopupBtn: TscPopupBtn;
   R: TRect;
begin
   aPopupBtn := cpbNone;
   for ii := 1 to 5 do begin
      R := GetButtonRect( TscPopupBtn(ii));
      if PtInRect( R, P) then begin
         aPopupBtn := TscPopupBtn(ii);
         Break;
      end;
   end;
   if not (aPopupBtn in [cpbleft, cpbRight]) and not (coShowPopups in Options) then
      aPopupBtn := cpbNone;
   if (aPopupBtn = cpbToday) and not (coShowTodayBtn in FOptions) then
      aPopupBtn := cpbNone;
   if aPopupBtn <> FLastPopupBtn then begin
      if FLastPopupBtn > cpbNone then
         PaintPopupBtn( GetButtonRect( FLastPopupBtn), cbsNormal);
      if (aPopupBtn > cpbNone) then
         PaintPopupBtn( R, cbsUp);
      FLastPopupBtn := aPopupBtn;
   end;
end;

procedure TsCustomCalendar.PaintPopupBtn(R: TRect; state: TscBtnState);
begin
   if state <> cbsNormal then begin
      PaintButton( TscPopupBtn(FTrackState), state = cbsDown);
      DrawFlatEDGE(R, state = cbsDown);
   end else
      FrameRect(R, 2)
end;

procedure TsCustomCalendar.Paint;
begin
   inherited;
   PaintBackground;
   with Canvas do begin
      Brush.Color := Color;
      Brush.Style := bsSolid;
   end;
   PaintBevel;
   PaintHeader;
   PaintFooter;
   PaintDates;
   PaintFocusFrame(FDate.CurrentDateIndex, TRUE);
   if coShowToday in FOptions then
      PaintToday;
end;

procedure TsCustomCalendar.InvalidateDates;
   function GetCaptionRect: TRect;
   begin
      with Result do begin
         Left := BORDER + TEXT_INDENT + FCellWidth + 10;
   	   Top := BORDER;
         Right := Width - BORDER - TEXT_INDENT - FCellWidth - 10;
         Bottom := Top + FCellHeight;
      end;
   end;
var
   R: TRect;
begin
   inherited;
   R := GetCalendarRect;
   InflateRect(R, FhOffset+BORDER, 0);
   Dec(R.Left, FwOffset);
   PaintBackgroundRect(R);
   R := GetCaptionRect;
   PaintBackgroundRect(R);
   with Canvas do begin
      Brush.Color := Color;
      Brush.Style := bsClear;
      Font.Style := [fsBold];
      Font.Color := FColors.Titles;
   end;
   PaintText(Canvas.Handle, PChar(FCaption), length(FCaption), R,
      DT_CENTER or DT_VCENTER or DT_SINGLELINE, FFont3d, nil, hsNone);
   PaintDates;
   PaintFocusFrame(FDate.CurrentDateIndex, TRUE);
end;

procedure TsCustomCalendar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      if PtInRect(GetCalendarRect, Point(X, Y)) then begin
         FTrackState := tsDay;
         FPrevIndex := FDate.CurrentdateIndex;
      end else if PtInRect( GetButtonRect(cpbLeft), Point(X, Y)) then
         FTrackState := tsLeft
  		else if PtInRect(GetButtonRect( cpbRight), Point(X, Y)) then
         FTrackState := tsRight
      else if PtInRect( GetButtonRect(cpbToday), Point(X, Y)) and
         (coShowTodayBtn in FOptions) then
         FTrackState := tsToday
      else if PtInRect( GetButtonRect( cpbMonth), Point(X, Y)) then
         CallPopupMonth( GetButtonRect( cpbMonth))
      else if PtInRect( GetButtonRect( cpbYear),Point(X, Y)) then
         CallPopupYear(GetButtonRect( cpbYear));
      Track(X, Y);
      MouseCapture := FTrackState <> tsNone;
   end;
   inherited;
end;

procedure TsCustomCalendar.MouseMove( Shift: TShiftState; X, Y: Integer);
begin
   if ([csLoading, csReading] * ComponentState = []) and (FTrackState = tsNone) and
      ((FYearPopup = nil) or not FYearPopup.Visible) then begin
      PaintHilightFrame(GetIndexFromPoint( X, Y ));
      PaintHilightButton( Point(X, Y));
   end else
      Track(X, Y);
   inherited;
end;

procedure TsCustomCalendar.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer );
var
   index: Integer;
begin
   MouseCapture := FALSE;
  	if (Button = mbLeft) then begin
      if cstPressed in FState then case FtrackState of
         tsLeft: begin
            if ssCtrl in Shift then
               FDate.PrevYear
            else
               FDate.PrevMonth;
         end;
         tsRight: begin
            if ssCtrl in Shift then
               FDate.NextYear
            else
               FDate.NextMonth;
         end;
         tstoday: begin
            FDate.ForceReload := not (coFullDisplay in Options);
            FDate.AsDateTime := SysUtils.Date;
            Accept;
         end;
         tsday: begin
            index := GetIndexFromPoint( X, Y);
            if coFullDisplay in FOptions then begin
               FDate.CurrentDateIndex := index;
               Accept;
            end else case FDate.MajorMonth(index) of
               -1:
                  if coMonthSwitch in FOptions then
                     FDate.PrevMonth;
               0: begin
                  FDate.CurrentDateIndex := index;
                  Accept;
               end;
               1:
                  if coMonthSwitch in FOptions then
                     FDate.NextMonth;
            end;
         end;
      end else if FtrackState = tsDay then begin
         if coCancelIfOut in FOptions then
            PaintFocusFrame(FPrevIndex, TRUE)
         else begin
            FDate.CurrentDateIndex := FPrevIndex;
            Accept;
         end;
      end;
      StopTracking(X, Y);
   end else
   	inherited;
end;

procedure TsCustomCalendar.KeyDown(var Key: Word; Shift: TShiftState);
begin
   case key of
      VK_Left:
         FDate.PrevDay;
      VK_Right:
         FDate.NextDay;
      VK_Up :
         FDate.PrevWeek;
      VK_Down :
         FDate.NextWeek;
      VK_Prior:
         if ssCtrl in Shift then
            FDate.PrevYear
         else
            FDate.PrevMonth;
      Vk_Next :
         if ssCtrl in Shift then
            FDate.NextYear
         else
            FDate.NextMonth;
      VK_Home :
         FDate.AsDateTime := SysUtils.Date;
   end;
   inherited;
end;

procedure TsCustomCalendar.KeyPress(var Key: Char);
begin
   if Key = Char(VK_Return) then
   	Accept
   else if Key = Char(VK_Escape) then
   	Cancel
   else
      inherited;
end;

procedure TsCustomCalendar.Track(X,Y: Integer);
const
   btnStates: array[Boolean] of TscBtnState = (cbsUp, cbsDown);
var
  	NewState: Boolean;
  	R: TRect;
begin
   case FTrackState of
      tsleft, tsRight, tsToday: begin
	      R := GetButtonRect( TscPopupBtn(FTrackState));
         NewState := PtInRect(R, Point(X, Y));
  	      if (cstPressed in FState) <> NewState then begin
            if NewState then
               Include( FState, cstPressed)
            else
               Exclude( FState, cstPressed);
            PaintPopupBtn(R, btnStates[newState]);
         end;
      end;
      tsDay: begin
         R := GetCalendarRect;
         NewState := PtInRect(R, Point(X, Y));
  	      if (cstPressed in FState) <> NewState then begin
            if NewState then
               Include( FState, cstPressed)
            else
               Exclude( FState, cstPressed);
         end;
         PaintFocusFrame(GetIndexFromPoint( X, Y), FALSE);
      end;
  	end;
end;

procedure TsCustomCalendar.StopTracking(X, Y: Integer);
begin
   if FTrackState <> tsNone then begin
    	Track(-1, -1);
      FTrackState := tsNone;
    	MouseCapture := False;
      PaintHilightButton( Point(X, Y));
  	end;
end;

const
   hCellsno: array[Boolean] of integer = (21, 25);
   wCellsNo: array[Boolean] of integer = (8, 9);

procedure TsCustomCalendar.CalculateValues( var AHeight, AWidth: Integer);
begin
   FvOffset := Ord(BorderStyle = bsSingle);
   if (coShowTodayBtn in FOptions) then begin
      FCellHeight := (AHeight - BORDER - 8 * Ord(BorderStyle = bsSingle) - 22) div 8;
      if (AHeight - BORDER - 2 * FvOffset - 22) <= (BORDER + FCellHeight * 8 + 2) then
         Dec(FCellHeight);
   end else
      FCellHeight := (AHeight - BORDER - 8 * Ord(BorderStyle = bsSingle)) div 8;

   FCellWidth := (AWidth - (2* BORDER) - (2* TEXT_INDENT)) div
      hCellsNo[coShowWeeks in FOptions];
   FhOffset := (AWidth - ((FCellWidth * hCellsNo[coShowWeeks in FOptions]) +
      (2* BORDER) + (2* TEXT_INDENT) + Ord(BorderStyle = bsSingle))) div 2;
   FwOffset := FCellWidth * 4 * Ord(coShowWeeks in FOptions);
end;

procedure TsCustomCalendar.Cancel;
begin
	if Assigned( FOnCancel) then
      FOnCancel(self);
end;

procedure TsCustomCalendar.Accept;
begin
	if Assigned( FOnAccept) then
      FOnAccept(self);
end;

procedure TsCustomCalendar.DoMouseEnter;
begin
	if Assigned( FOnMouseEnter) then
      FOnMouseEnter(self);
end;

procedure TsCustomCalendar.DoMouseLeave;
begin
	if Assigned( FOnMouseLeave) then
      FOnMouseLeave(self);
end;

function TsCustomCalendar.GetCalendarRect: TRect;
begin
  	with Result do begin
   	Top := FvOffset + FCellHeight * 2 + BORDER;
  		Left := FwOffset + FhOffset + BORDER + TEXT_INDENT;
      Bottom := Top + FCellHeight * 6;
      Right := Left + FCellWidth * 21;
   end;
end;

function TsCustomCalendar.GetButtonRect( btn: TscPopupBtn): TRect;
var
   S: String;
   p: Integer;
begin
	with Result do begin
      if btn = cpbToday then begin
         Bottom := Height - BORDER - 2 * FvOffset;
         Top := Bottom - 20;
      end else begin
   	   Top := FvOffset + BORDER;
   	   Bottom := Top + FCellHeight;
      end;

      case btn of
         cpbLeft: begin
            Left := BORDER + TEXT_INDENT;
   	      Right := Left + FCellHeight;
         end;
         cpbRight: begin
   	      Right := Width - BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle);
   	      Left := Right - FCellHeight;
         end;
         cpbMonth: begin
            Left := BORDER + TEXT_INDENT + FCellHeight;
            S := FCaption;
            SetLength(S, Pos(',', S));
            Canvas.Font.Style := [fsBold];
   	      Right := Left +
               (Width - 2 * BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle) -
               2 * FCellHeight - Canvas.TextWidth(FCaption)) div 2 + Canvas.TextWidth(S);
         end;
         cpbYear: begin
            Right := Width - BORDER - TEXT_INDENT - FCellHeight -
               TEXT_INDENT * Ord(FBorderStyle = bsSingle);
            S := FCaption;
            p := Pos(',', S);
            S := Copy(S, p + 1, length(S) - p);
            Canvas.Font.Style := [fsBold];
   	      Left := Right - Canvas.TextWidth(S) -
               ((Width - 2 * BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle)) -
               2 * FCellHeight - Canvas.TextWidth(FCaption)) div 2;
         end;
         cpbToday: begin
            Left := BORDER + TEXT_INDENT;
            Right := Width - BORDER - TEXT_INDENT - TEXT_INDENT * Ord(FBorderStyle = bsSingle);
         end;
      end;
	end;
end;

function TsCustomCalendar.GetRectFromIndex(index : Integer): TRect;
var
  	nWeek : Integer;
  	nDay : Integer;
begin
   nWeek := (index -1) div 7;
   nDay := index - (nWeek *7);
  	with Result do begin
   	Left := FwOffset + FhOffset + BORDER + TEXT_INDENT + ((FCellWidth * 3) * (nDay-1));
      Top := {FvOffset + }BORDER + FCellHeight * (nWeek + 2);
      Bottom := Top +  FCellHeight;
      Right := Left + FCellWidth * 3;
   end;
end;

function TsCustomCalendar.GetIndexFromPoint(nLeft : Integer ; nTop : Integer) : Integer;
var
  	nWeek, nDay: Integer;
begin
  	Result := -1;
	if PtInRect( GetCalendarRect, Point( nLeft, nTop)) then begin
      nWeek := (nTop - BORDER) div (FCellHeight);
      nday := (nLeft - (FwOffset + FhOffset + BORDER + TEXT_INDENT)) div (FCellWidth * 3);
      Result := nDay + ((nWeek - 2) * 7) + 1;
   end;
end;

procedure TsCustomCalendar.CallPopupYear(R: TRect);
begin
   if coShowPopups in FOptions then begin
      PaintPopupBtn( R, cbsNormal);
      if FLastPopupbtn = cpbYear then
         FLastPopupbtn := cpbNone;
      if FYearPopup = nil then
         FYearPopup := TsYearSpin.Create(self);
      TsYearSpin(FYearPopup).Popup;
   end;
end;

procedure TsCustomCalendar.CallPopupMonth(R: TRect);
var
   Msg: TMsg;
begin
   if coShowPopups in FOptions then begin
      PaintPopupBtn( R, cbsDown);
      MonthPopup.PopupOwner := self;
      MonthPopup.Date := FDate;
      MonthPopup.Alignment := paRight;
      MonthPopup.Popup(R.Right, R.Bottom);
      while PeekMessage(Msg, 0, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do;
      PaintPopupBtn( R, cbsNormal);
   end;
end;

procedure TsCustomCalendar.DateChanged(Sender: TObject);
begin
	if (FDate.Reloaded) then begin
      SetCaption;
      InvalidateDates;
   end else
      PaintFocusFrame(FDate.CurrentDateIndex, FALSE);
   if Assigned( FOnDateChange) then
      FOnDateChange( self);
end;

procedure TsCustomCalendar.SetBevel(Value: TCalendarBevelStyle);
begin
	if Value <> FBevel then begin
		FBevel := Value;
      PaintBevel;
   end;
end;

procedure TsCustomCalendar.SetBorderStyle(Value: TBorderStyle);
begin
  	if FBorderStyle <> Value then begin
    	FBorderStyle := Value;
      RecreateWnd;
  	end;
end;

procedure TsCustomCalendar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
   CalculateValues( AHeight, AWidth);
   if HandleAllocated then
      SetCaption;
   inherited SetBounds(ALeft, ATop, AWidth, AHeight);
end;

function TsCustomCalendar.GetBeginingOfWeek: TWeekDay;
begin
	Result := FDate.BeginingOfWeek;
end;

procedure TsCustomCalendar.SetBeginingOfWeek(Value: TWeekDay);
begin
	if BeginingOfWeek <> Value then begin
	   FDate.BeginingOfWeek := Value;
   	Invalidate;
   end;
end;

procedure TsCustomCalendar.SetOptions(Value: TCalendarOptions);
begin
   if FOptions <> Value then begin
      FOptions := Value;
      SetBounds(Left, Top, Width, Height);
      FDate.SmartReload := not (coMonthSwitch in FOptions);
      if ([coShowTodayBtn, coShowToday] * FOptions <> []) and (TodayBitmap = nil) then begin
         TodayBitmap := TBitmap.Create;
         TodayBitmap.Handle := LoadBitmap(hInstance, sTodayBmp);
      end;
      Invalidate;
   end;
end;

procedure TsCustomCalendar.SetTodayCaption(Value: String);
begin
   if FTodayCaption <> Value then begin
      FTodayCaption := Value;
      Invalidate;
   end;
end;

function TsCustomCalendar.GetTodayCaptionDefault: String;
begin
   Result := 'Today';
end;

function TsCustomCalendar.TodayCaptionStored: Boolean;
begin
   Result := GetTodayCaptionDefault <> FTodayCaption;
end;

procedure TsCustomCalendar.SetColors( Value: TsCalendarColors);
begin
   if FColors <> Value then
      FColors.Assign(Value);
end;

procedure TsCustomCalendar.SetWeekEnds(Value: TWeekEnds);
begin
   if FWeekEnds <> Value then begin
      FWeekEnds := Value;
      Invalidate;
   end;
end;

procedure TsCustomCalendar.SetFont3d(Value: TFont3d);
begin
   if FFont3d <> Value then
      FFont3d.Assign(Value);
end;

procedure TsCustomCalendar.Font3dChanged(Sender: Tobject);
begin
   Invalidate;
end;

procedure TsCustomCalendar.SetHilight3d(Value: TFont3d);
begin
   if FHilight3d <> Value then
      FHilight3d.Assign( Value);
end;

procedure TsCustomCalendar.SetHilightStyle(Value: THilightStyle);
begin
   if Value <> FHilightStyle then
      FHilightStyle := Value;
end;

procedure TsCustomCalendar.SetSelectStyle(Value: TSelectStyle);
begin
   if Value <> FSelectStyle then begin
      FSelectStyle := Value;
      PaintFocusFrame(FDate.CurrentdateIndex, TRUE);
   end;
end;
{*******************************************************************************
   TsCalendar
*******************************************************************************}

constructor TsCalendar.Create( AOwner: TComponent );
begin
   inherited;
   SetBounds(Left, Top, 155, 141);
end;

procedure TsCalendar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   SetFocus;
   inherited;
end;

{ TsPopupCalendar }
constructor TsPopupCalendar.Create( AOwner: TComponent);
begin
   inherited;
   Visible := FALSE;
   ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable];
   Color := clBtnFace;
   Ctl3D := False;
   ParentCtl3D := False;
   BorderStyle := TBorderStyle(bsNone);
   FBevel := bsRaised;
   Parent := AOwner as TWinControl;
end;

procedure TsPopupCalendar.CreateParams(var Params: TCreateParams);
begin
	inherited CreateParams( Params);
  	with Params do begin
      Style := WS_POPUP {or WS_BORDER }or WS_CLIPCHILDREN;
      ExStyle := WS_EX_TOOLWINDOW;
    	WindowClass.Style := WindowClass.Style or CS_SAVEBITS;
	end;
end;

procedure TsPopupCalendar.WMKillFocus(var Message: TMessage);
begin
   if Visible and (Message.WParam <> Handle)  then begin
      if (FYearPopup = nil) or (Message.WParam <> FYearPopup.Handle) then
         Cancel;
   end;
   inherited;
end;

procedure TsPopupCalendar.CalculateValues( var AHeight, AWidth: Integer);
begin
	with GetFontMetrics( Font) do begin
      FCellWidth := Round(tmAveCharWidth * 1.4);
      FCellHeight := Round((tmHeight + tmHeight / 8)  * 1.4);
   end;
 	AHeight := (FCellHeight * wCellsNo[coShowTodayBtn in FOptions]) + 2 * BORDER + 4 * TEXT_INDENT;
  	AWidth := (FCellWidth * hCellsNo[coShowWeeks in FOptions]) + 2 * BORDER + 4 * TEXT_INDENT;
   FhOffset := 0;
   FvOffset := 0;
   FwOffset := FCellWidth * 4 * Ord(coShowWeeks in FOptions);
end;

procedure TsPopupCalendar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if (FYearPopup <> nil) and (FYearPopup.Visible) then
      FYearPopup.Perform( CM_CANCELMODE, 0, Integer(self));
   inherited;
end;

procedure TsPopupCalendar.CalculatePosition;
var
   pt: TPoint;
   ALeft, ATop: Integer;
   d: Integer;
begin
   if FCaller <> nil then
      pt := FCaller.ClientToScreen( FPopupPoint)
   else
      pt := Parent.ClientToScreen( FPopupPoint);
   ATop := pt.Y + 1;
   ALeft := pt.X + 1;
   SetBounds(ALeft, ATop, 0, 0);

   d := Screen.Height - (Top + Height);
   if d < 0 then
      Top := Top + d;
   d := Screen.Width - (Left + Width);
   if d < 0 then
      left := Left + d;
end;

procedure TsPopupCalendar.Popup(X, Y: Integer);
begin
   FPopupPoint := Point(X, Y);
   CalculatePosition;
   Visible := True;
   SetFocus;
end;

procedure TsPopupCalendar.Cancel;
begin
   inherited;
   Visible := FALSE;
   if FCaller <> nil then
      FCaller.SetFocus;
end;

procedure TsPopupCalendar.Accept;
begin
   inherited;
   Visible := FALSE;
   if FCaller <> nil then
      FCaller.SetFocus;
end;

{******************** TsPopupCalendarProps ************************}
constructor TsPopupCalendarProps.Create(calendar: TsPopupCalendar);
begin
   FCalendar := calendar;
end;

procedure TsPopupCalendarProps.SetBeginingOfWeek(Value: TWeekDay);
begin
   FCalendar.BeginingOfWeek := Value
end;

function TsPopupCalendarProps.GetBeginingOfWeek: TWeekDay;
begin
   Result := FCalendar.BeginingOfWeek;
end;

procedure TsPopupCalendarProps.SetColor( Value: TColor);
begin
   FCalendar.Color := Value
end;

function TsPopupCalendarProps.GetColor: TColor;
begin
   Result := FCalendar.Color;
end;

procedure TsPopupCalendarProps.SetColors( Value: TsCalendarColors);
begin
   FCalendar.Colors.Assign(Value);
end;

function TsPopupCalendarProps.GetColors: TsCalendarColors;
begin
   Result := FCalendar.Colors;
end;

procedure TsPopupCalendarProps.Set3d(index: Integer; Value: TFont3D);
begin
   if index = 1 then
      FCalendar.Font3D.Assign(Value)
   else
      FCalendar.Hilight3D.Assign(Value);
end;

function TsPopupCalendarProps.Get3D(index: Integer): TFont3D;
begin
   if index = 1 then
      Result := FCalendar.Font3D
   else
      Result := FCalendar.Hilight3D;
end;

procedure TsPopupCalendarProps.SetHilightStyle(Value: THilightStyle);
begin
   FCalendar.HilightStyle := Value;
end;

function TsPopupCalendarProps.GetHilightStyle: THilightStyle;
begin
   Result := FCalendar.HilightStyle;
end;

procedure TsPopupCalendarProps.SetSelectStyle(Value: TSelectStyle);
begin
   FCalendar.SelectStyle := Value;
end;

function TsPopupCalendarProps.GetSelectStyle: TSelectStyle;
begin
   Result := FCalendar.SelectStyle;
end;

procedure TsPopupCalendarProps.SetOptions(Value: TCalendarOptions);
begin
   FCalendar.Options := Value;
end;

function TsPopupCalendarProps.GetOptions: TCalendarOptions;
begin
   Result := FCalendar.Options
end;

procedure TsPopupCalendarProps.SetTodayCaption(Value: String);
begin
   FCalendar.TodayCaption := Value;
end;

function TsPopupCalendarProps.GetTodayCaption: String;
begin
   Result := FCalendar.TodayCaption;
end;

function TsPopupCalendarProps.TodayCaptionStored: Boolean;
begin
   Result := FCalendar.GetTodayCaptionDefault <>  TodayCaption
end;

procedure TsPopupCalendarProps.SetWeekEnds(Value: TWeekEnds);
begin
   FCalendar.WeekEnds := Value
end;

function TsPopupCalendarProps.GetWeekEnds: TWeekEnds;
begin
   Result := FCalendar.WeekEnds
end;


initialization
   MonthPopup := TsMonthPopup.Create(nil);

finalization
   MonthPopup.Free;
   TodayBitmap.Free;
   TodayBitmap := nil;




end.
