{*******************************************************************************
   Utit:
      sDate.pas
   Description:
      Declares TsDate and TsCalendar date handling classes - used in
      TsDateEdit and TsCalndar.
   Versions:
      2.0
   Author(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**

*     I did not track the versions before, so let's consider it as 2.0
**    Some functions here were copied (some of them modified) from RX's dateUtils
      unit.
*******************************************************************************}

unit sDate;

interface
{$B-,V-,R-,Q-}

uses Classes, SysUtils;

type
   TDateChangeEvent = procedure (Sender: TObject; aDate: TDateTime) of Object;
   TDateOrder = (doDMY, doMDY, doYMD);

   TsDate = class
   private
      FYear, FMonth, FDay: Word;
      FBeforeDateChange: TDateChangeEvent;
      FOnDateChange: TNotifyEvent;
      FDateOrder: TDateOrder;
      function GetAsDateTime: TDateTime;
      procedure SetAsDateTime(const Value: TDateTime);
      function GetAsString: string;
      procedure SetAsString(const Value: string);
      procedure SetYear(const Value: Word);
      procedure SetDay(const Value: Word);
      function GetIsNull: Boolean;
   protected
      procedure SetMonth(const Value: Word); virtual;
      procedure DateBeforeChange( Value: TDateTime); virtual;
      procedure DateChanged; virtual;
   public
      constructor Create;
      procedure Date( y, m, d: Word);
      function IsLeapYear: Boolean;
      function DaysInMonth(const nMonth: Word): Integer;
      procedure PrevDay;
      procedure NextDay;
      procedure PrevWeek;
      procedure NextWeek;
      procedure PrevMonth; dynamic;
      procedure NextMonth; dynamic;
      procedure NextYear;
      procedure PrevYear;
      procedure IncDate(const nDays: Integer);
      procedure SetNull;
      property DateOrder: TDateOrder read FDateOrder write FDateOrder;
      property AsDateTime: TDateTime read GetAsDateTime write SetAsDateTime;
      property AsString: string read GetAsString write SetAsString;
      property Year: Word read FYear write SetYear;
      property Month: Word read FMonth write SetMonth;
      property Day: Word read FDay write SetDay;
      property IsNull: Boolean read GetIsNull;
      property BeforeDateChange: TDateChangeEvent read FBeforeDateChange write FBeforeDateChange;
      property OnDateChange: TNotifyEvent read FOnDateChange write FOnDateChange;
   end;

   TWeekDay = ( Mon, Tue, Wed, Thu, Fri, Sat, Sun);
   TWeekEnd = Sat..Sun;
   TWeekEnds = set of TWeekEnd;

   TDateArrayType = array[1..42] of ShortInt;
   TWeekArrayType = array[1..6] of ShortInt;

   TsCalendarDate = class(TsDate)
   private
      FBeginingOfWeek: TWeekDay;
      FStartDate: TDateTime;
      FMonthStart, FMonthEnd: Word;
      FDateArray: TDateArrayType;
      FWeeksArray: TWeekArrayType;
      FReloaded: Boolean;
      FSmartReload: Boolean;
      FForceReload: Boolean;
      FCurrentDateIndex: ShortInt;
      FTodayIndex: ShortInt;
      function NeedsReload: Boolean;
      procedure LoadDateArray;
      procedure SetBeginingOfWeek(const Value: TWeekDay);
      procedure SetCurrentDateIndex(const Value: ShortInt);
      function GetMonthBegin(Year, Month: Word): ShortInt;
      function GetIndexFromDate: Integer;
   protected
      procedure SetMonth(const Value: Word); override;
      procedure DateBeforeChange( Value: TDateTime); override;
      procedure DateChanged; override;
   public
      procedure PrevMonth; override;
      procedure NextMonth; override;
      function MajorMonth(index: ShortInt): ShortInt;
      function DayOfWeek(index: ShortInt): TWeekDay;
      property BeginingOfWeek: TWeekDay read FBeginingOfWeek write SetBeginingOfWeek;
      property DateArray: TDateArrayType read FDateArray;
      property WeeksArray: TWeekArrayType read FWeeksArray;
      property CurrentDateIndex: ShortInt read FCurrentDateIndex write SetCurrentDateIndex;
      property TodayIndex: ShortInt read FTodayIndex;
      property Reloaded: Boolean read FReloaded;
      property SmartReload: Boolean read FSmartReload write FSmartReload;
      property ForceReload: Boolean write FForceReload;
   end;

function ExtractDay(ADate: TDateTime): Word;
function ExtractMonth(ADate: TDateTime): Word;
function ExtractYear(ADate: TDateTime): Word;

function IncDate(ADate: TDateTime; Days, Months, Years: Integer): TDateTime;
function IncDateEx(ADate: TDateTime; Days, Months, Years: Integer; floatingMonth: Boolean): TDateTime;
function IncDay(ADate: TDateTime; Delta: Integer): TDateTime;
function IncMonth(ADate: TDateTime; Delta: Integer): TDateTime;
function IncYear(ADate: TDateTime; Delta: Integer): TDateTime;

procedure ValidateDate( var Year, Month, Day: Word);
function IsValidDate(year, Month, Day: Word): Boolean;
function IsLeapYear(AYear: Integer): Boolean;
function DaysPerMonth(AYear, AMonth: Integer): Integer;

function GetDateOrder(const DateFormat: string): TDateOrder;
function StrToDateDef(const S: string; Default: TDateTime): TDateTime;
function StrToDateFmt(const DateFormat, S: string): TDateTime;
function StrToDateFmtDef(const DateFormat, S: string; Default: TDateTime): TDateTime;
function DefDateFormat: string;
function DefDateMask(BlanksChar: Char): string;

const { affects DefDateFormat and DefDateMask }
   FourDigitYear: Boolean = True;

var
   WeekDays: array[1..7] of String;


implementation

uses StdUtils;

constructor TsDate.Create;
begin
   AsDateTime := SysUtils.Date;
end;

function TsDate.IsLeapYear: Boolean;
begin
   Result := sDate.IsleapYear(FYear);
end;

function TsDate.DaysInMonth(const nMonth: Word): Integer;
begin
   Result := sDate.DaysPerMonth(FYear, nMonth);
end;

procedure TsDate.PrevDay;
begin
   AsDateTime := sDate.IncDate(AsDateTime, -1, 0, 0);
end;

procedure TsDate.NextDay;
begin
   AsDateTime := sDate.IncDateEx(AsDateTime, 1, 0, 0, TRUE);
end;

procedure TsDate.PrevWeek;
begin
   AsDateTime := sDate.IncDate(AsDateTime, -7, 0, 0);
end;

procedure TsDate.NextWeek;
begin
   AsDateTime := sDate.IncDateEx(AsDateTime, 7, 0, 0, TRUE);
end;

procedure TsDate.PrevMonth;
begin
   AsDateTime := sDate.IncDate(AsDateTime, 0, -1, 0);
end;

procedure TsDate.NextMonth;
begin
   AsDateTime := sDate.IncDate(AsDateTime, 0, 1, 0);
end;

procedure TsDate.PrevYear;
begin
   AsDateTime := sDate.IncDate(AsDateTime, 0, 0, -1);
end;

procedure TsDate.NextYear;
begin
   AsDateTime := sDate.IncDate(AsDateTime, 0, 0, 1);
end;

procedure TsDate.IncDate(const nDays: Integer);
begin
   AsDateTime := AsDateTime + nDays;
end;

function TsDate.GetAsDateTime: TDateTime;
begin
   Result := 0.0;
   if (FYear > 0) and (FMonth in [1..12]) and (FDay in [1..31]) then try
      Result := EncodeDate(FYear, FMonth, FDay);
   except
   end;
end;

procedure TsDate.SetAsDateTime(const Value: TDateTime);
var
   d, m, y: Word;
begin
   try
      if Value <= 0 then
         Abort;
      DecodeDate(Value, y, m, d);
   except
      y := 0;
      m := 0;
      d := 0;
   end;
   if (FYear <> y) or (FMonth <> m) or (FDay <> d) then begin
      DateBeforeChange( Value);
      FYear := y;
      FMonth := m;
      FDay := d;
      DateChanged;
   end;
end;

function TsDate.GetAsString: string;
var
   d, m, y: word;
begin
   try
      d := FDay;
      m := FMonth;
      y := FYear;
      if FDateOrder = doMDY then
         ExchangeWordValues(d, m)
      else if FDateOrder = doYMD then
         ExchangeWordValues(d, y);
      Result := Format('%d/%d/%d', [d, m, y]);
   except
      Result := '';
   end;
end;

procedure TsDate.SetAsString(const Value: string);
var
   d, m, y: Word;

   function ParceString: Boolean;
   var
      Cell: Integer;
      ii, cnt: Integer;
      Val, S: string;
      sep: Char;
   begin
      Result := FALSE;

      val := StripWS(Value);

      sep := #0;
      cell := 0;
      cnt := 0;

      // analize string
      for ii := 1 to Length(Val) do begin
         if Val[ii] in ['0'..'9'] then begin
            Inc(cnt);
            if (cnt > 4) then
               Exit; // wrong format
         end else begin
            if sep = #0 then
               sep := Val[ii] // first ocurance of separator
            else if sep <> Val[ii] then
               Exit; // assumes that separators shuld be the same
            Inc(cell);
            if (cell > 2) or (cnt = 0) then
               Exit;
            cnt := 0;
         end;
      end;
      if (cell < 2) or (cnt = 0) then
         Exit;
      S := GetToken(Val, sep, FALSE);
      d := StrToInt(S);
      S := GetToken(Val, sep, FALSE);
      m := StrToInt(S);
      S := GetToken(Val, sep, FALSE);
      y := StrToInt(S);
      if FDateOrder = doMDY then
         ExchangeWordValues(d, m)
      else if FDateOrder = doYMD then
         ExchangeWordValues(d, y);
      Result := IsValidDate(y, m, d);
   end;
begin
   if ParceString then begin
      AsDateTime := EncodeDate(y, m, d);
   end else
      AsDateTime := -1;
end;

procedure TsDate.Date(y, m, d: Word);
begin
   ValidateDate( y, m, d);
   AsDateTime := EncodeDate(y, m, d);
end;

procedure TsDate.SetYear(const Value: Word);
begin
   if Value <> FYear then
      Date(Value, FMonth, FDay);
end;

procedure TsDate.SetMonth(const Value: Word);
begin
   if Value <> FMonth then
      Date(FYear, Value, FDay);
end;

procedure TsDate.SetDay(const Value: Word);
begin
   if Value <> FDay then
      Date(FYear, FMonth, Value);
end;

procedure TsDate.DateBeforeChange( Value: TDateTime);
begin
   if Assigned(FBeforeDateChange) then
      FBeforeDateChange(self, Value);
end;

procedure TsDate.DateChanged;
begin
   if Assigned(FOnDateChange) then
      FOnDateChange(self);
end;

function TsDate.GetIsNull: Boolean;
begin
   Result := (FYear = 0) or (FMonth = 0) or (FDay = 0); // invalid date
end;

procedure TsDate.SetNull;
begin
   FYear := 0;
   FMonth := 0;
   FDay := 0;
end;

{TsCalendarDate}

procedure TsCalendarDate.LoadDateArray;
var
   ii: Integer;
   y, m, d, prevDays: Word;
begin
   FMonthStart := GetMonthBegin(FYear, FMonth);
   if FMonthStart = -1 then
      Exit;
   FMonthEnd := FMonthStart + DaysInMonth(FMonth) - 1;

   y := FYear;
   m := FMonth - 1;
   ValidateDate( y, m, d);
   prevDays := DaysInMonth(m);
   d := prevDays - FMonthStart + 2;
   if d > prevDays then
      FStartDate := EncodeDate( FYear, FMonth, 1)
   else
      FStartDate := EncodeDate( y, m, d);

   for ii := 1 to 42 do begin
      if ii < FMonthStart then
         FDateArray[ii] := ii + prevDays - FMonthStart + 1
      else if ii > FMonthEnd then
         FDateArray[ii] := ii - FMonthEnd
      else
         FDateArray[ii] := ii - FMonthStart + 1;
   end;

   prevDays := trunc(FStartDate - EncodeDate( y, 1, 1));
   FWeeksArray[1] := Round((FStartDate - EncodeDate( y, 1, 1)) / 7 )+ 1;

   Inc(prevDays, 7);
   if prevDays > 365 then begin
      // next year started, so i do not care about exact values of prevDays,
      //since it is anyway less then 365
      prevDays := 0;
      FWeeksArray[1] := 1;
   end;

   for ii := 2 to 6 do begin
      if prevDays > 365 then begin
         prevDays := 0;
         FWeeksArray[ii] := 1
      end else
         FWeeksArray[ii] := FWeeksArray[ii-1] + 1;
      Inc(prevDays, 7);
   end;

   FTodayIndex := -1;
   ii := Trunc(SysUtils.Date - FStartDate) + 1;
   if ii in [1..42] then
      FTodayIndex := ii;
end;

function TsCalendarDate.GetMonthBegin(Year, Month: Word): ShortInt;
begin
   try
      Result := SysUtils.DayOfWeek(EncodeDate(Year, Month, 1)) - Ord(FBeginingOfWeek) - 1;
      if Result < 1 then
         Inc(Result, 7);
   except
      Result := -1;
   end;
end;

function TsCalendarDate.MajorMonth(index: ShortInt): ShortInt;
begin
   if index < FMonthStart then
      Result := -1
   else if index > FMonthEnd then
      Result := 1
   else
      Result := 0;
end;

function TsCalendarDate.DayOfWeek(index: ShortInt): TWeekDay;
var
   i: Integer;
begin
   i := (index mod 7) + Ord(FBeginingOfWeek) + 6;
   while i > 6 do
      Dec(i, 7);
   Result := TWeekDay(i);
end;

function TsCalendarDate.GetIndexFromDate: Integer;
begin
   Result := Trunc(AsDateTime) - Trunc(FStartDate) + 1;
   if not (Result in [1..42]) then
      Result := -1;
end;

procedure TsCalendarDate.SetBeginingOfWeek(const Value: TWeekDay);
begin
   if Value <> FBeginingOfWeek then begin
      FBeginingOfWeek := Value;
      LoadDateArray;
      FCurrentDateIndex := GetIndexFromDate;
   end;
end;

procedure TsCalendarDate.SetCurrentDateIndex(const Value: ShortInt);
begin
   if (Value <> FCurrentDateIndex) and (Value in [0..42]) then
      IncDate( Value - CurrentDateIndex);
end;

function TsCalendarDate.NeedsReload: Boolean;
var
   index: Integer;
begin
   Result := FForceReload;
   FForceReload := FALSE;
   if not Result then begin
      index := GetIndexFromDate;
      if FSmartReload then
         Result := index = -1
      else
         Result := (index < FMonthStart) or (index > FMonthEnd);
   end;
end;

procedure TsCalendarDate.DateBeforeChange( Value: TDateTime);
begin
   inherited;
end;

procedure TsCalendarDate.DateChanged;
begin
   FReloaded := FALSE;
   if NeedsReload then begin
      LoadDateArray;
      FReloaded := TRUE;
   end;
   FCurrentDateIndex := GetIndexFromDate;
   inherited;
end;

procedure TsCalendarDate.SetMonth(const Value: Word);
begin
   FForceReload := TRUE;
   inherited;
end;

procedure TsCalendarDate.PrevMonth;
begin
   if FSmartReload then begin
      FMonth := FMonth - MajorMonth(GetIndexFromDate);
      ValidateDate( FYear, FMonth, FDay);
   end;
   FForceReload := TRUE;
   inherited;
end;

procedure TsCalendarDate.NextMonth;
begin
   if FSmartReload then begin
      FMonth := FMonth - MajorMonth(GetIndexFromDate);
      ValidateDate( FYear, FMonth, FDay);
   end;
   FForceReload := TRUE;
   inherited;
end;

{
   functions
}

function ExtractDay(ADate: TDateTime): Word;
var
   M, Y: Word;
begin
   DecodeDate(ADate, Y, M, Result);
end;

function ExtractMonth(ADate: TDateTime): Word;
var
   D, Y: Word;
begin
   DecodeDate(ADate, Y, Result, D);
end;

function ExtractYear(ADate: TDateTime): Word;
var
   D, M: Word;
begin
   DecodeDate(ADate, Result, M, D);
end;

function IsLeapYear(AYear: Integer): Boolean;
begin
   Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

function DaysPerMonth(AYear, AMonth: Integer): Integer;
const
   DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
   Result := DaysInMonth[AMonth];
   if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

procedure ValidateDate( var Year, Month, Day: Word);
begin
   if Month < 1 then begin
      Month := 12;
      Dec(Year);
   end else if Month > 12 then begin
      Month := 1;
      Inc(Year);
   end;
   if Day > DaysPerMonth( Year, Month) then
      Day := DaysPerMonth( Year, Month);
end;

function IsValidDate(year, Month, Day: Word): Boolean;
begin
   Result := Month in [1..12];
   if Result then
      Result := (Day > 0) and (Day <= DaysPerMonth(Year, Month));
end;

function IncDate(ADate: TDateTime; Days, Months, Years: Integer): TDateTime;
begin
   Result := IncDateEx( ADate, Days, Months, Years, FALSE);
end;

function IncDateEx(ADate: TDateTime; Days, Months, Years: Integer; floatingMonth: Boolean): TDateTime;
var
   D, M, Y: Word;
   Year, Month, Day, Day28Delta: LongInt;
begin
   DecodeDate(ADate, Y, M, D);
   Year := y;
   Month := m;
   Day := d;
   Day28Delta := Day - 28;
   if Day28Delta < 0 then
      Day28Delta := 0
   else
      Day := 28;
   Inc(Year, Years);
   Inc(Year, Months div 12);
   Inc(Month, Months mod 12);
   if Month < 1 then begin
      Inc(Month, 12);
      Dec(Year);
   end else if Month > 12 then begin
      Dec(Month, 12);
      Inc(Year);
   end;
   Day := Day + Days + Day28Delta;
   while Day < 1 do begin
      Dec(Month);
      if Month < 1 then begin
         Inc(Month, 12);
         Dec(Year);
      end;
      Day := DaysPerMonth(Year, Month) - Abs(Day);
   end;

   if floatingMonth then begin
      while Day > DaysPerMonth(Year, Month) do begin
         dec(day, DaysPerMonth(Year, Month));
         Inc( month);
         if Month > 12 then begin
            Month := 1;
            Inc(year);
         end;
      end;
   end else begin
      if day > DaysPerMonth(Year, Month) then
         day := DaysPerMonth(Year, Month);
   end;
   Result := EncodeDate(Year, Month, Day);
end
;

function IncDay(ADate: TDateTime; Delta: Integer): TDateTime;
begin
   Result := IncDate(ADate, Delta, 0, 0);
end;

function IncMonth(ADate: TDateTime; Delta: Integer): TDateTime;
begin
   Result := IncDate(ADate, 0, Delta, 0);
end;

function IncYear(ADate: TDateTime; Delta: Integer): TDateTime;
begin
   Result := IncDate(ADate, 0, 0, Delta);
end;

{ String to date conversions    }
{ Copied from SYSUTILS.PAS unit }

procedure ScanBlanks(const S: string; var Pos: Integer);
var
   I: Integer;
begin
   I := Pos;
   while (I <= Length(S)) and (S[I] = ' ') do Inc(I);
   Pos := I;
end;

function ScanNumber(const S: string; var Pos: Integer; var Number: Word): Boolean;
var
   I: Integer;
   N: Word;
begin
   Result := False;
   ScanBlanks(S, Pos);
   I := Pos;
   N := 0;
   while (I <= Length(S)) and (S[I] in ['0'..'9']) and (N < 1000) do begin
      N := N * 10 + (Ord(S[I]) - Ord('0'));
      Inc(I);
   end;
   if I > Pos then begin
      Pos := I;
      Number := N;
      Result := True;
   end;
end;

function ScanChar(const S: string; var Pos: Integer; Ch: Char): Boolean;
begin
   Result := False;
   ScanBlanks(S, Pos);
   if (Pos <= Length(S)) and (S[Pos] = Ch) then begin
      Inc(Pos);
      Result := True;
   end;
end;

function GetDateOrder(const DateFormat: string): TDateOrder;
var
   I: Integer;
begin
   Result := doDMY;
   I := 1;
   while I <= Length(DateFormat) do begin
      case Chr(Ord(DateFormat[I]) and $DF) of
         'Y': Result := doYMD;
         'M': Result := doMDY;
         'D': Result := doDMY;
      else
         Inc(I);
         Continue;
      end;
      Exit;
   end;
end;

function ScanDate(const S, DateFormat: string; var Pos: Integer; var Date: TDateTime): Boolean;
var
   DateOrder: TDateOrder;
   N1, N2, N3, Y, M, D: Word;
begin
   Result := False;
   DateOrder := GetDateOrder(DateFormat);
   if not (ScanNumber(S, Pos, N1) and ScanChar(S, Pos, DateSeparator) and
      ScanNumber(S, Pos, N2)) then Exit;
   if ScanChar(S, Pos, DateSeparator) then begin
      if not ScanNumber(S, Pos, N3) then
         Exit;
      Y := 0;
      M := 0;
      D := 0; // to bypass compiler Warning
      case DateOrder of
         doMDY: begin Y := N3;
               M := N1;
               D := N2;
            end;
         doDMY: begin Y := N3;
               M := N2;
               D := N1;
            end;
         doYMD: begin Y := N1;
               M := N2;
               D := N3;
            end;
      end;
      if Y <= 99 then
         Y := Y + (ExtractYear(SysUtils.Date) div 100) * 100;
   end else begin
      Y := ExtractYear(SysUtils.Date);
      if DateOrder = doDMY then begin
         D := N1;
         M := N2
      end else begin
         M := N1;
         D := N2;
      end;
   end;
   ScanBlanks(S, Pos);
   try
      Date := EncodeDate(Y, M, D);
      Result := True;
   except
      Result := False;
   end;
end;

function StrToDateFmt(const DateFormat, S: string): TDateTime;
var
   Pos: Integer;
begin
   Pos := 1;
   if not ScanDate(S, DateFormat, Pos, Result) or (Pos <= Length(S)) then
      raise EConvertError.CreateFmt( SInvalidDate, [S]);
end;

function StrToDateDef(const S: string; Default: TDateTime): TDateTime;
var
   Pos: Integer;
begin
   Pos := 1;
   if not ScanDate(S, ShortDateFormat, Pos, Result) or (Pos <= Length(S)) then
      Result := Trunc(Default);
end;

function StrToDateFmtDef(const DateFormat, S: string; Default: TDateTime): TDateTime;
var
   Pos: Integer;
begin
   Pos := 1;
   if not ScanDate(S, DateFormat, Pos, Result) or (Pos <= Length(S)) then
      Result := Trunc(Default);
end;

function DefDateFormat: string;
begin
   if FourDigitYear then begin
      case GetDateOrder(ShortDateFormat) of
         doMDY: Result := 'MM/DD/YYYY';
         doDMY: Result := 'DD/MM/YYYY';
         doYMD: Result := 'YYYY/MM/DD';
      end;
   end else begin
      case GetDateOrder(ShortDateFormat) of
         doMDY: Result := 'MM/DD/YY';
         doDMY: Result := 'DD/MM/YY';
         doYMD: Result := 'YY/MM/DD';
      end;
   end;
end;

function DefDateMask(BlanksChar: Char): string;
begin
   if FourDigitYear then begin
      case GetDateOrder(ShortDateFormat) of
         doMDY, doDMY: Result := '!99/99/9999;1;';
         doYMD: Result := '!9999/99/99;1;';
      end;
   end else begin
      case GetDateOrder(ShortDateFormat) of
         doMDY, doDMY: Result := '!99/99/99;1;';
         doYMD: Result := '!99/99/99;1;';
      end;
   end;
   if Result <> '' then Result := Result + BlanksChar;
end;

procedure Initlocals; far;
var
   ii: Integer;
begin
   for ii := 1 to 7 do
      WeekDays[ii] := FormatDateTime( 'ddd', Encodedate( 1901, 12, ii)); // 1/12/1901 is sunday
end;

initialization
   FourDigitYear := Pos('YYYY', AnsiUpperCase(ShortDateFormat)) > 0;
   Initlocals;
end.

