{*******************************************************************************
   Unit
      sCtrls.pas
   Description:
      Transparent, BWCC style CheckBox and RadioButton
   Versions:
      2.0*
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   Comments:
      This components are part of sEditTools. Why BWCC ? Whell, I believe that
      it's style is more "flat" than Microsoft's check box and radio button. If you
      are not agree - use transparent controls from Dream Company, they are great!

*     I did not track the versions before, so let's consider it as 2.0
**    Enrico Lodolo - ldlc18k1@bo.nettuno.it -
         Idea and some code in the box drawing part was taken from his
         TBorRadio and TBorCheck. Thanks!
*******************************************************************************}
unit sCtrls;

interface

uses
   SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
   Forms, Dialogs, StdCtrls, Menus, sTrans, sGraphics;

type
   TTransparentMode = (tmOpaque, tmSemiTransparent, tmTransparent);

   TsCtrlState = (scsMouseInControl, scsFocused, scsInvalidateCaption);
   TsCtrlStates = set of TsCtrlState;

   TVerticalAlignment = (vaTop, vaBottom, vaCenter);
   TsBWCCBoxStyle = (bwsRaised, bwsSunken);

   TsBWCCControl = class(TsTransControl)
   private
      FAlignment: TLeftRight;
      FCheckColor: TColor;
      FCtrlState: TsCtrlStates;
      FDown: Boolean;
      FFont3d: TFont3d;
      FHilightStyle: TTextHilightStyle;
      FHilight3d: TFont3d;
      FTextAlignment: TAlignment;
      FTransparentMode: TTransparentMode;
      FVerticalAlignment: TVerticalAlignment;
      FWordWrap: Boolean;
      FOnMouseEnter: TNotifyEvent;
      FOnMouseLeave: TNotifyEvent;
      procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure Font3dChanged(Sender: Tobject);
      function GetCaption: TCaption;
      procedure SetAlignment(Value: TLeftRight);
      procedure SetCaption(const Value: TCaption);
      procedure SetCheckColor(Value: TColor);
      procedure SetFont3d(Value: TFont3d);
      procedure SetHilight3d(Value: TFont3d);
      procedure SetHilightStyle(Value: TTextHilightStyle);
      procedure SetTextAlignment(Value: TAlignment);
      procedure SetTransparentMode(Value: TTransparentMode);
      procedure SetVerticalAlignment(Value: TVerticalAlignment);
      procedure SetWordWrap(Value: Boolean);
      procedure PaintCaption;
   protected
      FState: TCheckBoxState;
      procedure Paint; override;
      procedure PaintBox(R: TRect); virtual; abstract;
      procedure Toggle; virtual; abstract;
      procedure SetChecked(Value: Boolean); virtual;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
      procedure KeyDown(var Key: Word; Shift: TShiftSTate); override;
      procedure KeyUp(var Key: Word; Shift: TShiftSTate); override;
      procedure DoEnter; override;
      procedure DoExit; override;
      procedure DoMouseEnter; virtual;
      procedure DoMouseLeave; virtual;
      function EditCanModify: Boolean; virtual;
      function GetChecked: Boolean;
      property Alignment: TLeftRight read FAlignment write SetAlignment default taLeftJustify;
      property Caption: TCaption read GetCaption write SetCaption;
      property CheckColor: TColor read FCheckColor write SetCheckColor default clBlack;
      property Checked: Boolean read GetChecked write SetChecked default False;
      property Font3d: TFont3d read FFont3d write SetFont3d;
      property HilightStyle: TTextHilightStyle read FHilightStyle write SetHilightStyle default hsFlushing;
      property Hilight3d: TFont3d read FHilight3d write SetHilight3d;
      property TextAlignment: TAlignment read FTextAlignment write SetTextAlignment default taLeftJustify;
      property TransparentMode: TTransparentMode read FTransparentMode write SetTransparentMode default tmOpaque;
      property VerticalAlignment: TVerticalAlignment read FVerticalAlignment write SetVerticalAlignment default vaTop;
      property WordWrap: Boolean read FWordWrap write SetWordWrap default FALSE;
      property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
      property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
   end;

   TsCustomCheckBox = class(TsBWCCControl)
   private
      FAllowGrayed: Boolean;
      FStyle: TsBWCCBoxStyle;
      procedure SetStyle( Value: TsBWCCBoxStyle);
      procedure SetState(Value: TCheckBoxState);
   protected
      procedure PaintBox(R: TRect); override;
      property AllowGrayed: Boolean read FAllowGrayed write FAllowGrayed default FALSE;
      property State: TCheckBoxState read FState write SetState default cbUnchecked;
      property Style: TsBWCCBoxStyle read FStyle write SetStyle default bwsRaised;
   public
      procedure Toggle; override;
   end;

   TsCheckBox = class(TsCustomCheckBox)
   published
      property Alignment;
      property AllowGrayed;
      property Caption;
      property CheckColor;
      property Checked;
      property Color;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Font;
      property Font3d;
      property Hilight3d;
      property HilightStyle;
      property ParentFont;
      property PopupMenu;
      property ShowHint;
      property State;
      property Style;
      property TabOrder;
      property TabStop;
      property TextAlignment;
      property TransparentMode;
      property VerticalAlignment;
      property WordWrap;
      property OnClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnMouseEnter;
      property OnMouseLeave;
   end;

   TsCustomRadioButton = class(TsBWCCControl)
   private
      FGroupIndex: Byte;
   protected
      procedure PaintBox(R: TRect); override;
      procedure SetChecked(Value: Boolean); override;
      property GroupIndex: Byte read FGroupIndex write FGroupIndex default 0;
   public
      procedure Toggle; override;
   end;

   TsRadioButton = class(TsCustomRadioButton)
   published
      property Alignment;
      property Caption;
      property CheckColor;
      property Checked;
      property Color;
      property DragCursor;
      property DragMode;
      property Enabled;
      property Font;
      property Font3d;
      property GroupIndex;
      property Hilight3d;
      property HilightStyle;
      property ParentFont;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property TextAlignment;
      property TransparentMode;
      property VerticalAlignment;
      property WordWrap;
      property OnClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnMouseEnter;
      property OnMouseLeave;
   end;


implementation

const
   BoxWidth = 12;

constructor TsBWCCControl.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   Color := clBtnFace;
   FFont3d := TFont3d.Create(Font3dChanged);
   FHilight3d := TFont3d.Create(Font3dChanged);
   Width := 98;
   Height := 20;
   ParentColor := FALSE;
   FTransparentMode := tmOpaque;
   FAlignment := taLeftJustify;
   FVerticalAlignment := vaTop;
   FTextAlignment := taLeftJustify;
   FHilightStyle := hsFlushing;
end;

destructor TsBWCCControl.Destroy;
begin
   FHilight3d.Free;
   FFont3d.Free;
   inherited;
end;

procedure TsBWCCControl.CMMouseEnter(var Message: TMessage);
begin
   Include(FCtrlState, scsMouseInControl);
   DoMouseEnter;
   if not (scsFocused in FCtrlState) then begin
      Include( FCtrlState, scsInvalidateCaption);
      Paint;
   end;
end;

procedure TsBWCCControl.CMMouseLeave(var Message: TMessage);
begin
   Exclude(FCtrlState, scsMouseInControl);
   DoMouseLeave;
   if not (scsFocused in FCtrlState) then begin
      Include( FCtrlState, scsInvalidateCaption);
      Paint;
   end;
end;

procedure TsBWCCControl.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
   if Transparent then
      inherited
   else
      Msg.Result := 1;
end;

procedure TsBWCCControl.Paint;
var
   iR, R: TRect;
begin
   if BackgroundChanged or FDown or (csDesigning in ComponentState) then
      FCtrlState := FCtrlState - [scsInvalidateCaption];
   inherited;
   with Canvas do begin
      case FVerticalAlignment of
         vaTop: begin
               R.Top := 1;
               R.Bottom := BoxWidth + 1;
            end;
         vaBottom: begin
               R.Top := Height - BoxWidth - 1;
               R.Bottom := Height - 1;
            end;
         else begin // vaCenter
            R.Top := (Height div 2) - (BoxWidth div 2);
            R.Bottom := R.Top + BoxWidth;
         end;
      end;

      if FAlignment = taRightJustify then begin
         R.Left := Width - 1 - BoxWidth;
         R.Right := Width - 1;
      end else begin
         R.Left := 1;
         R.Right := BoxWidth + 1;
      end;
   end;
   if scsInvalidateCaption in FCtrlState then begin
      iR := Bounds(0, 0, Width - BoxWidth - 4, Height - 2);
      if FAlignment = taLeftJustify then
         OffsetRect( iR, BoxWidth + 4, 0);
   end else
      iR := ClientRect;

   if Transparent then
      PaintBackgroundRect(iR)
   else with Canvas do begin
      Brush.Style := bsSolid;
      Brush.Color := Color;
      FillRect(iR);
   end;

   Canvas.Font := Font;
   if not (scsInvalidateCaption in FCtrlState) then
      PaintBox(R);
   PaintCaption;
   FCtrlState := FCtrlState - [scsInvalidateCaption];
end;

procedure TsBWCCControl.PaintCaption;
const
   Alignments: array [TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
   WordWraps: array [Boolean] of Word = (0, DT_WORDBREAK);
var
   oldColor: TColor;
   DrawStyle: Integer;
   R: TRect;
   hilightStyle: TTextHilightStyle;
   X: Integer;
begin
   DrawStyle := DT_EXPANDTABS or WordWraps [FWordWrap] or Alignments [FTextAlignment];

   if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
      hilightStyle := FHilightStyle
   else
      hilightStyle := hsNone;

   if FAlignment = taLeftJustify then
      X := BoxWidth + 7
   else
      X := 1;
   R := Bounds(X, 1, Width - BoxWidth - 9, Height - 2);

   PaintText( Canvas.Handle, PChar(Caption), Length(Caption), R,
      DrawStyle or DT_CALCRECT, FFont3d, FHilight3d, hilightStyle);


   R.left := X;
   R.Right := X + Width - BoxWidth - 9;

   if FVerticalAlignment = vaBottom then
      OffsetRect(R, 0, Height - R.Bottom)
   else if FVerticalAlignment = vaCenter then
      OffsetRect(R, 0, (Height - R.Bottom) div 2);

   with Canvas do begin
      Brush.Style := bsClear;
      oldColor := Font.Color;
      if not Enabled then
         Font.Color := clGray;

      if FDown then
         OffsetRect(R, 1, 1);
      PaintText(Handle, PChar(Caption), Length(Caption), R, DrawStyle,
         FFont3d, FHilight3d, hilightStyle);

      if not Enabled then
         Font.Color := oldColor;

      if (scsFocused in FCtrlState) and (hilightStyle <> hsFlushing) then begin
         PaintText( Handle, PChar(Caption), Length(Caption), R, DrawStyle or DT_CALCRECT,
            FFont3d, FHilight3d, hilightStyle);
         case FTextAlignment of
            taCenter:
               OffsetRect(R, (Width - BoxWidth - 9 - R.Right + R.Left) div 2, 0);
            taRightJustify:
               OffsetRect(R, (Width - BoxWidth - 9 - R.Right + R.Left), 0);
         end;
         InflateRect(R, 2, 0);
         Brush.Color := clBtnFace;
         FrameRect(R);
         DrawFocusRect(R);
      end;
   end;
end;

procedure TsBWCCControl.SetTransparentMode(Value: TTransparentMode);
begin
   if FTransparentMode <> Value then begin
      FTransparentMode := Value;
      Transparent := Value <> tmOpaque;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetAlignment(Value: TLeftRight);
begin
   if Value <> FAlignment then begin
      FAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetTextAlignment(Value: TAlignment);
begin
   if FTextAlignment <> Value then begin
      FTextAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetVerticalAlignment(Value: TVerticalAlignment);
begin
   if FVerticalAlignment <> Value then begin
      FVerticalAlignment := Value;
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetWordWrap(Value: Boolean);
begin
   if FWordWrap <> Value then begin
      FWordWrap := Value;
      Include(FCtrlState, scsInvalidateCaption);
      Invalidate;
   end;
end;

function TsBWCCControl.GetCaption: TCaption;
var
   Buf: array [0..256] of Char;
begin
   GetTextBuf(Buf, 256);
   Result := StrPas(Buf);
end;

procedure TsBWCCControl.SetCaption(const Value: TCaption);
var
   Buffer: array [0..255] of Char;
begin
   if GetCaption <> Value then
      SetTextBuf(StrPCopy(Buffer, Value));
   Include(FCtrlState, scsInvalidateCaption);
   Invalidate;
end;

procedure TsBWCCControl.SetFont3d(Value: TFont3d);
begin
   if FFont3d <> Value then
      FFont3d.Assign(Value);
end;

procedure TsBWCCControl.Font3dChanged(Sender: Tobject);
begin
   Invalidate;
end;

procedure TsBWCCControl.SetHilight3d(Value: TFont3d);
begin
   if FHilight3d <> Value then
      FHilight3d.Assign( Value);
end;

procedure TsBWCCControl.SetHilightStyle(Value: TTextHilightStyle);
begin
   if Value <> FHilightStyle then begin
      FHilightStyle := Value;
      if [scsMouseInControl, scsFocused] * FCtrlState <> [] then
         Invalidate;
   end;
end;

function TsBWCCControl.GetChecked: Boolean;
begin
   Result := FState = cbChecked;
end;

function TsBWCCControl.EditCanModify: Boolean;
begin
   Result := TRUE;
end;

procedure TsBWCCControl.SetChecked(Value: Boolean);
const
   checkStates: array [Boolean] of TCheckBoxState = (cbUnchecked, cbChecked);
begin
   if EditCanModify and (Value <> Checked) then begin
      FState := checkStates [Value];
      Invalidate;
   end;
end;

procedure TsBWCCControl.SetCheckColor(Value: TColor);
begin
   FCheckColor := Value;
   Invalidate;
end;

procedure TsBWCCControl.DoEnter;
begin
   inherited DoEnter;
   Include( FCtrlState, scsFocused);
   Invalidate;
end;

procedure TsBWCCControl.DoExit;
begin
   inherited DoExit;
   Exclude( FCtrlState, scsFocused);
   Invalidate;
end;

procedure TsBWCCControl.DoMouseEnter;
begin
	if Assigned( FOnMouseEnter) then
      FOnMouseEnter(self);
end;

procedure TsBWCCControl.DoMouseLeave;
begin
	if Assigned( FOnMouseLeave) then
      FOnMouseLeave(self);
end;

procedure TsBWCCControl.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      SetFocus;
      inherited;
      MouseCapture := True;
      FDown := TRUE;
      Invalidate;
   end else
      inherited;
end;

procedure TsBWCCControl.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
   if Button = mbLeft then begin
      MouseCapture := False;
      FDown := FALSE;
      if PtInRect(ClientRect, Point(X, Y)) then
         Toggle;
   end;
   inherited;
end;

procedure TsBWCCControl.MouseMove(Shift: TShiftState; X, Y: Integer);
var
   oldDown: Boolean;
begin
   if MouseCapture then begin
      oldDown := FDown;
      FDown := (X >= 0) and (X <= Width) and (Y >= 0) and (Y <= Height);
      if oldDown <> FDown then
         Invalidate;
   end;
   inherited;
end;

procedure TsBWCCControl.KeyDown(var Key: Word; Shift: TShiftSTate);
begin
   if (Key = vk_Space) and not FDown then begin
      FDown := TRUE;
      Invalidate;
   end;
   inherited KeyDown(Key, Shift);
end;

procedure TsBWCCControl.KeyUp(var Key: Word; Shift: TShiftSTate);
begin
   if Key = vk_Space then begin
      FDown := False;
      Toggle;
   end;
end;

{*******************************************************************************
   TsCheckBox
*******************************************************************************}

procedure TsCustomCheckBox.SetStyle( Value: TsBWCCBoxStyle);
begin
   if FStyle <> Value then begin
      FStyle := Value;
      Invalidate;
   end;
end;

procedure TsCustomCheckBox.SetState(Value: TCheckBoxState);
begin
   if EditCanModify and (FState <> Value) then begin
      FState := Value;
      Invalidate;
   end;
end;

procedure TsCustomCheckBox.Toggle;
begin
   if EditCanModify then begin
      if FAllowGrayed and (FState = cbUnchecked) then
         State := cbGrayed
      else
         Checked := not Checked;
      Click;
   end else
      Invalidate;
end;

const
   ColorMap: array[1..2, Boolean] of TColor =
      ((clBtnHighLight, clBtnShadow), (clBtnShadow, clBtnHighLight));

procedure TsCustomCheckBox.PaintBox(R: TRect);
begin
   with Canvas do begin
      Brush.Color := Color;
      if transparentMode = tmSemiTransparent then
         Canvas.FillRect(R);
      if FDown then begin
         Pen.Color := clBlack;
         Pen.Width := 2;
         Rectangle( R.Left + 1, R.Top + 1, R.Right + 1, R.Bottom + 1);
         Pen.Width := 1;
      end else begin
         Brush.Color  := Color;
         FrameRect(R);
         Pen.Color := ColorMap[1, Boolean(FStyle)];
         PolyLine([Point(R.Left, R.Bottom), R.TopLeft, Point(R.Right, R.Top)]);
         Pen.Color := ColorMap[2, Boolean(FStyle)];
         PolyLine([Point(R.Right, R.Top), R.BottomRight, Point(R.Left, R.Bottom)]);
      end;

      case State of
         cbChecked: begin
               if FStyle = bwsSunken then
                  OffsetRect(R, 0, -1);

               Pen.Color := FCheckColor;
               Pen.Width := 1;
               Dec(R.Top);
               Dec(R.Bottom);
               Polyline([Point(R.Left + 2, R.Top + BoxWidth div 2 + 1), Point(R.Left + 2, R.Bottom - 1)]);
               Polyline([Point(R.Left + 3, R.Top + BoxWidth div 2), Point(R.Left + 3, R.Bottom - 2)]);
               Polyline([Point(R.Left + 2, R.Bottom - 1), Point(R.Right - 2, R.Top + 3)]);
               Polyline([Point(R.Left + 3, R.Bottom - 1), Point(R.Right - 1, R.Top + 3)]);
            end;
         cbGrayed: begin
               OffsetRect( R, 2, 2);
               Dec(R.Right, 3);
               Dec(R.Bottom, 3);
               if FDown then begin
                  Pen.Color := clBtnFace;
                  FrameRect(R);
               end;
               Pen.Color := ColorMap[1, Boolean(FStyle)];
               Brush.Color := FCheckColor;
               Rectangle(R.Left, R.Top, R.Right, R.Bottom);
         end;
      end;
   end;
end;

{*******************************************************************************
   TsRadioButton
*******************************************************************************}

procedure TsCustomRadioButton.Toggle;
begin
   if EditCanModify then begin
      if not Checked then begin
         Checked := TRUE;
         Click;
      end else
         Invalidate;
   end;
end;

procedure TsCustomRadioButton.SetChecked(Value: Boolean);
var
   ii: Integer;
   ctrl: TsCustomRadioButton;
begin
   inherited;
   if Value and (Parent <> nil) then for ii := 0 to Parent.ControlCount - 1 do
      if Parent.Controls[ii] is TsCustomRadioButton then begin
         ctrl := TsCustomRadioButton(Parent.Controls[ii]);
         if( ctrl <> Self) and ( ctrl.FGroupIndex = FGroupIndex) then
            Ctrl.SetChecked(FALSE);
      end;
end;

procedure TsCustomRadioButton.PaintBox(R: TRect);
var
   HalfBoxWidth: Integer;
begin
   HalfBoxWidth := BoxWidth div 2;
   with Canvas do begin
      if FDown then begin
         Pen.Color := clBlack;
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top)]);
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top + 1), Point(R.Left + 1, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom - 1), Point(R.Right - 1, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top + 1)]);
      end else begin
         Pen.Color := ColorMap[1, Checked];
         PolyLine([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Bottom)]);
         Pen.Color := ColorMap[2, Checked];
         PolyLine([Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
            Point(R.Left + HalfBoxWidth, R.Top)]);
         if FTransparentMode = tmSemiTransparent then begin
            Pen.Color := Color;
            Brush.Color := Color;
            InflateRect(R, -1, -1);
            dec(HalfBoxWidth);
            Polygon([Point(R.Left + HalfBoxWidth, R.Top), Point(R.Left, R.Top + HalfBoxWidth),
                     Point(R.Left + HalfBoxWidth, R.Bottom), Point(R.Right, R.Top + HalfBoxWidth),
                     Point(R.Left + HalfBoxWidth, R.Top)]);
            InflateRect(R, 1, 1);
            Inc(HalfBoxWidth);
         end;
      end;
      if Checked then begin
         Pen.Color := FCheckColor;
         Inc(R.Left, HalfBoxWidth);
         Inc(R.Top, HalfBoxWidth);

         PolyLine([Point(R.Left - 1, R.Top - 1), Point(R.Left + 2, R.Top - 1)]);
         PolyLine([Point(R.Left - 2, R.Top), Point(R.Left + 3, R.Top)]);
         PolyLine([Point(R.Left - 1, R.Top + 1), Point(R.Left + 2, R.Top + 1)]);
         PolyLine([Point(R.Left, R.Top - 2), Point(R.Left, R.Top + 3)]);
      end;
   end;
end;


end.

