{*******************************************************************************
   Unit
      sCombos.pas
   Description:
      Flat style combo box controls.
   Versions:
      2.0*
   Autor(s):
      Dimitry Statilko - dstatus@iname.com, dima@mobitel.com**
   Comments:
*     I did not track the versions before, so let's consider it as 1.5
**    TsColorComboBox and TsFontComboBox are taken from RX package with a minor
      modifications.
*******************************************************************************}
unit sCombos;

interface

{$I S.Inc}

uses Windows, Messages, Classes, Forms, Controls, Graphics, sGraphics, StdCtrls;

type
   TsComboState = set of (msMouseInControl, msButtonPainted);

   TsCustomComboBox = class(TCustomComboBox)
   private
      FListColor: TColor;
      FChildHandle: HWND;
      FListHandle: HWND;
      FListInstance: Pointer;
      FDefListProc: Pointer;
      FDisabledFont: TFont;
      FFlat: Boolean;
      FColor: TColor;
      FParentColor: Boolean;
      FEditState: TsComboState;
      FButtonWidth: Integer;
      FSysBtnWidth: Integer;
      FSolidBorder: Boolean;
      FDisabledStyle: TsDisabledStyle;
      procedure SetFlat(const Value: Boolean);
      procedure SetDisabledStyle(Value: TsDisabledStyle);
      procedure SetDisabledFont(Value: TFont);
      function GetButtonRect: TRect;
      procedure PaintDisabled(PS: TPaintStruct);
      procedure PaintButton;
      procedure PaintButtonBorder;
      procedure PaintListFrame;
      procedure RedrawBorders;
      procedure SetupEnabled;
      procedure InvalidateSelection;
      function GetSolidBorder: Boolean;
      procedure SetSolidBorder;
      procedure ListWndProc(var Message: TMessage);
      procedure WMSetFocus(var Message: TMessage); message WM_SETFOCUS;
      procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS;
      procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
      procedure CMEnabledChanged(var Msg: TMessage); message CM_ENABLEDCHANGED;
      procedure WMKeyDown(var Message: TMessage); message WM_KEYDOWN;
      procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
      procedure WMNCPaint(var Message: TMessage); message WM_NCPAINT;
      procedure CNCommand(var Message: TWMCommand); message CN_COMMAND;
      procedure CNDrawItem(var Message: TWMDrawItem); message CN_DRAWITEM;
   protected
      procedure WndProc(var Message: TMessage); override;
      procedure ComboWndProc(var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer); override;
      procedure CreateWnd; override;
      procedure CreateParams(var Params: TCreateParams); override;
{$IFDEF V1_COMP}      {Back compatibility}
      procedure ReadDefaultDraw(Reader: TReader);
      procedure WriteNothing(Writer: TWriter);
      procedure DefineProperties(Filer: TFiler); override;
{$ENDIF}
      property SolidBorder: Boolean read FSolidBorder;
      property DisabledFont: TFont read FDisabledFont write SetDisabledFont;
      property DisabledStyle: TsDisabledStyle read FdisabledStyle write SetDisabledStyle default dsDefault;
      property Flat: Boolean read FFlat write SetFlat;
      property ListColor: TColor read FListColor write FListColor default clBtnFace;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
   end;

   TsComboBox = class(TsCustomComboBox)
   published
      property Style; {Must be published before Items}
      property Color;
      property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
      property DragMode;
      property DragCursor;
      property DropDownCount;
      property Enabled;
      property Flat;
      property Font;
      property ItemHeight;
      property Items;
      property ListColor;
      property MaxLength;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property Sorted;
      property TabOrder;
      property TabStop;
      property Text;
      property Visible;
      property ItemIndex;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnDrawItem;
      property OnDropDown;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMeasureItem;
      property OnStartDrag;
   end;

{ TColorComboBox }
   TsColorComboBox = class(TsCustomComboBox)
   private
      FColorValue: TColor;
      FDisplayNames: Boolean;
      FFullScheme: Boolean;
      FOnChange: TNotifyEvent;
      procedure SetColorValue(NewValue: TColor);
      procedure SetDisplayNames(Value: Boolean);
      procedure SetFullScheme(Value: Boolean);
      procedure ResetItemHeight;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
   protected
      procedure CreateWnd; override;
      procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
      procedure Click; override;
      procedure BuildList; virtual;
      procedure Change; override;
   public
      constructor Create(AOwner: TComponent); override;
      property Text;
   published
      property ColorValue: TColor read FColorValue write SetColorValue default clBlack;
      property DisabledFont;
      property DisabledStyle;
      property DisplayNames: Boolean read FDisplayNames write SetDisplayNames default True;
      property Color;
      property Ctl3D;
      property DragMode;
      property DragCursor;
      property Enabled;
      property Flat;
      property Font;
      property FullScheme: Boolean read FFullScheme write SetFullScheme;
      property ImeMode;
      property ImeName;
      property ListColor;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnDropDown;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnStartDrag;
   end;

{ TsFontComboBox }
   TFontDevice = (fdScreen, fdPrinter, fdBoth);
   TFontListOption = (foAnsiOnly, foTrueTypeOnly, foFixedPitchOnly, foNoOEMFonts, foOEMFontsOnly, foScalableOnly);
   TFontListOptions = set of TFontListOption;

   TsFontComboBox = class(TsCustomComboBox)
   private
      TrueTypeBMP, DeviceBMP: TBitmap;
      FOnChange: TNotifyEvent;
      FDevice: TFontDevice;
      FPreviewFont: Boolean;
      FUpdate: Boolean;
      FOptions: TFontListOptions;
      procedure SetFontName(const NewFontName: string);
      function GetFontName: string;
      function GetTrueTypeOnly: Boolean;
      procedure SetDevice(Value: TFontDevice);
      procedure SetOptions(Value: TFontListOptions);
      procedure SetTrueTypeOnly(Value: Boolean);
      procedure SetPreviewFont(Value: Boolean);
      procedure ResetItemHeight;
      procedure Reset;
      procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
      procedure WMFontChange(var Message: TMessage); message WM_FONTCHANGE;
   protected
      procedure BuildList; virtual;
      procedure Click; override;
      procedure Change; override;
      procedure CreateWnd; override;
      procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Text;
   published
      property Color;
      property Ctl3D;
      property DisabledStyle;
      property DisabledFont;
      property Device: TFontDevice read FDevice write SetDevice default fdScreen;
      property DragMode;
      property DragCursor;
      property Enabled;
      property Flat;
      property Font;
      property FontName: string read GetFontName write SetFontName;
      property Options: TFontListOptions read FOptions write SetOptions default [];
      property ImeMode;
      property ImeName;
      property ListColor;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property PreviewFont: Boolean read FPreviewFont write SetPreviewFont;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property TrueTypeOnly: Boolean read GetTrueTypeOnly write SetTrueTypeOnly stored False; { obsolete, use Options instead }
      property Visible;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnDropDown;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnStartDrag;
   end;

   TsFilesListOption = (flArchive, flDirectory, flDrives, flExclusive,
      flHidden, flReadOnly, flReadWrite, flSystem);
   TsFilesListOptions = set of TsFilesListOption;

   TsPathComboBox = class(TsCustomComboBox)
   private
      FIconHeight: Integer;
      FPath: String;
      FFileMask: String;
      FOptions: TsFilesListOptions;
      FLocked: Boolean;
      FImageList: THandle;
      function FileMaskStored: Boolean;
      function GetDisplayIcons: Boolean;
      procedure SetDisplayIcons(Value: Boolean);
      procedure SetFileMask(Value: String);
      procedure SetOptions(Value: TsFilesListOptions);
      procedure SetPath(Value: String);
      procedure CreateWnd; override;
   protected
      procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure BeginUpdate;
      procedure EndUpdate;
      procedure FillList;
      property Items;
      property Text;
   published
      property Color;
      property Ctl3D;
      property DisabledFont;
      property DisabledStyle;
      property DisplayIcons: Boolean read GetDisplayIcons write SetDisplayIcons default TRUE;
      property DragMode;
      property DragCursor;
      property DropDownCount;
      property Enabled;
      property FileMask: String read FFileMask write SetFileMask stored FileMaskStored;
      property Flat;
      property Font;
      property Path: String read FPath write SetPath;
      property ImeMode;
      property ImeName;
      property ListColor;
      property Options: TsFilesListOptions read FOptions write SetOptions default [flReadWrite];
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property Sorted;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnDropDown;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMeasureItem;
      property OnStartDrag;
   end;


implementation
{$R *.RES}

uses sFlat, SysUtils, stdUtils, Printers, sFileUtils, ShellApi, Commctrl;

{ TsCustomComboBox}

constructor TsCustomComboBox.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   ControlStyle := ControlStyle - [csFixedHeight];
   TControlCanvas(Canvas).Control := self;
   FDisabledFont := TFont.Create;
   FDisabledFont.Assign(Font);
   FDisabledStyle := dsDefault;
   FButtonWidth := 11;
   FSysBtnWidth := GetSystemMetrics(SM_CXVSCROLL);
   FColor := inherited Color;
   FParentColor := inherited ParentColor;
   FListInstance := MakeObjectInstance(ListWndProc);
   FListColor := clBtnFace;
   height := 10;
end;

destructor TsCustomComboBox.Destroy;
begin
   FDisabledFont.Free;
   inherited;
   FreeObjectInstance(FListInstance);
end;

procedure TsCustomComboBox.CreateWnd;
begin
   inherited;
   if FChildHandle <> EditHandle then
      FListHandle := FChildHandle;
   FDefListProc := Pointer(GetWindowLong(FListHandle, GWL_WNDPROC));
   SetWindowLong(FListHandle, GWL_WNDPROC, Longint(FListInstance));
end;

procedure TsCustomComboBox.WndProc(var Message: TMessage);
begin
   if (Message.Msg = WM_PARENTNOTIFY) then case LoWord(Message.wParam) of
      WM_CREATE:
         if FChildHandle = 0 then
            FChildHandle := Message.lParam
         else
            FListHandle := Message.lParam;
   end;
   inherited;
   if Message.Msg = WM_CTLCOLORLISTBOX then begin
      SetBkColor( Message.wParam, ColorToRGB(FListColor));
      Message.Result := CreateSolidBrush( ColorToRGB(FListColor));
   end;
end;

procedure TsCustomComboBox.ListWndProc(var Message: TMessage);
   procedure CallDefaultProc;
   begin
      with Message do
         Result := CallWindowProc(FDefListProc, FListHandle, Msg, WParam, LParam);
   end;
begin
   case Message.Msg of
      WM_NCPAINT: begin
         CallDefaultProc;
         PaintListFrame;
      end;
      else
         CallDefaultProc;
   end;
end;

procedure TsCustomComboBox.PaintListFrame;
var
   DC: HDC;
   R: TRect;
begin
   GetWindowRect(FListHandle, R);
   OffsetRect (R, -R.Left, -R.Top);
   DC := GetWindowDC(FListHandle);
   DrawEdge(DC, R, EDGE_RAISED, BF_RECT or BF_ADJUST);
   ReleaseDC(DC, FListHandle);
end;

procedure TsCustomComboBox.ComboWndProc(var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer);
begin
   inherited;
   if (ComboWnd = EditHandle) then case Message.Msg of
      WM_SETFOCUS, WM_KILLFOCUS:
         SetSolidBorder;
   end;
end;

procedure TsCustomComboBox.WMSetFocus(var Message: TMessage);
begin
   inherited;
   if not (csDesigning in ComponentState) then begin
      SetSolidBorder;
      if FFlat and not (Style in [csSimple, csDropDown]) then
         InvalidateSelection;
   end;
end;

procedure TsCustomComboBox.WMKillFocus(var Message: TMessage);
begin
   inherited;
   if not (csDesigning in ComponentState) then begin
      SetSolidBorder;
      if FFlat and not (Style in [csSimple, csDropDown]) then
         InvalidateSelection;
   end;
end;

procedure TsCustomComboBox.CMMouseEnter(var Message: TMessage);
begin
   inherited;
   if not (msMouseInControl in FEditState) and Enabled then begin
      Include(FEditState, msMouseInControl);
      SetSolidBorder;
   end;
end;

procedure TsCustomComboBox.CMMouseLeave(var Message: TMessage);
begin
   inherited;
   if (msMouseInControl in FEditState) and Enabled then begin
      Exclude(FEditState, msMouseInControl);
      SetSolidBorder;
   end;
end;

procedure TsCustomComboBox.CMEnabledChanged(var Msg: TMessage);
begin
   inherited;
   SetupEnabled;
end;

procedure TsCustomComboBox.CNCommand(var Message: TWMCommand);
var
   R: TRect;
begin
   inherited;
   if Message.NotifyCode in [1, 9, CBN_DROPDOWN, CBN_SELCHANGE] then begin
      if FFlat and not (Style in [csSimple, csDropDown]) then
         InvalidateSelection;
   end;
   if (Message.NotifyCode = CBN_CLOSEUP) and FFlat then begin
      R := GetButtonRect;
      Exclude(FEditState, msButtonPainted);
      InvalidateRect( Handle, @R, FALSE);
   end;
end;

procedure TsCustomComboBox.WMKeyDown(var Message: TMessage);
var
   S: String;
begin
   S := Text;
   inherited;
   if FFlat and not (Style in [csSimple, csDropDown]) and (Text <> S)then
      InvalidateSelection;
end;

procedure TsCustomComboBox.WMPaint(var Message: TWMPaint);
var
   R: TRect;
   DC: HDC;
   PS: TPaintStruct;
begin
   if not FFlat then begin
      inherited;
      Exit;
   end;
   if Style = csSimple then
      inherited
   else begin
      DC := BeginPaint(Handle, PS);
      try
         if Enabled or (FDisabledStyle <> dsEditTools) then begin
            R := PS.rcPaint;
            if R.Right > Width - FButtonWidth - 4 then
               R.Right := Width - FButtonWidth - 4;
            FillRect(DC, R, Brush.Handle);
            if RectInRect(GetButtonRect, PS.rcPaint) and not (msButtonPainted in FEditState) then
               PaintButton;
            ExcludeClipRect( DC, ClientWidth - FSysBtnWidth - 2, 0, ClientWidth, ClientHeight);
            PaintWindow(DC);
            if (Style = csDropDown) and DroppedDown then begin
               R := ClientRect;
               InflateRect(R, -2, -2);
               R.Right := Width - FButtonWidth - 3;
               Canvas.Brush.Color := clWindow;
               Canvas.FrameRect(R);
            end else if Style <> csDropDown then
               InvalidateSelection;
         end else
            PaintDisabled( PS);
      finally
         EndPaint(Handle, PS);
      end;
   end;
   RedrawBorders;
   Message.Result := 0;
end;

procedure TsCustomComboBox.WMNCPaint(var Message: TMessage);
begin
   inherited;
   if FFlat then
      RedrawBorders;
//      InternalRedrawBorder( Handle, SolidBorder);
end;

procedure TsCustomComboBox.CNDrawItem(var Message: TWMDrawItem);
var
   State: TOwnerDrawState;
begin
   with Message.DrawItemStruct^ do begin
      State := TOwnerDrawState(WordRec(LongRec(itemState).Lo).Lo);
      Canvas.Handle := hDC;
      Canvas.Font := Font;
      Canvas.Brush := Brush;
      if ItemState and ODS_COMBOBOXEDIT = 0 then
         Canvas.Brush.Color := FListColor;
      if odSelected in State then begin
         Canvas.Brush.Color := clHighlight;
         Canvas.Font.Color := clHighlightText
      end;
      if Integer(itemID) >= 0 then
         DrawItem(itemID, rcItem, State)
      else
         Canvas.FillRect(rcItem);
      if odFocused in State then
         DrawFocusRect(hDC, rcItem);
      Canvas.Handle := 0;
   end;
end;

procedure TsCustomComboBox.CreateParams(var Params: TCreateParams);
begin
   inherited CreateParams(Params);
   with Params do
      Style := Style or CBS_NOINTEGRALHEIGHT;
   ControlStyle := ControlStyle - [csFixedHeight];
end;

{$IFDEF V1_COMP}
procedure TsCustomComboBox.ReadDefaultDraw(Reader: TReader);
begin
   if Reader.ReadBoolean then
      DisabledStyle := dsDefault
   else
      DisabledStyle := dsOffice97;
end;

procedure TsCustomComboBox.WriteNothing(Writer: TWriter);
begin
end;

procedure TsCustomComboBox.DefineProperties(Filer: TFiler);
begin
   inherited;
   Filer.DefineProperty('DefaultDraw', ReadDefaultDraw, WriteNothing, FALSE);
end;
{$ENDIF}

procedure TsCustomComboBox.SetFlat(const Value: Boolean);
begin
   if Value <> FFlat then begin
      FFlat := Value;
      if FFlat then
         SetupEnabled;
      Invalidate;
   end;
end;

procedure TsCustomComboBox.SetDisabledStyle(Value: TsDisabledStyle);
begin
   if FDisabledStyle <> Value then begin
      FDisabledStyle := Value;
      if not Enabled then begin
         SetupEnabled;
         Refresh;
      end;
   end;
end;

procedure TsCustomComboBox.SetDisabledFont(Value: TFont);
begin
   FDisabledFont.Assign(Value);
   if not Enabled then
      Invalidate;
end;

procedure TsCustomComboBox.SetupEnabled;
begin
   if ([csLoading,csReading] * ComponentState = []) then begin
      if Flat then begin
         if not Enabled and (FdisabledStyle = dsOffice97) then begin
            FParentColor := inherited Parentcolor;
            FColor := inherited Color;
            inherited Parentcolor := TRUE;
         end else begin
            inherited ParentColor := FParentColor;
            inherited Color := FColor;
         end;
      end;
      if Enabled then
         Font.Assign(Canvas.Font)
      else begin
         Canvas.Font.Assign(Font);
         Font.Assign(FDisabledFont);
      end;
   end;
end;

procedure TsCustomComboBox.PaintDisabled(PS: TPaintStruct);
const
   alignments: array[TAlignment] of Word = (0, DT_RIGHT, DT_CENTER);
var
   R: TRect;
   H: THandle;
begin
   Canvas.Brush.Color := Color;
   Canvas.FillRect(ClientRect);
   H := SelectObject(Canvas.Handle, FDisabledFont.Handle);
   R := ClientRect;
   DrawText(Canvas.Handle, PChar(Text), Length(Text), R, DT_SINGLELINE or DT_VCENTER or DT_NOCLIP {or Alignments[Alignment]});
   SelectObject(Canvas.Handle, H);
end;

procedure TsCustomComboBox.InvalidateSelection;
var
   R: TRect;
begin
   R := ClientRect;
   InflateRect(R, -2, -3);
   R.Left := R.Right - FButtonWidth - 8;
   Dec(R.Right, FButtonWidth + 3);
   with Canvas do begin
      if Focused and not DroppedDown then
         Brush.Color := clHighlight
      else
         Brush.Color := Color;
      Brush.Style := bsSolid;
      FillRect(R);
      if Focused and not DroppedDown then begin
         R := ClientRect;
         InflateRect(R, -3, -3);
         Dec( R.Right, FButtonWidth + 2);
         FrameRect(R);
         Brush.Color := clWindow;
         DrawFocusRect(R);
      end;
      ExcludeClipRect( Handle, ClientWidth - FSysBtnWidth - 2, 0, ClientWidth, ClientHeight);
   end;
end;

function TsCustomComboBox.GetButtonRect: TRect;
begin
   GetWindowRect(Handle, Result);
   OffsetRect(Result, -Result.Left, -Result.Top);
   Inc(Result.Left, ClientWidth - FButtonWidth);
   InflateRect(Result, 0, -1);
   OffsetRect(Result, -2, 0);
end;

procedure TsCustomComboBox.PaintButton;
var
   R: TRect;
   state: Integer;
begin
   R := GetButtonRect;
   InflateRect(R, 1, 0);
   state := DFCS_SCROLLDOWN or DFCS_FLAT;
   if not Enabled then
      state := state or DFCS_INACTIVE;
   if DroppedDown then begin
      state := state or DFCS_PUSHED;
      Include(FEditState, msButtonPainted);
   end;
   DrawFrameControl(Canvas.Handle, R, DFC_SCROLL, state);
   PaintButtonBorder;
   ExcludeClipRect(Canvas.Handle, ClientWidth - FSysBtnWidth, 0, ClientWidth, ClientHeight);
end;

function TsCustomComboBox.GetSolidBorder: Boolean;
begin
   Result := ((csDesigning in ComponentState) and Enabled) or
      (not(csDesigning in ComponentState) and
      (DroppedDown or Focused or (GetFocus = EditHandle) or
      ((msMouseInControl in FEditState) and IsFlatApplied)));
end;

procedure TsCustomComboBox.SetSolidBorder;
var
   sb: Boolean;
begin
   sb := GetSolidBorder;
   if sb <> FSolidBorder then begin
      FSolidBorder := sb;
      RedrawBorders;
   end;
end;

procedure TsCustomComboBox.PaintButtonBorder;
const
   flags: array[Boolean] of Integer = ( EDGE_RAISED, EDGE_SUNKEN);
   Colors: array[Boolean] of TColor = (clBtnShadow, clWindow);
var
   R: TRect;
begin
   R := GetButtonRect;
   InflateRect(R, 0, -1);
   if FFlat then with Canvas do begin
      if SolidBorder then begin
         Brush.Color := Colors[DroppedDown];
         FrameRect(R);
         Pen.Color := Colors[not DroppedDown];
         Polyline([Point(R.Right-2, R.Top), Point(R.Left, R.Top),
               Point(R.Left, R.Bottom-1)]);
         Pen.Color := clWindow;
         PolyLine([Point(R.Left - 2, R.Top), Point(R.Left - 2, R.Bottom)]);
         Pen.Color := clBtnFace;
         Polyline([Point(R.Left-1, R.Top), Point(R.Left-1, R.Bottom)]);
      end else begin
         Brush.Color := clWindow;
         FrameRect(R);
         InflateRect(R, 0, -1);
         Pen.Color := Color;
         Polyline([Point(R.Left-2, R.Top), Point(R.Left-2, R.Bottom)]);
         Polyline([Point(R.Left-1, R.Top), Point(R.Left-1, R.Bottom)]);
      end;
   end;
end;

procedure TsCustomComboBox.RedrawBorders;
begin
   if not FFlat then
      Exit;
   InternalRedrawBorder( Handle, SolidBorder);
   if Style <> csSimple then
      PaintButtonBorder;
end;

{ TsColorComboBox }
const
   ColorsInList = 16;
   ColorValues: array[1..ColorsInList] of TColor = (
      clBlack, clMaroon, clGreen, clOlive, clNavy, clPurple, clTeal, clGray,
      clSilver, clRed, clLime, clYellow, clBlue, clFuchsia, clAqua, clWhite);
   SysColorsInList = 25;
   SysColorValues: array[1..SysColorsInList] of TColor = (
      clScrollBar, clBackground, clActiveCaption, clInactiveCaption, clMenu,
      clWindow, clWindowFrame, clMenuText, clWindowText, clCaptionText,
      clActiveBorder, clInactiveBorder, clAppWorkSpace, clHighlight,
      clHighlightText, clBtnFace, clBtnShadow, clGrayText, clBtnText,
      clInactiveCaptionText, clBtnHighlight, cl3DDkShadow, cl3DLight,
      clInfoText, clInfoBk);

constructor TsColorComboBox.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   Style := csOwnerDrawFixed;
   FColorValue := clBlack; { make default color selected }
   FDisplayNames := True;
end;

procedure TsColorComboBox.BuildList;
var
   ii: Integer;
   ColorName: string;
begin
   Clear;
   for ii := 1 to ColorsInList do begin
      ColorName := Copy(ColorToString(ColorValues[ii]), 3, 40);
      Items.AddObject(ColorName, TObject(ColorValues[ii]));
   end;
   if FFullScheme then for ii := 1 to SysColorsInList do begin
      ColorName := Copy(ColorToString(SysColorValues[ii]), 3, 40);
      Items.AddObject(ColorName, TObject(SysColorValues[ii]));
   end;
end;

procedure TsColorComboBox.SetDisplayNames(Value: Boolean);
begin
   if DisplayNames <> Value then begin
      FDisplayNames := Value;
      SetColorValue(FColorValue);
   end;
end;

procedure TsColorComboBox.SetColorValue(NewValue: TColor);
var
   Item: Integer;
   CurrentColor: TColor;
begin
   if not FFullScheme then
      NewValue := TColor(ColorToRGB(NewValue));
   if (ItemIndex < 0) or (NewValue <> FColorValue) then
      for Item := 0 to Pred(Items.Count) do begin
         CurrentColor := TColor(Items.Objects[Item]);
         if CurrentColor = NewValue then begin
            FColorValue := NewValue;
            if ItemIndex <> Item then
               ItemIndex := Item;
            Change;
            Break;
         end;
      end;
end;

procedure TsColorComboBox.SetFullScheme(Value: Boolean);
begin
   if FFullScheme <> Value then begin
      FFullScheme := Value;
      BuildList;
      SetColorValue(FColorValue);
   end;
end;

procedure TsColorComboBox.CreateWnd;
begin
   inherited CreateWnd;
   BuildList;
   SetColorValue(FColorValue);
end;

procedure TsColorComboBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
const
   ColorWidth = 22;
var
   ARect: TRect;
   Text: array[0..255] of Char;
   Safer: TColor;
begin
   ARect := Rect;
   Inc(ARect.Top, 2);
   Inc(ARect.Left, 2);
   Dec(ARect.Bottom, 2);
   if FDisplayNames then
      ARect.Right := ARect.Left + ColorWidth
   else
      Dec(ARect.Right, 3);
   with Canvas do begin
      FillRect(Rect);
      Safer := Brush.Color;
      Pen.Color := clBlack;
      Rectangle(ARect.Left, ARect.Top, ARect.Right, ARect.Bottom);
      Brush.Color := TColor(Items.Objects[Index]);
      try
         InflateRect(ARect, -1, -1);
         FillRect(ARect);
      finally
         Brush.Color := Safer;
      end;
      if FDisplayNames then begin
         StrPCopy(Text, Items[Index]);
         Rect.Left := Rect.Left + ColorWidth + 6;
         DrawText(Canvas.Handle, Text, StrLen(Text), Rect, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
      end;
   end;
end;

procedure TsColorComboBox.Click;
begin
   if ItemIndex >= 0 then
      ColorValue := TColor(Items.Objects[ItemIndex]);
   inherited Click;
end;

procedure TsColorComboBox.CMFontChanged(var Message: TMessage);
begin
   inherited;
   ResetItemHeight;
   RecreateWnd;
end;

procedure TsColorComboBox.ResetItemHeight;
begin
   ItemHeight := Max(GetFontHeight(Font), 9);
end;

procedure TsColorComboBox.Change;
begin
   if Assigned(FOnChange) then
      FOnChange(Self);
end;

{ TsFontComboBox }
function CreateBitmap(ResName: PChar): TBitmap;
begin
   Result := TBitmap.Create;
   Result.Handle := LoadBitmap(HInstance, ResName);
   if Result.Handle = 0 then begin
      Result.Free;
      Result := nil;
   end;
   if Result = nil then
      Raise Exception.Create( 'ResourceNotFound');
end;

function EnumFontsProc(var LogFont: TLogFont; var TextMetric: TTextMetric;
   FontType: Integer; Data: Pointer): Integer; export; stdcall;
var
   Box: TsFontComboBox;
   function IsValidFont: Boolean;
   begin
      Result := True;
      if foAnsiOnly in Box.Options then
         Result := Result and (LogFont.lfCharSet = ANSI_CHARSET);
      if foTrueTypeOnly in Box.Options then
         Result := Result and (FontType and TRUETYPE_FONTTYPE = TRUETYPE_FONTTYPE);
      if foFixedPitchOnly in Box.Options then
         Result := Result and (LogFont.lfPitchAndFamily and FIXED_PITCH = FIXED_PITCH);
      if foOEMFontsOnly in Box.Options then
         Result := Result and (LogFont.lfCharSet = OEM_CHARSET);
      if foNoOEMFonts in Box.Options then
         Result := Result and (LogFont.lfCharSet <> OEM_CHARSET);
      if foScalableOnly in Box.Options then
         Result := Result and (FontType and RASTER_FONTTYPE = 0);
   end;
begin
   Box := TsFontComboBox(Data);
   if (Box.Items.IndexOf(StrPas(LogFont.lfFaceName)) < 0) and IsValidFont then
      Box.Items.AddObject(StrPas(LogFont.lfFaceName), TObject(FontType));
   Result := 1;
end;

constructor TsFontComboBox.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   TrueTypeBMP := CreateBitmap('TTF_FONT');
   DeviceBMP := CreateBitmap('PRN_FONT');
   FDevice := fdScreen;
   Style := csOwnerDrawFixed;
   Sorted := True;
   FUpdate := False;
   ResetItemHeight;
end;

destructor TsFontComboBox.Destroy;
begin
   TrueTypeBMP.Free;
   DeviceBMP.Free;
   inherited Destroy;
end;

procedure TsFontComboBox.BuildList;
var
   DC: HDC;
   Proc: TFarProc;
begin
   if not HandleAllocated then
      Exit;
   Clear;
   DC := GetDC(0);
   try
      Proc := MakeProcInstance(@EnumFontsProc, HInstance);
      try
         if (FDevice = fdScreen) or (FDevice = fdBoth) then
            EnumFonts(DC, nil, Proc, Pointer(Self));
         if (FDevice = fdPrinter) or (FDevice = fdBoth) then try
            EnumFonts(Printer.Handle, nil, Proc, Pointer(Self));
         except
         end;
      finally
         FreeProcInstance(Proc);
      end;
   finally
      ReleaseDC(0, DC);
   end;
end;

procedure TsFontComboBox.SetFontName(const NewFontName: string);
var
   I, Item: Integer;
begin
   if FontName <> NewFontName then begin
      HandleNeeded;
      I := -1;
      for Item := 0 to Items.Count - 1 do
         if (AnsiUpperCase(Items[Item]) = AnsiUpperCase(NewFontName)) then begin
            I := Item;
            Break;
         end;
      ItemIndex := I;
      Change;
   end;
end;

function TsFontComboBox.GetFontName: string;
begin
   Result := Text;
{
   result := '';
   if ItemIndex > 0 then
      Result := AnsiUpperCase(Items[ItemIndex];
}
end;

function TsFontComboBox.GetTrueTypeOnly: Boolean;
begin
   Result := foTrueTypeOnly in FOptions;
end;

procedure TsFontComboBox.SetOptions(Value: TFontListOptions);
begin
   if Value <> FOptions then begin
      FOptions := Value;
      Reset;
   end;
end;

procedure TsFontComboBox.SetTrueTypeOnly(Value: Boolean);
begin
   if Value <> TrueTypeOnly then begin
      if Value then
         FOptions := FOptions + [foTrueTypeOnly]
      else
         FOptions := FOptions - [foTrueTypeOnly];
      Reset;
   end;
end;

procedure TsFontComboBox.SetPreviewFont(Value: Boolean);
begin
   if Value <> FPreviewFont then begin
      FPreviewFont := Value;
      Invalidate;
   end;
end;

procedure TsFontComboBox.SetDevice(Value: TFontDevice);
begin
   if Value <> FDevice then begin
      FDevice := Value;
      Reset;
   end;
end;

procedure TsFontComboBox.CreateWnd;
var
   OldFont: string;
begin
   OldFont := FontName;
   inherited CreateWnd;
   BuildList;
   SetFontName(OldFont);
end;

procedure TsFontComboBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
   Bitmap: TBitmap;
   BmpWidth: Integer;
   Text: array[0..255] of Char;
begin
   with Canvas do begin
      FillRect(Rect);
      BmpWidth := 20;
      Bitmap := nil;
      if (Integer(Items.Objects[Index]) and TRUETYPE_FONTTYPE) <> 0 then
         Bitmap := TrueTypeBMP
      else if (Integer(Items.Objects[Index]) and DEVICE_FONTTYPE) <> 0 then
         Bitmap := DeviceBMP;
      if Bitmap <> nil then begin
         BmpWidth := Bitmap.Width;
         BrushCopy(Bounds(Rect.Left + 2, (Rect.Top + Rect.Bottom - Bitmap.Height)
            div 2, Bitmap.Width, Bitmap.Height), Bitmap, Bounds(0, 0, Bitmap.Width,
            Bitmap.Height), Bitmap.TransparentColor);
      end;
      { uses DrawText instead of TextOut in order to get clipping against the combo box button }
      {TextOut(Rect.Left + bmpWidth + 6, Rect.Top, Items[Index])}
      StrPCopy(Text, Items[Index]);
      if FPreviewFont then
         Canvas.Font.Name := Text;
      Rect.Left := Rect.Left + BmpWidth + 6;
      DrawText(Canvas.Handle, Text, StrLen(Text), Rect, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
   end;
end;

procedure TsFontComboBox.WMFontChange(var Message: TMessage);
begin
   inherited;
   Reset;
end;

procedure TsFontComboBox.CMFontChanged(var Message: TMessage);
begin
   inherited;
   ResetItemHeight;
   RecreateWnd;
end;

procedure TsFontComboBox.ResetItemHeight;
begin
   ItemHeight := MaxInteger([GetFontHeight(Font), TrueTypeBMP.Height - 1, 9]);
end;

procedure TsFontComboBox.Click;
begin
   inherited Click;
   Change;
end;

procedure TsFontComboBox.Change;
begin
   if not FUpdate and Assigned(FOnChange) then
      FOnChange(Self);
end;

procedure TsFontComboBox.Reset;
var
   SaveName: string;
begin
   if HandleAllocated then begin
      Inc(FUpdate);
      try
         SaveName := FontName;
         BuildList;
         FontName := SaveName;
      finally
         Dec(FUpdate);
         if FontName <> SaveName then
            Change;
      end;
   end;
end;

const
   defaultMask = '*.*';

constructor TsPathComboBox.Create(AOwner: TComponent);
begin
   inherited;
   FFileMask := defaultMask;
   FOptions := [flReadWrite];
   Style := csOwnerDrawFixed;
end;

destructor TsPathComboBox.Destroy;
begin
   inherited;
end;

procedure TsPathComboBox.CreateWnd;
begin
   inherited;
   FillList;
end;

procedure TsPathComboBox.FillList;
var
   S: String;
   Ptr: array[0..124] of Char;
   flag: Integer;
   ii, w: Integer;
   ShellInfo: TSHFILEINFO;
begin
   Items.Clear;
   S := TerminateDir(LDN2Alternate(UnTerminateDir(FPath))) + FFileMask;
   StrPCopy(Ptr, S);
   flag := 0;
   if flArchive in FOptions then
      flag := DDL_ARCHIVE;
   if flDirectory in Options then
      flag := flag or DDL_DIRECTORY;
   if flDrives in Options then
      flag := flag or DDL_DRIVES;
   if flExclusive in Options then
      flag := flag or DDL_EXCLUSIVE;
   if flHidden in Options then
      flag := flag or DDL_HIDDEN;
   if flReadonly in Options then
      flag := flag or DDL_READONLY;
   if flReadWrite in Options then
      flag := flag or DDL_READWRITE;
   if flSystem in Options then
      flag := flag or DDL_SYSTEM;
   Perform( CB_DIR, flag, Integer(@Ptr));
   FIconHeight := 0;
   FImageList := 0;
   for ii := items.Count - 1 downto 0 do begin
      if Items[ii][1] <> '[' then
         StrPCopy(Ptr, TerminateDir(FPath) + Items[ii])
      else begin
         if Items[ii][2] = '-' then
            StrPCopy(Ptr, Items[ii][3] + ':\')
         else
            StrPCopy(Ptr,
               TerminateDir(FPath) + Copy(Items[ii], 2, Length(Items[ii]) - 2));
      end;
      w := SHGetFileInfo( Ptr, 0, ShellInfo, SizeOf(ShellInfo),
         SHGFI_DISPLAYNAME or SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
      if FImageList = 0 then begin
         FImageList := w;
         ImageList_GetIconSize( FImageList, w, FIconHeight);
      end;
      if ShellInfo.szDisplayName = '' then
         Items.Delete(ii)
      else begin
         if Items[ii][1] <> '[' then
            Items[ii] := ExtractFileName(ShellInfo.szDisplayName)
         else if Items[ii][2] <> '-' then
            Items[ii] := '[' + String(ShellInfo.szDisplayName) + ']';
         Items.Objects[ii] := TObject(ShellInfo.iIcon);
      end;
   end;
end;

procedure TsPathComboBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
   Text: array[0..255] of Char;
begin
   with Canvas do begin
      FillRect(Rect);
      ImageList_Draw(FImageList, Integer(Items.Objects[Index]), Canvas.Handle,
         Rect.Left + 2, (Rect.Top + Rect.Bottom - FIconHeight) div 2,
         ILD_TRANSPARENT);
      StrPCopy(Text, Items[Index]);
      Rect.Left := Rect.Left + 26;
      DrawText(Canvas.Handle, Text, StrLen(Text), Rect, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
   end;
end;

procedure TsPathComboBox.BeginUpdate;
begin
   Inc(FLocked);
end;

procedure TsPathComboBox.EndUpdate;
begin
   Dec(FLocked);
end;

function TsPathComboBox.GetDisplayIcons: Boolean;
begin
   Result := Style = csOwnerDrawFixed;
end;

procedure TsPathComboBox.SetDisplayIcons(Value: Boolean);
const
   styles: array[Boolean] of TComboBoxStyle = ( csDropDownList, csOwnerDrawFixed);
begin
   if Value <> DisplayIcons then
      Style := styles[Value];
end;

procedure TsPathComboBox.SetFileMask(Value: String);
begin
   if FFileMask <> Value then begin
      FFileMask := Value;
      if not FLocked then
         FillList;
   end;
end;

procedure TsPathComboBox.SetOptions(Value: TsFilesListOptions);
begin
   if FOptions <> Value then begin
      FOptions := Value;
      if not FLocked then
         FillList;
   end;
end;

procedure TsPathComboBox.SetPath(Value: String);
begin
   if Value <> FPath then begin
      FPath := Value;
      if not FLocked then
         FillList;
   end;
end;

function TsPathComboBox.FileMaskStored: Boolean;
begin
   Result := FFileMask <> defaultMask;
end;

initialization
   RegisterFlatControl(TsCustomComboBox);

end.

