unit ResStore;

{ ******************************************************************************
  The TRessourceStore has been designed by Alexandre GUILLIEN

  It allows you to add any external data to your projects.

  To add Datas, drop the component on your form, edit the Items property, add
  as many items as you want and for each Item, edit the Name property by clicking
  on the '...' button. This allow you to select the file where are stored your
  custom datas. Once loaded in the project, they remain as loaded the first time.
  If you update the file, reload it.

  You can edit the name of each item freely, however, each name must be unique.

  To use your stored datas, use the GetResByName to get the TPackedItem and with
  the TPackedItem, use as you need SaveToStream, SaveToFile or GetAsStream methods.
  The LoadFromStream and LoadFromFile methods allow you to modify the datas during
  execution, but any change will be lost when the App is launched again.

  Note : this component allow you to easily build a Sfx.

  ************
  Please note that the ZLib unit is used by default (provided with D3 extras). If
  you don't have it, just remove the line "$DEFINE USEZIP" which is located line 36.
  ************

  This is Freeware. You can use it anywhere, but I'd be happy to see my name in the
  apps where you used it.

  E-Mail = AGuillien@csi.com
  Mail : 12, rue Rhonat      69100 Villeurbanne      FRANCE
****************************************************************************** }

{$DEFINE USEZIP}

interface

uses
  SysUtils, Classes;

type
  ERessourceStoreError = class(Exception);
  TPackedRessources = class;
  TPackedItem = class;

  TRessourceStore = class(TComponent)
  private
    FRessources: TPackedRessources;
    procedure SetRessources(Ressources: TPackedRessources);
    function GetSize: Integer;
    procedure SetSize(NewSize: Integer);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    {}
    function ItemIndex(ItemName: string): Integer;
    function GetResByName(ResName: string): TPackedItem;
  published
    property Items: TPackedRessources read FRessources write SetRessources;
    property Size: Integer read GetSize write SetSize;
  end;

  TPackedRessources = class(TCollection)
  private
    FOwner: TPersistent;
    function GetOwner: TPersistent; override;
  protected
    property Owner: TPersistent read FOwner write FOwner;
  public
    constructor Create;
  end;

  TStoreMode = (smStore{$IFDEF USEZIP}, smZipLow, smZipMed, smZipMax{$ENDIF});

  TPackedItem = class(TCollectionItem)
  private
    FData: TMemoryStream;
    FStoreMode: TStoreMode;
    FName: string;
    procedure SetStoreMode(StoreMode: TStoreMode);
    procedure ReadData(Stream: TStream);
    procedure WriteData(Stream: TStream);
    function GetSize: Integer;
    procedure SetSize(NewSize: Integer);
    procedure SetName(Name: string);
  protected
    function GetDisplayName: string; override;
    procedure DefineProperties(Filer: TFiler); override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    {}
    procedure SaveToStream(Stream: TStream);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(FileName: string);
    procedure LoadFromFile(FileName: string);
    {}
    function GetAsStream: TStream;
  published
    property StoreMode: TStoreMode read FStoreMode write SetStoreMode {$IFDEF USEZIP}default smZipMax{$ENDIF};
    property Size: Integer read GetSize write SetSize;
    property Name: string read FName write SetName;
  end;

procedure register;

implementation

uses {$IFDEF USEZIP}ZLib, {$ENDIF}Dialogs, DsgnIntf;

{$R *.DCR}

{ ****************************** TRessourceStore ****************************** }

constructor TRessourceStore.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FRessources:= TPackedRessources.Create;
  FRessources.Owner:= Self;
end;

destructor TRessourceStore.Destroy;
begin
  FRessources.Free;
  inherited Destroy;
end;

function TRessourceStore.ItemIndex(ItemName: string): Integer;
var i: Integer;
begin
  Result:= -1; i:= 0;
  while (i < FRessources.Count) and (CompareText(ItemName, TPackedItem(FRessources.Items[i]).Name) <> 0) do
    Inc(i);
  if i < FRessources.Count then
    Result:= i;
end;

function TRessourceStore.GetResByName(ResName: string): TPackedItem;
var Idx: Integer;
begin
  Idx:= ItemIndex(ResName);
  if Idx <> -1 then
    Result:= TPackedItem(FRessources.Items[Idx])
  else raise ERessourceStoreError.Create('La ressource "' + ResName + '" n''existe pas !');
end;

function TRessourceStore.GetSize: Integer;
var i: Integer;
begin
  Result:= 0;
  for i:= 0 to FRessources.Count - 1 do
    Result:= Result + TPackedItem(FRessources.Items[i]).Size;
end;

procedure TRessourceStore.SetSize(NewSize: Integer);
begin end;

procedure TRessourceStore.SetRessources(Ressources: TPackedRessources);
begin
  FRessources.Assign(Ressources);
end;

{ ******************************** TPackedItem ******************************** }

constructor TPackedItem.Create(Collection: TCollection);
begin
  {$IFDEF USEZIP}FStoreMode:= smZipMax;{$ENDIF}
  inherited Create(Collection);
  FData:= TMemoryStream.Create;
end;

destructor TPackedItem.Destroy;
begin
  FData.Free; FData:= nil;
  inherited Destroy;
end;

{$IFDEF USEZIP}
procedure TPackedItem.SaveToStream(Stream: TStream);
var S: TDecompressionStream;
    L: Integer;
begin
  FData.Position:= 0;
  if StoreMode <> smStore then
  begin
    S:= TDecompressionStream.Create(FData);
    try
      S.Read(L, SizeOf(Integer));
      Stream.CopyFrom(S, L);
    finally
      S.Free;
    end;
  end else
    Stream.CopyFrom(FData, FData.Size);
end;

procedure TPackedItem.LoadFromStream(Stream: TStream);
var S: TCompressionStream;
    L: TCompressionLevel;
    Tmp: Integer;
begin
  case StoreMode of
    smZipLow: L:= clFastest;
    smZipMed: L:= clDefault;
    smZipMax: L:= clMax
    else L:= clNone;
  end;
  Stream.Position:= 0;
  FData.Clear;
  if L <> clNone then
  begin
    S:= TCompressionStream.Create(L, FData);
    try
      Tmp:= Stream.Size;
      S.Write(Tmp, SizeOf(Integer));
      S.CopyFrom(Stream, Tmp);
    finally
      S.Free;
    end;
  end else
    FData.CopyFrom(Stream, 0);
end;
{$ELSE}
procedure TPackedItem.SaveToStream(Stream: TStream);
begin
  FData.Position:= 0;
  Stream.CopyFrom(FData, 0);
end;

procedure TPackedItem.LoadFromStream(Stream: TStream);
begin
  FData.CopyFrom(Stream, 0);
end;
{$ENDIF}

procedure TPackedItem.SaveToFile(FileName: string);
var F: TFileStream;
begin
  F:= TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(F);
  finally
    F.Free;
  end;
end;

procedure TPackedItem.LoadFromFile(FileName: string);
var F: TFileStream;
begin
  F:= TFileStream.Create(FileName, fmOpenRead or fmShareDenyNone);
  try
    LoadFromStream(F);
  finally
    F.Free;
  end;
end;

function TPackedItem.GetAsStream: TStream;
begin
  SetStoreMode(smStore);
  FData.Position:= 0;
  Result:= FData;
end;

procedure TPackedItem.SetStoreMode(StoreMode: TStoreMode);
{$IFDEF USEZIP}
var M: TMemoryStream;
begin
  if csLoading in TRessourceStore(TPackedRessources(GetOwner).GetOwner).ComponentState then
    FStoreMode:= StoreMode
  else if StoreMode <> FStoreMode then
  begin
    M:= TMemoryStream.Create;
    try
      SaveToStream(M);
      FStoreMode:= StoreMode;
      LoadFromStream(M);
    finally
      M.Free;
    end;
  end;
end;
{$ELSE}
begin
  FStoreMode:= smStore;
end;
{$ENDIF}

function TPackedItem.GetDisplayName: string;
begin
  if FData.Size = 0 then
    Result:= ClassName
  else
    Result:= FName;
end;

function TPackedItem.GetSize: Integer;
begin
  Result:= FData.Size;
end;

procedure TPackedItem.SetSize(NewSize: Integer);
begin end;

procedure TPackedItem.SetName(Name: string);
var i: Integer;
begin
  for i:= 0 to Collection.Count - 1 do
  begin
    if (Collection.Items[i].ID <> ID) and (CompareText(Name, TPackedItem(Collection.Items[i]).Name) = 0) then
      raise ERessourceStoreError.Create('Le nom "' + Name + '" est dj utilis !');
  end;
  FName:= Name;
end;

procedure TPackedItem.ReadData(Stream: TStream);
var L: Integer;
begin
  FData.Clear;
  Stream.Read(L, SizeOf(Integer));
  FData.Size:= L; { Allocates what's needed }
  FData.CopyFrom(Stream, L);
end;

procedure TPackedItem.WriteData(Stream: TStream);
var L: Integer;
begin
  FData.Position:= 0;
  L:= FData.Size;
  Stream.Write(L, SizeOf(Integer));
  Stream.CopyFrom(FData, FData.Size);
end;

procedure TPackedItem.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('ResData', ReadData, WriteData, Assigned(FData));
end;

{ ***************************** TPackedRessources ***************************** }

function TPackedRessources.GetOwner: TPersistent;
begin
  Result:= FOwner;
end;

constructor TPackedRessources.Create;
begin
  FOwner:= nil;
  inherited Create(TPackedItem);
end;

{ ******************************** Prop Editors ******************************** }

type
  TROIntProperty = class(TIntegerProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
  end;

function TROIntProperty.GetAttributes: TPropertyAttributes;
begin
  Result:= [paReadOnly];
end;

type
  TPackedItemName = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

function TPackedItemName.GetAttributes: TPropertyAttributes;
begin
  Result:= inherited GetAttributes + [paDialog];
end;

procedure TPackedItemName.Edit;
var Dialog: TOpenDialog;
    F: TFileStream;
  procedure SetUniqueName(Name: string);
  var i: Integer;
      Base: string;
  begin
    try
      SetStrValue(Name);
    except
      Base:= Name; i:= 0;
      while Name <> GetStrValue do
      begin
        Inc(i);
        Name:= Base + IntToStr(i);
        try
          SetStrValue(Name);
        except end;
      end;
    end;
  end;
begin
  Dialog:= TOpenDialog.Create(nil);
  try
    if Dialog.Execute then
    begin
      F:= TFileStream.Create(Dialog.FileName, fmOpenRead or fmShareDenyNone);
      try
        SetUniqueName(ExtractFileName(Dialog.FileName));
        TPackedItem(GetComponent(0)).LoadFromStream(F);
        Modified;
      finally
        F.Free;
      end;
    end;
  finally
    Dialog.Free;
  end;
end;

procedure register;
begin
  RegisterComponents('ExtComps', [TRessourceStore]);
  RegisterPropertyEditor(TypeInfo(Integer), TRessourceStore, 'Size', TROIntProperty);
  RegisterPropertyEditor(TypeInfo(Integer), TPackedItem, 'Size', TROIntProperty);
  RegisterPropertyEditor(TypeInfo(string), TPackedItem, 'Name', TPackedItemName); 
end;

end.
