{$I DFS.INC}  { Standard defines for all Delphi Free Stuff components }

{-----------------------------------------------------------------------------}
{ TPageSetupDialog v2.04                                                      }
{-----------------------------------------------------------------------------}
{ A component to wrap the Win95 PageSetupDlg common dialog API function.      }
{ Borland seems to have forgotten this new common dialog in Delphi 2.0.       }
{ Copyright 1998, Brad Stowers.  All Rights Reserved.                         }
{ This component can be freely used and distributed in commercial and private }
{ environments, provided this notice is not modified in any way and there is  }
{ no charge for it other than nominal handling fees.  Contact me directly for }
{ modifications to this agreement.                                            }
{-----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at bstowers@pobox.com.                                                      }
{ The lateset version will always be available on the web at:                 }
{   http://www.pobox.com/~bstowers/delphi/                                    }
{ See PgSetup.txt for notes, known issues, and revision history.              }
{-----------------------------------------------------------------------------}
{ Date last modified:  September 21, 1998                                     }
{-----------------------------------------------------------------------------}


// Make sure we have RTTI available for the TPSRect class below.
{$M+}

unit PgSetup;

interface

{$IFNDEF DFS_WIN32}
  ERROR!  This unit only available for Delphi 2.0 or later!!!
{$ENDIF}

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
{$IFDEF DFS_DEBUG}
  debugmsg,mmsystem,
{$ENDIF}
  CommDlg, DFSAbout;


const
  { This shuts up C++Builder 3 about the redefiniton being different. There
    seems to be no equivalent in C1.  Sorry. }
  {$IFDEF DFS_CPPB_3_UP}
  {$EXTERNALSYM DFS_COMPONENT_VERSION}
  {$ENDIF}
  DFS_COMPONENT_VERSION = 'TPageSetupDialog v2.04';

type
  TPageSetupOption = (
       poDefaultMinMargins, poDisableMargins, poDisableOrientation,
       poDisablePagePainting, poDisablePaper, poDisablePrinter, poNoWarning,
       poShowHelp
     );
  TPageSetupOptions = set of TPageSetupOption;
  TPSPaperType = (ptPaper, ptEnvelope);
  TPSPaperOrientation = (poPortrait, poLandscape);
  TPSPrinterType = (ptDotMatrix, ptHPPCL);
  TPSPaintWhat = (pwFullPage, pwMinimumMargins, pwMargins,
                  pwGreekText, pwEnvStamp, pwYAFullPage);

  TPSMeasureVal = Double;
  TPSMeasurements = (pmDefault, pmMillimeters, pmInches);
  TPSPrinterEvent = procedure(Sender: TObject; Wnd: HWND) of object;

  (* PPSDlgData is simply redeclared as PPageSetupDlg (COMMDLG.PAS) to prevent
     compile errors in units that have this event.  They won't compile unless
     you add CommDlg to their units.  This circumvents the problem.           *)
  PPSDlgData = ^TPSDlgData;
  TPSDlgData = TPageSetupDlg;
  { PaperSize: See DEVMODE help topic, dmPaperSize member. DMPAPER_* constants.}
  TPSInitPaintPageEvent = function(Sender: TObject; PaperSize: short;
     PaperType: TPSPaperType; PaperOrientation: TPSPaperOrientation;
     PrinterType: TPSPrinterType; pSetupData: PPSDlgData): boolean of object;
  TPSPaintPageEvent = function(Sender: TObject; PaintWhat: TPSPaintWhat;
     Canvas: TCanvas; Rect: TRect): boolean of object;

  (* TPSRect is used for published properties that would normally be of TRect
     type.  Can't publish properties that are record types, so this is used.  *)
  TPSRect = class(TPersistent)
  private
    FRect: TRect;
  public
    function Compare(Other: TPSRect): boolean;

    property Rect: TRect
       read FRect
       write FRect;
  published
    property Left: integer
       read FRect.Left
       write FRect.Left;
    property Right: integer
       read FRect.Right
       write FRect.Right;
    property Top: integer
       read FRect.Top
       write FRect.Top;
    property Bottom: integer
       read FRect.Bottom
       write FRect.Bottom;
  end;

  (* TPSPoint is needed for the same reason as TPSRect above.                 *)
  TPSPoint = class(TPersistent)
  private
    FPoint: TPoint;
  protected
    function GetX: longint;
    procedure SetX(Val: longint);
    function GetY: longint;
    procedure SetY(Val: longint);
  public
    function Compare(Other: TPSPoint): boolean;

    property Point: TPoint
       read FPoint
       write FPoint;
  published
    property X: longint
       read GetX
       write SetX;
    property Y: longint
       read GetY
       write SetY;
  end;


  TPageSetupDialog = class(TCommonDialog)
  private
    FGettingDefaults: boolean;
    FCentered: boolean;
    FOptions: TPageSetupOptions;
    FCustomData: LPARAM;
    FPaperSize: TPSPoint;
    FMinimumMargins: TPSRect;
    FMargins: TPSRect;
    FMeasurements: TPSMeasurements;
    FOnPrinter: TPSPrinterEvent;
    FOnInitPaintPage: TPSInitPaintPageEvent;
    FOnPaintPage: TPSPaintPageEvent;

    function DoPrinter(Wnd: HWND): boolean;
    function DoExecute(Func: pointer): boolean;
  protected
    procedure SetName(const NewName: TComponentName); override;
    function Printer(Wnd: HWND): boolean; virtual;

    procedure SetPaperSize(const Val: TPSPoint);
    function StorePaperSize: boolean;
    procedure SetMinimumMargins(const Val: TPSRect);
    function StoreMinimumMargins: boolean;
    procedure SetMargins(const Val: TPSRect);
    function StoreMargins: boolean;
    procedure SetMeasurements(Val: TPSMeasurements);
    function GetDefaultMeasurements: TPSMeasurements;
    function GetCurrentMeasurements: TPSMeasurements;
    function GetVersion: TDFSVersion;
    procedure SetVersion(const Val: TDFSVersion);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { Delphi and C++Builder 3 finally got it right! }
    function Execute: boolean;
       {$IFDEF DFS_COMPILER_3_UP} override; {$ELSE} virtual; {$ENDIF}
    function ReadCurrentValues: boolean; virtual;
    function FromMeasurementVal(Val: integer): TPSMeasureVal;
    function ToMeasurementVal(Val: TPSMeasureVal): integer;

    { How does the user's system like to measure things? }
    property DefaultMeasurements: TPSMeasurements
       read GetDefaultMeasurements;
    { What are we using currently, i.e. translate pmDefault value }
    property CurrentMeasurements: TPSMeasurements
       read GetCurrentMeasurements;

    { It is the user's responsibility to clean up this pointer if necessary. }
    property CustomData: LPARAM
       read FCustomData
       write FCustomData;
  published
    property Version: TDFSVersion
       read GetVersion
       write SetVersion;
    // Measurements property has to be declared before PaperSize, MinimumMargins
    // and Margins because of streaming quirks.
    property Measurements: TPSMeasurements
       read FMeasurements
       write SetMeasurements
       nodefault;

    property PaperSize: TPSPoint
       read FPaperSize
       write SetPaperSize
       stored StorePaperSize;
    property MinimumMargins: TPSRect
       read FMinimumMargins
       write SetMinimumMargins
       stored StoreMinimumMargins;
    property Margins: TPSRect
       read FMargins
       write SetMargins
       stored StoreMargins;

    property Centered: boolean
       read FCentered
       write FCentered
       default TRUE;
    property Options: TPageSetupOptions
       read FOptions
       write FOptions
       default [poDefaultMinMargins, poShowHelp];

    { Events }
    property OnPrinter: TPSPrinterEvent
       read FOnPrinter
       write FOnPrinter;
    property OnInitPaintPage: TPSInitPaintPageEvent
       read FOnInitPaintPage
       write FOnInitPaintPage;
    property OnPaintPage: TPSPaintPageEvent
       read FOnPaintPage
       write FOnPaintPage;
  end;

implementation

uses
{$IFDEF DFS_COMPILER_3_UP}
  Dlgs,
{$ENDIF}
  Printers;

const
  IDPRINTERBTN = {$IFDEF DFS_COMPILER_3_UP} Dlgs.psh3 {$ELSE} $0402 {$ENDIF};

{ Private globals }
var
  HelpMsg: Integer;
  DefPaperSizeI: TPSPoint;
  DefMinimumMarginsI: TPSRect;
  DefMarginsI: TPSRect;
  DefPaperSizeM: TPSPoint;
  DefMinimumMarginsM: TPSRect;
  DefMarginsM: TPSRect;
  HookCtl3D: boolean;
  PageSetupDialog: TPageSetupDialog;


{ Center the given window on the screen }
procedure CenterWindow(Wnd: HWnd);
var
  Rect: TRect;
begin
  GetWindowRect(Wnd, Rect);
  SetWindowPos(Wnd, 0,
     (GetSystemMetrics(SM_CXSCREEN) - Rect.Right + Rect.Left) div 2,
     (GetSystemMetrics(SM_CYSCREEN) - Rect.Bottom + Rect.Top) div 3,
     0, 0, SWP_NOACTIVATE or SWP_NOSIZE or SWP_NOZORDER);
end;

{ Generic dialog hook. Centers the dialog on the screen in response to
  the WM_INITDIALOG message }
function DialogHook(Wnd: HWnd; Msg: UINT; WParam: WPARAM; LParam: LPARAM): UINT; stdcall;
begin
  Result := 0;
  case Msg of
    WM_INITDIALOG:
      begin
        if HookCtl3D then
        begin
          Subclass3DDlg(Wnd, CTL3D_ALL);
          SetAutoSubClass(True);
        end;
        if PageSetupDialog.Centered then
          CenterWindow(Wnd);
        Result := 1;
      end;
    WM_DESTROY:
      if HookCtl3D then
        SetAutoSubClass(False);
  end;
end;

function PageSetupDialogHook(Wnd: HWnd; Msg: UINT; WParam: WPARAM;
                             LParam: LPARAM): UINT; stdcall;
const
  PagePaintWhat: array[WM_PSD_FULLPAGERECT..
                       WM_PSD_YAFULLPAGERECT] of TPSPaintWhat = (
    pwFullPage, pwMinimumMargins, pwMargins,
    pwGreekText, pwEnvStamp, pwYAFullPage
  );
  PRINTER_MASK = $00000002;
  ORIENT_MASK  = $00000004;
  PAPER_MASK   = $00000008;
var
  PaperData: word;
  Paper: TPSPaperType;
  Orient: TPSPaperOrientation;
  Printer: TPSPrinterType;
  PaintRect: TRect;
  PaintCanvas: TCanvas;
begin
  if (Msg = WM_COMMAND) and (LongRec(WParam).Lo = IDPRINTERBTN) and
     (LongRec(WParam).Hi = BN_CLICKED) then
  begin
    // if hander is assigned, use it.  If not, let system do it.
    Result := ord(PageSetupDialog.DoPrinter(Wnd));
  end else begin
    if assigned(PageSetupDialog.FOnInitPaintPage) and
       assigned(PageSetupDialog.FOnPaintPage) then
    begin
      case Msg of
        WM_PSD_PAGESETUPDLG:
          begin
            PaperData := HiWord(WParam);
            if (PaperData AND PAPER_MASK > 0) then
              Paper := ptEnvelope
            else
              Paper := ptPaper;
            if (PaperData AND ORIENT_MASK > 0) then
              Orient := poPortrait
            else
              Orient := poLandscape;
            if (PaperData AND PAPER_MASK > 0) then
              Printer := ptHPPCL
            else
              Printer := ptDotMatrix;
            Result := Ord(PageSetupDialog.FOnInitPaintPage(PageSetupDialog,
               LoWord(WParam), Paper, Orient, Printer, PPSDlgData(LParam)));
          end;
        WM_PSD_FULLPAGERECT,
        WM_PSD_MINMARGINRECT,
        WM_PSD_MARGINRECT,
        WM_PSD_GREEKTEXTRECT,
        WM_PSD_ENVSTAMPRECT,
        WM_PSD_YAFULLPAGERECT:
          begin
            if LParam <> 0 then
              PaintRect := PRect(LParam)^
            else
              PaintRect := Rect(0,0,0,0);
            PaintCanvas := TCanvas.Create;
            PaintCanvas.Handle := HDC(WParam);
            try
              Result := Ord(PageSetupDialog.FOnPaintPage(PageSetupDialog,
                 PagePaintWhat[Msg], PaintCanvas, PaintRect));
            finally
              PaintCanvas.Free;   { This better not be deleting the DC! }
            end;
          end;
      else
        Result := DialogHook(Wnd, Msg, wParam, lParam);
      end;
    end else
      Result := DialogHook(Wnd, Msg, wParam, lParam);
  end;
end;


function TPSRect.Compare(Other: TPSRect): boolean;
begin
  Result := EqualRect(Rect, Other.Rect);
end;

function TPSPoint.Compare(Other: TPSPoint): boolean;
begin
  Result := (X = Other.X) and (Y = Other.Y);
end;

function TPSPoint.GetX: longint;
begin
  Result := FPoint.X;
end;

procedure TPSPoint.SetX(Val: longint);
begin
  FPoint.X := Val;
end;

function TPSPoint.GetY: longint;
begin
  Result := FPoint.Y;
end;

procedure TPSPoint.SetY(Val: longint);
begin
  FPoint.Y := Val;
end;



constructor TPageSetupDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FCentered := TRUE;
  FOptions := [poDefaultMinMargins, poShowHelp];
  FOnPrinter := NIL;
  FOnInitPaintPage := NIL;
  FOnPaintPage := NIL;
  FCustomData := 0;
  FMeasurements := pmDefault;
  FPaperSize := TPSPoint.Create;
  FMinimumMargins := TPSRect.Create;
  FMargins := TPSRect.Create;
  if CurrentMeasurements = pmInches then
  begin
    FPaperSize.Point := DefPaperSizeI.Point;
    FMinimumMargins.Rect := DefMinimumMarginsI.Rect;
    FMargins.Rect := DefMarginsI.Rect;
  end else begin
    FPaperSize.Point := DefPaperSizeM.Point;
    FMinimumMargins.Rect := DefMinimumMarginsM.Rect;
    FMargins.Rect := DefMarginsM.Rect;
  end;
end;

destructor TPageSetupDialog.Destroy;
begin
  FPaperSize.Free;
  FMinimumMargins.Free;
  FMargins.Free;

  inherited Destroy;
end;

procedure TPageSetupDialog.SetName(const NewName: TComponentName);
begin
  inherited Setname(NewName);
  if not (csLoading in ComponentState) then
    ReadCurrentValues;
end;

procedure TPageSetupDialog.SetPaperSize(const Val: TPSPoint);
begin
  FPaperSize.Point := Val.Point;
end;

function TPageSetupDialog.StorePaperSize: boolean;
begin
  if CurrentMeasurements = pmInches then
    Result := not PaperSize.Compare(DefPaperSizeI)
  else
    Result := not PaperSize.Compare(DefPaperSizeM);
end;

procedure TPageSetupDialog.SetMinimumMargins(const Val: TPSRect);
begin
  FMinimumMargins.Rect := Val.Rect;
end;

function TPageSetupDialog.StoreMinimumMargins: boolean;
begin
  if CurrentMeasurements = pmInches then
    Result := not MinimumMargins.Compare(DefMinimumMarginsI)
  else
    Result := not MinimumMargins.Compare(DefMinimumMarginsM);
end;

procedure TPageSetupDialog.SetMargins(const Val: TPSRect);
begin
  FMargins.Rect := Val.Rect;
end;

function TPageSetupDialog.StoreMargins: boolean;
begin
  if CurrentMeasurements = pmInches then
    Result := not Margins.Compare(DefMarginsI)
  else
    Result := not Margins.Compare(DefMarginsM);
end;

procedure TPageSetupDialog.SetMeasurements(Val: TPSMeasurements);
var
  TempVal: TPSMeasurements;
begin
  if Val = pmDefault then
    TempVal := DefaultMeasurements
  else
    TempVal := Val;
  if CurrentMeasurements <> TempVal then
  begin
    if TempVal = pmInches then
    begin
      // Convert to thousandths of an inch
      PaperSize.X := Round(PaperSize.X / 2.54);
      PaperSize.Y := Round(PaperSize.Y / 2.54);
      MinimumMargins.Top := Round(MinimumMargins.Top / 2.54);
      MinimumMargins.Left := Round(MinimumMargins.Left / 2.54);
      MinimumMargins.Right := Round(MinimumMargins.Right / 2.54);
      MinimumMargins.Bottom := Round(MinimumMargins.Bottom / 2.54);
      Margins.Top := Round(Margins.Top / 2.54);
      Margins.Left := Round(Margins.Left / 2.54);
      Margins.Right := Round(Margins.Right / 2.54);
      Margins.Bottom := Round(Margins.Bottom / 2.54);
    end else begin
      // Convert to millimeters
      PaperSize.X := Round(PaperSize.X * 2.54);
      PaperSize.Y := Round(PaperSize.Y * 2.54);
      MinimumMargins.Top := Round(MinimumMargins.Top * 2.54);
      MinimumMargins.Left := Round(MinimumMargins.Left * 2.54);
      MinimumMargins.Right := Round(MinimumMargins.Right * 2.54);
      MinimumMargins.Bottom := Round(MinimumMargins.Bottom * 2.54);
      Margins.Top := Round(Margins.Top * 2.54);
      Margins.Left := Round(Margins.Left * 2.54);
      Margins.Right := Round(Margins.Right * 2.54);
      Margins.Bottom := Round(Margins.Bottom * 2.54);
    end;
  end;
  FMeasurements := Val;
  ReadCurrentValues;
end;

function TPageSetupDialog.GetDefaultMeasurements: TPSMeasurements;
begin
  if GetLocaleChar(LOCALE_USER_DEFAULT,LOCALE_IMEASURE,'0') = '0' then
    Result:= pmMillimeters
  else
    Result:= pmInches;
end;

function TPageSetupDialog.GetCurrentMeasurements: TPSMeasurements;
begin
  if FMeasurements = pmDefault then
    Result := DefaultMeasurements
  else
    Result := FMeasurements;
end;

procedure GetPrinter(var DeviceMode, DeviceNames: THandle);
var
  Device, Driver, Port: array[0..79] of char;
  DevNames: PDevNames;
  Offset: PChar;
begin
  Printer.GetPrinter(Device, Driver, Port, DeviceMode);
  if DeviceMode <> 0 then
  begin
    DeviceNames := GlobalAlloc(GHND, SizeOf(TDevNames) + StrLen(Device) +
       StrLen(Driver) + StrLen(Port) + 3);
    DevNames := PDevNames(GlobalLock(DeviceNames));
    try
      Offset := PChar(DevNames) + SizeOf(TDevnames);
      with DevNames^ do
      begin
        wDriverOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Driver) + 1;
        wDeviceOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Device) + 1;
        wOutputOffset := Longint(Offset) - Longint(DevNames);;
        StrCopy(Offset, Port);
      end;
    finally
      GlobalUnlock(DeviceNames);
    end;
  end;
end;

procedure SetPrinter(DeviceMode, DeviceNames: THandle);
var
  DevNames: PDevNames;
begin
  DevNames := PDevNames(GlobalLock(DeviceNames));
  try
    with DevNames^ do
      Printer.SetPrinter(PChar(DevNames) + wDeviceOffset, PChar(DevNames) +
         wDriverOffset, PChar(DevNames) + wOutputOffset, DeviceMode);
  finally
    GlobalUnlock(DeviceNames);
    GlobalFree(DeviceNames);
  end;
end;

function CopyData(Handle: THandle): THandle;
var
  Src, Dest: PChar;
  Size: Integer;
begin
  if Handle <> 0 then
  begin
    Size := GlobalSize(Handle);
    Result := GlobalAlloc(GHND, Size);
    if Result <> 0 then
      try
        Src := GlobalLock(Handle);
        Dest := GlobalLock(Result);
        if (Src <> nil) and (Dest <> nil) then
           Move(Src^, Dest^, Size);
      finally
        GlobalUnlock(Handle);
        GlobalUnlock(Result);
      end
  end else
    Result := 0;
end;

function TPageSetupDialog.DoExecute(Func: pointer): boolean;
const
  PageSetupOptions: array [TPageSetupOption] of DWORD = (
     PSD_DEFAULTMINMARGINS, PSD_DISABLEMARGINS, PSD_DISABLEORIENTATION,
     PSD_DISABLEPAGEPAINTING, PSD_DISABLEPAPER, PSD_DISABLEPRINTER,
     PSD_NOWARNING, PSD_SHOWHELP
    );
  PageSetupMeasurements: array [TPSMeasurements] of DWORD = (
     0, PSD_INHUNDREDTHSOFMILLIMETERS, PSD_INTHOUSANDTHSOFINCHES
    );
var
  Option: TPageSetupOption;
  PageSetup: TPageSetupDlg;
  SavePageSetupDialog: TPageSetupDialog;
  DevHandle: THandle;
begin
  FillChar(PageSetup, SizeOf(PageSetup), 0);
  PageSetup.hInstance := HInstance;
  with PageSetup do
  try
    lStructSize := SizeOf(TPageSetupDlg);

    if FGettingDefaults then
    begin
      // Using millimeters always fails to retreive margins and minimum margins.
      // Only inches seems to work so I use that and convert.
      Flags := PSD_MARGINS or PSD_DEFAULTMINMARGINS or PSD_RETURNDEFAULT or
         PSD_INTHOUSANDTHSOFINCHES;
    end else begin
      Flags := PSD_MARGINS;
      Flags := Flags OR PageSetupMeasurements[CurrentMeasurements];
      if not (poDefaultMinMargins in FOptions) then
        Flags := Flags or PSD_MINMARGINS;

      if assigned(FOnPrinter) or assigned(FOnInitPaintPage) or
         assigned(FOnPaintPage) or FCentered then
      begin
        Flags := Flags or PSD_ENABLEPAGESETUPHOOK;
        lpfnPageSetupHook := PageSetupDialogHook;
      end;

      for Option := Low(Option) to High(Option) do
        if Option in FOptions then
          Flags := Flags OR PageSetupOptions[Option];
  {    if not assigned(FOnPrinter) then
        Flags := Flags OR PSD_DISABLEPRINTER;}
      if assigned(FOnInitPaintPage) and assigned(FOnPaintPage) then
      begin
        Flags := Flags OR PSD_ENABLEPAGEPAINTHOOK;
        lpfnPagePaintHook := PageSetupDialogHook;
      end;
      HookCtl3D := Ctl3D;
      lCustData := FCustomData;

      GetPrinter(DevHandle, hDevNames);
      hDevMode := CopyData(DevHandle);

      // This appears to do nothing.
      ptPaperSize := FPaperSize.Point;
      rtMinMargin := FMinimumMargins.Rect;
      rtMargin := FMargins.Rect;
      if (Flags and PSD_MINMARGINS) <> 0 then
      begin
        // rtMargin can not be smaller than rtMinMargin or dialog call will fail!
        if rtMargin.Left < rtMinMargin.Left then
          rtMargin.Left := rtMinMargin.Left;
        if rtMargin.Right < rtMinMargin.Right then
          rtMargin.Right := rtMinMargin.Right;
        if rtMargin.Top < rtMinMargin.Top then
          rtMargin.Top := rtMinMargin.Top;
        if rtMargin.Bottom < rtMinMargin.Bottom then
          rtMargin.Bottom := rtMinMargin.Bottom;
      end;
    end;

    hWndOwner := Application.Handle;

    SavePageSetupDialog := PageSetupDialog;
    PageSetupDialog := Self;
    if FGettingDefaults then
      Result := PageSetupDlg(PageSetup)
    else
      Result := TaskModalDialog(Func, PageSetup);
    PageSetupDialog := SavePageSetupDialog;

    if Result then
    begin
      // don't stomp on values that don't match defaults!
      if FGettingDefaults and (CurrentMeasurements = pmMillimeters) then
      begin
        // Defaults are always retreived in inches because it the API won't
        // cooperate with defaults in millimeters.  Have to convert by hand.
        if (csLoading in ComponentState) or
           (DefPaperSizeM.Compare(FPaperSize)) then
        begin
          FPaperSize.X := Round(ptPaperSize.X * 2.54);
          FPaperSize.Y := Round(ptPaperSize.Y * 2.54);
        end;
        if (csLoading in ComponentState) or
           (DefMinimumMarginsM.Compare(FMinimumMargins)) then
        begin
          FMinimumMargins.Left := Round(rtMinMargin.Left * 2.54);
          FMinimumMargins.Top := Round(rtMinMargin.Top * 2.54);
          FMinimumMargins.Right := Round(rtMinMargin.Right * 2.54);
          FMinimumMargins.Bottom := Round(rtMinMargin.Bottom * 2.54);
        end;
        if (csLoading in ComponentState) or
           (DefMarginsM.Compare(FMargins)) then
        begin
          FMargins.Left := Round(rtMargin.Left * 2.54);
          FMargins.Top := Round(rtMargin.Top * 2.54);
          FMargins.Right := Round(rtMargin.Right * 2.54);
          FMargins.Bottom := Round(rtMargin.Bottom * 2.54);
        end;
      end else begin
        FPaperSize.Point := ptPaperSize;
        FMinimumMargins.Rect := rtMinMargin;
        FMargins.Rect := rtMargin;
      end;

      // Only do this if not getting defaults
      if not FGettingDefaults then
        SetPrinter(hDevMode, hDevNames);
    end else begin
      if hDevMode <> 0 then GlobalFree(hDevMode);
      if hDevNames <> 0 then GlobalFree(hDevNames);
    end;
  finally
    { Nothing yet }
  end;
end;

function TPageSetupDialog.ReadCurrentValues: boolean;
begin
  FGettingDefaults := TRUE;
  try
    Result := DoExecute(@PageSetupDlg);
  finally
    FGettingDefaults := FALSE;
  end;
end;

const
  MeasurementsDiv : array [pmMillimeters..pmInches] of TPSMeasureVal = (
     100.0,1000.0
  );

function TPageSetupDialog.FromMeasurementVal(Val: integer): TPSMeasureVal;
begin
  Result := Val / MeasurementsDiv[CurrentMeasurements];
end;

function TPageSetupDialog.ToMeasurementVal(Val: TPSMeasureVal): integer;
const
  MeasurementsDiv : array [pmMillimeters..pmInches] of TPSMeasureVal = (
     100.0,1000.0
  );
begin
  Result := Round(Val * MeasurementsDiv[CurrentMeasurements]);
end;

function TPageSetupDialog.Execute: boolean;
begin
  FGettingDefaults := FALSE; // just in case
  Result := DoExecute(@PageSetupDlg);
end;

function TPageSetupDialog.Printer(Wnd: HWND): boolean;
begin
  Result :=  assigned(FOnPrinter);
  if Result then
    FOnPrinter(Self, Wnd);
end;

function TPageSetupDialog.DoPrinter(Wnd: HWND): boolean;
begin
  try
    Result := Printer(Wnd);
  except
    Result := FALSE;
    Application.HandleException(Self);
  end;
end;

function TPageSetupDialog.GetVersion: TDFSVersion;
begin
  Result := DFS_COMPONENT_VERSION;
end;

procedure TPageSetupDialog.SetVersion(const Val: TDFSVersion);
begin
  { empty write method, just needed to get it to show up in Object Inspector }
end;



{ Initialization and cleanup }

procedure InitGlobals;
var
  PageSetup: TPageSetupDlg;
begin
  HelpMsg := RegisterWindowMessage(HelpMsgString);

  DefPaperSizeI := TPSPoint.Create;
  DefMinimumMarginsI := TPSRect.Create;
  DefMarginsI := TPSRect.Create;

  FillChar(PageSetup, SizeOf(PageSetup), 0);
  PageSetup.hInstance := HInstance;
  with PageSetup do
  begin
    lStructSize := SizeOf(TPageSetupDlg);
    hWndOwner := Application.Handle;
    Flags := PSD_MARGINS or PSD_DEFAULTMINMARGINS or PSD_INTHOUSANDTHSOFINCHES
       or PSD_RETURNDEFAULT;
    if PageSetupDlg(PageSetup) then
    begin
      DefPaperSizeI.Point := ptPaperSize;
      DefMinimumMarginsI.Rect := rtMinMargin;
      DefMarginsI.Rect := rtMargin;
    end;
    if hDevMode <> 0 then GlobalFree(hDevMode);
    if hDevNames <> 0 then GlobalFree(hDevNames);
  end;

  DefPaperSizeM := TPSPoint.Create;
  DefMinimumMarginsM := TPSRect.Create;
  DefMarginsM := TPSRect.Create;

  // convert 1/1000 of inches to 1/100 of millimeters
  DefPaperSizeM.X := Round(DefPaperSizeI.X * 2.54);
  DefPaperSizeM.Y := Round(DefPaperSizeI.Y * 2.54);
  DefMinimumMarginsM.Top := Round(DefMinimumMarginsI.Top * 2.54);
  DefMinimumMarginsM.Left := Round(DefMinimumMarginsI.Left * 2.54);
  DefMinimumMarginsM.Right := Round(DefMinimumMarginsI.Right * 2.54);
  DefMinimumMarginsM.Bottom := Round(DefMinimumMarginsI.Bottom * 2.54);
  DefMarginsM.Top := Round(DefMarginsI.Top * 2.54);
  DefMarginsM.Left := Round(DefMarginsI.Left * 2.54);
  DefMarginsM.Right := Round(DefMarginsI.Right * 2.54);
  DefMarginsM.Bottom := Round(DefMarginsI.Bottom * 2.54);
end;

procedure DoneGlobals;
begin
  DefPaperSizeI.Free;
  DefMinimumMarginsI.Free;
  DefMarginsI.Free;
  DefPaperSizeM.Free;
  DefMinimumMarginsM.Free;
  DefMarginsM.Free;
end;

{$IFDEF DFS_DEBUG}
var
  t: dword;
{$ENDIF}

initialization
{$IFDEF DFS_DEBUG}
  t := timegettime;
{$ENDIF}
  InitGlobals;
{$IFDEF DFS_DEBUG}
  odm('Milliseconds: ', timegettime - t);
{$ENDIF}

finalization
  DoneGlobals;
end.

