 (*************************************************)
 (*                                               *)
 (*   Text Reader v3.01 for Delphi 3/4/5/6/7      *)
 (*                                               *)
 (*   Copiright 1999 - 2003 by Qarsoft            *)
 (*   All rights reserved                         *)
 (*                                               *)
 (*************************************************)

{$D+,L+,Y+,B+}

unit Reader;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DList;

const
  MaxPos=6-1;

type
  TStatus=(sNone,
           sWord,
           sInteger,
           sStandardNumber,
           sScientificNumber,
           sKeyWord,
           sDelimiter);

  TNumericFormat = (nfNone, nfInteger, nfPeriod, nfComma);

  TDisposeProperties = procedure(Sender: TObject;
    Structure: Integer; Properties: Pointer) of Object;
  TRegList = procedure(Sender: TObject; Str: string;
    Status: TStatus; Structure: Integer; Properties: Pointer) of Object;

  TReader = class(TComponent)
  private
    { Private declarations }
    FOrWildCard, FAndWildCard: char;
    FNumericFormat: TNumericFormat;
    FWildCardEnabled, FScientificNumber: boolean;
    FOnDisposeProperties: TDisposeProperties;
    FOnRegList: TRegList;
    FCaseSensitivity: boolean;
  protected
    { Protected declarations }
    FreeSymbols: array[char] of boolean;
    DecimalPoint: char;
    procedure DisposeProperties(Sender: TObject; Structure:integer;
                                Properties: pointer);
    procedure Clear(Status: TStatus);
    procedure SetNumericFormat(NM: TNumericFormat);
    procedure SetCaseSensitivity(CS: boolean);
  public
    { Public declarations }
    DList: TDList;
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    Procedure AddKeyWord(Str: string; Struct: integer; Properties: pointer);
    Procedure AddDelimiter(Str: string; Struct: integer; Properties: pointer);
    procedure RegStandardDelimiters;
    Procedure AddFreeSymbol(c:char);
    procedure RegStandardFreeSymbols;
    procedure RegList;
    function IsDelimiter(s: string): boolean;
    function IsFreeSymbol(c: char): boolean;
    function GetProperties(Str: string; var Status: TStatus;
      var Structure: integer; var Properties: pointer): boolean;
    procedure Remove(Str: string);
    procedure RemoveFreeSymbol(c: char);
    procedure ClearKeyWords;
    procedure ClearFreeSymbols;
    procedure ClearDelimiters;
    procedure ClearAll;
  published
    { Published declarations }
    property WildCardEnabled: boolean read FWildCardEnabled
      write FWildCardEnabled stored true;
    property OrWildCard: char read FOrWildCard write FOrWildCard stored true;
    property AndWildCard: char
      read FAndWildCard write FAndWildCard stored true;
    property NumericFormat: TNumericFormat
      read FNumericFormat write SetNumericFormat stored true;
    property ScientificNumber: boolean
      read FScientificNumber write FScientificNumber stored true;
    property CaseSensitivity: boolean
      read FCaseSensitivity write SetCaseSensitivity stored true;
    property OnDisposeProperties: TDisposeProperties
      read FOnDisposeProperties write FOnDisposeProperties;
    property OnRegList: TRegList read FOnRegList write FOnRegList;
  end;

  PSymProperties=^TSymProperties;
  TSymProperties=record
    OrigName: String;
    Status: TStatus;
    Additional: pointer;
  end;

  TPosition=record
    ActivePos, Structure: integer;
    ActiveStr, FreeStr: string;
    Status: TStatus;
    Additional: pointer;
  end;

  TPosList=array [0..MaxPos] of TPosition;

  TScroller = class
  private
    { Private declarations }
    Reader: TReader;
    Tracer: TTracer;
    MainText: string;
    TextLen: integer;
    procedure ResetPosition;
    procedure LoadNext;
    procedure CheckStatus;
  public
    { Public declarations }
    PrevPos: TPosition;
    PosList: TPosList;
    constructor Create(SymSet: TReader);
    destructor Destroy; override;
    procedure First(Text: string; StartPos: integer);
    procedure Next;
  end;

implementation

{TReader}
constructor TReader.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  DList:=TDList.Create(Self);
  DList.Separator:=#0;
  DList.OnDisposeStructure:=DisposeProperties;
  ScientificNumber:=true;
  WildCardEnabled:=false;
  OrWildCard:='~';
  AndWildCard:='`';
  NumericFormat:=nfPeriod;
  CaseSensitivity:=false;
  ClearFreeSymbols;
end;

Destructor TReader.Destroy;
begin
  DList.Destroy;
  inherited Destroy;
end;

procedure TReader.SetNumericFormat(NM: TNumericFormat);
begin
  FNumericFormat:=NM;
  if NM = nfComma then DecimalPoint:=','
  else DecimalPoint:='.';
end;

procedure TReader.SetCaseSensitivity(CS: boolean);
begin
  FCaseSensitivity := CS;
  ClearKeyWords;
  ClearDelimiters;
end;

procedure TReader.DisposeProperties(Sender: TObject; Structure: integer;
  Properties: pointer);
begin
  if PSymProperties(Properties)<>nil then begin
    if PSymProperties(Properties)^.Additional<>nil then
      if Assigned(FOnDisposeProperties) then
        FOnDisposeProperties(Self, Structure,
          PSymProperties(Properties)^.Additional);
    FreeMem(Properties, SizeOf(TSymProperties));
  end;
end;

procedure TReader.RegList;
var
  Str: string;
  Struct: integer;
  p: pointer;
begin
  if Assigned(FOnRegList) then with DList do begin
    SetFilter('', 0, 0);
    if First then repeat
      if GetName(Str) and GetStructure(Struct) and GetProperties(p) then begin
        if PSymProperties(p)^.OrigName = '' then
          FOnRegList(Self, Str, PSymProperties(p)^.Status, Struct,
            PSymProperties(p)^.Additional)
        else FOnRegList(Self, PSymProperties(p)^.OrigName,
          PSymProperties(p)^.Status, Struct, PSymProperties(p)^.Additional);
      end;
    until not Next;
  end;
end;

procedure TReader.AddKeyWord(Str: string; Struct: integer; Properties: pointer);
var
  SP:PSymProperties;
begin
  New(SP);
  with SP^ do begin
    Status:=sKeyWord;
    OrigName:=Str;
    Additional:=properties;
  end;
  if CaseSensitivity then DList.Add(Str,Struct,SP)
  else DList.Add(AnsiLowerCase(Str),Struct,SP);
end;

procedure TReader.AddDelimiter(Str: string; Struct: integer; Properties: pointer);
var
  SP: PSymProperties;
  p: pointer;
  OldStruct, i: integer;
procedure AddDel;
begin
  New(SP);
  with SP^ do begin
    Status:=sDelimiter;
    OrigName:=Str[i];
    Additional:=nil;
  end;
  if (Str[i]<>OrWildCard) and (Str[i]<>AndWildCard) then
    DList.Add(Str[i], 0, SP);
end;
begin
  if not CaseSensitivity then Str:=AnsiLowerCase(Str);
  if Length(Str)>1 then for i:=1 to Length(Str) do begin
    if not DList.getFields(Str[i], OldStruct, p) then AddDel
    else if p=nil then AddDel
    else if PSymProperties(p)^.Status<>sDelimiter then AddDel;
  end;
  New(SP);
  with SP^ do begin
    Status:=sDelimiter;
    OrigName:=Str;
    Additional:=properties;
  end;
  DList.Add(Str, Struct, SP);
end;

procedure TReader.RegStandardDelimiters;
begin
  ClearDelimiters;
  AddDelimiter('!', 0, nil);
  AddDelimiter('@', 0, nil);
  AddDelimiter('#', 0, nil);
  AddDelimiter('$', 0, nil);
  AddDelimiter('%', 0, nil);
  AddDelimiter('^', 0, nil);
  AddDelimiter('&', 0, nil);
  AddDelimiter('*', 0, nil);
  AddDelimiter('(', 0, nil);
  AddDelimiter(')', 0, nil);
//  AddDelimiter('_', 0, nil);
  AddDelimiter('+', 0, nil);
  AddDelimiter('|', 0, nil);
  AddDelimiter('-', 0, nil);
  AddDelimiter('=', 0, nil);
  AddDelimiter('\', 0, nil);
  AddDelimiter('[', 0, nil);
  AddDelimiter(']', 0, nil);
  AddDelimiter('{', 0, nil);
  AddDelimiter('}', 0, nil);
  AddDelimiter(';', 0, nil);
  AddDelimiter('''', 0, nil);
  AddDelimiter(':', 0, nil);
  AddDelimiter('"', 0, nil);
  AddDelimiter(',', 0, nil);
  AddDelimiter('.', 0, nil);
  AddDelimiter('/', 0, nil);
  AddDelimiter('<', 0, nil);
  AddDelimiter('>', 0, nil);
  AddDelimiter('?', 0, nil);
end;

Procedure TReader.AddFreeSymbol(c: char);
begin
  FreeSymbols[c]:=true;
end;

procedure TReader.RegStandardFreeSymbols;
var
  i: byte;
begin
  ClearFreeSymbols;
  for i:=0 to 32 do FreeSymbols[chr(i)]:=true;
end;

function TReader.IsDelimiter(s: string): boolean;
var
  Struct: integer;
  p: pointer;
  Found: boolean;
begin
  if CaseSensitivity then Found := DList.GetFields(s, Struct, p)
  else Found := DList.GetFields(AnsiLowerCase(s), Struct, p);
  if Found then
    if p=nil then Result:=false
    else Result:=PSymProperties(p)^.Status=sDelimiter
  else Result:=false;
end;

function TReader.IsFreeSymbol(c: char): boolean;
begin
  Result:=FreeSymbols[c];
end;

function TReader.GetProperties(Str: string; var Status: TStatus;
  var Structure: integer; var Properties: pointer): boolean;
begin
  if not CaseSensitivity then Str:=AnsiLowerCase(Str);
  if DList.GetFields(Str, Structure, Properties) then begin
    Result:=true;
    Status:=PSymProperties(Properties)^.Status;
    Properties:=PSymProperties(Properties)^.Additional;
  end
  else Result:=false;
end;

procedure TReader.Remove(Str: string);
begin
  if CaseSensitivity then DList.Remove(Str)
  else DList.Remove(AnsiLowerCase(Str));
end;

procedure TReader.RemoveFreeSymbol(c: char);
begin
  FreeSymbols[c]:=false;
end;

procedure TReader.Clear(Status: TStatus);
var
  ClearTable: TDList;
  p: pointer;
  s: string;
begin
  ClearTable:=TDList.Create(Self);
  try
    with DList do if First then repeat
      if GetProperties(p) then
        if PSymProperties(p)^.Status=Status then begin
          GetName(s);
          ClearTable.Add(s, 0, nil);
        end;
    until not Next;
    with ClearTable do if first then repeat
      GetName(s);
      DList.Remove(s);
    until not Next;
  finally
    ClearTable.Destroy;
  end;
end;

procedure TReader.ClearKeyWords;
begin
  Clear(sKeyWord);
end;

procedure TReader.ClearFreeSymbols;
var
  c: char;
begin
  for c:=Low(char) to High(char) do FreeSymbols[c]:=false;
end;

procedure TReader.ClearDelimiters;
begin
  Clear(sDelimiter);
end;

procedure TReader.ClearAll;
begin
  DList.ClearAll;
  ClearFreeSymbols;
end;

{TScroller}
constructor TScroller.Create(SymSet: TReader);
begin
  inherited Create;
  Reader:=SymSet;
  Tracer:=TTracer.Create(Reader.DList);
end;

destructor TScroller.Destroy;
begin
  Tracer.Destroy;
  inherited Destroy;
end;

procedure TScroller.ResetPosition;
begin
  with PosList[MaxPos] do begin
    ActiveStr:='';
    FreeStr:='';
    Structure:=0;
    Status:=sNone;
    Additional:=nil;
  end;
end;

procedure TScroller.LoadNext;
type
  TPos = record
    TracePos: TTracePos;
    Active: boolean;
    Properties: pointer;
    Position, Structure, MarkSize: integer;
    Str: string;
  end;
var
  CurPos, LastActive, CurStructure, MaxSize, CurMarkSize: integer;
  CurStr, s: string;
  CurActive, DelimiterExpected, Tracing, EndScroll,
    Compatible, b: boolean;
  CurProperties: pointer;
  WCPos: TPos;
procedure SavePos(var Pos: TPos);
begin
  with Pos do begin
    Tracer.SaveTracePos(TracePos);
    Active:=CurActive;
    Properties:=CurProperties;
    Position:=CurPos;
    Structure:=CurStructure;
    Str:=CurStr;
    MarkSize:=CurMarkSize;
  end;
end;
procedure RestorePos(Pos: TPos);
begin
  with Pos do begin
    Tracer.RestoreTracePos(TracePos);
    CurActive:=Active;
    CurProperties:=Properties;
    CurPos:=Position;
    CurStructure:=Structure;
    CurStr:=Str;
    CurMarkSize:=MarkSize;
  end;
end;
procedure FixWord;
begin
  with PosList[MaxPos] do begin
    ActivePos:=CurPos;
    ActiveStr:=CurStr;
    Status:=sWord;
    Structure:=0;
    Additional:=nil;
  end;
end;
procedure Fix(var Res: boolean);
var
  Comp: boolean;
begin
  if CurActive then Comp:=(PSymProperties(CurProperties)^.Status=sDelimiter) =
    DelimiterExpected else Comp:=false;
  if Comp then with PosList[MaxPos] do begin
    if not DelimiterExpected or (MaxSize < Length(CurStr)) then begin
      ActivePos:=CurPos;
      LastActive:=ActivePos;
      ActiveStr:=CurStr;
      MaxSize:=Length(ActiveStr);
      Status:=PSymProperties(CurProperties)^.Status;
      Structure:=CurStructure;
      Additional:=PSymProperties(CurProperties)^.Additional;
      Res:=true;
    end;
  end
  else if not DelimiterExpected then begin
    FixWord;
    Res:=false;
  end;
end;
function NewTraceComplet(OrWildCardMode: boolean; OrWildCardPos: TPos;
  Marker1, Marker2: string): boolean;
var
  Pos1, Pos2: TPos;
begin
  Result:=false;
  while true do begin
    if Marker1[CurMarkSize] = '0' then begin
      SavePos(Pos1);
      while Tracer.TraceNext(Reader.OrWildCard,
        CurActive, CurStructure, CurProperties) do begin
        SavePos(Pos2);
        RestorePos(Pos1);
        s:=Marker1;
        s[CurMarkSize]:='1';
        if NewTraceComplet(OrWildCardMode, OrWildCardPos, s, Marker2) then begin
          Result:=true;
          if not DelimiterExpected then Exit;
        end;
        RestorePos(Pos2);
        if CurActive then b:=(PSymProperties(CurProperties)^.Status=sDelimiter)=
          DelimiterExpected else b:=false;
        with PosList[MaxPos] do if b and (LastActive < CurPos-1) then begin
          if EndScroll then s:=CurStr
          else s:=Copy(CurStr, 1, Length(CurStr)-1);
          if not DelimiterExpected or (MaxSize < Length(s)) then begin
            ActivePos:=CurPos-1;
            LastActive:=ActivePos;
            ActiveStr:=s;
            MaxSize:=Length(ActiveStr);
            Structure:=CurStructure;
            Status:=PSymProperties(CurProperties)^.Status;
            Additional:=PSymProperties(CurProperties)^.Additional;
            Result:=true;
          end;
        end;
        OrWildCardMode:=true;
        inc(CurMarkSize);
        if CurMarkSize > Length(Marker1) then begin
          Marker1:=Marker1+'0';
          Marker2:=Marker2+'0';
        end;
        SavePos(OrWildCardPos);
        SavePos(Pos1);
      end;
    end;
    if CurPos > TextLen then Exit;
    if Reader.IsFreeSymbol(MainText[CurPos]) then Exit;
    if DelimiterExpected <> Reader.IsDelimiter(MainText[CurPos]) then Exit;
    if Marker2[CurMarkSize] = '0' then begin
      SavePos(Pos1);
      if Tracer.TraceNext(Reader.AndWildCard, CurActive,
        CurStructure, CurProperties) then begin
        if CurMarkSize = Length(Marker2) then begin
          SavePos(Pos2);
          RestorePos(Pos1);
          if NewTraceComplet(OrWildCardMode, OrWildCardPos, Marker1,
            Copy(Marker2, 1, Length(Marker2)-1)+'1') then begin
            Result:=true;
            if not DelimiterExpected then Exit;
          end;
          RestorePos(Pos2);
        end;
        inc(CurMarkSize);
        if CurMarkSize > Length(Marker1) then begin
          Marker1:=Marker1+'0';
          Marker2:=Marker2+'0';
        end;
        Compatible:=true;
       end
      else Compatible:=false;
    end
    else Compatible:=false;
    if not Compatible then begin
      if Reader.CaseSensitivity then
        b:=Tracer.TraceNext(MainText[CurPos], CurActive, CurStructure, CurProperties)
      else b:=Tracer.TraceNext(AnsiLowerCase(MainText[CurPos])[1], CurActive,
        CurStructure, CurProperties);
      if b then begin
        inc(CurMarkSize);
        if CurMarkSize > Length(Marker1) then begin
          Marker1:=Marker1+'0';
          Marker2:=Marker2+'0';
        end;
      end
      else if OrWildCardMode then begin
        RestorePos(OrWildCardPos);
        with OrWildCardPos do if Position < TextLen then begin
          inc(Position);
          Str:=Str+MainText[Position];
        end;
      end
      else begin
        if not DelimiterExpected then Result:=false;
        Exit;
      end;
    end;
    Fix(Result);
    Inc(CurPos);
    if CurPos > TextLen then EndScroll:=true
    else if Reader.IsFreeSymbol(MainText[CurPos]) then EndScroll:=true
    else begin
      CurStr:=CurStr+MainText[CurPos];
      EndScroll:=false;
    end;
  end;
end;
begin
  with PosList[MaxPos] do begin
    ResetPosition;
    CurPos:=ActivePos;
    repeat
      inc(CurPos);
      if CurPos>TextLen then Exit;
    until not Reader.IsFreeSymbol(MainText[CurPos]);
    FreeStr:=Copy(MainText, ActivePos + 1, CurPos - ActivePos - 1);
    DelimiterExpected:=Reader.IsDelimiter(MainText[CurPos]);
    CurActive:=false;
    MaxSize:=0;
    Tracer.BeginTrace('');
    if Reader.WildCardEnabled then begin
      CurStr:=MainText[CurPos];
      LastActive:=0;
      CurMarkSize:=1;
      EndScroll:=false;
      if not NewTraceComplet(false, WCPos, '0', '0') then
        while CurPos<=TextLen do begin
          if DelimiterExpected or Reader.IsFreeSymbol(MainText[CurPos]) or
            Reader.IsDelimiter(MainText[CurPos]) then Exit;
          FixWord;
          Inc(CurPos);
          CurStr:=CurStr+MainText[CurPos];
        end;
    end
    else begin
      CurStr:='';
      Tracing:=true;
      repeat
        CurStr:=CurStr+MainText[CurPos];
        if Tracing then begin
          if Reader.CaseSensitivity then
            b:=Tracer.TraceNext(MainText[CurPos], CurActive, CurStructure, CurProperties)
          else b:=Tracer.TraceNext(AnsiLowerCase(MainText[CurPos])[1], CurActive,
            CurStructure, CurProperties);
          if b then begin
            if DelimiterExpected=Reader.IsDelimiter(MainText[CurPos]) then Fix(b)
            else Exit;
          end
          else Tracing:=false;
        end;
        if not Tracing then begin
          if DelimiterExpected or Reader.IsDelimiter(MainText[CurPos]) or
            Reader.IsFreeSymbol(MainText[CurPos]) then Exit;
          FixWord;
        end;
        Inc(CurPos);
      until CurPos>TextLen;
    end;
  end;
end;

procedure TScroller.CheckStatus;
type
  TPosListShadow = array[0..MaxPos] of integer; 
var
  CurPosition, PrevPosition, i: integer;
  IsStandardNumber, IsScientificNumber: boolean;
  Str: string;
  PosListShadow: TPosListShadow;
procedure UpdatePosList;
var
  i: integer;
begin
  with PosList[0] do begin
    ActivePos:=PosList[1].ActivePos;
    ActiveStr:=ActiveStr+PosList[1].ActiveStr;
    Structure:=0;
    Additional:=nil;
  end;
  for i:=1 to MaxPos-1 do PosList[i]:=PosList[i+1];
  LoadNext;
end;
begin
  if (Reader.NumericFormat<>nfNone) and
    (PosList[0].Status<>sNone) and (PosList[0].Status<>sKeyWord)and
    (PosList[0].ActiveStr<>'-') and (PosList[0].ActiveStr<>'+') then begin
    IsStandardNumber:=false;
    IsScientificNumber:=false;
    Str:=PosList[0].ActiveStr;
    PosListShadow[0]:=Length(Str);
    i:=1;
    while (i<MaxPos) and
      (PosList[i].Status<>sNone) and (PosList[i].Status<>sKeyWord) and
      (PosList[i].FreeStr='') do begin
      Str:=Str+PosList[i].ActiveStr;
      PosListShadow[i]:=Length(Str);
      inc(i);
    end;
    if Length(Str)>0 then begin
      CurPosition:=0;
      if Str[1] in ['0'..'9'] then begin
        i:=2;
        if PosListShadow[CurPosition]=1 then begin
          PosList[0].Status:=sInteger;
          inc(CurPosition);
        end;
      end
      else if (Str[1]='-') or (Str[1]='+') then begin
        if not (Str[2] in ['0'..'9']) then Exit;
        i:=3;
        if PosListShadow[CurPosition]=2 then begin
          PosList[0].Status:=sInteger;
          inc(CurPosition);
        end;
      end
      else Exit;
      PrevPosition:=0;
      while i<=Length(Str) do begin
        if not (Str[i] in ['0'..'9']) then begin
          if Reader.NumericFormat = nfInteger then Exit;
          if not IsScientificNumber and Reader.ScientificNumber and
            (LowerCase(Str[i])='e') then begin
            if i=length(Str) then Exit;
            if (Str[i+1]='-') or (Str[i+1]='+') then begin
              if i+1=length(Str) then Exit;
              if PosListShadow[CurPosition]=i then inc(CurPosition);
              inc(i);
            end;
            if not (Str[i+1] in ['0'..'9']) then Exit;
            IsScientificNumber:=true;
            if PosListShadow[CurPosition]=i then inc(CurPosition);
            inc(i);
          end
          else if not IsStandardNumber and not IsScientificNumber and
            (Str[i]=Reader.DecimalPoint) then begin
            if i=length(Str) then Exit;
            if not (Str[i+1] in ['0'..'9']) then Exit;
            IsStandardNumber:=true;
            if PosListShadow[CurPosition]=i then inc(CurPosition);
            inc(i);
          end
          else Exit;
        end;
        if PosListShadow[CurPosition] = i then begin
          if IsScientificNumber then PosList[0].Status:=sScientificNumber
          else if IsStandardNumber then PosList[0].Status:=sStandardNumber
          else PosList[0].Status:=sInteger;
          while PrevPosition < CurPosition do begin
            UpdatePosList;
            inc(PrevPosition);
          end;
          inc(CurPosition);
        end;
        inc(i);
      end;
    end;
  end;
end;

procedure TScroller.First(Text: string; StartPos: integer);
var
  i: integer;
begin
  MainText:=Text;
  TextLen:=Length(MainText);
  with PosList[0] do begin
    ActivePos:=StartPos;
    ActiveStr:='';
    FreeStr:='';
    Status:=sNone;
    Structure:=0;
    Additional:=nil;
  end;
  PosList[MaxPos].ActivePos:=StartPos;
  for i:=1 to MaxPos do begin
    LoadNext;
    if i<MaxPos then PosList[i]:=PosList[MaxPos];
  end;
  Next;
end;

procedure TScroller.Next;
var
  i: integer;
begin
  PrevPos:=PosList[0];
  for i:=0 to MaxPos-1 do PosList[i]:=PosList[i+1];
  LoadNext;
  CheckStatus;
end;

end.


