unit QfelDriveComboBox;

interface

uses
  Windows, Messages, Classes, Graphics, Controls, StdCtrls;

type

  TQfelDriveComboBox = class(TCustomComboBox)
    private
      Drives         : TStrings;
      Images         : TImagelist;
      DriveItemIndex : Integer;
      FDrive         : Char;
      FOnChange      : TNotifyEvent;
      procedure CMFontChanged(var Msg:TMessage); message CM_FONTCHANGED;
      procedure ResetItemHeight;
    protected
      procedure CreateWnd; override;
      procedure SetDrive(Value:char);
      procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
      procedure CNCommand(var Msg: TWMCommand); message CN_COMMAND;
      procedure BuildList; virtual;
      procedure Change; override;
    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure UpdateDrives;
    published
      property Color;
      property Ctl3D;
      property DragMode;
      property DragCursor;
      property Drive: char read FDrive write SetDrive;
      property Enabled;
      property Font;
      property ImeMode;
      property ImeName;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnDropDown;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnStartDrag;
  end;

procedure Register;

implementation

uses
  ShellAPI, ImgList;

const
  SHGFI = SHGFI_SYSICONINDEX or SHGFI_SMALLICON;

function GetItemHeight(Font:TFont):Integer;
var
  DC       : HDC;
  SaveFont : HFont;
  Metrics  : TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC,Font.Handle);
  GetTextMetrics(DC,Metrics);
  SelectObject(DC,SaveFont);
  ReleaseDC(0,DC);
  Result := Metrics.tmHeight;
end;

constructor TQfelDriveComboBox.Create(AOwner: TComponent);
begin
  //inherited
  inherited Create(AOwner);
  Style := csOwnerDrawFixed;
  ImeMode := imDisable;
  //private objects
  Drives := TStringList.Create;
  Images := TImagelist.CreateSize(GetSystemMetrics(SM_CXSMICON),GetSystemMetrics(SM_CYSMICON));
  with Images do
    begin
      DrawingStyle := dsTransparent;
      ShareImages := True;
    end;
  //initialization
  ResetItemHeight;
end;

destructor TQfelDriveComboBox.Destroy;
begin
  Drives.Free;
  Images.Free;
  inherited Destroy;
end;

procedure TQfelDriveComboBox.BuildList;
var
  Info      : TSHFileInfo;
  DriveChar : Char;
  CurrDrive : string;
  DriveType:Integer;
begin
  if Items.Count>0 then
    begin
      if ItemIndex>-1 then
        DriveItemIndex := ItemIndex;
      Items.Clear;
    end;
  Images.Handle := SHGetFileInfo('',0,Info,SizeOf(TShFileInfo),SHGFI);
  for DriveChar:='A' to 'Z' do
    begin
      CurrDrive := DriveChar + ':\';
      DriveType := GetDriveType(PChar(CurrDrive));
      if DriveType in [0,1] then
        Continue; // Invalid drive specification
    SHGetFileInfo(PChar(CurrDrive),0,Info,SizeOf(TShFileInfo),SHGFI_DISPLAYNAME or SHGFI);
    Items.AddObject(Info.szDisplayName,TObject(Info.iIcon));
    Drives.Add(DriveChar);
  end;
  SetDrive(Drives[DriveItemIndex][1]);
  Update;
end;


procedure TQfelDriveComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

procedure TQfelDriveComboBox.DrawItem(Index:Integer;Rect:TRect;State:TOwnerDrawState);
var
  ImageTop : Integer;
begin
  with Canvas do
    begin
      FillRect(Rect);
      if Images.Count>0 then
        begin
          ImageTop := Rect.Top + ((Rect.Bottom-Rect.Top-Images.Height) div 2);
          Images.Draw(Canvas,Rect.Left+4,ImageTop,Integer(Items.Objects[Index]));
          Rect.Left := Rect.Left + Images.Width + 8;
        end;
      DrawText(Canvas.Handle,PChar(Items[Index]),-1,Rect,DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX or DT_NOCLIP);
  end;
end;

procedure TQfelDriveComboBox.CMFontChanged(var Msg:TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TQfelDriveComboBox.ResetItemHeight;
var
  NewHeight : Integer;
  MinHeight : Integer;
begin
  MinHeight := Images.Height + 2;
  NewHeight := GetItemHeight(Font);
  if NewHeight<MinHeight then
    NewHeight := MinHeight;
  ItemHeight := NewHeight;
end;

procedure TQfelDriveComboBox.SetDrive(Value:Char);
var
  i : Integer;
  j : Integer;
begin
  j := 0;
  if DriveItemIndex<>-1 then
    j := DriveItemIndex;
  Value := UpCase(Value);
  if FDrive<>Value then
    begin
      for i := 0 to Items.Count - 1 do
        if Drives[i][1] = Value then
          begin
            FDrive := Value;
            DriveItemIndex := i;
            ItemIndex := i;
            Exit;
          end;
    end
  else
    ItemIndex := j;
end;

procedure TQfelDriveComboBox.Change;
begin
 if ItemIndex<>-1 then
   DriveItemIndex := ItemIndex;
 SetDrive(Drives[DriveItemIndex][1]);
 if Assigned(FOnChange) then
   FOnChange(Self);
end;

procedure TQfelDriveComboBox.CNCommand(var Msg: TWMCommand);
begin
  inherited;
  if Msg.NotifyCode=CBN_SELCHANGE then
    Change;
end;

procedure TQfelDriveComboBox.UpdateDrives;
var
  Info : TSHFileInfo;
begin
  LockWindowUpdate(Handle);
  if Assigned(Images) then
    Images.Free;
  Images := TImagelist.CreateSize(GetSystemMetrics(SM_CXSMICON),GetSystemMetrics(SM_CYSMICON));
  with Images do
    begin
      DrawingStyle := dsTransparent;
      ShareImages := True;
    end;
  Images.Handle := SHGetFileInfo('',0,Info,SizeOf(TShFileInfo),SHGFI);
  BuildList;
  LockWindowUpdate(0);
end;

procedure Register;
begin
  RegisterComponents('Qfel', [TQfelDriveComboBox]);
end;

end.
