unit PLabel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TTextLayout = (tlTop, tlCenter, tlBottom);

  TPLabel = class(TGraphicControl)
  private
    FFocusControl: TWinControl;
    FAlignment: TAlignment;
    FAutoSize: Boolean;
    FLayout: TTextLayout;
    FWordWrap: Boolean;
    FShowAccelChar: Boolean;
    FShowColors: Boolean;
    FSelColor: TColor;
    FSelStart: Integer;
    procedure AdjustBounds;
    procedure DoDrawText(var Rect: TRect; Flags: Word);
    procedure DoDrawColorText;
    function GetTransparent: Boolean;
    procedure SetAlignment(Value: TAlignment);
    procedure SetFocusControl(Value: TWinControl);
    procedure SetShowAccelChar(Value: Boolean);
    procedure SetTransparent(Value: Boolean);
    procedure SetLayout(Value: TTextLayout);
    procedure SetWordWrap(Value: Boolean);
    procedure SetShowColors(Value: Boolean);
    procedure SetSelColor(Value: TColor);
    procedure SetSelStart(Value: Integer);
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    function GetLabelText: string; virtual;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Paint; override;
    procedure SetAutoSize(Value: Boolean); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    property Canvas;
  published
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property FocusControl: TWinControl read FFocusControl write SetFocusControl;
    property ShowAccelChar: Boolean read FShowAccelChar write SetShowAccelChar default True;
    property Transparent: Boolean read GetTransparent write SetTransparent default False;
    property Layout: TTextLayout read FLayout write SetLayout default tlTop;
    property WordWrap: Boolean read FWordWrap write SetWordWrap default False;
    property ShowColors: Boolean read FShowColors write SetShowColors default True;
    property SelColor: TColor read FSelColor write SetSelColor;
    property SelStart: Integer read FSelStart write SetSelStart;
    property Align;
    property Caption;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;


procedure Register;

implementation

constructor TPLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque, csReplicatable];
  Width := 65;
  Height := 17;
  FShowAccelChar := True;
  FShowColors:=True;
  FSelColor:=clBlue;
  FAlignment:=taCenter;
  FLayout:=tlCenter;
  FAutoSize:=False;
end;

function TPLabel.GetLabelText: string;
begin
  Result := Caption;
end;

procedure TPLabel.DoDrawText(var Rect: TRect; Flags: Word);
var
  Text: string;
begin
  Text := GetLabelText;
  if (Flags and DT_CALCRECT <> 0) and ((Text = '') or FShowAccelChar and
    (Text[1] = '&') and (Text[2] = #0)) then Text := Text + ' ';
  if (not FShowAccelChar) then Flags := Flags or DT_NOPREFIX;
  Canvas.Font := Font;
  if not Enabled then
  begin
    OffsetRect(Rect, 1, 1);
    Canvas.Font.Color := clBtnHighlight;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
    OffsetRect(Rect, -1, -1);
    Canvas.Font.Color := clBtnShadow;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
  end
  else begin
    If FShowColors then DoDrawColorText
    else DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
  end;
end;

procedure TPLabel.DoDrawColorText;
Var Rect: TRect;
    TH, TW, X: Integer;
    Ch: Array [0..0] of Char;
begin
  If Length(Text)=0 then Exit;
  TH:=Canvas.TextHeight(Text);
  TW:=Canvas.TextWidth(Text);
  Case FLayout of
    tlTop: begin
      Rect.Top:=0;
      Rect.Bottom:=TH;
    end;
    tlCenter: begin
      Rect.Top:=(Height-Canvas.TextHeight(Text)) div 2;
      Rect.Bottom:=Rect.Top+Height;
    end;
    tlBottom: begin
      Rect.Top:=Height-TH;
      Rect.Bottom:=Height;
    end;
  end;
  Case FAlignment of
    taLeftJustify: Rect.Left:=0;
    taCenter: Rect.Left:=(Width-TW) div 2;
    taRightJustify: Rect.Left:=Width-TW;
  end;
  For X:=1 to Length(Text) do begin
    Rect.Right:=Rect.Left+Canvas.TextWidth(Text[X])+5;
    CH[0]:=Text[X];
    if (X=SelStart) then Canvas.Font.Color:=FSelColor
    else Canvas.Font.Color:=Font.Color;
    DrawText(Canvas.Handle, CH , 1, Rect, DT_EXPANDTABS or DT_LEFT);
    Rect.Left:=Rect.Left+Canvas.TextWidth(Text[X]);
  end;
end;

procedure TPLabel.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect, CalcRect: TRect;
  DrawStyle: Integer;
begin
  with Canvas do
  begin
    if not Transparent then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    DrawStyle := DT_EXPANDTABS or WordWraps[FWordWrap] or Alignments[FAlignment];
    { Calculate vertical layout }
    if FLayout <> tlTop then
    begin
      CalcRect := Rect;
      DoDrawText(CalcRect, DrawStyle or DT_CALCRECT);
      if FLayout = tlBottom then OffsetRect(Rect, 0, Height - CalcRect.Bottom)
      else OffsetRect(Rect, 0, (Height - CalcRect.Bottom) div 2);
    end;
    DoDrawText(Rect, DrawStyle);
  end;
end;

procedure TPLabel.Loaded;
begin
  inherited Loaded;
//  If not FShowColors then AdjustBounds;
  AdjustBounds;
end;

procedure TPLabel.AdjustBounds;
const
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  DC: HDC;
  X: Integer;
  Rect: TRect;
begin
  if not (csReading in ComponentState) and FAutoSize then
  begin
    Rect := ClientRect;
    DC := GetDC(0);
    Canvas.Handle := DC;
    DoDrawText(Rect, (DT_EXPANDTABS or DT_CALCRECT) or WordWraps[FWordWrap]);
    Canvas.Handle := 0;
    ReleaseDC(0, DC);
    X := Left;
    if FAlignment = taRightJustify then Inc(X, Width - Rect.Right);
    SetBounds(X, Top, Rect.Right, Rect.Bottom);
  end;
end;

procedure TPLabel.SetAlignment(Value: TAlignment);
begin
  if FAlignment <> Value then
  begin
    FAlignment := Value;
    Invalidate;
  end;
end;

procedure TPLabel.SetAutoSize(Value: Boolean);
begin
  if FAutoSize <> Value then
  begin
    FAutoSize := Value;
    AdjustBounds;
  end;
end;

function TPLabel.GetTransparent: Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TPLabel.SetFocusControl(Value: TWinControl);
begin
  FFocusControl := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

procedure TPLabel.SetShowAccelChar(Value: Boolean);
begin
  if FShowAccelChar <> Value then
  begin
    FShowAccelChar := Value;
    Invalidate;
  end;
end;

procedure TPLabel.SetTransparent(Value: Boolean);
begin
  if Transparent <> Value then
  begin
    if Value then
      ControlStyle := ControlStyle - [csOpaque] else
      ControlStyle := ControlStyle + [csOpaque];
    Invalidate;
  end;
end;

procedure TPLabel.SetLayout(Value: TTextLayout);
begin
  if FLayout <> Value then
  begin
    FLayout := Value;
    Invalidate;
  end;
end;

procedure TPLabel.SetWordWrap(Value: Boolean);
begin
  if FWordWrap <> Value then
  begin
    FWordWrap := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TPLabel.SetShowColors(Value: Boolean);
begin
  if FShowColors<>Value then begin
    FShowColors:=Value;
    Invalidate;
  end;
end;

procedure TPLabel.SetSelColor(Value: TColor);
begin
  If FSelColor<>Value then begin
    FSelColor:=Value;
    Paint;
  end;
end;

procedure TPLabel.SetSelStart(Value: Integer);
begin
  If (FSelStart<>Value) and (Value>=0) then begin
    FSelStart:=Value;
    Paint;
  end;
end;

procedure TPLabel.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FFocusControl) then
    FFocusControl := nil;
end;

procedure TPLabel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
  AdjustBounds;
end;

procedure TPLabel.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustBounds;
end;

procedure TPLabel.CMDialogChar(var Message: TCMDialogChar);
begin
  if (FFocusControl <> nil) and Enabled and ShowAccelChar and
    IsAccel(Message.CharCode, Caption) then
    with FFocusControl do
      if CanFocus then
      begin
        SetFocus;
        Message.Result := 1;
      end;
end;


procedure Register;
begin
  RegisterComponents('PControls', [TPLabel]);
end;

end.
