{
 ***   Delphi component for USPS Postnet barcodes.
 ***   Written by Fergus (c) 1998.
 ***   Version 0.1 - Wednesday, April 22, 1998.
 ***   Freeware component - along with source
       as long as these initial comment lines are saved along.
 ***   Feel free to modify as per your requirements.
 ***   Please communicate any bugs/emails to 'sinha@purdue.edu'.
 ***   Let me know if you use/like this component thru email
 ***   Please check http://expert.cc.purdue.edu/~suvinay
       for updates and new components.      
 ***   About the component:

           * Uses the ZIP Code and the number in House Address to
             generate the USPS Postnet bar code. For example,
             the address is: 214, Hollywood St. IN 47907-2614.
             Use  Postnet.Zipcode := '47907-2614';
                  Postnet.StreetAddress := '412' ;
             to generate the postnet barcodes.
          * The StreetAddress is optional. When not supplied,
            only 52 bars are generated. If supplied 62 bars
            are generated.
          * Blanks and characters other than 0 to 9 and a hyphen
            in Zipcode are not allowed. They will be substituted
            by zero when generating barcodes.
}
unit Postnet;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;


{ The Postnet class declaration }
type
  TPostnet = class(TGraphicControl)
  private
    { Private declarations }
    FPen: TPen ;
    FBrush: TBrush ;
    FZipCode: String ;
    FStreetAddress: String ;
    FBorder: Boolean ;
    FLongBar: Integer ;
    FShortBar: Integer ;
    FSpacing: Integer ;
    procedure SetZipCode (VZipCode: String) ;
    procedure SetStreetAddress (VStreetAddress: String) ;
    procedure SetLongBar (VLongBar: Integer) ;
    procedure SetShortBar (VShortBar: Integer) ;
    procedure SetSpacing (VSpacing: Integer) ;
    procedure SetBorder (VBorder: Boolean) ;
    procedure PrintBar (Code, X, Y : Integer ) ;
    procedure SetPen (Value: TPen) ;
    procedure SetBrush (Value: TBrush) ;
    procedure RepaintRequest(Sender: TObject) ;
  protected
    { Protected declarations }
    XMove: Integer ;
    procedure Paint; override ;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Width default 310;
    property Height default 20;
    property Visible default TRUE;
    property ZipCode: String
     read FZipCode write SetZipCode  ;
    property StreetAddress: String
     read FStreetAddress write SetStreetAddress  ;
    property LongBar: Integer
     read FLongBar write SetLongBar default 10 ;
    property ShortBar: Integer
     read FShortBar write SetShortBar default 10 ;
    property Spacing: Integer
     read FSpacing write SetSpacing default 10 ;
    property Border: Boolean
     read FBorder write SetBorder default TRUE ;
    property Pen: TPen
     read FPen write SetPen ;
    property Brush: TBrush
     read FBrush write SetBrush ;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
end;


procedure Register;

implementation

{$R *.RES}

{ procedure to change to new zipcode }
procedure TPostnet.SetZipCode (VZipCode: String) ;
begin
 if FZipCode <> VZipCode then
 begin
  FZipCode := VZipCode ;
  Invalidate ;
 end ;
end ;

{ procedure to change to new StreetAddress }
procedure TPostnet.SetStreetAddress (VStreetAddress: String) ;
begin
 if FStreetAddress <> VStreetAddress then
 begin
  FStreetAddress := VStreetAddress ;
  Invalidate ;
 end ;
end ;

{ procedure to change to toggle border }
procedure TPostnet.SetBorder (VBorder: Boolean) ;
begin
 if FBorder <> VBorder then
 begin
  FBorder := VBorder ;
  Invalidate ;
 end ;
end ;

{ procedure to change to new long bar length in barcode }
procedure TPostnet.SetLongBar (VLongBar: Integer) ;
begin
 if FLongBar <> VLongBar then
 begin
  FLongBar := VLongBar ;
  Invalidate ;
 end ;
end ;

{ procedure to change to new short bar length in barcode }
procedure TPostnet.SetShortBar (VShortBar: Integer) ;
begin
 if FShortBar <> VShortBar then
 begin
  FShortBar := VShortBar ;
  Invalidate ;
 end ;
end ;

{ procedure to change to new spacing beetween bars in barcode }
procedure TPostnet.SetSpacing (VSpacing: Integer) ;
begin
 if FSpacing <> VSpacing then
 begin
  FSpacing := VSpacing ;
  Invalidate ;
 end ;
end ;

{ procedure to select new pen type }
procedure TPostnet.SetPen (Value: TPen) ;
begin
 FPen.Assign(Value) ;
 Invalidate ;
end ;

{ procedure to select new brush type }
procedure TPostnet.SetBrush (Value: TBrush) ;
begin
 FBrush.Assign(Value) ;
 Invalidate ;
end ;

{ procedure to handle new pen/brush change }
procedure TPostnet.RepaintRequest(Sender: TObject) ;
begin
 Invalidate ;
end;

{
  procedure that actually creates parses zipcode and
  street address to create the bars
}
procedure TPostnet.Paint;
const
 LongBar : Integer = 10 ;
 ShortBar: Integer = 5 ;
 SpaceOut : Integer = 5 ;
var
 XLoc, YLoc, iT, iL, i, CheckSum: Integer ;
 Code : array [1..10] of Integer ;
 MyZip1: String[6] ;
 MyZip2: String[4] ;
 MyZipCode: String[9] ;
 szTemp: String[2] ;
begin
 if FLongBar <= 0 then
  FLongBar := LongBar ;
 if FShortBar <= 0 then
  FShortBar := ShortBar ;
 if FSpacing <= 0 then
  FSpacing := SpaceOut ;
 // the width and height of control calculated at max.
 if Width < (Left+62*FSpacing) then
  Width := Left + 63*FSpacing ;
 if Height <= (FLongBar*2) then
  Height := FLongBar * 2 ;

 CheckSum := 0 ;
 Code[1]  := 184;
 Code[2]  := 163;
 Code[3]  := 165;
 Code[4]  := 166;
 Code[5]  := 169;
 Code[6]  := 170;
 Code[7]  := 172;
 Code[8]  := 177;
 Code[9]  := 178;
 Code[10] := 180;

 iL := Length(FZipCode) ;
 if iL > 10 then
  FZipCode := Copy(FZipCode, 1, 10) ;
 iT := Pos('-', FZipCode) ;
 if (iT > 0) then
  begin
   MyZip1 := Copy(FZipCode, 1, iT-1) ;
   MyZip2 := Copy(FZipCode, iT+1, 4) ;
   MyZipCode := MyZip1 + MyZip2 ;
  end
 else
  MyZipCode := FZipCode ;
 //draw border (optional)
 if FBorder then
  begin
   with Canvas do
   begin
    Brush := FBrush ;
    Pen := FPen ;
    MoveTo (Left, 0) ;
    Rectangle (Left, 0, Width+2, Height+2) ;
   end ;
  end ;
 iL := 9 ;
 XLoc := Left+5 ;
 YLoc := Height-2  ;
 //print long bar
 with Canvas do
 begin
  MoveTo(XLoc, YLoc) ;
  LineTo(XLoc, YLoc-FLongBar) ;
  XMove := XLoc + FSpacing ;
 end ;
 //print zipcode
 for i := 1 to iL do
 begin
  szTemp := MyZipCode[i] ;
  Printbar(Code[StrToInt(szTemp)+1], XMove, YLoc) ;
  Checksum := Checksum + StrToInt(szTemp) ;
 end ;
 //check for street address
 if FStreetAddress <> '' then
  begin
   iL := Length(FStreetAddress) ;
   if iL > 2 then
    FStreetAddress := Copy(FStreetAddress, iL-1, 2) ;
   for i := 1 to 2 do
   begin
    szTemp := FStreetAddress[i] ;
    Printbar(Code[StrToInt(szTemp)+1], XMove, YLoc) ;
    Checksum := Checksum + StrToInt(szTemp) ;
   end ; //end of for loop
  end ;

 Checksum := Checksum mod 10 ;
 Printbar(Code[10-Checksum+1], XMove, YLoc) ;
 //end with longbar
 with Canvas do
 begin
  Pen := FPen ;
  MoveTo(XMove, YLoc) ;
  LineTo(XMove, YLoc-FLongBar) ;
 end ;
end ;

{ procedure that actually draws bars on screen }
procedure TPostnet.PrintBar(Code, X, Y: Integer);
const
 LongBar : Integer = 10 ;
 ShortBar: Integer = 5 ;
 SpaceOut : Integer = 5 ;
var
 i, Len : Integer ;
begin
 if FLongBar <= 0 then
  FLongBar := LongBar ;
 if FShortBar <= 0 then
  FShortBar := ShortBar ;
 if FSpacing <= 0 then
  FSpacing := SpaceOut ;
 // print bars
 len := (Code shr 5) and $07 ;
 for i:= len-1 downto 0 do
  begin
   if ((Code and (1 shl i)) <> 0) then
   begin    // long bar
    with Canvas do
     begin
      Pen := FPen ;
      MoveTo(X, Y) ;
      LineTo(X, Y-FLongBar) ;
      X := X + FSpacing ;
     end ;
   end
   else
    begin    // short bar
     with Canvas do
      begin
       Pen := FPen ;
       MoveTo(X, Y) ;
       LineTo(X, Y-FShortBar) ;
       X := X + FSpacing ;
      end ;
   end ; // end of else
 end ; // end of for loop
 XMove := X ;
end;

{ Create the component with default values }
constructor TPostnet.Create (AOwner: TComponent) ;
begin
 inherited Create(AOwner) ; //call parent constructor
 FZipCode := '47906-2600' ; //set defualt zip code
 StreetAddress := '412' ;   //set default address
 FBorder := True ;          //border - true/false
 FPen := TPen.Create ;      //color of bars in postnet
 FBrush := TBrush.Create ;  // fill in borders
 FPen.OnChange := RepaintRequest ;
 FBrush.OnChange := RepaintRequest ;
end ;

{ Destroy the component }
destructor TPostnet.Destroy;
begin
 FPen.Free ;
 FBrush.Free ;
 inherited Destroy;  //call parent destructor
end;

{ Register the component }
procedure Register;
begin
  RegisterComponents('FRGTools', [TPostnet]);
end;

end.

{ end of component code }
{ thanks for using/looking thru this component }
