//==============================================
//       rspeedbutton.pas
//
//         Delphi.
//       SpeedButton  .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rSpeedButton;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Buttons, Menus, rUtils, rConst;

type
  TrSeparatorStyle = (rspSeparator, rspDivider);
  TMenuItemClickOption = (micoNone, micoBefore, micoAfter);

  TrWinControl = class(TWinControl)
  published
    property OnKeyUp;
  end;

  TrSpeedButton = class(TSpeedButton)
  private
    { Private declarations }
    FGlyphResKind: TBitmapResKind;
    FGlyphResource : string;
    FMenuItem      : TMenuItem;
    FPopupMenuItem : TMenuItem;
    FMenuShortCut: TShortCut;
    FOldMenuShortCut: TShortCut;
    FOldPopupShortCut: TShortCut;
    FOldMenuEnabled: Boolean;
    FMenuCallOption: TMenuItemClickOption;
    FOldMenuClick  : TNotifyEvent;
    FOldPopupClick : TNotifyEvent;
    FOldPopupEnabled: Boolean;
    FControlFocused:  TWinControl;
    FControlShortCut: TShortCut;
    FOldControlKeyUp: TKeyEvent;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure SetMenuItem(Value: TMenuItem);
    procedure SetPopupMenuItem(Value: TMenuItem);
    procedure SetMenuShortCut(Value: TShortCut);
    procedure UpdateGlyph;
    procedure SetGlyphResKind(Value: TBitmapResKind);
    procedure SetGlyphResource(Value: string);
    function IsGlyph: Boolean;
    procedure SetControlFocused(Value: TWinControl);
    procedure SetControlShortCut(Value: TShortCut);
  protected
    { Protected declarations }
    procedure SetMenuEnabled(AMenuItem: TMenuItem);
    procedure Loaded; override;
    procedure InternalClick(Sender: TObject);
    procedure InternalKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
  public
    { Public declarations }
    FControlKey: Word;
    FControlShift: TShiftState;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
  published
    { Published declarations }
    property Align;
    property Flat default True;
    property MenuItem: TMenuItem read  FMenuItem write SetMenuItem;
    property PopupMenuItem: TMenuItem read  FPopupMenuItem write SetPopupMenuItem;
    property MenuShortCut: TShortCut read FMenuShortCut write SetMenuShortCut
      default 0;
    property Glyph stored IsGlyph;
    property MenuCallOption: TMenuItemClickOption read FMenuCallOption
                                                  write FMenuCallOption
                                                  default micoNone;
    property GlyphResKind:  TBitmapResKind read FGlyphResKind write SetGlyphResKind
                                           default DefBRK;
    property GlyphResource: string read FGlyphResource write SetGlyphResource;
    property ControlFocused: TWinControl read FControlFocused write SetControlFocused;
    property ControlShortCut: TShortCut read FControlShortCut write SetControlShortCut
      default 0;
  end;

  TrSeparator = class(TGraphicControl)
  private
    FSeparatorStyle: TrSeparatorStyle;
    procedure SetSeparatorStyle(Value: TrSeparatorStyle);
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align default alLeft;
    property Width default 8;
    property ShowHint;
    property Visible;
    property SeparatorStyle: TrSeparatorStyle read FSeparatorStyle
                                              write SetSeparatorStyle
                                              default rspSeparator;
  end;

implementation

{ TrSpeedButton }

constructor TrSpeedButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMenuItem       := nil;
  FPopupMenuItem  := nil;
  FOldMenuClick   := nil;
  FOldPopupCLick  := nil;
  FMenuCallOption := micoNone;
  FGlyphResKind := DefBRK;
  FControlShortCut := 0;
  Flat := True;
end;

destructor TrSpeedButton.Destroy;
begin
  MenuItem       := nil;
  PopupMenuItem  := nil;
  ControlFocused := nil;
  inherited Destroy;
end;

function TrSpeedButton.IsGlyph: Boolean;
begin
  Result := (FGlyphResKind = rkNone) or (Trim(FGlyphResource) = '');
end;

procedure TrSpeedButton.SetMenuEnabled(AMenuItem: TMenuItem);
begin
  if Assigned(AMenuItem) then
    AMenuItem.Enabled := Enabled;
end;

procedure TrSpeedButton.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  SetMenuEnabled(FMenuItem);
  SetMenuEnabled(FPopupMenuItem);
end;

procedure TrSpeedButton.Notification(AComponent: TComponent;
                                     Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove)
  then begin
    if (AComponent = FMenuItem) and (FMenuItem <> nil)
    then begin
      MenuItem := nil;
      FOldMenuClick := nil;
    end else
    if (AComponent = FPopupMenuItem) and (FPopupMenuItem <> nil)
    then begin
      PopupMenuItem := nil;
      FOldPopupClick := nil;
    end else
    if AComponent = FControlFocused then begin
      FControlFocused := nil;
      FOldControlKeyUp := nil;
    end
  end
end;

procedure TrSpeedButton.InternalClick(Sender: TObject);
  procedure CallMenuItem;
  begin
    if (Sender=FMenuItem) and Assigned(FOldMenuClick)
    then FOldMenuClick(Self);
    if (Sender=FPopupMenuItem) and Assigned(FOldPopupClick)
    then FOldPopupClick(Self);
  end;
begin
  if Assigned(Sender) and (FMenuCallOption = micoBefore)
  then CallMenuItem;
  Click;
  if Assigned(Sender) and (FMenuCallOption = micoAfter)
  then CallMenuItem;
end;

procedure TrSpeedButton.SetMenuItem(Value: TMenuItem);
begin
  if FMenuItem <> Value then begin
    if (FMenuItem <> nil) then begin
      if Assigned(FOldMenuClick) then begin
        FMenuItem.OnClick := FOldMenuClick;
        FOldMenuClick := nil;
        FMenuItem.Enabled := FOldMenuEnabled;
        FMenuItem.ShortCut := FOldMenuShortCut;
      end;
      FMenuItem.Enabled := FOldMenuEnabled;
    end;
    FMenuItem := Value;
    if Value <> nil then begin
      FOldMenuClick := FMenuItem.OnClick;
      FMenuItem.OnClick := InternalClick;
      FOldMenuEnabled := FMenuItem.Enabled;
      FOldMenuShortCut := FMenuItem.ShortCut;
      if (FMenuItem.ShortCut = 0) and not(csDesigning in ComponentState) then
        FMenuItem.ShortCut := FMenuShortCut;
    end;
    SetMenuEnabled(FMenuItem);
  end;
end;

procedure TrSpeedButton.SetPopupMenuItem(Value: TMenuItem);
begin
  if FPopupMenuItem <> Value then begin
    if (FPopupMenuItem <> nil) then begin
      if Assigned(FOldPopupClick) then begin
        FPopupMenuItem.OnClick := FOldPopupClick;
        FOldPopupClick := nil;
        FPopupMenuItem.Enabled := FOldPopupEnabled;
        FPopupMenuItem.ShortCut := FOldPopupShortCut;
      end;
      FPopupMenuItem.Enabled := FOldPopupEnabled;
    end;
    FPopupMenuItem := Value;
    if Value <> nil then begin
      FOldPopupClick := FPopupMenuItem.OnClick;
      FPopupMenuItem.OnClick := InternalClick;
      FOldPopupEnabled := FPopupMenuItem.Enabled;
      FOldPopupShortCut := FPopupMenuItem.ShortCut;
      if (FPopupMenuItem.ShortCut = 0) and not(csDesigning in ComponentState) then
        FPopupMenuItem.ShortCut := FMenuShortCut;
    end;
    SetMenuEnabled(FPopupMenuItem);
  end;
end;

procedure TrSpeedButton.SetMenuShortCut(Value: TShortCut);
begin
  if FMenuShortCut <> Value then begin
    FMenuShortCut := Value;
    if not(csDesigning in ComponentState) then begin
      if FMenuItem <> nil then
        if FOldMenuShortCut = 0 then
          FMenuItem.ShortCut := FMenuShortCut;
      if FPopupMenuItem <> nil then
        if FOldPopupShortCut = 0 then
          FPopupMenuItem.ShortCut := FMenuShortCut;
    end;
  end;
end;

procedure TrSpeedButton.UpdateGlyph;
begin
  if IsGlyph or (csReading in ComponentState) then exit;
  try
    LoadBitmapFromResource(Glyph,hInstance,FGlyphResource,FGlyphResKind);
  except
    on EResNotFound do if not(csDesigning in ComponentState) then raise;
    else raise;
  end;
end;

procedure TrSpeedButton.SetGlyphResource;
begin
  if FGlyphResource <> Value then begin
    FGlyphResource := Value;
    if Trim(FGlyphResource) <> '' then UpdateGlyph;
  end;
end;

procedure TrSpeedButton.SetGlyphResKind;
begin
  if FGlyphResKind <> Value then begin
    FGlyphResKind := Value;
    if FGlyphResKind <> rkNone then UpdateGlyph;
  end;
end;

procedure TrSpeedButton.Loaded;
begin
  inherited;
  UpdateGlyph;
end;

procedure TrSpeedButton.InternalKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(FOldControlKeyUp) then FOldControlKeyUp(Sender,Key,Shift);
  if (Key = FControlKey) and (Shift = FControlShift) and Enabled then Click;
end;

procedure TrSpeedButton.SetControlFocused(Value: TWinControl);
begin
  if FControlFocused <> Value then begin
    if Assigned(FControlFocused) and Assigned(FOldControlKeyUp) then begin
      TrWinControl(FControlFocused).OnKeyUp := FOldControlKeyUp;
      FOldControlKeyUp := nil;
    end;
    FControlFocused := Value;
    if Assigned(FControlFocused) and (FControlShortCut<>0) then begin
      FOldControlKeyUp := TrWinControl(FControlFocused).OnKeyUp;
      TrWinControl(FControlFocused).OnKeyUp := InternalKeyUp;
    end;
  end;
end;

procedure TrSpeedButton.SetControlShortCut(Value: TShortCut);
begin
  if FControlShortCut <> Value then begin
    FControlShortCut := Value;
    ShortCutToKey(Value,FControlKey,FControlShift);
  end;
end;

{ TrSeparator }

constructor TrSeparator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Align := alLeft;
  Width := 8;
end;

procedure TrSeparator.SetSeparatorStyle(Value: TrSeparatorStyle);
begin
  if (Align in [alNone, alClient]) and (Value = rspDivider)
  then raise Exception.Create(srSepError);
  FSeparatorStyle := Value;
  Invalidate;
end;

procedure TrSeparator.Paint;
var
  R: TRect;
begin
  if FSeparatorStyle = rspDivider
  then with Canvas do if not (Align in [alNone, alClient]) then begin
    if Align in [alLeft, alRight]
    then begin
      R := Rect(Width div 2 - 1, 1, Width, Height - 1);
      DrawEdge(Handle, R, EDGE_ETCHED, BF_Left);
    end
    else begin
      R := Rect( 1, Height div 2 - 1, Width - 1, Height);
      DrawEdge(Handle, R, EDGE_ETCHED, BF_Top);
    end;
  end;
  if csDesigning in ComponentState then
   with Canvas do
   begin
     Pen.Style := psDot;
     Pen.Color := clBtnShadow;
     Brush.Style := bsClear;
     Rectangle(0, 0, ClientWidth, ClientHeight);
   end
end;

end.
